/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sedona.common.raster;

import java.awt.image.WritableRaster;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeSet;
import javax.media.jai.RasterFactory;
import org.apache.sedona.common.Functions;
import org.apache.sedona.common.raster.RasterAccessors;
import org.apache.sedona.common.raster.RasterPredicates;
import org.apache.sedona.common.utils.RasterUtils;
import org.geotools.api.geometry.BoundingBox;
import org.geotools.api.geometry.Bounds;
import org.geotools.api.referencing.FactoryException;
import org.geotools.coverage.grid.GridCoverage2D;
import org.geotools.coverage.grid.GridCoverageFactory;
import org.geotools.geometry.jts.JTS;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.LineSegment;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.Polygon;

public class Rasterization {
    protected static List<Object> rasterize(Geometry geom, GridCoverage2D raster, String pixelType, double value, boolean useGeometryExtent, boolean allTouched) throws FactoryException {
        double[] metadata = RasterAccessors.metadata(raster);
        Rasterization.validateRasterMetadata(metadata);
        if (!RasterPredicates.rsIntersects(raster, geom)) {
            throw new IllegalArgumentException("Geometry does not intersect Raster.");
        }
        ReferencedEnvelope rasterExtent = raster.getEnvelope2D();
        ReferencedEnvelope geomExtent = Rasterization.rasterizeGeomExtent(geom, raster, metadata, allTouched);
        RasterizationParams params = Rasterization.calculateRasterizationParams(raster, useGeometryExtent, metadata, geomExtent, pixelType);
        Rasterization.rasterizeGeometry(raster, metadata, geom, params, geomExtent, value, allTouched);
        GridCoverageFactory coverageFactory = new GridCoverageFactory();
        GridCoverage2D rasterized = useGeometryExtent ? coverageFactory.create((CharSequence)"rasterized", params.writableRaster, (Bounds)geomExtent) : coverageFactory.create((CharSequence)"rasterized", params.writableRaster, (Bounds)rasterExtent);
        ArrayList<Object> objects = new ArrayList<Object>();
        objects.add(params.writableRaster);
        objects.add(rasterized);
        return objects;
    }

    private static void rasterizeGeometry(GridCoverage2D raster, double[] metadata, Geometry geom, RasterizationParams params, ReferencedEnvelope geomExtent, double value, boolean allTouched) throws FactoryException {
        if (geomExtent == null) {
            return;
        }
        switch (geom.getGeometryType()) {
            case "GeometryCollection": 
            case "MultiPolygon": 
            case "MultiPoint": {
                Rasterization.rasterizeGeometryCollection(raster, metadata, geom, params, value, allTouched);
                break;
            }
            case "Point": {
                Rasterization.rasterizePoint(geom, params, geomExtent, value);
                break;
            }
            case "LineString": 
            case "MultiLineString": 
            case "LinearRing": {
                Rasterization.rasterizeLineString(geom, params, value, geomExtent);
                break;
            }
            default: {
                Rasterization.rasterizePolygon(geom, params, geomExtent, value, allTouched);
            }
        }
    }

    private static void rasterizeGeometryCollection(GridCoverage2D raster, double[] metadata, Geometry geom, RasterizationParams params, double value, boolean allTouched) throws FactoryException {
        for (int i = 0; i < geom.getNumGeometries(); ++i) {
            Geometry subGeom = geom.getGeometryN(i);
            ReferencedEnvelope subGeomExtent = Rasterization.rasterizeGeomExtent(subGeom, raster, metadata, allTouched);
            Rasterization.rasterizeGeometry(raster, metadata, subGeom, params, subGeomExtent, value, allTouched);
        }
    }

    private static void rasterizePoint(Geometry geom, RasterizationParams params, ReferencedEnvelope geomExtent, double value) {
        int startX = (int)Math.round((geomExtent.getMinX() - params.upperLeftX) / params.scaleX);
        int startY = (int)Math.round((geomExtent.getMinY() - params.upperLeftY) / params.scaleY);
        int x = startX;
        int y = startY;
        double worldY = geomExtent.getMinY();
        while (worldY < geomExtent.getMaxY()) {
            x = startX;
            double worldX = geomExtent.getMinX();
            while (worldX < geomExtent.getMaxX()) {
                int yIndex = -y - 1;
                double cellMaxX = worldX + params.scaleX;
                double cellMaxY = worldY + params.scaleY;
                boolean intersects = geom.intersects((Geometry)JTS.toGeometry((Envelope)new Envelope(worldX, cellMaxX, worldY, cellMaxY)));
                if (intersects) {
                    params.writableRaster.setSample(x, yIndex, 0, value);
                }
                worldX += params.scaleX;
                ++x;
            }
            worldY += params.scaleY;
            ++y;
        }
    }

    private static void rasterizeLineString(Geometry geom, RasterizationParams params, double value, ReferencedEnvelope geomExtent) {
        for (int i = 0; i < geom.getNumGeometries(); ++i) {
            LineString line = (LineString)geom.getGeometryN(i);
            Coordinate[] coords = line.getCoordinates();
            for (int j = 0; j < coords.length - 1; ++j) {
                LineSegment clippedSegment = Rasterization.clipSegmentToRasterBounds(coords[j], coords[j + 1], geomExtent);
                if (clippedSegment == null) continue;
                Coordinate start = new Coordinate(clippedSegment.p0.x, clippedSegment.p0.y);
                Coordinate end = new Coordinate(clippedSegment.p1.x, clippedSegment.p1.y);
                double x0 = (start.x - params.upperLeftX) / params.scaleX;
                double y0 = (params.upperLeftY - start.y) / params.scaleY;
                double x1 = (end.x - params.upperLeftX) / params.scaleX;
                double y1 = (params.upperLeftY - end.y) / params.scaleY;
                Rasterization.drawLineBresenham(params, x0, y0, x1, y1, value, 0.2);
            }
        }
    }

    private static void drawLineBresenham(RasterizationParams params, double x0, double y0, double x1, double y1, double value, double stepSize) {
        double dx = x1 - x0;
        double dy = y1 - y0;
        double distance = Math.sqrt(dx * dx + dy * dy);
        int steps = (int)Math.ceil(distance / stepSize);
        double stepX = dx / (double)steps;
        double stepY = dy / (double)steps;
        double x = x0;
        double y = y0;
        for (int i = 0; i <= steps; ++i) {
            int rasterX = (int)Math.floor(x);
            int rasterY = (int)Math.floor(y);
            if (rasterX >= 0 && rasterX < params.writableRaster.getWidth() && rasterY >= 0 && rasterY < params.writableRaster.getHeight()) {
                params.writableRaster.setSample(rasterX, rasterY, 0, value);
            }
            x += stepX;
            y += stepY;
        }
    }

    private static LineSegment clipSegmentToRasterBounds(Coordinate p1, Coordinate p2, ReferencedEnvelope geomExtent) {
        double temp;
        double tMax;
        double tMin;
        double minX = geomExtent.getMinX();
        double maxX = geomExtent.getMaxX();
        double minY = geomExtent.getMinY();
        double maxY = geomExtent.getMaxY();
        double x1 = p1.x;
        double y1 = p1.y;
        double x2 = p2.x;
        double y2 = p2.y;
        boolean p1Inside = Rasterization.isInsideBounds(x1, y1, minX, maxX, minY, maxY);
        boolean p2Inside = Rasterization.isInsideBounds(x2, y2, minX, maxX, minY, maxY);
        if (p1Inside && p2Inside) {
            return new LineSegment(p1, p2);
        }
        if (!p1Inside && !p2Inside) {
            return null;
        }
        double dx = x2 - x1;
        double dy = y2 - y1;
        double[] tValues = new double[]{0.0, 1.0};
        if (dx != 0.0) {
            tMin = (minX - x1) / dx;
            tMax = (maxX - x1) / dx;
            if (dx < 0.0) {
                temp = tMin;
                tMin = tMax;
                tMax = temp;
            }
            tValues[0] = Math.max(tValues[0], tMin);
            tValues[1] = Math.min(tValues[1], tMax);
        }
        if (dy != 0.0) {
            tMin = (minY - y1) / dy;
            tMax = (maxY - y1) / dy;
            if (dy < 0.0) {
                temp = tMin;
                tMin = tMax;
                tMax = temp;
            }
            tValues[0] = Math.max(tValues[0], tMin);
            tValues[1] = Math.min(tValues[1], tMax);
        }
        if (tValues[0] > tValues[1]) {
            return null;
        }
        Coordinate newP1 = new Coordinate(x1 + tValues[0] * dx, y1 + tValues[0] * dy);
        Coordinate newP2 = new Coordinate(x1 + tValues[1] * dx, y1 + tValues[1] * dy);
        return new LineSegment(newP1, newP2);
    }

    private static boolean isInsideBounds(double x, double y, double minX, double maxX, double minY, double maxY) {
        return x >= minX && x <= maxX && y >= minY && y <= maxY;
    }

    static ReferencedEnvelope rasterizeGeomExtent(Geometry geom, GridCoverage2D raster, double[] metadata, boolean allTouched) {
        Rasterization.validateRasterMetadata(metadata);
        if (Objects.equals(geom.getGeometryType(), "MultiLineString")) {
            allTouched = true;
        }
        if (Objects.equals(geom.getGeometryType(), "MultiPoint")) {
            allTouched = true;
        }
        ReferencedEnvelope geomExtent = new ReferencedEnvelope(geom.getEnvelopeInternal(), raster.getCoordinateReferenceSystem2D());
        double upperLeftX = metadata[0];
        double upperLeftY = metadata[1];
        double scaleX = metadata[4];
        double scaleY = metadata[5];
        double alignedMinX = Rasterization.toWorldCoordinate(Rasterization.toPixelIndex(geomExtent.getMinX(), scaleX, upperLeftX, true), scaleX, upperLeftX);
        double alignedMinY = Rasterization.toWorldCoordinate(Rasterization.toPixelIndex(geomExtent.getMinY(), scaleY, upperLeftY, true), scaleY, upperLeftY);
        double alignedMaxX = Rasterization.toWorldCoordinate(Rasterization.toPixelIndex(geomExtent.getMaxX(), scaleX, upperLeftX, false), scaleX, upperLeftX);
        double alignedMaxY = Rasterization.toWorldCoordinate(Rasterization.toPixelIndex(geomExtent.getMaxY(), scaleY, upperLeftY, false), scaleY, upperLeftY);
        if (alignedMaxX == alignedMinX) {
            alignedMinX -= scaleX;
            alignedMaxX += scaleX;
        }
        if (alignedMaxY == alignedMinY) {
            alignedMinY += scaleY;
            alignedMaxY -= scaleY;
        }
        double originalMinX = raster.getEnvelope().getMinimum(0);
        double originalMinY = raster.getEnvelope().getMinimum(1);
        double originalMaxX = raster.getEnvelope().getMaximum(0);
        double originalMaxY = raster.getEnvelope().getMaximum(1);
        if (alignedMinX >= originalMaxX || alignedMaxX <= originalMinX || alignedMinY >= originalMaxY || alignedMaxY <= originalMinY) {
            return null;
        }
        if (allTouched) {
            alignedMinX -= geomExtent.getMinX() == alignedMinX ? scaleX : 0.0;
            alignedMinY += geomExtent.getMinY() == alignedMinY ? scaleY : 0.0;
            alignedMaxX += geomExtent.getMaxX() == alignedMaxX ? scaleX : 0.0;
            alignedMaxY -= geomExtent.getMaxY() == alignedMaxY ? scaleY : 0.0;
        }
        alignedMinX = Math.max(alignedMinX, originalMinX);
        alignedMinY = Math.max(alignedMinY, originalMinY);
        alignedMaxX = Math.min(alignedMaxX, originalMaxX);
        alignedMaxY = Math.min(alignedMaxY, originalMaxY);
        ReferencedEnvelope alignedRasterExtent = new ReferencedEnvelope(alignedMinX, alignedMaxX, alignedMinY, alignedMaxY, geomExtent.getCoordinateReferenceSystem());
        return alignedRasterExtent;
    }

    private static double toPixelIndex(double coord, double scale, double upperLeft, boolean isMin) {
        BigDecimal rel = BigDecimal.valueOf(coord).subtract(BigDecimal.valueOf(upperLeft));
        BigDecimal px = rel.divide(BigDecimal.valueOf(scale), 16, RoundingMode.FLOOR);
        if (scale > 0.0) {
            return isMin ? px.setScale(0, RoundingMode.FLOOR).doubleValue() : px.setScale(0, RoundingMode.CEILING).doubleValue();
        }
        return isMin ? px.setScale(0, RoundingMode.CEILING).doubleValue() : px.setScale(0, RoundingMode.FLOOR).doubleValue();
    }

    private static double toWorldCoordinate(double pixelIndex, double scale, double upperLeft) {
        return BigDecimal.valueOf(pixelIndex).multiply(BigDecimal.valueOf(scale)).add(BigDecimal.valueOf(upperLeft)).doubleValue();
    }

    private static RasterizationParams calculateRasterizationParams(GridCoverage2D raster, boolean useGeometryExtent, double[] metadata, ReferencedEnvelope geomExtent, String pixelType) {
        WritableRaster writableRaster;
        double upperLeftX = 0.0;
        double upperLeftY = 0.0;
        if (useGeometryExtent) {
            upperLeftX = geomExtent.getMinX();
            upperLeftY = geomExtent.getMaxY();
        } else {
            upperLeftX = metadata[0];
            upperLeftY = metadata[1];
        }
        if (useGeometryExtent) {
            int geomExtentWidth = (int)Math.round(geomExtent.getWidth() / metadata[4]);
            int geomExtentHeight = (int)Math.round(geomExtent.getHeight() / -metadata[5]);
            writableRaster = RasterFactory.createBandedRaster((int)RasterUtils.getDataTypeCode(pixelType), (int)geomExtentWidth, (int)geomExtentHeight, (int)1, null);
        } else {
            int rasterWidth = (int)raster.getGridGeometry().getGridRange2D().getWidth();
            int rasterHeight = (int)raster.getGridGeometry().getGridRange2D().getHeight();
            writableRaster = RasterFactory.createBandedRaster((int)RasterUtils.getDataTypeCode(pixelType), (int)rasterWidth, (int)rasterHeight, (int)1, null);
        }
        return new RasterizationParams(writableRaster, pixelType, metadata[4], -metadata[5], upperLeftX, upperLeftY);
    }

    private static void validateRasterMetadata(double[] rasterMetadata) {
        if (rasterMetadata[4] < 0.0) {
            throw new IllegalArgumentException("ScaleX cannot be negative");
        }
        if (rasterMetadata[5] > 0.0) {
            throw new IllegalArgumentException("ScaleY must be negative");
        }
        if (rasterMetadata[6] != 0.0 || rasterMetadata[7] != 0.0) {
            throw new IllegalArgumentException("SkewX and SkewY must be zero");
        }
    }

    public static void rasterizePolygon(Geometry geom, RasterizationParams params, ReferencedEnvelope geomExtent, double value, boolean allTouched) {
        if (!(geom instanceof Polygon)) {
            throw new IllegalArgumentException("Only Polygon geometry is supported");
        }
        Geometry clippedGeom = Functions.intersection((Geometry)JTS.toGeometry((BoundingBox)geomExtent), Functions.buffer(geom, 0.0));
        if (Objects.equals(clippedGeom.getGeometryType(), "MultiPolygon")) {
            for (int i = 0; i < clippedGeom.getNumGeometries(); ++i) {
                Geometry subGeom = clippedGeom.getGeometryN(i);
                Rasterization.rasterizePolygon(subGeom, params, geomExtent, value, allTouched);
            }
            return;
        }
        Polygon polygon = (Polygon)clippedGeom;
        Map<Double, TreeSet<Double>> scanlineIntersections = Rasterization.computeScanlineIntersections(polygon, params, value, geomExtent, allTouched);
        Map<Integer, List<int[]>> scanlineFillRanges = Rasterization.computeFillRanges(scanlineIntersections);
        Rasterization.fillPolygon(scanlineFillRanges, params, value);
    }

    private static Map<Double, TreeSet<Double>> computeScanlineIntersections(Polygon polygon, RasterizationParams params, double value, ReferencedEnvelope geomExtent, boolean allTouched) {
        HashMap<Double, TreeSet<Double>> scanlineIntersections = new HashMap<Double, TreeSet<Double>>();
        ArrayList<LinearRing> allRings = new ArrayList<LinearRing>();
        allRings.add(polygon.getExteriorRing());
        for (int i = 0; i < polygon.getNumInteriorRing(); ++i) {
            allRings.add(polygon.getInteriorRingN(i));
        }
        for (LineString lineString : allRings) {
            Coordinate[] coords = lineString.getCoordinates();
            int numPoints = coords.length;
            if (allTouched) {
                Rasterization.rasterizeLineString((Geometry)lineString, params, value, geomExtent);
            }
            for (int i = 0; i < numPoints - 1; ++i) {
                Coordinate worldP1 = coords[i];
                Coordinate worldP2 = coords[i + 1];
                if (worldP1.y > worldP2.y) {
                    Coordinate temp = worldP1;
                    worldP1 = worldP2;
                    worldP2 = temp;
                }
                if (worldP1.y == worldP2.y) continue;
                double yStart = Math.round(BigDecimal.valueOf(params.upperLeftY).subtract(BigDecimal.valueOf(worldP1.y)).divide(BigDecimal.valueOf(params.scaleY), RoundingMode.CEILING).doubleValue());
                double yEnd = Math.round(BigDecimal.valueOf(params.upperLeftY).subtract(BigDecimal.valueOf(worldP2.y)).divide(BigDecimal.valueOf(params.scaleY), RoundingMode.FLOOR).doubleValue());
                yEnd = Math.max(0.5, Math.abs(yEnd) + 0.5);
                yStart = Math.min((double)params.writableRaster.getHeight() - 0.5, Math.abs(yStart) - 0.5);
                double p1X = (worldP1.x - params.upperLeftX) / params.scaleX;
                double p1Y = (params.upperLeftY - worldP1.y) / params.scaleY;
                if (worldP1.x == worldP2.x) {
                    for (double y = yStart; y >= yEnd; y -= 1.0) {
                        double xIntercept = p1X;
                        if (xIntercept < 0.0 || xIntercept > (double)params.writableRaster.getWidth()) continue;
                        scanlineIntersections.computeIfAbsent(y, k -> new TreeSet()).add(xIntercept);
                    }
                    continue;
                }
                double slope = (worldP2.y - worldP1.y) / (worldP2.x - worldP1.x);
                double xMin = (geomExtent.getMinX() - params.upperLeftX) / params.scaleX;
                double xMax = (geomExtent.getMaxX() - params.upperLeftX) / params.scaleX;
                for (double y = yStart; y >= yEnd; y -= 1.0) {
                    double xIntercept = p1X + (p1Y - y) / slope;
                    if (xIntercept < xMin || xIntercept >= xMax) continue;
                    scanlineIntersections.computeIfAbsent(y, k -> new TreeSet()).add(xIntercept);
                }
            }
        }
        return scanlineIntersections;
    }

    private static Map<Integer, List<int[]>> computeFillRanges(Map<Double, TreeSet<Double>> scanlineIntersections) {
        HashMap<Integer, List<int[]>> scanlineFillRanges = new HashMap<Integer, List<int[]>>();
        for (Map.Entry<Double, TreeSet<Double>> entry : scanlineIntersections.entrySet()) {
            double y = entry.getKey();
            TreeSet<Double> xIntercepts = entry.getValue();
            ArrayList<int[]> ranges = new ArrayList<int[]>();
            Iterator<Double> it = xIntercepts.iterator();
            while (it.hasNext()) {
                double x1 = it.next();
                if (!it.hasNext()) {
                    ranges.add(new int[]{(int)Math.floor(x1)});
                    continue;
                }
                double x2 = it.next();
                double firstCentroid = Math.ceil(x1 - 0.5) + 0.5;
                double lastCentroid = Math.floor(x2 + 0.5) - 0.5;
                if (lastCentroid < firstCentroid) continue;
                if (lastCentroid == firstCentroid) {
                    ranges.add(new int[]{(int)Math.floor(firstCentroid)});
                    continue;
                }
                ranges.add(new int[]{(int)Math.floor(firstCentroid), (int)Math.floor(lastCentroid)});
            }
            scanlineFillRanges.put((int)Math.floor(y), ranges);
        }
        return scanlineFillRanges;
    }

    private static void fillPolygon(Map<Integer, List<int[]>> scanlineFillRanges, RasterizationParams params, double value) {
        for (Map.Entry<Integer, List<int[]>> entry : scanlineFillRanges.entrySet()) {
            int y = entry.getKey();
            for (int[] range : entry.getValue()) {
                if (range.length == 1) {
                    params.writableRaster.setSample(range[0], y, 0, value);
                    continue;
                }
                int xStart = range[0];
                int xEnd = range[1];
                for (int x = xStart; x <= xEnd; ++x) {
                    params.writableRaster.setSample(x, y, 0, value);
                }
            }
        }
    }

    private static class RasterizationParams {
        WritableRaster writableRaster;
        String pixelType;
        double scaleX;
        double scaleY;
        double upperLeftX;
        double upperLeftY;

        RasterizationParams(WritableRaster writableRaster, String pixelType, double scaleX, double scaleY, double upperLeftX, double upperLeftY) {
            this.writableRaster = writableRaster;
            this.pixelType = pixelType;
            this.scaleX = scaleX;
            this.scaleY = scaleY;
            this.upperLeftX = upperLeftX;
            this.upperLeftY = upperLeftY;
        }
    }
}

