/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.ext.web.handler.impl;

import io.vertx.core.Vertx;
import io.vertx.core.VertxException;
import io.vertx.core.http.Cookie;
import io.vertx.core.http.CookieSameSite;
import io.vertx.core.impl.logging.Logger;
import io.vertx.core.impl.logging.LoggerFactory;
import io.vertx.ext.auth.impl.Codec;
import io.vertx.ext.auth.prng.VertxContextPRNG;
import io.vertx.ext.web.RoutingContext;
import io.vertx.ext.web.Session;
import io.vertx.ext.web.handler.CSRFHandler;
import io.vertx.ext.web.impl.Origin;
import io.vertx.ext.web.impl.RoutingContextInternal;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

public class CSRFHandlerImpl
implements CSRFHandler {
    private static final Logger LOG = LoggerFactory.getLogger(CSRFHandlerImpl.class);
    private final VertxContextPRNG random;
    private final Mac mac;
    private boolean nagHttps;
    private String cookieName = "XSRF-TOKEN";
    private String cookiePath = "/";
    private String headerName = "X-XSRF-TOKEN";
    private long timeout = 1800000L;
    private Origin origin;
    private boolean httpOnly;
    private boolean cookieSecure;

    public CSRFHandlerImpl(Vertx vertx, String secret) {
        try {
            if (secret.length() <= 8) {
                LOG.warn((Object)"CSRF secret is very short (<= 8 bytes)");
            }
            this.random = VertxContextPRNG.current((Vertx)vertx);
            this.mac = Mac.getInstance("HmacSHA256");
            this.mac.init(new SecretKeySpec(secret.getBytes(StandardCharsets.UTF_8), "HmacSHA256"));
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public CSRFHandler setOrigin(String origin) {
        this.origin = Origin.parse(origin);
        return this;
    }

    @Override
    public CSRFHandler setCookieName(String cookieName) {
        this.cookieName = cookieName;
        return this;
    }

    @Override
    public CSRFHandler setCookiePath(String cookiePath) {
        this.cookiePath = cookiePath;
        return this;
    }

    @Override
    public CSRFHandler setCookieHttpOnly(boolean httpOnly) {
        this.httpOnly = httpOnly;
        return this;
    }

    @Override
    public CSRFHandler setCookieSecure(boolean secure) {
        this.cookieSecure = secure;
        return this;
    }

    @Override
    public CSRFHandler setHeaderName(String headerName) {
        this.headerName = headerName;
        return this;
    }

    @Override
    public CSRFHandler setTimeout(long timeout) {
        this.timeout = timeout;
        return this;
    }

    @Override
    public CSRFHandler setNagHttps(boolean nag) {
        this.nagHttps = nag;
        return this;
    }

    private String generateAndStoreToken(RoutingContext ctx) {
        byte[] salt = new byte[32];
        this.random.nextBytes(salt);
        String saltPlusToken = Codec.base64UrlEncode((byte[])salt) + "." + System.currentTimeMillis();
        String signature = Codec.base64UrlEncode((byte[])this.mac.doFinal(saltPlusToken.getBytes(StandardCharsets.US_ASCII)));
        String token = saltPlusToken + "." + signature;
        Session session = ctx.session();
        if (session != null) {
            session.put(this.headerName, session.id() + "/" + token);
        }
        return token;
    }

    private String getTokenFromSession(RoutingContext ctx) {
        int idx;
        Session session = ctx.session();
        if (session == null) {
            return null;
        }
        String sessionToken = (String)session.get(this.headerName);
        if (sessionToken != null && (idx = sessionToken.indexOf(47)) != -1 && session.id() != null && session.id().equals(sessionToken.substring(0, idx))) {
            return sessionToken.substring(idx + 1);
        }
        return null;
    }

    private static boolean isBlank(String s) {
        return s == null || s.trim().isEmpty();
    }

    private static long parseLong(String s) {
        if (CSRFHandlerImpl.isBlank(s)) {
            return -1L;
        }
        try {
            return Long.parseLong(s);
        }
        catch (NumberFormatException e) {
            LOG.trace((Object)"Invalid Token format", (Throwable)e);
            return -1L;
        }
    }

    public void handle(RoutingContext ctx) {
        String methodName;
        String uri;
        if (((RoutingContextInternal)ctx).seenHandler(16)) {
            ctx.next();
            return;
        }
        ((RoutingContextInternal)ctx).visitHandler(16);
        if (this.nagHttps && LOG.isTraceEnabled() && (uri = ctx.request().absoluteURI()) != null && !uri.startsWith("https:")) {
            LOG.trace((Object)("Using session cookies without https could make you susceptible to session hijacking: " + uri));
        }
        if (!Origin.check(this.origin, ctx)) {
            ctx.fail(403, (Throwable)new VertxException("Invalid Origin", true));
            return;
        }
        switch (methodName = ctx.request().method().name()) {
            case "HEAD": 
            case "GET": {
                this.handleSafeMethod(ctx);
                break;
            }
            case "POST": 
            case "PUT": 
            case "DELETE": 
            case "PATCH": {
                this.handleUnsafeMethod(ctx);
                break;
            }
            default: {
                ctx.next();
            }
        }
    }

    private void handleSafeMethod(RoutingContext ctx) {
        String token;
        boolean sendCookie = true;
        Session session = ctx.session();
        if (session == null) {
            token = this.generateAndStoreToken(ctx);
        } else {
            String sessionToken = this.getTokenFromSession(ctx);
            if (sessionToken == null) {
                token = this.generateAndStoreToken(ctx);
            } else {
                String[] parts = sessionToken.split("\\.");
                long ts = CSRFHandlerImpl.parseLong(parts[1]);
                if (ts == -1L) {
                    token = this.generateAndStoreToken(ctx);
                } else if (System.currentTimeMillis() <= ts + this.timeout) {
                    token = sessionToken;
                    Cookie cookie = ctx.request().getCookie(this.cookieName);
                    if (cookie != null && token.equals(cookie.getValue())) {
                        sendCookie = false;
                    }
                } else {
                    token = this.generateAndStoreToken(ctx);
                }
            }
        }
        this.synchronizeWithClient(ctx, token, sendCookie);
        ctx.next();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void handleUnsafeMethod(RoutingContext ctx) {
        String[] tokens;
        Object challenge;
        byte[] cookieBytes;
        Cookie cookie = ctx.request().getCookie(this.cookieName);
        String header = ctx.request().getHeader(this.headerName);
        if (header == null) {
            if (!ctx.body().available()) {
                ctx.fail((Throwable)new VertxException("BodyHandler is required to process unsafe methods", true));
                return;
            }
            header = ctx.request().getFormAttribute(this.headerName);
        }
        if (header == null || cookie == null || CSRFHandlerImpl.isBlank(header)) {
            ctx.fail(403, new IllegalArgumentException("Token provided via HTTP Header/Form is absent/empty"));
            return;
        }
        String cookieValue = cookie.getValue();
        if (cookieValue == null || CSRFHandlerImpl.isBlank(cookieValue)) {
            ctx.fail(403, new IllegalArgumentException("Token provided via HTTP Header/Form is absent/empty"));
            return;
        }
        byte[] headerBytes = header.getBytes(StandardCharsets.UTF_8);
        if (!MessageDigest.isEqual(headerBytes, cookieBytes = cookieValue.getBytes(StandardCharsets.UTF_8))) {
            ctx.fail(403, new IllegalArgumentException("Token provided via HTTP Header and via Cookie are not equal"));
            return;
        }
        Session session = ctx.session();
        if (session != null) {
            String sessionToken = (String)session.get(this.headerName);
            if (sessionToken == null) {
                ctx.fail(403, new IllegalArgumentException("No Token has been added to the session"));
                return;
            }
            int idx = sessionToken.indexOf(47);
            if (idx != -1 && session.id() != null && session.id().equals(sessionToken.substring(0, idx))) {
                challenge = sessionToken.substring(idx + 1);
                if (!MessageDigest.isEqual(((String)challenge).getBytes(StandardCharsets.UTF_8), headerBytes)) {
                    ctx.fail(403, new IllegalArgumentException("Token has been used or is outdated"));
                    return;
                }
            } else {
                ctx.fail(403, new IllegalArgumentException("Token has been issued for a different session"));
                return;
            }
        }
        if ((tokens = header.split("\\.")).length != 3) {
            if (session != null) {
                session.remove(this.headerName);
            }
            ctx.fail(403);
            return;
        }
        byte[] saltPlusToken = (tokens[0] + "." + tokens[1]).getBytes(StandardCharsets.US_ASCII);
        challenge = this.mac;
        synchronized (challenge) {
            saltPlusToken = this.mac.doFinal(saltPlusToken);
        }
        byte[] signature = Codec.base64UrlEncode((byte[])saltPlusToken).getBytes(StandardCharsets.US_ASCII);
        if (!MessageDigest.isEqual(signature, tokens[2].getBytes(StandardCharsets.US_ASCII))) {
            ctx.fail(403, new IllegalArgumentException("Token signature does not match"));
            return;
        }
        long ts = CSRFHandlerImpl.parseLong(tokens[1]);
        if (ts == -1L) {
            if (session != null) {
                session.remove(this.headerName);
            }
            ctx.fail(403);
            return;
        }
        if (System.currentTimeMillis() <= ts + this.timeout) {
            String token = this.generateAndStoreToken(ctx);
            this.synchronizeWithClient(ctx, token, true);
            ctx.next();
            return;
        }
        if (session != null) {
            session.remove(this.headerName);
        }
        ctx.fail(403, new IllegalArgumentException("CSRF validity expired"));
    }

    private void synchronizeWithClient(RoutingContext ctx, String token, boolean cookie) {
        if (cookie) {
            ctx.response().addCookie(Cookie.cookie((String)this.cookieName, (String)token).setPath(this.cookiePath).setHttpOnly(this.httpOnly).setSecure(this.cookieSecure).setSameSite(CookieSameSite.STRICT));
        }
        ctx.put(this.headerName, token);
    }
}

