/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.hdds.client.ECReplicationConfig;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.DatanodeAdminError;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManager;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.node.DatanodeAdminMonitor;
import org.apache.hadoop.hdds.scm.node.DatanodeAdminMonitorImpl;
import org.apache.hadoop.hdds.scm.node.InvalidHostStringException;
import org.apache.hadoop.hdds.scm.node.InvalidNodeStateException;
import org.apache.hadoop.hdds.scm.node.NodeDecommissionMetrics;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeDecommissionManager {
    private final ScheduledExecutorService executor;
    private final DatanodeAdminMonitor monitor;
    private final NodeManager nodeManager;
    private ContainerManager containerManager;
    private final SCMContext scmContext;
    private final boolean useHostnames;
    private Integer maintenanceReplicaMinimum;
    private Integer maintenanceRemainingRedundancy;
    private final NodeDecommissionMetrics metrics;
    private static final Logger LOG = LoggerFactory.getLogger(NodeDecommissionManager.class);

    private List<DatanodeDetails> mapHostnamesToDatanodes(List<String> hosts, List<DatanodeAdminError> errors) {
        LinkedList<DatanodeDetails> results = new LinkedList<DatanodeDetails>();
        for (String hostString : hosts) {
            String msg;
            InetAddress addr;
            HostDefinition host;
            try {
                host = new HostDefinition(hostString);
                addr = InetAddress.getByName(host.getHostname());
            }
            catch (UnknownHostException | InvalidHostStringException e) {
                LOG.warn("Unable to resolve host {} ", (Object)hostString, (Object)e);
                errors.add(new DatanodeAdminError(hostString, e.getMessage()));
                continue;
            }
            String dnsName = this.useHostnames ? addr.getHostName() : addr.getHostAddress();
            List<DatanodeDetails> found = this.nodeManager.getNodesByAddress(dnsName);
            if (found.isEmpty()) {
                msg = "Host " + host.getRawHostname() + " (" + dnsName + ") is not running any datanodes registered with SCM. Please check the host name.";
                LOG.warn(msg);
                errors.add(new DatanodeAdminError(host.getRawHostname(), msg));
                continue;
            }
            if (found.size() == 1) {
                if (host.getPort() != -1 && !this.validateDNPortMatch(host.getPort(), found.get(0))) {
                    msg = "Host " + host.getRawHostname() + " is running a datanode registered with SCM, but the port number doesn't match. Please check the port number.";
                    LOG.warn(msg);
                    errors.add(new DatanodeAdminError(host.getRawHostname(), msg));
                    continue;
                }
                results.add(found.get(0));
                continue;
            }
            if (host.getPort() != -1) {
                HostDefinition finalHost = host;
                found.removeIf(dn -> !this.validateDNPortMatch(finalHost.getPort(), (DatanodeDetails)dn));
            }
            if (found.isEmpty()) {
                msg = "Host " + host.getRawHostname() + " is running multiple datanodes registered with SCM, but no port numbers match. Please check the port number.";
                LOG.warn(msg);
                errors.add(new DatanodeAdminError(host.getRawHostname(), msg));
                continue;
            }
            if (found.size() == 1) {
                results.add(found.get(0));
                continue;
            }
            if (this.allPortsMatch(found)) {
                DatanodeDetails mostRecent = this.findDnWithMostRecentHeartbeat(found);
                if (mostRecent == null) {
                    msg = "Host " + host.getRawHostname() + " has multiple datanodes registered with SCM. All have identical ports, but none have a newest heartbeat.";
                    LOG.warn(msg);
                    errors.add(new DatanodeAdminError(host.getRawHostname(), msg));
                    continue;
                }
                results.add(mostRecent);
                continue;
            }
            msg = "Host " + host.getRawHostname() + " is running multiple datanodes registered with SCM, but no port numbers match. Please check the port number.";
            LOG.warn(msg);
            errors.add(new DatanodeAdminError(host.getRawHostname(), msg));
        }
        return results;
    }

    private boolean allPortsMatch(List<DatanodeDetails> dns) {
        if (dns.size() < 2) {
            return true;
        }
        int port = dns.get(0).getRatisPort().getValue();
        for (int i = 1; i < dns.size(); ++i) {
            if (dns.get(i).getRatisPort().getValue() == port) continue;
            return false;
        }
        return true;
    }

    private DatanodeDetails findDnWithMostRecentHeartbeat(List<DatanodeDetails> dns) {
        if (dns.size() < 2) {
            return dns.isEmpty() ? null : dns.get(0);
        }
        List dnsWithHeartbeat = dns.stream().map(dn -> Pair.of((Object)dn, (Object)this.nodeManager.getLastHeartbeat((DatanodeDetails)dn))).sorted(Comparator.comparingLong(Pair::getRight)).collect(Collectors.toList());
        Pair last = (Pair)dnsWithHeartbeat.get(dnsWithHeartbeat.size() - 1);
        if ((Long)last.getRight() > (Long)((Pair)dnsWithHeartbeat.get(dnsWithHeartbeat.size() - 2)).getRight()) {
            return (DatanodeDetails)last.getLeft();
        }
        return null;
    }

    private boolean validateDNPortMatch(int port, DatanodeDetails dn) {
        return dn.hasPort(port);
    }

    public NodeDecommissionManager(OzoneConfiguration config, NodeManager nm, ContainerManager cm, SCMContext scmContext, EventPublisher eventQueue, ReplicationManager rm) {
        this.nodeManager = nm;
        this.containerManager = cm;
        this.scmContext = scmContext;
        this.executor = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat(scmContext.threadNamePrefix() + "DatanodeAdminManager-%d").setDaemon(true).build());
        this.useHostnames = config.getBoolean("hdds.datanode.use.datanode.hostname", false);
        long monitorIntervalMs = config.getOrFixDuration(LOG, "ozone.scm.datanode.admin.monitor.interval", "30s", TimeUnit.MILLISECONDS);
        this.setMaintenanceConfigs(config.getInt("hdds.scm.replication.maintenance.replica.minimum", 2), config.getInt("hdds.scm.replication.maintenance.remaining.redundancy", 1));
        this.monitor = new DatanodeAdminMonitorImpl(config, eventQueue, this.nodeManager, rm);
        this.metrics = NodeDecommissionMetrics.create();
        this.monitor.setMetrics(this.metrics);
        this.executor.scheduleAtFixedRate(this.monitor, monitorIntervalMs, monitorIntervalMs, TimeUnit.MILLISECONDS);
    }

    public Map<String, List<ContainerID>> getContainersPendingReplication(DatanodeDetails dn) throws NodeNotFoundException {
        return this.getMonitor().getContainersPendingReplication(dn);
    }

    @VisibleForTesting
    public DatanodeAdminMonitor getMonitor() {
        return this.monitor;
    }

    public synchronized List<DatanodeAdminError> decommissionNodes(List<String> nodes, boolean force) {
        ArrayList<DatanodeAdminError> errors = new ArrayList<DatanodeAdminError>();
        List<DatanodeDetails> dns = this.mapHostnamesToDatanodes(nodes, errors);
        if (!force) {
            LOG.info("Force flag = {}. Checking if decommission is possible for dns: {}", (Object)force, dns);
            boolean decommissionPossible = this.checkIfDecommissionPossible(dns, errors);
            if (!decommissionPossible) {
                LOG.error("Cannot decommission nodes as sufficient node are not available.");
                return errors;
            }
        } else {
            LOG.info("Force flag = {}. Skip checking if decommission is possible for dns: {}", (Object)force, dns);
        }
        for (DatanodeDetails dn : dns) {
            try {
                this.startDecommission(dn);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("The host {} was not found in SCM. Ignoring the request to decommission it", (Object)dn.getHostName());
                errors.add(new DatanodeAdminError(dn.getHostName(), "The host was not found in SCM"));
            }
            catch (InvalidNodeStateException e) {
                errors.add(new DatanodeAdminError(dn.getHostName(), e.getMessage()));
            }
        }
        return errors;
    }

    public synchronized void continueAdminForNode(DatanodeDetails dn) throws NodeNotFoundException {
        if (!this.scmContext.isLeader()) {
            LOG.info("follower SCM ignored continue admin for datanode {}", (Object)dn);
            return;
        }
        HddsProtos.NodeOperationalState opState = this.getNodeStatus(dn).getOperationalState();
        if (opState == HddsProtos.NodeOperationalState.DECOMMISSIONING || opState == HddsProtos.NodeOperationalState.ENTERING_MAINTENANCE || opState == HddsProtos.NodeOperationalState.IN_MAINTENANCE) {
            LOG.info("Continue admin for datanode {}", (Object)dn);
            this.monitor.startMonitoring(dn);
        }
    }

    public synchronized void startDecommission(DatanodeDetails dn) throws NodeNotFoundException, InvalidNodeStateException {
        NodeStatus nodeStatus = this.getNodeStatus(dn);
        HddsProtos.NodeOperationalState opState = nodeStatus.getOperationalState();
        if (opState == HddsProtos.NodeOperationalState.IN_SERVICE) {
            LOG.info("Starting Decommission for node {}", (Object)dn);
            this.nodeManager.setNodeOperationalState(dn, HddsProtos.NodeOperationalState.DECOMMISSIONING);
            this.monitor.startMonitoring(dn);
        } else if (nodeStatus.isDecommission()) {
            LOG.info("Start Decommission called on node {} in state {}. Nothing to do.", (Object)dn, (Object)opState);
        } else {
            LOG.error("Cannot decommission node {} in state {}", (Object)dn, (Object)opState);
            throw new InvalidNodeStateException("Cannot decommission node " + dn + " in state " + opState);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized boolean checkIfDecommissionPossible(List<DatanodeDetails> dns, List<DatanodeAdminError> errors) {
        int numDecom = dns.size();
        ArrayList<DatanodeDetails> validDns = new ArrayList<DatanodeDetails>(dns);
        int inServiceTotal = this.nodeManager.getNodeCount(NodeStatus.inServiceHealthy());
        for (DatanodeDetails dn : dns) {
            try {
                NodeStatus nodeStatus = this.getNodeStatus(dn);
                HddsProtos.NodeOperationalState opState = nodeStatus.getOperationalState();
                if (opState == HddsProtos.NodeOperationalState.IN_SERVICE) continue;
                --numDecom;
                validDns.remove(dn);
                LOG.warn("Cannot decommission {} because it is not IN-SERVICE", (Object)dn.getHostName());
            }
            catch (NodeNotFoundException ex) {
                --numDecom;
                validDns.remove(dn);
                LOG.warn("Cannot decommission {} because it is not found in SCM", (Object)dn.getHostName());
            }
        }
        for (DatanodeDetails dn : validDns) {
            Set<ContainerID> containers;
            try {
                containers = this.nodeManager.getContainers(dn);
            }
            catch (NodeNotFoundException ex) {
                LOG.warn("The host {} was not found in SCM. Ignoring the request to decommission it", (Object)dn.getHostName());
                continue;
            }
            for (ContainerID cid : containers) {
                ContainerInfo cif;
                try {
                    cif = this.containerManager.getContainer(cid);
                }
                catch (ContainerNotFoundException ex) {
                    LOG.warn("Could not find container info for container {}.", (Object)cid);
                    continue;
                }
                ContainerInfo containerInfo = cif;
                synchronized (containerInfo) {
                    if (cif.getState().equals((Object)HddsProtos.LifeCycleState.DELETED) || cif.getState().equals((Object)HddsProtos.LifeCycleState.DELETING)) {
                        continue;
                    }
                    int reqNodes = cif.getReplicationConfig().getRequiredNodes();
                    if (inServiceTotal - numDecom < reqNodes) {
                        int unHealthyTotal = this.nodeManager.getAllNodeCount() - inServiceTotal;
                        String errorMsg = "Insufficient nodes. Tried to decommission " + dns.size() + " nodes out of " + inServiceTotal + " IN-SERVICE HEALTHY and " + unHealthyTotal + " not IN-SERVICE or not HEALTHY nodes. Cannot decommission as a minimum of " + reqNodes + " IN-SERVICE HEALTHY nodes are required to maintain replication after decommission. ";
                        LOG.info(errorMsg + "Failing due to datanode : {}, container : {}", (Object)dn, (Object)cid);
                        errors.add(new DatanodeAdminError("AllHosts", errorMsg));
                        return false;
                    }
                }
            }
        }
        return true;
    }

    public synchronized List<DatanodeAdminError> recommissionNodes(List<String> nodes) {
        ArrayList<DatanodeAdminError> errors = new ArrayList<DatanodeAdminError>();
        List<DatanodeDetails> dns = this.mapHostnamesToDatanodes(nodes, errors);
        for (DatanodeDetails dn : dns) {
            try {
                this.recommission(dn);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Host {} was not found in SCM. Ignoring the request to recommission it.", (Object)dn.getHostName());
                errors.add(new DatanodeAdminError(dn.getHostName(), "The host was not found in SCM"));
            }
        }
        return errors;
    }

    public synchronized void recommission(DatanodeDetails dn) throws NodeNotFoundException {
        NodeStatus nodeStatus = this.getNodeStatus(dn);
        HddsProtos.NodeOperationalState opState = nodeStatus.getOperationalState();
        if (opState != HddsProtos.NodeOperationalState.IN_SERVICE) {
            this.monitor.stopMonitoring(dn);
            LOG.info("Queued node {} for recommission", (Object)dn);
        } else {
            LOG.info("Recommission called on node {} with state {}. Nothing to do.", (Object)dn, (Object)opState);
        }
    }

    public synchronized List<DatanodeAdminError> startMaintenanceNodes(List<String> nodes, int endInHours, boolean force) {
        ArrayList<DatanodeAdminError> errors = new ArrayList<DatanodeAdminError>();
        List<DatanodeDetails> dns = this.mapHostnamesToDatanodes(nodes, errors);
        if (!force) {
            LOG.info("Force flag = {}. Checking if maintenance is possible for dns: {}", (Object)force, dns);
            boolean maintenancePossible = this.checkIfMaintenancePossible(dns, errors);
            if (!maintenancePossible) {
                LOG.error("Cannot put nodes to maintenance as sufficient node are not available.");
                return errors;
            }
        } else {
            LOG.info("Force flag = {}. Skip checking if maintenance is possible for dns: {}", (Object)force, dns);
        }
        for (DatanodeDetails dn : dns) {
            try {
                this.startMaintenance(dn, endInHours);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("The host {} was not found in SCM. Ignoring the request to start maintenance on it", (Object)dn.getHostName());
            }
            catch (InvalidNodeStateException e) {
                errors.add(new DatanodeAdminError(dn.getHostName(), e.getMessage()));
            }
        }
        return errors;
    }

    public synchronized void startMaintenance(DatanodeDetails dn, int endInHours) throws NodeNotFoundException, InvalidNodeStateException {
        NodeStatus nodeStatus = this.getNodeStatus(dn);
        HddsProtos.NodeOperationalState opState = nodeStatus.getOperationalState();
        long maintenanceEnd = 0L;
        if (endInHours != 0) {
            maintenanceEnd = System.currentTimeMillis() / 1000L + (long)endInHours * 60L * 60L;
        }
        if (opState == HddsProtos.NodeOperationalState.IN_SERVICE) {
            this.nodeManager.setNodeOperationalState(dn, HddsProtos.NodeOperationalState.ENTERING_MAINTENANCE, maintenanceEnd);
            this.monitor.startMonitoring(dn);
            LOG.info("Starting Maintenance for node {}", (Object)dn);
        } else if (nodeStatus.isMaintenance()) {
            LOG.info("Starting Maintenance called on node {} with state {}. Nothing to do.", (Object)dn, (Object)opState);
        } else {
            LOG.error("Cannot start maintenance on node {} in state {}", (Object)dn, (Object)opState);
            throw new InvalidNodeStateException("Cannot start maintenance on node " + dn + " in state " + opState);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized boolean checkIfMaintenancePossible(List<DatanodeDetails> dns, List<DatanodeAdminError> errors) {
        int numMaintenance = dns.size();
        List validDns = dns.stream().collect(Collectors.toList());
        Collections.copy(validDns, dns);
        int inServiceTotal = this.nodeManager.getNodeCount(NodeStatus.inServiceHealthy());
        for (DatanodeDetails dn : dns) {
            try {
                NodeStatus nodeStatus = this.getNodeStatus(dn);
                HddsProtos.NodeOperationalState opState = nodeStatus.getOperationalState();
                if (opState == HddsProtos.NodeOperationalState.IN_SERVICE) continue;
                --numMaintenance;
                validDns.remove(dn);
                LOG.warn("{} cannot enter maintenance because it is not IN-SERVICE", (Object)dn.getHostName());
            }
            catch (NodeNotFoundException ex) {
                --numMaintenance;
                validDns.remove(dn);
                LOG.warn("{} cannot enter maintenance because it is not found in SCM", (Object)dn.getHostName());
            }
        }
        for (DatanodeDetails dn : validDns) {
            Set<ContainerID> containers;
            try {
                containers = this.nodeManager.getContainers(dn);
            }
            catch (NodeNotFoundException ex) {
                LOG.warn("The host {} was not found in SCM. Ignoring the request to enter maintenance", (Object)dn.getHostName());
                errors.add(new DatanodeAdminError(dn.getHostName(), "The host was not found in SCM"));
                continue;
            }
            for (ContainerID cid : containers) {
                ContainerInfo cif;
                try {
                    cif = this.containerManager.getContainer(cid);
                }
                catch (ContainerNotFoundException ex) {
                    continue;
                }
                ContainerInfo containerInfo = cif;
                synchronized (containerInfo) {
                    int minInService;
                    if (cif.getState().equals((Object)HddsProtos.LifeCycleState.DELETED) || cif.getState().equals((Object)HddsProtos.LifeCycleState.DELETING)) {
                        continue;
                    }
                    HddsProtos.ReplicationType replicationType = cif.getReplicationType();
                    if (replicationType.equals((Object)HddsProtos.ReplicationType.EC)) {
                        int reqNodes = cif.getReplicationConfig().getRequiredNodes();
                        int data = ((ECReplicationConfig)cif.getReplicationConfig()).getData();
                        minInService = Math.min(data + this.maintenanceRemainingRedundancy, reqNodes);
                    } else {
                        minInService = this.maintenanceReplicaMinimum;
                    }
                    if (inServiceTotal - numMaintenance < minInService) {
                        int unHealthyTotal = this.nodeManager.getAllNodeCount() - inServiceTotal;
                        String errorMsg = "Insufficient nodes. Tried to start maintenance for " + dns.size() + " nodes out of " + inServiceTotal + " IN-SERVICE HEALTHY and " + unHealthyTotal + " not IN-SERVICE or not HEALTHY nodes. Cannot enter maintenance mode as a minimum of " + minInService + " IN-SERVICE HEALTHY nodes are required to maintain replication after maintenance. ";
                        LOG.info(errorMsg + "Failing due to datanode : {}, container : {}", (Object)dn, (Object)cid);
                        errors.add(new DatanodeAdminError("AllHosts", errorMsg));
                        return false;
                    }
                }
            }
        }
        return true;
    }

    public void stop() {
        this.metrics.unRegister();
        if (this.executor != null) {
            this.executor.shutdown();
        }
    }

    private NodeStatus getNodeStatus(DatanodeDetails dn) throws NodeNotFoundException {
        return this.nodeManager.getNodeStatus(dn);
    }

    public void onBecomeLeader() {
        this.nodeManager.getAllNodes().forEach(datanodeDetails -> {
            try {
                this.continueAdminForNode((DatanodeDetails)datanodeDetails);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("NodeNotFound when adding the node to the decommissionManager", (Throwable)e);
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void setMaintenanceConfigs(int replicaMinimum, int remainingRedundancy) {
        NodeDecommissionManager nodeDecommissionManager = this;
        synchronized (nodeDecommissionManager) {
            this.maintenanceRemainingRedundancy = remainingRedundancy;
            this.maintenanceReplicaMinimum = replicaMinimum;
        }
    }

    static final class HostDefinition {
        private String rawHostname;
        private String hostname;
        private int port;

        HostDefinition(String hostname) throws InvalidHostStringException {
            this.rawHostname = hostname;
            this.parseHostname();
        }

        public String getRawHostname() {
            return this.rawHostname;
        }

        public String getHostname() {
            return this.hostname;
        }

        public int getPort() {
            return this.port;
        }

        private void parseHostname() throws InvalidHostStringException {
            try {
                URI uri = new URI("empty://" + this.rawHostname.trim());
                this.hostname = uri.getHost();
                this.port = uri.getPort();
                if (this.hostname == null) {
                    throw new InvalidHostStringException("The string " + this.rawHostname + " does not contain a value hostname or hostname:port definition");
                }
            }
            catch (URISyntaxException e) {
                throw new InvalidHostStringException("Unable to parse the host string " + this.rawHostname, e);
            }
        }
    }
}

