/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.storage;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.apache.hadoop.hdds.DatanodeVersion;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.ContainerClientMetrics;
import org.apache.hadoop.hdds.scm.OzoneClientConfig;
import org.apache.hadoop.hdds.scm.StreamBufferArgs;
import org.apache.hadoop.hdds.scm.XceiverClientFactory;
import org.apache.hadoop.hdds.scm.XceiverClientReply;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.storage.BufferPool;
import org.apache.hadoop.hdds.scm.storage.ContainerProtocolCalls;
import org.apache.hadoop.hdds.scm.storage.RatisBlockOutputStream;
import org.apache.hadoop.ozone.common.Checksum;
import org.apache.hadoop.ozone.common.ChecksumData;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.apache.hadoop.ozone.common.OzoneChecksumException;
import org.apache.hadoop.ozone.util.MetricUtil;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.util.DirectBufferPool;
import org.apache.hadoop.util.Time;
import org.apache.ozone.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.ozone.shaded.org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockOutputStream
extends OutputStream {
    static final Logger LOG = LoggerFactory.getLogger(BlockOutputStream.class);
    public static final String EXCEPTION_MSG = "Unexpected Storage Container Exception: ";
    public static final ContainerProtos.KeyValue INCREMENTAL_CHUNK_LIST_KV = ContainerProtos.KeyValue.newBuilder().setKey("incremental").build();
    public static final String FULL_CHUNK = "full";
    public static final ContainerProtos.KeyValue FULL_CHUNK_KV = ContainerProtos.KeyValue.newBuilder().setKey("full").build();
    private AtomicReference<BlockID> blockID;
    private long blockSize;
    private AtomicBoolean eofSent = new AtomicBoolean(false);
    private final AtomicReference<ContainerProtos.ChunkInfo> previousChunkInfo = new AtomicReference();
    private final ContainerProtos.BlockData.Builder containerBlockData;
    private volatile XceiverClientFactory xceiverClientFactory;
    private XceiverClientSpi xceiverClient;
    private OzoneClientConfig config;
    private StreamBufferArgs streamBufferArgs;
    private int chunkIndex;
    private final AtomicLong chunkOffset = new AtomicLong();
    private final BufferPool bufferPool;
    private static final DirectBufferPool DIRECT_BUFFER_POOL = new DirectBufferPool();
    private final AtomicReference<IOException> ioException;
    private final ExecutorService responseExecutor;
    private long totalWriteChunkLength;
    private long totalPutBlockLength;
    private long writtenDataLength;
    private List<ChunkBuffer> bufferList;
    private final List<DatanodeDetails> failedServers;
    private final Checksum checksum;
    private int flushPeriod;
    private int currentBufferRemaining;
    private ChunkBuffer currentBuffer;
    private ByteBuffer lastChunkBuffer;
    private long lastChunkOffset;
    private final Token<? extends TokenIdentifier> token;
    private final String tokenString;
    private int replicationIndex;
    private Pipeline pipeline;
    private final ContainerClientMetrics clientMetrics;
    private boolean allowPutBlockPiggybacking;
    private boolean supportIncrementalChunkList;
    private CompletableFuture<Void> lastFlushFuture;
    private CompletableFuture<Void> allPendingFlushFutures = CompletableFuture.completedFuture(null);

    public BlockOutputStream(BlockID blockID, long blockSize, XceiverClientFactory xceiverClientManager, Pipeline pipeline, BufferPool bufferPool, OzoneClientConfig config, Token<? extends TokenIdentifier> token, ContainerClientMetrics clientMetrics, StreamBufferArgs streamBufferArgs, Supplier<ExecutorService> blockOutputStreamResourceProvider) throws IOException {
        this.xceiverClientFactory = xceiverClientManager;
        this.config = config;
        this.blockID = new AtomicReference<BlockID>(blockID);
        this.blockSize = blockSize;
        this.replicationIndex = pipeline.getReplicaIndex(pipeline.getClosestNode());
        ContainerProtos.KeyValue keyValue = ContainerProtos.KeyValue.newBuilder().setKey("TYPE").setValue("KEY").build();
        ContainerProtos.DatanodeBlockID.Builder blkIDBuilder = ContainerProtos.DatanodeBlockID.newBuilder().setContainerID(blockID.getContainerID()).setLocalID(blockID.getLocalID()).setBlockCommitSequenceId(blockID.getBlockCommitSequenceId());
        if (this.replicationIndex > 0) {
            blkIDBuilder.setReplicaIndex(this.replicationIndex);
        }
        this.containerBlockData = ContainerProtos.BlockData.newBuilder().setBlockID(blkIDBuilder.build()).addMetadata(keyValue);
        this.pipeline = pipeline;
        this.supportIncrementalChunkList = this.canEnableIncrementalChunkList();
        LOG.debug("incrementalChunkList is {}", (Object)this.supportIncrementalChunkList);
        if (this.supportIncrementalChunkList) {
            this.containerBlockData.addMetadata(INCREMENTAL_CHUNK_LIST_KV);
            this.lastChunkBuffer = DIRECT_BUFFER_POOL.getBuffer(config.getStreamBufferSize());
            this.lastChunkOffset = 0L;
        } else {
            this.lastChunkBuffer = null;
        }
        this.xceiverClient = xceiverClientManager.acquireClient(pipeline);
        this.bufferPool = bufferPool;
        this.token = token;
        this.tokenString = this.token == null ? null : this.token.encodeToUrlString();
        this.currentBuffer = null;
        this.currentBufferRemaining = 0;
        this.flushPeriod = (int)(streamBufferArgs.getStreamBufferFlushSize() / (long)streamBufferArgs.getStreamBufferSize());
        Preconditions.checkArgument((long)this.flushPeriod * (long)streamBufferArgs.getStreamBufferSize() == streamBufferArgs.getStreamBufferFlushSize());
        this.responseExecutor = blockOutputStreamResourceProvider.get();
        this.bufferList = null;
        this.totalWriteChunkLength = 0L;
        this.totalPutBlockLength = 0L;
        this.writtenDataLength = 0L;
        this.failedServers = new CopyOnWriteArrayList<DatanodeDetails>();
        this.ioException = new AtomicReference<Object>(null);
        this.checksum = new Checksum(config.getChecksumType(), config.getBytesPerChecksum(), true);
        this.clientMetrics = clientMetrics;
        this.streamBufferArgs = streamBufferArgs;
        this.allowPutBlockPiggybacking = this.canEnablePutblockPiggybacking();
        LOG.debug("PutBlock piggybacking is {}", (Object)this.allowPutBlockPiggybacking);
    }

    private boolean canEnableIncrementalChunkList() {
        boolean confEnableIncrementalChunkList = this.config.getIncrementalChunkList();
        if (!confEnableIncrementalChunkList) {
            return false;
        }
        if (!(this instanceof RatisBlockOutputStream)) {
            LOG.debug("Unable to enable incrementalChunkList because BlockOutputStream is not a RatisBlockOutputStream");
            return false;
        }
        if (!this.allDataNodesSupportPiggybacking()) {
            LOG.debug("Unable to enable incrementalChunkList because not all datanodes support piggybacking");
            return false;
        }
        return confEnableIncrementalChunkList;
    }

    private boolean canEnablePutblockPiggybacking() {
        boolean confEnablePutblockPiggybacking = this.config.getEnablePutblockPiggybacking();
        if (!confEnablePutblockPiggybacking) {
            return false;
        }
        if (!this.allDataNodesSupportPiggybacking()) {
            LOG.debug("Unable to enable PutBlock piggybacking because not all datanodes support piggybacking");
            return false;
        }
        return confEnablePutblockPiggybacking;
    }

    private boolean allDataNodesSupportPiggybacking() {
        for (DatanodeDetails dn : this.pipeline.getNodes()) {
            LOG.debug("dn = {}, version = {}", (Object)dn, (Object)dn.getCurrentVersion());
            if (dn.getCurrentVersion() >= DatanodeVersion.COMBINED_PUTBLOCK_WRITECHUNK_RPC.toProtoValue()) continue;
            return false;
        }
        return true;
    }

    public BlockID getBlockID() {
        return this.blockID.get();
    }

    public long getTotalAckDataLength() {
        return 0L;
    }

    public synchronized long getWrittenDataLength() {
        return this.writtenDataLength;
    }

    public List<DatanodeDetails> getFailedServers() {
        return this.failedServers;
    }

    @VisibleForTesting
    public XceiverClientSpi getXceiverClient() {
        return this.xceiverClient;
    }

    @VisibleForTesting
    public long getTotalDataFlushedLength() {
        return this.totalPutBlockLength;
    }

    @VisibleForTesting
    public BufferPool getBufferPool() {
        return this.bufferPool;
    }

    public IOException getIoException() {
        return this.ioException.get();
    }

    public ContainerProtos.BlockData.Builder getContainerBlockData() {
        return this.containerBlockData;
    }

    public Pipeline getPipeline() {
        return this.pipeline;
    }

    protected String getTokenString() {
        return this.tokenString;
    }

    ExecutorService getResponseExecutor() {
        return this.responseExecutor;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void write(int b) throws IOException {
        this.checkOpen();
        BlockOutputStream blockOutputStream = this;
        synchronized (blockOutputStream) {
            this.allocateNewBufferIfNeeded();
            this.currentBuffer.put((byte)b);
            --this.currentBufferRemaining;
            this.updateWrittenDataLength(1);
            this.writeChunkIfNeeded();
            this.doFlushOrWatchIfNeeded();
        }
    }

    private void writeChunkIfNeeded() throws IOException {
        if (this.currentBufferRemaining == 0) {
            LOG.debug("WriteChunk from write(), buffer = {}", (Object)this.currentBuffer);
            this.clientMetrics.getWriteChunksDuringWrite().incr();
            this.writeChunk(this.currentBuffer);
            this.updateWriteChunkLength();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        this.checkOpen();
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
            throw new IndexOutOfBoundsException("Offset=" + off + " and len=" + len + " don't match the array length of " + b.length);
        }
        if (len == 0) {
            return;
        }
        BlockOutputStream blockOutputStream = this;
        synchronized (blockOutputStream) {
            while (len > 0) {
                this.allocateNewBufferIfNeeded();
                int writeLen = Math.min(this.currentBufferRemaining, len);
                this.currentBuffer.put(b, off, writeLen);
                this.currentBufferRemaining -= writeLen;
                this.updateWrittenDataLength(writeLen);
                this.writeChunkIfNeeded();
                off += writeLen;
                len -= writeLen;
                this.doFlushOrWatchIfNeeded();
            }
        }
    }

    protected synchronized void updateWrittenDataLength(int writeLen) {
        this.writtenDataLength += (long)writeLen;
    }

    private void doFlushOrWatchIfNeeded() throws IOException {
        if (this.currentBufferRemaining == 0) {
            if (this.bufferPool.getNumberOfUsedBuffers() % this.flushPeriod == 0) {
                this.updatePutBlockLength();
                CompletableFuture<PutBlockResult> putBlockFuture = this.executePutBlock(false, false);
                this.recordWatchForCommitAsync(putBlockFuture);
                this.clientMetrics.getFlushesDuringWrite().incr();
            }
            if (this.bufferPool.isAtCapacity()) {
                this.handleFullBuffer();
            }
        }
    }

    private void recordWatchForCommitAsync(CompletableFuture<PutBlockResult> putBlockResultFuture) {
        CompletionStage flushFuture = putBlockResultFuture.thenCompose(x -> this.watchForCommit(((PutBlockResult)x).commitIndex));
        Preconditions.checkState(Thread.holdsLock(this));
        this.lastFlushFuture = flushFuture;
        this.allPendingFlushFutures = this.allPendingFlushFutures.thenCombine(flushFuture, (last, curr) -> null);
    }

    private void allocateNewBufferIfNeeded() throws IOException {
        if (this.currentBufferRemaining == 0) {
            try {
                this.currentBuffer = this.bufferPool.allocateBuffer(this.config.getBufferIncrement());
                this.currentBufferRemaining = this.currentBuffer.remaining();
                LOG.debug("Allocated new buffer {}, used = {}, capacity = {}", new Object[]{this.currentBuffer, this.bufferPool.getNumberOfUsedBuffers(), this.bufferPool.getCapacity()});
            }
            catch (InterruptedException e) {
                this.handleInterruptedException(e, false);
            }
        }
    }

    private void updateWriteChunkLength() {
        Preconditions.checkState(Thread.holdsLock(this));
        this.totalWriteChunkLength = this.writtenDataLength;
    }

    private void updatePutBlockLength() {
        Preconditions.checkState(Thread.holdsLock(this));
        this.totalPutBlockLength = this.totalWriteChunkLength;
    }

    public synchronized void writeOnRetry(long len) throws IOException {
        if (len == 0L) {
            return;
        }
        List<ChunkBuffer> allocatedBuffers = this.bufferPool.getAllocatedBuffers();
        if (LOG.isDebugEnabled()) {
            LOG.debug("{}: Retrying write length {} on target blockID {}, {} buffers", new Object[]{this, len, this.blockID, allocatedBuffers.size()});
        }
        Preconditions.checkArgument(len <= this.streamBufferArgs.getStreamBufferMaxSize());
        int count = 0;
        while (len > 0L) {
            CompletableFuture<PutBlockResult> putBlockFuture;
            ChunkBuffer buffer = allocatedBuffers.get(count);
            long writeLen = Math.min((long)buffer.position(), len);
            len -= writeLen;
            ++count;
            this.writtenDataLength += writeLen;
            this.updateWriteChunkLength();
            this.updatePutBlockLength();
            LOG.debug("Write chunk on retry buffer = {}", (Object)buffer);
            if (this.allowPutBlockPiggybacking) {
                putBlockFuture = this.writeChunkAndPutBlock(buffer, false);
            } else {
                this.writeChunk(buffer);
                putBlockFuture = this.executePutBlock(false, false);
            }
            CompletionStage watchForCommitAsync = putBlockFuture.thenCompose(x -> this.watchForCommit(((PutBlockResult)x).commitIndex));
            try {
                ((CompletableFuture)watchForCommitAsync).get();
            }
            catch (InterruptedException e) {
                this.handleInterruptedException(e, true);
            }
            catch (ExecutionException e) {
                this.handleExecutionException(e);
            }
        }
    }

    private void handleFullBuffer() throws IOException {
        try {
            this.checkOpen();
            this.waitOnFlushFuture();
        }
        catch (ExecutionException e) {
            this.handleExecutionException(e);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            this.handleInterruptedException(ex, true);
        }
    }

    void releaseBuffersOnException() {
    }

    CompletableFuture<XceiverClientReply> sendWatchForCommit(long index) {
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Void> watchForCommit(long commitIndex) {
        try {
            this.checkOpen();
        }
        catch (IOException e2) {
            throw new FlushRuntimeException(e2);
        }
        LOG.debug("Entering watchForCommit commitIndex = {}", (Object)commitIndex);
        long start = Time.monotonicNowNanos();
        return ((CompletableFuture)((CompletableFuture)this.sendWatchForCommit(commitIndex).thenAccept(this::checkReply)).exceptionally(e -> {
            throw new FlushRuntimeException(this.setIoException((Throwable)e));
        })).whenComplete((r, e) -> {
            LOG.debug("Leaving watchForCommit commitIndex = {}", (Object)commitIndex);
            this.clientMetrics.getHsyncWatchForCommitNs().add(Time.monotonicNowNanos() - start);
        });
    }

    private void checkReply(XceiverClientReply reply) {
        if (reply == null) {
            return;
        }
        List<DatanodeDetails> dnList = reply.getDatanodes();
        if (dnList.isEmpty()) {
            return;
        }
        LOG.warn("Failed to commit BlockId {} on {}. Failed nodes: {}", new Object[]{this.blockID, this.xceiverClient.getPipeline(), dnList});
        this.failedServers.addAll(dnList);
    }

    void updateCommitInfo(XceiverClientReply reply, List<ChunkBuffer> buffers) {
    }

    CompletableFuture<PutBlockResult> executePutBlock(boolean close, boolean force) throws IOException {
        CompletionStage flushFuture;
        XceiverClientReply asyncReply;
        List<ChunkBuffer> byteBufferList;
        this.checkOpen();
        long flushPos = this.totalWriteChunkLength;
        if (!force) {
            Preconditions.checkNotNull(this.bufferList);
            byteBufferList = this.bufferList;
            this.bufferList = null;
            Preconditions.checkNotNull(byteBufferList);
        } else {
            byteBufferList = null;
        }
        try {
            ContainerProtos.BlockData blockData = this.containerBlockData.build();
            LOG.debug("sending PutBlock {} flushPos {}", (Object)blockData, (Object)flushPos);
            if (this.supportIncrementalChunkList) {
                this.containerBlockData.clearChunks();
            }
            boolean isBlockFull = this.blockSize != -1L && flushPos == this.blockSize;
            asyncReply = ContainerProtocolCalls.putBlockAsync(this.xceiverClient, blockData, close || isBlockFull, this.tokenString);
            CompletableFuture<ContainerProtos.ContainerCommandResponseProto> future = asyncReply.getResponse();
            flushFuture = ((CompletableFuture)future.thenApplyAsync(e -> {
                try {
                    this.validateResponse((ContainerProtos.ContainerCommandResponseProto)e);
                }
                catch (IOException sce) {
                    throw new CompletionException(sce);
                }
                if (this.getIoException() == null && !force) {
                    this.handleSuccessfulPutBlock(e.getPutBlock().getCommittedBlockLength(), asyncReply, flushPos, byteBufferList);
                    this.eofSent.set(close || isBlockFull);
                }
                return e;
            }, (Executor)this.responseExecutor)).exceptionally(e -> {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("putBlock failed for blockID {} with exception {}", this.blockID, (Object)e.getLocalizedMessage());
                }
                CompletionException ce = new CompletionException((Throwable)e);
                this.setIoException(ce);
                throw ce;
            });
        }
        catch (IOException | ExecutionException e2) {
            throw new IOException(EXCEPTION_MSG + e2.toString(), e2);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            this.handleInterruptedException(ex, false);
            return null;
        }
        return ((CompletableFuture)flushFuture).thenApply(r -> new PutBlockResult(asyncReply.getLogIndex(), (ContainerProtos.ContainerCommandResponseProto)r));
    }

    @Override
    public void flush() throws IOException {
        if (!(this.xceiverClientFactory == null || this.xceiverClient == null || this.bufferPool == null || this.bufferPool.getSize() <= 0 || this.streamBufferArgs.isStreamBufferFlushDelay() && this.unflushedLength() < (long)this.streamBufferArgs.getStreamBufferSize())) {
            this.handleFlush(false);
        }
    }

    private synchronized long unflushedLength() {
        return this.writtenDataLength - this.totalPutBlockLength;
    }

    private void writeChunkCommon(ChunkBuffer buffer) throws IOException {
        if (this.bufferList == null) {
            this.bufferList = new ArrayList<ChunkBuffer>();
        }
        this.bufferList.add(buffer);
    }

    private void writeChunk(ChunkBuffer buffer) throws IOException {
        this.writeChunkCommon(buffer);
        this.writeChunkToContainer(buffer.duplicate(0, buffer.position()), false, false);
    }

    private CompletableFuture<PutBlockResult> writeChunkAndPutBlock(ChunkBuffer buffer, boolean close) throws IOException {
        LOG.debug("WriteChunk and Putblock from flush, buffer={}", (Object)buffer);
        this.writeChunkCommon(buffer);
        return this.writeChunkToContainer(buffer.duplicate(0, buffer.position()), true, close);
    }

    protected void handleFlush(boolean close) throws IOException {
        try {
            this.handleFlushInternal(close);
            if (close) {
                this.waitForAllPendingFlushes();
            }
        }
        catch (ExecutionException e) {
            this.handleExecutionException(e);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            this.handleInterruptedException(ex, true);
        }
        catch (Throwable e) {
            String msg = "Failed to flush. error: " + e.getMessage();
            LOG.error(msg, e);
            throw e;
        }
        finally {
            if (close) {
                this.cleanup(false);
            }
        }
    }

    private void handleFlushInternal(boolean close) throws IOException, InterruptedException, ExecutionException {
        this.checkOpen();
        LOG.debug("Start handleFlushInternal close={}", (Object)close);
        CompletableFuture toWaitFor = MetricUtil.captureLatencyNs(this.clientMetrics.getHsyncSynchronizedWorkNs(), () -> this.handleFlushInternalSynchronized(close));
        if (toWaitFor != null) {
            LOG.debug("Waiting for flush");
            try {
                long startWaiting = Time.monotonicNowNanos();
                toWaitFor.get();
                this.clientMetrics.getHsyncWaitForFlushNs().add(Time.monotonicNowNanos() - startWaiting);
            }
            catch (ExecutionException ex) {
                if (ex.getCause() instanceof FlushRuntimeException) {
                    throw ((FlushRuntimeException)ex.getCause()).cause;
                }
                throw ex;
            }
            LOG.debug("Flush done.");
        }
        if (close) {
            this.allPendingFlushFutures.get();
        }
    }

    public void waitForAllPendingFlushes() throws IOException {
        try {
            this.allPendingFlushFutures.get();
        }
        catch (InterruptedException e) {
            this.handleInterruptedException(e, true);
        }
        catch (ExecutionException e) {
            this.handleExecutionException(e);
        }
    }

    private synchronized CompletableFuture<Void> handleFlushInternalSynchronized(boolean close) throws IOException {
        long start = Time.monotonicNowNanos();
        CompletableFuture<PutBlockResult> putBlockResultFuture = null;
        if (this.totalWriteChunkLength < this.writtenDataLength) {
            Preconditions.checkArgument(this.currentBuffer.position() > 0);
            this.updateWriteChunkLength();
            this.updatePutBlockLength();
            if (this.currentBuffer.hasRemaining()) {
                if (this.allowPutBlockPiggybacking) {
                    putBlockResultFuture = this.writeChunkAndPutBlock(this.currentBuffer, close);
                } else {
                    this.writeChunk(this.currentBuffer);
                    putBlockResultFuture = this.executePutBlock(close, false);
                }
                if (!close) {
                    this.currentBuffer = null;
                    this.currentBufferRemaining = 0;
                }
            } else {
                putBlockResultFuture = this.executePutBlock(close, false);
            }
        } else if (this.totalPutBlockLength < this.totalWriteChunkLength) {
            this.updatePutBlockLength();
            putBlockResultFuture = this.executePutBlock(close, false);
        } else if (close && !this.eofSent.get()) {
            this.updatePutBlockLength();
            putBlockResultFuture = this.executePutBlock(true, true);
        } else {
            LOG.debug("Flushing without data");
        }
        if (putBlockResultFuture != null) {
            this.recordWatchForCommitAsync(putBlockResultFuture);
        }
        this.clientMetrics.getHsyncSendWriteChunkNs().add(Time.monotonicNowNanos() - start);
        return this.lastFlushFuture;
    }

    @Override
    public void close() throws IOException {
        if (this.xceiverClientFactory != null && this.xceiverClient != null) {
            if (this.bufferPool != null && this.bufferPool.getSize() > 0) {
                this.handleFlush(true);
            } else {
                this.waitForAllPendingFlushes();
                this.cleanup(false);
            }
        }
    }

    void waitOnFlushFuture() throws InterruptedException, ExecutionException {
    }

    void validateResponse(ContainerProtos.ContainerCommandResponseProto responseProto) throws IOException {
        try {
            IOException exception = this.getIoException();
            if (exception != null) {
                throw exception;
            }
            ContainerProtocolCalls.validateContainerResponse(responseProto);
        }
        catch (StorageContainerException sce) {
            this.setIoException(sce);
            throw sce;
        }
    }

    public IOException setIoException(Throwable e) {
        IOException ioe = this.getIoException();
        if (ioe == null) {
            IOException exception = new IOException(EXCEPTION_MSG + e.toString(), e);
            this.ioException.compareAndSet(null, exception);
            LOG.debug("Exception: for block ID: " + this.blockID, e);
        } else {
            LOG.debug("Previous request had already failed with {} so subsequent request also encounters Storage Container Exception {}", (Object)ioe, (Object)e);
        }
        return this.getIoException();
    }

    void cleanup() {
    }

    public synchronized void cleanup(boolean invalidateClient) {
        if (this.xceiverClientFactory != null) {
            this.xceiverClientFactory.releaseClient(this.xceiverClient, invalidateClient);
        }
        this.xceiverClientFactory = null;
        this.xceiverClient = null;
        this.cleanup();
        if (this.bufferList != null) {
            this.bufferList.clear();
        }
        this.bufferList = null;
        if (this.lastChunkBuffer != null) {
            DIRECT_BUFFER_POOL.returnBuffer(this.lastChunkBuffer);
            this.lastChunkBuffer = null;
            this.checksum.clearChecksumCache();
        }
    }

    void checkOpen() throws IOException {
        if (this.isClosed()) {
            throw new IOException("BlockOutputStream has been closed.");
        }
        if (this.getIoException() != null) {
            throw this.getIoException();
        }
    }

    public boolean isClosed() {
        return this.xceiverClient == null;
    }

    CompletableFuture<ContainerProtos.ContainerCommandResponseProto> writeChunkToContainer(ChunkBuffer chunk) throws IOException {
        return this.writeChunkToContainer(chunk, false, false).thenApply(x -> ((PutBlockResult)x).response);
    }

    private CompletableFuture<PutBlockResult> writeChunkToContainer(ChunkBuffer chunk, boolean putBlockPiggybacking, boolean close) throws IOException {
        XceiverClientReply asyncReply;
        long expectedOffset;
        ContainerProtos.ChunkInfo previous;
        int effectiveChunkSize = chunk.remaining();
        long offset = this.chunkOffset.getAndAdd(effectiveChunkSize);
        ByteString data = chunk.toByteString(this.bufferPool.byteStringConversion());
        ChecksumData checksumData = this.checksum.computeChecksum(chunk, false);
        ContainerProtos.ChunkInfo chunkInfo = ContainerProtos.ChunkInfo.newBuilder().setChunkName(this.blockID.get().getLocalID() + "_chunk_" + ++this.chunkIndex).setOffset(offset).setLen(effectiveChunkSize).setChecksumData(checksumData.getProtoBufMessage()).build();
        long flushPos = this.totalWriteChunkLength;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Writing chunk {} length {} at offset {}", new Object[]{chunkInfo.getChunkName(), effectiveChunkSize, offset});
        }
        long l = (previous = this.previousChunkInfo.getAndSet(chunkInfo)) == null ? 0L : (expectedOffset = chunkInfo.getChunkName().equals(previous.getChunkName()) ? previous.getOffset() : previous.getOffset() + previous.getLen());
        if (chunkInfo.getOffset() != expectedOffset) {
            throw new IOException("Unexpected offset: " + chunkInfo.getOffset() + "(actual) != " + expectedOffset + "(expected), " + this.blockID + ", chunkInfo = " + chunkInfo + ", previous = " + previous);
        }
        CompletionStage validateFuture = null;
        try {
            List<ChunkBuffer> byteBufferList;
            ContainerProtos.BlockData blockData = null;
            if (this.supportIncrementalChunkList) {
                this.updateBlockDataForWriteChunk(chunk);
            } else {
                this.containerBlockData.addChunks(chunkInfo);
            }
            if (putBlockPiggybacking) {
                Preconditions.checkNotNull(this.bufferList);
                byteBufferList = this.bufferList;
                this.bufferList = null;
                Preconditions.checkNotNull(byteBufferList);
                blockData = this.containerBlockData.build();
                LOG.debug("piggyback chunk list {}", (Object)blockData);
                if (this.supportIncrementalChunkList) {
                    this.containerBlockData.clearChunks();
                }
            } else {
                byteBufferList = null;
            }
            asyncReply = ContainerProtocolCalls.writeChunkAsync(this.xceiverClient, chunkInfo, this.blockID.get(), data, this.tokenString, this.replicationIndex, blockData, close);
            CompletableFuture<ContainerProtos.ContainerCommandResponseProto> respFuture = asyncReply.getResponse();
            validateFuture = ((CompletableFuture)respFuture.thenApplyAsync(e -> {
                try {
                    this.validateResponse((ContainerProtos.ContainerCommandResponseProto)e);
                }
                catch (IOException sce) {
                    respFuture.completeExceptionally(sce);
                }
                if (this.getIoException() == null && putBlockPiggybacking) {
                    this.handleSuccessfulPutBlock(e.getWriteChunk().getCommittedBlockLength(), asyncReply, flushPos, byteBufferList);
                }
                return e;
            }, (Executor)this.responseExecutor)).exceptionally(e -> {
                String msg = "Failed to write chunk " + chunkInfo.getChunkName() + " into block " + this.blockID;
                LOG.debug("{}, exception: {}", (Object)msg, (Object)e.getLocalizedMessage());
                CompletionException ce = new CompletionException(msg, (Throwable)e);
                this.setIoException(ce);
                throw ce;
            });
            this.clientMetrics.recordWriteChunk(this.pipeline, chunkInfo.getLen());
        }
        catch (IOException | ExecutionException e2) {
            throw new IOException(EXCEPTION_MSG + e2.toString(), e2);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            this.handleInterruptedException(ex, false);
            return null;
        }
        return ((CompletableFuture)validateFuture).thenApply(x -> new PutBlockResult(asyncReply.getLogIndex(), (ContainerProtos.ContainerCommandResponseProto)x));
    }

    private void handleSuccessfulPutBlock(ContainerProtos.GetCommittedBlockLengthResponseProto e, XceiverClientReply asyncReply, long flushPos, List<ChunkBuffer> byteBufferList) {
        BlockID responseBlockID = BlockID.getFromProtobuf(e.getBlockID());
        Preconditions.checkState(this.blockID.get().getContainerBlockID().equals(responseBlockID.getContainerBlockID()));
        this.blockID.set(responseBlockID);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Adding index " + asyncReply.getLogIndex() + " flushLength " + flushPos + " numBuffers " + byteBufferList.size() + " blockID " + this.blockID + " bufferPool size " + this.bufferPool.getSize());
        }
        this.updateCommitInfo(asyncReply, byteBufferList);
    }

    private void updateBlockDataForWriteChunk(ChunkBuffer chunk) throws OzoneChecksumException {
        this.removeLastPartialChunk();
        chunk.rewind();
        LOG.debug("Adding chunk pos {} limit {} remaining {}.lastChunkBuffer pos {} limit {} remaining {} lastChunkOffset = {}", new Object[]{chunk.position(), chunk.limit(), chunk.remaining(), this.lastChunkBuffer.position(), this.lastChunkBuffer.limit(), this.lastChunkBuffer.remaining(), this.lastChunkOffset});
        if (this.lastChunkBuffer.position() + chunk.remaining() <= this.lastChunkBuffer.capacity()) {
            this.appendLastChunkBuffer(chunk, 0, chunk.remaining());
        } else {
            int remainingBufferSize = this.lastChunkBuffer.capacity() - this.lastChunkBuffer.position();
            this.appendLastChunkBuffer(chunk, 0, remainingBufferSize);
            this.updateBlockDataWithLastChunkBuffer();
            this.appendLastChunkBuffer(chunk, remainingBufferSize, chunk.remaining() - remainingBufferSize);
        }
        LOG.debug("after append, lastChunkBuffer={} lastChunkOffset={}", (Object)this.lastChunkBuffer, (Object)this.lastChunkOffset);
        this.updateBlockDataWithLastChunkBuffer();
    }

    private void updateBlockDataWithLastChunkBuffer() throws OzoneChecksumException {
        ContainerProtos.ChunkInfo lastChunkInfo = this.createChunkInfo(this.lastChunkOffset);
        LOG.debug("lastChunkInfo = {}", (Object)lastChunkInfo);
        long lastChunkSize = lastChunkInfo.getLen();
        this.addToBlockData(lastChunkInfo);
        this.lastChunkBuffer.clear();
        if (lastChunkSize == (long)this.config.getStreamBufferSize()) {
            this.lastChunkOffset += (long)this.config.getStreamBufferSize();
            this.checksum.clearChecksumCache();
        } else {
            this.lastChunkBuffer.position((int)lastChunkSize);
        }
    }

    private void appendLastChunkBuffer(ChunkBuffer chunkBuffer, int offset, int length) {
        LOG.debug("copying to last chunk buffer offset={} length={}", (Object)offset, (Object)length);
        int pos = 0;
        int uncopied = length;
        for (ByteBuffer bb : chunkBuffer.asByteBufferList()) {
            if (pos + bb.remaining() >= offset) {
                int copyStart = offset < pos ? 0 : offset - pos;
                int copyLen = Math.min(uncopied, bb.remaining());
                try {
                    LOG.debug("put into last chunk buffer start = {} len = {}", (Object)copyStart, (Object)copyLen);
                    int origPos = bb.position();
                    int origLimit = bb.limit();
                    bb.position(copyStart).limit(copyStart + copyLen);
                    this.lastChunkBuffer.put(bb);
                    bb.position(origPos).limit(origLimit);
                }
                catch (BufferOverflowException e) {
                    LOG.error("appending from " + copyStart + " for len=" + copyLen + ". lastChunkBuffer remaining=" + this.lastChunkBuffer.remaining() + " pos=" + this.lastChunkBuffer.position() + " limit=" + this.lastChunkBuffer.limit() + " capacity=" + this.lastChunkBuffer.capacity());
                    throw e;
                }
                uncopied -= copyLen;
            }
            if ((pos += bb.remaining()) >= offset + length) {
                return;
            }
            if (uncopied != 0) continue;
            return;
        }
    }

    private void removeLastPartialChunk() {
        if (this.containerBlockData.getChunksList().isEmpty()) {
            return;
        }
        int lastChunkIndex = this.containerBlockData.getChunksCount() - 1;
        ContainerProtos.ChunkInfo lastChunkInBlockData = this.containerBlockData.getChunks(lastChunkIndex);
        if (!this.isFullChunk(lastChunkInBlockData)) {
            this.containerBlockData.removeChunks(lastChunkIndex);
        }
    }

    private ContainerProtos.ChunkInfo createChunkInfo(long lastPartialChunkOffset) throws OzoneChecksumException {
        this.lastChunkBuffer.flip();
        int revisedChunkSize = this.lastChunkBuffer.remaining();
        ChecksumData revisedChecksumData = this.checksum.computeChecksum(this.lastChunkBuffer, true);
        long chunkID = lastPartialChunkOffset / (long)this.config.getStreamBufferSize();
        ContainerProtos.ChunkInfo.Builder revisedChunkInfo = ContainerProtos.ChunkInfo.newBuilder().setChunkName(this.blockID.get().getLocalID() + "_chunk_" + chunkID).setOffset(lastPartialChunkOffset).setLen(revisedChunkSize).setChecksumData(revisedChecksumData.getProtoBufMessage());
        if (revisedChunkSize == this.config.getStreamBufferSize()) {
            revisedChunkInfo.addMetadata(FULL_CHUNK_KV);
        }
        return revisedChunkInfo.build();
    }

    private boolean isFullChunk(ContainerProtos.ChunkInfo chunkInfo) {
        Preconditions.checkState(chunkInfo.getLen() <= (long)this.config.getStreamBufferSize());
        return chunkInfo.getLen() == (long)this.config.getStreamBufferSize();
    }

    private void addToBlockData(ContainerProtos.ChunkInfo revisedChunkInfo) {
        LOG.debug("containerBlockData chunk: {}", (Object)this.containerBlockData);
        if (this.containerBlockData.getChunksCount() > 0) {
            ContainerProtos.ChunkInfo lastChunk = this.containerBlockData.getChunks(this.containerBlockData.getChunksCount() - 1);
            LOG.debug("revisedChunkInfo chunk: {}", (Object)revisedChunkInfo);
            Preconditions.checkState(lastChunk.getOffset() + lastChunk.getLen() == revisedChunkInfo.getOffset(), "lastChunk.getOffset() + lastChunk.getLen() != revisedChunkInfo.getOffset()");
        }
        this.containerBlockData.addChunks(revisedChunkInfo);
    }

    @VisibleForTesting
    public void setXceiverClient(XceiverClientSpi xceiverClient) {
        this.xceiverClient = xceiverClient;
    }

    void handleInterruptedException(Exception ex, boolean processExecutionException) throws IOException {
        LOG.error("Command execution was interrupted.");
        if (!processExecutionException) {
            throw new IOException(EXCEPTION_MSG + ex.toString(), ex);
        }
        this.handleExecutionException(ex);
    }

    private void handleExecutionException(Exception ex) throws IOException {
        this.setIoException(ex);
        throw this.getIoException();
    }

    protected synchronized CompletableFuture<Void> getLastFlushFuture() {
        return this.lastFlushFuture;
    }

    public int getReplicationIndex() {
        return this.replicationIndex;
    }

    private static class FlushRuntimeException
    extends RuntimeException {
        private final IOException cause;

        FlushRuntimeException(IOException cause) {
            this.cause = cause;
        }
    }

    static class PutBlockResult {
        private final long commitIndex;
        private final ContainerProtos.ContainerCommandResponseProto response;

        PutBlockResult(long commitIndex, ContainerProtos.ContainerCommandResponseProto response) {
            this.commitIndex = commitIndex;
            this.response = response;
        }

        ContainerProtos.ContainerCommandResponseProto getResponse() {
            return this.response;
        }
    }
}

