/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.helpers;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.utils.MetadataKeyFilters;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.container.checksum.ContainerChecksumTreeManager;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.interfaces.BlockIterator;
import org.apache.hadoop.ozone.container.common.interfaces.DBHandle;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.utils.ContainerInspectorUtil;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.BlockUtils;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerLocationUtil;
import org.apache.hadoop.ozone.container.metadata.AbstractDatanodeStore;
import org.apache.hadoop.ozone.container.metadata.DatanodeStore;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreSchemaOneImpl;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreSchemaTwoImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KeyValueContainerUtil {
    private static final Logger LOG = LoggerFactory.getLogger(KeyValueContainerUtil.class);

    private KeyValueContainerUtil() {
    }

    public static void createContainerMetaData(File containerMetaDataPath, File chunksPath, File dbFile, String schemaVersion, ConfigurationSource conf) throws IOException {
        AbstractDatanodeStore store;
        Preconditions.checkNotNull((Object)containerMetaDataPath);
        Preconditions.checkNotNull((Object)conf);
        if (!containerMetaDataPath.mkdirs()) {
            LOG.error("Unable to create directory for metadata storage. Path: {}", (Object)containerMetaDataPath);
            throw new IOException("Unable to create directory for metadata storage. Path: " + containerMetaDataPath);
        }
        if (!chunksPath.mkdirs()) {
            LOG.error("Unable to create chunks directory Container {}", (Object)chunksPath);
            FileUtils.deleteDirectory((File)containerMetaDataPath);
            FileUtils.deleteDirectory((File)containerMetaDataPath.getParentFile());
            throw new IOException("Unable to create directory for data storage. Path: " + chunksPath);
        }
        if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "1")) {
            store = new DatanodeStoreSchemaOneImpl(conf, dbFile.getAbsolutePath(), false);
        } else if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "2")) {
            store = new DatanodeStoreSchemaTwoImpl(conf, dbFile.getAbsolutePath(), false);
        } else {
            if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "3")) {
                return;
            }
            throw new IllegalArgumentException("Unrecognized schema version for container: " + schemaVersion);
        }
        BlockUtils.addDB(store, dbFile.getAbsolutePath(), conf, schemaVersion);
    }

    public static void removeContainer(KeyValueContainerData containerData, ConfigurationSource conf) throws IOException {
        Preconditions.checkNotNull((Object)containerData);
        KeyValueContainerUtil.removeContainerDB(containerData, conf);
        KeyValueContainerUtil.moveToDeletedContainerDir(containerData, containerData.getVolume());
    }

    public static void removeContainerDB(KeyValueContainerData containerData, ConfigurationSource conf) throws IOException {
        if (containerData.hasSchema("3")) {
            try {
                BlockUtils.removeContainerFromDB(containerData, conf);
            }
            catch (IOException ex) {
                LOG.error("DB failure, unable to remove container. Disk need to be replaced.", (Throwable)ex);
                throw ex;
            }
        } else {
            BlockUtils.removeDB(containerData, conf);
        }
    }

    public static boolean noBlocksInContainer(DatanodeStore store, KeyValueContainerData containerData, boolean bCheckChunksFilePath) throws IOException {
        Preconditions.checkNotNull((Object)store);
        Preconditions.checkNotNull((Object)containerData);
        if (containerData.isOpen()) {
            return false;
        }
        try (BlockIterator<BlockData> blockIterator = store.getBlockIterator(containerData.getContainerID());){
            if (blockIterator.hasNext()) {
                boolean bl = false;
                return bl;
            }
        }
        if (bCheckChunksFilePath) {
            File chunksPath = new File(containerData.getChunksPath());
            Preconditions.checkArgument((boolean)chunksPath.isDirectory());
            try (DirectoryStream<Path> dir = Files.newDirectoryStream(chunksPath.toPath());){
                boolean bl = !dir.iterator().hasNext();
                return bl;
            }
        }
        return true;
    }

    public static void parseKVContainerData(KeyValueContainerData kvContainerData, ConfigurationSource config) throws IOException {
        KeyValueContainerUtil.parseKVContainerData(kvContainerData, config, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void parseKVContainerData(KeyValueContainerData kvContainerData, ConfigurationSource config, boolean skipVerifyChecksum) throws IOException {
        File dbFile;
        long containerID = kvContainerData.getContainerID();
        if (!skipVerifyChecksum) {
            ContainerUtils.verifyContainerFileChecksum(kvContainerData, config);
        }
        if (kvContainerData.getSchemaVersion() == null) {
            kvContainerData.setSchemaVersion("1");
        }
        if (!(dbFile = KeyValueContainerLocationUtil.getContainerDBFile(kvContainerData)).exists()) {
            LOG.error("Container DB file is missing at {} for ContainerID {}. Skipping loading of this container.", (Object)dbFile, (Object)containerID);
            throw new IOException("Container DB file is missing for containerID " + containerID);
        }
        kvContainerData.setDbFile(dbFile);
        DatanodeConfiguration dnConf = (DatanodeConfiguration)((Object)config.getObject(DatanodeConfiguration.class));
        boolean bCheckChunksFilePath = dnConf.getCheckEmptyContainerDir();
        if (kvContainerData.hasSchema("3")) {
            try (DBHandle db = BlockUtils.getDB(kvContainerData, config);){
                KeyValueContainerUtil.populateContainerMetadata(kvContainerData, db.getStore(), bCheckChunksFilePath);
            }
            return;
        }
        DBHandle cachedDB = null;
        DatanodeStore store = null;
        try {
            try {
                store = BlockUtils.getUncachedDatanodeStore(kvContainerData, config, false);
            }
            catch (IOException e) {
                cachedDB = BlockUtils.getDB(kvContainerData, config);
                store = cachedDB.getStore();
                LOG.warn("Attempt to get an uncached RocksDB handle failed and an instance was retrieved from the cache. This should only happen in tests");
            }
            KeyValueContainerUtil.populateContainerMetadata(kvContainerData, store, bCheckChunksFilePath);
        }
        finally {
            if (cachedDB != null) {
                cachedDB.close();
            } else if (store != null) {
                store.stop();
            }
        }
    }

    private static void loadAndSetContainerDataChecksum(KeyValueContainerData kvContainerData, Table<String, Long> metadataTable) {
        if (kvContainerData.isOpen()) {
            return;
        }
        try {
            Long containerDataChecksum = (Long)metadataTable.get((Object)kvContainerData.getContainerDataChecksumKey());
            if (containerDataChecksum != null && kvContainerData.needsDataChecksum()) {
                kvContainerData.setDataChecksum(containerDataChecksum);
                return;
            }
            ContainerProtos.ContainerChecksumInfo containerChecksumInfo = ContainerChecksumTreeManager.readChecksumInfo(kvContainerData);
            if (ContainerChecksumTreeManager.hasDataChecksum(containerChecksumInfo) && kvContainerData.needsDataChecksum()) {
                containerDataChecksum = containerChecksumInfo.getContainerMerkleTree().getDataChecksum();
                kvContainerData.setDataChecksum(containerDataChecksum);
                metadataTable.put((Object)kvContainerData.getContainerDataChecksumKey(), (Object)containerDataChecksum);
            }
        }
        catch (IOException ex) {
            LOG.warn("Failed to read checksum info for container {}", (Object)kvContainerData.getContainerID(), (Object)ex);
        }
    }

    private static void populateContainerMetadata(KeyValueContainerData kvContainerData, DatanodeStore store, boolean bCheckChunksFilePath) throws IOException {
        long blockCount;
        long blockBytes;
        Long bcsId;
        long blockPendingDeletion;
        Table<String, Long> metadataTable = store.getMetadataTable();
        Long pendingDeleteBlockCount = (Long)metadataTable.get((Object)kvContainerData.getPendingDeleteBlockCountKey());
        if (pendingDeleteBlockCount != null) {
            blockPendingDeletion = pendingDeleteBlockCount;
        } else {
            LOG.warn("Missing pendingDeleteBlockCount from {}: recalculate them from block table", (Object)metadataTable.getName());
            MetadataKeyFilters.KeyPrefixFilter filter = kvContainerData.getDeletingBlockKeyFilter();
            blockPendingDeletion = store.getBlockDataTable().getRangeKVs((Object)kvContainerData.startKeyEmpty(), Integer.MAX_VALUE, (Object)kvContainerData.containerPrefix(), filter, true).size();
        }
        Long delTxnId = (Long)metadataTable.get((Object)kvContainerData.getLatestDeleteTxnKey());
        if (delTxnId != null) {
            kvContainerData.updateDeleteTransactionId(delTxnId);
        }
        if ((bcsId = (Long)metadataTable.get((Object)kvContainerData.getBcsIdKey())) != null) {
            kvContainerData.updateBlockCommitSequenceId(bcsId);
        }
        Long metadataTableBytesUsed = (Long)metadataTable.get((Object)kvContainerData.getBytesUsedKey());
        Long metadataTableBlockCount = (Long)metadataTable.get((Object)kvContainerData.getBlockCountKey());
        if (metadataTableBytesUsed != null && metadataTableBlockCount != null) {
            blockBytes = metadataTableBytesUsed;
            blockCount = metadataTableBlockCount;
        } else {
            LOG.warn("Missing bytesUsed={} or blockCount={} from {}: recalculate them from block table", new Object[]{metadataTableBytesUsed, metadataTableBlockCount, metadataTable.getName()});
            ContainerData.BlockByteAndCounts b = KeyValueContainerUtil.getUsedBytesAndBlockCount(store, kvContainerData);
            blockBytes = b.getBytes();
            blockCount = b.getCount();
        }
        kvContainerData.getStatistics().updateBlocks(blockBytes, blockCount, blockPendingDeletion);
        File chunksDir = new File(kvContainerData.getChunksPath());
        if (!chunksDir.exists()) {
            Files.createDirectories(chunksDir.toPath(), new FileAttribute[0]);
        }
        if (KeyValueContainerUtil.noBlocksInContainer(store, kvContainerData, bCheckChunksFilePath)) {
            kvContainerData.markAsEmpty();
        }
        KeyValueContainerUtil.loadAndSetContainerDataChecksum(kvContainerData, metadataTable);
        ContainerInspectorUtil.process(kvContainerData, store);
        KeyValueContainerUtil.populateContainerFinalizeBlock(kvContainerData, store);
    }

    private static void populateContainerFinalizeBlock(KeyValueContainerData kvContainerData, DatanodeStore store) throws IOException {
        if (store.getFinalizeBlocksTable() != null) {
            try (BlockIterator<Long> iter = store.getFinalizeBlockIterator(kvContainerData.getContainerID(), kvContainerData.getUnprefixedKeyFilter());){
                while (iter.hasNext()) {
                    kvContainerData.addToFinalizedBlockSet(iter.nextBlock());
                }
            }
        }
    }

    private static ContainerData.BlockByteAndCounts getUsedBytesAndBlockCount(DatanodeStore store, KeyValueContainerData kvData) throws IOException {
        long blockCount = 0L;
        long usedBytes = 0L;
        try (BlockIterator<BlockData> blockIter = store.getBlockIterator(kvData.getContainerID(), kvData.getUnprefixedKeyFilter());){
            while (blockIter.hasNext()) {
                ++blockCount;
                usedBytes += KeyValueContainerUtil.getBlockLengthTryCatch(blockIter.nextBlock());
            }
        }
        blockIter = store.getBlockIterator(kvData.getContainerID(), kvData.getDeletingBlockKeyFilter());
        try {
            while (blockIter.hasNext()) {
                ++blockCount;
                usedBytes += KeyValueContainerUtil.getBlockLengthTryCatch(blockIter.nextBlock());
            }
        }
        finally {
            if (blockIter != null) {
                blockIter.close();
            }
        }
        return new ContainerData.BlockByteAndCounts(usedBytes, blockCount, 0L);
    }

    public static long getBlockLengthTryCatch(BlockData block) {
        try {
            return block.getChunks().stream().mapToLong(ContainerProtos.ChunkInfo::getLen).sum();
        }
        catch (Exception e) {
            LOG.error("Failed to getBlockLength for block {}", (Object)block.getBlockID(), (Object)e);
            return 0L;
        }
    }

    public static long getBlockLength(BlockData block) {
        return block.getChunks().stream().mapToLong(ContainerProtos.ChunkInfo::getLen).sum();
    }

    public static boolean isSameSchemaVersion(String schema, String other) {
        String effective1 = schema != null ? schema : "1";
        String effective2 = other != null ? other : "1";
        return effective1.equals(effective2);
    }

    public static void moveToDeletedContainerDir(KeyValueContainerData keyValueContainerData, HddsVolume hddsVolume) throws IOException {
        String containerPath = keyValueContainerData.getContainerPath();
        File container = new File(containerPath);
        Path destinationDirPath = KeyValueContainerUtil.getTmpDirectoryPath(keyValueContainerData, hddsVolume);
        File destinationDirFile = destinationDirPath.toFile();
        if (destinationDirFile.exists()) {
            FileUtils.deleteDirectory((File)destinationDirFile);
        }
        Files.move(container.toPath(), destinationDirPath, new CopyOption[0]);
        LOG.debug("Container {} has been successfully moved under {}", (Object)container.getName(), (Object)hddsVolume.getDeletedContainerDir());
    }

    public static Path getTmpDirectoryPath(KeyValueContainerData keyValueContainerData, HddsVolume hddsVolume) {
        String containerPath = keyValueContainerData.getContainerPath();
        File container = new File(containerPath);
        String containerDirName = container.getName();
        Path destinationDirPath = hddsVolume.getDeletedContainerDir().toPath().resolve(Paths.get(containerDirName, new String[0]));
        return destinationDirPath;
    }
}

