/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.kafka.common.DirectoryId;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.errors.BrokerIdNotRegisteredException;
import org.apache.kafka.common.errors.DuplicateBrokerRegistrationException;
import org.apache.kafka.common.errors.InconsistentClusterIdException;
import org.apache.kafka.common.errors.InvalidRegistrationException;
import org.apache.kafka.common.errors.StaleBrokerEpochException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.message.BrokerRegistrationRequestData;
import org.apache.kafka.common.message.ControllerRegistrationRequestData;
import org.apache.kafka.common.metadata.BrokerRegistrationChangeRecord;
import org.apache.kafka.common.metadata.FenceBrokerRecord;
import org.apache.kafka.common.metadata.RegisterBrokerRecord;
import org.apache.kafka.common.metadata.RegisterControllerRecord;
import org.apache.kafka.common.metadata.UnfenceBrokerRecord;
import org.apache.kafka.common.metadata.UnregisterBrokerRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.BrokerHeartbeatManager;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.controller.FeatureControlManager;
import org.apache.kafka.metadata.BrokerRegistration;
import org.apache.kafka.metadata.BrokerRegistrationFencingChange;
import org.apache.kafka.metadata.BrokerRegistrationInControlledShutdownChange;
import org.apache.kafka.metadata.BrokerRegistrationReply;
import org.apache.kafka.metadata.ControllerRegistration;
import org.apache.kafka.metadata.FinalizedControllerFeatures;
import org.apache.kafka.metadata.ListenerInfo;
import org.apache.kafka.metadata.VersionRange;
import org.apache.kafka.metadata.placement.ReplicaPlacer;
import org.apache.kafka.metadata.placement.StripedReplicaPlacer;
import org.apache.kafka.metadata.placement.UsableBroker;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.common.MetadataVersion;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.apache.kafka.timeline.TimelineHashMap;
import org.slf4j.Logger;

public class ClusterControlManager {
    static final long DEFAULT_SESSION_TIMEOUT_NS = TimeUnit.NANOSECONDS.convert(9L, TimeUnit.SECONDS);
    private final LogContext logContext;
    private final String clusterId;
    private final Logger log;
    private final Time time;
    private final long sessionTimeoutNs;
    private final ReplicaPlacer replicaPlacer;
    private final TimelineHashMap<Integer, BrokerRegistration> brokerRegistrations;
    private final TimelineHashMap<Integer, Long> registerBrokerRecordOffsets;
    private BrokerHeartbeatManager heartbeatManager;
    private Optional<ReadyBrokersFuture> readyBrokersFuture;
    private final FeatureControlManager featureControl;
    private final boolean zkMigrationEnabled;
    private final TimelineHashMap<Integer, ControllerRegistration> controllerRegistrations;
    private final TimelineHashMap<Uuid, Integer> directoryToBroker;

    private ClusterControlManager(LogContext logContext, String clusterId, Time time, SnapshotRegistry snapshotRegistry, long sessionTimeoutNs, ReplicaPlacer replicaPlacer, FeatureControlManager featureControl, boolean zkMigrationEnabled) {
        this.logContext = logContext;
        this.clusterId = clusterId;
        this.log = logContext.logger(ClusterControlManager.class);
        this.time = time;
        this.sessionTimeoutNs = sessionTimeoutNs;
        this.replicaPlacer = replicaPlacer;
        this.brokerRegistrations = new TimelineHashMap(snapshotRegistry, 0);
        this.registerBrokerRecordOffsets = new TimelineHashMap(snapshotRegistry, 0);
        this.heartbeatManager = null;
        this.readyBrokersFuture = Optional.empty();
        this.featureControl = featureControl;
        this.zkMigrationEnabled = zkMigrationEnabled;
        this.controllerRegistrations = new TimelineHashMap(snapshotRegistry, 0);
        this.directoryToBroker = new TimelineHashMap(snapshotRegistry, 0);
    }

    ReplicaPlacer replicaPlacer() {
        return this.replicaPlacer;
    }

    public void activate() {
        this.heartbeatManager = new BrokerHeartbeatManager(this.logContext, this.time, this.sessionTimeoutNs);
        for (BrokerRegistration registration : this.brokerRegistrations.values()) {
            this.heartbeatManager.register(registration.id(), registration.fenced());
        }
    }

    public void deactivate() {
        this.heartbeatManager = null;
    }

    Map<Integer, BrokerRegistration> brokerRegistrations() {
        return this.brokerRegistrations;
    }

    Set<Integer> fencedBrokerIds() {
        return this.brokerRegistrations.values().stream().filter(BrokerRegistration::fenced).map(BrokerRegistration::id).collect(Collectors.toSet());
    }

    boolean zkRegistrationAllowed() {
        return this.zkMigrationEnabled && this.featureControl.metadataVersion().isMigrationSupported();
    }

    public ControllerResult<BrokerRegistrationReply> registerBroker(BrokerRegistrationRequestData request, long brokerEpoch, FinalizedControllerFeatures finalizedFeatures, short version) {
        if (this.heartbeatManager == null) {
            throw new RuntimeException("ClusterControlManager is not active.");
        }
        if (!this.clusterId.equals(request.clusterId())) {
            throw new InconsistentClusterIdException("Expected cluster ID " + this.clusterId + ", but got cluster ID " + request.clusterId());
        }
        int brokerId = request.brokerId();
        BrokerRegistration existing = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (version < 2 || existing == null || request.previousBrokerEpoch() != existing.epoch()) {
            this.log.debug("Received an unclean shutdown request");
        }
        if (existing != null) {
            if (this.heartbeatManager.hasValidSession(brokerId)) {
                if (!existing.incarnationId().equals((Object)request.incarnationId())) {
                    throw new DuplicateBrokerRegistrationException("Another broker is registered with that broker id.");
                }
            } else if (!existing.incarnationId().equals((Object)request.incarnationId())) {
                this.heartbeatManager.remove(brokerId);
            }
        }
        if (request.isMigratingZkBroker() && !this.zkRegistrationAllowed()) {
            throw new BrokerIdNotRegisteredException("Controller does not support registering ZK brokers.");
        }
        if (!request.isMigratingZkBroker() && this.featureControl.inPreMigrationMode()) {
            throw new BrokerIdNotRegisteredException("Controller is in pre-migration mode and cannot register KRaft brokers until the metadata migration is complete.");
        }
        if (this.featureControl.metadataVersion().isDirectoryAssignmentSupported()) {
            if (request.logDirs().isEmpty()) {
                throw new InvalidRegistrationException("No directories specified in request");
            }
            if (request.logDirs().stream().anyMatch(DirectoryId::reserved)) {
                throw new InvalidRegistrationException("Reserved directory ID in request");
            }
            HashSet set = new HashSet(request.logDirs());
            if (set.size() != request.logDirs().size()) {
                throw new InvalidRegistrationException("Duplicate directory ID in request");
            }
            for (Iterator directory : request.logDirs()) {
                Integer dirBrokerId = (Integer)this.directoryToBroker.get((Object)directory);
                if (dirBrokerId == null || dirBrokerId == brokerId) continue;
                throw new InvalidRegistrationException("Broker " + dirBrokerId + " is already registered with directory " + directory);
            }
        }
        ListenerInfo listenerInfo = ListenerInfo.fromBrokerRegistrationRequest(request.listeners());
        RegisterBrokerRecord record = new RegisterBrokerRecord().setBrokerId(brokerId).setIsMigratingZkBroker(request.isMigratingZkBroker()).setIncarnationId(request.incarnationId()).setBrokerEpoch(brokerEpoch).setRack(request.rack()).setEndPoints(listenerInfo.toBrokerRegistrationRecord());
        for (BrokerRegistrationRequestData.Feature feature : request.features()) {
            record.features().add(this.processRegistrationFeature(brokerId, finalizedFeatures, feature));
        }
        if (request.features().find("metadata.version") == null) {
            record.features().add(this.processRegistrationFeature(brokerId, finalizedFeatures, new BrokerRegistrationRequestData.Feature().setName("metadata.version").setMinSupportedVersion(MetadataVersion.MINIMUM_KRAFT_VERSION.featureLevel()).setMaxSupportedVersion(MetadataVersion.MINIMUM_KRAFT_VERSION.featureLevel())));
        }
        if (this.featureControl.metadataVersion().isDirectoryAssignmentSupported()) {
            record.setLogDirs(request.logDirs());
        }
        this.heartbeatManager.register(brokerId, record.fenced());
        ArrayList<ApiMessageAndVersion> records = new ArrayList<ApiMessageAndVersion>();
        records.add(new ApiMessageAndVersion((ApiMessage)record, this.featureControl.metadataVersion().registerBrokerRecordVersion()));
        return ControllerResult.atomicOf(records, new BrokerRegistrationReply(brokerEpoch));
    }

    ControllerResult<Void> registerController(ControllerRegistrationRequestData request) {
        if (!this.featureControl.metadataVersion().isControllerRegistrationSupported()) {
            throw new UnsupportedVersionException("The current MetadataVersion is too old to support controller registrations.");
        }
        ListenerInfo listenerInfo = ListenerInfo.fromControllerRegistrationRequest(request.listeners());
        RegisterControllerRecord.ControllerFeatureCollection features = new RegisterControllerRecord.ControllerFeatureCollection();
        request.features().forEach(feature -> features.add(new RegisterControllerRecord.ControllerFeature().setName(feature.name()).setMaxSupportedVersion(feature.maxSupportedVersion()).setMinSupportedVersion(feature.minSupportedVersion())));
        ArrayList<ApiMessageAndVersion> records = new ArrayList<ApiMessageAndVersion>();
        records.add(new ApiMessageAndVersion((ApiMessage)new RegisterControllerRecord().setControllerId(request.controllerId()).setIncarnationId(request.incarnationId()).setZkMigrationReady(request.zkMigrationReady()).setEndPoints(listenerInfo.toControllerRegistrationRecord()).setFeatures(features), 0));
        return ControllerResult.atomicOf(records, null);
    }

    RegisterBrokerRecord.BrokerFeature processRegistrationFeature(int brokerId, FinalizedControllerFeatures finalizedFeatures, BrokerRegistrationRequestData.Feature feature) {
        Optional<Short> finalized = finalizedFeatures.get(feature.name());
        if (finalized.isPresent()) {
            if (!VersionRange.of(feature.minSupportedVersion(), feature.maxSupportedVersion()).contains(finalized.get())) {
                throw new UnsupportedVersionException("Unable to register because the broker does not support version " + finalized.get() + " of " + feature.name() + ". It wants a version between " + feature.minSupportedVersion() + " and " + feature.maxSupportedVersion() + ", inclusive.");
            }
        } else {
            this.log.warn("Broker {} registered with feature {} that is unknown to the controller", (Object)brokerId, (Object)feature.name());
        }
        return new RegisterBrokerRecord.BrokerFeature().setName(feature.name()).setMinSupportedVersion(feature.minSupportedVersion()).setMaxSupportedVersion(feature.maxSupportedVersion());
    }

    public OptionalLong registerBrokerRecordOffset(int brokerId) {
        Long registrationOffset = (Long)this.registerBrokerRecordOffsets.get((Object)brokerId);
        if (registrationOffset != null) {
            return OptionalLong.of(registrationOffset);
        }
        return OptionalLong.empty();
    }

    public void replay(RegisterBrokerRecord record, long offset) {
        this.registerBrokerRecordOffsets.put((Object)record.brokerId(), (Object)offset);
        int brokerId = record.brokerId();
        ListenerInfo listenerInfo = ListenerInfo.fromBrokerRegistrationRecord(record.endPoints());
        HashMap<String, VersionRange> features = new HashMap<String, VersionRange>();
        Iterator iterator = record.features().iterator();
        while (iterator.hasNext()) {
            RegisterBrokerRecord.BrokerFeature feature = (RegisterBrokerRecord.BrokerFeature)iterator.next();
            features.put(feature.name(), VersionRange.of(feature.minSupportedVersion(), feature.maxSupportedVersion()));
        }
        BrokerRegistration prevRegistration = (BrokerRegistration)this.brokerRegistrations.put((Object)brokerId, (Object)new BrokerRegistration.Builder().setId(brokerId).setEpoch(record.brokerEpoch()).setIncarnationId(record.incarnationId()).setListeners(listenerInfo.listeners()).setSupportedFeatures(features).setRack(Optional.ofNullable(record.rack())).setFenced(record.fenced()).setInControlledShutdown(record.inControlledShutdown()).setIsMigratingZkBroker(record.isMigratingZkBroker()).setDirectories(record.logDirs()).build());
        this.updateDirectories(brokerId, prevRegistration == null ? null : prevRegistration.directories(), record.logDirs());
        if (this.heartbeatManager != null) {
            if (prevRegistration != null) {
                this.heartbeatManager.remove(brokerId);
            }
            this.heartbeatManager.register(brokerId, record.fenced());
        }
        if (prevRegistration == null) {
            this.log.info("Replayed initial RegisterBrokerRecord for broker {}: {}", (Object)record.brokerId(), (Object)record);
        } else if (prevRegistration.incarnationId().equals((Object)record.incarnationId())) {
            this.log.info("Replayed RegisterBrokerRecord modifying the registration for broker {}: {}", (Object)record.brokerId(), (Object)record);
        } else {
            this.log.info("Replayed RegisterBrokerRecord establishing a new incarnation of broker {}: {}", (Object)record.brokerId(), (Object)record);
        }
    }

    public void replay(UnregisterBrokerRecord record) {
        this.registerBrokerRecordOffsets.remove((Object)record.brokerId());
        int brokerId = record.brokerId();
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            throw new RuntimeException(String.format("Unable to replay %s: no broker registration found for that id", record));
        }
        if (registration.epoch() != record.brokerEpoch()) {
            throw new RuntimeException(String.format("Unable to replay %s: no broker registration with that epoch found", record));
        }
        if (this.heartbeatManager != null) {
            this.heartbeatManager.remove(brokerId);
        }
        this.updateDirectories(brokerId, registration.directories(), null);
        this.brokerRegistrations.remove((Object)brokerId);
        this.log.info("Replayed {}", (Object)record);
    }

    public void replay(FenceBrokerRecord record) {
        this.replayRegistrationChange(record, record.id(), record.epoch(), BrokerRegistrationFencingChange.FENCE.asBoolean(), BrokerRegistrationInControlledShutdownChange.NONE.asBoolean(), Optional.empty());
    }

    public void replay(UnfenceBrokerRecord record) {
        this.replayRegistrationChange(record, record.id(), record.epoch(), BrokerRegistrationFencingChange.UNFENCE.asBoolean(), BrokerRegistrationInControlledShutdownChange.NONE.asBoolean(), Optional.empty());
    }

    public void replay(BrokerRegistrationChangeRecord record) {
        BrokerRegistrationFencingChange fencingChange = BrokerRegistrationFencingChange.fromValue(record.fenced()).orElseThrow(() -> new IllegalStateException(String.format("Unable to replay %s: unknown value for fenced field: %x", record, record.fenced())));
        BrokerRegistrationInControlledShutdownChange inControlledShutdownChange = BrokerRegistrationInControlledShutdownChange.fromValue(record.inControlledShutdown()).orElseThrow(() -> new IllegalStateException(String.format("Unable to replay %s: unknown value for inControlledShutdown field: %x", record, record.inControlledShutdown())));
        Optional<List<Uuid>> directoriesChange = Optional.ofNullable(record.logDirs()).filter(list -> !list.isEmpty());
        this.replayRegistrationChange(record, record.brokerId(), record.brokerEpoch(), fencingChange.asBoolean(), inControlledShutdownChange.asBoolean(), directoriesChange);
    }

    private void replayRegistrationChange(ApiMessage record, int brokerId, long brokerEpoch, Optional<Boolean> fencingChange, Optional<Boolean> inControlledShutdownChange, Optional<List<Uuid>> directoriesChange) {
        BrokerRegistration curRegistration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (curRegistration == null) {
            throw new RuntimeException(String.format("Unable to replay %s: no broker registration found for that id", record.toString()));
        }
        if (curRegistration.epoch() != brokerEpoch) {
            throw new RuntimeException(String.format("Unable to replay %s: no broker registration with that epoch found", record.toString()));
        }
        BrokerRegistration nextRegistration = curRegistration.cloneWith(fencingChange, inControlledShutdownChange, directoriesChange);
        if (!curRegistration.equals(nextRegistration)) {
            this.log.info("Replayed {} modifying the registration for broker {}: {}", new Object[]{record.getClass().getSimpleName(), brokerId, record});
            this.brokerRegistrations.put((Object)brokerId, (Object)nextRegistration);
        } else {
            this.log.info("Ignoring no-op registration change for {}", (Object)curRegistration);
        }
        this.updateDirectories(brokerId, curRegistration.directories(), nextRegistration.directories());
        if (this.heartbeatManager != null) {
            this.heartbeatManager.register(brokerId, nextRegistration.fenced());
        }
        if (this.readyBrokersFuture.isPresent() && this.readyBrokersFuture.get().check()) {
            this.readyBrokersFuture.get().future.complete(null);
            this.readyBrokersFuture = Optional.empty();
        }
    }

    public void replay(RegisterControllerRecord record) {
        ControllerRegistration newRegistration = new ControllerRegistration.Builder(record).build();
        ControllerRegistration prevRegistration = (ControllerRegistration)this.controllerRegistrations.put((Object)record.controllerId(), (Object)newRegistration);
        this.log.info("Replayed RegisterControllerRecord contaning {}.{}", (Object)newRegistration, (Object)(prevRegistration == null ? "" : " Previous incarnation was " + prevRegistration.incarnationId()));
    }

    Iterator<UsableBroker> usableBrokers() {
        if (this.heartbeatManager == null) {
            throw new RuntimeException("ClusterControlManager is not active.");
        }
        return this.heartbeatManager.usableBrokers(id -> ((BrokerRegistration)this.brokerRegistrations.get(id)).rack());
    }

    public boolean isUnfenced(int brokerId) {
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            return false;
        }
        return !registration.fenced();
    }

    public BrokerRegistration registration(int brokerId) {
        return (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
    }

    public boolean inControlledShutdown(int brokerId) {
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            return false;
        }
        return registration.inControlledShutdown();
    }

    public boolean isActive(int brokerId) {
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            return false;
        }
        return !registration.inControlledShutdown() && !registration.fenced();
    }

    BrokerHeartbeatManager heartbeatManager() {
        if (this.heartbeatManager == null) {
            throw new RuntimeException("ClusterControlManager is not active.");
        }
        return this.heartbeatManager;
    }

    public void checkBrokerEpoch(int brokerId, long brokerEpoch) {
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            throw new StaleBrokerEpochException("No broker registration found for broker id " + brokerId);
        }
        if (registration.epoch() != brokerEpoch) {
            throw new StaleBrokerEpochException("Expected broker epoch " + registration.epoch() + ", but got broker epoch " + brokerEpoch);
        }
    }

    public void addReadyBrokersFuture(CompletableFuture<Void> future, int minBrokers) {
        this.readyBrokersFuture = Optional.of(new ReadyBrokersFuture(future, minBrokers));
        if (this.readyBrokersFuture.get().check()) {
            this.readyBrokersFuture.get().future.complete(null);
            this.readyBrokersFuture = Optional.empty();
        }
    }

    public boolean hasOnlineDir(int brokerId, Uuid directoryId) {
        BrokerRegistration registration = this.registration(brokerId);
        return registration != null && registration.hasOnlineDir(directoryId);
    }

    public Uuid defaultDir(int brokerId) {
        BrokerRegistration registration = this.registration(brokerId);
        if (registration == null) {
            return DirectoryId.UNASSIGNED;
        }
        List<Uuid> directories = registration.directories();
        if (directories.isEmpty()) {
            return DirectoryId.MIGRATING;
        }
        if (directories.size() == 1) {
            return directories.get(0);
        }
        return DirectoryId.UNASSIGNED;
    }

    void updateDirectories(int brokerId, List<Uuid> dirsToRemove, List<Uuid> dirsToAdd) {
        if (dirsToRemove != null) {
            for (Uuid directory : dirsToRemove) {
                if (this.directoryToBroker.remove((Object)directory, (Object)brokerId)) continue;
                throw new IllegalStateException("BUG: directory " + directory + " not assigned to broker " + brokerId);
            }
        }
        if (dirsToAdd != null) {
            for (Uuid directory : dirsToAdd) {
                Integer existingId = (Integer)this.directoryToBroker.putIfAbsent((Object)directory, (Object)brokerId);
                if (existingId == null || existingId == brokerId) continue;
                throw new IllegalStateException("BUG: directory " + directory + " already assigned to broker " + existingId);
            }
        }
    }

    Iterator<Map.Entry<Integer, Map<String, VersionRange>>> brokerSupportedFeatures() {
        return new Iterator<Map.Entry<Integer, Map<String, VersionRange>>>(){
            private final Iterator<BrokerRegistration> iter;
            {
                this.iter = ClusterControlManager.this.brokerRegistrations.values().iterator();
            }

            @Override
            public boolean hasNext() {
                return this.iter.hasNext();
            }

            @Override
            public Map.Entry<Integer, Map<String, VersionRange>> next() {
                BrokerRegistration registration = this.iter.next();
                return new AbstractMap.SimpleImmutableEntry<Integer, Map<String, VersionRange>>(registration.id(), registration.supportedFeatures());
            }
        };
    }

    Iterator<Map.Entry<Integer, Map<String, VersionRange>>> controllerSupportedFeatures() {
        if (!this.featureControl.metadataVersion().isControllerRegistrationSupported()) {
            throw new UnsupportedVersionException("The current MetadataVersion is too old to support controller registrations.");
        }
        return new Iterator<Map.Entry<Integer, Map<String, VersionRange>>>(){
            private final Iterator<ControllerRegistration> iter;
            {
                this.iter = ClusterControlManager.this.controllerRegistrations.values().iterator();
            }

            @Override
            public boolean hasNext() {
                return this.iter.hasNext();
            }

            @Override
            public Map.Entry<Integer, Map<String, VersionRange>> next() {
                ControllerRegistration registration = this.iter.next();
                return new AbstractMap.SimpleImmutableEntry<Integer, Map<String, VersionRange>>(registration.id(), registration.supportedFeatures());
            }
        };
    }

    class ReadyBrokersFuture {
        private final CompletableFuture<Void> future;
        private final int minBrokers;

        ReadyBrokersFuture(CompletableFuture<Void> future, int minBrokers) {
            this.future = future;
            this.minBrokers = minBrokers;
        }

        boolean check() {
            int numUnfenced = 0;
            for (BrokerRegistration registration : ClusterControlManager.this.brokerRegistrations.values()) {
                if (!registration.fenced()) {
                    ++numUnfenced;
                }
                if (numUnfenced < this.minBrokers) continue;
                return true;
            }
            return false;
        }
    }

    static class Builder {
        private LogContext logContext = null;
        private String clusterId = null;
        private Time time = Time.SYSTEM;
        private SnapshotRegistry snapshotRegistry = null;
        private long sessionTimeoutNs = DEFAULT_SESSION_TIMEOUT_NS;
        private ReplicaPlacer replicaPlacer = null;
        private FeatureControlManager featureControl = null;
        private boolean zkMigrationEnabled = false;

        Builder() {
        }

        Builder setLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        Builder setClusterId(String clusterId) {
            this.clusterId = clusterId;
            return this;
        }

        Builder setTime(Time time) {
            this.time = time;
            return this;
        }

        Builder setSnapshotRegistry(SnapshotRegistry snapshotRegistry) {
            this.snapshotRegistry = snapshotRegistry;
            return this;
        }

        Builder setSessionTimeoutNs(long sessionTimeoutNs) {
            this.sessionTimeoutNs = sessionTimeoutNs;
            return this;
        }

        Builder setReplicaPlacer(ReplicaPlacer replicaPlacer) {
            this.replicaPlacer = replicaPlacer;
            return this;
        }

        Builder setFeatureControlManager(FeatureControlManager featureControl) {
            this.featureControl = featureControl;
            return this;
        }

        Builder setZkMigrationEnabled(boolean zkMigrationEnabled) {
            this.zkMigrationEnabled = zkMigrationEnabled;
            return this;
        }

        ClusterControlManager build() {
            if (this.logContext == null) {
                this.logContext = new LogContext();
            }
            if (this.clusterId == null) {
                this.clusterId = Uuid.randomUuid().toString();
            }
            if (this.snapshotRegistry == null) {
                this.snapshotRegistry = new SnapshotRegistry(this.logContext);
            }
            if (this.replicaPlacer == null) {
                this.replicaPlacer = new StripedReplicaPlacer(new Random());
            }
            if (this.featureControl == null) {
                throw new RuntimeException("You must specify FeatureControlManager");
            }
            return new ClusterControlManager(this.logContext, this.clusterId, this.time, this.snapshotRegistry, this.sessionTimeoutNs, this.replicaPlacer, this.featureControl, this.zkMigrationEnabled);
        }
    }
}

