/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.rest.matcher;

import static org.apache.juneau.commons.utils.StringUtils.*;
import static org.apache.juneau.rest.annotation.RestOpAnnotation.*;

import org.apache.juneau.commons.lang.*;
import org.apache.juneau.commons.reflect.*;
import jakarta.servlet.http.*;

/**
 * Specialized matcher for matching client versions.
 *
 * <h5 class='section'>See Also:</h5><ul>
 * 	<li class='link'><a class="doclink" href="https://juneau.apache.org/docs/topics/ClientVersioning">Client Versioning</a>
 * </ul>
 */
public class ClientVersionMatcher extends RestMatcher {

	private static final AnnotationProvider AP = AnnotationProvider.INSTANCE;

	private final String clientVersionHeader;
	private final VersionRange range;

	/**
	 * Constructor.
	 *
	 * @param clientVersionHeader
	 * 	The HTTP request header name containing the client version.
	 * 	If <jk>null</jk> or an empty string, uses <js>"Client-Version"</js>
	 * @param mi The version string that the client version must match.
	 */
	public ClientVersionMatcher(String clientVersionHeader, MethodInfo mi) {
		this.clientVersionHeader = isEmpty(clientVersionHeader) ? "Client-Version" : clientVersionHeader;
		// @formatter:off
		var clientVersion = AP.find(mi)
			.stream()
			.filter(REST_OP_GROUP)
			.flatMap(ai -> ai.getValue(String.class, "clientVersion").stream())
			.filter(NOT_EMPTY)
			.findFirst()
			.orElse(null);
		// @formatter:on
		range = new VersionRange(clientVersion);
	}

	@Override /* Overridden from RestMatcher */
	public boolean matches(HttpServletRequest req) {
		return range.matches(req.getHeader(clientVersionHeader));
	}

	@Override /* Overridden from RestMatcher */
	public boolean required() {
		return true;
	}
}