/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.s3a.impl.StoreContext;
import org.apache.hadoop.fs.s3a.impl.UploadContentProviders;
import org.apache.hadoop.fs.s3a.statistics.BlockOutputStreamStatistics;
import org.apache.hadoop.fs.store.DataBlocks;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.DirectBufferPool;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.functional.BiFunctionRaisingIOE;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class S3ADataBlocks {
    private static final Logger LOG = LoggerFactory.getLogger(S3ADataBlocks.class);

    private S3ADataBlocks() {
    }

    static void validateWriteArgs(byte[] b, int off, int len) throws IOException {
        DataBlocks.validateWriteArgs((byte[])b, (int)off, (int)len);
    }

    static BlockFactory createFactory(StoreContext owner, String name) {
        switch (name) {
            case "array": {
                return new ArrayBlockFactory(owner);
            }
            case "disk": {
                return new DiskBlockFactory(owner);
            }
            case "bytebuffer": {
                return new ByteBufferBlockFactory(owner);
            }
        }
        throw new IllegalArgumentException("Unsupported block buffer \"" + name + '\"');
    }

    static class DiskBlock
    extends DataBlock {
        private long bytesWritten;
        private final File bufferFile;
        private final long limit;
        private BufferedOutputStream out;
        private final AtomicBoolean closed = new AtomicBoolean(false);

        DiskBlock(File bufferFile, long limit, long index, BlockOutputStreamStatistics statistics) throws FileNotFoundException {
            super(index, statistics);
            this.limit = limit;
            this.bufferFile = Objects.requireNonNull(bufferFile);
            this.blockAllocated();
            this.out = new BufferedOutputStream(new FileOutputStream(bufferFile));
        }

        @Override
        long dataSize() {
            return this.bytesWritten;
        }

        private boolean unlimited() {
            return this.limit < 0L;
        }

        @Override
        boolean hasCapacity(long bytes) {
            return this.unlimited() || this.dataSize() + bytes <= this.limit;
        }

        @Override
        long remainingCapacity() {
            return this.unlimited() ? Integer.MAX_VALUE : this.limit - this.bytesWritten;
        }

        @Override
        int write(byte[] b, int offset, int len) throws IOException {
            super.write(b, offset, len);
            int written = (int)Math.min(this.remainingCapacity(), (long)len);
            this.out.write(b, offset, written);
            this.bytesWritten += (long)written;
            return written;
        }

        @Override
        BlockUploadData startUpload() throws IOException {
            super.startUpload();
            try {
                this.out.flush();
            }
            finally {
                this.out.close();
                this.out = null;
            }
            return new BlockUploadData(this.bufferFile, this::isUploading);
        }

        @Override
        protected void innerClose() throws IOException {
            DataBlock.DestState state = this.getState();
            LOG.debug("Closing {}", (Object)this);
            switch (state) {
                case Writing: {
                    if (!this.bufferFile.exists()) break;
                    LOG.debug("Block[{}]: Deleting buffer file as upload did not start", (Object)this.index);
                    this.closeBlock();
                    break;
                }
                case Upload: {
                    LOG.debug("Block[{}]: Buffer file {} exists \u2014close upload stream", (Object)this.index, (Object)this.bufferFile);
                    break;
                }
                case Closed: {
                    this.closeBlock();
                    break;
                }
            }
        }

        @Override
        void flush() throws IOException {
            super.flush();
            this.out.flush();
        }

        public String toString() {
            String sb = "FileBlock{index=" + this.index + ", destFile=" + this.bufferFile + ", state=" + (Object)((Object)this.getState()) + ", dataSize=" + this.dataSize() + ", limit=" + this.limit + '}';
            return sb;
        }

        void closeBlock() {
            LOG.debug("block[{}]: closeBlock()", (Object)this.index);
            if (!this.closed.getAndSet(true)) {
                this.blockReleased();
                if (!this.bufferFile.delete() && this.bufferFile.exists()) {
                    LOG.warn("delete({}) returned false", (Object)this.bufferFile.getAbsoluteFile());
                }
            } else {
                LOG.debug("block[{}]: skipping re-entrant closeBlock()", (Object)this.index);
            }
        }
    }

    static class DiskBlockFactory
    extends BlockFactory {
        private final BiFunctionRaisingIOE<Long, Long, File> tempFileFn;

        DiskBlockFactory(StoreContext owner) {
            super(owner);
            this.tempFileFn = (index, limit) -> owner.createTempFile(String.format("s3ablock-%04d-", index), (long)limit);
        }

        @VisibleForTesting
        DiskBlockFactory(BiFunctionRaisingIOE<Long, Long, File> tempFileFn) {
            super(null);
            this.tempFileFn = Objects.requireNonNull(tempFileFn);
        }

        @Override
        DataBlock create(long index, long limit, BlockOutputStreamStatistics statistics) throws IOException {
            Preconditions.checkArgument((limit != 0L ? 1 : 0) != 0, (String)"Invalid block size: %d", (Object[])new Object[]{limit});
            File destFile = (File)this.tempFileFn.apply((Object)index, (Object)limit);
            return new DiskBlock(destFile, limit, index, statistics);
        }
    }

    static class ByteBufferBlockFactory
    extends BlockFactory {
        private final DirectBufferPool bufferPool = new DirectBufferPool();
        private final AtomicInteger buffersOutstanding = new AtomicInteger(0);

        ByteBufferBlockFactory(StoreContext owner) {
            super(owner);
        }

        @Override
        ByteBufferBlock create(long index, long limit, BlockOutputStreamStatistics statistics) throws IOException {
            Preconditions.checkArgument((limit > 0L ? 1 : 0) != 0, (String)"Invalid block size: %d", (Object[])new Object[]{limit});
            return new ByteBufferBlock(index, limit, statistics);
        }

        private ByteBuffer requestBuffer(int limit) {
            LOG.debug("Requesting buffer of size {}", (Object)limit);
            this.buffersOutstanding.incrementAndGet();
            return this.bufferPool.getBuffer(limit);
        }

        private void releaseBuffer(ByteBuffer buffer) {
            LOG.debug("Releasing buffer");
            this.bufferPool.returnBuffer(buffer);
            this.buffersOutstanding.decrementAndGet();
        }

        public int getOutstandingBufferCount() {
            return this.buffersOutstanding.get();
        }

        public String toString() {
            return "ByteBufferBlockFactory{buffersOutstanding=" + this.buffersOutstanding + '}';
        }

        class ByteBufferBlock
        extends DataBlock {
            private ByteBuffer blockBuffer;
            private final int bufferSize;
            private Integer dataSize;

            ByteBufferBlock(long index, long bufferSize, BlockOutputStreamStatistics statistics) {
                super(index, statistics);
                this.bufferSize = bufferSize > Integer.MAX_VALUE ? Integer.MAX_VALUE : (int)bufferSize;
                this.blockBuffer = ByteBufferBlockFactory.this.requestBuffer(this.bufferSize);
                this.blockAllocated();
            }

            @Override
            long dataSize() {
                return this.dataSize != null ? (long)this.dataSize.intValue() : (long)this.bufferCapacityUsed();
            }

            @Override
            BlockUploadData startUpload() throws IOException {
                super.startUpload();
                this.dataSize = this.bufferCapacityUsed();
                return new BlockUploadData(UploadContentProviders.byteBufferContentProvider(this.blockBuffer, this.dataSize, this::isUploading));
            }

            @Override
            public boolean hasCapacity(long bytes) {
                return bytes <= this.remainingCapacity();
            }

            @Override
            public long remainingCapacity() {
                return this.blockBuffer != null ? (long)this.blockBuffer.remaining() : 0L;
            }

            private int bufferCapacityUsed() {
                return this.blockBuffer.capacity() - this.blockBuffer.remaining();
            }

            @Override
            int write(byte[] b, int offset, int len) throws IOException {
                super.write(b, offset, len);
                int written = (int)Math.min(this.remainingCapacity(), (long)len);
                this.blockBuffer.put(b, offset, written);
                return written;
            }

            @Override
            protected void innerClose() {
                if (this.blockBuffer != null) {
                    this.blockReleased();
                    ByteBufferBlockFactory.this.releaseBuffer(this.blockBuffer);
                    this.blockBuffer = null;
                }
            }

            public String toString() {
                return "ByteBufferBlock{index=" + this.index + ", state=" + (Object)((Object)this.getState()) + ", dataSize=" + this.dataSize() + ", limit=" + this.bufferSize + ", remainingCapacity=" + this.remainingCapacity() + '}';
            }
        }
    }

    static class ByteArrayBlock
    extends DataBlock {
        private S3AByteArrayOutputStream buffer;
        private final int limit;
        private Integer dataSize;

        ByteArrayBlock(long index, long limit, BlockOutputStreamStatistics statistics) {
            super(index, statistics);
            this.limit = limit > Integer.MAX_VALUE ? Integer.MAX_VALUE : (int)limit;
            this.buffer = new S3AByteArrayOutputStream(this.limit);
            this.blockAllocated();
        }

        @Override
        long dataSize() {
            return this.dataSize != null ? (long)this.dataSize.intValue() : (long)this.buffer.size();
        }

        @Override
        BlockUploadData startUpload() throws IOException {
            super.startUpload();
            this.dataSize = this.buffer.size();
            byte[] bytes = this.buffer.getBuffer();
            this.buffer = null;
            return new BlockUploadData(UploadContentProviders.byteArrayContentProvider(bytes, 0, this.dataSize, this::isUploading));
        }

        @Override
        boolean hasCapacity(long bytes) {
            return this.dataSize() + bytes <= (long)this.limit;
        }

        @Override
        long remainingCapacity() {
            return (long)this.limit - this.dataSize();
        }

        @Override
        int write(byte[] b, int offset, int len) throws IOException {
            super.write(b, offset, len);
            int written = (int)Math.min(this.remainingCapacity(), (long)len);
            this.buffer.write(b, offset, written);
            return written;
        }

        @Override
        protected void innerClose() {
            this.buffer = null;
            this.blockReleased();
        }

        public String toString() {
            return "ByteArrayBlock{index=" + this.index + ", state=" + (Object)((Object)this.getState()) + ", limit=" + this.limit + ", dataSize=" + this.dataSize + '}';
        }
    }

    static class S3AByteArrayOutputStream
    extends ByteArrayOutputStream {
        S3AByteArrayOutputStream(int size) {
            super(size);
        }

        public byte[] getBuffer() {
            return this.buf;
        }
    }

    static class ArrayBlockFactory
    extends BlockFactory {
        ArrayBlockFactory(StoreContext owner) {
            super(owner);
        }

        @Override
        DataBlock create(long index, long limit, BlockOutputStreamStatistics statistics) throws IOException {
            Preconditions.checkArgument((limit > 0L ? 1 : 0) != 0, (String)"Invalid block size: %d", (Object[])new Object[]{limit});
            return new ByteArrayBlock(0L, limit, statistics);
        }
    }

    static abstract class DataBlock
    implements Closeable {
        private volatile DestState state = DestState.Writing;
        protected final long index;
        private final BlockOutputStreamStatistics statistics;

        protected DataBlock(long index, BlockOutputStreamStatistics statistics) {
            this.index = index;
            this.statistics = statistics;
        }

        protected final synchronized void enterState(DestState current, DestState next) throws IllegalStateException {
            this.verifyState(current);
            LOG.debug("{}: entering state {}", (Object)this, (Object)next);
            this.state = next;
        }

        protected final void verifyState(DestState expected) throws IllegalStateException {
            if (expected != null && this.state != expected) {
                throw new IllegalStateException("Expected stream state " + (Object)((Object)expected) + " -but actual state is " + (Object)((Object)this.state) + " in " + this);
            }
        }

        final DestState getState() {
            return this.state;
        }

        final boolean isUploading() {
            return this.state == DestState.Upload;
        }

        abstract long dataSize();

        abstract boolean hasCapacity(long var1);

        boolean hasData() {
            return this.dataSize() > 0L;
        }

        abstract long remainingCapacity();

        int write(byte[] buffer, int offset, int length) throws IOException {
            this.verifyState(DestState.Writing);
            Preconditions.checkArgument((buffer != null ? 1 : 0) != 0, (Object)"Null buffer");
            Preconditions.checkArgument((length >= 0 ? 1 : 0) != 0, (Object)"length is negative");
            Preconditions.checkArgument((offset >= 0 ? 1 : 0) != 0, (Object)"offset is negative");
            Preconditions.checkArgument((buffer.length - offset >= length ? 1 : 0) != 0, (Object)"buffer shorter than amount of data to write");
            return 0;
        }

        void flush() throws IOException {
            this.verifyState(DestState.Writing);
        }

        BlockUploadData startUpload() throws IOException {
            LOG.debug("Start datablock[{}] upload", (Object)this.index);
            this.enterState(DestState.Writing, DestState.Upload);
            return null;
        }

        protected synchronized boolean enterClosedState() {
            if (!this.state.equals((Object)DestState.Closed)) {
                this.enterState(null, DestState.Closed);
                return true;
            }
            return false;
        }

        @Override
        public void close() throws IOException {
            if (this.enterClosedState()) {
                LOG.debug("Closed {}", (Object)this);
                this.innerClose();
            }
        }

        protected void innerClose() throws IOException {
        }

        protected final void blockAllocated() {
            if (this.statistics != null) {
                this.statistics.blockAllocated();
            }
        }

        protected final void blockReleased() {
            if (this.statistics != null) {
                this.statistics.blockReleased();
            }
        }

        protected BlockOutputStreamStatistics getStatistics() {
            return this.statistics;
        }

        static enum DestState {
            Writing,
            Upload,
            Closed;

        }
    }

    public static abstract class BlockFactory
    implements Closeable {
        private final StoreContext owner;

        protected BlockFactory(StoreContext owner) {
            this.owner = owner;
        }

        abstract DataBlock create(long var1, long var3, BlockOutputStreamStatistics var5) throws IOException;

        @Override
        public void close() throws IOException {
        }

        protected StoreContext getOwner() {
            return this.owner;
        }
    }

    public static final class BlockUploadData
    implements Closeable {
        private final UploadContentProviders.BaseContentProvider<?> contentProvider;

        public BlockUploadData(UploadContentProviders.BaseContentProvider<?> contentProvider) {
            this.contentProvider = Objects.requireNonNull(contentProvider);
        }

        public UploadContentProviders.BaseContentProvider<?> getContentProvider() {
            return this.contentProvider;
        }

        public BlockUploadData(File file, Supplier<Boolean> isOpen) {
            Preconditions.checkArgument((boolean)file.exists(), (Object)("No file: " + file));
            long length = file.length();
            Preconditions.checkArgument((length <= Integer.MAX_VALUE ? 1 : 0) != 0, (String)"File %s is too long to upload: %d", (Object[])new Object[]{file, length});
            this.contentProvider = UploadContentProviders.fileContentProvider(file, 0L, (int)length, isOpen);
        }

        public BlockUploadData(byte[] bytes, int offset, int size, Supplier<Boolean> isOpen) {
            this.contentProvider = UploadContentProviders.byteArrayContentProvider(bytes, offset, size, isOpen);
        }

        public BlockUploadData(byte[] bytes, Supplier<Boolean> isOpen) {
            this.contentProvider = UploadContentProviders.byteArrayContentProvider(bytes, isOpen);
        }

        int getSize() {
            return this.contentProvider.getSize();
        }

        @Override
        public void close() throws IOException {
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.contentProvider});
        }
    }
}

