// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

#pragma once

#include <parallel_hashmap/phmap.h>

#include <memory>
#include <string_view>
#include <unordered_map>
#include <utility>

#include "common/exception.h"
#include "common/status.h"
#include "io/io_common.h"
#include "olap/field.h"
#include "olap/iterators.h"
#include "olap/rowset/segment_v2/column_reader.h"
#include "olap/rowset/segment_v2/stream_reader.h"
#include "olap/rowset/segment_v2/variant/variant_column_reader.h"
#include "olap/schema.h"
#include "olap/tablet_schema.h"
#include "vec/columns/column.h"
#include "vec/columns/column_nullable.h"
#include "vec/columns/column_variant.h"
#include "vec/columns/subcolumn_tree.h"
#include "vec/common/assert_cast.h"
#include "vec/core/column_with_type_and_name.h"
#include "vec/core/columns_with_type_and_name.h"
#include "vec/core/types.h"
#include "vec/data_types/data_type.h"
#include "vec/data_types/data_type_array.h"
#include "vec/data_types/data_type_nullable.h"
#include "vec/data_types/data_type_string.h"
#include "vec/data_types/data_type_variant.h"
#include "vec/functions/function_helpers.h"
#include "vec/json/path_in_data.h"

namespace doris::segment_v2 {

#include "common/compile_check_begin.h"

// Base class for sparse column processors with common functionality
class BaseSparseColumnProcessor : public ColumnIterator {
protected:
    const StorageReadOptions* _read_opts;
    SparseColumnCacheSPtr _sparse_column_cache;
    // Pure virtual method for data processing when encounter existing sparse columns(to be implemented by subclasses)
    virtual void _process_data_with_existing_sparse_column(vectorized::MutableColumnPtr& dst,
                                                           size_t num_rows) = 0;

    // Pure virtual method for data processing when no sparse columns(to be implemented by subclasses)
    virtual void _process_data_without_sparse_column(vectorized::MutableColumnPtr& dst,
                                                     size_t num_rows) = 0;

public:
    BaseSparseColumnProcessor(SparseColumnCacheSPtr sparse_column_cache,
                              const StorageReadOptions* opts)
            : _read_opts(opts), _sparse_column_cache(std::move(sparse_column_cache)) {}

    // Common initialization for all processors
    Status init(const ColumnIteratorOptions& opts) override {
        return _sparse_column_cache->init(opts);
    }

    Status seek_to_ordinal(ordinal_t ord) override {
        return _sparse_column_cache->seek_to_ordinal(ord);
    }

    ordinal_t get_current_ordinal() const override {
        throw doris::Exception(ErrorCode::NOT_IMPLEMENTED_ERROR, "not implement");
    }

    // Template method pattern for batch processing
    template <typename ReadMethod>
    Status _process_batch(ReadMethod&& read_method, size_t nrows,
                          vectorized::MutableColumnPtr& dst) {
        {
            SCOPED_RAW_TIMER(&_read_opts->stats->variant_scan_sparse_column_timer_ns);
            int64_t before_size = _read_opts->stats->uncompressed_bytes_read;
            RETURN_IF_ERROR(read_method());
            _read_opts->stats->variant_scan_sparse_column_bytes +=
                    _read_opts->stats->uncompressed_bytes_read - before_size;
        }

        SCOPED_RAW_TIMER(&_read_opts->stats->variant_fill_path_from_sparse_column_timer_ns);
        const auto& offsets =
                assert_cast<const vectorized::ColumnMap&>(*_sparse_column_cache->sparse_column)
                        .get_offsets();
        if (offsets.back() == offsets[-1]) {
            // no sparse column in this batch
            _process_data_without_sparse_column(dst, nrows);
        } else {
            // merge subcolumns to existing sparse columns
            _process_data_with_existing_sparse_column(dst, nrows);
        }
        return Status::OK();
    }
};

// Implementation for path extraction processor
class SparseColumnExtractIterator : public BaseSparseColumnProcessor {
public:
    SparseColumnExtractIterator(std::string_view path, SparseColumnCacheSPtr sparse_column_cache,
                                const StorageReadOptions* opts)
            : BaseSparseColumnProcessor(std::move(sparse_column_cache), opts), _path(path) {}

    // Batch processing using template method
    Status next_batch(size_t* n, vectorized::MutableColumnPtr& dst, bool* has_null) override {
        return _process_batch([&]() { return _sparse_column_cache->next_batch(n, has_null); }, *n,
                              dst);
    }

    // RowID-based read using template method
    Status read_by_rowids(const rowid_t* rowids, const size_t count,
                          vectorized::MutableColumnPtr& dst) override {
        return _process_batch([&]() { return _sparse_column_cache->read_by_rowids(rowids, count); },
                              count, dst);
    }

private:
    std::string _path;

    // Fill column by finding path in sparse column
    void _process_data_with_existing_sparse_column(vectorized::MutableColumnPtr& dst,
                                                   size_t num_rows) override {
        _fill_path_column(dst);
    }

    void _fill_path_column(vectorized::MutableColumnPtr& dst) {
        vectorized::ColumnNullable* nullable_column = nullptr;
        if (dst->is_nullable()) {
            nullable_column = assert_cast<vectorized::ColumnNullable*>(dst.get());
        }
        vectorized::ColumnVariant& var = nullable_column != nullptr
                                                 ? assert_cast<vectorized::ColumnVariant&>(
                                                           nullable_column->get_nested_column())
                                                 : assert_cast<vectorized::ColumnVariant&>(*dst);
        if (var.is_null_root()) {
            var.add_sub_column({}, dst->size());
        }
        vectorized::NullMap* null_map =
                nullable_column ? &nullable_column->get_null_map_data() : nullptr;
        vectorized::ColumnVariant::fill_path_column_from_sparse_data(
                *var.get_subcolumn({}) /*root*/, null_map, StringRef {_path.data(), _path.size()},
                _sparse_column_cache->sparse_column->get_ptr(), 0,
                _sparse_column_cache->sparse_column->size());
        var.incr_num_rows(_sparse_column_cache->sparse_column->size());
        var.get_sparse_column()->assume_mutable()->resize(var.rows());
        ENABLE_CHECK_CONSISTENCY(&var);
    }

    void _process_data_without_sparse_column(vectorized::MutableColumnPtr& dst,
                                             size_t num_rows) override {
        dst->insert_many_defaults(num_rows);
    }
};

#include "common/compile_check_end.h"

} // namespace doris::segment_v2
