/*
 * Decompiled with CFR 0.152.
 */
package io.kubernetes.client.extended.leaderelection;

import io.kubernetes.client.extended.leaderelection.LeaderElectionConfig;
import io.kubernetes.client.extended.leaderelection.LeaderElectionRecord;
import io.kubernetes.client.extended.leaderelection.Lock;
import io.kubernetes.client.openapi.ApiException;
import io.kubernetes.client.util.Threads;
import java.time.Duration;
import java.util.Date;
import java.util.LinkedList;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeaderElector
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(LeaderElector.class);
    private static final double JITTER_FACTOR = 1.2;
    private final LeaderElectionConfig config;
    private LeaderElectionRecord observedRecord;
    private long observedTimeMilliSeconds;
    private final Consumer<Throwable> exceptionHandler;
    private String reportedLeader;
    private Consumer<String> onNewLeaderHook;
    private final ScheduledExecutorService scheduledWorkers = Executors.newSingleThreadScheduledExecutor(Threads.threadFactory((String)"leader-elector-scheduled-worker-%d"));
    private final ExecutorService leaseWorkers = Executors.newSingleThreadExecutor(Threads.threadFactory((String)"leader-elector-lease-worker-%d"));
    private final ExecutorService hookWorkers = Executors.newSingleThreadExecutor(Threads.threadFactory((String)"leader-elector-hook-worker-%d"));

    public LeaderElector(LeaderElectionConfig config) {
        this(config, t -> log.error("Unexpected error on acquiring or renewing the lease", t));
    }

    public LeaderElector(LeaderElectionConfig config, Consumer<Throwable> exceptionHandler) {
        if (config == null) {
            throw new IllegalArgumentException("Config must be provided.");
        }
        LinkedList<String> errors = new LinkedList<String>();
        if (config.getLock() == null) {
            errors.add("Lock must be provided.");
        }
        if (config.getLeaseDuration() == null) {
            errors.add("LeaseDuration must be provided.");
        }
        if (config.getRetryPeriod() == null) {
            errors.add("RetryPeriod must be provided.");
        }
        if (config.getRenewDeadline() == null) {
            errors.add("RenewDeadline must be provided.");
        }
        if (config.getLeaseDuration().compareTo(config.getRenewDeadline()) <= 0) {
            errors.add("LeaseDuration must be greater than renewDeadline.");
        }
        if (config.getRenewDeadline().compareTo(config.getRetryPeriod()) <= 0) {
            errors.add("RenewDeadline must be greater than retryPeriod.");
        }
        if (config.getLeaseDuration().isZero() || config.getLeaseDuration().isNegative()) {
            errors.add("LeaseDuration must be greater than zero.");
        }
        if (config.getRenewDeadline().isZero() || config.getRenewDeadline().isNegative()) {
            errors.add("RenewDeadline must be greater than zero.");
        }
        if (config.getRetryPeriod().isZero() || config.getRetryPeriod().isNegative()) {
            errors.add("RetryPeriod must be greater than zero.");
        }
        if (errors.size() > 0) {
            throw new IllegalArgumentException(String.join((CharSequence)",", errors));
        }
        this.config = config;
        this.exceptionHandler = exceptionHandler;
    }

    public void run(Runnable startLeadingHook, Runnable stopLeadingHook) {
        this.run(startLeadingHook, stopLeadingHook, null);
    }

    public void run(Runnable startLeadingHook, Runnable stopLeadingHook, Consumer<String> onNewLeaderHook) {
        this.onNewLeaderHook = onNewLeaderHook;
        log.info("Start leader election with lock {}", (Object)this.config.getLock().describe());
        try {
            if (!this.acquire()) {
                return;
            }
            log.info("Successfully acquired lease, became leader");
            this.hookWorkers.submit(startLeadingHook);
            this.renewLoop();
            log.info("Failed to renew lease, lose leadership");
            stopLeadingHook.run();
        }
        catch (Throwable t) {
            stopLeadingHook.run();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean acquire() {
        if (log.isDebugEnabled()) {
            log.debug("Attempting to acquire leader lease...");
        }
        Duration retryPeriod = this.config.getRetryPeriod();
        long retryPeriodMillis = retryPeriod.toMillis();
        AtomicBoolean acquired = new AtomicBoolean(false);
        ScheduledFuture<?> scheduledFuture = this.scheduledWorkers.scheduleWithFixedDelay(() -> {
            Future<Boolean> future = this.leaseWorkers.submit(this::tryAcquireOrRenew);
            try {
                Boolean success = future.get(retryPeriodMillis, TimeUnit.MILLISECONDS);
                if (log.isDebugEnabled()) {
                    log.debug("The tryAcquireOrRenew result is {}", (Object)success);
                }
                acquired.set(success);
            }
            catch (CancellationException e) {
                log.info("Processing tryAcquireOrRenew successfully canceled");
            }
            catch (Throwable t) {
                this.exceptionHandler.accept(t);
                future.cancel(true);
            }
            finally {
                this.maybeReportTransition();
            }
        }, 0L, Double.valueOf((double)retryPeriodMillis * (1.2 * Math.random() + 1.0)).longValue(), TimeUnit.MILLISECONDS);
        try {
            while (!acquired.get()) {
                Thread.sleep(retryPeriodMillis);
            }
        }
        catch (InterruptedException e) {
            log.error("LeaderElection acquire loop gets interrupted", (Throwable)e);
            boolean bl = false;
            return bl;
        }
        finally {
            scheduledFuture.cancel(true);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void renewLoop() {
        if (log.isDebugEnabled()) {
            log.debug("Attempting to renew leader lease...");
        }
        Duration retryPeriod = this.config.getRetryPeriod();
        long retryPeriodMillis = retryPeriod.toMillis();
        Duration renewDeadline = this.config.getRenewDeadline();
        long renewDeadlineMillis = renewDeadline.toMillis();
        try {
            while (true) {
                boolean renewResult;
                Future<Boolean> future = this.leaseWorkers.submit(() -> {
                    try {
                        while (!this.tryAcquireOrRenew()) {
                            Thread.sleep(retryPeriodMillis);
                            this.maybeReportTransition();
                        }
                    }
                    catch (InterruptedException e) {
                        return false;
                    }
                    return true;
                });
                try {
                    renewResult = future.get(renewDeadlineMillis, TimeUnit.MILLISECONDS);
                }
                catch (ExecutionException | TimeoutException t) {
                    if (log.isDebugEnabled()) {
                        log.debug("failed to tryAcquireOrRenew", (Throwable)t);
                    }
                    renewResult = false;
                }
                finally {
                    future.cancel(true);
                }
                if (renewResult) {
                    if (log.isDebugEnabled()) {
                        log.debug("Successfully renewed lease");
                    }
                    Thread.sleep(retryPeriodMillis);
                    continue;
                }
                break;
            }
        }
        catch (InterruptedException e) {
            log.error("LeaderElection renew loop gets interrupted", (Throwable)e);
        }
    }

    private boolean tryAcquireOrRenew() {
        boolean renewalStatus;
        LeaderElectionRecord oldLeaderElectionRecord;
        Date now = new Date();
        Lock lock = this.config.getLock();
        LeaderElectionRecord leaderElectionRecord = new LeaderElectionRecord(lock.identity(), Long.valueOf(this.config.getLeaseDuration().getSeconds()).intValue(), now, now, 0, this.config.getOwnerReference());
        try {
            oldLeaderElectionRecord = lock.get();
        }
        catch (ApiException e) {
            if (e.getCode() != 404) {
                log.error("Error retrieving resource lock {}", (Object)lock.describe(), (Object)e);
                return false;
            }
            if (log.isDebugEnabled()) {
                log.debug("Lock not found, try to create it");
            }
            return this.createLock(lock, leaderElectionRecord);
        }
        if (oldLeaderElectionRecord == null || oldLeaderElectionRecord.getAcquireTime() == null || oldLeaderElectionRecord.getRenewTime() == null || oldLeaderElectionRecord.getHolderIdentity() == null) {
            if (log.isDebugEnabled()) {
                log.debug("Update lock to get lease");
            }
            if (oldLeaderElectionRecord != null) {
                leaderElectionRecord.setLeaderTransitions(oldLeaderElectionRecord.getLeaderTransitions() + 1);
            }
            return this.updateLock(lock, leaderElectionRecord);
        }
        if (!oldLeaderElectionRecord.equals(this.observedRecord)) {
            this.observedRecord = oldLeaderElectionRecord;
            this.observedTimeMilliSeconds = System.currentTimeMillis();
        }
        if (this.observedTimeMilliSeconds + this.config.getLeaseDuration().toMillis() > now.getTime() && !this.isLeader()) {
            log.debug("Lock is held by {} and has not yet expired", (Object)oldLeaderElectionRecord.getHolderIdentity());
            return false;
        }
        if (this.isLeader()) {
            leaderElectionRecord.setAcquireTime(oldLeaderElectionRecord.getAcquireTime());
            leaderElectionRecord.setLeaderTransitions(oldLeaderElectionRecord.getLeaderTransitions());
        } else {
            leaderElectionRecord.setLeaderTransitions(oldLeaderElectionRecord.getLeaderTransitions() + 1);
        }
        if (log.isDebugEnabled()) {
            log.debug("Update lock to renew lease");
        }
        if ((renewalStatus = this.updateLock(lock, leaderElectionRecord)) && log.isDebugEnabled()) {
            log.debug("TryAcquireOrRenew return success");
        }
        return renewalStatus;
    }

    private boolean createLock(Lock lock, LeaderElectionRecord leaderElectionRecord) {
        boolean createSuccess = lock.create(leaderElectionRecord);
        if (!createSuccess) {
            return false;
        }
        this.observedRecord = leaderElectionRecord;
        this.observedTimeMilliSeconds = System.currentTimeMillis();
        return true;
    }

    private boolean updateLock(Lock lock, LeaderElectionRecord leaderElectionRecord) {
        boolean updateSuccess = lock.update(leaderElectionRecord);
        if (!updateSuccess) {
            return false;
        }
        this.observedRecord = leaderElectionRecord;
        this.observedTimeMilliSeconds = System.currentTimeMillis();
        return true;
    }

    private boolean isLeader() {
        return this.config.getLock().identity().equals(this.observedRecord.getHolderIdentity());
    }

    private void maybeReportTransition() {
        if (this.observedRecord == null) {
            return;
        }
        if (this.observedRecord.getHolderIdentity().equals(this.reportedLeader)) {
            return;
        }
        this.reportedLeader = this.observedRecord.getHolderIdentity();
        if (this.onNewLeaderHook != null) {
            this.hookWorkers.submit(() -> {
                log.info("LeaderElection lock is currently held by {}", (Object)this.observedRecord.getHolderIdentity());
                this.onNewLeaderHook.accept(this.reportedLeader);
            });
        }
    }

    @Override
    public void close() {
        log.info("Closing...");
        this.scheduledWorkers.shutdownNow();
        this.leaseWorkers.shutdownNow();
        this.hookWorkers.shutdownNow();
        if (this.observedRecord != null && this.isLeader()) {
            try {
                boolean isTerminated = this.scheduledWorkers.awaitTermination(this.config.getRetryPeriod().getSeconds(), TimeUnit.SECONDS);
                if (!isTerminated) {
                    log.warn("scheduledWorkers executor termination didn't finish.");
                    return;
                }
                isTerminated = this.leaseWorkers.awaitTermination(this.config.getRetryPeriod().getSeconds(), TimeUnit.SECONDS);
                if (!isTerminated) {
                    log.warn("leaseWorkers executor termination didn't finish.");
                    return;
                }
                isTerminated = this.hookWorkers.awaitTermination(this.config.getRetryPeriod().getSeconds(), TimeUnit.SECONDS);
                if (!isTerminated) {
                    log.warn("hookWorkers executor termination didn't finish.");
                    return;
                }
            }
            catch (InterruptedException ex) {
                log.warn("Failed to ensure executors termination.", (Throwable)ex);
                return;
            }
            log.info("Giving up the lock....");
            LeaderElectionRecord emptyRecord = new LeaderElectionRecord();
            emptyRecord.setLeaderTransitions(this.observedRecord.getLeaderTransitions());
            emptyRecord.setLeaseDurationSeconds(Long.valueOf(this.config.getLeaseDuration().getSeconds()).intValue());
            boolean status = this.config.getLock().update(emptyRecord);
            if (!status) {
                log.warn("Failed to give up the lock.");
            }
        }
        log.info("Closed");
    }
}

