use warnings;
use strict;
use utf8;
use FindBin '$Bin';
use Test::More;
my $builder = Test::More->builder;
binmode $builder->output, ":encoding(utf8)";
binmode $builder->failure_output, ":encoding(utf8)";
binmode $builder->todo_output, ":encoding(utf8)";
binmode STDOUT, ":encoding(utf8)";
binmode STDERR, ":encoding(utf8)";

use Image::SVG::Path 'extract_path_info';

# These paths were extracted from the "Super Tiny Icons" collection of
# SVG icons. They are the cases where the pre-version-0.34 parsing of
# elliptical arc arguments failed.

# See https://github.com/edent/SuperTinyIcons for licence information.

my @paths = (
{
    file => 'acast.svg',
    path => 'M256 64a192 192 0 1039 380l152-212c-12-95-93-168-191-168zm8 74c76 0 113 37 113 71l-93 134c-12 19-40 33-68 33-48 0-79-23-79-58 0-64 106-68 145-68v-42c0-8 0-24-19-24-17 0-18 11-18 14l2 14c0 14-18 24-45 24-29 0-47-15-47-39 0-30 39-59 109-59zm-28 169c0 13 8 20 21 20s23-10 23-22l1-31c-30 0-45 15-45 33z',
},
{
    file => 'amazon_alexa.svg',
    path => 'M256 64a192 192 0 00-26 382v-39c0-11 -7 -21 -17 -24a134 134 0 11177-122v1a141 141 0 010 5l0 1a126 126 0 010 3a130 130 0 010 4l0 2C372 385 231 446 230 446A194 194 0 00256 448a192 192 0 000-384',
},
{
    file => 'andotp.svg',
    path => 'M256 117a139 139 0 00-139 139 139 139 0 00139 139 139 139 0 00139-139 139 139 0 00-139-139zm0 38c45 0 85 30 97 73H159c12-43 52-73 97-73zm0 115a20 20 0 0120 20 20 20 0 01-12 18l16 36h-48l15-36a20 20 0 01-11-18 20 20 0 0120-20z',
},
{
    file => 'angellist.svg',
    path => 'M342 230c39 4 47 43 47 75 0 65-33 128-95 154-51 21-114 13-155-25-25-22-39-51-40-84 0-30 15-49 44-57-3-8-9-19-9-28 0-18 20-40 38-40 9 0 17 3 24 8-15-58-75-175-5-181 34-3 69 125 78 153 11-26 45-145 80-145 66 0 7 130-7 170zM138 348c13 13 22 41 45 41 29-3-18-76-39-76-17 0-26 21-26 36 0 38 31 74 65 89 50 23 111 12 149-29 33-37 43-88 34-136-3-24-35-29-54-33-23-6-47-8-71-8-30 0-17 36 3 44 26 9 55 9 83 9 6 0 9 6 9 12-13 10-30 13-43 23a79 79 0 00-34 86c2 7 4 14 4 21-26 0-27-37-27-54-5 5-13 4-20 3 4 18-5 38-27 38-24 0-55-26-55-52 0-4 1-11 4-14zm86 6c35 0-8-67-15-77-10-17-28-38-46-20s7 50 19 65c10 14 23 32 42 32zm28-140l-32-87c-4-12-18-56-32-56-26 0-7 51-4 62 7 22 17 51 31 85 10-7 26-6 37-4zm32 88c-15 0-32-2-45-8 5 13 10 24 13 38 9-12 20-22 32-30zm39-77c13-34 22-64 30-86 3-10 22-59-1-59-17 0-32 43-36 56l-29 82 36 7z',
},
{
    file => 'bitbucket.svg',
    path => 'M422 130a10 10 0 00-9.9-11.7H100.5a10 10 0 00-10 11.7L136 409a10 10 0 009.9 8.4h221c5 0 9.2-3.5 10 -8.4L422 130zM291 316.8h-69.3l-18.7-98h104.8z',
},
{
    file => 'chromium.svg',
    path => 'M256 140h228a256 256 0 01-240 371.7',
},
{
    file => 'digitalocean.svg',
    path => 'M78 373v-47h47v104h57V300h74v147A191 191 0 1065 256h74a117 117 0 11117 117',
},
{
    file => 'disqus.svg',
    path => 'M83 387l26-65a159 159 0 1142 56zm168-52q90-6 90 -81t-89-80h-61v161zm2-39h-18v-82h18q43 2 43 40t-43 42',
},
{
    file => 'espressif.svg',
    path => 'M126 120.53a188.46 188.46 0 00266.53 266.52a7.69 7.69 0 1110.87 10.87a203.83 203.83 0 01-288.26-288.26A7.69 7.69 0 01126 120.53zm52.5 240.22a26 26 0 100 1M409 334a273 273 0 00-231.5-231.5A150 150 0 00139 130v26a216 216 0 01217 216h26a111 111 0 0027-39M240 83l-4 12a298 298 0 01181 180l11-4A172 172 0 00240 83zm5 342a118 118 0 00-95-171a10.61 10.61 0 013-21a140 140 0 01120 177l31 9l26.5-10a195 195 0 00-163-230a66.6 66.6 0 10-18 131.5a63 63 0 0142 95.5l21 14z',
},
{
    file => 'finder.svg',
    path => 'M435.2 0H274.4c-21.2 49.2-59.2 129.6-60.8 283.4a9.9 9.9 0 0010 10.1h58.7a9.9 9.9 0 019.9 10.2A933.3 933.3 0 00311.3 512h123.9a76.8 76.8 0 0076.8-76.8V76.8A76.8 76.8 0 00435.2 0z',
},
{
    file => 'firefox.svg',
    path => 'M48 49s-3-9-1-16c-9 2-33 35-33 35a51 48 0 1087-32s5 9 5 15c-3-9-20-25-26-37-24 13-16 39-16 39',
},
{
    file => 'freecodecamp.svg',
    path => 'M376.8 128.6c-2.2 0-3.3 3.3-3.3 4.5 0 3.3 4.4 7.7 11 15.5 29 27.8 44.6 62.3 44.6 103.5 0 45.6-16.7 83.5-46.7 111.3-5.6 5.5-9 10-9 14.4l3.4 4.5a8.9 8.9 0 005.6 3.3c6.6 0 16.7-7.8 28.9-24.5 24.5-30 35.6-63.4 36.7-109 0-45.6-13.3-76.8-42.3-109-10-11.2-17.8-16.7-23.3-16.7-2 .1-4 .9-5.6 2.2zm-87.9 131.3c-11.1-2.2 33.4-55.6-46.7-119 0 0 11 33.3-42.3 107.9-52.3 74.5 23.4 118 23.4 118s-35.7-19 6.6-86.9c6.7-11.1 16.7-22.2 29-49 0 0 11 15.6 5.5 48-8.9 50 35.6 36.6 36.7 36.6 15.6 17.8-13.3 51.2-14.4 52.3s81.2-51.2 22.2-128c-4.4 4.5-10 23.4-21.1 20zM135.3 128.6a6.7 6.7 0 012.3 4.5c0 3.3-3.4 7.7-11.2 15.5A137.7 137.7 0 0083 252.1c0 45.6 15.6 82.3 45.7 111.3 6.6 5.5 8.9 11.1 8.9 13.3l-2.3 5.6a8.9 8.9 0 01-5.5 3.3c-7.8 0-16.7-8.9-30-24.5a164.7 164.7 0 01-35.7-109c0-45.6 13.4-76.8 41.2-109 11.1-11.2 19-16.7 24.5-16.7 2 .1 4 .9 5.5 2.2z',
},
{
    file => 'gandi.svg',
    path => 'M255 66a54 54 0 1054 54 54 54 0 00-54-54zm0 71a17 17 0 1117-17 17 17 0 01-17 17z',
},
{
    file => 'git.svg',
    path => 'M440.8 238.9L273 71.2a24.7 24.7 0 00-35 0l-34.8 34.9l44.2 44.1a29.4 29.4 0 0137.2 37.5l42.5 42.5a29.4 29.4 0 11-17.6 16.6l-39.7-39.7v104.5a29.4 29.4 0 11-24.2-.9V205.3a29.4 29.4 0 01-16-38.6l-43.5-43.5l-115 115a24.7 24.7 0 000 34.9L239 440.8a24.7 24.7 0 0035 0l166.9-167a24.7 24.7 0 000-34.9',
},
{
    file => 'gitea.svg',
    path => 'M418.773 149.5c-58 3-92 5-121 5v65l-9-4v-61c-34 0-63-2-119-5l-27-1c-27 0-63 19-61 67 3 75 71 82 99 83 3 14 35 62 59 65h104c63-5 109-213 75-214zm-311 67c-3-21 7-42 42-42 3 39 10 61 22 96-32-5-59-15-64-54zm236 48l-29 60c-3 7-13 10-20 7l-60-29c-8-4-11-13-8-21l30-60c3-8 13-11 20-7l11 5-8 16c-5.497.533-9.521 5.421-8.989 10.918a9.994 9.994 0 001.989 5.082l-14 27c-5.522.091-9.925 4.641-9.834 10.163.022 1.321.305 2.626.834 3.837 1.946 5.169 7.714 7.781 12.883 5.834s7.781-7.714 5.834-12.883a10 10 0 00-2.717-3.952l13-27c2.83.515 5.745-.214 8-2l1 1c10 5 16 7 17 11 2 4-3 13-11 29v1c-5.483-.666-10.467 3.239-11.132 8.722s3.239 10.467 8.722 11.132a10.009 10.009 0 005.41-.854c5.176-1.926 7.81-7.684 5.884-12.86a9.99 9.99 0 00-1.884-3.14c10-20 14-29 11-36-2-6-9-9-21-15l-1-1a10.001 10.001 0 00-3-7l8-16 42 21c8 4.001 11 13.001 7 21.001z',
},
{
    file => 'glitch.svg',
    path => 'M78 189q20 25-15 30 50 40 97 32 14 35 76 25l-2-16s72 2 138-26c12-3 79 35 64 9s6-22-5-54',
},
{
    file => 'gojek.svg',
    path => 'M256 91a171 171 0 00-66 328 32 32 0 0044-24 32 32 0 00-19-35 106 106 0 0141-205 107 107 0 0141 205 32 32 0 0013 61 33 33 0 0012-3 170 170 0 00-66-327',
},
{
    file => 'google_maps.svg',
    path => 'M375 136a133 133 0 00-79-66 136 136 0 00-40-6 133 133 0 00-103 48 133 133 0 00-31 86c0 38 13 64 13 64 15 32 42 61 61 86a399 399 0 0130 45 222 222 0 0117 42c3 10 6 13 13 13s11-5 13-13a228 228 0 0116-41 472 472 0 0145-63c5-6 32-39 45-64 0 0 15-29 15-68 0-37-15-63-15-63z',
},
{
    file => 'gradle.svg',
    path => 'M427 132.7a61 61 0 00-85-1a6 6 0 000 9l7 8a6 6 0 008 1a35 35 0 0146 53c-48 48-113 -87 -259 -17a20 20 0 00-9 28l25 43a20 20 0 0027 7l1 0l0 0l11-6a257 257 0 0035-26a6 6 0 018 0v0a6 6 0 010 9a263 263 0 01-37 28h0l-11 6a31 31 0 01-15 4a32 32 0 01-28-16L126 219C81 259 53 314 68.13 392.26a6 6 0 006 4.74H100.6a6 6 0 005.93-5.3a40 40 0 0178.62 0a6 6 0 005.72 5.08h26.2a6 6 0 005.7-5.1a40 40 0 0178.6 0a6 6 0 005.7 5h26a6 6 0 005.8-5.72c1-37 10 -79 38.7 -100c98-73 72 -136 49.4 -158.3zm-100 110l-19-9v0a12 12 0 1119 9z',
},
{
    file => 'grafana.svg',
    path => 'M390 146c5-22-7-40-7-40-20-2-34 6-38 10-13-6-26-9-39-11-9-29-35-41-35-41a78 78 0 00-34 45c-16 5-31 12-43 21-24-10-53-9-77 2-3 43 16 70 20 74-5 14-9 29-10 42-37 19-48 56-48 56 16 17 43 38 67 38 7 12 15 22 24 32-11 32 1 58 1 58 35 2 57-15 62-18 19 5 37 9 56 8 11 14 27 23 45 26 11-16 21-33 21-49 14-10 26-23 35-37 22 2 39-14 39-14-3-14-10-30-22-38 3-23-2-46-13-65-19-30-51-47-85-47-23 0-44 11-58 27a71 71 0 00-15 65c5 19 21 35 38 39s34 2 43-7c3-3-1-8-5-6-7 3-14 4-22 3-59-9-45-86 10-89 11-2 26 3 35 7 32 21 39 52 26 84a92 92 0 01-76 51c-18 0-35-5-49-12-34-20-56-53-58-92-3-76 41-110 90-121 29-6 61-3 87 10 36 18 61 52 72 91 1 4 6 4 6-1 0-38-17-75-43-101z',
},
{
    file => 'haml.svg',
    path => 'M115 91l81 165-43 184c27 19 61 25 90 17a11062 11062 0 0028-153c18 21 47 29 72 32-1 5-7 16-2 18 17 10 40 13 55 0 8-7 12-15 12-25 12-6 23-16 21-29-1-19-18-32-31-41-11-8-34-10-40-21-8-13 2-24 2-37l28-24 3 2s9 1 14-12l-2-2c3-3 10-9 9-10-2-2-10 4-14 7l-9-6 18-15c5 1 14 8 16 3-7-16-37-30-47-32-2 0-3 2-5 3l-9-3-112 95-46-99h3c1-2 3-13-9-11 0-2-7-22-11-23-15-6-53 10-72 17zm68-6l6 12-13 1-6-11zm12 24l52 101-8 7-56-106zm188 9c7 1 35 20 36 22-3 0-35-20-36-22zm-39 94c-4 11-5 24 2 32 11 15 31 14 46 23 12 7 27 18 28 32 1 10-5 17-12 23 0-1-3-20-5-15-9 24-28 15-45 8l-3 1-10 15c-31-3-64-39-72-44l2-23c23-16 46-34 69-52zm-75 53l2 3-32 183-3 1 26-185 114-99 6 5z',
},
{
    file => 'homekit.svg',
    path => 'M118 406a11 11 0 01-5 0 13 13 0 010-5V218c0-6-5-11-11-11H82L256 69l104 82c8 5 18 0 18-9v-25h15v55a11 11 0 004 8l34 27h-21c-6 0-11 5-11 11v183a13 13 0 010 4 11 11 0 01-5 1zM241 83l-114 90c-7 5-14 14-14 29v177c0 15 9 25 24 25h238c15 0 24-10 24-25V202c0-15-7-24-14-29L271 83c-10-6-22-6-30 0zm-67 261V217c0-4 1-5 2-6l80-63 80 63c1 1 2 1 2 6v127zm82-189c-6 0-9 1-14 5l-58 45c-9 7-9 15-9 20v97c0 12 8 20 19 20h124c11 0 19-8 19-20v-97c0-5 0-13-9-20l-58-46c-5-3-9-4-14-4zm-28 134v-49l28-21 28 21v48zm28-66c-4 0-6 2-10 5l-11 9a15 15 0 00-6 11v26c0 8 6 13 13 13h28c7 0 13-6 13-13v-26a15 15 0 00-6-11l-10-9-11-5',
},
{
    file => 'intercom.svg',
    path => 'M365 272a10 10 0 01-11 10 10 10 0 01-10-10v-91a10 10 0 0110-10c3 0 6 1 8 3s3 4 3 7v91zm-4 63c-1 2-39 33-108 33s-106-31-108-33a10 10 0 1113-15c1 1 34 28 95 28s95-28 95-28a10 10 0 0114 1 10 10 0 01-1 14zM142 181a10 10 0 0110-10 10 10 0 0110 10v91a10 10 0 01-10 10 10 10 0 01-10-10v-91zm50-20a10 10 0 0111-10 10 10 0 0110 10v135a10 10 0 01-11 10 10 10 0 01-10-10V161zm51-5a10 10 0 0110-10 10 10 0 0111 10v146a10 10 0 01-11 11 10 10 0 01-10-11V156zm51 5a10 10 0 0110-10 10 10 0 0110 10v135a10 10 0 01-10 10 10 10 0 01-10-10V161zm73-56H139a38 38 0 00-35 24c-2 4-3 9-3 14v228a38 38 0 0024 35c4 2 9 3 14 3h228a38 38 0 0035-24c2-4 3-9 3-14V143a38 38 0 00-23-35l-15-3',
},
{
    file => 'itch_io.svg',
    path => 'M118 95c-16 10-49 47-49 56v16c0 21 19 38 36 38 21 0 38-17 38-37 0 20 17 37 38 37 20 0 36-17 36-37 0 20 18 37 39 37s39-17 39-37c0 20 16 37 36 37 21 0 38-17 38-37 0 20 17 37 38 37 17 0 36-17 36-38v-16c0-9-33-46-49-56a3511 3511 0 00-276 0zm99 101l-7 9a43 43 0 01-68-9l-7 9c-8 8-19 13-31 13l-4-1-2 46v18c0 36-4 118 16 138 30 7 86 10 142 10s112-3 142-10c20-20 16-102 16-138v-18l-2-46-4 1c-12 0-23-5-31-13l-7-9-7 9a43 43 0 01-68-9 43 43 0 01-38 22h-1-1a43 43 0 01-38-22zm-31 40c12 0 23 0 37 15l33-2 33 2c14-15 25-15 37-15 6 0 29 0 45 46l18 63c13 46-4 47-26 47-31-1-49-24-49-47a371 371 0 01-117 0c1 23-17 46-48 47-22 0-39-1-26-47l18-63c16-46 39-46 45-46zm70 36s-33 31-39 42l22-1v19h34v-19l22 1c-6-11-39-42-39-42z',
},
{
    file => 'keepassdx.svg',
    path => 'M282.79 195.24l-173.43 173.6-.53 34.98 34.97-.53 36.32-36.28h39.14V330.2h36.8v-39.06l57.25-57.18-30.52-38.72z',
},
{
    file => 'keskonfai.svg',
    path => 'M284.2 404L236 324.8l-29.6 28v43.4q0 15.8-8.4 24.4-8.2 8.4-19 8.4-12.6 0-19.8-8.4-7.2-8.4-7.2-24.8V163q0-18.2 7-27.6 7-9.6 20-9.6 12.6 0 20 8.6t7.4 25.4v132.4l61.4-64.4q11.4-12 17.4-16.4 6-4.4 14.6-4.4 10.2 0 17 6.6 6.8 6.4 6.8 16.2 0 12-22.2 32l-29 26.6 56 88q6.2 9.8 8.8 15 2.8 5 2.8 9.6 0 13-7.2 20.6-7 7.4-18.6 7.4-10 0-15.4-5.4T284.2 404z',
},
{
    file => 'linux_mint.svg',
    path => 'M59.6 97.5v100h40v101a121 121 0 00120 120h198v-201a121 121 0 00-120-120z',
},
{
    file => 'macos.svg',
    path => 'M282 170v-4c-52 0-5 34 0 4zm24-18c7-21 43-23 47 3h-10c-3-15-28-16-28 11 0 15 23 24 28 6h10c-6 33-59 21-47-20zm-146-16h10v9c5-12 27-13 31 1 7-15 35-14 35 7v37h-11v-34c0-15-22-15-22 1v33h-11v-35c-2.447-9.36-14.915-11.23-20-3l-2 5v33h-10zm23 259c-47 0-76-33-76-86s29-85 76-85 77 33 77 85-29 86-77 86zm88-205c-29 7-33-30-3-31l14-1v-4c1-12-19-13-22-2h-10a14 14 0 012-7c8-14 41-14 41 8v37h-10v-9a18 18 0 01-12 9zm68 205c-36-2-61-19-63-49h24c23 72 146-5 25-30-19-4-33-13-39-24-38-74 109-96 113-20h-23c-7-49-98-22-65 12 14 14 43 13 64 22 50 23 26 91-36 89zM183 245c-32 0-52 25-52 64s20 64 52 64 53-24 53-64-20-64-53-64z',
},
{
    file => 'mattermost.svg',
    path => 'M429.6 174.4a190 190 0 00-55.3-69.8l2 41.2l2.2 2.9a141.3 141.3 0 0125.3 113.6a144.8 144.8 0 01-179 112.7C153 357 105 283.4 118.6 210.6c10.7-57.5 44.7 -95.8 99.9 -115.2l2-.8l1.6-1.2l20.1-29A190.9 190.9 0 0065 237.5a191.5 191.5 0 00152.4 206.1a191.2 191.2 0 00222-132c14.3-46.7 11 -92.8 -9.7 -137.3zm-236 61.7a69.7 69.7 0 0078.5 62.4a70 70 0 0060.3-81.3c-5.7-30.2 -12.1 -60.3 -18.2 -90.4l-7.8-38.1l-2 1.7l-40.2 39.5l-49 48a68.1 68.1 0 00-21.5 58.2z',
},
{
    file => 'messenger.svg',
    path => 'M256 78.2c-102.4 0-181.8 75 -181.8 176.4c0 53 21.7 98.8 57 130.4a14.7 14.7 0 015 10.4l1 32.3a14.6 14.6 0 0020.4 12.9l36-16a14.5 14.5 0 019.8-.7a197.8 197.8 0 0052.6 7c102.4 0 181.8-75 181.8 -176.3S358.4 78.2 256 78.2z',
},
{
    file => 'ok.svg',
    path => 'M256 137a36.5 36.5 0 0136.5 36.5a36.5 36.5 0 01-36.5 36.5a36.5 36.5 0 01-36.5-36.5a36.5 36.5 0 0136.5-36.5zm0 124.5c48.6 0 88-39.5 88 -88s-39.5-88 -88 -88s-88 39.5-88 88s39.5 88 88 88m35.6 71.9a164.8 164.8 0 0051-21a25.8 25.8 0 008-35.6a25.8 25.8 0 00-35.6-8a111.9 111.9 0 01-118.6 0a25.8 25.8 0 00-35.6 8a25.8 25.8 0 008 35.6c15.9 10 33 17 51 21l-49 49.3a25.8 25.8 0 0036.5 36.5l48.4-48.5l48.4 48.4a25.8 25.8 0 0036.5 0a25.8 25.8 0 000-36.5l-49.3-49.3',
},
{
    file => 'opensource.svg',
    path => 'M192 430a180 180 0 11128 0l-40-109a63 63 0 10-87-58c0 26 16 49 39 58z',
},
{
    file => 'pixelfed.svg',
    path => 'M254 256c32-42 23-102-20-134s-108-15-140 27l11-19c52-63 143-89 210-58 50 17 78 72 62 123s-72 78-123 61',
},
{
    file => 'pocketcasts.svg',
    path => 'M256 400a144 144 0 11144-144h48a192 192 0 10-192 192v-48zm0-71a73 73 0 1173-73h42a115 115 0 10-115 115v-42z',
},
{
    file => 'qq.svg',
    path => 'M261 398a57 32 0 00114 0 57 32 0 00-114 0zm-124 0a57 32 0 00114 0 57 32 0 00-114 0z',
},
{
    file => 'redhat.svg',
    path => 'M379 255l3 20c5 89-241 29-237-46 0-5 1-9 3-14l7-17s170 44 220 39z',
},
{
    file => 'rust.svg',
    path => 'M20-183 6-206c-3-5-9-5-12 0l-14 23m0 366 14 23c3 5 9 5 12 0l14-23',
},
{
    file => 'samsung_internet.svg',
    path => 'M46 221a228 80 15 10441 118',
},
{
    file => 'samsung_swoop.svg',
    path => 'M118 232c-12 0-17 4-16 15 2 9 19 14 20 17v3c0 2-1 3-4 3s-4-2-4-4v-4h-12c-1 12 8 16 16 16 10 0 16-3 16-13 1-14-18-15-20-21v-2c0-1 1-3 4-3s4 2 4 4v3h11v-3c0-10-9-11-15-11zm265 11v25c1 16 29 13 29 0v-15h-13v6h4v8c-1 4-10 4-10 0v-23c0-4 8-4 9 0v4h10v-5c-1-15-29-13-29 0zm-120-11c-12 0-17 4-16 15 2 9 19 14 20 17v3c0 2-1 3-4 3s-4-2-4-4v-4h-12c-1 12 8 16 16 16 10 0 16-3 16-13 1-14-18-15-20-21v-2c0-1 1-3 4-3s4 2 4 4v3h11v-3c0-10-9-11-15-11zm-75 1l-1 44h11v-40l8 40h11l7-40v40h11l-1-44h-18l-5 34-5-34zm101 0v34c1 8 7 11 16 11 8 0 15-3 15-11v-34h-11v34c0 1-1 3-5 3-3 0-4-2-4-3v-34zm-139 0l-8 44h12l6-40 6 40h12l-8-44zm182 0v44h11v-36l11 36h16v-44h-11v35l-10-35zm136-14a61 208 80 01-203 98 61 217 80 01-225-25 63 211 80 01204-97 61 217 80 01224 23',
},
{
    file => 'sentry.svg',
    path => 'M283 122a31 31 0 00-54 0l-44 77 11 6a216 216 0 01108 174h-31a185 185 0 00-92-147l-11-7-42 72 12 6a104 104 0 0150 76h-71a5 5 0 01-4-7l20-34a72 72 0 00-23-13l-19 34a31 31 0 0026 46h98v-13a129 129 0 00-54-104l16-27a160 160 0 0169 131v13h82v-13a242 242 0 00-110-203l32-54a5 5 0 018 0l137 237a5 5 0 01-5 7h-32v26h33a31 31 0 0026-46z',
},
{
    file => 'signal.svg',
    path => 'M256 100a156 156 0 00-132 239l-15 64 64-15a156 156 0 1083-288z',
},
{
    file => 'slack.svg',
    path => 'M149 305a39 39 0 01-78 0c0-22 17 -39 39 -39h39zM168 305a39 39 0 0178 0v97a39 39 0 01-78 0z',
},
{
    file => 'svelte.svg',
    path => 'M395 115c-35-51-106-66-157-34l-89 57a103 103 0 00-46 69 108 108 0 0010 69 103 103 0 00-15 39 109 109 0 0019 82c35 51 106 66 157 34l89-57a103 103 0 0046-69 108 108 0 00-10-69 103 103 0 0015-39 109 109 0 00-19-82',
},
{
    file => 'teamspeak.svg',
    path => 'M407 336c5 5 9 18 9 26s0 15-2 21c0 3-6 33-43 51l-23 8-14 3c-64 12-148-3-84-10l17-2a1532 1532 0 0042-10h1a19086 19086 0 0012-4c16-7 33-16 47-29 10-10 19-22 26-37l6-15c0-2 4-3 6-2zm-60-56c17-3 28 5 29 19 1 13-3 25-13 35-8 9-18 13-32 12a249 249 0 01-40-5c-19-5-26-19-14-30 7-7 16-12 25-16l23-9c7-3 15-5 22-6zm-211-15c3-18 18-26 34-18a1135 1135 0 0142 28 141 141 0 0133 39l1 10c-2 10-9 15-18 19l-13 1h-19l-16-2c-15-1-26-7-34-19a77 77 0 01-10-58zm256-153l3 2c4 4 8 9 3 16l-1 1v2l2 2c16 22 27 46 32 73l2 4 2 1c14 7 21 18 21 33v19l-1 21c-4 20-23 33-42 29-6-1-7-6-7-11a11534 11534 0 010-66 150 150 0 00-300-1v69c0 6-4 9-10 9-24 1-40-14-40-38v-27l1-10c2-12 8-21 19-27h1l1-1c2-1 3-3 3-5 5-26 15-50 30-71l2-3v-5c-3-4-1-7 1-11h1c22-25 48-43 80-53 75-23 141-7 197 48z',
},
{
    file => 'tutanota.svg',
    path => 'M113 97c-17 0-31 14.09-31.455 31.451v290.461c0 1.102.063 2.212.17 3.284a1700 1700 0 0037-14c91-36 165-67 165-102 0-1.12-.077-2.25-.236-3.38-4.717-34.512-87.191-45-87-61.028.01-.84.248-1.719.754-2 9-17 49-16 63-17 14-1 48-.996 50-11.336.048-.32.078-.637.078-.956.039-9-23-13-23-13s28 4 28 15c0 .54-.066 1.1-.217 1.67-3.055 11-28 14-44 14-15.615.782-39 2-39 10.186-.01.445.067.908.226 1 3 11 90 16 146 45 32 16 48 44 55 73V128c0-17-14-31-31-31H113z',
},
{
    file => 'twitter.svg',
    path => 'M437 152a72 72 0 01-40 12a72 72 0 0032-40a72 72 0 01-45 17a72 72 0 00-122 65a200 200 0 01-145-74a72 72 0 0022 94a72 72 0 01-32-7a72 72 0 0056 69a72 72 0 01-32 1a72 72 0 0067 50a200 200 0 01-105 29a200 200 0 00309-179a200 200 0 0035-37',
},
{
    file => 'ubiquiti.svg',
    path => 'M112 94v18h18V94h-18zm288 0c-82 0-90 31-90 61v172a147 147 0 01-3 28c43-9 72-36 86-82l7-23V94zm-234 18v18h18v-18h-18zm-18 18v18h18v-18h-18zm36 9v18h18v-18h-18zm-72 4v147c0 73 53 128 144 128 0 0-54-30-54-91V197h-18v66h-18v-39h-18v17h-18v-98h-18zm54 18v18h18v-18h-18zm-18 27v18h18v-18h-18zm252 87c-19 64-65 92-131 89-24-1-43-7-57-16 10 42 46 63 48 64l10 6c82-5 130-59 130-128v-15z',
},
{
    file => 'ubisoft.svg',
    path => 'M403 327c-28 63-85 96-146 95-117-6-151-141-73-190l3 4c-85 110 82 216 159 89 63-132-94-262-230-162l-3-3c36-57 106-82 172-65a170 170 0 01118 232zm-110-2a52 52 0 01-89-5c-19-40 8-76 42-77 42-2 59 45 33 72l14 10zm-183-5l-5 2c-45-173 177-218 221-81l-4 2c-66-94-218-61-212 77zm337-64a201 201 0 00-380-89l15 11c-13 29-19 57-19 86 6 128 100 192 194 192 94-1 191-75 190-200',
},
{
    file => 'unicode.svg',
    path => 'M428 199h-37c-2 0-3 1-3 3v65a779 779 0 001 50v9l-10-20-16-30-19-35-20-37-19-36-16-29-2-2-3-1h-54c-2 0-3 1-3 3v150c0 12-1 22-4 29s-6 14-11 19c-4 5-10 8-16 10l-21 3c-19 0-32-5-41-16s-14-26-14-44V139c0-2-1-3-3-3H77c-2 0-3 2-3 4v154c0 16 2 31 7 43a78 78 0 0051 48 138 138 0 0081 1c12-4 23-10 32-18 8-8 16-18 21-31s8-28 8-46v-96a613 613 0 0025 50 4630 4630 0 0080 142h49c2 0 3-1 3-3V203c0-3-1-4-3-4z',
},
{
    file => 'uphold.svg',
    path => 'M360 277c-9 19-20 36-33 50 9-34 5-78-15-120-10-22-24-42-40-59 25-19 51-25 71-16 13 6 23 17 28 33 12 31 7 72-11 112m-208 0c-18-40-23-81-11-112 5-16 15-27 28-33 20-9 46-3 71 16-16 17-30 37-40 59-20 42-24 86-15 120-13-14-24-31-33-50m123 86a47 47 0 01-38 0c-38-17-46-85-17-148 10-20 22-37 36-52 14 15 26 32 36 52 29 63 21 131-17 148m52-256c-23-1-48 8-71 27-23-19-48-28-71-27a126 126 0 01142 0m66 51c-20-56-74-94-137-94s-118 38-137 94v1c-13 35-8 82 12 127 28 60 79 102 124 102h2c45 0 96-42 124-102 20-45 25-92 12-128m-98 263c-13 4-26 6-38 6h-2c-12 0-25-2-37-6-6-1-13 2-14 7-2 6 1 12 7 14 15 4 30 6 44 6h2c14 0 29-2 44-6 6-2 9-8 8-13-2-6-9-9-14-8',
},
{
    file => 'uplay.svg',
    path => 'M403 327c-28 63-85 96-146 95-117-6-151-141-73-190l3 4c-85 110 82 216 159 89 63-132-94-262-230-162l-3-3c36-57 106-82 172-65a170 170 0 01118 232zm-110-2a52 52 0 01-89-5c-19-40 8-76 42-77 42-2 59 45 33 72l14 10zm-183-5l-5 2c-45-173 177-218 221-81l-4 2c-66-94-218-61-212 77zm337-64a201 201 0 00-380-89l15 11c-13 29-19 57-19 86 6 128 100 192 194 192 94-1 191-75 190-200',
},
{
    file => 'wechat.svg',
    path => 'M402 369c23-17 38 -42 38 -70c0-51 -50 -92 -111 -92s-110 41-110 92s49 92 110 92c13 0 25-2 36 -5c4-1 8 0 9 1l25 14c3 2 6 0 5-4l-6-22c0-3 2 -5 4 -6m-110-85a15 15 0 110-29a15 15 0 010 29m74 0a15 15 0 110-29a15 15 0 010 29',
},
{
    file => 'wekan.svg',
    path => 'M372 309s5-7 6-11l1-3-6 1c-1 1-5 13-19 26-8 7-17 9-25 8-6 0-10-4-13-11-2-7-3-15-1-25 4-19 9-29 16-29 8 7-3 35-12 42l-3 3h4c10-3 18-6 26-11 7-5 12-10 14-15 1-6-1-11-6-15s-12-6-20-7c-8 0-16 1-23 5 10-19 21-38 27-57 5-17 4-30-5-38-3-3-7-5-13-6-12-3-24-7-15 6 2 2 3 5 3 7 5 31-7 56-18 79-11 22-26 54-47 66l-1-3c-1-5 0-9 1-14l7-38 8-37 3-15c1-4 0-6-1-8l-17-10c-8-5-8-5-11 3l-11 31a680 680 0 01-36 77c-6 10-11 15-15 15-4-3-2-9-1-12 1-12 5-25 7-35l9-32 7-27 6-19c0-3-2-6-5-8l-15-9c-5-3-7-2-12-1a459 459 0 00-76 37c-11 8-18 14-21 20-4 8-3 15 1 22 2 4 5 8 9 11 15 10 10 4 12-8 0-4 2-8 4-12 5-8 13-18 27-29 14-10 29-19 45-26l-9 36-9 38c-3 14-8 27-8 41 2 12 12 21 24 21 8 0 15-5 22-15a481 481 0 0036-78l-6 32-4 28c-1 8 0 16 4 22 5 7 13 10 21 11 8 0 18-7 32-20l9-11c0 10 3 17 10 23 9 6 19 10 31 10 19-1 42-11 52-26z',
},
{
    file => 'whatsapp.svg',
    path => 'M123 393l14-65a138 138 0 1150 47z',
},
{
    file => 'wifi.svg',
    path => 'M392 220a11 11 0 110-1zm-152 0a11 11 0 110-1zm-111-48a63 63 0 00-63 63v41a63 63 0 0063 63h254a63 63 0 0063-63v-41a63 63 0 00-63-63zm243 66v60h18v-60zm-79-27v87h19v-34h44v-18h-44v-18h47v-17zm-176 0l13 56l17-56h15l17 58l14-58h17l-23 87h-16l-17-60l-18 60h-15l-23-87zm121 27v60h-18v-60zm202 39a56 56 0 01-56 56H209a56 56 0 0056-56v-43a56 56 0 0156-56h63a56 56 0 0156 56zm-290 74a142 142 0 00212 0zm212-191a142 142 0 00-212 0z',
},
{
    file => 'wire.svg',
    path => 'M99 127v163a1 1 0 00186 0V157a1 1 0 00-58 0v133a1 1 0 00186 0V127',
},
);

for my $path (@paths) {
    eval {
	my @foo = extract_path_info ($path->{path});
    };
    ok (!$@, "No error parsing 'a' element of $path->{file}");
    if ($@) {
	diag ("Error $@ parsing $path->{path}");
    }
}

done_testing ();
