#!/usr/bin/perl

use strict;
use warnings;

use DateTime::Format::Natural;
use DateTime::Format::Natural::Test ':set';
use Test::More;

my @simple = (
    { 'now'                            => [ '24.11.2006 01:13:08',     unaltered ] },
    { 'yesterday'                      => [ '23.11.2006 00:00:00',     truncated ] },
    { 'today'                          => [ '24.11.2006 00:00:00',     truncated ] },
    { 'tomorrow'                       => [ '25.11.2006 00:00:00',     truncated ] },
    { 'morning'                        => [ '24.11.2006 08:00:00',     truncated ] },
    { 'afternoon'                      => [ '24.11.2006 14:00:00',     truncated ] },
    { 'evening'                        => [ '24.11.2006 20:00:00',     truncated ] },
    { 'noon'                           => [ '24.11.2006 12:00:00',     truncated ] },
    { 'midnight'                       => [ '24.11.2006 00:00:00',     truncated ] },
    { 'yesterday {at} noon'            => [ '23.11.2006 12:00:00',     truncated ] },
    { 'yesterday {at} midnight'        => [ '23.11.2006 00:00:00',     truncated ] },
    { 'today {at} noon'                => [ '24.11.2006 12:00:00',     truncated ] },
    { 'today {at} midnight'            => [ '24.11.2006 00:00:00',     truncated ] },
    { 'tomorrow {at} noon'             => [ '25.11.2006 12:00:00',     truncated ] },
    { 'tomorrow {at} midnight'         => [ '25.11.2006 00:00:00',     truncated ] },
    { 'this morning'                   => [ '24.11.2006 08:00:00',     truncated ] },
    { 'this afternoon'                 => [ '24.11.2006 14:00:00',     truncated ] },
    { 'this evening'                   => [ '24.11.2006 20:00:00',     truncated ] },
    { 'yesterday morning'              => [ '23.11.2006 08:00:00',     truncated ] },
    { 'yesterday afternoon'            => [ '23.11.2006 14:00:00',     truncated ] },
    { 'yesterday evening'              => [ '23.11.2006 20:00:00',     truncated ] },
    { 'today morning'                  => [ '24.11.2006 08:00:00',     truncated ] },
    { 'today afternoon'                => [ '24.11.2006 14:00:00',     truncated ] },
    { 'today evening'                  => [ '24.11.2006 20:00:00',     truncated ] },
    { 'tomorrow morning'               => [ '25.11.2006 08:00:00',     truncated ] },
    { 'tomorrow afternoon'             => [ '25.11.2006 14:00:00',     truncated ] },
    { 'tomorrow evening'               => [ '25.11.2006 20:00:00',     truncated ] },
    { 'thursday morning'               => [ '23.11.2006 08:00:00',     truncated ] },
    { 'thursday afternoon'             => [ '23.11.2006 14:00:00',     truncated ] },
    { 'thursday evening'               => [ '23.11.2006 20:00:00',     truncated ] },
    { '6:00{sec} yesterday'            => [ '23.11.2006 06:00:{sec}',  truncated ] },
    { '6:00{sec} today'                => [ '24.11.2006 06:00:{sec}',  truncated ] },
    { '6:00{sec} tomorrow'             => [ '25.11.2006 06:00:{sec}',  truncated ] },
    { '5{min_sec}{ }am yesterday'      => [ '23.11.2006 05:{min_sec}', truncated ] },
    { '5{min_sec}{ }am today'          => [ '24.11.2006 05:{min_sec}', truncated ] },
    { '5{min_sec}{ }am tomorrow'       => [ '25.11.2006 05:{min_sec}', truncated ] },
    { '4{min_sec}{ }pm yesterday'      => [ '23.11.2006 16:{min_sec}', truncated ] },
    { '4{min_sec}{ }pm today'          => [ '24.11.2006 16:{min_sec}', truncated ] },
    { '4{min_sec}{ }pm tomorrow'       => [ '25.11.2006 16:{min_sec}', truncated ] },
    { 'last second'                    => [ '24.11.2006 01:13:07',     truncated ] },
    { 'this second'                    => [ '24.11.2006 01:13:08',     truncated ] },
    { 'next second'                    => [ '24.11.2006 01:13:09',     truncated ] },
    { 'last minute'                    => [ '24.11.2006 01:12:00',     truncated ] },
    { 'this minute'                    => [ '24.11.2006 01:13:00',     truncated ] },
    { 'next minute'                    => [ '24.11.2006 01:14:00',     truncated ] },
    { 'last hour'                      => [ '24.11.2006 00:00:00',     truncated ] },
    { 'this hour'                      => [ '24.11.2006 01:00:00',     truncated ] },
    { 'next hour'                      => [ '24.11.2006 02:00:00',     truncated ] },
    { 'last day'                       => [ '23.11.2006 00:00:00',     truncated ] },
    { 'this day'                       => [ '24.11.2006 00:00:00',     truncated ] },
    { 'next day'                       => [ '25.11.2006 00:00:00',     truncated ] },
    { 'last week'                      => [ '17.11.2006 00:00:00',     truncated ] },
    { 'this week'                      => [ '24.11.2006 00:00:00',     truncated ] },
    { 'next week'                      => [ '01.12.2006 00:00:00',     truncated ] },
    { 'last month'                     => [ '01.10.2006 00:00:00',     truncated ] },
    { 'this month'                     => [ '01.11.2006 00:00:00',     truncated ] },
    { 'next month'                     => [ '01.12.2006 00:00:00',     truncated ] },
    { 'last quarter'                   => [ '01.07.2006 00:00:00',     truncated ] },
    { 'this quarter'                   => [ '01.10.2006 00:00:00',     truncated ] },
    { 'next quarter'                   => [ '01.01.2007 00:00:00',     truncated ] },
    { 'last year'                      => [ '01.01.2005 00:00:00',     truncated ] },
    { 'this year'                      => [ '01.01.2006 00:00:00',     truncated ] },
    { 'next year'                      => [ '01.01.2007 00:00:00',     truncated ] },
    { 'last friday'                    => [ '17.11.2006 00:00:00',     truncated ] },
    { 'this friday'                    => [ '24.11.2006 00:00:00',     truncated ] },
    { 'next friday'                    => [ '01.12.2006 00:00:00',     truncated ] },
    { 'tuesday last week'              => [ '14.11.2006 00:00:00',     truncated ] },
    { 'tuesday this week'              => [ '21.11.2006 00:00:00',     truncated ] },
    { 'tuesday next week'              => [ '28.11.2006 00:00:00',     truncated ] },
    { 'last week wednesday'            => [ '15.11.2006 00:00:00',     truncated ] },
    { 'this week wednesday'            => [ '22.11.2006 00:00:00',     truncated ] },
    { 'next week wednesday'            => [ '29.11.2006 00:00:00',     truncated ] },
    { '10 seconds ago'                 => [ '24.11.2006 01:12:58',     unaltered ] },
    { '10 minutes ago'                 => [ '24.11.2006 01:03:08',     unaltered ] },
    { '10 hours ago'                   => [ '23.11.2006 15:13:08',     unaltered ] },
    { '10 days ago'                    => [ '14.11.2006 01:13:08',     unaltered ] },
    { '10 weeks ago'                   => [ '15.09.2006 01:13:08',     unaltered ] },
    { '10 months ago'                  => [ '24.01.2006 01:13:08',     unaltered ] },
    { '10 years ago'                   => [ '24.11.1996 01:13:08',     unaltered ] },
    { 'in 5 seconds'                   => [ '24.11.2006 01:13:13',     unaltered ] },
    { 'in 5 minutes'                   => [ '24.11.2006 01:18:08',     unaltered ] },
    { 'in 5 hours'                     => [ '24.11.2006 06:13:08',     unaltered ] },
    { 'in 5 days'                      => [ '29.11.2006 01:13:08',     unaltered ] },
    { 'in 5 weeks'                     => [ '29.12.2006 01:13:08',     unaltered ] },
    { 'in 5 months'                    => [ '24.04.2007 01:13:08',     unaltered ] },
    { 'in 5 years'                     => [ '24.11.2011 01:13:08',     unaltered ] },
    { 'saturday'                       => [ '25.11.2006 00:00:00',     truncated ] },
    { 'yesterday {at} 4:00{sec}'       => [ '23.11.2006 04:00:{sec}',  truncated ] },
    { 'today {at} 4:00{sec}'           => [ '24.11.2006 04:00:{sec}',  truncated ] },
    { 'tomorrow {at} 4:00{sec}'        => [ '25.11.2006 04:00:{sec}',  truncated ] },
    { 'yesterday {at} 6:45{sec}{ }am'  => [ '23.11.2006 06:45:{sec}',  truncated ] },
    { 'today {at} 6:45{sec}{ }am'      => [ '24.11.2006 06:45:{sec}',  truncated ] },
    { 'tomorrow {at} 6:45{sec}{ }am'   => [ '25.11.2006 06:45:{sec}',  truncated ] },
    { 'yesterday {at} 6:45{sec}{ }pm'  => [ '23.11.2006 18:45:{sec}',  truncated ] },
    { 'today {at} 6:45{sec}{ }pm'      => [ '24.11.2006 18:45:{sec}',  truncated ] },
    { 'tomorrow {at} 6:45{sec}{ }pm'   => [ '25.11.2006 18:45:{sec}',  truncated ] },
    { 'wednesday {at} 14:30{sec}'      => [ '22.11.2006 14:30:{sec}',  truncated ] },
    { 'wednesday {at} 02:30{sec}{ }am' => [ '22.11.2006 02:30:{sec}',  truncated ] },
    { 'wednesday {at} 02:30{sec}{ }pm' => [ '22.11.2006 14:30:{sec}',  truncated ] },
    { '2nd monday'                     => [ '13.11.2006 00:00:00',     truncated ] },
    { '100th day'                      => [ '10.04.2006 00:00:00',     truncated ] },
    { '4th february'                   => [ '04.02.2006 00:00:00',     truncated ] },
    { 'november 3rd'                   => [ '03.11.2006 00:00:00',     truncated ] },
    { 'last june'                      => [ '01.06.2005 00:00:00',     truncated ] },
    { 'next october'                   => [ '01.10.2007 00:00:00',     truncated ] },
    { '5{min_sec}{ }am'                => [ '24.11.2006 05:{min_sec}', truncated ] },
    { '5:30{sec}{ }am'                 => [ '24.11.2006 05:30:{sec}',  truncated ] },
    { '4{min_sec}{ }pm'                => [ '24.11.2006 16:{min_sec}', truncated ] },
    { '4:20{sec}{ }pm'                 => [ '24.11.2006 16:20:{sec}',  truncated ] },
    { '06:56:06{ }am'                  => [ '24.11.2006 06:56:06',     truncated ] },
    { '06.56.06{ }am'                  => [ '24.11.2006 06:56:06',     truncated ] },
    { '06.56.06{ }pm'                  => [ '24.11.2006 18:56:06',     truncated ] },
    { '06:56:06{ }pm'                  => [ '24.11.2006 18:56:06',     truncated ] },
    { 'mon 2:35{sec}'                  => [ '20.11.2006 02:35:{sec}',  truncated ] },
    { '1:00{sec} sun'                  => [ '26.11.2006 01:00:{sec}',  truncated ] },
    { '1{min_sec}{ }am sun'            => [ '26.11.2006 01:{min_sec}', truncated ] },
    { '1{min_sec}{ }pm sun'            => [ '26.11.2006 13:{min_sec}', truncated ] },
    { '1:00{sec} on sun'               => [ '26.11.2006 01:00:{sec}',  truncated ] },
    { '1{min_sec}{ }am on sun'         => [ '26.11.2006 01:{min_sec}', truncated ] },
    { '1{min_sec}{ }pm on sun'         => [ '26.11.2006 13:{min_sec}', truncated ] },
    { '12:14{sec}{ }PM'                => [ '24.11.2006 12:14:{sec}',  truncated ] },
    { '12:14{sec}{ }P.M.'              => [ '24.11.2006 12:14:{sec}',  truncated ] },
    { '12:14{sec}{ }P.M'               => [ '24.11.2006 12:14:{sec}',  truncated ] },
    { '12:14{sec}{ }AM'                => [ '24.11.2006 00:14:{sec}',  truncated ] },
    { '12:14{sec}{ }A.M.'              => [ '24.11.2006 00:14:{sec}',  truncated ] },
    { '12:14{sec}{ }A.M'               => [ '24.11.2006 00:14:{sec}',  truncated ] },

);

my @complex = (
    { 'yesterday 7 seconds ago'                    => [ '23.11.2006 01:13:01',     unaltered ] },
    { 'yesterday 7 minutes ago'                    => [ '23.11.2006 01:06:08',     unaltered ] },
    { 'yesterday 7 hours ago'                      => [ '22.11.2006 18:13:08',     unaltered ] },
    { 'yesterday 7 days ago'                       => [ '16.11.2006 00:00:00',     truncated ] },
    { 'yesterday 7 weeks ago'                      => [ '05.10.2006 00:00:00',     truncated ] },
    { 'yesterday 7 months ago'                     => [ '23.04.2006 00:00:00',     truncated ] },
    { 'yesterday 7 years ago'                      => [ '23.11.1999 00:00:00',     truncated ] },
    { 'today 5 seconds ago'                        => [ '24.11.2006 01:13:03',     unaltered ] },
    { 'today 5 minutes ago'                        => [ '24.11.2006 01:08:08',     unaltered ] },
    { 'today 5 hours ago'                          => [ '23.11.2006 20:13:08',     unaltered ] },
    { 'today 5 days ago'                           => [ '19.11.2006 00:00:00',     truncated ] },
    { 'today 5 weeks ago'                          => [ '20.10.2006 00:00:00',     truncated ] },
    { 'today 5 months ago'                         => [ '24.06.2006 00:00:00',     truncated ] },
    { 'today 5 years ago'                          => [ '24.11.2001 00:00:00',     truncated ] },
    { 'tomorrow 3 seconds ago'                     => [ '25.11.2006 01:13:05',     unaltered ] },
    { 'tomorrow 3 minutes ago'                     => [ '25.11.2006 01:10:08',     unaltered ] },
    { 'tomorrow 3 hours ago'                       => [ '24.11.2006 22:13:08',     unaltered ] },
    { 'tomorrow 3 days ago'                        => [ '22.11.2006 00:00:00',     truncated ] },
    { 'tomorrow 3 weeks ago'                       => [ '04.11.2006 00:00:00',     truncated ] },
    { 'tomorrow 3 months ago'                      => [ '25.08.2006 00:00:00',     truncated ] },
    { 'tomorrow 3 years ago'                       => [ '25.11.2003 00:00:00',     truncated ] },
    { '2 seconds before now'                       => [ '24.11.2006 01:13:06',     unaltered ] },
    { '2 minutes before now'                       => [ '24.11.2006 01:11:08',     unaltered ] },
    { '2 hours before now'                         => [ '23.11.2006 23:13:08',     unaltered ] },
    { '2 days before now'                          => [ '22.11.2006 01:13:08',     unaltered ] },
    { '2 weeks before now'                         => [ '10.11.2006 01:13:08',     unaltered ] },
    { '2 months before now'                        => [ '24.09.2006 01:13:08',     unaltered ] },
    { '2 years before now'                         => [ '24.11.2004 01:13:08',     unaltered ] },
    { '4 seconds from now'                         => [ '24.11.2006 01:13:12',     unaltered ] },
    { '4 minutes from now'                         => [ '24.11.2006 01:17:08',     unaltered ] },
    { '4 hours from now'                           => [ '24.11.2006 05:13:08',     unaltered ] },
    { '4 days from now'                            => [ '28.11.2006 01:13:08',     unaltered ] },
    { '4 weeks from now'                           => [ '22.12.2006 01:13:08',     unaltered ] },
    { '4 months from now'                          => [ '24.03.2007 01:13:08',     unaltered ] },
    { '4 years from now'                           => [ '24.11.2010 01:13:08',     unaltered ] },
    { '6 in the morning'                           => [ '24.11.2006 06:00:00',     truncated ] },
    { '4 in the afternoon'                         => [ '24.11.2006 16:00:00',     truncated ] },
    { '9 in the evening'                           => [ '24.11.2006 21:00:00',     truncated ] },
    { 'monday 6 in the morning'                    => [ '20.11.2006 06:00:00',     truncated ] },
    { 'monday 4 in the afternoon'                  => [ '20.11.2006 16:00:00',     truncated ] },
    { 'monday 9 in the evening'                    => [ '20.11.2006 21:00:00',     truncated ] },
    { 'monday last week'                           => [ '13.11.2006 00:00:00',     truncated ] },
    { '6th day last week'                          => [ '18.11.2006 00:00:00',     truncated ] },
    { '6th day this week'                          => [ '25.11.2006 00:00:00',     truncated ] },
    { '6th day next week'                          => [ '02.12.2006 00:00:00',     truncated ] },
    { '12th day last month'                        => [ '12.10.2006 00:00:00',     truncated ] },
    { '12th day this month'                        => [ '12.11.2006 00:00:00',     truncated ] },
    { '12th day next month'                        => [ '12.12.2006 00:00:00',     truncated ] },
    { '1st day last year'                          => [ '01.01.2005 00:00:00',     truncated ] },
    { '1st day this year'                          => [ '01.01.2006 00:00:00',     truncated ] },
    { '1st day next year'                          => [ '01.01.2007 00:00:00',     truncated ] },
    { '1st tuesday last november'                  => [ '01.11.2005 00:00:00',     truncated ] },
    { '1st tuesday this november'                  => [ '07.11.2006 00:00:00',     truncated ] },
    { '1st tuesday next november'                  => [ '06.11.2007 00:00:00',     truncated ] },
    { '11 january next year'                       => [ '11.01.2007 00:00:00',     truncated ] },
    { '11 january this year'                       => [ '11.01.2006 00:00:00',     truncated ] },
    { '11 january last year'                       => [ '11.01.2005 00:00:00',     truncated ] },
    { '6 seconds before yesterday'                 => [ '22.11.2006 23:59:54',     truncated ] },
    { '6 minutes before yesterday'                 => [ '22.11.2006 23:54:00',     truncated ] },
    { '6 hours before yesterday'                   => [ '22.11.2006 18:00:00',     truncated ] },
    { '6 seconds before today'                     => [ '23.11.2006 23:59:54',     truncated ] },
    { '6 minutes before today'                     => [ '23.11.2006 23:54:00',     truncated ] },
    { '6 hours before today'                       => [ '23.11.2006 18:00:00',     truncated ] },
    { '6 seconds before tomorrow'                  => [ '24.11.2006 23:59:54',     truncated ] },
    { '6 minutes before tomorrow'                  => [ '24.11.2006 23:54:00',     truncated ] },
    { '6 hours before tomorrow'                    => [ '24.11.2006 18:00:00',     truncated ] },
    { '3 seconds after yesterday'                  => [ '23.11.2006 00:00:03',     truncated ] },
    { '3 minutes after yesterday'                  => [ '23.11.2006 00:03:00',     truncated ] },
    { '3 hours after yesterday'                    => [ '23.11.2006 03:00:00',     truncated ] },
    { '3 seconds after today'                      => [ '24.11.2006 00:00:03',     truncated ] },
    { '3 minutes after today'                      => [ '24.11.2006 00:03:00',     truncated ] },
    { '3 hours after today'                        => [ '24.11.2006 03:00:00',     truncated ] },
    { '3 seconds after tomorrow'                   => [ '25.11.2006 00:00:03',     truncated ] },
    { '3 minutes after tomorrow'                   => [ '25.11.2006 00:03:00',     truncated ] },
    { '3 hours after tomorrow'                     => [ '25.11.2006 03:00:00',     truncated ] },
    { '10 seconds before noon'                     => [ '24.11.2006 11:59:50',     truncated ] },
    { '10 minutes before noon'                     => [ '24.11.2006 11:50:00',     truncated ] },
    { '10 hours before noon'                       => [ '24.11.2006 02:00:00',     truncated ] },
    { '10 seconds before midnight'                 => [ '23.11.2006 23:59:50',     truncated ] },
    { '10 minutes before midnight'                 => [ '23.11.2006 23:50:00',     truncated ] },
    { '10 hours before midnight'                   => [ '23.11.2006 14:00:00',     truncated ] },
    { '5 seconds after noon'                       => [ '24.11.2006 12:00:05',     truncated ] },
    { '5 minutes after noon'                       => [ '24.11.2006 12:05:00',     truncated ] },
    { '5 hours after noon'                         => [ '24.11.2006 17:00:00',     truncated ] },
    { '5 seconds after midnight'                   => [ '24.11.2006 00:00:05',     truncated ] },
    { '5 minutes after midnight'                   => [ '24.11.2006 00:05:00',     truncated ] },
    { '5 hours after midnight'                     => [ '24.11.2006 05:00:00',     truncated ] },
    { 'noon last friday'                           => [ '17.11.2006 12:00:00',     truncated ] },
    { 'midnight last friday'                       => [ '17.11.2006 00:00:00',     truncated ] },
    { 'noon this friday'                           => [ '24.11.2006 12:00:00',     truncated ] },
    { 'midnight this friday'                       => [ '24.11.2006 00:00:00',     truncated ] },
    { 'noon next friday'                           => [ '01.12.2006 12:00:00',     truncated ] },
    { 'midnight next friday'                       => [ '01.12.2006 00:00:00',     truncated ] },
    { 'last friday {at} 20:00{sec}'                => [ '17.11.2006 20:00:{sec}',  truncated ] },
    { 'this friday {at} 20:00{sec}'                => [ '24.11.2006 20:00:{sec}',  truncated ] },
    { 'next friday {at} 20:00{sec}'                => [ '01.12.2006 20:00:{sec}',  truncated ] },
    { '1:00{sec} last friday'                      => [ '17.11.2006 01:00:{sec}',  truncated ] },
    { '1:00{sec} this friday'                      => [ '24.11.2006 01:00:{sec}',  truncated ] },
    { '1:00{sec} next friday'                      => [ '01.12.2006 01:00:{sec}',  truncated ] },
    { '1{min_sec}{ }am last friday'                => [ '17.11.2006 01:{min_sec}', truncated ] },
    { '1{min_sec}{ }am this friday'                => [ '24.11.2006 01:{min_sec}', truncated ] },
    { '1{min_sec}{ }am next friday'                => [ '01.12.2006 01:{min_sec}', truncated ] },
    { '1{min_sec}{ }pm last friday'                => [ '17.11.2006 13:{min_sec}', truncated ] },
    { '1{min_sec}{ }pm this friday'                => [ '24.11.2006 13:{min_sec}', truncated ] },
    { '1{min_sec}{ }pm next friday'                => [ '01.12.2006 13:{min_sec}', truncated ] },
    { 'last wednesday {at} 7{min_sec}{ }am'        => [ '15.11.2006 07:{min_sec}', truncated ] },
    { 'this wednesday {at} 7{min_sec}{ }am'        => [ '22.11.2006 07:{min_sec}', truncated ] },
    { 'next wednesday {at} 7{min_sec}{ }am'        => [ '29.11.2006 07:{min_sec}', truncated ] },
    { 'last wednesday {at} 7{min_sec}{ }pm'        => [ '15.11.2006 19:{min_sec}', truncated ] },
    { 'this wednesday {at} 7{min_sec}{ }pm'        => [ '22.11.2006 19:{min_sec}', truncated ] },
    { 'next wednesday {at} 7{min_sec}{ }pm'        => [ '29.11.2006 19:{min_sec}', truncated ] },
    { '2nd friday in august'                       => [ '11.08.2006 00:00:00',     truncated ] },
    { '3rd wednesday in november'                  => [ '15.11.2006 00:00:00',     truncated ] },
    { 'tomorrow 1 year ago'                        => [ '25.11.2005 00:00:00',     truncated ] },
    { 'saturday 3 months ago {at} 17:00{sec}'      => [ '26.08.2006 17:00:{sec}',  truncated ] },
    { 'saturday 3 months ago {at} 5:00{sec}{ }am'  => [ '26.08.2006 05:00:{sec}',  truncated ] },
    { 'saturday 3 months ago {at} 5:00{sec}{ }pm'  => [ '26.08.2006 17:00:{sec}',  truncated ] },
    { '11 january 2 years ago'                     => [ '11.01.2004 00:00:00',     truncated ] },
    { '4th day last week'                          => [ '16.11.2006 00:00:00',     truncated ] },
    { '8th month last year'                        => [ '01.08.2005 00:00:00',     truncated ] },
    { '8th month this year'                        => [ '01.08.2006 00:00:00',     truncated ] },
    { '8th month next year'                        => [ '01.08.2007 00:00:00',     truncated ] },
    { '6 mondays from now'                         => [ '01.01.2007 00:00:00',     truncated ] },
    { 'fri 3 months ago at 5{min_sec}{ }am'        => [ '25.08.2006 05:{min_sec}', truncated ] },
    { 'wednesday 1 month ago at 8{min_sec}{ }pm'   => [ '25.10.2006 20:{min_sec}', truncated ] },
    { 'final thursday in april'                    => [ '27.04.2006 00:00:00',     truncated ] },
    { 'final sunday in april'                      => [ '30.04.2006 00:00:00',     truncated ] }, # edge case
    { 'last thursday in april'                     => [ '27.04.2006 00:00:00',     truncated ] },
    { 'beginning of last month'                    => [ '01.10.2006 00:00:00',     truncated ] },
    { 'end of last month'                          => [ '31.10.2006 00:00:00',     truncated ] },
);

my @specific = (
    { 'march'                              => [ '01.03.2006 00:00:00',     truncated ] },
    { 'january 11'                         => [ '11.01.2006 00:00:00',     truncated ] },
    { '11 january'                         => [ '11.01.2006 00:00:00',     truncated ] },
    { '18 oct {at} 17:00{sec}'             => [ '18.10.2006 17:00:{sec}',  truncated ] },
    { '18 oct 2001 {at} 17:00{sec}'        => [ '18.10.2001 17:00:{sec}',  truncated ] },
    { '18 oct {at} 5{min_sec}{ }am'        => [ '18.10.2006 05:{min_sec}', truncated ] },
    { '18 oct {at} 5{min_sec}{ }pm'        => [ '18.10.2006 17:{min_sec}', truncated ] },
    { 'dec 25'                             => [ '25.12.2006 00:00:00',     truncated ] },
    { 'feb 28 {at} 3:00{sec}'              => [ '28.02.2006 03:00:{sec}',  truncated ] },
    { 'feb 28 {at} 3.00 pm'                => [ '28.02.2006 15:00:00',     truncated ] },
    { 'feb 28 {at} 3.00 p.m.'              => [ '28.02.2006 15:00:00',     truncated ] },
    { 'feb 28 {at} 3.00.15 a.m'            => [ '28.02.2006 03:00:15',     truncated ] },
    { 'feb 28 2001 {at} 3:00{sec}'         => [ '28.02.2001 03:00:{sec}',  truncated ] },
    { 'feb 28 {at} 3{min_sec}{ }am'        => [ '28.02.2006 03:{min_sec}', truncated ] },
    { 'feb 28 {at} 3{min_sec}{ }pm'        => [ '28.02.2006 15:{min_sec}', truncated ] },
    { '19:00{sec} jul 1'                   => [ '01.07.2006 19:00:{sec}',  truncated ] },
    { '7{min_sec}{ }am jul 1'              => [ '01.07.2006 07:{min_sec}', truncated ] },
    { '7{min_sec}{ }pm jul 1'              => [ '01.07.2006 19:{min_sec}', truncated ] },
    { 'jan 24, 2011 {at} 12:00{sec}'       => [ '24.01.2011 12:00:{sec}',  truncated ] },
    { 'jan 24, 2011 {at} 12{min_sec}{ }am' => [ '24.01.2011 00:{min_sec}', truncated ] },
    { 'jan 24, 2011 {at} 12{min_sec}{ }pm' => [ '24.01.2011 12:{min_sec}', truncated ] },
    { 'may 27th'                           => [ '27.05.2006 00:00:00',     truncated ] },
  # { '2005'                               => [ '01.01.2005 00:00:00',     truncated ] },
    { 'march 1st 2009'                     => [ '01.03.2009 00:00:00',     truncated ] },
    { 'October 2006'                       => [ '01.10.2006 00:00:00',     truncated ] },
    { 'february 14, 2004'                  => [ '14.02.2004 00:00:00',     truncated ] },
    { 'jan 3 2010'                         => [ '03.01.2010 00:00:00',     truncated ] },
    { 'jan 3 2010 {at} 17:23'              => [ '03.01.2010 17:23:00',     truncated ] },
    { 'jan 3 2010 {at} 5{min_sec}{ }pm'    => [ '03.01.2010 17:{min_sec}', truncated ] },
    { '3 jan 2000'                         => [ '03.01.2000 00:00:00',     truncated ] },
    { '3 jan 2000 {at} 03:02'              => [ '03.01.2000 03:02:00',     truncated ] },
    { '3 jan 2000 {at} 3{min_sec}{ }am'    => [ '03.01.2000 03:{min_sec}', truncated ] },
    { '2010 october 28'                    => [ '28.10.2010 00:00:00',     truncated ] },
    { '27/5/1979'                          => [ '27.05.1979 00:00:00',     truncated ] },
    { '6'                                  => [ '24.11.2006 06:00:00',     truncated ] },
    { '4:00'                               => [ '24.11.2006 04:00:00',     truncated ] },
    { '17:00'                              => [ '24.11.2006 17:00:00',     truncated ] },
    { '17.00'                              => [ '24.11.2006 17:00:00',     truncated ] },
    { 'Nov 24th 2006 {at} 17.00.00'        => [ '24.11.2006 17:00:00',     truncated ] },
    { '3:20:00'                            => [ '24.11.2006 03:20:00',     truncated ] },
    { '-5min'                              => [ '24.11.2006 01:08:08',     unaltered ] },
    { '+2d'                                => [ '26.11.2006 01:13:08',     unaltered ] },
);

_run_tests(720, [ [ \@simple ], [ \@complex ], [ \@specific ] ], \&compare);

sub compare
{
    my $aref = shift;

    foreach my $href (@$aref) {
        my $key = (keys %$href)[0];
        foreach my $entry ($time_entries->($key, $href->{$key})) {
            foreach my $string ($case_strings->($entry->[0])) {
                compare_strings($string, $entry->[1]);
            }
        }
    }
}

sub compare_strings
{
    my ($string, $result) = @_;

    my $parser = DateTime::Format::Natural->new;
    $parser->_set_datetime(\%time);

    my $dt = $parser->parse_datetime($string);

    if ($parser->success && $parser->_get_truncated == $result->[1]) {
        is(_result_string($dt), $result->[0], _message($string));
    }
    else {
        fail(_message($string));
    }
}
