/*
 * Copyright (C) 1990-1998 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)tpread.c	2.78 06/04/98 CERN IT-PDP/DM Antoine Trannoy";
#endif /* not lint */

/* tpread.c             SHIFT remote tape copy client command.          */

#include <stdio.h>
#include <malloc.h>
#if ! defined(apollo) 
#include <unistd.h>
#endif 	/* ! apollo */
#include "rtcopy.h"

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/stat.h>

#if defined(apollo)
#include <strings.h>
#else
#include <string.h>
#endif	/* apollo */

#include <pwd.h>
#include <grp.h>
#if defined(_AIX) && defined(_IBMR2)
#include <sys/select.h>
#endif /* _AIX */
#if defined(SOLARIS)
#include <sys/filio.h>
#endif

/*
 * Process handling.
 */
#include <signal.h>		/* Signal handling	*/

#if defined(apollo)
#define SIG_ERR (void(*)())-1
#endif  /* apollo */

/*
 * Networking.
 */
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <marshall.h>
/*
 * TRACE specification
 */
#include <trace.h>

extern int (*recvfunc)() ;
extern int (*sendfunc)() ;
extern int setnetio() ; 
#define netread		(*recvfunc)
#define netwrite	(*sendfunc)
#define KEYPORT (WORD) ( 10000 + getpid()%55500 )
#define MAXFILENAMSIZE 1024
#define RFIO_NETOPT 	2
#define RFIO_NONET	1
#define UPPERCASE(a) \
{char *_c; \
  for (_c=a; *_c != '\0'; _c++)  *_c=toupper(*_c); \
}

extern char * getconfent() ;
extern int solveln() ;

/* 
 * Error and message handling.
 */
#include <errno.h>
#include <serrno.h>
#include <log.h>

extern int errno ; 
extern int sys_nerr ;
#ifndef linux
extern char *sys_errlist[] ;
#endif /* linux */
extern int rfio_parse() ;
extern int rfio_mstat() ;

/*
 * Variable errfault is used to known
 * in the main() function who is responsible
 * for the error. It can take the following 
 * values: USERR, SYERR, UNERR, SEERR.
 */
static int errfault= UNERR ; 

/*
 * Buffer used to build, send and receive 
 * the messages.
 */
static char buffer[SYCONFLIM] ;

/*
 * Name of the command.
 * tpread() ot tpwrite() ?
 */
static char * command ; 

/*
 * Is the staging process 
 * done for the group ?
 */
static int Gopt = 0 ; 
static int Guid ;
static char Gname[10] ;
/*
 * For option info
 */
static int alg_info = 0 ; 
static int istpselect = 0 ;
static int isout    = 0 ; /* Is request for an out of site server ? */
static char serversite[10]= "";
static int nfile = 0 ;         /* Number of disk files */
static char **dirpath = NULL ;

/*
 * Setting flags for the G option.
 */
static int setGopt() 
{
	struct passwd * pw ;
	struct group  * gr ;
	char * cp ; 
	extern char * getconfent() ;

	Gopt ++ ;
	if ( (pw= getpwuid(geteuid())) == NULL ) {
		log(LOG_ERR,"! invalid user: %d\n",geteuid()) ;
		errfault= SYERR ;
		return -1 ;
	}
	if ( (gr= getgrgid(getegid())) == NULL ) {
		log(LOG_ERR,"! invalid group: %d\n",getegid()) ;
		errfault= SYERR ;
		return -1 ;
	}
	if ( (cp= getconfent("GRPUSER",gr->gr_name,0)) == NULL ) {
		log(LOG_ERR,"! Option G not allowed for user of group %s\n",gr->gr_name) ;
		errfault= USERR ;
		return -1 ;
	}
	(void) strcpy(Gname,cp) ;
	if ( (pw= getpwnam(Gname)) == NULL ) {		
		log(LOG_ERR,"! invalid user: %d\n",Gname) ;
		errfault= SYERR ;
		return -1 ;
	}
	Guid= pw->pw_uid ; 
	return 0 ;
}

/*
 * SIGINT interrupt  handler.
 */
static int ABORT= 0 ; 
void actionSIGINT()
{
	ABORT ++ ;
#if defined(sgi)
	signal(SIGINT,(void (*)())actionSIGINT) ; 
#endif	/* sgi	*/
}

/*
 * Receive message in buffer.
 *
 * Return code is:
 *   -1 in case of an error. The socket is closed.
 *   RQST ot ACKN type otherwise. 
 */
static int rcvmsg(sock) 
	int sock ; 
{
	int msglen ;
	int reqtyp ;
	int  rcode ;
	int  magic ;
	char * ptr ;

	/*
	 * Getting header message.
	 */
	if ( (rcode= netread(sock,buffer,LONGSIZE*3)) == -1 ) { 
		log(LOG_ERR,"! netread(): %s\n",sys_errlist[errno]) ; 
		(void) close(sock) ; 
		errfault= SYERR ;
		return -1 ;
	}
	if ( rcode == 0 ) {
		log(LOG_ERR,"! connection closed by remote end\n") ; 
		(void) close(sock) ; 
		errfault= SYERR ;
 		return -1 ; 
	}
	ptr= buffer ; 
	unmarshall_LONG(ptr,magic) ; 
	unmarshall_LONG(ptr,reqtyp) ; 
	unmarshall_LONG(ptr,msglen) ; 
	if ( magic != RTCOPY_MAGIC ) { 
		serrno= SEBADVERSION ;
		log(LOG_ERR,"! %s\n",sstrerror(serrno)) ; 
		(void) close(sock) ; 
		errfault= SEERR ;
		return -1 ; 
	}

	/*
	 * Returns, if no body, the message type.
	 */
	if ( msglen == 0 ) 
		return reqtyp ; 

	/* 
	 * Getting message body.
	 */
	if ( (rcode= netread(sock,buffer,msglen)) == -1 ) { 
		log(LOG_ERR,"! netread(): %s\n",sys_errlist[errno]) ; 
		(void) close(sock) ; 
		errfault= SYERR ;
		return -1 ;
	}
	if ( rcode == 0 ) {
		log(LOG_ERR,"! connection closed by server\n") ; 
		(void) close(sock) ; 
		errfault= SYERR ;
 		return -1 ; 
	}
	/*
	 * No problem,
	 * sending back message type.
	 */
	log(LOG_DEBUG,"rcvmsg(): returning request type %d\n", reqtyp) ;
	return reqtyp; 
}

/*
 * Finding out the type of device needed if specified. 
 */
static int getdevtyp(argc,argv,device,host)
	int    	 argc ;
	char  ** argv ; 
	char * device ;
	char *   host ; 
{
	extern char   * optarg ; 	
	extern int      optind ; 
	int 		     c ;
	struct    stat statbuf ;
	char filename[MAXFILENAMSIZE] ;
	char 	       fopt[5] ;
	char   tms_device[10] ; /* device selected by tms */
	int		    i ; 
        char            *uvsn = NULL ;
        char            *uvid = NULL ;
        char            *cp   ;
        char        buff[100] ;
	int 	  stgflag = 0 ;
	char	     copt[16] ;
#if defined (TMS)
        char          req[80] ; /* Request command      */
        int            reqlen ; /* Request length       */
        char        info[132] ; /* Info buffer          */
        int           infolen ; /* Info length          */
        extern int   sysreq() ; /* External declaration */
        extern char *  getconfent() ;
	char 	      *server ;
        char 	     *ptr[10] ;
        int 	n ,rcode ,ok=0;
        int 	 RETRY, SLEEP ;
#endif

	/*
	 * Initialization
	 */
	optind= 1 ; 
	*device = '\0' ;
	*host= '\0' ; 
	*fopt= '\0' ;
	*tms_device = '\0' ;

	/*
	 * Parsing options to eventually 
	 * get device type needed.
	 */
	while((c= getopt(argc,argv,"A:E:xF:GL:N:S:V:b:d:f:g:i:l:noq:t:v:IR:X:Z:Ts:c:C:")) != EOF) {
		switch(c) {
			case 'A':
				break;
			case 'F':
				(void) strcpy(fopt,optarg) ;
				break ; 
			case 'G':
				if ( setGopt() == -1 ) {
					return -1 ; 
				}
				break ; 
			case 'L':
			case 'N':
				break ; 
			case 'S':
				(void) strcpy(host,optarg) ; 
				break ; 
			case 'V':
                               	uvid = (char *) malloc(strlen(optarg)+1) ;
                                (void) strcpy(uvid,optarg) ;
				break ; 
			case 'b':
			case 'd':
			case 'f':
				break ;
			case 'Z':	/* stager's signal */
				stgflag ++ ;
				break ; 
			case 'g':
				(void) strcpy(device,optarg) ; 
				break ; 
			case 'i':
			case 'l':
			case 'n':
			case 'o':
			case 'q':
			case 't':
				break ; 
			case 'v':
                                uvsn = (char *) malloc(strlen(optarg)+1) ;
                                (void) strcpy(uvsn,optarg) ; 
				break ; 
			case 'I':
				alg_info++;
				break;
			case 'R':
				(void) strcpy(serversite,optarg);
				for (i=0;i< (int)strlen(optarg);i++)
					serversite[i]= isupper(serversite[i]) ? tolower(serversite[i]):serversite[i] ;
				isout ++;
				break ;
			case 'x':
				(void) initlog(command,LOG_DEBUG,"") ;
				break ;
			case 'X':break ;
			case 'T':break ;
			case 's':break ;
			case 'E':break ;
			case 'c':
				strcpy (copt, optarg) ;
				break ;
			case 'C':break ;
			default:
				log(LOG_ERR,"! '%c' is not a valid option\n",c) ; 
				errfault= USERR ;
				return -1 ; 
		}
	}

	nfile = argc-optind ;
   if (!alg_info && !stgflag) {
	/* 
	 * Check the c options 
	 */
	if ( !strcmp(copt, "off") || !strcmp(copt,"OFF") && !stgflag ) {
		log(LOG_ERR," The 'c' option set to OFF is valid with the stager only !\n");
		errfault= USERR ;
                return -1 ;
        }

	/*
	 * vsn or vid have to be specified.
	 */
	if ( uvid == NULL && uvsn == NULL ) {
		log(LOG_ERR,"! vid or vsn must be specified\n") ; 
		errfault= USERR ;
		return -1 ; 
	}


	/* Number of files specified */
	if ( nfile <= 0 ) {
		log(LOG_ERR,"! Disk files have not been specified\n");
		errfault= USERR ;
		return -1 ;
	}

	/* 
	  Dynamically allocate the dirpath array
	 */
	if (dirpath == NULL) {
		dirpath = (char **)malloc(sizeof(char *) * nfile);
		if (dirpath == NULL) {
		  log(LOG_ERR,"! malloc() failed allocating %d bytes\n",sizeof(char *) * nfile) ;
		  log(LOG_INFO,"command failed\n\n") ;
		  exit(SYERR) ;
		}
	}
	/*
	 * checking that the file exists &
	 * contains data in case of a tpwrite
	 */
	for ( i=optind ; i< argc; i++ ) {
		(void) strcpy(filename,argv[i]);
	        /* Filename is a dot, in this case this means that the filename is not given
		   and has to be allocated by the stager itself (deferred space allocation) 
		   In this case, the tpread client doesn't change this special filename  */
		if (strcmp(filename,".") != 0) { 
			/*
			 * File name is relative
			 */
			if ( filename[0] != '/' && strstr(filename,":/")==NULL ) {
				char *cp2 ;
				dirpath[i-optind] = (char *) malloc( MAXFILENAMSIZE ) ;
				if ( dirpath[i-optind] == NULL ) {
					log(LOG_ERR,"! malloc() failed\n") ;
					return -1 ;
				}
				getcwd( dirpath[i-optind], MAXFILENAMSIZE ) ;
				strcat( dirpath[i-optind] , "/" ) ;
				if ( (cp2 = strchr(filename,':')) != NULL) 
					strcat ( dirpath[i-optind], cp2+1 ) ;
				else
					strcat ( dirpath[i-optind], filename ) ;
				argv[i] = dirpath[i-optind] ;
				log(LOG_DEBUG, "File name %s changed into: %s\n", filename, dirpath[i-optind] ) ;
				strcpy( filename , dirpath[i-optind]) ;
        		}
		}

		/*
 		 * Setting rfio option not to read the NET entry 
		 * in configuration file
		 */
		c = RFIO_NONET ;
		if ( rfiosetopt(RFIO_NETOPT, &c , 4) < 0 ) {
				log(LOG_INFO,"! Warning: Could not set rfio option properly !\n") ;
		}
		if ((strcmp(filename,".") != 0) && (strstr(argv[0],"tpwrite") != NULL)) {
			if ( rfio_mstat(filename,&statbuf) < 0 ) {
				log(LOG_ERR,"! File %s does not exist or is unreachable\n",filename);
				errfault= USERR ;
                		return -1 ;
			}
			if ( S_ISDIR(statbuf.st_mode) ) {
				log(LOG_ERR,"! File %s is a directory !\n",filename);
				errfault= USERR ;
				return -1 ;
			}
			if (statbuf.st_size == 0) {
				log(LOG_ERR,"! File %s is empty !\n",filename);
				errfault= USERR ;
                        	return -1 ;
                	}
		}
		if ((strcmp(filename,".") != 0) && (strstr(argv[0],"tpread") != NULL)) {
			if ( rfio_mstat(filename,&statbuf) >= 0 && S_ISDIR(statbuf.st_mode) ) {
				log(LOG_ERR,"! File %s is a directory !\n",filename);
				errfault= USERR ;
				return -1 ;
			}
		}
	} /* for */
	rfio_end() ;
    } /* !alg_info && !stgflag */
 

    if ( ( !alg_info || ( alg_info && (uvid != NULL || uvsn != NULL))) ) {
#if defined(TMS)
      
	if (!isout) {
                /*
                 * If TMS is installed, the device
                 * type of the tape is retrieved.
                 */

                if ( uvid != NULL ) 
			strcpy(buff,uvid) ;
		else {
			strcpy(buff,uvsn) ;
			uvid = uvsn ;
		}

                for (cp = strtok(buff,":"),i=0 ; cp!= NULL ; i++ ,cp=strtok(NULL,":")) {
                        ptr[i] = (char *)malloc(20) ;
                        strcpy(ptr[i], cp);
                }
                for (n=0;n<i;n++) {
                        (void) sprintf(req,"VIDMAP %s QUERY (GENERIC SHIFT MESSAGE",ptr[n]) ;
                        reqlen= strlen(req) ;
                        ok = 0 ;
                        while (!ok) {
                           infolen= 132 ;
                           rcode = sysreq("TMS",req,&reqlen,info,&infolen) ;
                           switch ( rcode ) {
                                case 0 :
				        chkvidmap(info,device,tms_device);
                                        ok ++ ;
                                        break ;
                                case 8 :
                                        log(LOG_ERR,"! volume ID %s has wrong syntax\n",ptr[n]) ;
                                        errfault= USERR ;
                                        return -1 ;
                                        break ;
                                case 100 :
                                        log(LOG_ERR,"! volume %s does not exist\n",ptr[n]) ;
                                        errfault= USERR ;
                                        return -1 ;
                                        break ;
                                case 312 :
                                        log(LOG_ERR,"! volume ID %s is unavailable\n",ptr[n]);
                                        errfault= USERR ;
                                        return -1 ;
                                        break ;
                                case 313 :
                                        log(LOG_ERR,"! tms: Unknown generic system SHIFT\n") ;
                                        errfault= USERR ;
                                        return -1 ;
                                        break ;
                                case 314 :
                                        log(LOG_ERR,"! tms: Unknown system\n");
                                        errfault= USERR ;
                                        return -1 ;
                                        break ;
                                case 315 :
                                        log(LOG_ERR,"! Volume %s unmountable\n",ptr[n]) ;
                                        errfault= USERR ;
                                        return -1 ;
                                        break ;
                                default :
                                        if (rcode <0) {
						sperror("TMS NOT RESPONDING");
						rcode = -rcode ;
					}
					if ( (int)strlen(info) >0 ) fprintf(stderr,"%s\n", info) ;
					
                                        log(LOG_ERR,"will retry in a while...\n");
                                        RETRY= ( (cp = getconfent("RTCOPY","RETRY",0)) ==NULL ) ? 0 : atoi(cp) ;
                                        SLEEP= ( (cp = getconfent("RTCOPY","SLEEP",0)) ==NULL ) ? 0 : atoi(cp) ;
                                        if ( RETRY == 0 || SLEEP == 0 ) {
                                                RETRY= 0 ;
                                                SLEEP= 0 ;
                                        }
                                        sleep(SLEEP) ;
                                        break ;
                        } /* switch */
                 } /* while */
           } /* for */
	} /* !isout */

#endif	/* TMS	*/

    } /* !alg_info && !stgflag */

    /*
     * Freeing buffers
     */
    if (uvid != NULL ) (void) free(uvid) ;
    if (uvsn != NULL && uvsn != uvid) (void) free(uvsn) ;

    if ( *device == '\0' || *tms_device != '\0') {
	if ( *device == '\0' && *tms_device == '\0') {
		/*
		 * The device type is still unknown.
		 * By default it is set to CART.
		 */
		(void) strcpy(device,"CART") ;			
	}
	else {
	     if ( *device == '\0' && *tms_device != '\0') 
		(void) strcpy(device,tms_device);
	     else if ( tms_device != '\0' ) {
		if ( !strcmp(tms_device,"CT1") ) {
			(void) strcpy(tms_device,"CART");
		}
		if ( !strcmp(device,"CT1") ) {
			(void) strcpy(device,"CART");
		}
		if ( strcmp(tms_device,device) ) {
			log(LOG_ERR,"! Confusion: %s (user) <--> %s (tms data)\n",device,tms_device);
			errfault= USERR ;
                        return -1 ;
                }
	     }
	}
    }
	return 0 ; 
}

/*
 * Check VIDMAP output to see if the volume is really mapped and if
 * so reset device to the new device type.
 */
int chkvidmap(info,device,tms_device)
char *info,*device,*tms_device;
{
  char *cp;
  int i;
  int ifirst = 0;
  int ilast = 0;
  int ipart = 0;
  for (cp = strtok(info," \t"), i=0; cp != NULL; 
       i++, cp = strtok(NULL," \t")) {
    if ( i == 3 ) strcpy(tms_device,cp);
    if ( i == 15 ) ifirst = atoi(cp);
    if ( i == 16 ) ilast = atoi(cp);
    if ( i == 17 ) ipart = atoi(cp);
  }
  if ( ifirst > 0 && ilast >= ifirst ) strcpy(device,tms_device);
}

/*
 * Getting possible tape servers.
 */
static int gettpserv(device,hosts) 
	char * device ;
	char *** hosts ; 
{
        char ** p ;
        char * lp ;
	char * lptmp ;
        char * cp ;
        int    rc ;
	enum {r, w} readwrite ;
	int    lplen = 0 ;
	int count ;
	int avtpserv = 0;
	char infoserver[MAXHOSTNAMELEN] ;
	char  *ptrlist ;
	char   perptrlist[1024] ; 	/* Permanent ptr list */
	char st[100] ;
	char sccontent[1024] ;
	int scfull =0 ;			/* Is there info in config file ? */
	int sccount = 0 ;		/* How many available servers among those ? */

	ptrlist = (char *) malloc(1024) ;
	ptrlist[0]='\0' ;
	perptrlist[0]='\0' ;
	infoserver[0]='\0';
	sccontent[0]='\0';
	readwrite = (strstr(command,"tpread") != NULL) ? r : w ;

	if (strlen(serversite)) {
		sprintf(st,"SERVER-%s",serversite);
		cp = getconfent("TPSERVLIST",st,1) ;
		strcpy( infoserver, (cp == NULL ? "": strtok(cp," \t")) );
	}
	lptmp = NULL;
	if ( (lp=getconfent("TPSERV",device,1)) != NULL ) {
	  lplen = strlen(lp);
	  if ( (lptmp = (char *) malloc(lplen+2)) == NULL ) {
	    log(LOG_ERR,"! malloc() failed allocating %d bytes\n",lplen+2);
	    exit(SYERR);
	  }
	  (void) strcpy(lptmp,lp);
	  /* 
	   * Add an extra blank to avoid concatenated hostnames when
	   * appending TPSERVR(W) to TPSERV.
	   */
	  (void)strcat(lptmp," "); 
	  lplen = strlen(lptmp);
	}
	/* 
	 * Allow for "read" and "write" specific servers. This permits
	 * to use 3490 drives to read (but not write) 3480 tapes.
	 */
	if ((readwrite == r && (lp=getconfent("TPSERVR",device,1)) != NULL) ||
	    (readwrite == w && (lp=getconfent("TPSERVW",device,1)) != NULL)) {
	  if ( lptmp != NULL ) {
	    if ((lptmp = (char *)realloc(lptmp,lplen+strlen(lp)+3)) == NULL) {
	      log(LOG_ERR,"! realloc() failed allocating %d bytes\n",
		  lplen+strlen(lp)+3);
	      exit(SYERR);
	    }
	    (void) strcat(lptmp,lp);
	    (void)strcat(lptmp," "); 
	    lplen = strlen(lptmp); /* To facilitate debugging.... */
	  }
	}
	if ( lptmp != NULL ) lp = lptmp;
	if ( lp != NULL ) {
		(void) strcpy(sccontent,lp)  ;
		scfull =1 ;
        	if (strlen(serversite) ) {
                	for(cp= strtok(lp," \t\n"),count=0; cp != NULL ;  cp= strtok((char *)NULL," \t\n"))
                        if ( strstr(cp,serversite)!= NULL )
                                count ++ ;
        	}
        	else {
                	for(cp= strtok(lp," \t\n"),count=0; cp != NULL ; cp= strtok((char *)NULL," \t\n"))
                        if ( strstr(cp,".")==NULL )
                                count ++ ;
        	}
		sccount = count ;
	}
	if ( lptmp != NULL ) free(lptmp);
/*
 * START CHOICE ALGORITHM
 */
	if ( strlen(infoserver) && ((isout && !scfull) || (isout && scfull && sccount==0 )) ) {
		rc = rqst_info(infoserver, ptrlist, 1024, serversite, device) ;
                switch (rc) {
                        case -1 :
				log(LOG_ERR,"! No tape server availaible in configuration file\n");
                                log(LOG_INFO,"!  Server %s has no info about %s@%s\n",infoserver, device, serversite);
                                errfault= USERR ;
                                return -1  ;
                        case -2:
                                log(LOG_ERR,"! No tape server availaible in configuration file\n");
				log(LOG_ERR,"! No info daemon responding: Can't connect %s\n",infoserver) ;
                                errfault= USERR ;
                                return -1  ;
                        default:
                                (void) strcpy(perptrlist,ptrlist);
                                lp = ptrlist ;
                                break ;
                }
		        /*
         	 	 * Counting possible tape servers
         		 */
        	if (strlen(serversite) ) {
                	for(cp= strtok(lp," \t\n"),count=0; cp != NULL ;  cp= strtok((char *)NULL," \t\n"))
                        	if ( strstr(cp,serversite)!= NULL )
                                	count ++ ;
        	}
        	else {
                	for(cp= strtok(lp," \t\n"),count=0; cp != NULL ; cp= strtok((char *)NULL," \t\n"))
                        	if ( strstr(cp,".")==NULL )
                                	count ++ ;
        	}
		avtpserv = count ;
	}


	if ( (isout && scfull && sccount) || (!isout && scfull) ) {
		avtpserv = sccount ;
		(void) strcpy(perptrlist, sccontent) ;
	}

	/*
	 * if no tape server available, exit 
	 */
	if ( avtpserv == 0 ) {
		log(LOG_ERR,"! No tape server and no tape information daemon found\n");
		errfault= SYERR ;
		return -1 ;
	}
	
	if (!isout && !scfull) {
		log(LOG_ERR,"! device type %s not defined in configuration file\n",device) ;
		errfault= USERR ;
		return -1 ;
	}
	(void) strcpy(ptrlist,perptrlist) ;

        if ( (p= ( char **) malloc(avtpserv*sizeof(char *))) == NULL ) {
                log(LOG_ERR,"! malloc(): %s\n",sys_errlist[errno]) ;
                errfault= SYERR ;
                return -1 ;
        }
        /*
         * FILLING THE ARRAY HOSTS.
         */

	lp = ptrlist ;
	if ( strlen(serversite) ) {
        	for(count=0,cp=strtok(lp," \t\n"); cp != NULL ;  cp= strtok((char *)NULL," \t\n"))
			if ( strstr(cp,serversite) !=NULL ) {
					p[count]= cp ;
					count ++ ;
			}
	}
	else {
		for(count=0,cp= strtok(lp," \t\n"); cp != NULL ; cp= strtok((char *)NULL," \t\n")) 
                	if ( strstr(cp,".") == NULL ) {
					p[count]= cp ;
					count ++ ;
			}
	}

        *hosts=  p ;
        return  avtpserv;
}

/*
 * Select a server.
 * Depending on the type of device requested,
 * and the queue of the different servers.
 *
 * Connection with the different tape servers are asynchronous
 * except when running on apollo where I did not manage to make it work.
 */
static int selectserver(argc,argv,host)
	int     argc    ; 
	char ** argv    ;
	char  * host    ; 
{
	char 	** hosts ;		/* All possible hosts		*/
	struct tpservinfo {
		int 	sock ; 		/* Socket descriptor		*/
		int   status ;		/* Request current status	*/
		int    queue ; 		/* Tape server load		*/
		int	num  ;		/* Number of cart servers	*/
	} * s ; 

	extern char * getconfent() ; 	/* External declarations	*/
	extern char *     getenv() ; 
	struct sockaddr_in     sin ; 	/* Internet socket address	*/
	struct passwd	     *  pw ;	/* Password entry	 	*/

	char 	   sendbuf[BUFSIZ] ;	/* Sending buffer		*/
	int 		bestserver ;	/* Best available server 	*/
	int 		  nbtpserv ;	/* Number of possible tp server	*/
	char 		 device[8] ;	/* Type of device needed	*/
	char    	     * ptr ;	/* Pointer to the buffer	*/
	int  		    msglen ;	/* Message length		*/
	char		     * env ; 	/* To store env variables	*/
	int		tryagain=0 ; 	/* Is it worth to try again ?	*/
	int 			 i ; 	/* Loop counter			*/
	int		       sum ;	/* Sum of queues abolute values */
	int		     bestc ;    /* server with hightest capacity */

	/*
	 * Getting device type. 
	 * Host will also be set if option S is specified.
	 */
	if ( getdevtyp(argc,argv,device,host) == -1 ) {
		return -1 ;
	}
	/*
	 * Getting a-priori possible tape servers.
	 */
	if ( *host ) {
		if ((hosts= (char **)malloc(sizeof(char *))) == NULL ) {
			log(LOG_ERR,"! malloc(): %s\n",sys_errlist[errno]) ; 
			errfault= SYERR ;
			return -1 ;
		}
		*hosts= host ;
		nbtpserv= 1 ;
	}
	else if ( (nbtpserv= gettpserv(device,&hosts)) == -1 ) {
		return -1 ; 
	} 
	/*
	 * Allocating space to store information
	 */
	if ((s=(struct tpservinfo *) malloc(nbtpserv*sizeof(struct tpservinfo))) == NULL) {
		log(LOG_ERR,"! malloc(): %s\n",sys_errlist[errno]) ; 
		errfault= SYERR ;
		return -1 ;
	}
	/*
	 * Preparing the info request message.
	 */
	if ( (pw= getpwuid(geteuid())) == NULL ) {
		log(LOG_ERR,"! invalid user: %d\n",geteuid()) ;
		errfault= SYERR ;
		return -1 ;
	}
	if ( Gopt ) 
		msglen= 2*WORDSIZE+ strlen(Gname)+1+strlen(device)+1 ;
	else
		msglen= 2*WORDSIZE+ strlen(pw->pw_name)+1+strlen(device)+1 ;
	ptr= sendbuf ;
	marshall_LONG(ptr,RTCOPY_MAGIC) ;
	marshall_LONG(ptr,RQST_INFO) ;
	marshall_LONG(ptr,msglen) ; 
	if ( Gopt ) {
		marshall_STRING(ptr,Gname) ; 
		marshall_WORD(ptr,(WORD) Guid) ; 
	}
	else	{
		marshall_STRING(ptr,pw->pw_name) ; 
		marshall_WORD(ptr,(WORD) pw->pw_uid) ; 
	}
	marshall_WORD(ptr,(WORD) getegid()) ; 
	marshall_STRING(ptr,device) ; 	

	/*
	 * Building Internet address
	 */
	sin.sin_family= AF_INET ; 
	if ( (env=getenv("RTCOPYPORT")) == NULL ) {
		struct servent	* sp ; 

		if ((sp= getservbyname("rtcopy","tcp")) == NULL) {
			serrno = SENOSSERV;
			log(LOG_ERR,"! getservbyname(): %s\n",sstrerror(serrno)) ;
			errfault= SYERR ;
			return -1 ; 
		}
		sin.sin_port= sp->s_port ;
	}
	else	{
		sin.sin_port= htons(atoi(env)) ;
	}

	/*
	 * Connecting servers.
	 */
	for(i= 0; i< nbtpserv; i ++) {
		struct hostent * hp ; 
		int     nonblocking ; 

		s[i].status= CONNECTING ;
		s[i].num= 1;
		if ( (hp= gethostbyname(hosts[i])) == NULL ) {
			s[i].status= IMPOSSIBLE;
			s[i].queue= errno;
			log(LOG_ERR,"! tape server %s not found by gethostbyname()\n",hosts[i]) ; 
			continue ; 
		}
		sin.sin_addr.s_addr= ((struct in_addr *)(hp->h_addr))->s_addr ;	
		if ( (s[i].sock= socket(AF_INET,SOCK_STREAM,0)) == -1 ) {
			log(LOG_ERR,"! socket(): %s\n",sys_errlist[errno]) ; 
			errfault= SYERR ;
			return -1 ; 
		}

#if !defined(apollo)
		/* 
		 * Asynchronously if not running on an apollo
		 */
		nonblocking= 1 ; 
		if ( ioctl(s[i].sock,FIONBIO,&nonblocking) == -1 ) {
			log(LOG_ERR,"! ioctl(): %s\n", sys_errlist[errno]) ; 
			errfault= SYERR ;
			return -1 ; 
		}
#endif	/* ! apollo */

		if (connect(s[i].sock,(struct sockaddr *)&sin,sizeof(sin))== -1 && errno!=EINPROGRESS) {	
			(void) close(s[i].sock) ; 
			s[i].status= IMPOSSIBLE ;
			s[i].queue= errno ;
			continue ;
		}
#if defined(apollo)
		/*
		 * Getting tape server status now.
		 */
		if ( netwrite(s[i].sock,sendbuf,msglen+3*LONGSIZE) == -1 ) {
			(void) close(s[i].sock) ; 
			s[i].status= SYERROR ;
			continue ;
		}

		switch(rcvmsg(s[i].sock)) {
			case -1:
				s[i].status= SYERROR ;
				(void) close(s[i].sock) ;
				break ;
			case GIVE_INFO:
				ptr= buffer ;
				unmarshall_LONG(ptr,s[i].status) ; 
				unmarshall_LONG(ptr,s[i].queue ) ; 
				unmarshall_LONG(ptr,s[i].num);
				(void) close(s[i].sock) ; 
				break ;
			default:
				s[i].status= SYERROR ;
				(void) close(s[i].sock) ;
				break ;
		}
#endif	/* apollo */
	} 
	
#if !defined(apollo)
	/*
	 * Sending request and getting 
	 * answer as soon as possible.
	 */
	for(;;) {
		fd_set	rdfds ;	/* Sets of file descriptors	*/
		fd_set  wrfds ;	/* Sets of file descriptors	*/
		int        nb ; /* Loop counters		*/
		
		/*
		 * Initializing FD_SETs
		 * We exit the for(;;) loop if 
	 	 * there is nothing to wait for.
		 */
		FD_ZERO(&wrfds) ; 
		FD_ZERO(&rdfds) ; 
		for(i= 0,nb= 0; i<nbtpserv; i++) {
			if ( s[i].status == CONNECTING ) {
				FD_SET(s[i].sock,&wrfds) ; 
				nb ++ ;
			}
			else if ( s[i].status == SENDING ) {
				FD_SET(s[i].sock,&rdfds) ; 
				nb ++ ;
			}
		}
		if ( nb == 0 ) 
			break ; 

		while( select(FD_SETSIZE,&rdfds,&wrfds,(fd_set *)0,(struct timeval *)NULL) == -1 && errno == EINTR ) 
			;
		for(i=0; i<nbtpserv;i++) {
			/*
			 * A connection  should be completed.
			 */
			if ( s[i].status==CONNECTING && FD_ISSET(s[i].sock,&wrfds) ) {
				int err ;			/* To check connection	*/
				int errlen= sizeof(err) ; 

				/*
				 * Checking connection.
				 */
				if (getsockopt(s[i].sock,SOL_SOCKET,SO_ERROR,(char *)&err,&errlen) == -1) {
				        log(LOG_ERR,"! getsockopt(): %s\n",sys_errlist[errno]) ; 
					s[i].status = IMPOSSIBLE;
					s[i].queue = errno;
					(void)close(s[i].sock);
				}
				else if ( err ) {
					s[i].status= IMPOSSIBLE ;
					s[i].queue = err ;
					(void) close(s[i].sock) ;
				}
				else	{
					/*
					 * Connection is correct.
					 * Switching to blocking mode again, and,
					 * sending request.
					 */
					int blocking= 0 ; 
					if ( ioctl(s[i].sock,FIONBIO,&blocking) == -1 ) {
						log(LOG_ERR,"! ioctl(): %s\n",sys_errlist[errno]) ;
						errfault= SYERR ;
						return -1 ; 
					}
					if ( setnetio(s[i].sock) == -1 ) {
						log(LOG_ERR,"! error in setnetio()\n") ;
						errfault= SYERR ;
						return -1 ; 
					}
					s[i].status= SENDING ;
					if ( netwrite(s[i].sock,sendbuf,msglen+3*LONGSIZE) == -1 ) {
						s[i].status= SYERROR ;
						(void) close(s[i].sock) ;
						continue ;
					}
				}
			}
			/*
			 * An answer has been received.
			 */
			else if ( s[i].status==SENDING && FD_ISSET(s[i].sock,&rdfds) ) {
				switch(rcvmsg(s[i].sock)) {
					case -1:
						s[i].status= SYERROR ;
						(void) close(s[i].sock) ;
						break ;
					case GIVE_INFO:
						ptr= buffer ;
						unmarshall_LONG(ptr,s[i].status) ; 
						unmarshall_LONG(ptr,s[i].queue ) ; 
						unmarshall_LONG(ptr,s[i].num);
						(void) close(s[i].sock) ; 
						break ;
					default:
						s[i].status= SYERROR ;
						(void) close(s[i].sock) ;
						break ;
				}
			}
		}
	} 
	if (alg_info && !istpselect ) {
		printf("\n");
		for (i=0;i<nbtpserv;i++) {
	    		if ( s[i].status == AVAILABLE )
				printf("\tServer: %s\t Capacity: %d\t Queue: %d\n",hosts[i],s[i].num, s[i].queue);
	    		else
				printf("\tServer: %s\t Not responding\n",hosts[i]);
		}
		printf("\n");
	}
#endif	/* ! apollo */

	/*
	 * Choosing server.
	 */
	if (!istpselect) 
		log(LOG_INFO,"selecting tape server ...\n") ; 
	sum=0;
	for(i=0,bestserver= -1,bestc= -1;i<nbtpserv;i++) {
		switch(s[i].status) {
			case IMPOSSIBLE:
				switch(s[i].queue) {
				   case ETIMEDOUT:
					log(LOG_INFO,"!  ! * %s connection timed out.\n",hosts[i]) ; 
					tryagain ++ ;
					break ; 
				   case ECONNREFUSED:
					log(LOG_INFO,"! * %s refused connection.\n",hosts[i]) ; 
					break ; 
				   case ENETUNREACH:
					log(LOG_INFO,"! * %s is unreachable.\n",hosts[i]) ; 
					break ; 
				   default:
					log(LOG_INFO,"! * %s was not connectable.\n",hosts[i]) ; 
					break ; 
				}
				break ;
			case SYERROR:
				log(LOG_INFO,"! * %s was unable to compute or to send us its status.\n",hosts[i]) ; 
				break ; 
			case CONNECTING:
			case SENDING:
				log(LOG_INFO,"! * %s unreachable or not answering within timeout.\n",hosts[i]) ;
				break ; 
			case PERMDENIED:
				if ( Gopt )
				  log(LOG_INFO,"! * %s does not accept requests from client having a uid (%d) smaller than 100.\n",hosts[i],Guid) ; 
				else
				  log(LOG_INFO,"! * %s does not accept requests from client having a uid (%d) smaller than 100.\n",hosts[i],pw->pw_uid) ; 
				break ;
			case UNKNOWNUID:
				if ( Gopt ) 
				  log(LOG_INFO,"! * %s - uid (%d) is not defined\n",hosts[i],Guid) ; 
				else
				  log(LOG_INFO,"! * %s - uid (%d) is not defined.\n",hosts[i],pw->pw_uid) ; 
				break ; 
			case UNKNOWNGID:
				log(LOG_INFO,"! * %s - gid (%d) is not defined or you don't belong to the corresponding group.\n",hosts[i],getegid()) ; 
				break ; 
			case UIDMISMATCH:
				if ( Gopt ) 
				  log(LOG_INFO,"! * %s - uid (%d) and user name (%s) don't match.\n",hosts[i],Guid,Gname) ; 
				else
				  log(LOG_INFO,"! * %s - uid (%d) and user name (%s) don't match.\n",hosts[i],pw->pw_uid,pw->pw_name) ; 
				break ; 
			case NOTAUTH:
				log(LOG_INFO,"! * %s does not know you.\n",hosts[i]) ; 
				break ; 
                        case HOSTNOTAUTH:
                                log(LOG_INFO,"* Your host is not authorized by %s.Sorry.\n",hosts[i]); 
                                break;
			case ALLDOWN:
			case NOTAVAIL:
				tryagain ++ ;
				log(LOG_INFO,"* %s tape service momentarily interrupted.\n",hosts[i]) ; 
				break ; 
			case AVAILABLE:
				if (!istpselect)
					log(LOG_INFO,"* %s is a possible tape server.\n",hosts[i]) ; 
				break ; 
			case HOSTMAINT:
				if (!istpselect)
					log(LOG_INFO,"* %s is unavailable .\n",hosts[i]) ;
				break;
			case MUTEINFO:
				if (!istpselect)
					log(LOG_DEBUG,"* %s is giving no response.\n",hosts[i]) ;
				break;
			case HOSTNOTSUIT:
				if (!istpselect)
					log(LOG_INFO,"* %s is not adapted to suit request.\n",hosts[i]) ;
				break;
		}
		if ( s[i].status == AVAILABLE ) {
			if ( s[i].queue < 0 ) {
#if !defined(CRAY)
				int j;
				int leastused;
				/*
				 * They are some free devices:
				 * Choose the one with the lowest number of 
				 * units in use.
				 */
				leastused=i;
				for (j=i; j < nbtpserv ; j++) 
					if (s[j].queue < 0 && s[j].queue+s[j].num < s[leastused].queue +s[leastused].num )
						leastused=j;
				(void) strcpy(host,hosts[leastused]) ; 
                                if (leastused != i && !istpselect)
                                        log(LOG_INFO,"* %s is a possible tape server.\n",hosts[leastused]) ;

#else
				void) strcpy(host,hosts[i]);
#endif
				if (!istpselect)
					log(LOG_INFO,"! selected tape server is %s.\n\n",host) ;
				else
					printf("%s\n",host);
				return 0 ; 
			}
			sum += abs(s[i].queue) ;
			if ( bestc == -1 || s[bestc].num < s[i].num )
				bestc=i;
			if( bestserver== -1 || ( (100*s[i].queue)/s[i].num< (100*s[bestserver].queue)/s[bestserver].num	) )
				bestserver= i ;
		}
	}
	/* If all the queues are zero
 	 * then choose the one with the 
	 * tape server with the highest capacity
	 */
	if (sum==0 && bestc != -1 && s[bestc].status == AVAILABLE && bestserver != -1) 
			bestserver= bestc;
	if ( bestserver == -1 ) {
		if ( tryagain ) {
			log(LOG_INFO,"tape service is momentarily interrupted.\n\n") ;
			errfault= 0 ; 
		}
		else	{	
			log(LOG_INFO,"! sorry, no tape server available.\n\n") ;
			errfault= SYERR ;
		}
		return -1 ; 
	}
	else	{
		(void) strcpy(host,hosts[bestserver]) ; 
		log(LOG_INFO,"selected tape server is %s.\n\n",host) ; 
		return 0 ;
	}
}

/*
 * Connect the client to the server.
 */
static int connectserver(host)
	char * host ;
{
	struct servent 		*sp ; 		/* Service entry pointer.	*/
	struct hostent		*hp ;		/* Host entry pointer.		*/
	struct sockaddr_in	sin ;		/* An Internet socket address.	*/ 
	int 		       sock ; 		/* Socket descriptor.		*/
	extern char      * getenv() ; 		/* Getting env variables	*/
	char 		      * env ;		/* To store env variables 	*/

	/*
	 * Creating socket.
	 */
	if (( sock= socket(AF_INET,SOCK_STREAM,0)) == -1 ) {
		log(LOG_ERR,"! socket(): %s\n",sys_errlist[errno]) ; 
		errfault= SYERR ;
		return -1 ;
	}

	if ((hp= gethostbyname(host)) == NULL ) {
		serrno = SENOSHOST;
		log(LOG_ERR,"! gethostbyname(): %s\n",sstrerror(serrno)) ;
		errfault= SYERR ;
		return -1 ; 
	}

	/*
	 * Building Daemon Internet address.
	 */
	if ( (env=getenv("RTCOPYPORT")) == NULL ) {
		if ((sp= getservbyname("rtcopy","tcp")) == NULL) {
			serrno = SENOSSERV;
			log(LOG_ERR,"! getservbyname(): %s\n",sstrerror(serrno)) ;
			errfault= SYERR ;
			return -1 ; 
		}
		sin.sin_port= sp->s_port ;
	}
	else	{
		sin.sin_port= htons(atoi(env)) ;
	}
	sin.sin_family= AF_INET ;
	sin.sin_addr.s_addr= ((struct in_addr *)(hp->h_addr))->s_addr ;

	/*
	 * Connecting the socket.
	 */
	if ( connect(sock, (struct sockaddr *) &sin, sizeof(sin))  == -1 ) {
		log(LOG_ERR,"! connect(): %s\n",sys_errlist[errno]) ; 
		errfault= SYERR ;
		return -1 ;
	}
	/* 
	 * Checking wether tape server is really out of site
   	 * or not.
	 */
	if( !alg_info) 
		isout = isremote(sin.sin_addr,host) ;
	log(LOG_DEBUG,"tape server chosen is in site ? %s\n",(isout ? "no":"yes"));

	if (isout && !strlen(serversite) ) {
		log(LOG_ERR,"! request's destination is out of site, but option -R <serversite> not specified\n");
		return -1 ;
	}
	if ( setnetio(sock) == -1 ) {
		log(LOG_ERR,"! error in setnetio()\n") ; 
		(void) close(sock) ; 
		errfault= SYERR ;
		return -1 ; 
	}
	return sock ; 
}


/*
 * Sending a PING.
 * In case of error the socket is closed.
 */
static int ping(sock) 
	int sock ; 
{
	char * ptr ; 

	ptr= buffer ;
	marshall_LONG(ptr,RTCOPY_MAGIC) ; 
	marshall_LONG( ptr,RQST_PING) ; 
	marshall_LONG(ptr,0) ;
	/* setrtimo(PING_TIMEOUT) ;     disabled for now */
	if ( netwrite(sock,buffer,3*LONGSIZE) == -1 ) {
		log(LOG_ERR,"! netwrite(): %s\n", sys_errlist[errno]) ; 
		(void) close(sock) ; 
		errfault= SYERR ;
		return -1 ; 
	}
	return 0 ; 
}

/*
 * Sending abort request to the server
 * to kill the current job.
 */
static void sendabort(sock) 
	int sock ; 
{
	char abortmsg[3*LONGSIZE] ;
	char * ptr ; 

	ptr= abortmsg ; 
	marshall_LONG(ptr,RTCOPY_MAGIC) ; 
	marshall_LONG(ptr,RQST_ABORT) ;
	marshall_LONG(ptr,0) ;
	if ( netwrite(sock,abortmsg,3*LONGSIZE) == -1 ) {
		return ; 
	}	
	switch(rcvmsg(sock)) {
		case -1:
			break ;
		case ACKN_ABORT:
			break ;
		default:
			break ;
	}
	(void) close(sock) ; 
	return ; 
}

/*
 * Sending request 
 */
static int sendreq(sock,argc,argv,k,rfiosock) 
	int     sock ;
	int     argc ; 
	char ** argv ; 
	u_short	  *k ;
	int *rfiosock ;
{
	char   * ptr ;		/* Pointer to the buffer	*/
	int   msglen ;		/* Message length		*/
	int        i ;		/* Loop index			*/
	struct passwd * pw ; 	/* Password entry.		*/

				/* File creation mode mask	*/
#if defined(CRAY) || defined(sgi) || defined(hpux) || defined(_AIX)  || ( defined(__osf__) && defined(__alpha) )
	mode_t  mask ;	
#endif	/* CRAY || sgi || hpux || _AIX */
#if defined(sun) || defined(ultrix) || defined(apollo) || defined(linux) || defined(__Lynx__)
	int 	mask ; 		
#endif	/* sun || ultrix || apollo || linux || __Lynx__*/

	/*
	 * Getting password entry.
	 */
	if ( ( pw= getpwuid(geteuid())) == NULL ) {
		log(LOG_ERR,"! invalid user: %d\n",geteuid()) ;
		errfault= SYERR ;
		return -1 ;
	}

	/*
	 * Getting file creation mode mask.
	 */
	(void) umask(mask= umask(0)) ;

	/*
	 * Building the message.
	 *
	 * + Message header:
	 * -   Magic Number
	 * -   Request type
	 * -   Message length
	 * + Message body:
	 * -   Login name
	 * -   User identifier
	 * -   Group identifier
	 * -   File Creation Mode mask
	 * -   Nb of arguments
	 * -   Arguments.
	 */
	ptr= buffer ;
	marshall_LONG(ptr,RTCOPY_MAGIC) ;
	if ( strstr(command,"tpread")!= NULL ) {
		marshall_LONG(ptr,RQST_TPDK) ; 
	} 
	else if ( strstr(command,"tpwrite") != NULL )	{
		marshall_LONG(ptr,RQST_DKTP) ;
	}
        else {
                log(LOG_ERR,"! Command %s does not have it original name !",command) ;
                exit(SYERR);
        }

	if ( Gopt ) 
		msglen= 4*WORDSIZE + LONGSIZE + strlen(Gname)+1 ; 
	else
		msglen= 4*WORDSIZE + LONGSIZE + strlen(pw->pw_name)+1 ; 

	for(i=1;i<argc;i++)
		msglen+= strlen(argv[i])+1 ;
	marshall_LONG(ptr,msglen) ; 

	if ( Gopt ) {
		marshall_STRING(ptr,Gname) ; 
		marshall_WORD(ptr,(WORD) Guid) ; 
	}
	else	{
		marshall_STRING(ptr,pw->pw_name) ; 
		marshall_WORD(ptr,(WORD) pw->pw_uid) ; 
	}

	marshall_WORD(ptr,(WORD) getegid()) ;
		/* find key & port & send it */
	*k = KEYPORT ;
	while ( isout && ( *rfiosock = connectrfiod ((u_short)*k) ) == -EADDRINUSE )
		*k ++ ;
	if ( isout && *rfiosock < 0 ) {
		log(LOG_ERR,"! connectrfiod() failed\n");
		exit(1);
	}
	marshall_WORD(ptr, *k ) ;
	marshall_WORD(ptr,(WORD) mask) ;
	marshall_LONG(ptr,argc ) ; 
	for(i=1;i<argc;i++)  {
		marshall_STRING(ptr,argv[i]) ; 
	}
	/*
 	 * Sending request.
 	 */
	if ( netwrite(sock,buffer,msglen+3*LONGSIZE) == -1 ) {
		log(LOG_ERR,"! netwrite(): %s\n", sys_errlist[errno]) ; 
		errfault= SYERR ;
		return -1 ;
	}

	/*
	 * Waiting for ackn.
	 */
	switch( rcvmsg(sock)) {
		case -1:
			return -1 ;
		case ACKN_TPDK:
			if ( strstr(command,"tpread")!=NULL ) {
				return 0 ; 
			} else 	{
				log(LOG_ERR,"! internal error\n") ;
				errfault= SEERR ;
				return -1 ; 
			}
		case ACKN_DKTP:
			if ( strstr(command,"tpwrite")!= NULL ) {
				return 0 ; 
			} else 	{
				log(LOG_ERR,"! internal error\n") ;
				errfault= SEERR ;
				return -1 ; 
			}
		default:
			log(LOG_ERR,"! internal error\n") ;
			errfault= SEERR ;
			return -1 ; 
	} 
}

/*
 * tpread() or tpwrite()
 */
main(argc,argv)
	int     argc ; 
	char ** argv ; 
{
	char * getconfent() ;	/* External declaration	*/
	char 	   host[64] ;	/* Tape server hostname	*/
	int	       sock ; 	/* Socket Identifier	*/
	int	    	  i ; 	/* Loop index		*/
	char 	      * ptr ; 	/* Char pointer		*/
	int 	      RETRY ; 	/* Number of possible retries	*/
	int 	      SLEEP ; 	/* Interval between two retries	*/
	static char **argv1;    /* Store filenames after link were solved */
	char **dskservlist;
	char buf[MAXFILENAMSIZE] ;
	int rfiosock ;		
	int checked ;		/* Do we expect more key checks ? 	  */
	u_short key_p ;		/* To get key number 			  */


	INIT_TRACE("RFIO_TRACE");
	/*
	 * Stripping the command name from its full path name.
	 */
	if ( (command= strrchr(argv[0],'/')) == NULL ) 
		command= argv[0] ;
	else 
		command ++ ; 
	
	if ( strstr(argv[0],"tpselect") != NULL ) {
		alg_info++ ;
		istpselect ++ ;
	}
	/*
	 * Starting LOG on stderr.
	 */
	(void) initlog(command,LOG_INFO,"") ;
	
	/* 
	 * The following ugly flag enables to perform a 
	 * complete tape re-selection 
	 */
	restart:

	/*
	 * Selecting tape server.
	 */
	RETRY= ( (ptr= getconfent("RTCOPY","RETRY",0)) == NULL ) ? 0 : atoi(ptr) ;
	SLEEP= ( (ptr= getconfent("RTCOPY","SLEEP",0)) == NULL ) ? 0 : atoi(ptr) ;
	if ( RETRY == 0 || SLEEP == 0 ) {
		RETRY= 0 ; 
		SLEEP= 0 ; 
	}
	for(i= 0; i<= RETRY; i ++) {
		if ( i ) {
			log(LOG_INFO,"retry selection in a little while...\n") ; 
			sleep(SLEEP) ; 
		}
		if ( selectserver(argc,argv,host) == -1 ) {
			if ( errfault || i == RETRY ) {
				log(LOG_INFO,"command failed\n\n") ; 
				exit(errfault? errfault:SYERR) ; 
			}
		}
		else
			break ; 
	}
	 
	
	/* 
	 * Only info has been requested
	 */
	if (alg_info) 
		exit(0);
	/*
	 * Setting SIGINTR interrupt handler.
	 * To catch ^C.
	 */
	if ( signal(SIGINT,( void (*) ()) actionSIGINT) == SIG_ERR ) {
		log(LOG_ERR,"! signal(SIGINT,actionINT): %s\n",sys_errlist[errno]) ; 
		log(LOG_INFO,"command failed\n\n") ; 
		exit(SYERR) ;
	}

	/*
	 * Connecting tape server.
	 */
	if ( (sock= connectserver(host)) == -1 ) {
		log(LOG_INFO,"command failed\n\n") ; 
		exit(errfault) ; 
	}

	/* 
	  Dynamically allocate the argv1 and dskservlist arrays
	 */
	argv1 = (char **)malloc(sizeof(char *) * nfile);
	dskservlist = (char **)malloc(sizeof(char *) * nfile);
	if ((argv1 == NULL) || (dskservlist == NULL)) {
	  log(LOG_ERR,"! malloc() failed allocating %d bytes\n",sizeof(char *) * nfile) ;
	  log(LOG_INFO,"command failed\n\n") ;
	  exit(SYERR) ;
	}

	/*
	 * Transforming file names by solving links &
 	 * adding names. After argument index argc - nfile,
         * all arguments are file names.
         */
        for (i=argc-nfile; i< argc ; i++ )  {
		/*
		 * argv[] character chains might not be long enough to
		 * support changes occuring in solveln() & add_domain()
		 */
		argv1[i-argc+nfile]= (char *)malloc(MAXFILENAMSIZE) ;
		if ( argv1[i-argc+nfile] == NULL ) {
			log(LOG_ERR,"! malloc() failed allocating %d bytes\n",MAXFILENAMSIZE) ;
			log(LOG_INFO,"command failed\n\n") ;
			exit(SYERR) ;
		}
		(void) strcpy(argv1[i-argc+nfile],argv[i]) ;
		argv[i]=argv1[i-argc+nfile] ;

		/* Deferred allocation, do not touch the "." special filename */
		if (strcmp(argv[i],".") != 0) {
               	 	if ( !solveln(argv[i], buf, MAXFILENAMSIZE ) )
				(void) strcpy(argv[i],buf) ;
		}
                if (isout && !add_domain(argv[i],buf) ) {
			log(LOG_DEBUG, "Adding domain to %s\n",argv[i]) ;
                        (void) strcpy(argv[i], buf) ;
		}
        }

        if ( isout ) {
		char filename[MAXFILENAMSIZE];
                for ( i=argc-nfile; i<argc; i++ ) {
                        char *path;
                        char *dskhost ;
                        (void) strcpy(filename, argv[i]);

                        /* Filename is local */
                        if ( !rfio_parse(filename , &dskhost, &path ) ) {
                                dskservlist[i-argc+nfile]=(char *)malloc(MAXHOSTNAMELEN) ;
                                gethostname(dskservlist[i-argc+nfile],MAXHOSTNAMELEN) ;
                        }
                        else {
                                dskservlist[i-argc+nfile]=(char *)malloc((unsigned)strlen(dskhost)+1) ;
                                (void) strcpy(dskservlist[i-argc+nfile],dskhost);
                        }
                }
        }


	/*
	 * Sending request.
	 */
	if ( sendreq(sock,argc,argv,&key_p ,&rfiosock) == -1 ) {
		log(LOG_INFO,"command failed\n\n") ; 
		exit(errfault) ; 
	}
	checked = 0 ;

	/*
	 * Waiting for answer.
	 */
	for(;;) {
#if !defined(SOLARIS)
		fd_set  rfds,wfds,efds ;	/* Set of file descriptor    */
		struct timeval     out ; 	/* To set time out in select */
#else
		struct  pollfd pollfds[2] ;
#endif
		int 		 rcode ;	/* Used for return code	     */

#if !defined(SOLARIS)
		FD_ZERO(&rfds) ; 
		FD_ZERO(&wfds) ; 
		FD_ZERO(&efds) ; 
		FD_SET(sock,&rfds) ; 
                if ( isout && !checked )
                        FD_SET ( rfiosock, &rfds) ;

		out.tv_sec= RTCOPY_TIMEOUT ;
		out.tv_usec= 0 ; 
#else
		
                pollfds[0].fd= sock ;
                pollfds[0].events= POLLIN ;
                pollfds[0].revents=0 ;
		if ( isout && !checked ) {
			pollfds[1].fd= rfiosock ;
                	pollfds[1].events= POLLIN ;
                	pollfds[1].revents=0 ;
		}	
#endif

		/*
		 * Did a SIGINT occured ?
		 */
		if ( ABORT ) {
			sendabort(sock) ; 
			log(LOG_INFO,"command failed\n\n") ; 
			exit(USERR) ;
		}
		/*
		 * Waiting for an event.
		 */
		log(LOG_DEBUG,"main() : select waiting for response\n");
#if !defined(SOLARIS)
		rcode= select(FD_SETSIZE,&rfds,&wfds,&efds,(struct timeval *)&out) ; 
#else
		rcode= poll( pollfds , ((isout && !checked) ? 2:1 ), RTCOPY_TIMEOUT );
#endif
		fflush(stderr);
		if ( rcode == -1 ) {
			if ( errno != EINTR) {
				log(LOG_ERR,"! select(): %s\n",sys_errlist[errno]) ; 
				log(LOG_INFO,"command failed\n\n") ; 
				(void) close(sock) ; 
				exit(SYERR) ;
			}
			else if ( ABORT ) {
				sendabort(sock) ; 
				log(LOG_INFO,"command failed\n\n") ; 
				exit(USERR) ;
			}
		}
		/*
		 * The key is checked
	 	 */
#if !defined(SOLARIS)
                if ( isout && FD_ISSET ( rfiosock, &rfds ) ) {
#else
		if ( isout && pollfds[1].revents ) {
#endif
                        int rc ;
                        rc = getkey ( rfiosock, (int)key_p ,dskservlist)  ;
                        if ( rc == -3 ) {
                        	log(LOG_ERR,"! rfio key inconsistent ! \n");
				sendabort(sock) ;
				log(LOG_INFO,"command failed\n\n") ;
                        	exit(USERR) ;
                	}
			if ( rc == -1 ) {
				log(LOG_ERR,"! rfio failed to connect tpread ! \n");
                                sendabort(sock) ;
                                log(LOG_INFO,"command failed\n\n") ;
				exit(USERR) ;
                        }
		}

		/*
		 * if there is no answer yet, a PING
		 * is sent to check the connection.
		 */
		if ( rcode == 0  &&  ping(sock) == -1 ) {
			log(LOG_INFO,"command failed\n\n") ; 
			exit(errfault) ;
		}
		
		/*
		 * Receiving answer.
		 */
#if !defined(SOLARIS)
		if ( FD_ISSET(sock,&rfds) ) {
#else
		if ( pollfds[0].revents ) {
#endif
		    log(LOG_DEBUG,"Entering in rcvmsg() \n");
		    switch(rcode= rcvmsg(sock)) {
			case -1:
				log(LOG_INFO,"command failed\n\n") ; 
				exit(errfault) ;
			case ACKN_PING:
				break ; 
			case GIVE_OUTP:	
				/*
				 * WARNING !!!
				 * Valid  only on ASCII machine
				 * buffer is NOT UNMARSHALLed as it should be.
				 */
				(void) fprintf(stderr,"%s",buffer) ;
				fflush(stderr);
				break ;
			case GIVE_RESU:
				ptr= buffer ;
				unmarshall_LONG(ptr,rcode) ; 
				log(LOG_DEBUG,"main(): GIVE_RESU returned with rcode %d\n",rcode);
				if (rcode == BLKSKPD) 
					log(LOG_INFO,"command partially successful since blocks were skipped\n");
				else if (rcode == TPE_LSZ )
					log(LOG_INFO,"command partially successful: blocks skipped and size limited by -s option\n");
				else if (rcode == MNYPARY )
					log(LOG_INFO,"command partially successful: blocks skipped or too many errors on tape\n");
				else if (rcode == ETMUSR || rcode == RSLCT ) {
					(void) close(sock) ;
					log(LOG_INFO,"Re-selecting another tape server\n");
					sleep(SLEEP) ;
					goto restart;
				}
				else if ( rcode == 0 || rcode == LIMBYSZ )
					log(LOG_INFO,"command successful\n");
				else 
					log(LOG_INFO,"command failed\n") ; 
				exit(rcode) ; 
			default:
				log(LOG_ERR,"! internal error\n") ; 
				log(LOG_INFO,"command failed\n\n") ; 
				exit(SEERR) ; 
		    }
		}
	}
}

#define OK	1
#define NOTOK	2

/*
 * Connects the client and transmits OK or NOTOK
 * if client sent the "key", he gets OK, otherwise NOTOK.
 * returns -1 if failure
 * key otherwise.
 */

/* 
 * Connects to rfiod & returns the socket descriptor,
 *  or -errno if an error occured
 */

int connectrfiod (keyport)
u_short keyport ;
{

                struct sockaddr_in              sin ;   /* Internet address     */
                int                            sock ;   /* Socket               */
                char                             *p ;
                extern               char *getenv() ;

                /*
                 * Creating socket.
                 */
		log(LOG_DEBUG,"connectrfiod(): keyport is %d\n",keyport);
                if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)  {
                        log(LOG_ERR,"! socket(): %s\n",sys_errlist[errno]);
                        return -errno ;
                }
                /*
                 * Finding port number.
                 */
                if ( (p = getenv ("RFIO2TPREAD")) == NULL ) 
                        sin.sin_port = keyport;
                else    {
                        sin.sin_port= htons(atoi(p)) ;
                }

                /*
                 * Binding socket.
                 */
                sin.sin_addr.s_addr = htonl(INADDR_ANY);
                sin.sin_family = AF_INET;
                if (bind(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
                        log(LOG_ERR,"! bind(): %s\n",sys_errlist[errno]);
                        return -errno ;
                }

                /*
                 * Listening on socket.
                 */
                if ( listen(sock, 5) == -1 ) {
                        log(LOG_ERR,"! listen(): %s\n",sys_errlist[errno]);
                        return -errno;
                }
		return ( sock ) ;


}

/*
 * returns -2 if it is contacted by a wrong host
 * 	   -3 if key received is inconsistent.
 */
int getkey( sock ,key,fromhostlist)
int sock , key;         /* Expected key */
char **fromhostlist  ;       /* Expected location of replier */
{
                struct            sockaddr_in  from ;  /* Client socket address  */
                int          fromlen = sizeof(from) ;
                struct hostent                 * hp ;  /* Client host structure  */
                int                           magic ;
                char                   buffer[1024] ;
                int                          reckey ;
                char     clienthost[MAXHOSTNAMELEN] ;
                char                            *ch ;
                int                           nsock ;
		int              	      rcode ;			
                char                           *ptr ;
		int 				  i ;
		int 			      found ;


                nsock = accept(sock, (struct sockaddr *)&from, &fromlen);
                if (nsock < 0) {
                         if (errno != EINTR) 
                              log(LOG_ERR,"! accept(): %s\n",sys_errlist[errno]);
			 return -1 ;
                }
                if ( setnetio(nsock) == -1 ) {
                        (void) close(nsock) ;
                        log(LOG_ERR,"! error in setnetio()\n") ;
			return -1 ;
                }

                if (getpeername(nsock,(struct sockaddr *)&from, &fromlen)< 0) {
                         log(LOG_ERR,"! getpeername: %s\n",sys_errlist[errno]);
                         (void) close(nsock) ;
                         return -1 ;
                }
                hp= gethostbyaddr((char *)(&from.sin_addr),sizeof(struct in_addr),from.sin_family) ;
                if ( hp == NULL )
                         ch= (char *)inet_ntoa(from.sin_addr) ;
                else
                         ch= hp->h_name ;
		strcpy(clienthost, ch) ;
                log(LOG_DEBUG , "Connection from %s established\n",clienthost);

                /*
                 * Aliasing clienthost might be useful
                 * for some networks configurations
                 */
                if ( (ptr=getconfent("NETWORK", clienthost, 0)) != NULL )
                        strcpy(clienthost, ptr) ;

		/* 
		 * Check that key arrived from one of the
		 * disk servers
	 	 */
		for ( i=0,found=0 ; i<nfile && !found ;i++)  
                	if (  !strcmp(clienthost,fromhostlist[i]) || 
			      !strncmp(clienthost,fromhostlist[i], strlen(clienthost)) ||
			      !strncmp(fromhostlist[i],clienthost,strlen(clienthost) ) ||
                              strstr(clienthost, fromhostlist[i]) != NULL ) 
			      
				found ++ ;
		if (!found) {
			log(LOG_ERR,"! Warning: key received from unexpected host %s !\n",clienthost);
			return -2 ;
		}
                log(LOG_DEBUG , "Connection OK.\n");
                rcode = netread(nsock,(char *)buffer,LONGSIZE*3) ;
                switch (rcode) {
                        case -1:
                                log(LOG_ERR,"! netread(): %s\n",sys_errlist[errno]) ;
                                (void) close(nsock);
                                return -1 ;
                        case  0:
                                log(LOG_ERR,"! netread(): Connection dropped at remote end\n") ;
                                (void) close(nsock);
                                return -1 ;
                }
                ptr = buffer ;
                unmarshall_LONG ( ptr, magic ) ;
                unmarshall_LONG (ptr, reckey ) ;
                if ( magic != RFIO2TPREAD_MAGIC ) {
                        log(LOG_ERR,"! Magic does not match\n");
                        (void) close(nsock);
                        log(LOG_ERR,"Magic received: %d\n",magic);
                        return -1 ;
                }
                ptr=buffer ;
                marshall_LONG(ptr,RFIO2TPREAD_MAGIC);
/*
 *
 * Replaced code as we assume that as connection occured
 * on right port the authentification of user is done
 *
 *              if ((int)reckey == (int)key ) {
 *                      log(LOG_DEBUG ,"Key received is correct\n");
 *                      marshall_LONG(ptr,OK) ;
 *              }
 *              else {
 *                      log(LOG_ERR,"! Key received from %s is incorrect \n",clienthost);
 *			log(LOG_DEBUG, "Expected %d received %d\n", key,reckey);
 *                      marshall_LONG(ptr,NOTOK) ;
 *			reckey = -3 ;
 *		}
 */
		marshall_LONG(ptr,OK) ;
                marshall_LONG(ptr,0);
                if  ( netwrite ( nsock, buffer, LONGSIZE*3) == -1 ) {
                        log(LOG_ERR,"! netwrite(): %s\n", sys_errlist[errno]) ;
                        return -1 ;
                }
                (void) close (nsock);

                return( (int)reckey );

}

/*
 * Add domain name to file path.
 * file path is assumed to be an absolute path.
 */

int add_domain( path, buffer)
char * path ;
char * buffer ;
{
	char *p ;
	char host[MAXHOSTNAMELEN] ;
	char *nfsroot ;
	extern char * getconfent ();

	nfsroot =  getconfent("RFIO","NFS_ROOT",0) ;
	if ( (p = strstr(path,":/")) != NULL ) {
		(void) strcpy ( host, path ) ;
		p = strstr( host ,":/") ;
		*p = '\0' ;
			
		/*
	 	 * Host name is not a full internet address
	 	 */
		if ( strstr(host,".") == NULL ) {
			sprintf( buffer , "%s.%s:%s", host, DOMAINNAME, p+1) ;
			return 0 ;
		}
		else {
			(void) strcpy(buffer, path) ;
			return 0 ;
		}
	}
		
	if ( !strncmp(path, nfsroot, strlen(nfsroot) ) ) {
		p = strstr(path+1,"/") ;
		if ( p ==  NULL )
			return -1 ;
		(void) strcpy( host, p+1 ) ;
		p = strchr(host,'/') ;
		if ( p == NULL )
			return -1 ;
		else
			*p = '\0' ;
		
		if ( strstr( host,".")==NULL ) {
			sprintf(buffer,"%s.%s:%s", host,DOMAINNAME, path) ;
			return 0 ;
		}
		else {
			sprintf(buffer,"%s",  path) ;
                        return 0 ;
		}
	}
	return -1 ;
}

