/*
 * Copyright (C) 1998 by CERN/IT/PDP/DM
 * All rights reserved
 */
 
#ifndef lint
static char sccsid[] = "@(#)rfio_hpss.c	1.17 08/23/99  CERN IT-PDP/DM Olof Barring";
#endif /* not lint */

#ifdef HPSS
#include <syslog.h>			/* System logger		*/
#include <log.h>                        /* Genralized error logger      */
#include <sys/param.h>                  /* System parameters            */

#if defined(__osf__) && defined(__alpha)
#include <sys/uswitch.h>
#include <sys/resource.h>
#endif /* __osf__ && __alpha */
#include <u_signed64.h>
#include <hpss_api.h>
#define RFIO_KERNEL     1               /* KERNEL part of the programs  */
#include <rfio.h>                       /* Remote file I/O              */
#include <rfio_hpss.h>
#include "../h/marshall.h"
void *rhpss_doit();
#define DCE_ERROR(X,Y) log(LOG_ERR,"(%s:%d) %s: status=%d, %s\n",__FILE__,__LINE__,#X,Y,sys_errlist[errno])

extern struct global_defs global[];

extern char     *sys_errlist[];         /* External error list          */
/*
 * pthread_exit() success status must be declared const.
 */
const int success=0;

/*
 * UID and user name to use for "root" access
 */
static uid_t root_uid;
#if !defined(RFIO_HPSSROOT)
#define RFIO_HPSSROOT "hpss_ssm"
#endif /* RFIO_HPSSROOT */


int rhpss_init(PrincipalName,KeyTabFile,privhosts,threadData,nb_threads)
char *PrincipalName,*KeyTabFile,*privhosts;
struct rfio_threadData *threadData[];
int *nb_threads;
{
  char *p;
  int i, status;
  extern char *rtuser;
  extern char *keepalive;
  extern int DISKBUFSIZE_READ;
  extern int DISKBUFSIZE_WRITE;
  pthread_attr_t attr;
  struct rfio_threadData *td;

#if defined(__osf__) && defined(__alpha)
  int uswitch_val;
  struct rlimit rl;
  int thr_stack_size;
#endif /* __osf__ && __alpha */

#if defined(__osf__) && defined(__alpha)
  uswitch_val = uswitch(USC_GET,0);
  uswitch(USC_SET,uswitch_val | USW_NULLP);
  rl.rlim_cur = rl.rlim_max = 32*1024*1024;	
  setrlimit(RLIMIT_STACK,&rl); 
#endif /* __osf__ && __alpha */

  /*
   * Moved from doit() because getconfent not re-entrant
   */
  if ( (p = getconfent("RFIOD","KEEPALIVE",0)) != NULL ) {
    keepalive = (char *)malloc(strlen(p)+1);
    strcpy(keepalive,p);
  }
  /* 
   * Get the list of privhosts, if any
   */
  *privhosts = '\0';
  if ( (p = getconfent("RFIOD","PRIVHOSTS",1)) != NULL )
    strcpy(privhosts,p);
  /*
   * Moved from ropen() because getconfent not re-entrant
   */
  if ( (p=getconfent ("RTUSER","CHECK",0) ) != NULL ) {
    rtuser = (char *)malloc(strlen(p)+1);
    strcpy(rtuser,p);
  }
  /*
   * Moved from rread_v3 because of getconfent not re-entrant
   */
  if ((p = getconfent("RFIO","DAEMONV3_RDSIZE",0)) != NULL)
    {
      if (atoi(p) > 0)
	DISKBUFSIZE_READ = atoi(p);
    }
  /*
   * Moved from rwrite_v3 because of getconfent not re-entrant
   */
  if ((p = getconfent("RFIO","DAEMONV3_WRSIZE",0)) != NULL)
    {
      if (atoi(p) > 0)
	DISKBUFSIZE_WRITE = atoi(p);
    }
  /*
   * Start thread to update credentials.
   */
  status = hpss_SetLoginContext(PrincipalName,KeyTabFile);
  if (status) {
    perror("hpss_SetLoginContext");
    exit(1);
  }
  /*
   * HPSS client API initialization.
   */
  rhpss_doit(NULL);
  /*
   * Mutex inits. for UNIX file access
   */
  runix_gethandle(0,NULL,-1,-1);
  /*
   * Initialize root dir handle
   */
  rhpss_gethandle(-1,NULL,-1,-1);
  
  if ( (status = pthread_attr_create(&attr)))
    DCE_ERROR(pthread_attr_create(),status);
  else {
    if ( (status = pthread_attr_setinheritsched(&attr, PTHREAD_DEFAULT_SCHED)) ) {
      DCE_ERROR(pthread_attr_setinheritsched(),status);
    }
  }
#if defined(__osf__) && defined(__alpha)
  if ( !status ) {
    /*
     * Need to increase the thread stack on DUNIX
     */
    thr_stack_size = 256*1024;
    if ( (status = pthread_attr_setstacksize(&attr,thr_stack_size)) ) {
      DCE_ERROR(pthread_attr_setstacksize(),status);
    }
  }
#endif /* __osf__ && __alpha */
  /*
   * Create thread data array
   */
  *threadData = NULL;
  *nb_threads = MAX_THREADS;
  if ((p = getconfent("RFIO","MAX_THREADS",0)) != NULL)
    {
      if (atoi(p) > 0)
	*nb_threads = atoi(p);
    }
  *threadData = (struct rfio_threadData *)malloc(*nb_threads*sizeof(struct rfio_threadData));
  if ( *threadData == NULL ) {
    log(LOG_ERR,"rhpss_init(): cannot allocate threadData array, %s\n",sys_errlist[errno]);
    exit(1);
  }

  /*
   * Initialize threadData mutexes and conds and kick off the workers
   */
  td = *threadData;
  for (i=0;i<*nb_threads;i++) {
    memset(td,'\0',sizeof(struct rfio_threadData));
    if ( (status = pthread_mutex_init(&td->inuse,
				      pthread_mutexattr_default)) ) {
      DCE_ERROR(pthread_mutex_init(),status);
      exit(1);
    }
    if ( (status = pthread_cond_init(&td->start,
				     pthread_condattr_default)) ) {
      DCE_ERROR(pthread_cond_init(),status);
      exit(1);
    }

    log(LOG_INFO,"Starting thread %d out of %d\n",i,*nb_threads-1);
    if ( (status = pthread_create((pthread_t *)&td->id,
#if defined(__osf__) && defined(__alpha)
				  attr,
#else
				  (pthread_attr_t)attr,
#endif
				  (pthread_startroutine_t)rhpss_doit,
				  (pthread_addr_t)td) ) ) {
      DCE_ERROR(pthread_create(),status);
      exit(1);
    }
    pthread_yield();
    td++;
  } /* for(...) */
  if ( (status = pthread_attr_delete(&attr)) ) {
    DCE_ERROR(pthread_attr_delete(),status);
  }
  return(0);
}

int rhpss_startreq(ns,from,fromlen,privhosts,threadData,nb_threads)
int ns;
struct sockaddr_in *from;
unsigned long fromlen;
char *privhosts;
struct rfio_threadData threadData[];
int nb_threads;
{
  static int last = 0;
  int i, _is_remote, status, rc;
  char privhosts_r[MAX_PRIVHOST_LEN],*privhost;
  char _from_host[MAXHOSTNAMELEN];
#if ! (defined(_AIX) || ( defined(__osf__) && defined(__alpha) ) )
  struct hostent *hp;
#endif
  struct hostent HostentPointer;
  struct hostent_data HostentData;

  if ( from == NULL || fromlen == 0 || threadData == NULL ) return(-1);
  /*
   * Moved from doit() because setnetio is not re-entrant
   */
  if (setnetio(ns) <0) {
    log(LOG_ERR,"setnetio(): %s\n",sys_errlist[errno]);
    return(-1);
  }
  /*
   * Get hostname and check if it is a privileged host
   */
  privhost = NULL;
  *_from_host = '\0';
#if defined(_AIX) || ( defined(__osf__) && defined(__alpha) ) 
  memset(&HostentData,'\0',sizeof(struct hostent_data));
  if ( gethostbyaddr_r((char *)(&from->sin_addr),
		       sizeof(struct in_addr),
		       from->sin_family,
		       &HostentPointer,
		       &HostentData) ) {
#else
  hp = gethostbyaddr((char *)from->sin_addr,sizeof(struct in_addr),
	             from->sin_family);
  if ( hp != NULL ) memcpy(&HostentPointer,hp,sizeof(struct hostent));
  if ( hp == NULL ) {
#endif
    log(LOG_ERR,"gethostbyaddr(_r)(): h_errno=%d %s\n",h_errno,
	sys_errlist[errno]);
    strcpy(_from_host,(char *)inet_ntoa(from->sin_addr));
    log(LOG_INFO, "connection from %s\n", inet_ntoa(from->sin_addr));
  } else {
    /*
     * isremote is non re-entrant but we only
     * call it in main thread
     */
    strcpy(_from_host,HostentPointer.h_name);
    log(LOG_INFO, "connection from %s\n", HostentPointer.h_name);
    _is_remote = 0;
    if ( isremote(from->sin_addr,HostentPointer.h_name) ) _is_remote = 1;
    if ( *privhosts != '\0' ) {
      HostentPointer.h_name = strtok(HostentPointer.h_name,".");
      strcpy(privhosts_r,privhosts);
      privhost = strtok(privhosts_r," ");
      while (privhost != NULL && 
	     strcmp(HostentPointer.h_name,privhost)) 
	privhost = strtok(NULL," ");
    }
  }
  /* 
   * Try to find a free worker. Best chances at last + 1
   */
  i=last;
  for (;;) {
    i++;
    i = i * ( 1 - i / nb_threads ) ;
    log(LOG_DEBUG,"test thread %d (last=%d) out of %d\n",i,last,nb_threads);
    if ( (status = pthread_mutex_trylock((pthread_mutex_t *)&threadData[i].inuse)) < 0 ||
	 status > 1 ) {
      DCE_ERROR(pthread_mutex_trylock(),status);
    } else if ( status == 0 ) {
      log(LOG_DEBUG,"thread %d in use\n",i);
    } else if ( status == 1 ) {
      if ( threadData[i].ns == -1 ) break;
      log(LOG_ERR,"Thread not ready: threadData[%d].ns = %d\n",i,threadData[i].ns);
      if ( (status = pthread_mutex_unlock((pthread_mutex_t *)&threadData[i].inuse)) ) 
	DCE_ERROR(pthread_mutex_unlock(),status);
    }
    if ( i == last ) {
      log(LOG_INFO,"exhausted threadpool (size=%d)\n",nb_threads);
      errno = EAGAIN;
      return(-1);
    }
  }
  last = i;
  threadData[i].ns = ns;
  threadData[i].mode = 0;
  threadData[i]._is_remote = _is_remote;
  memcpy(&threadData[i].from,(void *)from,fromlen);
  strcpy(threadData[i]._from_host,_from_host);
  /*
   * Start the worker
   */
  rc = 0;
  if ( (status = pthread_cond_signal((pthread_cond_t *)&threadData[i].start)) < 0 ) {
    DCE_ERROR(pthread_cond_signal(),status);
    rc = -1;
  }
  if ( (status = pthread_mutex_unlock((pthread_mutex_t *)&threadData[i].inuse)) < 0 ) {
    DCE_ERROR(pthread_mutex_unlock(),status);
    rc = -1;
  }
  if ( !rc ) pthread_yield();
  return(rc);
}

void *rhpss_doit(threadData)
struct rfio_threadData *threadData;
{
  int status;
  extern struct global_defs global[FD_SETSIZE];

  /*
   * Initialization
   */
  if ( threadData == NULL ) memset(&global[0],'\0',sizeof(struct global_defs)*FD_SETSIZE);
  else {
    if ( (status = pthread_mutex_lock(&threadData->inuse)) < 0 ) {
      DCE_ERROR(pthread_mutex_lock(),status);
      return((void *)&success);
    }

    for (;;) {
      threadData->ns = -1;
      /*
       * Wait for the start signal
       */
      while (threadData->ns < 0 ) {
	if ( (status = pthread_cond_wait(&threadData->start,&threadData->inuse)) < 0 ) {
	  DCE_ERROR(pthread_cond_wait(),status);
	  return((void *)&success);
	}
      }
      if ( threadData->_dirhandle != NULL ) 
	memset(threadData->_dirhandle,'\0',sizeof(ns_ObjHandle_t));
      if ( threadData->_user_cred != NULL ) 
	memset(threadData->_user_cred,'\0',sizeof(hsec_UserCred_t));
      if ( threadData->_dirent_cache != NULL ) 
	memset(threadData->_dirent_cache,'\0',DIRENT_CACHESIZE*sizeof(ns_DirEntry_t));
      *threadData->_current_dir = '\0';
      /* Save thread specific data in global */
      global[threadData->ns]._is_remote = threadData->_is_remote;
      global[threadData->ns]._data_s    = -1;
      global[threadData->ns]._data_sock = -1;
      global[threadData->ns]._ctrl_sock = -1;
      global[threadData->ns]._first_read = 0;
      global[threadData->ns]._first_write = 0;
      global[threadData->ns]._byte_read_from_network = 0;
      global[threadData->ns]._from_host = threadData->_from_host;
      global[threadData->ns]._rqstbuf = threadData->_rqstbuf;
      global[threadData->ns]._filename = threadData->_filename;
      global[threadData->ns]._iobuffer = threadData->_iobuffer;
      global[threadData->ns]._iobufsiz = threadData->_iobufsiz;
      global[threadData->ns]._current_dir = threadData->_current_dir;
      global[threadData->ns]._dirhandle = threadData->_dirhandle;
      global[threadData->ns]._user_cred = threadData->_user_cred;
      global[threadData->ns]._dirent_cache = threadData->_dirent_cache;
      global[threadData->ns]._current_dirent = NULL;
      log(LOG_INFO,"rhpss_doit(): running doit(s=%d,mode=%d)\n",threadData->ns,threadData->mode);
      /*
       * Now, doit
       */
      doit(threadData->ns,&threadData->from,threadData->mode);
      log(LOG_INFO,"rhpss_doit(): doit(s=%d,mode=%d) returned\n",threadData->ns,threadData->mode);
      /*
       * Give back the I/O memory buffer (malloc'ed in _v3 routines).
       */
      threadData->_iobuffer = global[threadData->ns]._iobuffer;
      threadData->_iobufsiz = global[threadData->ns]._iobufsiz;
      /*
       * Free the memory (in principle not necessary). 
       */
      free(threadData->_iobuffer);
      threadData->_iobuffer = NULL;
      threadData->_iobufsiz = 0;
      /*
       * Give back directory handle, dirent cache and user creds allocated in rhpss_gethandle()
       */
      threadData->_dirhandle = global[threadData->ns]._dirhandle;
      threadData->_dirent_cache = global[threadData->ns]._dirent_cache;
      threadData->_user_cred = global[threadData->ns]._user_cred;
       
      if ( global[threadData->ns]._data_s >= 0 ) {
	shutdown(global[threadData->ns]._data_s,2); 
	close(global[threadData->ns]._data_s);
      } 
      if ( global[threadData->ns]._data_sock >= 0 ) {
	shutdown(global[threadData->ns]._data_sock,2); 
	close(global[threadData->ns]._data_sock);
      }
      shutdown(threadData->ns,2); 
      close(threadData->ns);
      log(LOG_DEBUG,"rhpss_doit(s=%d): connection shutdown.\n",threadData->ns);
    }
  }
  return((void *)&success);
}	
/*
 * cleanup file descriptor and directory pointers if necessary
 */
int rhpss_cleanup(int s,int *fd, DIR *dirp,int status) {
  if ( *fd >= 0 ) {
    if ( global[s]._is_unix == 1 ) (void) close(*fd);
    else (void) hpss_Close(*fd);
  }
  *fd = -1;
  if ( dirp != NULL ) (void) rhpss_closedir(dirp,s,0,0);
  dirp = NULL;
  (void) rhpss_freehandle(s);
  return(status);
}

/************************************************************************/
/*                                                                      */
/*                          HPSS MISC.                                  */
/*                                                                      */
/************************************************************************/
int rhpss_get_jid(ns) {
  char *p;
  pthread_t me;
  me = pthread_self();
  if ( ns < 0 ) {
    if ((p = getconfent("ACCT", "RFIO", 0)) == NULL ||
        (strcmp (p, "YES") && strcmp (p, "yes"))) return(ns);
  }
  return((int)pthread_getunique_np(&me));
}

int runix_hpsspath(path)
char *path;
{
  int dir_desc;
  struct dirent dir_entry;
  int fd;

  static char buffer[BUFSIZ],**HPSSRootDirs;
  static int nb_HPSSRootDirs = 0;
  int rc,i,status;
  
  if ( path == NULL ) {
    memset(buffer,'\0',sizeof(buffer));
    dir_desc = hpss_Opendir("/");
    if ( dir_desc < 0 ) {
      log(LOG_ERR,"hpss_Opendir(/): %s\n",sys_errlist[-dir_desc]);
      exit(1);
    }
    rc = 0;
    while ( ( status = hpss_Readdir(dir_desc,&dir_entry) ) == 0 ) {
      if ( dir_entry.d_namlen == 0 && *dir_entry.d_name == '\0' ) break;
      if ( *dir_entry.d_name == '.' ) continue;
      if ( rc > BUFSIZ - dir_entry.d_namlen-1 ) {
	log(LOG_ERR,"runix_hpsspath(): out of buffer space\n");
	exit(1);
      }
      strcpy(buffer+rc,"/");
      rc++;
      strcpy(buffer+rc,dir_entry.d_name);
      rc += dir_entry.d_namlen + 1;
      nb_HPSSRootDirs++;
    }
    status = hpss_Closedir(dir_desc);
    HPSSRootDirs = (char **)malloc(nb_HPSSRootDirs*sizeof(char *));
    rc = 0;
    for (i=0;i<nb_HPSSRootDirs;i++) {
      HPSSRootDirs[i] = buffer + rc;
      rc += strlen(HPSSRootDirs[i])+1;
      log(LOG_INFO,"Detected HPSS root directory %s\n",HPSSRootDirs[i]);
    }
    return(0);
  }
  rc = 1;
  for (i=0; i<nb_HPSSRootDirs; i++) {
    if (!strncmp(HPSSRootDirs[i],path,strlen(HPSSRootDirs[i]))) {
      rc = 0;
      break;
    }
  }

  return ( rc );
}
  
int runix_gethandle(s,path,uid,gid)
int s;
char *path;
uid_t uid;
gid_t gid;
{
  static pthread_mutex_t euid_handle;
  static uid_t saved_euid;
  static gid_t saved_egid;
  int status;

  if ( s == 0 && path == NULL && uid == -1 ) {
    /*
     * Initialization, called once from main before creating threads
     */
    if ( (status = pthread_mutex_init(&euid_handle,pthread_mutexattr_default)) ) {
      DCE_ERROR(pthread_mutex_init(),status);
      exit(1);
    }
#if defined(_AIX)
    if ( setgroups(0,NULL) ) {
      log(LOG_ERR,"setgroups(): %s\n",sys_errlist[errno]);
    }
#endif /* _AIX */
    saved_euid = geteuid();
    saved_egid = getegid();
    runix_hpsspath(NULL);
    return(0);
  }
  if ( uid == -1 ) {
    /*
     * Privileged operation finished, reset saved effective UID and unlock mutex
     */
    if ( setegid(saved_egid) < 0 || seteuid(saved_euid) < 0 ) {
      log(LOG_ERR,"seteuid(%d)/setegid(%d): %s\n",saved_euid,saved_egid,sys_errlist[errno]);
      if ( (status = pthread_mutex_unlock(&euid_handle)) ) {
	DCE_ERROR(pthread_mutex_unlock(euid_handle),status);
      }
      return(-1);
    }
    if ( (status = pthread_mutex_unlock(&euid_handle)) ) {
      DCE_ERROR(pthread_mutex_unlock(euid_handle),status);
      return(-1);
    }
    return(0);
  }
  global[s]._is_unix = 0;
  log(LOG_DEBUG,"Check for UFS on path=%s\n",path);
  if ( runix_hpsspath(path) ) {
    /*
     * Privileged operation started, lock mutex and set effective UID.
     */
    if ( (status = pthread_mutex_lock(&euid_handle)) ) {
      DCE_ERROR(pthread_mutex_lock(euid_handle),status);
      return(-1);
    }
    if ( setegid(gid) < 0 || seteuid(uid) < 0 ) {
      log(LOG_ERR,"seteuid(%d)/setegid(%d): %s\n",uid,gid,sys_errlist[errno]);
      if ( (status = pthread_mutex_unlock(&euid_handle)) ) {
	DCE_ERROR(pthread_mutex_unlock(euid_handle),status);
      }
      return(-1);
    }
    log(LOG_DEBUG,"UFS access determined from path=%s, uid=%d\n",path,uid);
    global[s]._is_unix = 1;
    return(1);
  }
  return(0);
}
int runix_freehandle(s,path)
int s;
char *path;
{
  return(runix_gethandle(s,path,-1,-1));
}

int rhpss_gethandle(s,path,uid,gid)
int s;
char *path;              /* Directory path of the object */
uid_t uid;
gid_t gid;
{
  static hpss_fileattr_t topdirattr;
  gss_token_t AuthzTicket;
  hpss_vattr_t AttrOut;
  hsec_ucred_t ucred;
  char *ClientFullName = 0;
  int status;
  
  if ( s < 0 && path == NULL ) {
    /*
     * Initialization: Get "root_uid"
     */
    if ( (status = hsec_GetUcredNam(RFIO_HPSSROOT,&ucred,ClientFullName)) ) {
      log(LOG_ERR,"rhpss_gethandle(): hsec_GetUcredName(%s) return status %d, %s\n",RFIO_HPSSROOT,status,sys_errlist[(status<0?-status : errno)]);
    } else {
      root_uid = ucred->SecPWent.Uid;
      log(LOG_INFO,"rhpss_gethandle(): HPSS ROOT UID is %d\n",root_uid);
      hsec_FreeUcred(ucred);
    }

    /*
     * Get attributes for HPSS top directory "/"
     */
    status = hpss_FileGetAttributes("/",&topdirattr);
    if ( status ) {
      log(LOG_ERR,"rhpss_gethandle(): hpss_FileGetAttributes(/) return status %d, %s\n",
	  status,sys_errlist[(status<0?-status : errno)]);
      return(status);
    }
    return(0);
  }

  if ( dirhandle == NULL ) {
    global[s]._dirhandle=(ns_ObjHandle_t *)malloc(sizeof(ns_ObjHandle_t));
    memset(dirhandle,'\0',sizeof(ns_ObjHandle_t));
  }    
  if ( user_cred == NULL ) {
    global[s]._user_cred=(void *)malloc(sizeof(hsec_UserCred_t));
    memset(user_cred,'\0',sizeof(hsec_UserCred_t));
  }
  if ( s>=0 && (int)uid < 0 ) {
    /*
     * free handle
     */
    memset(dirhandle,'\0',sizeof(ns_ObjHandle_t));
    memset(user_cred,'\0',sizeof(hsec_UserCred_t));
    return(0);
  }

  /*
   * Get user credentials if necessary
   */
  if ( uid != (uid_t)user_cred->SecPWent.Uid || gid != (gid_t)user_cred->SecPWent.Gid ) {
    log(LOG_DEBUG,"Get user credentials for uid=%d, gid=%d\n",uid,gid);
    status = hsec_GetUcredUid(uid,&ucred,ClientFullName);    
    if ( status ) {
      log(LOG_ERR,"rhpss_gethandle(%d,%s,%d,%d): hsec_GetUcredUid() return status %d, %s\n",s,
	  path,uid,gid,status,sys_errlist[-status]);
      return(status);
    }
    ucred->SecPWent.Gid = (int)gid;
    ConvertUcredToUserCred(ucred,user_cred);
    user_cred->SecPWent.Gid = gid;
    hsec_FreeUcred(ucred);
    log(LOG_DEBUG,"Returned user credentials are uid=%d, gid=%d\n",user_cred->SecPWent.Uid,
	user_cred->SecPWent.Gid);
  } else log(LOG_DEBUG,"Use cached user credentials for uid=%d, gid=%d\n",uid,gid);

  /*
   * Get handle if needed
   */
  if ( strcmp(path,current_dir) ) {
    log(LOG_DEBUG,"Get attribute handle for directory=%s, uid=%d\n",path,uid);
    status = hpss_GetAttrHandle(&topdirattr.NSObjectHandle,path,user_cred,dirhandle,
				&AuthzTicket,&AttrOut);
    if ( status ) {
      log(LOG_ERR,"hpss_GetAttrHandle(%s,%d): status=%d, %s\n",path,uid,status,
	  sys_errlist[(status<0?-status : errno)]);
      return(status);
    }
    strcpy(current_dir,path);
  } else log(LOG_DEBUG,"Use cached directory handle for directory=%s, uid=%d\n",path,uid);
  return(0);
}
int rhpss_freehandle(s)
int s;
{
  log(LOG_DEBUG,"rhpss_freehandle(%d)\n",s);
  return(rhpss_gethandle(s,NULL,(uid_t)-1,(gid_t)-1));
}

static char *rhpss_splitpath(path,CwdName,basename)
char *path;
char **CwdName,**basename;
{
  static char RootDir[] = "/";
  char *p = NULL;
  if ( path == NULL ) return(NULL);
  if ( (p = strrchr(path,'/')) == NULL || p == path ) {
    *CwdName = RootDir;
    if ( p != NULL ) *basename = path+1;
    else *basename = path;
    return(NULL);
  } else {
    *p = '\0';
    *CwdName = path;
    *basename = p+1;
    return(p);
  }
  return(NULL);
}

static int rhpss_restorepath(p)
char *p;
{
  if ( p != NULL ) *p = '/';
  return(0);
}

/*
 * Non-privileged calls
 */
mode_t rhpss_umask(mode,s,uid,gid)
mode_t mode;
int s;
uid_t uid;
gid_t gid;
{
  if ( global[s]._is_unix == 1 ) {
    return(umask(mode));
  } else {
    return(hpss_Umask(mode));
  }
}


off_t rhpss_lseek(fd,offset,how,s,uid,gid)
int fd;
off_t offset;
int how;
int s;
uid_t uid;
gid_t gid;
{
  off_t status;
  if ( global[s]._is_unix == 1 ) status = lseek(fd,offset,how);
  else {
    status= hpss_Lseek(fd,offset,how);
    if ( status < 0 ) status = -1;
  }
  return(status);
}

int rhpss_close(fd,s,uid,gid)
int fd;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  if ( global[s]._is_unix == 1 ) status = close(fd);
  else {
    status = hpss_Close(fd);
    if ( status < 0 ) status = -1;
  }
  return(status);
}

ssize_t rhpss_read(fd,buffer,nb,s,uid,gid)
int fd;
char *buffer;
size_t nb;
int s;
uid_t uid;
gid_t gid;
{
  ssize_t status;
  if ( global[s]._is_unix == 1 ) status = read(fd,buffer,nb);
  else {
    status = hpss_Read(fd,buffer,nb);
    if ( status < 0 ) status = -1;
  }
  return(status);
}
ssize_t rhpss_write(fd,buffer,nb,s,uid,gid)
int fd;
char *buffer;
size_t nb;
int s;
uid_t uid;
gid_t gid;
{
  ssize_t status;
  if ( global[s]._is_unix == 1 ) status = write(fd,buffer,nb);
  else {
    status = hpss_Write(fd,buffer,nb);
    if ( status < 0 ) status = -1;
  }
  return(status);
}

FILE *rhpss_popen(cmd,type,s,uid,gid)
char *cmd;
char *type;
int s;
uid_t uid;
gid_t gid;
{
  FILE *fs;
  if ( runix_gethandle(s,cmd,uid,gid) ) {
    fs = popen(cmd,type);
    runix_freehandle(s,cmd);
  } else {
    /*
     * Currently not supported for HPSS
     */
    fs = NULL;
    errno = EACCES;
  }
  return(fs);
}

/*
 * Privileged calls
 */
int rhpss_chmod(hpss_filename, mode, s, uid,gid)
char *hpss_filename;
long mode;
int s;
uid_t uid;
gid_t gid;
{
  hpss_vattr_t            Attrs,AttrsOut;      /* OUT - File attributes */
  gss_token_t             AuthzTicket;   /* OUT - Client authorization */
  int status;
  char *CwdName, *basename, *p;

  u_signed64 flagin,flagout;

  if ( runix_gethandle(s,hpss_filename,uid,gid) > 0 ) {
    status = chmod(hpss_filename,mode);
    runix_freehandle(s,hpss_filename);
    return(status);
  }

  memset(&Attrs,'\0',sizeof(hpss_vattr_t));
  memset(&AttrsOut,'\0',sizeof(hpss_vattr_t));
  Attrs.va_mode = (hpss_mode_t)mode;
  flagin = cast64m(0);
  flagin = orbit64m(flagin,VA_MODE);
  flagout = cast64m(0);

  p = rhpss_splitpath(hpss_filename,&CwdName,&basename);
  if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(p); 
    return(-1);
  }
  status = hpss_SetAttrHandle(dirhandle,basename,user_cred,flagin,&Attrs,&flagout,&AttrsOut);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(p);
  if ( status ) {
    log(LOG_ERR,"rhpss_chmod(%s,%d): hpss_SetAttrHandle() (%d,%d), status=%d, %s\n",
	hpss_filename,mode,uid,gid,status,sys_errlist[(status<0?-status : errno)]);
    if ( status < 0 ) errno = -status;
    return(-1);
  }
  return(0);
}

int rhpss_chown(hpss_filename, owner, group, s, uid,gid)
char *hpss_filename;
int owner;
int group;
int s;
uid_t uid;
gid_t gid;
{
  hpss_vattr_t            Attrs,AttrsOut;      /* OUT - File attributes */
  gss_token_t             AuthzTicket;   /* OUT - Client authorization */
  int status;
  char *CwdName, *basename, *p;
  u_signed64 flagin,flagout;

  if ( runix_gethandle(s,hpss_filename,uid,gid) > 0 ) {
    status = chown(hpss_filename,owner,group);
    runix_freehandle(s,hpss_filename);
    return(status);
  }

  memset(&Attrs,'\0',sizeof(hpss_vattr_t));
  memset(&AttrsOut,'\0',sizeof(hpss_vattr_t));
  Attrs.va_uid = (hpss_uid_t)owner;
  Attrs.va_gid = (hpss_gid_t)group;
  flagin = cast64m(0);
  flagin = orbit64m(flagin,VA_UID);
  flagin = orbit64m(flagin,VA_GID);
  flagout = cast64m(0);

  p = rhpss_splitpath(hpss_filename,&CwdName,&basename);
  if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(p); 
    return(-1);
  }
  status = hpss_SetAttrHandle(dirhandle,basename,user_cred,flagin,&Attrs,&flagout,&AttrsOut);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(p);
  if ( status ) {
    log(LOG_ERR,"rhpss_chown(%s,%d,%d): hpss_SetAttrHandle() (%d,%d), status=%d, %s\n",
	hpss_filename,owner,group,uid,gid,status,sys_errlist[(status<0?-status : errno)]);
    if ( status < 0 ) errno = -status;
    return(-1);
  }
  return(0);
}

int rhpss_access(hpss_filename, mode, s, uid,gid)
char *hpss_filename;
int mode;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  char *CwdName, *basename, *p;

  if ( runix_gethandle(s,hpss_filename,uid,gid) > 0 ) {
    status = ( access(hpss_filename,mode) < 0 ) ? errno : 0;
    runix_freehandle(s,hpss_filename);
    return(status);
  }

  p = rhpss_splitpath(hpss_filename,&CwdName,&basename);
  if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(p); 
    return(-1);
  }
  status = hpss_AccessHandle(dirhandle,basename,mode,user_cred,NULL);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(p);
  if ( status ) {
    log(LOG_ERR,"access() %s, status=%d, %s\n",hpss_filename,
	status,sys_errlist[(status<0?-status : errno)]);
    if ( status < 0 ) errno = -status;
    return(-1);
  }
  return(0);
}

int rhpss_lockf(fd, op, size, s, uid, gid)
int fd;
int op;
long size;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  char *CwdName, *basename, *p;

  if ( global[s]._is_unix == 1 ) {
    status = ( lockf(fd,op,size) < 0 ) ? errno : 0;
    return(status);
  } else {
    log(LOG_ERR,"lockf currently not implemented for HPSS files\n");
    errno = EACCES;
    return(-1);
  }
}


int rhpss_open(hpss_filename, flags, mode, s, uid, gid)
char *hpss_filename;
int flags;
mode_t mode;
int s;
uid_t uid;
gid_t gid;
{
  hpss_cos_hints_t        HintsOut;      /* OUT - Granted class of service */
  hpss_vattr_t            AttrsOut;      /* OUT - File attributes */
  gss_token_t             AuthzTicket;   /* OUT - Client authorization */
  int fd, status;
  char *CwdName, *basename, *p;
  int local_flags = flags;

  if ( runix_gethandle(s,hpss_filename,uid,gid) > 0 ) {
    fd = open(hpss_filename,flags,mode);
    runix_freehandle(s,hpss_filename);
    return(fd);
  }
     
  p = rhpss_splitpath(hpss_filename,&CwdName,&basename);
  if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(p); 
    return(-1);
  }

  if ( local_flags && O_CREAT ) {
  /*
   * Create the file. Needed because of bug in hpss_OpenHandle()
   */
    memset((char *)&AttrsOut,0,sizeof(hpss_vattr_t));
    status = hpss_CreateHandle(dirhandle,basename,mode,user_cred,
			       NULL,NULL,NULL,&AttrsOut,NULL);
    if ( status && status != -EEXIST ) {
      log(LOG_ERR,"Cannot create HPSS file %s/%s, status=%d, %s\n",CwdName,
	  basename,status,sys_errlist[(status<0?-status : errno)]);
      (void) rhpss_freehandle(s);
      (void) rhpss_restorepath(p);
      if ( status < 0 ) errno = -status;
      return(-1);
    }
  }

  local_flags &= ~O_CREAT;

  /*
   * Open the file
   */
  fd= hpss_OpenHandle(dirhandle,basename,local_flags, mode, user_cred,
		      NULL, NULL,&HintsOut,&AttrsOut,&AuthzTicket);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(p);
  return(fd);
}

DIR *rhpss_opendir(hpss_filename,s,uid,gid)
char *hpss_filename;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  DIR *dirp;

  dirp = NULL;
  status = -1;
  if ( runix_gethandle(s,hpss_filename,uid,gid) > 0 ) {
    dirp = opendir(filename);
    runix_freehandle(s,hpss_filename);
    return(dirp);
  }

  /*
   * Get handle. Note that handle cannot be released here since it is needed for the
   * subsequent readdir()'s.
   */
  if ( (status = rhpss_gethandle(s,hpss_filename,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",hpss_filename,
	s,uid,gid,status,sys_errlist[errno]); 
    return(NULL);
  }

  /*
   * Save some thread global information for this directory to speed up
   * readdir()'s and stat()'s.
   */

  dir_offset = 0;
  if ( dirent_cache == (ns_DirEntry_t *)NULL ) 
    global[s]._dirent_cache = (void *)malloc(DIRENT_CACHESIZE*sizeof(ns_DirEntry_t));
  if ( dirent_cache == (ns_DirEntry_t *)NULL ) {
    log(LOG_ERR,"Cannot allocate direntry cache, %s\n",sys_errlist[errno]);
    errno = ENOMEM;
    return(NULL);
  }
  global[s]._current_dirent = NULL;

  dirp = (DIR *)malloc(sizeof(DIR));
  /*
   * Reserv large enough buffer to hold the dirent returned by hpss_ReaddirHandle()
   */
  if ( dirp != NULL ) {
    dirp->dd_buf = (char *)malloc(sizeof(struct dirent) + HPSS_MAX_FILE_NAME );
    /*
     * There is no "real" directory descriptor here but rather an object handle
     * addressed by the accept socket number "s".
     */
    if ( dirp->dd_buf != NULL ) {
      dirp->dd_fd = s;
      memset(dirp->dd_buf,'\0',sizeof(struct dirent) + HPSS_MAX_FILE_NAME);
    } else {
      free(dirp);
      dirp = NULL;
    }
  }
  if ( dirp == NULL ) {
    errno = ENOMEM;
  }
  return(dirp);
}

struct dirent *rhpss_readdir(dirp,s,uid,gid)
DIR *dirp;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  unsigned long dirbufsiz,end,getattr;
  struct dirent *dp = NULL;

  if ( global[s]._is_unix == 1 ) {
    dp = readdir(dirp);
  } else {
    dp = (struct dirent *)dirp->dd_buf;
    if ( dp == NULL ) {
      errno = ENOMEM;
    } else {
      if ( dirp->dd_fd != s ) {
	errno = EBADF;
      } else {
	log(LOG_DEBUG,"rhpss_readdir(0x%x,%d,%d,%d) called\n",dirp,s,uid,gid);
	/*
	 * Check if current offset already in cache 
	 */
	if ( current_dirent != NULL && current_dirent->Next != NULL ) {
	  log(LOG_DEBUG,"rhpss_readdir(0x%x,%d,%d,%d) next entry in cache\n",dirp,s,uid,gid);
	  global[s]._current_dirent = (void *)(current_dirent->Next);
	} else {
	  log(LOG_DEBUG,"rhpss_readdir(0x%x,%d,%d,%d) cache in new dir-entries\n",dirp,s,uid,gid);
	  dirbufsiz = DIRENT_CACHESIZE * sizeof(ns_DirEntry_t);
	  getattr = TRUE;
	  status = hpss_ReadAttrsHandle(dirhandle,dir_offset,user_cred,dirbufsiz,getattr,&end,&dir_offset,dirent_cache);
	  if ( status < 0 ) {
	    log(LOG_ERR,"hpss_ReadAttrHandle(%s): status=%d, %s\n",
		(current_dir!=NULL ? current_dir : ""),status,sys_errlist[-status]);
	    errno = -status;
	    dp = NULL;
	  }
	  global[s]._current_dirent = (void *)dirent_cache;
	}
	if ( dp != NULL ) {
	  strncpy(dp->d_name,current_dirent->Name,HPSS_MAX_FILE_NAME);
	  dp->d_namlen = strlen(current_dirent->Name);
	}
      }
    }
  }
  return(dp);
}

int rhpss_rewinddir(dirp,s,uid,gid)
DIR *dirp;
int s;
uid_t uid;
gid_t gid;
{
  unsigned long offset;
  int status;
  status = 0;
  if ( global[s]._is_unix == 1 ) {
    (void)rewinddir(dirp);
  } else {
    if ( dirp->dd_fd != s ) {
      errno = EBADF;
      status = -1;
    } else {
      dir_offset = 0;
      global[s]._current_dirent = NULL;
    }
  }
  return(status);
}

int rhpss_closedir(dirp,s,uid,gid)
DIR *dirp;
int s;
uid_t uid;
gid_t gid;
{
  int status,offset;
  status = 0;
  if ( global[s]._is_unix == 1 ) {
    status = closedir(dirp);
  } else {
    free(dirp->dd_buf);
    free(dirp);
    (void) rhpss_freehandle(s);
  }
  return(status);
}

int rhpss_fstat(fd,st,s,uid,gid)
int fd;
struct stat *st;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  if ( global[s]._is_unix == 1 ) status = fstat(fd,st);
  else {
    status= hpss_Fstat(fd, st) ;
    /*
     * These entries does not make sens for HPSS.
     * Client may use this pattern to distinguish HPSS from UNIX files
     */
    st->st_dev = 0; st->st_ino = 1;
  }
  return(status);
}
int rhpss_lstat(hpss_filename,st,s,uid,gid)
char *hpss_filename;
struct stat *st;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  ns_ObjHandle_t handle;
  gss_token_t token;
  hpss_vattr_t vattr;
  char *CwdName,*basename,*p;
  
  if ( runix_hpsspath(hpss_filename) )
    status = lstat(hpss_filename,st);
  else {
    if ( uid == 0 && gid == 0 ) {
      /*
       * This is an old stat client... Run on behalf of rfiod principal
       */
      status = hpss_Lstat(hpss_filename,st);
    } else {
      status = 0;
      p = rhpss_splitpath(hpss_filename,&CwdName,&basename);
      if ( current_dirent == NULL || 
	   strcmp(current_dir,CwdName) ||
	   strcmp(current_dirent->Name,basename) ) {
	/*
	 * No (or wrong) cached direntry. Do a normal stat.
	 */
	log(LOG_DEBUG,"rhpss_lstat(): No cached entries for s=%d, %s/%s, uid=%d\n",
	    s,CwdName,basename,uid);
	if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
	  log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	      s,uid,gid,status,sys_errlist[errno]); 
	  (void) rhpss_restorepath(p); 
	  return(-1);
	}

	status = hpss_GetAttrHandle(dirhandle,basename,user_cred,&handle,
				    &token,&vattr);
	(void) rhpss_restorepath(p);
	if ( status ) {
	  log(LOG_ERR,"hpss_GetAttrHandle(%s,%d): status=%d, %s\n",hpss_filename,uid,
	      status,sys_errlist[(status<0?-status : errno)]);
	  if ( status < 0 ) errno = -status;
	  return(-1);
	}
	st->st_mode = vattr.va_mode;
	st->st_nlink = (short) vattr.va_nlink;
	st->st_uid = (ushort) vattr.va_uid;
	st->st_gid = (ushort) vattr.va_gid;
	st->st_atime = vattr.va_atime;
	st->st_mtime = vattr.va_mtime;
	st->st_ctime = vattr.va_ctime;
	st->st_rdev = (dev_t) 0;
	st->st_size = low32m(vattr.va_size);
	st->st_blksize = low32m(vattr.va_blocksize);
	st->st_blocks = low32m(vattr.va_numblocks);
      } else {
	/*
	 * Use cached direntry
	 */
	(void) rhpss_restorepath(p);
	log(LOG_DEBUG,"rhpss_lstat(): use cached entries for s=%d, %s, uid=%d\n",
	    s,hpss_filename,uid);
	ConvertHpnsModeToPosixMode(&current_dirent->Attrs,&st->st_mode);
	st->st_nlink = (short) current_dirent->Attrs.LinkCount;
	st->st_uid = (ushort) current_dirent->Attrs.UID;
	st->st_gid = (ushort) current_dirent->Attrs.GID;
	st->st_atime = current_dirent->Attrs.TimeLastRead;
	st->st_mtime = current_dirent->Attrs.TimeLastWritten;
	st->st_ctime = current_dirent->Attrs.TimeOfMetadataUpdate;
	st->st_rdev = (dev_t) 0;
	st->st_size = low32m(current_dirent->Attrs.FileSize);
	st->st_blksize = 4096;
	st->st_blocks = low32m(div64m(add64m(current_dirent->Attrs.FileSize,cast64m(4095)),4096));
      }
    }
    if ( status < 0 ) status = -1;
    /*
     * These entries does not make sens for HPSS.
     * Client may use this pattern to distinguish HPSS from UNIX files
     */
    st->st_dev = 0; st->st_ino = 1;
  }
  return(status);
}

int rhpss_stat(hpss_filename,st,s,uid,gid)
char *hpss_filename;
struct stat *st;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  ns_ObjHandle_t handle;
  gss_token_t token;
  hpss_vattr_t vattr;
  int cnt;
  char *CwdName, *basename, *p;
  char buffer[HPSS_MAX_PATH_NAME];

  if ( runix_hpsspath(hpss_filename) )
    status = stat(hpss_filename,st);
  else {
    /*
     * If we have a HPSS root uid, use it if STATTRUST has been
     * granted for remote root access. Otherwise we try a normal hpss_Stat().
     */
    if ( root_uid != 0 && uid == 0 ) uid = root_uid;
    if ( uid == 0 ) {
      /*
       * This is an old stat client... or a remote root who has
       * passed the STATTRUST. Run on behalf of rfiod principal
       */
      status = hpss_Stat(hpss_filename,st);
    } else {
      status = 0;
      p = rhpss_splitpath(hpss_filename,&CwdName,&basename);
      if ( current_dirent == NULL || 
	   strcmp(current_dir,CwdName) ||
	   strcmp(current_dirent->Name,basename) ||
	   current_dirent->Attrs.Type == SYM_LINK_OBJECT ) {
	/*
	 * No (or wrong) cached direntry or symbolic link which needs to be chased. 
	 * Do a normal stat.
	 */
	log(LOG_DEBUG,"rhpss_stat(): No cached entries for s=%d, %s/%s, uid=%d\n",
	    s,CwdName,basename,uid);
	if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
	  log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	      s,uid,gid,status,sys_errlist[errno]); 
	  (void) rhpss_restorepath(p); 
	  return(-1);
	}

	log(LOG_DEBUG,"call hpss_GetAttrHandle() s=%d, basename=%s, uid=%d\n",s,basename,uid);
	status = hpss_GetAttrHandle(dirhandle,basename,user_cred,&handle,
				    &token,&vattr);
	(void) rhpss_restorepath(p);
	if ( status < 0 ) {
	  log(LOG_ERR,"hpss_GetAttrHandle(%s,%d): status=%d, %s\n",hpss_filename,uid,
	      status,sys_errlist[(status<0?-status : errno)]);
	  if ( status < 0) errno = -status;
	  return(-1);
	}
	cnt = 0;
	strcpy(buffer,(char *)basename);
	while ( vattr.va_type == F_VLNK && cnt < LINK_MAX ) {
	  /*
	   * Chase symbolic link
	   */
	  status = hpss_ReadlinkHandle(dirhandle,buffer,buffer,HPSS_MAX_PATH_NAME,user_cred);
	  if ( status < 0 ) {
	    log(LOG_ERR,"hpss_ReadlinkHandle(%s), status=%d, %s\n",
		buffer,status,sys_errlist[(status<0?-status : errno)]);
	    if ( status < 0) errno = -status;
	    return(-1);
	  }
	  status = hpss_GetAttrHandle(dirhandle,buffer,user_cred,&handle,&token,&vattr);
	  if ( status < 0 ) {
	    log(LOG_ERR,"hpss_GetAttrHandle(%s), status=%d, %s\n",
		buffer,status,sys_errlist[(status<0?-status : errno)]);
	    if (status < 0) errno = -status;
	    return(-1);
	  }
	  cnt++;
	}
	if ( cnt > LINK_MAX ) {
	  errno = ELOOP;
	  return(-1);
	}
	if ( cnt ) log(LOG_DEBUG,"link %s -> %s of depth %d found\n",hpss_filename,buffer,cnt);
	st->st_mode = vattr.va_mode;
	st->st_nlink = (short) vattr.va_nlink;
	st->st_uid = (ushort) vattr.va_uid;
	st->st_gid = (ushort) vattr.va_gid;
	st->st_atime = vattr.va_atime;
	st->st_mtime = vattr.va_mtime;
	st->st_ctime = vattr.va_ctime;
	st->st_rdev = (dev_t) 0;
	st->st_size = low32m(vattr.va_size);
	st->st_blksize = low32m(vattr.va_blocksize);
	st->st_blocks = low32m(vattr.va_numblocks);
      } else {
	/*
	 * Use cached direntry
	 */
	(void) rhpss_restorepath(p);
	log(LOG_DEBUG,"rhpss_stat(): use cached entries for s=%d, %s, uid=%d\n",s,hpss_filename,uid);
	ConvertHpnsModeToPosixMode(&current_dirent->Attrs,&st->st_mode);
	st->st_nlink = (short) current_dirent->Attrs.LinkCount;
	st->st_uid = (ushort) current_dirent->Attrs.UID;
	st->st_gid = (ushort) current_dirent->Attrs.GID;
	st->st_atime = current_dirent->Attrs.TimeLastRead;
	st->st_mtime = current_dirent->Attrs.TimeLastWritten;
	st->st_ctime = current_dirent->Attrs.TimeOfMetadataUpdate;
	st->st_rdev = (dev_t) 0;
	st->st_size = low32m(current_dirent->Attrs.FileSize);
	st->st_blksize = 4096;
	st->st_blocks = low32m(div64m(add64m(current_dirent->Attrs.FileSize,cast64m(4095)),4096));
      }
    }
    if ( status < 0 ) status = -1;
    /*
     * These entries does not make sens for HPSS.
     * Client may use this pattern to distinguish HPSS from UNIX files
     */
    st->st_dev = 0; st->st_ino = 1;
  }
  return(status);
}

int rhpss_unlink(hpss_filename, s, uid, gid)
char *hpss_filename;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  char *CwdName, *basename, *p;
  struct stat st;

  if ( runix_gethandle(s,hpss_filename,uid,gid) > 0 ) {
    status = unlink(hpss_filename);
    runix_freehandle(s,hpss_filename);
    return(status);
  }

  p = rhpss_splitpath(hpss_filename,&CwdName,&basename);
  if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(p); 
    return(-1);
  }

  status = hpss_UnlinkHandle(dirhandle,basename,user_cred);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(p);
  if ( status ) {
    log(LOG_ERR,"Cannot unlink() HPSS file %s/%s, status=%d, %s\n",CwdName,
	basename,status,sys_errlist[(status<0?-status : errno)]);
    if ( status < 0 ) errno = -status;
    return(-1);
  }
  return(0);
}

int rhpss_symlink(hpss_filename, hpss_linkname, s, uid, gid)
char *hpss_filename, *hpss_linkname;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  char *CwdName, *basename, *p;

  if ( runix_gethandle(s,hpss_linkname,uid,gid) > 0 ) {
    status = symlink(hpss_filename,hpss_linkname);
    runix_freehandle(s,hpss_linkname);
    return(status);
  }

  p = rhpss_splitpath(hpss_linkname,&CwdName,&basename);
  if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(p); 
    return(-1);
  }

  status = hpss_SymlinkHandle(dirhandle,hpss_filename,basename,user_cred);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(p);
  if ( status ) {
    log(LOG_ERR,"Cannot create HPSS symbolic link %s -> %s, status=%d, %s\n",hpss_linkname,
	hpss_filename,status,sys_errlist[(status<0?-status : errno)]);
    if ( status < 0 ) errno = -status;
    return(-1);
  }
  return(0);
}

int rhpss_readlink(hpss_linkname, hpss_filename, bufsize, s, uid, gid)
char *hpss_filename, *hpss_linkname;
int bufsize;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  char *CwdName, *basename, *p;

  if ( hpss_linkname == NULL || *hpss_linkname == '\0' ) {
    errno = EINVAL;
    return(-1);
  }
  if ( runix_gethandle(s,hpss_linkname,uid,gid) > 0 ) {
    status = readlink(hpss_linkname,hpss_filename,bufsize);
    runix_freehandle(s,hpss_linkname);
    return(status);
  }

  p = rhpss_splitpath(hpss_linkname,&CwdName,&basename);
  if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(p); 
    return(-1);
  }

  status = hpss_ReadlinkHandle(dirhandle,basename,hpss_filename,bufsize,user_cred);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(p);
  if ( status < 0 ) {
    log(LOG_ERR,"Cannot read HPSS symbolic link %s, status=%d, %s\n",hpss_linkname,
	status,sys_errlist[(status<0?-status : errno)]);
    if ( status < 0 ) errno = -status;
    return(-1);
  }
  return(status);
}

int rhpss_rmdir(hpss_dirname,s,uid,gid)
char *hpss_dirname;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  char *CwdName, *basename, *p;

  if ( runix_gethandle(s,hpss_dirname,uid,gid) > 0 ) {
    status = rmdir(hpss_dirname);
    runix_freehandle(s,hpss_dirname);
    return(status);
  }
  p = rhpss_splitpath(hpss_dirname,&CwdName,&basename);
  if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(p); 
    return(-1);
  }

  status = hpss_RmdirHandle(dirhandle,basename,user_cred);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(p);
  if ( status ) {
    log(LOG_ERR,"Cannot remove HPSS directory %s, status=%d, %s\n",hpss_dirname,
	status,sys_errlist[(status<0?-status : errno)]);
    if ( status < 0 ) errno = -status;
    return(-1);
  }
  return(0);
}

int rhpss_mkdir(hpss_dirname,mode,s,uid,gid)
char *hpss_dirname;
mode_t mode;
int s;
uid_t uid;
gid_t gid;
{
  int status;
  char *CwdName, *basename, *p;

  if ( runix_gethandle(s,hpss_dirname,uid,gid) > 0 ) {
    status = mkdir(hpss_dirname,mode);
    runix_freehandle(s,hpss_dirname);
    return(status);
  }

  p = rhpss_splitpath(hpss_dirname,&CwdName,&basename);
  if ( (status = rhpss_gethandle(s,CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(p); 
    return(-1);
  }

  status = hpss_MkdirHandle(dirhandle,basename,mode,user_cred,NULL,NULL);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(p);
  if ( status ) {
    log(LOG_ERR,"Cannot make HPSS directory %s, status=%d, %s\n",hpss_dirname,
	status,sys_errlist[(status<0?-status : errno)]);
    if ( status < 0 ) errno = -status;
    return(-1);
  }
  return(0);
}

int rhpss_rename(hpss_oldpath,hpss_newpath,s,uid,gid)
char *hpss_oldpath,*hpss_newpath;
int s;
uid_t uid;
{
  ns_ObjHandle_t old_dirhandle;
  int status;
  char *old_CwdName, *old_basename, *old_p;
  char *new_CwdName, *new_basename, *new_p;

  if ( runix_gethandle(s,hpss_oldpath,uid,gid) > 0 ) {
    status = rename(hpss_oldpath,hpss_newpath);
    runix_freehandle(s,hpss_oldpath);
    return(status);
  }

  old_p = rhpss_splitpath(hpss_oldpath,&old_CwdName,&old_basename);
  if ( (status = rhpss_gethandle(s,old_CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",old_CwdName,
	s,uid,gid,status,sys_errlist[errno]); 
    (void) rhpss_restorepath(old_p);
    return(-1);
  }
  memcpy(&old_dirhandle,dirhandle,sizeof(ns_ObjHandle_t));

  new_p = rhpss_splitpath(hpss_newpath,&new_CwdName,&new_basename);
  if ( (status = rhpss_gethandle(s,new_CwdName,uid,gid)) ) {
    log(LOG_ERR,"Cannot get HPSS handle for %s (s=%d,uid=%d,gid=%d), status=%d, %s\n",new_CwdName,
	s,uid,gid,status,sys_errlist[errno]);
    (void) rhpss_freehandle(s);
    (void) rhpss_restorepath(new_p);
    (void) rhpss_restorepath(old_p);
    return(-1);
  }

  status = hpss_RenameHandle(&old_dirhandle,old_basename,
			     dirhandle,new_basename,user_cred);
  (void) rhpss_freehandle(s);
  (void) rhpss_restorepath(old_p);
  (void) rhpss_restorepath(new_p);
  if ( status ) {
    log(LOG_ERR,"Cannot rename HPSS path %s to %s, status=%d, %s\n",
	hpss_oldpath,hpss_newpath,status,sys_errlist[(status<0?-status : errno)]);
    if ( status < 0 ) errno = -status;
    return(-1);
  }
  return(0);
}

int srsetcos(s,fd)
int s,fd;
{
  char *p;
  int filesize,cosid;
  int status,rcode,n;
  int cos_req_size;

  log(LOG_INFO, "rsetcos(%d, %d)\n",s, fd);
  p= rqstbuf + 2*WORDSIZE ; 
  unmarshall_LONG(p, filesize) ;
  unmarshall_LONG(p,cosid) ;
  status = rhpss_setcos(fd,filesize,cosid);
  rcode = errno;
  p = rqstbuf;
  marshall_WORD(p,RQST_SETCOS) ;
  marshall_LONG(p,status) ;
  marshall_LONG(p,rcode) ;
  marshall_LONG(p,0) ;
#if defined(SACCT)
  rfioacct(RQST_SETCOS,0,0,s,cosid,filesize,status,rcode,NULL,NULL,NULL);
#endif /* SACCT */
  if ((n = netwrite(s,rqstbuf,RQSTSIZE)) != RQSTSIZE)  {
    log(LOG_ERR, "rsetcos: write(): %s\n", sys_errlist[errno]) ;
    return -1 ; 
  }
  log(LOG_INFO,"rsetcos: return status=%d rcode=%d\n",status,rcode);
  return(status);
}

int rhpss_setcos(fd,filesize, cosid)
int fd;
int filesize;
int cosid;
{
  /*
   * Set Class Of Service for a file which has been opened for write but
   * no data written yet.
   */
  struct stat fst;
  int status;
  unsigned32 flags = 0;
  hpss_cos_hints_t cos_hints;
  hpss_cos_priorities_t cos_priors;
  hpss_cos_md_t cos_out;

  /*
   * First check that no data has been written
   */
  log(LOG_INFO,"rhpss_setcos(%d,%d,%d) entered\n",fd,filesize,cosid);
  if ( fd<0 ) {
    errno=EBADF;
    return(-1);
  }
  status = hpss_Fstat(fd,&fst);
  if ( status ) {
    log(LOG_ERR,"Cannot fstat HPSS file descriptor, status=%d, %s\n",fd,
	status,sys_errlist[errno]);
    return(status);
  }
  if ( fst.st_size > 0 ) {
    log(LOG_ERR,"Cannot set COS on an open file which already contains data\n");
    return(1);
  }

  /*
   * do the work
   */
  memset(&cos_hints,0,sizeof(hpss_cos_hints_t));
  memset(&cos_out,0,sizeof(hpss_cos_md_t));
  cos_priors.COSIdPriority             = NO_PRIORITY;
  cos_priors.COSNamePriority           = NO_PRIORITY;
  cos_priors.OptimumAccessSizePriority = NO_PRIORITY;
  cos_priors.MinFileSizePriority       = NO_PRIORITY;
  cos_priors.MaxFileSizePriority       = NO_PRIORITY;
  cos_priors.AccessFrequencyPriority   = NO_PRIORITY;
  cos_priors.TransferRatePriority      = NO_PRIORITY;
  cos_priors.AvgLatencyPriority        = NO_PRIORITY;
  cos_priors.WriteOpsPriority          = NO_PRIORITY;
  cos_priors.ReadOpsPriority           = NO_PRIORITY;
  cos_priors.StageCodePriority         = NO_PRIORITY;
  if ( cosid > 0 ) {
    cos_hints.COSId = cosid;
    cos_priors.COSIdPriority = REQUIRED_PRIORITY;
  }
  if ( filesize > 0 ) {
    cos_hints.MinFileSize = cos_hints.MaxFileSize = cast64m(filesize);
    cos_priors.MinFileSizePriority = cos_priors.MaxFileSizePriority = HIGHLY_DESIRED_PRIORITY;
    flags = BFS_RESET_SEGSIZE;
  }
  status = hpss_SetCOSByHints(fd,flags,&cos_hints,&cos_priors,&cos_out);
  if ( status )
    log(LOG_ERR,"hpss_SetCOSByHints() failed with %d, %s\n",status,sys_errlist[-status]);

  return(status);
}

int srreadlist(s, infop, fd, from)
int     s, fd;
struct rfiostat * infop ;
struct sockaddr_in *from;
{
  int 	status ;	/* Return code		*/
  int      rcode ;	/* To send back errno	*/
  int	   how ;	/* lseek mode 		*/
  int	offset ;	/* lseek offset		*/
  int       size ;	/* Requested read size	*/
  int   len;
  int reqlen,i;
  int nb_ports,ports[RFIO_MAX_PORTS];
  stripeaddress_t netstrp[RFIO_MAX_PORTS];
  struct sockaddr_in addr[RFIO_MAX_PORTS];
  IOD_t iod;
  IOR_t ior;
  srcsinkdesc_t src,sink;
  int reqid,buflen;
  unsigned long RL_flag;
  u_signed64 remlen,filesize,fileoffset,start_fileoffset;
  char *p;

  log(LOG_DEBUG, "rreadlist(%d, %d)\n",s, fd);
  if ( fd < 0 ) {
    errno=EBADF;
    return(-1);
  }
  p= rqstbuf + 2*WORDSIZE ; 
  unmarshall_LONG(p,len);
  log(LOG_DEBUG,"rreadlist: reading %d bytes\n",len) ;
  if ((status = netread(s,rqstbuf,len)) != len) {
    log(LOG_ERR,"rreadlist: read(): %s\n",sys_errlist[errno]) ;
    return -1 ;
  }

  p = rqstbuf;
  unmarshall_LONG(p,size) ;
  unmarshall_LONG(p,how) ;
  unmarshall_LONG(p,offset) ; 
  unmarshall_LONG(p,reqid) ;
  unmarshall_LONG(p,buflen) ;
  unmarshall_LONG(p,nb_ports);
  log(LOG_INFO,"rreadlist(): size=%d, offset=%d, reqid=%d, buflen=%d, nb_ports=%d\n",
      size,offset,reqid,buflen,nb_ports);
  if ( nb_ports > RFIO_MAX_PORTS ) {
    log(LOG_ERR,"rreadlist(): too many (%d) ports specified. Maximum is %d\n",
	nb_ports,RFIO_MAX_PORTS);
    p = rqstbuf;
    status = -1;
    rcode = EINVAL;
    return(rhpss_readlist_return(s,status,rcode,0,0,0));
  }
  RL_flag = 0;
  if ( nb_ports <= 0 ) {
    nb_ports = 1;
    RL_flag = HPSS_READ_SEQUENTIAL;
    log(LOG_INFO,"rreadlist(): readlist Flag = HPSS_READ_SEQUENTIAL (0x%x)\n",RL_flag);
  }
  for ( i=0; i<nb_ports; i++ ) unmarshall_LONG(p,ports[i]);

  rcode = rhpss_mkaddr(from,nb_ports,ports,addr);
  if ( rcode ) {
    log(LOG_ERR,"rhpss_mkaddr(%s) failed with status %d\n",rcode);
    status = -1;
    return(rhpss_readlist_return(s,status,rcode,0,0,0));
  }


  memset(&iod,'\0',sizeof(iod));
  memset(&ior,'\0',sizeof(ior));
  memset(&src,'\0',sizeof(src));
  memset(&sink,'\0',sizeof(sink));

  filesize = cast64m(size);
  start_fileoffset = cast64m(offset);

  iod.Function = IOD_READ;
  iod.RequestID = reqid;
  iod.SrcDescLength = 1;
  iod.SinkDescLength = 1;
  iod.SrcDescList = &src;
  iod.SinkDescList = &sink;

  src.SrcSinkAddr.Type = CLIENTFILE_ADDRESS;
  src.SrcSinkAddr.Addr_u.ClientFileAddr.FileDes = fd;
  src.Next = sink.Next = (struct srcsinkdesc *) NULL;
  if ( nb_ports == 1 ) {
    sink.Flags = HOLD_RESOURCES | XFEROPT_IP | CONTROL_ADDR;
    sink.SrcSinkAddr.Type = NET_ADDRESS;
    sink.SrcSinkAddr.Addr_u.NetAddr.SockTransferID = cast64m(reqid);
    sink.SrcSinkAddr.Addr_u.NetAddr.SockAddr.addr = addr[0].sin_addr.s_addr;
    sink.SrcSinkAddr.Addr_u.NetAddr.SockAddr.port = addr[0].sin_port;
    sink.SrcSinkAddr.Addr_u.NetAddr.SockAddr.family = addr[0].sin_family;
    sink.SrcSinkAddr.Addr_u.NetAddr.SockOffset = cast64m(0);
  } else {
    memset(&netstrp[0],'\0',RFIO_MAX_PORTS*sizeof(stripeaddress_t));
    sink.SrcSinkAddr.Type = STRIPE_ADDRESS;
    for (i=0; i<nb_ports; i++) {
      netstrp[i].BlockSize = cast64m(buflen);
      netstrp[i].StripeWidth = cast64m(nb_ports);
      netstrp[i].AddrListLength = nb_ports;
      netstrp[i].Flags = CONTROL_ADDR | XFEROPT_IP;
      netstrp[i].Addr.Type = NET_ADDRESS;
      netstrp[i].Addr.Addr_u.NetAddr.SockTransferID = cast64m(reqid);
      netstrp[i].Addr.Addr_u.NetAddr.SockAddr.addr = addr[i].sin_addr.s_addr;
      netstrp[i].Addr.Addr_u.NetAddr.SockAddr.port = addr[i].sin_port;
      netstrp[i].Addr.Addr_u.NetAddr.SockAddr.family = addr[i].sin_family;
      netstrp[i].Addr.Addr_u.NetAddr.SockOffset = cast64m(buflen*i);
      netstrp[i].Next = (stripeaddress_t *)&netstrp[i+1];
    }
    netstrp[i-1].Next = NULL;
    memcpy(&sink.SrcSinkAddr.Addr_u.StripeAddr,&netstrp[0],sizeof(stripeaddress_t));
  }
  remlen = sub64m(filesize,start_fileoffset) ;
  status = rcode = 0;
  while ( status || neqz64m(remlen) ) {
    srcsinkreply_t *sofar;
    src.Length = sink.Length = remlen;
    fileoffset = add64m(start_fileoffset,sub64m(filesize,remlen));
    src.SrcSinkAddr.Addr_u.ClientFileAddr.FileOffset = fileoffset;
    src.Offset = sink.Offset = fileoffset;
    memset(&ior,'\0',sizeof(ior));
    ior.Flags = IOR_NOT_PROCESSED;
    log(LOG_DEBUG,"(s=%d) calling hpss_Readlist(0x%x,0x%x,0x%x), HPSS reqid=%lu\n",s,
	&ior,RL_flag,&iod,reqid);
    status = hpss_ReadList(&iod,RL_flag,&ior);
    log(LOG_DEBUG,"(s=%d) hpss_Readlist(0x%x,0x%x,0x%x) returned, HPSS reqid=%lu\n",s,
	&ior,RL_flag,&iod,reqid);
    sofar = ior.SinkReplyList;
    while ( sofar != (srcsinkreply_t *)NULL ) {
      remlen = sub64m(remlen,sofar->BytesMoved);
      infop->rnbr+= cast32m(sofar->BytesMoved) ;
      sofar = sofar->Next;
    }
    if ( status ) {
      rcode = -status;
      free_ior_mem(&ior);
      log(LOG_ERR,"hpss_ReadList() failed with status %d, %s\n",status,sys_errlist[-status]);
      break;
    } else if ( ior.Flags & IOR_GAPINFO_VALID ) { 
      u_signed64 gapsize,gapoffset;
      int gapsz,gapoff;
      gapoffset = ior.ReqSpecReply->ReqReply_s.ReqReply_u.GapInfo.Offset;
      gapsize = ior.ReqSpecReply->ReqReply_s.ReqReply_u.GapInfo.Length;
      free_ior_mem(&ior);
      remlen = sub64m(remlen,gapsize);
      gapoff = cast32m(gapoffset);
      gapsz = cast32m(gapsize);
      log(LOG_INFO,"A gap of size %d encountered at offset %d\n",gapsz,gapoff);
      rhpss_readlist_return(s,0,0,1,gapoff,gapsz);
    }
/*
    rpc_ss_client_free(ior.SrcReplyList);
    rpc_ss_client_free(ior.SinkReplyList);
*/
    if ( ! (ior.Flags & IOR_GAPINFO_VALID) ) {
      if ( ior.Status != HPSS_EOM ) status = ior.Status;
      else log(LOG_INFO,"rreadlist: ignoring IOR.Status==HPSS_EOM (%d)\n",ior.Status);
      rcode = 0;
      break;
    }
  }
  free_iod_mem(&iod);
#if defined(SACCT)
  rfioacct(RQST_READLIST,0,0,s,nb_ports,reqid,status,rcode,infop,NULL,NULL);
#endif /* SACCT */
  log(LOG_INFO, "rreadlist: returning status %d, rcode %d after moving %d bytes\n", 
      status, rcode, infop->rnbr ) ;
  return(rhpss_readlist_return(s,status,rcode,0,0,0));
}

rhpss_mkaddr(from,nb_ports,ports,addr)
struct sockaddr_in *from;
int nb_ports,ports[];
struct sockaddr_in addr[];
{
  int save_errno;
  int i;

  if ( from == NULL ) {
    log(LOG_ERR,"rhpss_mkaddr: host addr not supplied\n");
    return(-1);
  }

  for (i=0; i<nb_ports; i++) {
    memset(&addr[i],'\0',sizeof(struct sockaddr_in));
    addr[i].sin_family = AF_INET;
    /*
     * Everything should be in host order in the IOD
     */
    addr[i].sin_addr.s_addr = ntohl(from->sin_addr.s_addr);
    addr[i].sin_port = ports[i];
  }
  return(0);
}
int rhpss_readlist_return(s,status,rcode,gap,gapoffset,gapsize)
int s,status,rcode,gap,gapoffset,gapsize;
{
  char *p;
  int n, readlist_req_size;

  p = rqstbuf;
  readlist_req_size = WORDSIZE + 5*LONGSIZE;
  marshall_WORD(p,RQST_READLIST) ;
  marshall_LONG(p,status) ;
  marshall_LONG(p,rcode) ;
  marshall_LONG(p,gap) ;
  marshall_LONG(p,gapoffset) ;
  marshall_LONG(p,gapsize) ;
  if ((n = netwrite(s,rqstbuf,readlist_req_size)) != readlist_req_size)  {
    log(LOG_ERR, "rreadlist: write(): %s\n", sys_errlist[errno]) ;
    return -1 ; 
  }
  return(status);
}

int srwritelist(s, infop, fd, from)
int     s, fd;
struct rfiostat * infop ;
struct sockaddr_in *from;
{
  int 	status ;	/* Return code		*/
  int      rcode ;	/* To send back errno	*/
  int	   how ;	/* lseek mode 		*/
  int	offset ;	/* lseek offset		*/
  int       size ;	/* Requested read size	*/
  int   len;
  int reqlen,i;
  int nb_ports,ports[RFIO_MAX_PORTS];
  stripeaddress_t netstrp[RFIO_MAX_PORTS];
  struct sockaddr_in addr[RFIO_MAX_PORTS];
  IOD_t iod;
  IOR_t ior;
  srcsinkdesc_t src,sink;
  int reqid,buflen;
  u_signed64 filesize,fileoffset;
  char *p;

  log(LOG_DEBUG, "rwritelist(%d, %d)\n",s, fd);
  if ( fd < 0 ) {
    errno=EBADF;
    return(-1);
  }
  p= rqstbuf + 2*WORDSIZE ; 
  unmarshall_LONG(p,len);
  log(LOG_DEBUG,"rwritelist: reading %d bytes\n",len) ;
  if ((status = netread(s,rqstbuf,len)) != len) {
    log(LOG_ERR,"rwritelist: read(): %s\n",sys_errlist[errno]) ;
    return -1 ;
  }

  p = rqstbuf;
  unmarshall_LONG(p,size) ;
  unmarshall_LONG(p,how) ;
  unmarshall_LONG(p,offset) ; 
  unmarshall_LONG(p,reqid) ;
  unmarshall_LONG(p,buflen) ;
  unmarshall_LONG(p,nb_ports);
  log(LOG_INFO,"rwritelist(): size=%d, offset=%d, reqid=%d, buflen=%d, nb_ports=%d\n",
      size,offset,reqid,buflen,nb_ports);
  if ( nb_ports > RFIO_MAX_PORTS ) {
    log(LOG_ERR,"rwritelist(): too many (%d) ports specified. Maximum is %d\n",
	nb_ports,RFIO_MAX_PORTS);
    p = rqstbuf;
    status = -1;
    rcode = EINVAL;
    return(rhpss_writelist_return(s,status,rcode,0,0,0));
  }
  for ( i=0; i<nb_ports; i++ ) unmarshall_LONG(p,ports[i]);

  rcode = rhpss_mkaddr(from,nb_ports,ports,addr);
  if ( rcode ) {
    log(LOG_ERR,"rhpss_mkaddr(%s) failed with status %d\n",rcode);
    status = -1;
    return(rhpss_writelist_return(s,status,rcode,0,0,0));
  }


  memset(&iod,'\0',sizeof(iod));
  memset(&ior,'\0',sizeof(ior));
  memset(&src,'\0',sizeof(src));
  memset(&sink,'\0',sizeof(sink));

  filesize = cast64m(size);
  fileoffset = cast64m(offset);

  iod.Function = IOD_WRITE;
  iod.RequestID = reqid;
  iod.SrcDescLength = 1;
  iod.SinkDescLength = 1;
  iod.SrcDescList = &src;
  iod.SinkDescList = &sink;

  sink.SrcSinkAddr.Type = CLIENTFILE_ADDRESS;
  sink.SrcSinkAddr.Addr_u.ClientFileAddr.FileDes = fd;
  src.Length = sink.Length = filesize;
  src.Offset = sink.Offset = fileoffset;
  src.Next = sink.Next = (struct srcsinkdesc *) NULL;
  sink.SrcSinkAddr.Addr_u.ClientFileAddr.FileOffset = fileoffset;
  if ( nb_ports == 1 ) {
    src.Flags = HOLD_RESOURCES | XFEROPT_IP | CONTROL_ADDR;
    src.SrcSinkAddr.Type = NET_ADDRESS;
    src.SrcSinkAddr.Addr_u.NetAddr.SockTransferID = cast64m(reqid);
    src.SrcSinkAddr.Addr_u.NetAddr.SockAddr.addr = addr[0].sin_addr.s_addr;
    src.SrcSinkAddr.Addr_u.NetAddr.SockAddr.port = addr[0].sin_port;
    src.SrcSinkAddr.Addr_u.NetAddr.SockAddr.family = addr[0].sin_family;
    src.SrcSinkAddr.Addr_u.NetAddr.SockOffset = cast64m(0);
  } else {
    memset(&netstrp[0],'\0',RFIO_MAX_PORTS*sizeof(stripeaddress_t));
    src.SrcSinkAddr.Type = STRIPE_ADDRESS;
    for (i=0; i<nb_ports; i++) {
      netstrp[i].BlockSize = cast64m(buflen);
      netstrp[i].StripeWidth = cast64m(nb_ports);
      netstrp[i].AddrListLength = nb_ports;
      netstrp[i].Flags = CONTROL_ADDR | XFEROPT_IP;
      netstrp[i].Addr.Type = NET_ADDRESS;
      netstrp[i].Addr.Addr_u.NetAddr.SockTransferID = cast64m(reqid);
      netstrp[i].Addr.Addr_u.NetAddr.SockAddr.addr = addr[i].sin_addr.s_addr;
      netstrp[i].Addr.Addr_u.NetAddr.SockAddr.port = addr[i].sin_port;
      netstrp[i].Addr.Addr_u.NetAddr.SockAddr.family = addr[i].sin_family;
      netstrp[i].Addr.Addr_u.NetAddr.SockOffset = cast64m(buflen*i);
      netstrp[i].Next = (stripeaddress_t *)&netstrp[i+1];
    }
    netstrp[i-1].Next = NULL;
    memcpy(&src.SrcSinkAddr.Addr_u.StripeAddr,&netstrp[0],sizeof(stripeaddress_t));
  }
  status = rcode = 0;
  memset(&ior,'\0',sizeof(ior));
  ior.Flags = IOR_NOT_PROCESSED;
  log(LOG_DEBUG,"(s=%d) calling hpss_Writelist(0x%x,0,0x%x), HPSS reqid=%lu\n",s,
      &ior,&iod,reqid);
  status = hpss_WriteList(&iod,0,&ior);
  log(LOG_DEBUG,"(s=%d) hpss_Writelist(0x%x,0,0x%x) returned, HPSS reqid=%lu\n",s,
      &ior,&iod,reqid);
  if ( status ) {
    rcode = -status;
    log(LOG_ERR,"hpss_Writelist() failed with status %d, %s\n",status,sys_errlist[-status]);
  } 
  {
    srcsinkreply_t *sofar;
    sofar = ior.SinkReplyList;
    while ( sofar != (srcsinkreply_t *)NULL ) {
      infop->wnbr+= cast32m(sofar->BytesMoved) ;
      sofar = sofar->Next;
    }
  }
  free_ior_mem(&ior);
  free_iod_mem(&iod);
/*
  rpc_ss_client_free(ior.SrcReplyList);
  rpc_ss_client_free(ior.SinkReplyList);
*/
#if defined(SACCT)
  rfioacct(RQST_WRITELIST,0,0,s,nb_ports,reqid,status,rcode,infop,NULL,NULL);
#endif /* SACCT */
  log(LOG_INFO, "rwritelist: returning status %d, rcode %d after moving %d bytes\n", 
      status, rcode, infop->wnbr ) ;
  return(rhpss_writelist_return(s,status,rcode,0,0,0));
}
int rhpss_writelist_return(s,status,rcode)
int s,status,rcode;
{
  char *p;
  int n, writelist_req_size;

  p = rqstbuf;
  writelist_req_size = WORDSIZE + 5*LONGSIZE;
  marshall_WORD(p,RQST_WRITELIST) ;
  marshall_LONG(p,status) ;
  marshall_LONG(p,rcode) ;
  marshall_LONG(p,0) ;
  marshall_LONG(p,0) ;
  marshall_LONG(p,0) ;
  if ((n = netwrite(s,rqstbuf,writelist_req_size)) != writelist_req_size)  {
    log(LOG_ERR, "rwritelist: write(): %s\n", sys_errlist[errno]) ;
    return -1 ; 
  }
  return(status);
}
#ifdef TRACEMALLOC
#undef malloc
#undef free
char *rfio_malloc(int size, char *file, int line) {
  void *tmp;
  log(LOG_INFO,"rfio_malloc(%d) (%s:%d)\n",size,file,line);
  tmp = (void *)malloc(size);
  log(LOG_INFO,"rfio_malloc(%d) (%s:%d) -> 0x%x\n",size,file,line,tmp);
  return(tmp);
}
void rfio_free(void *tmp, char *file, int line) {
  log(LOG_INFO,"rfio_free(0x%x) (%s:%d)\n",tmp,file,line);
  free(tmp);
}
#endif /* TRACEMALLOC */
#endif /* HPSS */

