#ifndef lint
static char sccsid[] = "@(#)ibmconfidential.c	1.2 11/06/98 CERN IT/PDP/DM Olof Barring";
#endif /* not lint */
/* static char SccsId[] = "src/cs/pdata/mvrprotocol.c, gen, 3v2   12/3/97   14:30:55   "; */
/*============================================================================
 *
 * Name:        mvrprot.c
 *
 * Functions:
 *	build_initmsg_checksum - build checksum for initiator message
 *	build_ipaddr_checksum - build checksum for IP address message
 *	build_ipi3addr_checksum - build checksum for IPI-3 address message
 *	build_shmaddr_checksum - build checksum for shmem address message
 *	build_compmsg_checksum - build checksum for completion message
 *	mvrprot_recv_data - receive specified amount of data
 *	mvrprot_send_data - send specified amount of data
 *	mvrprot_decode_chars - decode a number of characters
 *	mvrprot_encode_chars - encode a number of characters
 *	mvrprot_decode_16 - decode a short integer
 *	mvrprot_encode_16 - encode a short integer
 *	mvrprot_decode_32 - decode a long integer
 *	mvrprot_encode_32 - encode a long integer
 *	mvrprot_decode_64 - decode a 64-bit integer
 *	mvrprot_encode_64 - encode a 64-bit integer
 *	mvrprot_recv_initmsg - receive an initiator message
 *	mvrprot_send_initmsg - send an initiator message
 *	mvrprot_recv_ipaddr - receive an IP address message
 *	mvrprot_send_ipaddr - send an IP address message
 *	mvrprot_recv_ipi3addr - receive an IPI-3 address message
 *	mvrprot_send_ipi3addr - send an IPI-3 address message
 *	mvrprot_recv_shmaddr - receive a shared memory address message
 *	mvrprot_send_shmaddr - send a shared memory address message
 *	mvrprot_recv_compmsg - receive a completion message
 *	mvrprot_send_compmsg - send a completion message
 *
 * Description:
 *	These routines are used by the mover to send and receive mover
 *	protocol messages.
 *
 * Traceability:
 *      Version         Author          Date            Description
 *      -------         ------          ----            -----------
 *      2.1             kwe             05/16/95	Initial D2 version.
 *      4.2             WHR             01/29/97	Added Portability code
 *      4.3	WHR	10/09/97	Removed "static" from void functions.
 *					Added ntohl and htonl calls as needed
 *					for BYTE8INT (non-cray) systems.
 *      4.4             kwe             11/12/97	Added timeout routines.
 *
 *  Notes:
 *       Licensed Materials
 *
 *       (C) Copyright International Business Machines Corp. 1994
 *       (C) Copyright Martin Marietta Energy Systems, Inc. and
 *                     Oak Ridge National Laboratory 1994
 *                     under CRADA No. Y1293-0203
 *       (C) Copyright Regents of The University of California and
 *                     Lawrence Livermore National Laboratory 1994
 *                     under CRADA No. T-253-92-C (as modified by T-325-92-C)
 *       (C) Copyright Regents of The University of California and
 *                     Los Alamos National Laboratory 1994
 *                     under CRADA No. LA93C10085
 *       (C) Copyright Sandia Corporation and Sandia National Laboratories 1994
 *                     under CRADA No. SC93/01198
 * 
 *       All rights reserved.
 * 
 *       This file is IBM Confidential.
 *
 *		This function is dependent on a 64-bit math library.
 *
 *-------------------------------------------------------------------------*/

#include <stdio.h>

#if defined(THREADS_ENABLED)
#include <pthread.h>
#endif

#include <sys/types.h>
#if ! defined(__PARAGON__) && ! defined(__HPUX__)  && ! defined(NeXT)
#include <sys/select.h>
#endif
#include <sys/time.h>
#include <netinet/in.h>
#include <errno.h>
#include "u_signed64.h"
#include "hpss_errno.h"
#include "mvr_protocol.h"

/*============================================================================
 *
 * Function:
 *	build_initmsg_checksum - builds checksum for an initiator message.
 *
 * Synopsis:
 *
 *	static void
 *	build_initmsg_checksum(
 *	initiator_msg_t		*InitMsg,
 *	u_signed64		*CheckSum
 *	)
 *
 * Description:
 *	Generates the checksum for the passed initiator message, returning
 *	it in the area pointed to by CheckSum.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	None.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

static void
build_initmsg_checksum(
initiator_msg_t		*InitMsg,
u_signed64		*CheckSum)
{
    u_signed64	checksum;

    checksum = add64m(InitMsg->Delimiter,cast64m(InitMsg->Flags));
    inc64m(checksum,cast64m(InitMsg->Type));
    inc64m(checksum,InitMsg->Offset);
    inc64m(checksum,InitMsg->Length);
    inc64m(checksum,InitMsg->BlockSize);
    inc64m(checksum,InitMsg->StripeWidth);
    inc64m(checksum,InitMsg->Stride);
    inc64m(checksum,InitMsg->TotalLength);

    *CheckSum = add64m(checksum,mem64m(InitMsg->SecurityTicket));
    return;
}

/*============================================================================
 *
 * Function:
 *	build_ipaddr_checksum - builds checksum for an IP address message.
 *
 * Synopsis:
 *
 *	static void
 *	build_ipaddr_checksum(
 *	initiator_ipaddr_t	*InitIpPtr,
 *	u_signed64		*CheckSum
 *	)
 *
 * Description:
 *	Generates the checksum for the passed IP address message, returning
 *	it in the aread pointed to by CheckSum.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	None.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

static void
build_ipaddr_checksum(
initiator_ipaddr_t	*InitIpPtr,
u_signed64		*CheckSum)
{
    u_signed64	checksum;

    checksum = add64m(InitIpPtr->Delimiter,cast64m(InitIpPtr->Flags));
    inc64m(checksum,InitIpPtr->IpAddr.SockTransferID);
    inc64m(checksum,cast64m(InitIpPtr->IpAddr.SockAddr.addr));
    inc64m(checksum,cast64m(InitIpPtr->IpAddr.SockAddr.port));
    inc64m(checksum,cast64m(InitIpPtr->IpAddr.SockAddr.family));
    inc64m(checksum,InitIpPtr->IpAddr.SockOffset);

    *CheckSum = add64m(checksum,mem64m(InitIpPtr->SecurityTicket));
    return;
}

/*============================================================================
 *
 * Function:
 *	build_ipi3addr_checksum - builds checksum for an IPI-3 address message.
 *
 * Synopsis:
 *
 *	static void
 *	build_ipi3addr_checksum(
 *	initiator_ipi3addr_t	*InitIpi3Ptr,
 *	u_signed64		*CheckSum
 *	)
 *
 * Description:
 *	Generates the checksum for the passed IPI-3 address message, returning
 *	it in the aread pointed to by CheckSum.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	None.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

static void
build_ipi3addr_checksum(
initiator_ipi3addr_t	*InitIpi3Ptr,
u_signed64		*CheckSum)
{
    u_signed64	checksum;

    checksum = add64m(InitIpi3Ptr->Delimiter,cast64m(InitIpi3Ptr->Flags));
    inc64m(checksum,cast64m(InitIpi3Ptr->Ipi3Addr.IPI3TransferID));
    inc64m(checksum,cast64m(InitIpi3Ptr->Ipi3Addr.IPI3Addr.Interface));

    inc64m(checksum,mem64m(InitIpi3Ptr->Ipi3Addr.IPI3Addr.Name));
    inc64m(checksum,mem64m(&InitIpi3Ptr->Ipi3Addr.IPI3Addr.Name[8]));
    inc64m(checksum,mem64m(&InitIpi3Ptr->Ipi3Addr.IPI3Addr.Name[16]));
    inc64m(checksum,mem64m(&InitIpi3Ptr->Ipi3Addr.IPI3Addr.Name[24]));

    *CheckSum = add64m(checksum,mem64m(InitIpi3Ptr->SecurityTicket));
    return;
}

/*============================================================================
 *
 * Function:
 *	build_shmaddr_checksum - builds checksum for a shared memory
 *				 address message.
 *
 * Synopsis:
 *
 *	static void
 *	build_shmaddr_checksum(
 *	initiator_shmaddr_t	*InitShmPtr,
 *	u_signed64		*CheckSum
 *	)
 *
 * Description:
 *	Generates the checksum for the passed shared memory address message,
 *	returning it in the aread pointed to by CheckSum.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	None.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

static void
build_shmaddr_checksum(
initiator_shmaddr_t	*InitShmPtr,
u_signed64		*CheckSum)
{
    u_signed64	checksum;

    checksum = add64m(InitShmPtr->Delimiter,cast64m(InitShmPtr->Flags));
    inc64m(checksum,cast64m(InitShmPtr->ShmAddr.Flags));
    inc64m(checksum,cast64m(InitShmPtr->ShmAddr.ShmID));
    inc64m(checksum,cast64m(InitShmPtr->ShmAddr.ShmOffset));

    *CheckSum = add64m(checksum,mem64m(InitShmPtr->SecurityTicket));
    return;
}

/*============================================================================
 *
 * Function:
 *	build_compmsg_checksum - builds checksum for an completion message.
 *
 * Synopsis:
 *
 *	static void
 *	build_compmsg_checksum(
 *	completion_msg_t	*CompPtr,
 *	u_signed64		*CheckSum
 *	)
 *
 * Description:
 *	Generates the checksum for the passed IPI-3 address message, returning
 *	it in the aread pointed to by CheckSum.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	None.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

static void
build_compmsg_checksum(
completion_msg_t	*CompPtr,
u_signed64		*CheckSum)
{
    u_signed64	checksum;

    checksum = add64m(CompPtr->Delimiter,cast64m(CompPtr->Flags));
    inc64m(checksum,cast64m(CompPtr->Status));
    inc64m(checksum,CompPtr->BytesMoved);

    *CheckSum = add64m(checksum,mem64m(CompPtr->SecurityTicket));
    return;
}

/*============================================================================
 *
 * Function:
 *	mvrprot_recv_data - attempts to receive the specified amount of data.
 *
 * Synopsis:
 *
 *	static long
 *	mvrprot_recv_data(
 *	int		SockFD,
 *	char		*CharPtr,
 *	int		NBytes
 *	)
 *
 * Description:
 *	This routine attempts to receive, using the socket specified by
 *	SockFD, the amount of data specified by NBytes into the buffer
 *	pointed to by CharPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	Return value:
 *		> 0		- bytes received
 *		HPSS_ECONN	- Connection was closed.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

static long
mvrprot_recv_data_timeout(
int			SockFD,
char			*CharPtr,
int			NBytes,
int			SecTimeout)
{
    register int	leng = 0;
    register int	ret,nfd;
    register char	*cptr;
    struct timeval	timeout,*timeptr;
    fd_set		rdlist,wrlist,exlist;
    fd_set		save_rdlist,save_wrlist,save_exlist;

    /*
     *  Set up receive lists.
     */

    FD_ZERO(&save_rdlist);
    FD_ZERO(&save_wrlist);
    FD_ZERO(&save_exlist);

    FD_SET(SockFD,&save_rdlist);

    cptr = CharPtr;
    while (leng < NBytes)
    {
	if (SecTimeout == 0)
	   timeptr = (struct timeval *)NULL;
	else
	{
	   timeout.tv_sec = SecTimeout;
	   timeout.tv_usec = 0;
	   timeptr = &timeout;
	}

	rdlist = save_rdlist;
	wrlist = save_wrlist;
	exlist = save_exlist;

	nfd = select(FD_SETSIZE,&rdlist,&wrlist,&exlist,timeptr);
	if (nfd < 0)
	{
	   if((errno == EINTR) || (errno == EAGAIN)) continue;

	   /*
	    *  If we got any other error, bail out.
	    */

	   return(-errno);
	}

	/*
	 *  If the select timed out, return -ETIMEDOUT.  Otherwise,
	 *  it should be safe to use the socket, since we only
	 *  asked about one.
	 */

	if (nfd == 0)
	   return(-ETIMEDOUT);

	ret = read(SockFD,cptr,NBytes - leng);
	if (ret < 0)
	{
	    if ((errno == EINTR) || (errno == EAGAIN))
	        continue;
	    /* mkg96feb29: Intel Paragon returns <ENETDOWN> if connection closed */
#if defined(__PARAGON__)
	    if (errno == ENETDOWN) 
	    return(HPSS_ECONN);
#endif /* __PARAGON__ */
	    return(-errno);
	}
	else if (ret == 0)
	{
	    /*
	     *  Connection must have closed.
	     */

	    return(HPSS_ECONN);
	}

	cptr += ret;
	leng += ret;
    }

    return(leng);
}

static long
mvrprot_recv_data(
int			SockFD,
char			*CharPtr,
int			NBytes)
{
    return(mvrprot_recv_data_timeout(SockFD,CharPtr,NBytes,0));
}

/*============================================================================
 *
 * Function:
 *	mvrprot_send_data - attempts to send the specified amount of data.
 *
 * Synopsis:
 *
 *	static long
 *	mvrprot_send_data(
 *	int		SockFD,
 *	char		*CharPtr,
 *	int		NBytes
 *	)
 *
 * Description:
 *	This routine attempts to send, using the socket specified by
 *	SockFD, the amount of data specified by NBytes from the buffer
 *	pointed to by CharPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	Return value:
 *		> 0		- bytes sent
 *		HPSS_ECONN	- Connection was closed.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

static long
mvrprot_send_data_timeout(
int			SockFD,
char			*CharPtr,
int			NBytes,
int			SecTimeout)
{
    register int	leng = 0;
    register int	ret,nfd;
    register char	*cptr;
    struct timeval	timeout,*timeptr;
    fd_set		rdlist,wrlist,exlist;
    fd_set		save_rdlist,save_wrlist,save_exlist;

    /*
     *  Set up receive lists.
     */

    FD_ZERO(&save_rdlist);
    FD_ZERO(&save_wrlist);
    FD_ZERO(&save_exlist);

    FD_SET(SockFD,&save_wrlist);


    cptr = CharPtr;
    while (leng < NBytes)
    {
	if (SecTimeout == 0)
	   timeptr = (struct timeval *)NULL;
	else
	{
	   timeout.tv_sec = SecTimeout;
	   timeout.tv_usec = 0;
	   timeptr = &timeout;
	}

	rdlist = save_rdlist;
	wrlist = save_wrlist;
	exlist = save_exlist;

	nfd = select(FD_SETSIZE,&rdlist,&wrlist,&exlist,timeptr);
	if (nfd < 0)
	{
	   if((errno == EINTR) || (errno == EAGAIN)) continue;

	   /*
	    *  If we got any other error, bail out.
	    */

	   return(-errno);
	}

	/*
	 *  If the select timed out, return -ETIMEDOUT.  Otherwise,
	 *  it should be safe to use the socket, since we only
	 *  asked about one.
	 */

	if (nfd == 0)
	   return(-ETIMEDOUT);

	ret = write(SockFD,cptr,NBytes - leng);
	if (ret < 0)
	{
	    if ((errno == EINTR) || (errno == EAGAIN))
	        continue;

	    return(-errno);
	}
	else if (ret == 0)
	{
	    /*
	     *  Connection must have closed.
	     */

	    return(HPSS_ECONN);
	}

	cptr += ret;
	leng += ret;
    }

    return(leng);
}

static long
mvrprot_send_data(
int			SockFD,
char			*CharPtr,
int			NBytes)
{
    return(mvrprot_send_data_timeout(SockFD,CharPtr,NBytes,0));
}

/*============================================================================
 *
 * Function:
 *	mvrprot_decode_chars - decodes the specified number of characters
 *		from the message buffer.
 *
 * Synopsis:
 *
 *	static void
 *	mvrprot_decode_chars(
 *	char		**DataPtr,
 *	char		*CharPtr,
 *	int		Count)
 *	)
 *
 * Description:
 *	This routine attempts to decode the number of characters,
 *	specified by Count, from the message buffer pointed to by
 *	the pointer which is itself pointed to by DataPtr.  The decoded
 *	characters are placed in the buffer pointed to by CharPtr.
 *
 * Other Inputs:
 *      CharPtr - decode characters.
 *
 * Outputs:
 *	None
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

void
mvrprot_decode_chars(
char			*DataPtr,
int			Index,
char			*CharPtr,
int			Count)
{
    memcpy(CharPtr,&DataPtr[Index],Count);

    return;
}

/*============================================================================
 *
 * Function:
 *	mvrprot_encode_chars - encodes the specified number of characters
 *		into the message buffer.
 *
 * Synopsis:
 *
 *	static void
 *	mvrprot_encode_chars(
 *	char		**DataPtr,
 *	char		*CharPtr,
 *	int		Count)
 *	)
 *
 * Description:
 *	This routine attempts to encode the number of characters,
 *	specified by Count, into the message buffer pointed to by
 *	the pointer which is itself pointed to by DataPtr.  The decoded
 *	characters are taken from the buffer pointed to by CharPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	None
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

void
mvrprot_encode_chars(
char			*DataPtr,
int			Index,
char			*CharPtr,
int			Count)
{
    memcpy(&DataPtr[Index],CharPtr,Count);

    return;
}

/*============================================================================
 *
 * Function:
 *	mvrprot_decode_16 - decodes the specified 16-bit value
 *		from the message buffer.
 *
 * Synopsis:
 *
 *	static void
 *	mvrprot_decode_16(
 *	char		**DataPtr,
 *	unsigned16	*ShortPtr
 *	)
 *
 * Description:
 *	This routine attempts to decode a 2-byte value from the
 *	message buffer pointed to by the pointer which is itself
 *	pointed to by DataPtr.  The decoded value will be placed in
 *	the area pointed to by ShortPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	ShortPtr - decoded value.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

void
mvrprot_decode_16(
char			*DataPtr,
int			Index,
unsigned16		*ShortPtr)
{
    *ShortPtr = ((DataPtr[Index] << 8) & 0xFF00) |
		 (DataPtr[Index + 1] & 0x00FF);

    return;
}

/*============================================================================
 *
 * Function:
 *	mvrprot_encode_16 - encodes the specified 16-bit value
 *		into the message buffer.
 *
 * Synopsis:
 *
 *	static void
 *	mvrprot_encode_16(
 *	char		**DataPtr,
 *	unsigned16	ShortValue
 *	)
 *
 * Description:
 *	This routine attempts to encode a 2-byte value specified
 *	by ShortValue into the message buffer pointed to by the
 *	pointer which is itself pointed to by DataPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

void
mvrprot_encode_16(
char			*DataPtr,
int			Index,
unsigned16		ShortValue)
{
    DataPtr[Index] = (ShortValue >> 8) & 0xFF;
    DataPtr[Index + 1] = ShortValue & 0xFF;

    return;
}

/*============================================================================
 *
 * Function:
 *	mvrprot_decode_32 - decodes the specified 32-bit value
 *		from the message buffer.
 *
 * Synopsis:
 *
 *	static void
 *	mvrprot_decode_32(
 *	char		**DataPtr,
 *	unsigned32	*LongPtr
 *	)
 *
 * Description:
 *	This routine attempts to decode a 4-byte value from the
 *	message buffer pointed to by the pointer which is itself
 *	pointed to by DataPtr.  The decoded value will be placed in
 *	the area pointed to by LongPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	LongPtr - decoded value.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

void
mvrprot_decode_32(
char			*DataPtr,
int			Index,
unsigned32		*LongPtr)
{
    *LongPtr = ((DataPtr[Index] << 24)     & 0xFF000000) |
	       ((DataPtr[Index + 1] << 16) & 0x00FF0000) |
	       ((DataPtr[Index + 2] << 8)  & 0x0000FF00) |
	       (DataPtr[Index + 3]         & 0x000000FF);

    return;
}

/*============================================================================
 *
 * Function:
 *	mvrprot_encode_32 - encodes the specified 16-bit value
 *		into the message buffer.
 *
 * Synopsis:
 *
 *	static void
 *	mvrprot_encode_32(
 *	char		**DataPtr,
 *	unsigned32	LongValue
 *	)
 *
 * Description:
 *	This routine attempts to encode a 4-byte value specified
 *	by LongValue into the message buffer pointed to by the
 *	pointer which is itself pointed to by DataPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

void
mvrprot_encode_32(
char			*DataPtr,
int			Index,
unsigned32		LongValue)
{
    DataPtr[Index] = (LongValue >> 24) & 0xFF;
    DataPtr[Index + 1] = (LongValue >> 16) & 0xFF;
    DataPtr[Index + 2] = (LongValue >> 8)  & 0xFF;
    DataPtr[Index + 3] = LongValue & 0xFF;

    return;
}

/*============================================================================
 *
 * Function:
 *	mvrprot_decode_64 - decodes the specified 64-bit value
 *		from the message buffer.
 *
 * Synopsis:
 *
 *	static void
 *	mvrprot_decode_64(
 *	char		**DataPtr,
 *	u_signed64	*BigPtr
 *	)
 *
 * Description:
 *	This routine attempts to decode a 8-byte value from the
 *	message buffer pointed to by the pointer which is itself
 *	pointed to by DataPtr.  The decoded value will be placed in
 *	the area pointed to by BigPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	BigPtr - decoded value.
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

void
mvrprot_decode_64(
char			*DataPtr,
int			Index,
u_signed64		*BigPtr)
{
    unsigned32		templong1,templong2;

    templong1 = ((DataPtr[Index] << 24) & 0xFF000000) |
		((DataPtr[Index + 1] << 16) & 0x00FF0000) |
	        ((DataPtr[Index + 2] << 8) & 0x0000FF00) |
		(DataPtr[Index + 3] & 0x000000FF);
    templong2 = ((DataPtr[Index + 4] << 24) & 0xFF000000) |
		((DataPtr[Index + 5] << 16) & 0x00FF0000) |
	        ((DataPtr[Index + 6] << 8) & 0x0000FF00) |
		(DataPtr[Index + 7] & 0x000000FF);

    *BigPtr = bld64m(templong1,templong2);
    return;
}

/*============================================================================
 *
 * Function:
 *	mvrprot_encode_64 - encodes the specified 64-bit value
 *		into the message buffer.
 *
 * Synopsis:
 *
 *	static void
 *	mvrprot_encode_64(
 *	char		**DataPtr,
 *	u_signed64	BigValue
 *	)
 *
 * Description:
 *	This routine attempts to encode a 8-byte value specified
 *	by BigValue into the message buffer pointed to by the
 *	pointer which is itself pointed to by DataPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

void
mvrprot_encode_64(
char			*DataPtr,
int			Index,
u_signed64		BigValue)
{
    unsigned32		templong1,templong2;

    templong1 = high32m(BigValue);
    templong2 = low32m(BigValue);

    DataPtr[Index] = (templong1 >> 24) & 0xFF;
    DataPtr[Index + 1] = (templong1 >> 16) & 0xFF;
    DataPtr[Index + 2] = (templong1 >> 8)  & 0xFF;
    DataPtr[Index + 3] = templong1 & 0xFF;
    DataPtr[Index + 4] = (templong2 >> 24) & 0xFF;
    DataPtr[Index + 5] = (templong2 >> 16) & 0xFF;
    DataPtr[Index + 6] = (templong2 >> 8)  & 0xFF;
    DataPtr[Index + 7] = templong2 & 0xFF;

    return;
}

/*============================================================================
 *
 * Function:
 *	mvrprot_recv_initmsg - attempts to receive an initiator message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_recv_initmsg(
 *	int			SockFD,
 *	initiator_msg_t		*InitPtr)
 *	)
 *
 * Description:
 *	This routine attempts to read an initiator message using
 *	the socket specified by SockFD.  The message will be placed in
 *	the area pointed to by InitPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	InitPtr - received initiator message.
 *
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		HPSS_EINVAL	- Incorrect checksum.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_recv_initmsg(
int			SockFD,
initiator_msg_t		*InitPtr)
{
    return(mvrprot_recv_initmsg_timeout(SockFD,InitPtr,0));
}

long
mvrprot_recv_initmsg_timeout(
int			SockFD,
initiator_msg_t		*InitPtr,
int			SecTimeout)
{
    register long	ret;
    u_signed64		checksum;
    char		buffer[MVRPROT_INITMSG_SIZE];

    /*
     *  First receive the data.
     */

    ret = mvrprot_recv_data_timeout(SockFD,
				    buffer,
				    MVRPROT_INITMSG_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_INITMSG_SIZE)
	return(ret);

    /*
     *  Now decode the fields.
     */

    mvrprot_decode_64(buffer,MVRPROT_DELIM_OFFSET,&InitPtr->Delimiter);
    mvrprot_decode_32(buffer,MVRPROT_FLAGS_OFFSET,&InitPtr->Flags);
    mvrprot_decode_32(buffer,MVRPROT_INIT_TYPE_OFFSET,&InitPtr->Type);
    mvrprot_decode_64(buffer,MVRPROT_INIT_OFFS_OFFSET,&InitPtr->Offset);
    mvrprot_decode_64(buffer,MVRPROT_INIT_LEN_OFFSET,&InitPtr->Length);
    mvrprot_decode_64(buffer,MVRPROT_INIT_BLKSZ_OFFSET,&InitPtr->BlockSize);
    mvrprot_decode_64(buffer,MVRPROT_INIT_STRW_OFFSET,&InitPtr->StripeWidth);
    mvrprot_decode_64(buffer,MVRPROT_INIT_STRIDE_OFFSET,&InitPtr->Stride);
    mvrprot_decode_64(buffer,MVRPROT_INIT_TOTLEN_OFFSET,&InitPtr->TotalLength);
    mvrprot_decode_chars(buffer,
			 MVRPROT_INIT_SECTKT_OFFSET,
			 InitPtr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_decode_64(buffer,MVRPROT_INIT_CHKSUM_OFFSET,&InitPtr->CheckSum);

    /*
     *  Verify the checksum.
     */

    build_initmsg_checksum(InitPtr,&checksum);
    if (neq64m(InitPtr->CheckSum,checksum))
	return(HPSS_EINVAL);

    return(HPSS_E_NOERROR);
}

/*============================================================================
 *
 * Function:
 *	mvrprot_send_initmsg - attempts to send an initiator message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_send_initmsg(
 *	int			SockFD,
 *	initiator_msg_t		*InitPtr)
 *	)
 *
 * Description:
 *	This routine attempts to send an initiator message using
 *	the socket specified by SockFD.  The message will be taken from
 *	the area pointed to by InitPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_send_initmsg(
int			SockFD,
initiator_msg_t		*InitPtr)
{
    return(mvrprot_send_initmsg_timeout(SockFD,InitPtr,0));
}

long
mvrprot_send_initmsg_timeout(
int			SockFD,
initiator_msg_t		*InitPtr,
int			SecTimeout)
{
    register long	ret;
    char		buffer[MVRPROT_INITMSG_SIZE];

    /*
     *  Build the delimiter and checksum.
     */

    MVRPROT_BUILDDELIM(InitPtr->Delimiter);
    build_initmsg_checksum(InitPtr,&InitPtr->CheckSum);

    /*
     *  Now encode the fields.
     */

    mvrprot_encode_64(buffer,MVRPROT_DELIM_OFFSET,InitPtr->Delimiter);
    mvrprot_encode_32(buffer,MVRPROT_FLAGS_OFFSET,InitPtr->Flags);
    mvrprot_encode_32(buffer,MVRPROT_INIT_TYPE_OFFSET,InitPtr->Type);
    mvrprot_encode_64(buffer,MVRPROT_INIT_OFFS_OFFSET,InitPtr->Offset);
    mvrprot_encode_64(buffer,MVRPROT_INIT_LEN_OFFSET,InitPtr->Length);
    mvrprot_encode_64(buffer,MVRPROT_INIT_BLKSZ_OFFSET,InitPtr->BlockSize);
    mvrprot_encode_64(buffer,MVRPROT_INIT_STRW_OFFSET,InitPtr->StripeWidth);
    mvrprot_encode_64(buffer,MVRPROT_INIT_STRIDE_OFFSET,InitPtr->Stride);
    mvrprot_encode_64(buffer,MVRPROT_INIT_TOTLEN_OFFSET,InitPtr->TotalLength);
    mvrprot_encode_chars(buffer,
			 MVRPROT_INIT_SECTKT_OFFSET,
			 InitPtr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_encode_64(buffer,MVRPROT_INIT_CHKSUM_OFFSET,InitPtr->CheckSum);

    /*
     *  Now send the buffer.
     */

    ret = mvrprot_send_data_timeout(SockFD,
				    buffer,
				    MVRPROT_INITMSG_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_INITMSG_SIZE)
	return(ret);

    return(HPSS_E_NOERROR);
}

/*============================================================================
 *
 * Function:
 *	mvrprot_recv_ipaddr - attempts to receive an IP address message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_recv_ipaddr(
 *	int			SockFD,
 *	initiator_ipaddr_t	*InitIpPtr
 *	)
 *
 * Description:
 *	This routine attempts to read an IP address message using
 *	the socket specified by SockFD.  The message will be placed in
 *	the area pointed to by InitIpPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	InitIpPtr - received IP address message.
 *
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		HPSS_EINVAL	- Incorrect checksum.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_recv_ipaddr(
int			SockFD,
initiator_ipaddr_t	*InitIpPtr)
{
    return(mvrprot_recv_ipaddr_timeout(SockFD,InitIpPtr,0));
}

long
mvrprot_recv_ipaddr_timeout(
int			SockFD,
initiator_ipaddr_t	*InitIpPtr,
int			SecTimeout)
{
    register long	ret;
    u_signed64		checksum;
    char		buffer[MVRPROT_IPADDR_SIZE];

    /*
     *  First receive the message.
     */

    ret = mvrprot_recv_data_timeout(SockFD,
				    buffer,
				    MVRPROT_IPADDR_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_IPADDR_SIZE)
	return(ret);

    /*
     *  Now decode the fields.
     */

    mvrprot_decode_64(buffer,MVRPROT_DELIM_OFFSET,&InitIpPtr->Delimiter);
    mvrprot_decode_32(buffer,MVRPROT_FLAGS_OFFSET,&InitIpPtr->Flags);
    mvrprot_decode_64(buffer,
		      MVRPROT_IPADDR_XFERID_OFFSET,
		      &InitIpPtr->IpAddr.SockTransferID);
    mvrprot_decode_32(buffer,
		      MVRPROT_IPADDR_ADDR_OFFSET,
		      (unsigned32 *)&InitIpPtr->IpAddr.SockAddr.addr);
    mvrprot_decode_32(buffer,
		      MVRPROT_IPADDR_PORT_OFFSET,
		      (unsigned32 *)&InitIpPtr->IpAddr.SockAddr.port);
    mvrprot_decode_32(buffer,
		      MVRPROT_IPADDR_FAMILY_OFFSET,
		      (unsigned32 *)&InitIpPtr->IpAddr.SockAddr.family);
    mvrprot_decode_64(buffer,
		      MVRPROT_IPADDR_OFFS_OFFSET,
		      &InitIpPtr->IpAddr.SockOffset);
    mvrprot_decode_chars(buffer,
			 MVRPROT_IPADDR_SECTKT_OFFSET,
			 InitIpPtr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_decode_64(buffer,
		      MVRPROT_IPADDR_CHKSUM_OFFSET,
		      &InitIpPtr->CheckSum);

    /*
     *  Verify the checksum.
     */

    build_ipaddr_checksum(InitIpPtr,&checksum);
    if (neq64m(InitIpPtr->CheckSum,checksum))
	return(HPSS_EINVAL);

    return(HPSS_E_NOERROR);
}

/*============================================================================
 *
 * Function:
 *	mvrprot_send_ipaddr - attempts to send an IP address message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_send_ipaddr(
 *	int			SockFD,
 *	initiator_ipaddr_t	*InitIpPtr)
 *	)
 *
 * Description:
 *	This routine attempts to send an IP address message using
 *	the socket specified by SockFD.  The message will be taken from
 *	the area pointed to by InitIpPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_send_ipaddr(
int			SockFD,
initiator_ipaddr_t	*InitIpPtr)
{
    return(mvrprot_send_ipaddr_timeout(SockFD,InitIpPtr,0));
}

long
mvrprot_send_ipaddr_timeout(
int			SockFD,
initiator_ipaddr_t	*InitIpPtr,
int			SecTimeout)
{
    register long	ret;
    char		buffer[MVRPROT_IPADDR_SIZE];

    /*
     *  Build the delimiter and checksum.
     */

    MVRPROT_BUILDDELIM(InitIpPtr->Delimiter);
    build_ipaddr_checksum(InitIpPtr,&InitIpPtr->CheckSum);

    /*
     *  First encode the fields.
     */

    mvrprot_encode_64(buffer,MVRPROT_DELIM_OFFSET,InitIpPtr->Delimiter);
    mvrprot_encode_32(buffer,MVRPROT_FLAGS_OFFSET,InitIpPtr->Flags);
    mvrprot_encode_64(buffer,
		      MVRPROT_IPADDR_XFERID_OFFSET,
		      InitIpPtr->IpAddr.SockTransferID);
    mvrprot_encode_32(buffer,
		      MVRPROT_IPADDR_ADDR_OFFSET,
		      InitIpPtr->IpAddr.SockAddr.addr);
    mvrprot_encode_32(buffer,
		      MVRPROT_IPADDR_PORT_OFFSET,
		      InitIpPtr->IpAddr.SockAddr.port);
    mvrprot_encode_32(buffer,
		      MVRPROT_IPADDR_FAMILY_OFFSET,
		      InitIpPtr->IpAddr.SockAddr.family);
    mvrprot_encode_64(buffer,
		      MVRPROT_IPADDR_OFFS_OFFSET,
		      InitIpPtr->IpAddr.SockOffset);
    mvrprot_encode_chars(buffer,
			 MVRPROT_IPADDR_SECTKT_OFFSET,
			 InitIpPtr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_encode_64(buffer,
		      MVRPROT_IPADDR_CHKSUM_OFFSET,
		      InitIpPtr->CheckSum);

    /*
     *  Now send the buffer.
     */

    ret = mvrprot_send_data_timeout(SockFD,
				    buffer,
				    MVRPROT_IPADDR_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_IPADDR_SIZE)
	return(ret);

    return(HPSS_E_NOERROR);
}

/*============================================================================
 *
 * Function:
 *	mvrprot_recv_ipi3addr - attempts to receive an IPI-3 address message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_recv_ipi3addr(
 *	int			SockFD,
 *	initiator_ipi3addr_t	*InitIpi3Ptr)
 *	)
 *
 * Description:
 *	This routine attempts to read an IPI-3 address message using
 *	the socket specified by SockFD.  The message will be placed in
 *	the area pointed to by InitIpi3Ptr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	InitIpi3Ptr - received IPI-3 address message.
 *
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		HPSS_EINVAL	- Incorrect checksum.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_recv_ipi3addr(
int			SockFD,
initiator_ipi3addr_t	*InitIpi3Ptr)
{
    return(mvrprot_recv_ipi3addr_timeout(SockFD,InitIpi3Ptr,0));
}

long
mvrprot_recv_ipi3addr_timeout(
int			SockFD,
initiator_ipi3addr_t	*InitIpi3Ptr,
int			SecTimeout)
{
    register long	ret;
    u_signed64		checksum;
    char		buffer[MVRPROT_IPI3ADDR_SIZE];

    /*
     *  First receive the data.
     */

    ret = mvrprot_recv_data_timeout(SockFD,
				    buffer,
				    MVRPROT_IPI3ADDR_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_IPI3ADDR_SIZE)
	return(ret);

    /*
     *  Now decode the fields.
     */

    mvrprot_decode_64(buffer,MVRPROT_DELIM_OFFSET,&InitIpi3Ptr->Delimiter);
    mvrprot_decode_32(buffer,MVRPROT_FLAGS_OFFSET,&InitIpi3Ptr->Flags);
    mvrprot_decode_32(buffer,
		      MVRPROT_IPIADDR_XFERID_OFFSET,
		      &InitIpi3Ptr->Ipi3Addr.IPI3TransferID);
    mvrprot_decode_16(buffer,
		      MVRPROT_IPIADDR_INTRFC_OFFSET,
		      (unsigned16 *)&InitIpi3Ptr->Ipi3Addr.IPI3Addr.Interface);
    mvrprot_decode_chars(buffer,
			 MVRPROT_IPIADDR_NAME_OFFSET,
			 (char *)InitIpi3Ptr->Ipi3Addr.IPI3Addr.Name,
			 sizeof(InitIpi3Ptr->Ipi3Addr.IPI3Addr.Name));
    mvrprot_decode_64(buffer,
		      MVRPROT_IPIADDR_OFFS_OFFSET,
		      &InitIpi3Ptr->Ipi3Addr.IPI3Offset);
    mvrprot_decode_chars(buffer,
			 MVRPROT_IPIADDR_SECTKT_OFFSET,
			 InitIpi3Ptr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_decode_64(buffer,
		      MVRPROT_IPIADDR_CHKSUM_OFFSET,
		      &InitIpi3Ptr->CheckSum);

    /*
     *  Verify the checksum.
     */

    build_ipi3addr_checksum(InitIpi3Ptr,&checksum);
    if (neq64m(InitIpi3Ptr->CheckSum,checksum))
	return(HPSS_EINVAL);

    return(HPSS_E_NOERROR);
}

/*============================================================================
 *
 * Function:
 *	mvrprot_send_ipi3addr - attempts to send an IPI-3 address message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_send_ipi3addr(
 *	int			SockFD,
 *	initiator_ipi3addr_t	*InitIpi3Ptr)
 *	)
 *
 * Description:
 *	This routine attempts to send an IPI-3 address message using
 *	the socket specified by SockFD.  The message will be taken from
 *	the area pointed to by InitIpi3Ptr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_send_ipi3addr(
int			SockFD,
initiator_ipi3addr_t	*InitIpi3Ptr)
{
    return(mvrprot_send_ipi3addr_timeout(SockFD,InitIpi3Ptr,0));
}

long
mvrprot_send_ipi3addr_timeout(
int			SockFD,
initiator_ipi3addr_t	*InitIpi3Ptr,
int			SecTimeout)
{
    register long	ret;
    char		buffer[MVRPROT_IPI3ADDR_SIZE];

    /*
     *  Build the delimiter and checksum.
     */

    MVRPROT_BUILDDELIM(InitIpi3Ptr->Delimiter);
    build_ipi3addr_checksum(InitIpi3Ptr,&InitIpi3Ptr->CheckSum);

    /*
     *  First encode the fields.
     */

    mvrprot_encode_64(buffer,MVRPROT_DELIM_OFFSET,InitIpi3Ptr->Delimiter);
    mvrprot_encode_32(buffer,MVRPROT_FLAGS_OFFSET,InitIpi3Ptr->Flags);
    mvrprot_encode_32(buffer,
		      MVRPROT_IPIADDR_XFERID_OFFSET,
		      (unsigned32)InitIpi3Ptr->Ipi3Addr.IPI3TransferID);
    mvrprot_encode_16(buffer,
		      MVRPROT_IPIADDR_INTRFC_OFFSET,
		      (unsigned16)InitIpi3Ptr->Ipi3Addr.IPI3Addr.Interface);
    mvrprot_encode_chars(buffer,
			 MVRPROT_IPIADDR_NAME_OFFSET,
			 (char *)InitIpi3Ptr->Ipi3Addr.IPI3Addr.Name,
			 sizeof(InitIpi3Ptr->Ipi3Addr.IPI3Addr.Name));
    mvrprot_encode_64(buffer,
		      MVRPROT_IPIADDR_OFFS_OFFSET,
		      InitIpi3Ptr->Ipi3Addr.IPI3Offset);
    mvrprot_encode_chars(buffer,
			 MVRPROT_IPIADDR_SECTKT_OFFSET,
			 InitIpi3Ptr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_encode_64(buffer,
		      MVRPROT_IPIADDR_CHKSUM_OFFSET,
		      InitIpi3Ptr->CheckSum);

    /*
     *  Now send the buffer.
     */

    ret = mvrprot_send_data_timeout(SockFD,
				    buffer,
				    MVRPROT_IPI3ADDR_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_IPI3ADDR_SIZE)
	return(ret);

    return(HPSS_E_NOERROR);
}

/*============================================================================
 *
 * Function:
 *	mvrprot_recv_shmaddr - attempts to receive a shared memory
 *				address message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_recv_shmaddr(
 *	int			SockFD,
 *	initiator_shmaddr_t	*InitShmPtr
 *	)
 *
 * Description:
 *	This routine attempts to read a shared memory address message using
 *	the socket specified by SockFD.  The message will be placed in
 *	the area pointed to by InitShmPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	InitShmPtr - received shared memory address message.
 *
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		HPSS_EINVAL	- Incorrect checksum.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_recv_shmaddr(
int			SockFD,
initiator_shmaddr_t	*InitShmPtr)
{
    return(mvrprot_recv_shmaddr_timeout(SockFD,InitShmPtr,0));
}

long
mvrprot_recv_shmaddr_timeout(
int			SockFD,
initiator_shmaddr_t	*InitShmPtr,
int			SecTimeout)
{
    register long	ret;
    u_signed64		checksum;
    char		buffer[MVRPROT_SHMADDR_SIZE];

    /*
     *  First receive the data.
     */

    ret = mvrprot_recv_data_timeout(SockFD,
				    buffer,
				    MVRPROT_SHMADDR_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_SHMADDR_SIZE)
	return(ret);

    /*
     *  Now decode the fields.
     */

    mvrprot_decode_64(buffer,MVRPROT_DELIM_OFFSET,&InitShmPtr->Delimiter);
    mvrprot_decode_32(buffer,MVRPROT_FLAGS_OFFSET,&InitShmPtr->Flags);
    mvrprot_decode_32(buffer,
		      MVRPROT_SHMADDR_FLAGS_OFFSET,
		      &InitShmPtr->ShmAddr.Flags);
    mvrprot_decode_32(buffer,
		      MVRPROT_SHMADDR_SHMID_OFFSET,
		      (unsigned32 *)&InitShmPtr->ShmAddr.ShmID);
    mvrprot_decode_32(buffer,
		      MVRPROT_SHMADDR_OFFS_OFFSET,
		      &InitShmPtr->ShmAddr.ShmOffset);
    mvrprot_decode_chars(buffer,
			 MVRPROT_SHMADDR_SECTKT_OFFSET,
			 InitShmPtr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_decode_64(buffer,
		      MVRPROT_SHMADDR_CHKSUM_OFFSET,
		      &InitShmPtr->CheckSum);

    /*
     *  Verify the checksum.
     */

    build_shmaddr_checksum(InitShmPtr,&checksum);
    if (neq64m(InitShmPtr->CheckSum,checksum))
	return(HPSS_EINVAL);

    return(HPSS_E_NOERROR);
}

/*============================================================================
 *
 * Function:
 *	mvrprot_send_shmaddr - attempts to send a shared memory address message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_send_shmaddr(
 *	int			SockFD,
 *	initiator_shmaddr_t	*InitShmPtr
 *	)
 *
 * Description:
 *	This routine attempts to send a shared memory address message using
 *	the socket specified by SockFD.  The message will be taken from
 *	the area pointed to by InitShmPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_send_shmaddr(
int			SockFD,
initiator_shmaddr_t	*InitShmPtr)
{
    return(mvrprot_send_shmaddr_timeout(SockFD,InitShmPtr,0));
}

long
mvrprot_send_shmaddr_timeout(
int			SockFD,
initiator_shmaddr_t	*InitShmPtr,
int			SecTimeout)
{
    register long	ret;
    char		buffer[MVRPROT_SHMADDR_SIZE];

    /*
     *  Build the delimiter and checksum.
     */

    MVRPROT_BUILDDELIM(InitShmPtr->Delimiter);
    build_shmaddr_checksum(InitShmPtr,&InitShmPtr->CheckSum);

    /*
     *  First encode the fields.
     */

    mvrprot_encode_64(buffer,MVRPROT_DELIM_OFFSET,InitShmPtr->Delimiter);
    mvrprot_encode_32(buffer,MVRPROT_FLAGS_OFFSET,InitShmPtr->Flags);
    mvrprot_encode_32(buffer,
		      MVRPROT_SHMADDR_FLAGS_OFFSET,
		      InitShmPtr->ShmAddr.Flags);
    mvrprot_encode_32(buffer,
		      MVRPROT_SHMADDR_SHMID_OFFSET,
		      InitShmPtr->ShmAddr.ShmID);
    mvrprot_encode_32(buffer,
		      MVRPROT_SHMADDR_OFFS_OFFSET,
		      InitShmPtr->ShmAddr.ShmOffset);
    mvrprot_encode_chars(buffer,
			 MVRPROT_SHMADDR_SECTKT_OFFSET,
			 InitShmPtr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_encode_64(buffer,
		      MVRPROT_SHMADDR_CHKSUM_OFFSET,
		      InitShmPtr->CheckSum);

    /*
     *  Now send the buffer.
     */

    ret = mvrprot_send_data_timeout(SockFD,
				    buffer,
				    MVRPROT_SHMADDR_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_SHMADDR_SIZE)
	return(ret);

    return(HPSS_E_NOERROR);
}

/*============================================================================
 *
 * Function:
 *	mvrprot_recv_compmsg - attempts to receive a completion message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_recv_compmsg(
 *	int			SockFD,
 *	completion_msg_t	*CompPtr
 *	)
 *
 * Description:
 *	This routine attempts to read a completion message using
 *	the socket specified by SockFD.  The message will be placed in
 *	the area pointed to by CompPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	CompPtr - received completion message.
 *
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		HPSS_EINVAL	- Incorrect checksum.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_recv_compmsg(
int			SockFD,
completion_msg_t	*CompPtr)
{
    return(mvrprot_recv_compmsg_timeout(SockFD,CompPtr,0));
}

long
mvrprot_recv_compmsg_timeout(
int			SockFD,
completion_msg_t	*CompPtr,
int			SecTimeout)
{
    register long	ret;
    u_signed64		checksum;
    char		buffer[MVRPROT_COMPMSG_SIZE];

    /*
     *  First receive the data.
     */

    ret = mvrprot_recv_data_timeout(SockFD,
				    buffer,
				    MVRPROT_COMPMSG_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_COMPMSG_SIZE)
	return(ret);

    /*
     *  Now decode the fields.
     */

    mvrprot_decode_64(buffer,MVRPROT_DELIM_OFFSET,&CompPtr->Delimiter);
    mvrprot_decode_32(buffer,MVRPROT_FLAGS_OFFSET,&CompPtr->Flags);
    mvrprot_decode_32(buffer,MVRPROT_COMP_STATUS_OFFSET,&CompPtr->Status);
    mvrprot_decode_64(buffer,MVRPROT_COMP_BMOVED_OFFSET,&CompPtr->BytesMoved);
    mvrprot_decode_chars(buffer,
			 MVRPROT_COMP_SECTKT_OFFSET,
			 CompPtr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_decode_64(buffer,MVRPROT_COMP_CHKSUM_OFFSET,&CompPtr->CheckSum);

    /*
     *  Verify the checksum.
     */

    build_compmsg_checksum(CompPtr,&checksum);
    if (neq64m(CompPtr->CheckSum,checksum))
	return(HPSS_EINVAL);

    return(HPSS_E_NOERROR);
}

/*============================================================================
 *
 * Function:
 *	mvrprot_send_compmsg - attempts to send a completion message.
 *
 * Synopsis:
 *
 *	long
 *	mvrprot_send_compmsg(
 *	int			SockFD,
 *	completion_msg_t	*CompPtr
 *	)
 *
 * Description:
 *	This routine attempts to send an initialization message using
 *	the socket specified by SockFD.  The message will be taken from
 *	the area pointed to by CompPtr.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	Return value:
 *		HPSS_E_NOERROR	- Success.
 *		
 *
 * Interfaces:
 *
 * Resources Used:
 *
 * Limitations:
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

long
mvrprot_send_compmsg(
int			SockFD,
completion_msg_t	*CompPtr)
{
    return(mvrprot_send_compmsg_timeout(SockFD,CompPtr,0));
}

long
mvrprot_send_compmsg_timeout(
int			SockFD,
completion_msg_t	*CompPtr,
int			SecTimeout)
{
    register long	ret;
    char		buffer[MVRPROT_COMPMSG_SIZE];

    /*
     *  Build the delimiter and checksum.
     */

    MVRPROT_BUILDDELIM(CompPtr->Delimiter);
    build_compmsg_checksum(CompPtr,&CompPtr->CheckSum);

    /*
     *  First encode the fields.
     */

    mvrprot_encode_64(buffer,MVRPROT_DELIM_OFFSET,CompPtr->Delimiter);
    mvrprot_encode_32(buffer,MVRPROT_FLAGS_OFFSET,CompPtr->Flags);
    mvrprot_encode_32(buffer,MVRPROT_COMP_STATUS_OFFSET,CompPtr->Status);
    mvrprot_encode_64(buffer,MVRPROT_COMP_BMOVED_OFFSET,CompPtr->BytesMoved);
    mvrprot_encode_chars(buffer,
			 MVRPROT_COMP_SECTKT_OFFSET,
			 CompPtr->SecurityTicket,
			 MVRPROT_TICKET_LENGTH);
    mvrprot_encode_64(buffer,MVRPROT_COMP_CHKSUM_OFFSET,CompPtr->CheckSum);

    /*
     *  Now send the buffer.
     */

    ret = mvrprot_send_data_timeout(SockFD,
				    buffer,
				    MVRPROT_COMPMSG_SIZE,
				    SecTimeout);
    if (ret != MVRPROT_COMPMSG_SIZE)
	return(ret);

    return(HPSS_E_NOERROR);
}
/* static char SccsId[] = "src/cs/pdata/pdata.c, gen, 3v2   12/4/97   14:58:48   "; */
/*============================================================================
 *
 * Name:        pdata.c
 *
 * Functions:
 *		build_pdata_checksum - Build a check sum for the packet header
 *				so that the receiver can verify that the packet
 *				was undamaged during transmission. This routine
 *				is machine dependent and will have to be
 *				altered when the system is ported to a machine
 *				other than a 32-bit big-endian.
 *              pdata_recv_hdr - receive the header that describes the data
 *				that has been sent in a parallel transfer.
 *              pdata_send_hdr - send a header that describes either the data
 *				that follows the transfer, or a header that
 *				will request data to be received.
 *              pdata_send_hdr_and_data - send a header that describes the
 *				data that will be sent.
 * Description:
 *              Routines to send and recv headers that describe parallel data,
 *		and send the data if desired.
 *
 * Traceability:
 *      Version         Author          Date            Description
 *      -------         ------          ----            -----------
 *      0.1             RPR             03/18/93	Initial
 *      1.2     			07/05/94   	Copyright Update
 *	3.1		kwe		10/04/95	Socket timeouts
 *	4.2		WHR		01/29/97	Added portability code
 *	4.3	WHR	01/29/97	Corrected Portability code - changed 
 *					_HPUX10 to __HPUX__  Defined cptr
 *					for BYTE8INT.  Added BYTE8INT, LITTLEEND
 *					requirements.
 *	4.4	WHR	11/04/97	Added #include <netinet/in.h>
 *	4.5	kwe	11/12/97	Fix SecurityTicket in checksum.
 *	4.6	WHR	12/12/97	Remove register statements
 *
 *  Notes:
 *       Licensed Materials
 *
 *       (C) Copyright International Business Machines Corp. 1994
 *       (C) Copyright Martin Marietta Energy Systems, Inc. and
 *                     Oak Ridge National Laboratory 1994
 *                     under CRADA No. Y1293-0203
 *       (C) Copyright Regents of The University of California and
 *                     Lawrence Livermore National Laboratory 1994
 *                     under CRADA No. T-253-92-C (as modified by T-325-92-C)
 *       (C) Copyright Regents of The University of California and
 *                     Los Alamos National Laboratory 1994
 *                     under CRADA No. LA93C10085
 *       (C) Copyright Sandia Corporation and Sandia National Laboratories 1994
 *                     under CRADA No. SC93/01198
 * 
 *       All rights reserved.
 * 
 *       This file is IBM Confidential.
 *
 *		These function is dependent on a 64-bit math library.
 *
 *-------------------------------------------------------------------------*/

#include <sys/types.h>
#include <stdio.h>
#include <sys/time.h>
#if ! defined(__PARAGON__) && ! defined(__HPUX__) && ! defined(NeXT)
#include <sys/select.h>
#endif /* __PARAGON__ */
#include <errno.h>

#if defined(THREADS_ENABLED)
#include <pthread.h>
#endif

#include <netinet/in.h>
#include "u_signed64.h"
#include "pdata.h"
#include "mvr_protocol.h"

/*============================================================================
 *
 * Function:
 *              build_pdata_checksum - build a check sum to protect against
 *					data corruption.
 *
 * Synopsis:
 *
 *	static void
 *	build_pdata_checksum(
 *	pdata_hdr_t	*PDHdr;   	** where to receive the header
 *	u_signed64	*CheckSum;	** return checksum
 *	)
 *
 * Description:
 *	Build a check sum to protect against data corruption. Give receiver the
 *	ability to double check the sender.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	None
 *
 * Interfaces:
 *	add64m
 *
 * Resources Used:
 *
 * Limitations:
 *	This code is currently not portable. Must modify to run an any machine
 *	other than a 32-bit big_endian machine.
 *
 * Assumptions:
 *
 * Notes:
 *	Initial implementation is for RISC System 6000. It should also run on
 *	any machine that is 32-bit big_endian.
 *
 *-------------------------------------------------------------------------*/

static void
build_pdata_checksum(
pdata_hdr_t	*PDHdr,
u_signed64	*CheckSum
)
{
	u_signed64	i1,i2,i3,sectkt;

	i1 = add64m(PDHdr->PdataDelimiter,PDHdr->XferID);
	i2 = add64m(PDHdr->Offset,PDHdr->Length);
	i3 = add64m(i1,i2);

	*CheckSum = add64m(i3,mem64m(PDHdr->SecurityTicket));
	return;
}

/*============================================================================
 *
 * Function:
 *              pdata_recv_hdr - receive and parallel data transfer header.
 *
 * Synopsis:
 *
 *	int
 *	pdata_recv_hdr(
 *	int		SocketDescriptor, ** socket to receive header on
 *	pdata_hdr_t	*PdataHeaderPtr   ** where to receive the header
 *	)
 *
 * Description:
 *      Receive and parallel data header and make sure that it has not been
 *	corrupted.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	positive:  the number of bytes sent (should == sizeof(pdata_hdr_t))
 *      0:      the sender closed the socket
 *      -1:	Error receiving header.
 *		errno value:
 *
 *		EBADF - The SocketDescriptor is not an open socket.
 *		EINVAL - internal pdata_recv_hdr failure (bad code).
 *		ENOTSOCK - The SocketDescriptor is not a socket.
 *		EFAULT - PdataHeaderPtr is an invalid address.
 *		EIO - The header was corrupted in transit.
 *
 * Interfaces:
 *	recv, memcmp, build_pdata_checksum
 *
 * Resources Used:
 *
 * Limitations:
 *	The pdata_hdr_t and the code to check sum the header will have to be
 *	changed to port to a machine other than 32-bit big-endian.
 *
 * Assumptions:
 *
 * Notes:
 *	Initial implementation is for RISC System 6000. It should also run on
 *	any machine that is 32-bit big_endian.
 *	- spg/whr/mkg: ported to Intel Paragon (32 bit little-endian machine)
 *
 *-------------------------------------------------------------------------*/

int
pdata_recv_hdr(
int		SocketDescriptor,	/* socket to receive header */
pdata_hdr_t	*PdataHeaderPtr		/* where to receive the header */
)
{
	return(pdata_recv_hdr_timeout(SocketDescriptor,PdataHeaderPtr,0));
}

int
pdata_recv_hdr_timeout(
int		SocketDescriptor,	/* socket to receive header */
pdata_hdr_t	*PdataHeaderPtr,	/* where to receive the header */
int		SecTimeout		/* socket timeout in seconds */
)
{
	int	leng = 0;
	int	ret,nfd;
	char		*cptr;
	struct timeval	timeout,*timeptr;
	fd_set		rdlist,wrlist,exlist;
	fd_set		save_rdlist,save_wrlist,save_exlist;
	u_signed64	checksum;
	char		buffer[PDATA_HDR_SIZE];

	/*
	 *  Set up receive lists.
	 */

	FD_ZERO(&save_rdlist);
	FD_ZERO(&save_wrlist);
	FD_ZERO(&save_exlist);

	FD_SET(SocketDescriptor,&save_rdlist);

	/*
	 *  try to receive in header
	 */

	cptr = buffer;
	while(leng < PDATA_HDR_SIZE)
	{
	   if (SecTimeout == 0)
	      timeptr = (struct timeval *)NULL;
	   else
	   {
	      timeout.tv_sec = SecTimeout;
	      timeout.tv_usec = 0;
	      timeptr = &timeout;
	   }

	   rdlist = save_rdlist;
	   wrlist = save_wrlist;
	   exlist = save_exlist;

	   nfd = select(FD_SETSIZE,&rdlist,&wrlist,&exlist,timeptr);
	   if (nfd < 0)
	   {
	      if((errno == EINTR) || (errno == EAGAIN)) continue;

	      /*
	       *  If we got any other error, bail out.
	       */

	      return(-errno);
	   }

	   /*
	    *  If the select timed out, return -ETIMEDOUT.  Otherwise,
	    *  it should be safe to use the socket, since we only
	    *  asked about one.
	    */

	   if (nfd == 0)
	      return(-ETIMEDOUT);

	   ret = read(SocketDescriptor,cptr,PDATA_HDR_SIZE - leng);
	   if(ret < 0)
	   {
	      if((errno == EINTR) || (errno == EAGAIN)) continue;

	      /*
	       *  Either we got an error or the sender closed.
	       */

	      return(-errno);
	   }
	   else if (ret == 0)
	   {
	      return(0);
	   }
	   cptr += ret;
	   leng += ret;
	}

	mvrprot_decode_64(buffer,
			  PDATA_DELIM_OFFSET,
			  &PdataHeaderPtr->PdataDelimiter);
	mvrprot_decode_64(buffer,
			  PDATA_XFERID_OFFSET,
			  &PdataHeaderPtr->XferID);
	mvrprot_decode_64(buffer,
			  PDATA_OFFS_OFFSET,
			  &PdataHeaderPtr->Offset);
	mvrprot_decode_64(buffer,
			  PDATA_LENGTH_OFFSET,
			  &PdataHeaderPtr->Length);
	mvrprot_decode_chars(buffer,
			PDATA_SECTKT_OFFSET,
			PdataHeaderPtr->SecurityTicket,
			PDATA_TICKET_LENGTH);
	mvrprot_decode_64(buffer,
			  PDATA_CHKSUM_OFFSET,
			  &PdataHeaderPtr->CheckSum);

	/*
	 *  Now check that the delimiter is set correctly
	 */

	if( !CHECKPDATADELIMITER(PdataHeaderPtr->PdataDelimiter) )
	{
	   return(-EINVAL);
	}
	/*
	 *  Now build the checksum and verify that it was not corrupted
	 */

	build_pdata_checksum(PdataHeaderPtr,&checksum);
	if(eq64m(checksum,PdataHeaderPtr->CheckSum))
	{
	   return(sizeof(pdata_hdr_t));
	}

	return(-EINVAL);
}

/*============================================================================
 *
 * Function:
 *              pdata_send_hdr - send a parallel data transfer header. This
 *				routine can be used with the standard call
 *				'send' to transfer data, or it can be used to
 *				request data across a parallel connection.
 *
 * Synopsis:
 *
 *	int
 *	pdata_send_hdr(
 *	int		SocketDescriptor, ** socket to send header on
 *	pdata_hdr_t	*PdataHeaderPtr   ** header to send
 *	)
 *
 * Description:
 *	Send a parallel data transfer header. This routine can be used to 
 *	request that data be sent across a parallel connection, or it can be
 *	used to send a header that will describe the data to follow.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	positive:  the number of bytes sent (should == sizeof(pdata_hdr_t))
 *      -1:	Error receiving header.
 *		errno value:
 *
 *		EBADF - The SocketDescriptor is not an open socket.
 *		EINVAL - internal pdata_recv_hdr failure (bad code).
 *		ENOTSOCK - The SocketDescriptor is not a socket.
 *		ENOTCONN - the receiver close to connection.
 *		EFAULT - PdataHeaderPtr is an invalid address.
 *		EIO - The header was corrupted in transit.
 *
 * Interfaces:
 *	send, build_pdata_checksum
 *
 * Resources Used:
 *
 * Limitations:
 *	The pdata_hdr_t and the code to check sum the header will have to be
 *	changed to port to a machine other than 32-bit big-endian.
 *
 * Assumptions:
 *
 * Notes:
 *	Initial implementation is for RISC System 6000. It should also run on
 *	any machine that is 32-bit big_endian.
 * - spg/whr/mkg: ported to Intel Paragon (32 bit little-endian machine)
 *
 *-------------------------------------------------------------------------*/

int
pdata_send_hdr(
int		SocketDescriptor,	/* socket to send header */
pdata_hdr_t	*PdataHeaderPtr		/* header to send */
)
{
	return(pdata_send_hdr_timeout(SocketDescriptor,PdataHeaderPtr,0));
}

int
pdata_send_hdr_timeout(
int		SocketDescriptor,	/* socket to send header */
pdata_hdr_t	*PdataHeaderPtr,	/* header to send */
int		SecTimeout
)
{
	int	leng = 0;
	int	ret,nfd;
	char		*cptr;
	struct timeval	timeout,*timeptr;
	fd_set		rdlist,wrlist,exlist;
	fd_set		save_rdlist,save_wrlist,save_exlist;
	u_signed64	checksum;
	char		buffer[PDATA_HDR_SIZE];

	/*
	 *  Load the delimiter and the check sum into the parallel transfer hdr
	 */

	SETPDATADELIMITER(PdataHeaderPtr->PdataDelimiter);
	build_pdata_checksum(PdataHeaderPtr,&PdataHeaderPtr->CheckSum);

	mvrprot_encode_64(buffer,
			  PDATA_DELIM_OFFSET,
			  PdataHeaderPtr->PdataDelimiter);
	mvrprot_encode_64(buffer,
			  PDATA_XFERID_OFFSET,
			  PdataHeaderPtr->XferID);
	mvrprot_encode_64(buffer,
			  PDATA_OFFS_OFFSET,
			  PdataHeaderPtr->Offset);
	mvrprot_encode_64(buffer,
			  PDATA_LENGTH_OFFSET,
			  PdataHeaderPtr->Length);
	mvrprot_encode_chars(buffer,
			PDATA_SECTKT_OFFSET,
			PdataHeaderPtr->SecurityTicket,
			PDATA_TICKET_LENGTH);
	mvrprot_encode_64(buffer,
			  PDATA_CHKSUM_OFFSET,
			  PdataHeaderPtr->CheckSum);

	/*
	 *  Set up send lists.
	 */

	FD_ZERO(&save_rdlist);
	FD_ZERO(&save_wrlist);
	FD_ZERO(&save_exlist);

	FD_SET(SocketDescriptor,&save_wrlist);

	/*
	 *  try to send in header
	 */

	cptr = buffer;
	while(leng < PDATA_HDR_SIZE)
	{
	   if (SecTimeout == 0)
	      timeptr = (struct timeval *)NULL;
	   else
	   {
	      timeout.tv_sec = SecTimeout;
	      timeout.tv_usec = 0;
	      timeptr = &timeout;
	   }

	   rdlist = save_rdlist;
	   wrlist = save_wrlist;
	   exlist = save_exlist;

	   nfd = select(FD_SETSIZE,&rdlist,&wrlist,&exlist,timeptr);
	   if (nfd < 0)
	   {
	      if((errno == EINTR) || (errno == EAGAIN)) continue;

	      /*
	       *  If we got any other error, bail out.
	       */

	      return(-errno);
	   }

	   /*
	    *  If the select timed out, return -ETIMEDOUT.  Otherwise,
	    *  it should be safe to use the socket, since we only
	    *  asked about one.
	    */

	   if (nfd == 0)
	      return(-ETIMEDOUT);

	   ret = write(SocketDescriptor,cptr,PDATA_HDR_SIZE - leng);
	   if(ret < 0)
	   {
	      if((errno == EINTR) || (errno == EAGAIN)) continue;

	      /*
	       *  We got a socket error.
	       */

	      return(-errno);
	   }
	   cptr += ret;
	   leng += ret;
	}
	/*
	 *  We sent the header OK
	 */

	return(PDATA_HDR_SIZE);
}

/*============================================================================
 *
 * Function:
 *              pdata_send_hdr_and_data - send a parallel data transfer header
 *					followed by the data.
 *
 * Synopsis:
 *
 *	int
 *	pdata_send_hdr_and_data(
 *	int		SocketDescriptor, ** socket to send header on
 *	pdata_hdr_t	*PdataHeaderPtr,  ** header to send
 *	char		*DataBuffer,	  ** data to be sent
 *	int		DataLength	  ** # bytes to send
 *	)
 *
 * Description:
 *	Send a parallel data transfer header followed by the header.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	positive:  the number of bytes sent (should == DataLength)
 *      -1:	Error receiving header.
 *		errno value:
 *
 *		EBADF - The SocketDescriptor is not an open socket.
 *		EINVAL - internal pdata_recv_hdr failure (bad code).
 *		ENOTSOCK - The SocketDescriptor is not a socket.
 *		ENOTCONN - the receiver close to connection.
 *		EFAULT - PdataHeaderPtr is an invalid address.
 *		EIO - The header was corrupted in transit.
 *
 * Interfaces:
 *	pdata_send_hdr_and_data, send, build_pdata_checksum
 *
 * Resources Used:
 *
 * Limitations:
 *	The pdata_hdr_t and the code to check sum the header will have to be
 *	changed to port to a machine other than 32-bit big-endian.
 *
 * Assumptions:
 *
 * Notes:
 *	Initial implementation is for RISC System 6000. It should also run on
 *	any machine that is 32-bit big_endian.
 *
 *-------------------------------------------------------------------------*/

int
pdata_send_hdr_and_data(
int		SocketDescriptor,	/* socket to send header */
pdata_hdr_t	*PdataHeaderPtr,	/* header to send */
char		*DataBuffer,		/* data to be sent */
int		DataLength		/* # bytes to transfer */
)
{
	return(pdata_send_hdr_and_data_timeout(SocketDescriptor,
					       PdataHeaderPtr,
					       DataBuffer,
					       DataLength,
					       0));
}

int
pdata_send_hdr_and_data_timeout(
int		SocketDescriptor,	/* socket to send header */
pdata_hdr_t	*PdataHeaderPtr,	/* header to send */
char		*DataBuffer,		/* data to be sent */
int		DataLength,		/* # bytes to transfer */
int		SecTimeout
)
{
	int	leng = 0;
	int	ret,nfd;
	char	*cptr;
	struct timeval	timeout,*timeptr;
	fd_set		rdlist,wrlist,exlist;
	fd_set		save_rdlist,save_wrlist,save_exlist;

	/*
	 *  Make sure the length in the header is equal to the DataLength
	 */

	if((high32m(PdataHeaderPtr->Length) != 0)
			||
	   (low32m(PdataHeaderPtr->Length) != DataLength))
	{
	   /*
	    *  The lengths differ
	    */

	   return(-EINVAL);
	}

	ret = pdata_send_hdr_timeout(SocketDescriptor,
				     PdataHeaderPtr,
				     SecTimeout);
	if(ret <= 0)
	{
	   /*
	    *  Could not send the header
	    */

	   return(ret);
	}

	/*
	 *  Set up send lists.
	 */

	FD_ZERO(&save_rdlist);
	FD_ZERO(&save_wrlist);
	FD_ZERO(&save_exlist);

	FD_SET(SocketDescriptor,&save_wrlist);

	/*
	 *  Now send the data
	 */

	while(leng < DataLength)
	{
	   if (SecTimeout == 0)
	      timeptr = (struct timeval *)NULL;
	   else
	   {
	      timeout.tv_sec = SecTimeout;
	      timeout.tv_usec = 0;
	      timeptr = &timeout;
	   }

	   rdlist = save_rdlist;
	   wrlist = save_wrlist;
	   exlist = save_exlist;

	   nfd = select(FD_SETSIZE,&rdlist,&wrlist,&exlist,timeptr);
	   if (nfd < 0)
	   {
	      if((errno == EINTR) || (errno == EAGAIN)) continue;

	      /*
	       *  If we got any other error, bail out.
	       */

	      return(-errno);
	   }

	   /*
	    *  If the select timed out, return -ETIMEDOUT.  Otherwise,
	    *  it should be safe to use the socket, since we only
	    *  asked about one.
	    */

	   if (nfd == 0)
	      return(-ETIMEDOUT);

	   ret = write(SocketDescriptor,DataBuffer,DataLength - leng);
	   if(ret < 0)
	   {
	      if((errno == EINTR) || (errno == EAGAIN)) continue;

	      /*
	       *  We got an error
	       */

	      return(-errno);
	   }
	   DataBuffer += ret;
	   leng += ret;
	}

	return(leng);
}
/* static char SccsId[] = "src/cs/u_signed64.c, gen, 3v2   12/3/97   14:26:25   "; */
/*  The Vesta Parallel File System
 *
 *  Designed and implemented by
 *
 *  Peter F. Corbett
 *  Dror G. Feitelson
 *
 *  IBM T. J. Watson Research Center
 *  P. O. Box 218
 *  Yorktown Heights, NY 10598
 *
 *  Code written beginning
 *  July 26, 1992.
 *
 *  This file is provides the 64 bit arithmetic routines used by HPSS, and
 *  is based off of code from IBM Yorktown.
 *
 *  Traceability:
 *       Version   Date       Description
 *       _______   ________   _______________________________________________ 
 *         1.0     02/18/93   Initial version
 *         1.2     06/21/94   Support div2x64m, div_cl2x64m, and mod2x64m
 *                            operations
 *                            Copyright statement update
 *         1.3     07/20/94   Correct little endian structure definitions
 *         1.4     12/12/95   Correct mul64 for 64 bit architectures
 *         4.1     05/10/96   AIX 4.1 upgrade: renamed unsigned64
 *         4.2     01/29/97   Modifications for portability
 *         4.3     06/25/97   Fixed Little Endian 32 bit u_signed64 structure.
 *         4.4     09/24/97   Corrected bad #elif in !BYTE8INT version of
 *			      mem64(). 32 BIT functions at top 64 BIT functions
 *			      at Botttom.
 *			      Changes to determine order differences between
 *			      i386 and i860.
 *         4.5     11/25/97   Correction for TERAFLOP
 *         4.8     11/25/97   Change mem64.
 *
 *  Notes:
 *       Licensed Materials
 *
 *       (C) Copyright International Business Machines Corp. 1994
 *       (C) Copyright Martin Marietta Energy Systems, Inc. and
 *                     Oak Ridge National Laboratory 1994
 *                     under CRADA No. Y1293-0203
 *       (C) Copyright Regents of The University of California and
 *                     Lawrence Livermore National Laboratory 1994
 *                     under CRADA No. T-253-92-C (as modified by T-325-92-C)
 *       (C) Copyright Regents of The University of California and
 *                     Los Alamos National Laboratory 1994
 *                     under CRADA No. LA93C10085
 *       (C) Copyright Sandia Corporation and Sandia National Laboratories 1994
 *                     under CRADA No. SC93/01198
 *
 *       All rights reserved.
 *
 */

/*
 * This file contains the functions for 64 bit arithmetic.
 * These functions are required for computations on offsets.
 */

#include <sys/types.h>

#ifndef _VESTA_ARITH64_C
#define _VESTA_ARITH64_C

#include "u_signed64.h"

#if !defined(BYTE8INT)

#if defined(LITTLEEND)

typedef union {
   u_signed64 d;
#if defined(__i386__) || defined(__TERAFLOP__)
   struct {
      unsigned16 d2;
      unsigned16 d3;
      unsigned16 d0;
      unsigned16 d1;
     } s;
#endif	/* __i386__ || __TERAFLOP__ */
#if defined(__i860__)
   struct {
      unsigned16 d0;
      unsigned16 d1;
      unsigned16 d2;
      unsigned16 d3;
     } s;
#endif	/* __i860__ */
  } lx16_t;

typedef union {
   unsigned32 d;
   struct {
      unsigned16 d0;
      unsigned16 d1;
     } s;
  } ix16_t;

#else

typedef union {
   u_signed64 d;
   struct {
      unsigned16 d3;
      unsigned16 d2;
      unsigned16 d1;
      unsigned16 d0;
     } s;
  } lx16_t;

typedef union {
   unsigned32 d;
   struct {
      unsigned16 d1;
      unsigned16 d0;
     } s;
  } ix16_t;

#endif	/* LITTLEEND */

/*
 * cast64: Cast a unsigned int 32 into a unsigned int 64.
 */
u_signed64 cast64 (unsigned32 n)
  {
   u_signed64 t;

   t.high = 0;
   t.low = n;
   return t;
  }

/*
 * bld64: build a 64-bit unsigned int out of two 32-bit unsigned int's
 */
u_signed64 bld64 (unsigned32 high,unsigned32 low)
  {
   u_signed64 t;

   t.high = high;
   t.low = low;
   return t;
  }

/*
 * mem64: builds a 64-bit unsigned int from an array of consecutive
 *        unsigned characters. The first 4 are used as the .high
 *        part, the next 4 as the .low part. 
 * Notes: 
 * - On 64-bit or big-endian machines, this is functionally identical
 *        to "*(u_signed64* pointer)". The <mem64m> macro can be used
 *        to generate this in-line.
 * - On little-endian machines, the input is assumed to be in network
 *   byte order, and is converted to host order 
 */
u_signed64 mem64(unsigned char *mem)
  {
   u_signed64 t;

   t.high = ((mem[0] << 24) & 0xFF000000) | ((mem[1] << 16) & 0x00FF0000) |
	    ((mem[2] <<  8) & 0x0000FF00) | (mem[3] & 0x000000FF);
   t.low = ((mem[4] << 24) & 0xFF000000) | ((mem[5] << 16) & 0x00FF0000) |
	   ((mem[6] <<  8) & 0x0000FF00) | (mem[7] & 0x000000FF);

   return t;
  }

/*
 * gt64: Return 1 if a1 > a2, else 0.
 */
int gt64 (u_signed64 a1, u_signed64 a2)
  {
   return ( (a1.high > a2.high) || ((a1.high == a2.high) && (a1.low > a2.low)) );
  }

/*
 * ge64: Return 1 if a1 >= a2, else 0.
 */
int ge64 (u_signed64 a1, u_signed64 a2)
  {
   return ( !((a1.high < a2.high) || ((a1.high == a2.high) && (a1.low < a2.low))) );
  }

/*
 * eq64: Return 1 if a1 = a2, else 0.
 */
int eq64 (u_signed64 a1, u_signed64 a2)
  {
   return ( (a1.high == a2.high) && (a1.low == a2.low) );
  }

/*
 * neq64: Return 1 if a1 != a2, else 0.
 */
int neq64 (u_signed64 a1, u_signed64 a2)
  {
   return ( !((a1.high == a2.high) && (a1.low == a2.low)) );
  }

/*
 * le64:Return 1 if a1 <= a2, else 0.
 */
int le64 (u_signed64 a1, u_signed64 a2)
  {
   return ( !((a1.high > a2.high) || ((a1.high == a2.high) && (a1.low > a2.low))) );
  }

/*
 * lt64: Return 1 if a1 < a2, else 0.
 */
int lt64 (u_signed64 a1, u_signed64 a2)
  {
   return ( (a1.high < a2.high) || ((a1.high == a2.high) && (a1.low < a2.low)) );
  }

/*
 * add64: Add two 64 bit unsigned numbers.  Return 64 bit unsigned result.
 */
u_signed64 add64 (u_signed64 a1, u_signed64 a2)
  {
   unsigned32 k, g;
   u_signed64 s;

   g = a1.low & a2.low;
   k = ~(a1.low | a2.low);

   s.low = a1.low + a2.low;
   s.high = a1.high + a2.high + ((g>k) ? 1 : 0);

   return (s);
  }

/*
 * sub64: Subtract two 64 bit unsigned numbers.  Return 64 bit unsigned result.
 */
u_signed64 sub64 (u_signed64 a1, u_signed64 a2)
  {
   u_signed64 s;

   s.low = a1.low - a2.low;
   s.high = (a1.high - a2.high) - ((a2.low > a1.low) ? 1 : 0);

   return (s);
  }

/*
 * mul64: Multiply a 64 bit unsigned number by a 32 bit unsigned number.
 */
u_signed64 mul64 (u_signed64 m1, unsigned32 m2)
  {
   lx16_t pp1, pp2, m1d;
   ix16_t  m2d, t, h;

   /* Cast the multipliers. */
   m1d.d = m1;
   m2d.d = m2;

   /* Compute pp1 = m1 * low half of m2. */
   t.d = m1d.s.d0 * m2d.s.d0;
   pp1.s.d0 = t.s.d0;
   pp1.s.d1 = t.s.d1;

   t.d = m1d.s.d1 * m2d.s.d0;
   h.d = pp1.s.d1 + t.s.d0;
   pp1.s.d1 = h.s.d0;
   pp1.s.d2 = t.s.d1 + h.s.d1;

   t.d = m1d.s.d2 * m2d.s.d0;
   h.d = pp1.s.d2 + t.s.d0;
   pp1.s.d2 = h.s.d0;
   pp1.s.d3 = t.s.d1 + h.s.d1;

   t.d = m1d.s.d3 * m2d.s.d0;
   h.d = pp1.s.d3 + t.s.d0;
   pp1.s.d3 = h.s.d0;

   /* Return if this is complete result. */
   if (m2d.s.d1 == 0)
      return (pp1.d);

   /* Compute pp2 = m1 * high half of m2. */
   t.d = m1d.s.d0 * m2d.s.d1;
   pp2.s.d0 = 0;
   pp2.s.d1 = t.s.d0;
   pp2.s.d2 = t.s.d1;

   t.d = m1d.s.d1 * m2d.s.d1;
   h.d = pp2.s.d2 + t.s.d0;
   pp2.s.d2 = h.s.d0;
   pp2.s.d3 = t.s.d1 + h.s.d1;

   t.d = m1d.s.d2 * m2d.s.d1;
   h.d = pp2.s.d3 + t.s.d0;
   pp2.s.d3 = h.s.d0;

   /* Now add the partial products. */
     {
      unsigned32 k, g;
      u_signed64 s;

      g = pp1.d.low & pp2.d.low;
      k = ~(pp1.d.low | pp2.d.low);

      s.low = pp1.d.low + pp2.d.low;
      s.high = pp1.d.high + pp2.d.high + ((g>k) ? 1 : 0);

      return (s);
     }
  }

 /*
  * div2x64: Divide an unsigned 64 bit integer by an unsigned 64 bit integer.
  */
 u_signed64 div2x64 (u_signed64 n, u_signed64 d)
   {
    if ( lt64m ( n, d ) )
      {
       u_signed64 q;
       q.high = q.low = 0;
       return ( q );
      }

    else if ((n.high == 0) && (d.high == 0))
      {
       n.low /= d.low;
       return (n);
      }

    else if ((d.high == 0) && (d.low < 0x10000))
      {
       lx16_t qx, rx;
       ix16_t pr;

       rx.d = n;

       qx.s.d3 = rx.s.d3 / d.low;
       rx.s.d3 %= d.low;

       pr.s.d1 = rx.s.d3;
       pr.s.d0 = rx.s.d2;

       qx.s.d2 = pr.d / d.low;
       rx.s.d2 = pr.d % d.low;

       pr.s.d1 = rx.s.d2;
       pr.s.d0 = rx.s.d1;

       qx.s.d1 = pr.d / d.low;
       rx.s.d1 = pr.d % d.low;

       pr.s.d1 = rx.s.d1;
       pr.s.d0 = rx.s.d0;

       qx.s.d0 = pr.d / d.low;

       return (qx.d);
      }

    else
      {
       /* This is a messy division, just shift and subtract in binary. */
       u_signed64 q, pq, r, dl, t, dk, overflow;
       register int w;

       r = n;
       dk.high = dl.high = d.high;
       dk.low = dl.low = d.low;

       pq.high = 0;
       pq.low = 1;

       for ( w = 32; w > 0; w >>=1 )
         {
          shr64_3m ( overflow, dl, 64 - w);
          shl64_3m ( t, dl, w );
          if ( (eqz64m(overflow)) && (le64m(t,r)) )
            {
             shl64_ipm (pq, w);
             dl = t;
            }
         }

       dec64m(r, dl);
       q = pq;

       while ( ge64m(r,dk) )
         {
          do
            {
             shr64_ipm (pq, 1);
             shr64_ipm (dl, 1);
            }
          while ( gt64m(dl,r) );

          dec64m (r, dl);
          q.high |= pq.high;
          q.low |= pq.low;
         }

       return (q);
      }
   }


 /*
  * div64: Divide an unsigned 64 bit integer by an unsigned 32 bit integer.
  */
 u_signed64 div64 (u_signed64 n, unsigned32 d)
   {
    u_signed64 d64, q;

    d64 = cast64 ( d );
    q = div2x64 ( n, d64 );
    return ( q );
   }

 /*
  * div_cl2x64: Divide an unsigned 64 bit integer by an unsigned 64 bit
  * 	     integer and
  *           return the integer ceiling of the result.
  */
 u_signed64 div_cl2x64 (u_signed64 n, u_signed64 d)
   {
    if ( eqz64m ( n ) )
      {
       u_signed64 q;
       q.high = q.low = 0;
       return ( q );
      }

    else if ( lt64m ( n, d ) )
      {
       u_signed64 q;
       q.high = 0;
       q.low = 1;
       return ( q );
      }

    else if ((n.high == 0) && (d.high == 0))
      {
       unsigned32 q;
       q = n.low / d.low;
       if ( (n.low % d.low) > 0 )
          q++;
       n.low = q;
       return (n);
      }

    else if ((d.high == 0) && (d.low < 0x10000))
      {
       lx16_t qx, rx;
       ix16_t pr;

       rx.d = n;

       qx.s.d3 = rx.s.d3 / d.low;
       rx.s.d3 %= d.low;

       pr.s.d1 = rx.s.d3;
       pr.s.d0 = rx.s.d2;

       qx.s.d2 = pr.d / d.low;
       rx.s.d2 = pr.d % d.low;

       pr.s.d1 = rx.s.d2;
       pr.s.d0 = rx.s.d1;

       qx.s.d1 = pr.d / d.low;
       rx.s.d1 = pr.d % d.low;

       pr.s.d1 = rx.s.d1;
       pr.s.d0 = rx.s.d0;

       qx.s.d0 = pr.d / d.low;
       rx.s.d0 = pr.d % d.low;

       if (rx.s.d0 != 0)
         {
          qx.d.low++;
          if (qx.d.low == 0)
             qx.d.high++;
         }
       return (qx.d);
      }

    else
      {
       /* This is a messy division, just shift and subtract in binary. */
       u_signed64 q, pq, r, dl, t, dk, overflow;
       register int w;

       r = n;
       dk.high = dl.high = d.high;
       dk.low = dl.low = d.low;

       pq.high = 0;
       pq.low = 1;

       for ( w = 32; w > 0; w >>=1 )
         {
          overflow = shr64m (dl, 64 - w);
          shl64_3m ( t, dl, w );
          if ( (eqz64m(overflow)) && (le64m(t,r)) )
            {
             shl64_ipm (pq, w);
             dl = t;
            }
         }

       dec64m(r, dl);
       q = pq;

       while ( ge64m(r,dk) )
         {
          do
            {
             shr64_ipm (pq, 1);
             shr64_ipm (dl, 1);
            }
          while ( gt64m(dl,r) );

          dec64m (r, dl);
          q.high |= pq.high;
          q.low |= pq.low;
         }

       if (r.low > 0)
         {
          q.low++;
          if (q.low == 0)
             q.high++;
         }
       return (q);
      }
   }

 /*
  * div_cl64: Divide an unsigned 64 bit integer by an unsigned 32 bit integer
  * and return the integer ceiling of the result.
  */
 u_signed64 div_cl64 (u_signed64 n, unsigned32 d)
   {
    u_signed64 d64, q;

    d64 = cast64 ( d );
    q = div_cl2x64 ( n, d64 );
    return ( q );
   }

 /*
  * mod2x64: Modulus an unsigned 64 bit integer by an unsigned 32 bit integer.
  */
 u_signed64 mod2x64 (u_signed64 n, u_signed64 d)
   {
    if ( lt64m ( n, d ) )
      {
       return ( n );
      }

    else if ((n.high == 0) && (d.high == 0))
      {
       n.low %= d.low;
       return (n);
      }

    else if ((d.high == 0) && (d.low < 0x10000))
      {
       lx16_t rx;
       ix16_t pr;

       rx.d = n;

       rx.s.d3 %= d.low;

       pr.s.d1 = rx.s.d3;
       pr.s.d0 = rx.s.d2;

       rx.s.d3 = 0;
       rx.s.d2 = pr.d % d.low;

       pr.s.d1 = rx.s.d2;
       pr.s.d0 = rx.s.d1;

       rx.s.d2 = 0;
       rx.s.d1 = pr.d % d.low;

       pr.s.d1 = rx.s.d1;
       pr.s.d0 = rx.s.d0;

       rx.s.d1 = 0;
       rx.s.d0 = pr.d % d.low;

       return (rx.d);
      }

   else
      {
       /* This is a messy division, just shift and subtract in binary. */
       u_signed64 r, dl, t, dk, overflow;
       register int w;

       r = n;
       dk.high = dl.high = d.high;
       dk.low = dl.low = d.low;

       for ( w = 32; w > 0; w >>= 1 )
         {
          overflow = shr64m (dl, 64 - w);
          shl64_3m ( t, dl, w );
          if ( (eqz64m(overflow)) && (le64m(t,r)) )
            {
             dl = t;
            }
         }

       dec64m(r, dl);

       while ( ge64m(r,dk) )
         {
          do
             shr64_ipm (dl, 1);
          while ( gt64m(dl,r) );

          dec64m (r, dl);
         }

       return (r);
      }
   }

 /*
  * mod64: Modulus an unsigned 64 bit integer by an unsigned 32 bit integer.
  */
 u_signed64 mod64 (u_signed64 n, unsigned32 d)
   {
    u_signed64 d64, r;

    d64 = cast64 ( d );
    r = mod2x64 ( n, d64 );
    return ( r );
   }
/*
 * shl64: Shift an unsigned 64 bit integer left by an unsigned 16 bit integer am
ount.
 */
u_signed64 shl64 (u_signed64 n, unsigned32 s)
  {
   u_signed64 t;

   if (s >= 64)
     {
      t.low = t.high = 0;
      return (t);
     }

   if (s >= 32)
     {
      t.low = 0;
      t.high = n.low << (s-32);
      return (t);
     }

   t.low = n.low << s;
   t.high = (n.high << s) | (n.low >> (32-s));
   return (t);
  }

/*
 * shr64: Shift an unsigned 64 bit integer right by an unsigned 16 bit integer a
mount.
 */
u_signed64 shr64 (u_signed64 n, unsigned32 s)
  {
   u_signed64 t;

   if (s >= 64)
     {
      t.low = t.high = 0;
      return (t);
     }

   if (s >= 32)
     {
      t.high = 0;
      t.low = n.high >> (s-32);
      return (t);
     }

   t.high = n.high >> s;
   t.low = (n.low >> s) | (n.high << (32-s));
   return (t);
  }

/*
 * and64: Find the bitwise AND of two 64-bit values
 */
u_signed64 and64 (u_signed64 a1, u_signed64 a2)
  {
   u_signed64 t;

   t.high = a1.high & a2.high;
   t.low = a1.low & a2.low;
   return (t);
  }

/*
 * or64: Find the bitwise OR of two 64-bit values
 */
u_signed64 or64 (u_signed64 a1, u_signed64 a2)
  {
   u_signed64 t;

   t.high = a1.high | a2.high;
   t.low = a1.low | a2.low;
   return (t);
  }

/*
 * not64: Find the bitwise NOT of a 64-bit value
 */
u_signed64 not64 (u_signed64 a)
  {
   u_signed64 t;

   t.high = ~a.high;
   t.low = ~a.low;
   return (t);
  }

#else	/* BYTE8INT  */

/*
 * cast64: Cast a unsigned int 32 into a unsigned int 64.
 */
u_signed64 cast64 (unsigned32 n)
  {
   u_signed64	t;

   t = n;
   return t;
  }

/*
 * bld64: build a 64-bit unsigned int out of two 32-bit unsigned int's
 */
u_signed64 bld64 (unsigned32 high,unsigned32 low)
  {
   u_signed64 t;

   t = ((unsigned long) high << 32) | (low & 0xFFFFFFFF);
   return t;
  }

/*
 * mem64: builds a 64-bit unsigned int from an array of consecutive
 *        unsigned characters. The first 4 are used as the .high
 *        part, the next 4 as the .low part. 
 * Notes: 
 * - On 64-bit or big-endian machines, this is functionally identical
 *        to "*(u_signed64* pointer)". The <mem64m> macro can be used
 *        to generate this in-line.
 * - On little-endian machines, the input is assumed to be in network
 *   byte order, and is converted to host order 
 */
u_signed64 mem64(unsigned char *mem)
  {
   u_signed64 t;

   t = ((unsigned long)((mem[0] << 24) | (mem[1] << 16) |
					(mem[2] << 8) | mem[3]) << 32) |
       ((mem[4] << 24) | (mem[5] << 16) | (mem[6] << 8) | mem[7]);

   return t;
  }

/*
 * gt64: Return 1 if a1 > a2, else 0.
 */
int gt64 (u_signed64 a1, u_signed64 a2)
  {
   return (a1 > a2);
  }

/*
 * ge64: Return 1 if a1 >= a2, else 0.
 */
int ge64 (u_signed64 a1, u_signed64 a2)
  {
   return ( a1 >= a2);
  }

/*
 * eq64: Return 1 if a1 = a2, else 0.
 */
int eq64 (u_signed64 a1, u_signed64 a2)
  {
   return (a1 == a2);
  }

/*
 * neq64: Return 1 if a1 != a2, else 0.
 */
int neq64 (u_signed64 a1, u_signed64 a2)
  {
   return (a1 != a2);
  }

/*
 * le64:Return 1 if a1 <= a2, else 0.
 */
int le64 (u_signed64 a1, u_signed64 a2)
  {
   return (a1 <= a2);
  }

/*
 * lt64: Return 1 if a1 < a2, else 0.
 */
int lt64 (u_signed64 a1, u_signed64 a2)
  {
   return (a1 < a2);
  }

/*
 * add64: Add two 64 bit unsigned numbers.  Return 64 bit unsigned result.
 */
u_signed64 add64 (u_signed64 a1, u_signed64 a2)
  {
   return (a1 + a2);
  }

/*
 * sub64: Subtract two 64 bit unsigned numbers.  Return 64 bit unsigned result.
 */
u_signed64 sub64 (u_signed64 a1, u_signed64 a2)
  {
   return (a1 - a2);
  }

/*
 * mul64: Multiply a 64 bit unsigned number by a 32 bit unsigned number.
 */
u_signed64 mul64 (u_signed64 m1, unsigned32 m2)
  {
   return (m1 * m2);
  }

/*
 * div64: Divide an unsigned 64 bit integer by an unsigned 32 bit integer.
 */
u_signed64 div64 (u_signed64 n, unsigned32 d)
  {
   return(n / d);
  }

/*
 * div_cl64: Divide an unsigned 64 bit integer by an unsigned 32 bit integer and
 *           return the integer ceiling of the result.
 */
u_signed64 div_cl64 (u_signed64 n, unsigned32 d)
  {
      unsigned32 q;
      q = n / d;
      if ( (n % d) > 0 )
         q++;
      n = q;
      return (n);
  }

/*
 * mod64: Modulus an unsigned 64 bit integer by an unsigned 32 bit integer.
 */
u_signed64 mod64 (u_signed64 n, unsigned32 d)
  {
   return (n % d);
  }

/*
 * shl64: Shift an unsigned 64 bit integer left by an unsigned 32 bit 
 * integer amount.
 */
u_signed64 shl64 (u_signed64 n, unsigned32 s)
  {
   u_signed64 t;

   t = n << s;
   return (t);
  }

/*
 * shr64: Shift an unsigned 64 bit integer right by an unsigned 32 bit 
 * integer amount.
 */
u_signed64 shr64 (u_signed64 n, unsigned32 s)
  {
   u_signed64 t;

   t = n >> s;
   return (t);
  }

/*
 * and64: Find the bitwise AND of two 64-bit values
 */
u_signed64 and64 (u_signed64 a1, u_signed64 a2)
  {
   return (a1 & a2);
  }

/*
 * or64: Find the bitwise OR of two 64-bit values
 */
u_signed64 or64 (u_signed64 a1, u_signed64 a2)
  {
   return (a1 | a2);
  }

/*
 * not64: Find the bitwise NOT of a 64-bit value
 */
u_signed64 not64 (u_signed64 a)
  {
   return (~a);
  }

#endif	/* BYTE8INT */

#endif	/* _VESTA_ARITH64_C */
/*============================================================================
 *
 * Function:
 *      mover_socket_recv_data - Receive parallel data and return the number
 *				of bytes received.
 *
 * Synopsis:
 *
 *	int
 *	mover_socket_recv_data(
 *	int		SD,		** socket for data transfer
 *	u_signed64	XferID,		** transfer ID to identify the data
 *	u_signed64	Offset,		** offset into the transfer
 *	char		*Buffer,	** where to store the data
 *	int		BufSize,	** size of the buffer
 *	int		*BytesRecvd,	** return the number of bytes received(>= 0)
 *	int		NumOfPackets	** number of packets to rrecv (-== until close)
 *	)
 *
 * Description:
 *	Wait for a parallel transfer header. Determine if the data is OK. Put
 *	the data into the buffer.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	positive - The # of bytes received (may be less then BufSize because
 *						other threads may be getting
 *						the rest of the data).
 *	0 - socket connection closed
 *	-1 - error receiving the data
 *
 * Interfaces:
 *	add64m,sub64m,bld64m,low32m
 *	pdata_recv_hdr,recv
 *
 * Resources Used:
 *
 * Limitations:
 *	Only 2 gigabytes of data may be received for any one transfer.
 *	Care should be taken, because data can be received in any order.
 *
 *	This code does not yet handle receiving data that is resent.
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

int
mover_socket_recv_data(
int		SD,		/* socket for data transfer */
u_signed64	XferID,		/* transfer ID that identifies the data */
u_signed64	Offset,		/* offset of the data into the transfer */
char		*Buffer,	/* where data is to be received */
int		BufSize,	/* Size of the buffer */
int		*BytesRecvd,	/* # bytes received (0 <= BytesRecvd <= BufSize) */
int		NumOfPackets	/* # of packets to receive before return */
)
{
	return(mover_socket_recv_data_timeout(SD,
					      XferID,
					      Offset,
					      Buffer,
					      BufSize,
					      BytesRecvd,
					      NumOfPackets,
					      0));
}

int
mover_socket_recv_data_timeout(
int		SD,		/* socket for data transfer */
u_signed64	XferID,		/* transfer ID that identifies the data */
u_signed64	Offset,		/* offset of the data into the transfer */
char		*Buffer,	/* where data is to be received */
int		BufSize,	/* Size of the buffer */
int		*BytesRecvd,	/* # bytes received (0 <= BytesRecvd <= BufSize) */
int		NumOfPackets,	/* # of packets to receive before return */
int		SecTimeout	/* socket timeout in seconds */
)
{
	pdata_hdr_t	phdr;
	u_signed64		bufoffset,endbyte,lastbyte;
	register int	leng,ret;
	register char	*bufptr;
	int		nfd;
	struct timeval	timeout,*timeptr;
	fd_set		rdlist,wrlist,exlist;
	fd_set		save_rdlist,save_wrlist,save_exlist;

	/*
	 *  Make sure the length makes some sense
	 */

	*BytesRecvd = 0;
	if(BufSize < 0)
	{
	   return(-EINVAL);
	}
	/*
	 *  Determine the boundries for the transfer
	 */

	endbyte = bld64m(0,BufSize);
	endbyte = add64m(Offset,endbyte);

	/*
	 *  Now loop receiving the headers and then getting the data
	 */

	while(NumOfPackets != 0)
	{
	   /*
	    *  Wait for a parallel data transfer header
	    */

	   ret = pdata_recv_hdr_timeout(SD,&phdr,SecTimeout);
	   if(ret != sizeof(pdata_hdr_t))
	   {
	      /*
	       *  I/O error or the socket closed.
	       */

	      if (ret > 0)
		ret = -EIO;
		
	      return(ret);
	   }
	   /*
	    * Make sure transfer ID is the same
	    */

	   if(!eq64m(phdr.XferID,XferID))
	   {
	      return(-EINVAL);
	   }
	   /*
	    *  Make sure the data is in range
	    */

	   if(gt64m(Offset,phdr.Offset))
	   {
	      /*
	       *  The data comes prior to the data we asked for, out of range
	       */

	      return(-EINVAL);
	   }
	   lastbyte = add64m(phdr.Offset,phdr.Length);
	   if(gt64m(lastbyte,endbyte))
	   {
	      /*
	       *  The data goes beyond what we asked for, out of range
	       */

	      return(-EINVAL);
	   }
	   /*
	    *  Determine how many bytes are to be received.
	    */

	   leng = low32m(phdr.Length);

	   /*
	    *  Determine where in the buffer to receive the data
	    */

	   bufoffset = sub64m(phdr.Offset,Offset);
	   bufptr = Buffer + low32m(bufoffset);

	   /*
	    *  Set up receive lists.
	    */

	   FD_ZERO(&save_rdlist);
	   FD_ZERO(&save_wrlist);
	   FD_ZERO(&save_exlist);

	   FD_SET(SD,&save_rdlist);

	   /*
	    *  Now receive the data into a piece of the buffer
	    */

	   while(leng > 0)
	   {
	      if (SecTimeout == 0)
	         timeptr = (struct timeval *)NULL;
	      else
	      {
	         timeout.tv_sec = SecTimeout;
	         timeout.tv_usec = 0;
	         timeptr = &timeout;
	      }

	      rdlist = save_rdlist;
	      wrlist = save_wrlist;
	      exlist = save_exlist;

	      nfd = select(FD_SETSIZE,&rdlist,&wrlist,&exlist,timeptr);
	      if (nfd < 0)
	      {
	         if((errno == EINTR) || (errno == EAGAIN)) continue;

	         /*
	          *  If we got any other error, bail out.
	          */

	         return(-errno);
	      }

	      /*
	       *  If the select timed out, return -ETIMEDOUT.  Otherwise,
	       *  it should be safe to use the socket, since we only
	       *  asked about one.
	       */

	      if (nfd == 0)
	         return(-ETIMEDOUT);

	      ret = read(SD,bufptr,leng);
	      if(ret < 0)
	      {
		 if((errno == EAGAIN) || (errno == EINTR)) continue;

		 /*
		  *  I/O error or the sender closed, both are bad.
	          */

	         return(-errno);
	      }
	      else if (ret == 0)
	      {
	         return(ret);
	      }
	      leng -= ret;
	      bufptr += ret;
	   }
	   *BytesRecvd += low32m(phdr.Length);
	   if(NumOfPackets > 0) NumOfPackets--;
	}
	return(*BytesRecvd);
}
/*============================================================================
 *
 * Function:
 *      mover_socket_send_requested_data - Wait for a request for data. Then
 *			determine which data was requested and send it to the
 *			requestor.
 *
 * Synopsis:
 *
 *	int
 *	mover_socket_send_requested_data(
 *	int		SD,		** socket for data transfer
 *	u_signed64	XferID,		** transfer ID to identify the data
 *	u_signed64	Offset,		** offset into the transfer
 *	char		*Buffer,	** where to get the data
 *	int		BufSize,	** size of the buffer
 *	int		*BytesSent	** return the number of bytes sent(>= 0)
 *	)
 *
 * Description:
 *	Wait for a parallel transfer header. This header will be a request for
 *	some parallel data. Make sure that the request for the data is valid
 *	and in range. Send the requested data to the requestor.
 *
 * Other Inputs:
 *      None
 *
 * Outputs:
 *	positive - The # of bytes sent (may be less then BufSize because
 *					other threads may be getting
 *					the rest of the data).
 *	0 - connection was closed.
 *	-1 - error sending the data
 *
 * Interfaces:
 *	add64m,sub64m,bld64m,low32m
 *	pdata_recv_hdr,pdata_send_hdr_and_data,recv
 *
 * Resources Used:
 *
 * Limitations:
 *	Only 2 gigabytes of data may be sent for any one transfer.
 *	Care should be taken, because data may be requested in any order.
 *
 *	This code does not yet handle data request retries. The BytesSent
 *	filed will return the wrong value.
 *
 * Assumptions:
 *
 * Notes:
 *
 *-------------------------------------------------------------------------*/

int
mover_socket_send_requested_data(
int		SD,		/* socket for data transfer */
u_signed64	XferID,		/* transfer ID that identifies the data */
u_signed64	Offset,		/* offset of the data into the transfer */
char		*Buffer,	/* where data is to be sent from */
int		BufSize,	/* Size of the buffer */
int		*BytesSent,	/* # bytes sent (0 <= Bytessent <= BufSize) */
int		NumOfPackets	/* # of packets to wait for */
)
{
	return(mover_socket_send_requested_data_timeout(SD,
							XferID,
							Offset,
							Buffer,
							BufSize,
							BytesSent,
							NumOfPackets,
							0));
}

int
mover_socket_send_requested_data_timeout(
int		SD,		/* socket for data transfer */
u_signed64	XferID,		/* transfer ID that identifies the data */
u_signed64	Offset,		/* offset of the data into the transfer */
char		*Buffer,	/* where data is to be sent from */
int		BufSize,	/* Size of the buffer */
int		*BytesSent,	/* # bytes sent (0 <= Bytessent <= BufSize) */
int		NumOfPackets,	/* # of packets to wait for */
int		SecTimeout	/* socket timeout in seconds */
)
{
	pdata_hdr_t	phdr;
	u_signed64		bufoffset,endbyte,lastbyte;
	register int	leng,ret;
	register char	*bufptr;

	/*
	 *  Make sure the length makes some sense
	 */

	*BytesSent = 0;
	if(BufSize < 0)
	{
	   return(-EINVAL);
	}
	/*
	 *  Determine the boundries for the transfer
	 */

	endbyte = bld64m(0,BufSize);
	endbyte = add64m(Offset,endbyte);

	/*
	 *  Now loop receiving the the request headers and then sending the
	 *  requested data back to the requestor.
	 */

	while(NumOfPackets != 0)
	{
	   /*
	    *  Wait for a parallel data transfer header. This is a request
	    *  for data.
	    */

	   ret = pdata_recv_hdr_timeout(SD,&phdr,SecTimeout);
	   if(ret != sizeof(pdata_hdr_t))
	   {
	      /*
	       *  I/O error or the socket closed.
	       */

	      if (ret > 0)
		ret = -EIO;

	      return(ret);
	   }
	   /*
	    * Make sure transfer ID is the same
	    */

	   if(!eq64m(phdr.XferID,XferID))
	   {
	      return(-EINVAL);
	   }
	   /*
	    *  Make sure the data is in range
	    */

	   if(gt64m(Offset,phdr.Offset))
	   {
	      /*
	       *  The data comes prior to the data we asked for, out of range
	       */

	      return(-EINVAL);
	   }
	   lastbyte = add64m(phdr.Offset,phdr.Length);
	   if(gt64m(lastbyte,endbyte))
	   {
	      /*
	       *  The data goes beyond what we asked for, out of range
	       */

	      return(-EINVAL);
	   }
	   /*
	    *  Determine how many bytes are to be received.
	    */

	   leng = low32m(phdr.Length);

	   /*
	    *  Determine where in the buffer to receive the data
	    */

	   bufoffset = sub64m(phdr.Offset,Offset);
	   bufptr = Buffer + low32m(bufoffset);

	   /*
	    *  Now send the data to the requestor.
	    */

	   ret = pdata_send_hdr_and_data_timeout(SD,
						 &phdr,
						 bufptr,
						 leng,
						 SecTimeout);
	   if(ret != leng)
	   {
	      /*
	       *  The data did not get sent correctly.
	       */

	      if (ret > 0)
		ret = -EIO;

	      return(ret);
	    }
	   *BytesSent += leng;
	   if(NumOfPackets >0) NumOfPackets--;
	}
	/*
	 *  We got here because we ran out of packets to send
	 */

	return(*BytesSent);
}
