<?php

namespace {

	/**
	 * <p>This class or <code>CURLStringFile</code> should be used to upload a file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
	 * <p>Unserialization of <b>CURLFile</b> instances is not allowed. As of PHP 7.4.0, serialization is forbidden in the first place.</p>
	 * @link https://php.net/manual/en/class.curlfile.php
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	class CURLFile {

		/**
		 * @var string <p>Name of the file to be uploaded.</p>
		 * @link https://php.net/manual/en/class.curlfile.php#curlfile.props.name
		 */
		public string $name = "";

		/**
		 * @var string <p>MIME type of the file (default is <code>application/octet-stream</code>).</p>
		 * @link https://php.net/manual/en/class.curlfile.php#curlfile.props.mime
		 */
		public string $mime = "";

		/**
		 * @var string <p>The name of the file in the upload data (defaults to the name property).</p>
		 * @link https://php.net/manual/en/class.curlfile.php#curlfile.props.postname
		 */
		public string $postname = "";

		/**
		 * Create a CURLFile object
		 * <p>Object-oriented style</p><p>Creates a <code>CURLFile</code> object, used to upload a file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
		 * @param string $filename <p>Path to the file which will be uploaded.</p>
		 * @param ?string $mime_type <p>Mimetype of the file.</p>
		 * @param ?string $posted_filename <p>Name of the file to be used in the upload data.</p>
		 * @return CURLFile <p>Returns a <code>CURLFile</code> object.</p>
		 * @link https://php.net/manual/en/curlfile.construct.php
		 * @see curl_setopt()
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function __construct(string $filename, ?string $mime_type = null, ?string $posted_filename = null) {}

		/**
		 * Get file name
		 * @return string <p>Returns file name.</p>
		 * @link https://php.net/manual/en/curlfile.getfilename.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function getFilename(): string {}

		/**
		 * Get MIME type
		 * @return string <p>Returns MIME type.</p>
		 * @link https://php.net/manual/en/curlfile.getmimetype.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function getMimeType(): string {}

		/**
		 * Get file name for POST
		 * @return string <p>Returns file name for POST.</p>
		 * @link https://php.net/manual/en/curlfile.getpostfilename.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function getPostFilename(): string {}

		/**
		 * Set MIME type
		 * @param string $mime_type <p>MIME type to be used in POST data.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/curlfile.setmimetype.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function setMimeType(string $mime_type): void {}

		/**
		 * Set file name for POST
		 * @param string $posted_filename <p>Filename to be used in POST data.</p>
		 * @return void <p>No value is returned.</p>
		 * @link https://php.net/manual/en/curlfile.setpostfilename.php
		 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
		 */
		public function setPostFilename(string $posted_filename): void {}
	}

	/**
	 * <p>A fully opaque class which replaces <code>curl</code> resources as of PHP 8.0.0.</p>
	 * @link https://php.net/manual/en/class.curlhandle.php
	 * @since PHP 8
	 */
	final class CurlHandle {
	}

	/**
	 * <p>A fully opaque class which replaces <code>curl_multi</code> resources as of PHP 8.0.0.</p>
	 * @link https://php.net/manual/en/class.curlmultihandle.php
	 * @since PHP 8
	 */
	final class CurlMultiHandle {
	}

	/**
	 * <p>A fully opaque class which replaces <code>curl_share</code> resources as of PHP 8.0.0.</p>
	 * @link https://php.net/manual/en/class.curlsharehandle.php
	 * @since PHP 8
	 */
	final class CurlShareHandle {
	}

	/**
	 * <p><b>CURLStringFile</b> makes it possible to upload a file directly from a variable. This is similar to <code>CURLFile</code>, but works with the contents of the file, not filename. This class or <code>CURLFile</code> should be used to upload the contents of the file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
	 * @link https://php.net/manual/en/class.curlstringfile.php
	 * @since PHP 8 >= 8.1.0
	 */
	class CURLStringFile {

		/**
		 * @var string <p>The contents to be uploaded.</p>
		 * @link https://php.net/manual/en/class.curlstringfile.php#curlstringfile.props.data
		 */
		public string $data;

		/**
		 * @var string <p>The name of the file to be used in the upload data.</p>
		 * @link https://php.net/manual/en/class.curlstringfile.php#curlstringfile.props.postname
		 */
		public string $postname;

		/**
		 * @var string <p>MIME type of the file (default is <code>application/octet-stream</code>).</p>
		 * @link https://php.net/manual/en/class.curlstringfile.php#curlstringfile.props.mime
		 */
		public string $mime;

		/**
		 * Create a CURLStringFile object
		 * <p>Creates a <code>CURLStringFile</code> object, used to upload a file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
		 * @param string $data <p>The contents to be uploaded.</p>
		 * @param string $postname <p>The name of the file to be used in the upload data.</p>
		 * @param string $mime <p>MIME type of the file (default is <code>application/octet-stream</code>).</p>
		 * @return self
		 * @link https://php.net/manual/en/curlstringfile.construct.php
		 * @see curl_setopt()
		 * @since PHP 8 >= 8.1.0
		 */
		public function __construct(string $data, string $postname, string $mime = "application/octet-stream") {}
	}

	/**
	 * Close a cURL session
	 * <p><b>Note</b>:</p><p>This function has no effect. Prior to PHP 8.0.0, this function was used to close the resource.</p><p>Closes a cURL session and frees all resources. The cURL handle, <code>handle</code>, is also deleted.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return void <p>No value is returned.</p>
	 * @link https://php.net/manual/en/function.curl-close.php
	 * @see curl_init(), curl_multi_close()
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_close(\CurlHandle $handle): void {}

	/**
	 * Copy a cURL handle along with all of its preferences
	 * <p>Copies a cURL handle keeping the same preferences.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return CurlHandle|false <p>Returns a new cURL handle, or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-copy-handle.php
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_copy_handle(\CurlHandle $handle): \CurlHandle|false {}

	/**
	 * Return the last error number
	 * <p>Returns the error number for the last cURL operation.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return int <p>Returns the error number or <code>0</code> (zero) if no error occurred.</p>
	 * @link https://php.net/manual/en/function.curl-errno.php
	 * @see curl_error()
	 * @since PHP 4 >= 4.0.3, PHP 5, PHP 7, PHP 8
	 */
	function curl_errno(\CurlHandle $handle): int {}

	/**
	 * Return a string containing the last error for the current session
	 * <p>Returns a clear text error message for the last cURL operation.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return string <p>Returns the error message or <code>''</code> (the empty string) if no error occurred.</p>
	 * @link https://php.net/manual/en/function.curl-error.php
	 * @see curl_errno()
	 * @since PHP 4 >= 4.0.3, PHP 5, PHP 7, PHP 8
	 */
	function curl_error(\CurlHandle $handle): string {}

	/**
	 * URL encodes the given string
	 * <p>This function URL encodes the given string according to &#xBB;&#xA0;RFC 3986.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param string $string <p>The string to be encoded.</p>
	 * @return string|false <p>Returns escaped string or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-escape.php
	 * @see curl_unescape(), urlencode(), rawurlencode()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_escape(\CurlHandle $handle, string $string): string|false {}

	/**
	 * Perform a cURL session
	 * <p>Execute the given cURL session.</p><p>This function should be called after initializing a cURL session and all the options for the session are set.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return string|bool <p>On success, this function flushes the result directly to the <code>stdout</code> and returns <b><code>true</code></b>, or <b><code>false</code></b> on failure. However, if the <b><code>CURLOPT_RETURNTRANSFER</code></b> option is set, it will return the result on success, <b><code>false</code></b> on failure.</p><p><b>Warning</b></p><p>This function may return Boolean <b><code>false</code></b>, but may also return a non-Boolean value which evaluates to <b><code>false</code></b>. Please read the section on Booleans for more information. Use the === operator for testing the return value of this function.</p><p><b>Note</b>:</p><p>Note that response status codes which indicate errors (such as <code>404 Not found</code>) are not regarded as failure. <code>curl_getinfo()</code> can be used to check for these.</p>
	 * @link https://php.net/manual/en/function.curl-exec.php
	 * @see curl_multi_exec()
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_exec(\CurlHandle $handle): string|bool {}

	/**
	 * Create a CURLFile object
	 * <p>Object-oriented style</p><p>Creates a <code>CURLFile</code> object, used to upload a file with <b><code>CURLOPT_POSTFIELDS</code></b>.</p>
	 * @param string $filename <p>Path to the file which will be uploaded.</p>
	 * @param ?string $mime_type <p>Mimetype of the file.</p>
	 * @param ?string $posted_filename <p>Name of the file to be used in the upload data.</p>
	 * @return CURLFile <p>Returns a <code>CURLFile</code> object.</p>
	 * @link https://php.net/manual/en/curlfile.construct.php
	 * @see curl_setopt()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_file_create(string $filename, ?string $mime_type = null, ?string $posted_filename = null): \CURLFile {}

	/**
	 * Get information regarding a specific transfer
	 * <p>Gets information about the last transfer.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param ?int $option <p>One of the <b><code>CURLINFO_&#42;</code></b> constants.</p>
	 * @return mixed <p>If <code>option</code> is given, returns its value. Otherwise, returns an associative array with the following elements (which correspond to <code>option</code>), or <b><code>false</code></b> on failure:</p><ul> <li>  "url"  </li> <li>  "content_type"  </li> <li>  "http_code"  </li> <li>  "header_size"  </li> <li>  "request_size"  </li> <li>  "filetime"  </li> <li>  "ssl_verify_result"  </li> <li>  "redirect_count"  </li> <li>  "total_time"  </li> <li>  "namelookup_time"  </li> <li>  "connect_time"  </li> <li>  "pretransfer_time"  </li> <li>  "size_upload"  </li> <li>  "size_download"  </li> <li>  "speed_download"  </li> <li>  "speed_upload"  </li> <li>  "download_content_length"  </li> <li>  "upload_content_length"  </li> <li>  "starttransfer_time"  </li> <li>  "redirect_time"  </li> <li>  "certinfo"  </li> <li>  "primary_ip"  </li> <li>  "primary_port"  </li> <li>  "local_ip"  </li> <li>  "local_port"  </li> <li>  "redirect_url"  </li> <li>  "request_header" (This is only set if the <b><code>CURLINFO_HEADER_OUT</code></b> is set by a previous call to <code>curl_setopt()</code>)  </li> <li>  "posttransfer_time_us" (Available as of PHP 8.4.0 and cURL 8.10.0)  </li> </ul> Note that private data is not included in the associative array and must be retrieved individually with the <b><code>CURLINFO_PRIVATE</code></b> option.
	 * @link https://php.net/manual/en/function.curl-getinfo.php
	 * @since PHP 4 >= 4.0.4, PHP 5, PHP 7, PHP 8
	 */
	function curl_getinfo(\CurlHandle $handle, ?int $option = null): mixed {}

	/**
	 * Initialize a cURL session
	 * <p>Initializes a new session and returns a cURL handle.</p>
	 * @param ?string $url <p>If provided, the <b><code>CURLOPT_URL</code></b> option will be set to its value. This can be set manually using the <code>curl_setopt()</code> function.</p> <p><b>Note</b>:</p><p>The <code>file</code> protocol is disabled by cURL if open_basedir is set.</p>
	 * @return CurlHandle|false <p>Returns a cURL handle on success, <b><code>false</code></b> on errors.</p>
	 * @link https://php.net/manual/en/function.curl-init.php
	 * @see curl_multi_init()
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_init(?string $url = null): \CurlHandle|false {}

	/**
	 * Add a normal cURL handle to a cURL multi handle
	 * <p>Adds the <code>handle</code> handle to the multi handle <code>multi_handle</code></p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return int <p>Returns 0 on success, or one of the <b><code>CURLM_&#42;</code></b> errors code.</p>
	 * @link https://php.net/manual/en/function.curl-multi-add-handle.php
	 * @see curl_multi_remove_handle(), curl_multi_init(), curl_init()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_add_handle(\CurlMultiHandle $multi_handle, \CurlHandle $handle): int {}

	/**
	 * Remove all cURL handles from a multi handle
	 * <p>Removes all <code>CurlHandle</code>s attached to the <code>CurlMultiHandle</code>, as if <code>curl_multi_remove_handle()</code> was called for each of them.</p><p>Prior to PHP 8.0.0 this function also closed the cURL multi handle resource, making it unusable.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @return void <p>No value is returned.</p>
	 * @link https://php.net/manual/en/function.curl-multi-close.php
	 * @see curl_multi_init(), curl_close()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_close(\CurlMultiHandle $multi_handle): void {}

	/**
	 * Return the last multi curl error number
	 * <p>Return an integer containing the last multi curl error number.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @return int <p>Return an integer containing the last multi curl error number.</p>
	 * @link https://php.net/manual/en/function.curl-multi-errno.php
	 * @see curl_errno()
	 * @since PHP 7 >= 7.1.0, PHP 8
	 */
	function curl_multi_errno(\CurlMultiHandle $multi_handle): int {}

	/**
	 * Run the sub-connections of the current cURL handle
	 * <p>Processes each of the handles in the stack. This method can be called whether or not a handle needs to read or write data.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param int $still_running <p>A reference to a flag to tell whether the operations are still running.</p>
	 * @return int <p>A cURL code defined in the cURL Predefined Constants.</p><p><b>Note</b>:</p><p>This only returns errors regarding the whole multi stack. There might still have occurred problems on individual transfers even when this function returns <b><code>CURLM_OK</code></b>.</p>
	 * @link https://php.net/manual/en/function.curl-multi-exec.php
	 * @see curl_multi_init(), curl_multi_select(), curl_exec()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_exec(\CurlMultiHandle $multi_handle, int &$still_running): int {}

	/**
	 * Return the content of a cURL handle if CURLOPT_RETURNTRANSFER is set
	 * <p>If <b><code>CURLOPT_RETURNTRANSFER</code></b> is an option that is set for a specific handle, then this function will return the content of that cURL handle in the form of a string.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return ?string <p>Return the content of a cURL handle if <b><code>CURLOPT_RETURNTRANSFER</code></b> is set or <b><code>null</code></b> if not set.</p>
	 * @link https://php.net/manual/en/function.curl-multi-getcontent.php
	 * @see curl_multi_init()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_getcontent(\CurlHandle $handle): ?string {}

	/**
	 * Get information about the current transfers
	 * <p>Ask the multi handle if there are any messages or information from the individual transfers. Messages may include information such as an error code from the transfer or just the fact that a transfer is completed.</p><p>Repeated calls to this function will return a new result each time, until a <b><code>false</code></b> is returned as a signal that there is no more to get at this point. The integer pointed to with <code>queued_messages</code> will contain the number of remaining messages after this function was called.</p><p>The data the returned resource points to will not survive calling <code>curl_multi_remove_handle()</code>.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param int $queued_messages <p>Number of messages that are still in the queue</p>
	 * @return array|false <p>On success, returns an associative array for the message, <b><code>false</code></b> on failure.</p> <b>Contents of the returned array</b>   Key: Value:     <code>msg</code> The <b><code>CURLMSG_DONE</code></b> constant. Other return values are currently not available.   <code>result</code> One of the <b><code>CURLE_&#42;</code></b> constants. If everything is OK, the <b><code>CURLE_OK</code></b> will be the result.   <code>handle</code> Resource of type curl indicates the handle which it concerns.
	 * @link https://php.net/manual/en/function.curl-multi-info-read.php
	 * @see curl_multi_init()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_info_read(\CurlMultiHandle $multi_handle, int &$queued_messages = null): array|false {}

	/**
	 * Returns a new cURL multi handle
	 * <p>Allows the processing of multiple cURL handles asynchronously.</p>
	 * @return CurlMultiHandle <p>Returns a cURL multi handle.</p>
	 * @link https://php.net/manual/en/function.curl-multi-init.php
	 * @see curl_init(), curl_multi_close()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_init(): \CurlMultiHandle {}

	/**
	 * Remove a handle from a set of cURL handles
	 * <p>Removes a given <code>handle</code> from the given <code>multi_handle</code>. When the <code>handle</code> has been removed, it is again perfectly legal to run <code>curl_exec()</code> on this handle. Removing the <code>handle</code> while being used, will effectively halt the transfer in progress involving that handle.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return int <p>Returns 0 on success, or one of the <b><code>CURLM_&#42;</code></b> error codes.</p>
	 * @link https://php.net/manual/en/function.curl-multi-remove-handle.php
	 * @see curl_init(), curl_multi_init(), curl_multi_add_handle()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_remove_handle(\CurlMultiHandle $multi_handle, \CurlHandle $handle): int {}

	/**
	 * Wait until reading or writing is possible for any cURL multi handle connection
	 * <p>Blocks the execution of the script until a cURL handle attached to the cURL multi handle would be able to make progress on the next call to <code>curl_multi_exec()</code> or until the timeout strikes (whichever comes first).</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param float $timeout <p>Time, in seconds, to wait for a response from the active cURL multi handle connections.</p>
	 * @return int <p>On success, returns the number of active descriptors contained in the descriptor sets. This may be <code>0</code> if there was no activity on any of the descriptors. On failure, this function will return <code>-1</code> on a select failure (from the underlying <code>select()</code> system call).</p>
	 * @link https://php.net/manual/en/function.curl-multi-select.php
	 * @see curl_multi_init()
	 * @since PHP 5, PHP 7, PHP 8
	 */
	function curl_multi_select(\CurlMultiHandle $multi_handle, float $timeout = 1.0): int {}

	/**
	 * Set a cURL multi option
	 * <p>Sets an option on the given cURL multi handle.</p>
	 * @param \CurlMultiHandle $multi_handle <p>A cURL multi handle returned by <code>curl_multi_init()</code>.</p>
	 * @param int $option <p>One of the <b><code>CURLMOPT_&#42;</code></b> constants.</p>
	 * @param mixed $value <p>The value to be set on <code>option</code>. See the description of the <b><code>CURLMOPT_&#42;</code></b> constants for details on the type of values each constant expects.</p>
	 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-multi-setopt.php
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_multi_setopt(\CurlMultiHandle $multi_handle, int $option, mixed $value): bool {}

	/**
	 * Return string describing error code
	 * <p>Returns a text error message describing the given <b><code>CURLM_&#42;</code></b> error code.</p>
	 * @param int $error_code <p>One of the <b><code>CURLM_&#42;</code></b> constants.</p>
	 * @return ?string <p>Returns error string for valid error code, <b><code>null</code></b> otherwise.</p>
	 * @link https://php.net/manual/en/function.curl-multi-strerror.php
	 * @see curl_strerror()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_multi_strerror(int $error_code): ?string {}

	/**
	 * Pause and unpause a connection
	 * <p>Pause or unpause a cURL session. A session can be paused while a transfer is in progress, in either the read, write or both directions, by calling this function from a callback registered with <code>curl_setopt()</code>.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param int $flags <p>One of <b><code>CURLPAUSE_&#42;</code></b> constants.</p>
	 * @return int <p>Returns an error code (<b><code>CURLE_OK</code></b> for no error).</p>
	 * @link https://php.net/manual/en/function.curl-pause.php
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_pause(\CurlHandle $handle, int $flags): int {}

	/**
	 * Reset all options of a libcurl session handle
	 * <p>This function re-initializes all options set on the given cURL handle to the default values.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return void <p>No value is returned.</p>
	 * @link https://php.net/manual/en/function.curl-reset.php
	 * @see curl_setopt()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_reset(\CurlHandle $handle): void {}

	/**
	 * Set an option for a cURL transfer
	 * <p>Sets an option on the given cURL session handle.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param int $option <p>The <b><code>CURLOPT_&#42;</code></b> option to set.</p>
	 * @param mixed $value <p>The value to be set on <code>option</code>. See the description of the <b><code>CURLOPT_&#42;</code></b> constants for details on the type of values each constant expects.</p>
	 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-setopt.php
	 * @see curl_setopt_array()
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_setopt(\CurlHandle $handle, int $option, mixed $value): bool {}

	/**
	 * Set multiple options for a cURL transfer
	 * <p>Sets multiple options for a cURL session. This function is useful for setting a large number of cURL options without repetitively calling <code>curl_setopt()</code>.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param array $options <p>An <code>array</code> specifying which options to set and their values. The keys should be valid <code>curl_setopt()</code> constants or their integer equivalents.</p>
	 * @return bool <p>Returns <b><code>true</code></b> if all options were successfully set. If an option could not be successfully set, <b><code>false</code></b> is immediately returned, ignoring any future options in the <code>options</code> array.</p>
	 * @link https://php.net/manual/en/function.curl-setopt-array.php
	 * @see curl_setopt()
	 * @since PHP 5 >= 5.1.3, PHP 7, PHP 8
	 */
	function curl_setopt_array(\CurlHandle $handle, array $options): bool {}

	/**
	 * Close a cURL share handle
	 * <p><b>Note</b>:</p><p>This function has no effect. Prior to PHP 8.0.0, this function was used to close the resource.</p><p>Closes a cURL share handle and frees all resources.</p>
	 * @param \CurlShareHandle $share_handle <p>A cURL share handle returned by <code>curl_share_init()</code>.</p>
	 * @return void <p>No value is returned.</p>
	 * @link https://php.net/manual/en/function.curl-share-close.php
	 * @see curl_share_init()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_share_close(\CurlShareHandle $share_handle): void {}

	/**
	 * Return the last share curl error number
	 * <p>Return an integer containing the last share curl error number.</p>
	 * @param \CurlShareHandle $share_handle <p>A cURL share handle returned by <code>curl_share_init()</code>.</p>
	 * @return int <p>Returns an integer containing the last share curl error number.</p>
	 * @link https://php.net/manual/en/function.curl-share-errno.php
	 * @see curl_errno()
	 * @since PHP 7 >= 7.1.0, PHP 8
	 */
	function curl_share_errno(\CurlShareHandle $share_handle): int {}

	/**
	 * Initialize a cURL share handle
	 * <p>Allows to share data between cURL handles.</p>
	 * @return CurlShareHandle <p>Returns a cURL share handle.</p>
	 * @link https://php.net/manual/en/function.curl-share-init.php
	 * @see curl_share_setopt(), curl_share_close()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_share_init(): \CurlShareHandle {}

	/**
	 * Set an option for a cURL share handle
	 * <p>Sets an option on the given cURL share handle.</p>
	 * @param \CurlShareHandle $share_handle <p>A cURL share handle returned by <code>curl_share_init()</code>.</p>
	 * @param int $option <p>One of the <b><code>CURLSHOPT_&#42;</code></b> constants.</p>
	 * @param mixed $value <p>One of the <b><code>CURL_LOCK_DATA_&#42;</code></b> constants.</p>
	 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-share-setopt.php
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_share_setopt(\CurlShareHandle $share_handle, int $option, mixed $value): bool {}

	/**
	 * Return string describing the given error code
	 * <p>Returns a text error message describing the given error code.</p>
	 * @param int $error_code <p>One of the &#xBB;&#xA0;cURL error codes constants.</p>
	 * @return ?string <p>Returns error description or <b><code>null</code></b> for invalid error code.</p>
	 * @link https://php.net/manual/en/function.curl-share-strerror.php
	 * @see curl_share_errno(), curl_strerror()
	 * @since PHP 7 >= 7.1.0, PHP 8
	 */
	function curl_share_strerror(int $error_code): ?string {}

	/**
	 * Return string describing the given error code
	 * <p>Returns a text error message describing the given error code.</p>
	 * @param int $error_code <p>One of the &#xBB;&#xA0;cURL error codes constants.</p>
	 * @return ?string <p>Returns error description or <b><code>null</code></b> for invalid error code.</p>
	 * @link https://php.net/manual/en/function.curl-strerror.php
	 * @see curl_errno(), curl_error()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_strerror(int $error_code): ?string {}

	/**
	 * Decodes the given URL encoded string
	 * <p>This function decodes the given URL encoded string.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @param string $string <p>The URL encoded string to be decoded.</p>
	 * @return string|false <p>Returns decoded string or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl-unescape.php
	 * @see curl_escape(), urlencode(), urldecode(), rawurlencode(), rawurldecode()
	 * @since PHP 5 >= 5.5.0, PHP 7, PHP 8
	 */
	function curl_unescape(\CurlHandle $handle, string $string): string|false {}

	/**
	 * Performs any connection upkeep checks
	 * <p>Available if built against libcurl &gt;= 7.62.0.</p><p>Some protocols have "connection upkeep" mechanisms. These mechanisms usually send some traffic on existing connections in order to keep them alive; this can prevent connections from being closed due to overzealous firewalls, for example.</p><p>Connection upkeep is currently available only for HTTP/2 connections. A small amount of traffic is usually sent to keep a connection alive. HTTP/2 maintains its connection by sending a HTTP/2 PING frame.</p>
	 * @param \CurlHandle $handle <p>A cURL handle returned by <code>curl_init()</code>.</p>
	 * @return bool <p>Returns <b><code>true</code></b> on success or <b><code>false</code></b> on failure.</p>
	 * @link https://php.net/manual/en/function.curl_upkeep.php
	 * @see curl_init()
	 * @since PHP 8 >= 8.2.0
	 */
	function curl_upkeep(\CurlHandle $handle): bool {}

	/**
	 * Gets cURL version information
	 * <p>Returns information about the cURL version.</p>
	 * @return array|false <p>Returns an associative array with the following elements:</p>   Key Value description     version_number cURL 24 bit version number   version cURL version number, as a string   ssl_version_number OpenSSL 24 bit version number   ssl_version OpenSSL version number, as a string   libz_version zlib version number, as a string   host Information about the host where cURL was built   age &#xA0;   features A bitmask of the <b><code>CURL_VERSION_&#42;</code></b> constants   protocols An array of protocols names supported by cURL   feature_list  An associative array of all known cURL features, and whether they are supported (<b><code>true</code></b>) or not (<b><code>false</code></b>)
	 * @link https://php.net/manual/en/function.curl-version.php
	 * @since PHP 4 >= 4.0.2, PHP 5, PHP 7, PHP 8
	 */
	function curl_version(): array|false {}

	/**
	 * Returned by the wildcard match callback function if an error occurred. Available as of cURL 7.21.0.
	 */
	define('CURL_FNMATCHFUNC_FAIL', 2);

	/**
	 * Returned by the wildcard match callback function if pattern matches the string. Available as of cURL 7.21.0.
	 */
	define('CURL_FNMATCHFUNC_MATCH', 0);

	/**
	 * Returned by the wildcard match callback function if pattern does not match the string. Available as of cURL 7.21.0.
	 */
	define('CURL_FNMATCHFUNC_NOMATCH', 1);

	define('CURL_HTTP_VERSION_1_0', 1);

	define('CURL_HTTP_VERSION_1_1', 2);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.43.0
	 */
	define('CURL_HTTP_VERSION_2', 3);

	/**
	 * Available as of cURL 7.33.0
	 */
	define('CURL_HTTP_VERSION_2_0', 3);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.49.0
	 */
	define('CURL_HTTP_VERSION_2_PRIOR_KNOWLEDGE', 5);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.47.0
	 */
	define('CURL_HTTP_VERSION_2TLS', 4);

	/**
	 * Available as of PHP 8.4.0 and cURL 7.66.0.
	 */
	define('CURL_HTTP_VERSION_3', 30);

	/**
	 * Available as of PHP 8.4.0 and cURL 7.88.0.
	 */
	define('CURL_HTTP_VERSION_3ONLY', null);

	define('CURL_HTTP_VERSION_NONE', 0);

	/**
	 * Use only IPv4 addresses when establishing a connection or choosing one from the connection pool. Available as of cURL 7.10.8.
	 */
	define('CURL_IPRESOLVE_V4', 1);

	/**
	 * Use only IPv6 addresses when establishing a connection or choosing one from the connection pool. Available as of cURL 7.10.8.
	 */
	define('CURL_IPRESOLVE_V6', 2);

	/**
	 * Use addresses of all IP versions allowed by the system. Available as of cURL 7.10.8.
	 */
	define('CURL_IPRESOLVE_WHATEVER', 0);

	/**
	 * Shares/unshares the connection cache. Available as of PHP 7.3.0 and cURL 7.10.3.
	 */
	define('CURL_LOCK_DATA_CONNECT', 5);

	/**
	 * Shares/unshares cookie data. Available as of cURL 7.10.3.
	 */
	define('CURL_LOCK_DATA_COOKIE', 2);

	/**
	 * Shares/unshares DNS cache. Note that when you use cURL multi handles, all handles added to the same multi handle will share DNS cache by default. Available as of cURL 7.10.3.
	 */
	define('CURL_LOCK_DATA_DNS', 3);

	/**
	 * Shares/unshares the Public Suffix List. Available as of PHP 7.3.0 and cURL 7.61.0.
	 */
	define('CURL_LOCK_DATA_PSL', 6);

	/**
	 * Shares/unshares SSL session IDs, reducing the time spent on the SSL handshake when reconnecting to the same server. Note that SSL session IDs are reused within the same handle by default. Available as of cURL 7.10.3.
	 */
	define('CURL_LOCK_DATA_SSL_SESSION', 4);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.53.0
	 */
	define('CURL_MAX_READ_SIZE', 524288);

	define('CURL_NETRC_IGNORED', 0);

	define('CURL_NETRC_OPTIONAL', 1);

	define('CURL_NETRC_REQUIRED', 2);

	/**
	 * Available as of PHP 8.4.0 and cURL 7.80.0.
	 */
	define('CURL_PREREQFUNC_ABORT', 1);

	/**
	 * Available as of PHP 8.4.0 and cURL 7.80.0.
	 */
	define('CURL_PREREQFUNC_OK', 0);

	/**
	 * Available as of PHP 7.1.0 and cURL 7.44.0
	 */
	define('CURL_PUSH_DENY', 1);

	/**
	 * Available as of PHP 7.1.0 and cURL 7.44.0
	 */
	define('CURL_PUSH_OK', 0);

	/**
	 * Available as of cURL 7.18.0.
	 */
	define('CURL_READFUNC_PAUSE', 268435457);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.18.2
	 */
	define('CURL_REDIR_POST_301', 1);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.18.2
	 */
	define('CURL_REDIR_POST_302', 2);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.25.1
	 */
	define('CURL_REDIR_POST_303', 4);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.18.2
	 */
	define('CURL_REDIR_POST_ALL', 7);

	/**
	 * When sent by a client, this method changes the description of the session. <code>ANNOUNCE</code> acts like an HTTP PUT or POST just like <b><code>CURL_RTSPREQ_SET_PARAMETER</code></b>. Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_ANNOUNCE', 3);

	/**
	 * Used to get the low level description of a stream. The application should note what formats it understands in the <code>Accept:</code> header. Unless set manually, libcurl automatically adds in <code>Accept: application/sdp</code>. Time-condition headers are added to DESCRIBE requests if the <b><code>CURLOPT_TIMECONDITION</code></b> option is used. Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_DESCRIBE', 2);

	/**
	 * Retrieve a parameter from the server. By default, libcurl adds a <code>Content-Type: text/parameters</code> header on all non-empty requests unless a custom one is set. <code>GET_PARAMETER</code> acts just like an HTTP PUT or POST. Applications wishing to send a heartbeat message should use an empty <code>GET_PARAMETER</code> request. Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_GET_PARAMETER', 8);

	/**
	 * Used to retrieve the available methods of the server. Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_OPTIONS', 1);

	/**
	 * Send a <code>PAUSE</code> command to the server. Use the <b><code>CURLOPT_RANGE</code></b> option with a single value to indicate when the stream should be halted (e.g. npt=25). Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_PAUSE', 6);

	/**
	 * Send a <code>PLAY</code> command to the server. Use the <b><code>CURLOPT_RANGE</code></b> option to modify the playback time (e.g. npt=10-15). Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_PLAY', 5);

	/**
	 * Set the RTSP request type to this value to receive interleaved RTP data. Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_RECEIVE', 11);

	/**
	 * Used to tell the server to record a session. Use the <b><code>CURLOPT_RANGE</code></b> option to modify the record time. Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_RECORD', 10);

	/**
	 * Set a parameter on the server. Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_SET_PARAMETER', 9);

	/**
	 * Used to initialize the transport layer for the session. Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_SETUP', 4);

	/**
	 * Terminates an RTSP session. Simply closing a connection does not terminate the RTSP session since it is valid to control an RTSP session over different connections. Available as of cURL 7.20.0.
	 */
	define('CURL_RTSPREQ_TEARDOWN', 7);

	define('CURL_SSLVERSION_DEFAULT', 0);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_DEFAULT', 65536);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_NONE', 0);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_TLSv1_0', 262144);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_TLSv1_1', 327680);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_TLSv1_2', 393216);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.54.0
	 */
	define('CURL_SSLVERSION_MAX_TLSv1_3', 458752);

	define('CURL_SSLVERSION_SSLv2', 2);

	define('CURL_SSLVERSION_SSLv3', 3);

	define('CURL_SSLVERSION_TLSv1', 1);

	define('CURL_SSLVERSION_TLSv1_0', 4);

	define('CURL_SSLVERSION_TLSv1_1', 5);

	define('CURL_SSLVERSION_TLSv1_2', 6);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURL_SSLVERSION_TLSv1_3', 7);

	define('CURL_TIMECOND_IFMODSINCE', 1);

	define('CURL_TIMECOND_IFUNMODSINCE', 2);

	define('CURL_TIMECOND_LASTMOD', 3);

	define('CURL_TIMECOND_NONE', 0);

	/**
	 * Available as of cURL 7.21.4.
	 */
	define('CURL_TLSAUTH_SRP', 1);

	/**
	 * Available as of PHP 7.3.6 and cURL 7.64.1
	 */
	define('CURL_VERSION_ALTSVC', 16777216);

	/**
	 * Asynchronous DNS resolves. Available as of PHP 7.3.0 and cURL 7.10.7
	 */
	define('CURL_VERSION_ASYNCHDNS', 128);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.57.0
	 */
	define('CURL_VERSION_BROTLI', 8388608);

	/**
	 * Character conversions supported. Available as of PHP 7.3.0 and cURL 7.15.4
	 */
	define('CURL_VERSION_CONV', 4096);

	/**
	 * Debug memory tracking supported. Available as of PHP 7.3.6 and cURL 7.19.6
	 */
	define('CURL_VERSION_CURLDEBUG', 8192);

	/**
	 * Built with debug capabilities. Available as of PHP 7.3.0 and cURL 7.10.6
	 */
	define('CURL_VERSION_DEBUG', 64);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.76.0
	 */
	define('CURL_VERSION_GSASL', 536870912);

	/**
	 * Built against a GSS-API library. Available as of PHP 7.3.0 and cURL 7.38.0
	 */
	define('CURL_VERSION_GSSAPI', 131072);

	/**
	 * Negotiate auth is supported. Available as of PHP 7.3.0 and cURL 7.10.6 (deprecated as of cURL 7.38.0)
	 */
	define('CURL_VERSION_GSSNEGOTIATE', 32);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.74.0
	 */
	define('CURL_VERSION_HSTS', 268435456);

	/**
	 * HTTP2 support built-in. Available as of cURL 7.33.0
	 */
	define('CURL_VERSION_HTTP2', 65536);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.66.0
	 */
	define('CURL_VERSION_HTTP3', 33554432);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURL_VERSION_HTTPS_PROXY', 2097152);

	/**
	 * Internationalized Domain Names are supported. Available as of PHP 7.3.0 and cURL 7.12.0
	 */
	define('CURL_VERSION_IDN', 1024);

	/**
	 * IPv6-enabled.
	 */
	define('CURL_VERSION_IPV6', 1);

	/**
	 * Kerberos V4 auth is supported.
	 */
	define('CURL_VERSION_KERBEROS4', 2);

	/**
	 * Kerberos V5 auth is supported. Available as of PHP 7.0.7 and cURL 7.40.0
	 */
	define('CURL_VERSION_KERBEROS5', 262144);

	/**
	 * Supports files larger than 2GB. Available as of cURL 7.33.0
	 */
	define('CURL_VERSION_LARGEFILE', 512);

	/**
	 * libz features are present.
	 */
	define('CURL_VERSION_LIBZ', 8);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.56.0
	 */
	define('CURL_VERSION_MULTI_SSL', 4194304);

	/**
	 * NTLM auth is supported. Available as of PHP 7.3.0 and cURL 7.10.6
	 */
	define('CURL_VERSION_NTLM', 16);

	/**
	 * NTLM delegation to winbind helper is supported. Available as of PHP 7.3.0 and cURL 7.22.0
	 */
	define('CURL_VERSION_NTLM_WB', 32768);

	/**
	 * Mozilla's Public Suffix List, used for cookie domain verification. Available as of PHP 7.3.6 and cURL 7.47.0
	 */
	define('CURL_VERSION_PSL', 1048576);

	/**
	 * SPNEGO auth is supported. Available as of PHP 7.3.0 and cURL 7.10.8
	 */
	define('CURL_VERSION_SPNEGO', 256);

	/**
	 * SSL options are present.
	 */
	define('CURL_VERSION_SSL', 4);

	/**
	 * Built against Windows SSPI. Available as of PHP 7.3.0 and cURL 7.13.2
	 */
	define('CURL_VERSION_SSPI', 2048);

	/**
	 * TLS-SRP auth is supported. Available as of PHP 7.3.0 and cURL 7.21.4
	 */
	define('CURL_VERSION_TLSAUTH_SRP', 16384);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.72.0
	 */
	define('CURL_VERSION_UNICODE', 134217728);

	/**
	 * Unix domain sockets support. Available as of PHP 7.0.7 and cURL 7.40.0
	 */
	define('CURL_VERSION_UNIX_SOCKETS', 524288);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.72.0
	 */
	define('CURL_VERSION_ZSTD', 67108864);

	/**
	 * Available as of cURL 7.18.0.
	 */
	define('CURL_WRITEFUNC_PAUSE', 268435457);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLALTSVC_H1', 8);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLALTSVC_H2', 16);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLALTSVC_H3', 32);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.64.1.
	 */
	define('CURLALTSVC_READONLYFILE', 4);

	define('CURLAUTH_ANY', -17);

	define('CURLAUTH_ANYSAFE', -18);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.75.0.
	 */
	define('CURLAUTH_AWS_SIGV4', 128);

	define('CURLAUTH_BASIC', 1);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.61.0.
	 */
	define('CURLAUTH_BEARER', 64);

	define('CURLAUTH_DIGEST', 2);

	/**
	 * Use HTTP Digest authentication with an IE flavor. Available as of cURL 7.19.3.
	 */
	define('CURLAUTH_DIGEST_IE', 16);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.54.1
	 */
	define('CURLAUTH_GSSAPI', 4);

	define('CURLAUTH_GSSNEGOTIATE', 4);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.38.0.
	 */
	define('CURLAUTH_NEGOTIATE', 4);

	/**
	 * Available as of cURL 7.10.6.
	 */
	define('CURLAUTH_NONE', 0);

	define('CURLAUTH_NTLM', 8);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.22.0
	 */
	define('CURLAUTH_NTLM_WB', 32);

	/**
	 * This is a meta symbol. OR this value together with a single specific auth value to force libcurl to probe for unrestricted auth and if not, only that single auth algorithm is acceptable. Available as of cURL 7.21.3.
	 */
	define('CURLAUTH_ONLY', 2147483648);

	/**
	 * Aborted by callback. A callback returned "abort" to libcurl.
	 */
	define('CURLE_ABORTED_BY_CALLBACK', 42);

	define('CURLE_BAD_CALLING_ORDER', 44);

	/**
	 * Unrecognized transfer encoding.
	 */
	define('CURLE_BAD_CONTENT_ENCODING', 61);

	/**
	 * The download could not be resumed because the specified offset was out of the file boundary.
	 */
	define('CURLE_BAD_DOWNLOAD_RESUME', 36);

	/**
	 * A function was called with a bad parameter.
	 */
	define('CURLE_BAD_FUNCTION_ARGUMENT', 43);

	define('CURLE_BAD_PASSWORD_ENTERED', 46);

	/**
	 * Failed to connect to host or proxy.
	 */
	define('CURLE_COULDNT_CONNECT', 7);

	/**
	 * Could not resolve host. The given remote host was not resolved.
	 */
	define('CURLE_COULDNT_RESOLVE_HOST', 6);

	/**
	 * Could not resolve proxy. The given proxy host could not be resolved.
	 */
	define('CURLE_COULDNT_RESOLVE_PROXY', 5);

	/**
	 * Early initialization code failed. This is likely to be an internal error or problem, or a resource problem where something fundamental could not get done at init time.
	 */
	define('CURLE_FAILED_INIT', 2);

	/**
	 * A file given with FILE:// could not be opened. Most likely because the file path does not identify an existing file or due to the lack of appropriate file permissions.
	 */
	define('CURLE_FILE_COULDNT_READ_FILE', 37);

	/**
	 * Maximum file size exceeded.
	 */
	define('CURLE_FILESIZE_EXCEEDED', 63);

	define('CURLE_FTP_ACCESS_DENIED', 9);

	define('CURLE_FTP_BAD_DOWNLOAD_RESUME', 36);

	/**
	 * An internal failure to lookup the host used for the new connection.
	 */
	define('CURLE_FTP_CANT_GET_HOST', 15);

	define('CURLE_FTP_CANT_RECONNECT', 16);

	define('CURLE_FTP_COULDNT_GET_SIZE', 32);

	/**
	 * This was either a unexpected reply to a 'RETR' command or a zero byte transfer complete.
	 */
	define('CURLE_FTP_COULDNT_RETR_FILE', 19);

	define('CURLE_FTP_COULDNT_SET_ASCII', 29);

	define('CURLE_FTP_COULDNT_SET_BINARY', 17);

	define('CURLE_FTP_COULDNT_STOR_FILE', 25);

	/**
	 * The FTP REST command returned error. This should never happen if the server is sane.
	 */
	define('CURLE_FTP_COULDNT_USE_REST', 31);

	define('CURLE_FTP_PARTIAL_FILE', 18);

	/**
	 * The FTP PORT command returned error. This mostly happens when a good enough address has not been specified for libcurl to use. See <b><code>CURLOPT_FTPPORT</code></b>.
	 */
	define('CURLE_FTP_PORT_FAILED', 30);

	define('CURLE_FTP_QUOTE_ERROR', 21);

	define('CURLE_FTP_SSL_FAILED', 64);

	define('CURLE_FTP_USER_PASSWORD_INCORRECT', 10);

	/**
	 * FTP servers return a 227-line as a response to a PASV command. If libcurl fails to parse that line, this return code is passed back.
	 */
	define('CURLE_FTP_WEIRD_227_FORMAT', 14);

	/**
	 * After having sent the FTP password to the server, libcurl expects a proper reply. This error code indicates that an unexpected code was returned.
	 */
	define('CURLE_FTP_WEIRD_PASS_REPLY', 11);

	/**
	 * libcurl failed to get a sensible result back from the server as a response to either a PASV or a EPSV command. The server is flawed.
	 */
	define('CURLE_FTP_WEIRD_PASV_REPLY', 13);

	/**
	 * The server sent data libcurl could not parse. This error code is known as <b><code>CURLE_WEIRD_SERVER_REPLY</code></b> as of cURL 7.51.0.
	 */
	define('CURLE_FTP_WEIRD_SERVER_REPLY', 8);

	define('CURLE_FTP_WEIRD_USER_REPLY', 12);

	define('CURLE_FTP_WRITE_ERROR', 20);

	/**
	 * Function not found. A required zlib function was not found.
	 */
	define('CURLE_FUNCTION_NOT_FOUND', 41);

	/**
	 * Nothing was returned from the server, and under the circumstances, getting nothing is considered an error.
	 */
	define('CURLE_GOT_NOTHING', 52);

	define('CURLE_HTTP_NOT_FOUND', 22);

	define('CURLE_HTTP_PORT_FAILED', 45);

	/**
	 * This is an odd error that mainly occurs due to internal confusion.
	 */
	define('CURLE_HTTP_POST_ERROR', 34);

	define('CURLE_HTTP_RANGE_ERROR', 33);

	/**
	 * This is returned if <b><code>CURLOPT_FAILONERROR</code></b> is set <b><code>true</code></b> and the HTTP server returns an error code that is greater than or equal to 400.
	 */
	define('CURLE_HTTP_RETURNED_ERROR', 22);

	/**
	 * LDAP cannot bind. LDAP bind operation failed.
	 */
	define('CURLE_LDAP_CANNOT_BIND', 38);

	define('CURLE_LDAP_INVALID_URL', 62);

	/**
	 * LDAP search failed.
	 */
	define('CURLE_LDAP_SEARCH_FAILED', 39);

	define('CURLE_LIBRARY_NOT_FOUND', 40);

	define('CURLE_MALFORMAT_USER', 24);

	define('CURLE_OBSOLETE', 50);

	/**
	 * All fine. Proceed as usual.
	 */
	define('CURLE_OK', 0);

	/**
	 * Operation timeout. The specified time-out period was reached according to the conditions.
	 */
	define('CURLE_OPERATION_TIMEDOUT', 28);

	define('CURLE_OPERATION_TIMEOUTED', 28);

	/**
	 * A memory allocation request failed.
	 */
	define('CURLE_OUT_OF_MEMORY', 27);

	/**
	 * A file transfer was shorter or larger than expected. This happens when the server first reports an expected transfer size, and then delivers data that does not match the previously given size.
	 */
	define('CURLE_PARTIAL_FILE', 18);

	/**
	 * Proxy handshake error. <b><code>CURLINFO_PROXY_ERROR</code></b> provides extra details on the specific problem. Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLE_PROXY', 97);

	/**
	 * There was a problem reading a local file or an error returned by the read callback.
	 */
	define('CURLE_READ_ERROR', 26);

	/**
	 * Failure with receiving network data.
	 */
	define('CURLE_RECV_ERROR', 56);

	/**
	 * Failed sending network data.
	 */
	define('CURLE_SEND_ERROR', 55);

	define('CURLE_SHARE_IN_USE', 57);

	/**
	 * An unspecified error occurred during the SSH session. Available as of cURL 7.16.1.
	 */
	define('CURLE_SSH', 79);

	define('CURLE_SSL_CACERT', 60);

	/**
	 * Problem with reading the SSL CA cert.
	 */
	define('CURLE_SSL_CACERT_BADFILE', 77);

	/**
	 * Problem with the local client certificate.
	 */
	define('CURLE_SSL_CERTPROBLEM', 58);

	/**
	 * Could not use specified cipher.
	 */
	define('CURLE_SSL_CIPHER', 59);

	/**
	 * A problem occurred somewhere in the SSL/TLS handshake. Reading the message in the error buffer provides more details on the problem. Could be certificates (file formats, paths, permissions), passwords, and others.
	 */
	define('CURLE_SSL_CONNECT_ERROR', 35);

	/**
	 * The specified crypto engine was not found.
	 */
	define('CURLE_SSL_ENGINE_NOTFOUND', 53);

	/**
	 * Failed setting the selected SSL crypto engine as default.
	 */
	define('CURLE_SSL_ENGINE_SETFAILED', 54);

	define('CURLE_SSL_PEER_CERTIFICATE', 60);

	/**
	 * Failed to match the pinned key specified with <b><code>CURLOPT_PINNEDPUBLICKEY</code></b>.
	 */
	define('CURLE_SSL_PINNEDPUBKEYNOTMATCH', 90);

	define('CURLE_TELNET_OPTION_SYNTAX', 49);

	/**
	 * Too many redirects. When following redirects, libcurl hit the maximum amount. The limit can be set with <b><code>CURLOPT_MAXREDIRS</code></b>.
	 */
	define('CURLE_TOO_MANY_REDIRECTS', 47);

	define('CURLE_UNKNOWN_TELNET_OPTION', 48);

	/**
	 * The URL passed to libcurl used a protocol that libcurl does not support. The issue might be a compile-time option that was not used, a misspelled protocol string or just a protocol libcurl has no code for.
	 */
	define('CURLE_UNSUPPORTED_PROTOCOL', 1);

	/**
	 * The URL was not properly formatted.
	 */
	define('CURLE_URL_MALFORMAT', 3);

	define('CURLE_URL_MALFORMAT_USER', 4);

	/**
	 * The server sent data libcurl could not parse. This error code was known as <b><code>CURLE_FTP_WEIRD_SERVER_REPLY</code></b> before cURL 7.51.0. Available as of PHP 7.3.0 and cURL 7.51.0
	 */
	define('CURLE_WEIRD_SERVER_REPLY', 8);

	/**
	 * An error occurred when writing received data to a local file, or an error was returned to libcurl from a write callback.
	 */
	define('CURLE_WRITE_ERROR', 23);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.19.3
	 */
	define('CURLFTP_CREATE_DIR', 1);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.19.3
	 */
	define('CURLFTP_CREATE_DIR_NONE', 0);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.19.3
	 */
	define('CURLFTP_CREATE_DIR_RETRY', 2);

	define('CURLFTPAUTH_DEFAULT', 0);

	define('CURLFTPAUTH_SSL', 1);

	define('CURLFTPAUTH_TLS', 2);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.15.3.
	 */
	define('CURLFTPMETHOD_DEFAULT', 0);

	/**
	 * Do a single <code>CWD</code> operation for each path part in the given URL. Available as of cURL 7.15.3.
	 */
	define('CURLFTPMETHOD_MULTICWD', 1);

	/**
	 * libcurl makes no <code>CWD</code> at all. libcurl does <code>SIZE</code>, <code>RETR</code>, <code>STOR</code> etc. and gives a full path to the server for all these commands. Available as of cURL 7.15.3.
	 */
	define('CURLFTPMETHOD_NOCWD', 2);

	/**
	 * libcurl does one <code>CWD</code> with the full target directory and then operates on the file like in the multicwd case. Available as of cURL 7.15.3.
	 */
	define('CURLFTPMETHOD_SINGLECWD', 3);

	define('CURLFTPSSL_ALL', 3);

	/**
	 * Initiate the shutdown and wait for a reply. Available as of cURL 7.16.2.
	 */
	define('CURLFTPSSL_CCC_ACTIVE', 2);

	/**
	 * Do not attempt to use CCC (Clear Command Channel). Available as of cURL 7.16.2.
	 */
	define('CURLFTPSSL_CCC_NONE', 0);

	/**
	 * Do not initiate the shutdown, but wait for the server to do it. Do not send a reply. Available as of cURL 7.16.1.
	 */
	define('CURLFTPSSL_CCC_PASSIVE', 1);

	define('CURLFTPSSL_CONTROL', 2);

	define('CURLFTPSSL_NONE', 0);

	define('CURLFTPSSL_TRY', 1);

	/**
	 * Allow unconditional GSSAPI credential delegation. Available as of cURL 7.22.0.
	 */
	define('CURLGSSAPI_DELEGATION_FLAG', 2);

	/**
	 * Delegate only if the <code>OK-AS-DELEGATE</code> flag is set in the service ticket if this feature is supported by the GSS-API implementation and the definition of <code>GSS_C_DELEG_POLICY_FLAG</code> was available at compile-time. Available as of cURL 7.22.0.
	 */
	define('CURLGSSAPI_DELEGATION_POLICY_FLAG', 1);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.37.0.
	 */
	define('CURLHEADER_SEPARATE', 1);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.37.0.
	 */
	define('CURLHEADER_UNIFIED', 0);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.74.0
	 */
	define('CURLHSTS_ENABLE', 1);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.74.0
	 */
	define('CURLHSTS_READONLYFILE', 2);

	/**
	 * Time in seconds it took from the start until the SSL/SSH connect/handshake to the remote host was completed
	 */
	define('CURLINFO_APPCONNECT_TIME', 3145761);

	/**
	 * Time, in microseconds, it took from the start until the SSL/SSH connect/handshake to the remote host was completed. Available as of PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_APPCONNECT_TIME_T', 6291512);

	/**
	 * Default built-in CA certificate path. Available as of PHP 8.3.0 and cURL 7.84.0
	 */
	define('CURLINFO_CAINFO', null);

	/**
	 * Default built-in CA path string. Available as of PHP 8.3.0 and cURL 7.84.0
	 */
	define('CURLINFO_CAPATH', null);

	/**
	 * TLS certificate chain
	 */
	define('CURLINFO_CERTINFO', 4194338);

	/**
	 * Info on unmet time conditional
	 */
	define('CURLINFO_CONDITION_UNMET', 2097187);

	/**
	 * Time in seconds it took to establish the connection
	 */
	define('CURLINFO_CONNECT_TIME', 3145733);

	/**
	 * Total time taken, in microseconds, from the start until the connection to the remote host (or proxy) was completed. Available as of PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_CONNECT_TIME_T', 6291508);

	/**
	 * Content length of download, read from Content-Length: field
	 */
	define('CURLINFO_CONTENT_LENGTH_DOWNLOAD', 3145743);

	/**
	 * The content-length of the download. This is the value read from the Content-Length: field. -1 if the size isn't known. Available as of PHP 7.3.0 and cURL 7.55.0
	 */
	define('CURLINFO_CONTENT_LENGTH_DOWNLOAD_T', 6291471);

	/**
	 * Specified size of upload
	 */
	define('CURLINFO_CONTENT_LENGTH_UPLOAD', 3145744);

	/**
	 * The specified size of the upload. -1 if the size isn't known. Available as of PHP 7.3.0 and cURL 7.55.0
	 */
	define('CURLINFO_CONTENT_LENGTH_UPLOAD_T', 6291472);

	/**
	 * <code>Content-Type</code> of the requested document. NULL indicates server did not send valid <code>Content-Type</code> header
	 */
	define('CURLINFO_CONTENT_TYPE', 1048594);

	/**
	 * All known cookies
	 */
	define('CURLINFO_COOKIELIST', 4194332);

	/**
	 * Unprocessed protocol data received from the peer. Even if the data is encoded or compressed, it is not provided decoded nor decompressed to this callback.
	 */
	define('CURLINFO_DATA_IN', 3);

	/**
	 * Protocol data sent to the peer.
	 */
	define('CURLINFO_DATA_OUT', 4);

	/**
	 * Get the last used HTTP method.
	 */
	define('CURLINFO_EFFECTIVE_METHOD', 1048634);

	/**
	 * Last effective URL
	 */
	define('CURLINFO_EFFECTIVE_URL', 1048577);

	/**
	 * Remote time of the retrieved document, with the <b><code>CURLOPT_FILETIME</code></b> enabled; if -1 is returned the time of the document is unknown
	 */
	define('CURLINFO_FILETIME', 2097166);

	/**
	 * Remote time of the retrieved document (as Unix timestamp), an alternative to <b><code>CURLINFO_FILETIME</code></b> to allow systems with 32 bit long variables to extract dates outside of the 32bit timestamp range. Available as of PHP 7.3.0 and cURL 7.59.0
	 */
	define('CURLINFO_FILETIME_T', 6291470);

	/**
	 * Entry path in FTP server
	 */
	define('CURLINFO_FTP_ENTRY_PATH', 1048606);

	/**
	 * Header (or header-like) data received from the peer.
	 */
	define('CURLINFO_HEADER_IN', 1);

	/**
	 * The request string sent. For this to work, add the <b><code>CURLINFO_HEADER_OUT</code></b> option to the handle by calling <code>curl_setopt()</code>
	 */
	define('CURLINFO_HEADER_OUT', 2);

	/**
	 * Total size of all headers received
	 */
	define('CURLINFO_HEADER_SIZE', 2097163);

	/**
	 * The last response code. As of cURL 7.10.8, this is a legacy alias of <b><code>CURLINFO_RESPONSE_CODE</code></b>.
	 */
	define('CURLINFO_HTTP_CODE', 2097154);

	/**
	 * The CONNECT response code
	 */
	define('CURLINFO_HTTP_CONNECTCODE', 2097174);

	/**
	 * The version used in the last HTTP connection. The return value will be one of the defined <b><code>CURL_HTTP_VERSION_&#42;</code></b> constants or 0 if the version can't be determined. Available as of PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_HTTP_VERSION', 2097198);

	/**
	 * Bitmask indicating the authentication method(s) available according to the previous response
	 */
	define('CURLINFO_HTTPAUTH_AVAIL', 2097175);

	/**
	 * The last enum value in the underlying <code>CURLINFO</code> enum in <code>libcurl</code>.
	 */
	define('CURLINFO_LASTONE', 60);

	/**
	 * Local (source) IP address of the most recent connection
	 */
	define('CURLINFO_LOCAL_IP', 1048617);

	/**
	 * Local (source) port of the most recent connection
	 */
	define('CURLINFO_LOCAL_PORT', 2097194);

	/**
	 * Time in seconds until name resolving was complete
	 */
	define('CURLINFO_NAMELOOKUP_TIME', 3145732);

	/**
	 * Time in microseconds from the start until the name resolving was completed. Available as of PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_NAMELOOKUP_TIME_T', 6291507);

	/**
	 * Number of connections curl had to create to achieve the previous transfer
	 */
	define('CURLINFO_NUM_CONNECTS', 2097178);

	/**
	 * Errno from a connect failure. The number is OS and system specific.
	 */
	define('CURLINFO_OS_ERRNO', 2097177);

	/**
	 * Time it took from the start until the last byte is sent, in microseconds. Available as of PHP 8.4.0 and cURL 8.10.0
	 */
	define('CURLINFO_POSTTRANSFER_TIME_T', null);

	/**
	 * Time in seconds from start until just before file transfer begins
	 */
	define('CURLINFO_PRETRANSFER_TIME', 3145734);

	/**
	 * Time taken from the start until the file transfer is just about to begin, in microseconds. Available as of PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_PRETRANSFER_TIME_T', 6291509);

	/**
	 * IP address of the most recent connection
	 */
	define('CURLINFO_PRIMARY_IP', 1048608);

	/**
	 * Destination port of the most recent connection
	 */
	define('CURLINFO_PRIMARY_PORT', 2097192);

	/**
	 * Private data associated with this cURL handle, previously set with the <b><code>CURLOPT_PRIVATE</code></b> option of <code>curl_setopt()</code>
	 */
	define('CURLINFO_PRIVATE', 1048597);

	/**
	 * The protocol used in the last HTTP connection. The returned value will be exactly one of the <b><code>CURLPROTO_&#42;</code></b> values. Available as of PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLINFO_PROTOCOL', 2097200);

	/**
	 * The detailed (SOCKS) proxy error code when the most recent transfer returned a <b><code>CURLE_PROXY</code></b> error. The returned value will be exactly one of the <b><code>CURLPX_&#42;</code></b> values. The error code will be <b><code>CURLPX_OK</code></b> if no response code was available. Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLINFO_PROXY_ERROR', 2097211);

	/**
	 * The result of the certificate verification that was requested (using the <b><code>CURLOPT_PROXY_SSL_VERIFYPEER</code></b> option). Only used for HTTPS proxies. Available as of PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLINFO_PROXY_SSL_VERIFYRESULT', 2097199);

	/**
	 * Bitmask indicating the proxy authentication method(s) available according to the previous response
	 */
	define('CURLINFO_PROXYAUTH_AVAIL', 2097176);

	/**
	 * Number of redirects, with the <b><code>CURLOPT_FOLLOWLOCATION</code></b> option enabled
	 */
	define('CURLINFO_REDIRECT_COUNT', 2097172);

	/**
	 * Time in seconds of all redirection steps before final transaction was started, with the <b><code>CURLOPT_FOLLOWLOCATION</code></b> option enabled
	 */
	define('CURLINFO_REDIRECT_TIME', 3145747);

	/**
	 * Total time, in microseconds, it took for all redirection steps include name lookup, connect, pretransfer and transfer before final transaction was started. Available as of PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_REDIRECT_TIME_T', 6291511);

	/**
	 * With the <b><code>CURLOPT_FOLLOWLOCATION</code></b> option disabled: redirect URL found in the last transaction, that should be requested manually next. With the <b><code>CURLOPT_FOLLOWLOCATION</code></b> option enabled: this is empty. The redirect URL in this case is available in <b><code>CURLINFO_EFFECTIVE_URL</code></b>
	 */
	define('CURLINFO_REDIRECT_URL', 1048607);

	/**
	 * The <code>Referer</code> header. Available as of PHP 8.2.0 and cURL 7.76.0
	 */
	define('CURLINFO_REFERER', 1048636);

	/**
	 * Total size of issued requests, currently only for HTTP requests
	 */
	define('CURLINFO_REQUEST_SIZE', 2097164);

	/**
	 * The last response code. Available as of cURL 7.10.8
	 */
	define('CURLINFO_RESPONSE_CODE', 2097154);

	/**
	 * The information from the <code>Retry-After</code> header, or zero if there was no valid header. Available as of PHP 8.2.0 and cURL 7.66.0
	 */
	define('CURLINFO_RETRY_AFTER', 6291513);

	/**
	 * Next RTSP client CSeq
	 */
	define('CURLINFO_RTSP_CLIENT_CSEQ', 2097189);

	/**
	 * Recently received CSeq
	 */
	define('CURLINFO_RTSP_CSEQ_RECV', 2097191);

	/**
	 * Next RTSP server CSeq
	 */
	define('CURLINFO_RTSP_SERVER_CSEQ', 2097190);

	/**
	 * RTSP session ID
	 */
	define('CURLINFO_RTSP_SESSION_ID', 1048612);

	/**
	 * The URL scheme used for the most recent connection. Available as of PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLINFO_SCHEME', 1048625);

	/**
	 * Total number of bytes downloaded
	 */
	define('CURLINFO_SIZE_DOWNLOAD', 3145736);

	/**
	 * Total number of bytes that were downloaded. The number is only for the latest transfer and will be reset again for each new transfer. Available as of PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_SIZE_DOWNLOAD_T', 6291464);

	/**
	 * Total number of bytes uploaded
	 */
	define('CURLINFO_SIZE_UPLOAD', 3145735);

	/**
	 * Total number of bytes that were uploaded. Available as of PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_SIZE_UPLOAD_T', 6291463);

	/**
	 * Average download speed
	 */
	define('CURLINFO_SPEED_DOWNLOAD', 3145737);

	/**
	 * The average download speed in bytes/second that curl measured for the complete download. Available as of PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_SPEED_DOWNLOAD_T', 6291465);

	/**
	 * Average upload speed
	 */
	define('CURLINFO_SPEED_UPLOAD', 3145738);

	/**
	 * The average upload speed in bytes/second that curl measured for the complete upload. Available as of PHP 7.3.0 and cURL 7.50.0
	 */
	define('CURLINFO_SPEED_UPLOAD_T', 6291466);

	/**
	 * SSL/TLS (binary) data received from the peer.
	 */
	define('CURLINFO_SSL_DATA_IN', 5);

	/**
	 * SSL/TLS (binary) data sent to the peer.
	 */
	define('CURLINFO_SSL_DATA_OUT', 6);

	/**
	 * OpenSSL crypto-engines supported
	 */
	define('CURLINFO_SSL_ENGINES', 4194331);

	/**
	 * Result of SSL certification verification requested by setting <b><code>CURLOPT_SSL_VERIFYPEER</code></b>
	 */
	define('CURLINFO_SSL_VERIFYRESULT', 2097165);

	/**
	 * Time in seconds until the first byte is about to be transferred
	 */
	define('CURLINFO_STARTTRANSFER_TIME', 3145745);

	/**
	 * Time, in microseconds, it took from the start until the first byte is received. Available as of PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_STARTTRANSFER_TIME_T', 6291510);

	/**
	 * Informational text.
	 */
	define('CURLINFO_TEXT', 0);

	/**
	 * Total transaction time in seconds for last transfer
	 */
	define('CURLINFO_TOTAL_TIME', 3145731);

	/**
	 * Total time in microseconds for the previous transfer, including name resolving, TCP connect etc. Available as of PHP 7.3.0 and cURL 7.61.0
	 */
	define('CURLINFO_TOTAL_TIME_T', 6291506);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.19.6
	 */
	define('CURLKHMATCH_LAST', 3);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.19.6
	 */
	define('CURLKHMATCH_MISMATCH', 1);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.19.6
	 */
	define('CURLKHMATCH_MISSING', 2);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.19.6
	 */
	define('CURLKHMATCH_OK', 0);

	/**
	 * An easy handle already added to a multi handle was attempted to get added a second time. Available as of cURL 7.32.1.
	 */
	define('CURLM_ADDED_ALREADY', 7);

	/**
	 * An easy handle was not good/valid. It could mean that it is not an easy handle at all, or possibly that the handle already is in use by this or another multi handle. Available as of cURL 7.9.6.
	 */
	define('CURLM_BAD_EASY_HANDLE', 2);

	/**
	 * The passed-in handle is not a valid multi handle. Available as of cURL 7.9.6.
	 */
	define('CURLM_BAD_HANDLE', 1);

	/**
	 * As of cURL 7.20.0, this constant is not used. Before cURL 7.20.0, this status could be returned by <code>curl_multi_exec()</code> when <code>curl_multi_select()</code> or a similar function was called before it returned any other constant. Available as of cURL 7.9.6.
	 */
	define('CURLM_CALL_MULTI_PERFORM', -1);

	/**
	 * Internal <code>libcurl</code> error. Available as of cURL 7.9.6.
	 */
	define('CURLM_INTERNAL_ERROR', 4);

	/**
	 * No errors. Available as of cURL 7.9.6.
	 */
	define('CURLM_OK', 0);

	/**
	 * Ran out of memory while processing multi handles. Available as of cURL 7.9.6.
	 */
	define('CURLM_OUT_OF_MEMORY', 3);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.81.0
	 */
	define('CURLMIMEOPT_FORMESCAPE', 1);

	/**
	 * Specifies the chunk length threshold for pipelining in bytes. Available as of PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_CHUNK_LENGTH_PENALTY_SIZE', 30010);

	/**
	 * Specifies the size threshold for pipelining penalty in bytes. Available as of PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_CONTENT_LENGTH_PENALTY_SIZE', 30009);

	/**
	 * Specifies the maximum number of concurrent streams for connections that cURL should support on connections using HTTP/2. Valid values range from <code>1</code> to <code>2147483647</code> (<code>2^31 - 1</code>). The value passed here would be honored based on other system resources properties. Default is <code>100</code>. Available as of PHP 8.2.0 and cURL 7.67.0.
	 */
	define('CURLMOPT_MAX_CONCURRENT_STREAMS', 16);

	/**
	 * Specifies the maximum number of connections to a single host. Available as of PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_MAX_HOST_CONNECTIONS', 7);

	/**
	 * Specifies the maximum number of requests in a pipeline. Available as of PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_MAX_PIPELINE_LENGTH', 8);

	/**
	 * Specifies the maximum number of simultaneously open connections. Available as of PHP 7.0.7 and cURL 7.30.0
	 */
	define('CURLMOPT_MAX_TOTAL_CONNECTIONS', 13);

	/**
	 * Specifies the maximum amount of simultaneously open connections that libcurl may cache. By default the size will be enlarged to fit four times the number of handles added via <code>curl_multi_add_handle()</code>. When the cache is full, curl closes the oldest one in the cache to prevent the number of open connections from increasing. Available as of cURL 7.16.3.
	 */
	define('CURLMOPT_MAXCONNECTS', 6);

	/**
	 * Pass 1 to enable or 0 to disable. Enabling pipelining on a multi handle will make it attempt to perform HTTP Pipelining as far as possible for transfers using this handle. This means that adding a second request that can use an already existing connection will "pipe" the second request on the same connection. As of cURL 7.43.0, the value is a bitmask, and passing 2 will try to multiplex the new transfer over an existing HTTP/2 connection. Passing 3 instructs cURL to ask for pipelining and multiplexing independently of each other. As of cURL 7.62.0, setting the pipelining bit has no effect. Instead of integer literals, the CURLPIPE_&#42; constants can also be used. Available as of cURL 7.16.0.
	 */
	define('CURLMOPT_PIPELINING', 3);

	/**
	 * <p>Pass a <code>callable</code> that will be registered to handle server pushes and should have the following signature:</p> pushfunction(<code>resource</code> <code>$parent_ch</code>, <code>resource</code> <code>$pushed_ch</code>, <code>array</code> <code>$headers</code>): <code>int</code>  <code>parent_ch</code>   The parent cURL handle (the request the client made).   <code>pushed_ch</code>   A new cURL handle for the pushed request.   <code>headers</code>   The push promise headers.    The push function is supposed to return either <b><code>CURL_PUSH_OK</code></b> if it can handle the push, or <b><code>CURL_PUSH_DENY</code></b> to reject it. Available as of PHP 7.1.0 and cURL 7.44.0
	 */
	define('CURLMOPT_PUSHFUNCTION', 20014);

	define('CURLMSG_DONE', 1);

	/**
	 * <p>Enables the use of an abstract Unix domain socket instead of establishing a TCP connection to a host and sets the path to the given <code>string</code>. This option shares the same semantics as <b><code>CURLOPT_UNIX_SOCKET_PATH</code></b>. These two options share the same storage and therefore only one of them can be set per handle. Available as of PHP 7.3.0 and cURL 7.53.0.</p>
	 */
	define('CURLOPT_ABSTRACT_UNIX_SOCKET', 10264);

	/**
	 * <p>Sets a <code>string</code> with the contents of the <code>Accept-Encoding:</code> header sent in an HTTP request. Set to <b><code>null</code></b> to disable sending the <code>Accept-Encoding:</code> header. Defaults to <b><code>null</code></b>. Available as of cURL 7.21.6.</p>
	 */
	define('CURLOPT_ACCEPT_ENCODING', 10102);

	/**
	 * <p>The maximum number of milliseconds to wait for a server to connect back to cURL when an active FTP connection is used. This option accepts any value that can be cast to a valid <code>int</code>. Defaults to <code>60000</code> milliseconds. Available as of cURL 7.24.0.</p>
	 */
	define('CURLOPT_ACCEPTTIMEOUT_MS', 212);

	/**
	 * <p>The scope id value to use when connecting to IPv6 addresses. This option accepts any value that can be cast to a valid <code>int</code>. Defaults to <code>0</code>. Available as of cURL 7.19.0.</p>
	 */
	define('CURLOPT_ADDRESS_SCOPE', 171);

	/**
	 * <p>Pass a <code>string</code> with the filename for cURL to use as the Alt-Svc cache file to read existing cache contents from and possibly also write it back to a after a transfer, unless <b><code>CURLALTSVC_READONLYFILE</code></b> is set via <b><code>CURLOPT_ALTSVC_CTRL</code></b>. Available as of PHP 8.2.0 and cURL 7.64.1.</p>
	 */
	define('CURLOPT_ALTSVC', 10287);

	/**
	 * <p>Populate the bitmask with the correct set of features to instruct cURL how to handle Alt-Svc for the transfers using this handle. cURL only accepts Alt-Svc headers over HTTPS. It will also only complete a request to an alternative origin if that origin is properly hosted over HTTPS. Setting any bit will enable the alt-svc engine. Set to any of the <b><code>CURLALTSVC_&#42;</code></b> constants. Defaults to Alt-Svc handling being disabled. Available as of PHP 8.2.0 and cURL 7.64.1.</p>
	 */
	define('CURLOPT_ALTSVC_CTRL', 286);

	/**
	 * <p>Setting this option to <code>1</code> will cause FTP uploads to append to the remote file instead of overwriting it. Defaults to <code>0</code>. Available as of cURL 7.17.0.</p>
	 */
	define('CURLOPT_APPEND', 50);

	/**
	 * <p><b><code>true</code></b> to automatically set the <code>Referer:</code> field in requests where it follows a <code>Location:</code> redirect. Defaults to <code>0</code>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_AUTOREFERER', 58);

	/**
	 * <p>Provides AWS V4 signature authentication on HTTP(S) header as a <code>string</code>. This option overrides any other authentication types that have been set in <b><code>CURLOPT_HTTPAUTH</code></b>. This method cannot be combined with other authentication types. Available as of PHP 8.2.0 and cURL 7.75.0.</p>
	 */
	define('CURLOPT_AWS_SIGV4', 10305);

	/**
	 * <p>This constant is no longer used as of PHP 5.5.0. Deprecated as of PHP 8.4.0.</p>
	 */
	define('CURLOPT_BINARYTRANSFER', 19914);

	/**
	 * <p>The size of the buffer to use for each read. There is no guarantee this request will be fulfilled, however. This option accepts any value that can be cast to a valid <code>int</code>. Defaults to <b><code>CURL_MAX_WRITE_SIZE</code></b> (currently, 16kB). Available as of cURL 7.10.</p>
	 */
	define('CURLOPT_BUFFERSIZE', 98);

	/**
	 * <p>Sets the maximum time in seconds any in memory cached CA certificate store may be kept and reused for new connections. This option accepts any value that can be cast to a valid <code>int</code>. Defaults to <code>86400</code> (24 hours). Available as of PHP 8.3.0 and cURL 7.87.0</p>
	 */
	define('CURLOPT_CA_CACHE_TIMEOUT', null);

	/**
	 * <p>A <code>string</code> with the name of a file holding one or more certificates to verify the peer with. This only makes sense when used in combination with <b><code>CURLOPT_SSL_VERIFYPEER</code></b>. Might require an absolute path. Available as of cURL 7.4.2.</p>
	 */
	define('CURLOPT_CAINFO', 10065);

	/**
	 * <p>A <code>string</code> with the name of a PEM file holding one or more certificates to verify the peer with. This option overrides <b><code>CURLOPT_CAINFO</code></b>. Available as of PHP 8.2.0 and cURL 7.77.0.</p>
	 */
	define('CURLOPT_CAINFO_BLOB', 40309);

	/**
	 * <p>A <code>string</code> with a directory that holds multiple CA certificates. Use this option alongside <b><code>CURLOPT_SSL_VERIFYPEER</code></b>. Available as of cURL 7.9.8.</p>
	 */
	define('CURLOPT_CAPATH', 10097);

	/**
	 * <p><b><code>true</code></b> to output SSL certification information to <b><code>STDERR</code></b> on secure transfers. Requires <b><code>CURLOPT_VERBOSE</code></b> to be on to have an effect. Defaults to <b><code>false</code></b>. Available as of cURL 7.19.1.</p>
	 */
	define('CURLOPT_CERTINFO', 172);

	/**
	 * <p><b><code>true</code></b> tells the library to perform all the required proxy authentication and connection setup, but no data transfer. This option is implemented for HTTP, SMTP and POP3. Defaults to <b><code>false</code></b>. Available as of cURL 7.15.2.</p>
	 */
	define('CURLOPT_CONNECT_ONLY', 141);

	/**
	 * <p>Connect to a specific host and port instead of the URL's host and port. Accepts an <code>array</code> of <code>string</code>s with the format <code>HOST:PORT:CONNECT-TO-HOST:CONNECT-TO-PORT</code>. Available as of PHP 7.0.7 and cURL 7.49.0.</p>
	 */
	define('CURLOPT_CONNECT_TO', 10243);

	/**
	 * <p>The number of seconds to wait while trying to connect. Use 0 to wait indefinitely. This option accepts any value that can be cast to a valid <code>int</code>. Defaults to <code>300</code>. Available as of cURL 7.7.0.</p>
	 */
	define('CURLOPT_CONNECTTIMEOUT', 78);

	/**
	 * <p>The number of milliseconds to wait while trying to connect. Use <code>0</code> to wait indefinitely. If cURL is built to use the standard system name resolver, that portion of the connect will still use full-second resolution for timeouts with a minimum timeout allowed of one second. This option accepts any value that can be cast to a valid <code>int</code>. Defaults to <code>300000</code>. Available as of cURL 7.16.2.</p>
	 */
	define('CURLOPT_CONNECTTIMEOUT_MS', 156);

	/**
	 * <p>A <code>string</code> with the contents of the <code>Cookie: </code> header to be used in the HTTP request. Note that multiple cookies are separated with a semicolon followed by a space (e.g., <code>fruit=apple; colour=red</code>). Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_COOKIE', 10022);

	/**
	 * <p>A <code>string</code> with the name of the file containing the cookie data. The cookie file can be in Netscape format, or just plain HTTP-style headers dumped into a file. If the name is an empty <code>string</code>, no cookies are loaded, but cookie handling is still enabled. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_COOKIEFILE', 10031);

	/**
	 * <p>A <code>string</code> with the name of a file to save all internal cookies to when the handle's destructor is called. Available as of cURL 7.9.0.</p><p><b>Warning</b></p> <p>As of PHP 8.0.0, <code>curl_close()</code> is a no-op and does <i>not</i> destroy the handle. If cookies need to be written prior to the handle being automatically destroyed, call <code>unset()</code> on the handle.</p>
	 */
	define('CURLOPT_COOKIEJAR', 10082);

	/**
	 * <p>A cookie <code>string</code> (i.e. a single line in Netscape/Mozilla format, or a regular HTTP-style Set-Cookie header) adds that single cookie to the internal cookie store.  <code>ALL</code> erases all cookies held in memory , <code>SESS</code> erases all session cookies held in memory , <code>FLUSH</code> writes all known cookies to the file specified by <b><code>CURLOPT_COOKIEJAR</code></b> , <code>RELOAD</code> loads all cookies from the files specified by <b><code>CURLOPT_COOKIEFILE</code></b> . Available as of cURL 7.14.1.</p>
	 */
	define('CURLOPT_COOKIELIST', 10135);

	/**
	 * <p><b><code>true</code></b> to mark this as a new cookie "session". It will force cURL to ignore all cookies it is about to load that are "session cookies" from the previous session. By default, cURL always stores and loads all cookies, independent if they are session cookies or not. Session cookies are cookies without expiry date and they are meant to be alive and existing for this "session" only. Available as of cURL 7.9.7.</p>
	 */
	define('CURLOPT_COOKIESESSION', 96);

	/**
	 * <p><b><code>true</code></b> to convert Unix newlines to CRLF newlines on transfers. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_CRLF', 27);

	/**
	 * <p>Pass a <code>string</code> naming a file with the concatenation of CRL (Certificate Revocation List) (in PEM format) to use in the certificate validation that occurs during the SSL exchange. When cURL is built to use GnuTLS, there is no way to influence the use of CRL passed to help in the verification process. When cURL is built with OpenSSL support, <code>X509_V_FLAG_CRL_CHECK</code> and <code>X509_V_FLAG_CRL_CHECK_ALL</code> are both set, requiring CRL check against all the elements of the certificate chain if a CRL file is passed. Also note that <b><code>CURLOPT_CRLFILE</code></b> implies <b><code>CURLSSLOPT_NO_PARTIALCHAIN</code></b> as of cURL 7.71.0 due to an OpenSSL bug. Available as of cURL 7.19.0.</p>
	 */
	define('CURLOPT_CRLFILE', 10169);

	/**
	 * <p>A custom request method to use instead of <code>GET</code> or <code>HEAD</code> when doing a HTTP request. This is useful for doing <code>DELETE</code> or other, more obscure HTTP requests. Valid values are things like <code>GET</code>, <code>POST</code>, <code>CONNECT</code> and so on; i.e. Do not enter a whole HTTP request line here. For instance, entering <code>GET /index.html HTTP/1.0\r\n\r\n</code> would be incorrect. This option accepts a <code>string</code> or <b><code>null</code></b>. Available as of cURL 7.1.0.</p><p><b>Note</b>:</p><p>Don't do this without making sure the server supports the custom request method first.</p>
	 */
	define('CURLOPT_CUSTOMREQUEST', 10036);

	/**
	 * <p>Available as of PHP 8.4.0. This option requires <b><code>CURLOPT_VERBOSE</code></b> option enabled. A <code>callable</code> to replace the standard cURL verbose output. This callback gets called during various stages of the request with verbose debug information. The callback should match the following signature:</p> callback(<code>CurlHandle</code> <code>$curlHandle</code>, <code>int</code> <code>$type</code>, <code>string</code> <code>$data</code>): <code>void</code>  <code>curlHandle</code>   The cURL handle.   <code>type</code>   One of the following constants indicating the type of the <code>data</code> value:    Constants Description    <b><code>CURLINFO_TEXT</code></b> (<code>int</code>)    Informational text.      <b><code>CURLINFO_HEADER_IN</code></b> (<code>int</code>)    Header (or header-like) data received from the peer.      <b><code>CURLINFO_HEADER_OUT</code></b> (<code>int</code>)    Header (or header-like) data sent to the peer.      <b><code>CURLINFO_DATA_IN</code></b> (<code>int</code>)    Unprocessed protocol data received from the peer. Even if the data is encoded or compressed, it is not provided decoded nor decompressed to this callback.      <b><code>CURLINFO_DATA_OUT</code></b> (<code>int</code>)    Protocol data sent to the peer.      <b><code>CURLINFO_SSL_DATA_IN</code></b> (<code>int</code>)    SSL/TLS (binary) data received from the peer.      <b><code>CURLINFO_SSL_DATA_OUT</code></b> (<code>int</code>)    SSL/TLS (binary) data sent to the peer.      <code>data</code>   Verbose debug data of the type indicate by the <code>type</code> parameter.
	 */
	define('CURLOPT_DEBUGFUNCTION', 20094);

	/**
	 * <p>A <code>string</code> with the default protocol to use if the URL is missing a scheme name. Available as of PHP 7.0.7 and cURL 7.45.0.</p>
	 */
	define('CURLOPT_DEFAULT_PROTOCOL', 10238);

	/**
	 * <p>Setting this option to <code>1</code> will have different effects based on the protocol it is used with. FTP and SFTP based URLs will list only the names of files in a directory. POP3 will list the email message or messages on the POP3 server. For FILE, this option has no effect as directories are always listed in this mode. Using this option with <b><code>CURLOPT_WILDCARDMATCH</code></b> will prevent the latter from having any effect. Defaults to <code>0</code>. Available as of cURL 7.17.0.</p>
	 */
	define('CURLOPT_DIRLISTONLY', 48);

	/**
	 * <p><b><code>true</code></b> to not allow URLs that include a username. Usernames are allowed by default. Available as of PHP 7.3.0 and cURL 7.61.0.</p>
	 */
	define('CURLOPT_DISALLOW_USERNAME_IN_URL', 278);

	/**
	 * <p>The number of seconds to keep DNS entries in memory. This option is set to <code>120</code> (2 minutes) by default. This option accepts any value that can be cast to a valid <code>int</code>. Available as of cURL 7.9.3.</p>
	 */
	define('CURLOPT_DNS_CACHE_TIMEOUT', 92);

	/**
	 * <p>Set the name of the network interface that the DNS resolver should bind to. This must be an interface name (not an address). This option accepts a <code>string</code> or <b><code>null</code></b>. Available as of PHP 7.0.7 and cURL 7.33.0</p>
	 */
	define('CURLOPT_DNS_INTERFACE', 10221);

	/**
	 * <p>Set the local IPv4 address that the resolver should bind to. The argument should contain a single numerical IPv4 address. This option accepts a <code>string</code> or <b><code>null</code></b>. Available as of PHP 7.0.7 and cURL 7.33.0.</p>
	 */
	define('CURLOPT_DNS_LOCAL_IP4', 10222);

	/**
	 * <p>Set the local IPv6 address that the resolver should bind to. The argument should contain a single numerical IPv6 address. This option accepts a <code>string</code> or <b><code>null</code></b>. Available as of PHP 7.0.7 and cURL 7.33.0.</p>
	 */
	define('CURLOPT_DNS_LOCAL_IP6', 10223);

	/**
	 * <p>Pass a <code>string</code> with a comma-separated list of DNS servers to be used instead of the system default (e.g.: <code>192.168.1.100,192.168.1.101:8080</code>). Available as of cURL 7.24.0.</p>
	 */
	define('CURLOPT_DNS_SERVERS', 10211);

	/**
	 * <p><b><code>true</code></b> to shuffle the order of all returned addresses so that they will be used in a random order, when a name is resolved and more than one IP address is returned. This may cause IPv4 to be used before IPv6 or vice versa. Available as of PHP 7.3.0 and cURL 7.60.0.</p>
	 */
	define('CURLOPT_DNS_SHUFFLE_ADDRESSES', 275);

	/**
	 * <p><b><code>true</code></b> to use a global DNS cache. This option is not thread-safe. It is conditionally enabled by default if PHP is built for non-threaded use (CLI, FCGI, Apache2-Prefork, etc.). Available as of cURL 7.9.3 and deprecated as of cURL 7.11.1. As of PHP 8.4, this option no longer has any effect.</p>
	 */
	define('CURLOPT_DNS_USE_GLOBAL_CACHE', 91);

	/**
	 * <p>Set to <code>2</code> to verify the DNS-over-HTTPS server's SSL certificate name fields against the host name. Available as of PHP 8.2.0 and cURL 7.76.0.</p>
	 */
	define('CURLOPT_DOH_SSL_VERIFYHOST', 307);

	/**
	 * <p>Set to <code>1</code> to enable and <code>0</code> to disable verification of the authenticity of the DNS-over-HTTPS server's SSL certificate. Available as of PHP 8.2.0 and cURL 7.76.0.</p>
	 */
	define('CURLOPT_DOH_SSL_VERIFYPEER', 306);

	/**
	 * <p>Set to <code>1</code> to enable and <code>0</code> to disable the verification of the status of the DNS-over-HTTPS server certificate using the "Certificate Status Request" TLS extension (OCSP stapling). Available as of PHP 8.2.0 and cURL 7.76.0.</p>
	 */
	define('CURLOPT_DOH_SSL_VERIFYSTATUS', 308);

	/**
	 * <p>Provides the DNS-over-HTTPS URL. This option accepts a <code>string</code> or <b><code>null</code></b>. Available as of PHP 8.1.0 and cURL 7.62.0.</p>
	 */
	define('CURLOPT_DOH_URL', 10279);

	/**
	 * <p>Like <b><code>CURLOPT_RANDOM_FILE</code></b>, except a filename to an Entropy Gathering Daemon socket. Available as of cURL 7.7.0 and deprecated as of cURL 7.84.0.</p>
	 */
	define('CURLOPT_EGDSOCKET', 10077);

	/**
	 * <p>The contents of the <code>Accept-Encoding: </code> header as a <code>string</code>. This enables decoding of the response. Supported encodings are: <code>identity</code>, <code>deflate</code>, <code>gzip</code>. If an empty <code>string</code> is set, a header containing all supported encoding types is sent. Available as of cURL 7.10 and deprecated as of cURL 7.21.6.</p>
	 */
	define('CURLOPT_ENCODING', 10102);

	/**
	 * <p>The timeout for <code>Expect: 100-continue</code> responses in milliseconds. Defaults to <code>1000</code> milliseconds. This option accepts any value that can be cast to a valid <code>int</code>. Available as of PHP 7.0.7 and cURL 7.36.0.</p>
	 */
	define('CURLOPT_EXPECT_100_TIMEOUT_MS', 227);

	/**
	 * <p><b><code>true</code></b> to fail verbosely if the HTTP code returned is greater than or equal to <code>400</code>. The default behavior is to return the page normally, ignoring the code. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_FAILONERROR', 45);

	/**
	 * <p>Accepts a file handle <code>resource</code> to the file that the transfer should be written to. The default is <b><code>STDOUT</code></b> (the browser window). Available as of cURL 7.1.0 and deprecated as of cURL 7.9.7.</p>
	 */
	define('CURLOPT_FILE', 10001);

	/**
	 * <p>Set to <b><code>true</code></b> to attempt to retrieve the modification date of the remote document. This value can be retrieved using the <b><code>CURLINFO_FILETIME</code></b> option with <code>curl_getinfo()</code>. Available as of cURL 7.5.0.</p>
	 */
	define('CURLOPT_FILETIME', 69);

	/**
	 * <p>Pass a <code>callable</code> that will be used for wildcard matching. The signature of the callback should be:</p> callback(<code>resource</code> <code>$curlHandle</code>, <code>string</code> <code>$pattern</code>, <code>string</code> <code>$string</code>): <code>int</code>  <code>curlHandle</code>   The cURL handle.   <code>pattern</code>   The wildcard pattern.   <code>string</code>   The <code>string</code> to run the wildcard pattern matching on.    The callback should return <b><code>CURL_FNMATCHFUNC_MATCH</code></b> if pattern matches the <code>string</code>, <b><code>CURL_FNMATCHFUNC_NOMATCH</code></b> if not or <b><code>CURL_FNMATCHFUNC_FAIL</code></b> if an error occurred. Available as of cURL 7.21.0.
	 */
	define('CURLOPT_FNMATCH_FUNCTION', 20200);

	/**
	 * <p>Set to <b><code>true</code></b> to follow any <code>Location: </code> header that the server sends as part of the HTTP header. See also <b><code>CURLOPT_MAXREDIRS</code></b>. This constant is not available when open_basedir is enabled. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_FOLLOWLOCATION', 52);

	/**
	 * <p>Set to <b><code>true</code></b> to force the connection to explicitly close when it has finished processing, and not be pooled for reuse. Available as of cURL 7.7.0.</p>
	 */
	define('CURLOPT_FORBID_REUSE', 75);

	/**
	 * <p>Set to <b><code>true</code></b> to force the use of a new connection instead of a cached one. Available as of cURL 7.7.0.</p>
	 */
	define('CURLOPT_FRESH_CONNECT', 74);

	/**
	 * <p>Pass a <code>string</code> that will be sent as account information over FTP (using the <code>ACCT</code> command) after username and password has been provided to the server. Set to <b><code>null</code></b> to disable sending the account information. Defaults to <b><code>null</code></b>. Available as of cURL 7.13.0.</p>
	 */
	define('CURLOPT_FTP_ACCOUNT', 10134);

	/**
	 * <p>Pass a <code>string</code> that will be used to try to authenticate over FTP if the <code>USER/PASS</code> negotiation fails. Available as of cURL 7.15.5.</p>
	 */
	define('CURLOPT_FTP_ALTERNATIVE_TO_USER', 10147);

	/**
	 * <p>Set to <b><code>true</code></b> to create missing directories when an FTP operation encounters a path that currently doesn't exist. Available as of cURL 7.10.7.</p>
	 */
	define('CURLOPT_FTP_CREATE_MISSING_DIRS', 110);

	/**
	 * <p>Tell cURL which method to use to reach a file on a FTP(S) server. Possible values are any of the <b><code>CURLFTPMETHOD_&#42;</code></b> constants. Defaults to <b><code>CURLFTPMETHOD_MULTICWD</code></b>. Available as of cURL 7.15.1.</p>
	 */
	define('CURLOPT_FTP_FILEMETHOD', 138);

	/**
	 * <p>A timeout in seconds cURL will wait for a response from an FTP server. This option overrides <b><code>CURLOPT_TIMEOUT</code></b>. This option accepts any value that can be cast to a valid <code>int</code>. This option name is replaced with <b><code>CURLOPT_SERVER_RESPONSE_TIMEOUT</code></b>, available as of PHP 8.4.0. Available as of cURL 7.10.8 and deprecated as of cURL 7.85.0.</p>
	 */
	define('CURLOPT_FTP_RESPONSE_TIMEOUT', 112);

	/**
	 * <p>If this option is set to <code>1</code> cURL will not use the IP address the server suggests in its 227-response to cURL's <code>PASV</code> command but will use the IP address it used for the connection. The port number received from the 227-response will not be ignored by cURL. Defaults to <code>1</code> as of cURL 7.74.0 and <code>0</code> prior to that. Available as of cURL 7.15.0.</p>
	 */
	define('CURLOPT_FTP_SKIP_PASV_IP', 137);

	/**
	 * <p>Available as of cURL 7.11.0 and deprecated as of cURL 7.16.4.</p>
	 */
	define('CURLOPT_FTP_SSL', 119);

	/**
	 * <p>This option makes cURL use CCC (Clear Command Channel) which shuts down the SSL/TLS layer after authenticating making the rest of the control channel communication unencrypted. Use one of the <b><code>CURLFTPSSL_CCC_&#42;</code></b> constants. Defaults to <b><code>CURLFTPSSL_CCC_NONE</code></b>. Available as of cURL 7.16.1.</p>
	 */
	define('CURLOPT_FTP_SSL_CCC', 154);

	/**
	 * <p>Set to <b><code>true</code></b> to use <code>EPRT</code> (and <code>LPRT</code>) when doing active FTP downloads. Set to <b><code>false</code></b> to disable <code>EPRT</code> and <code>LPRT</code> and use <code>PORT</code> only. Available as of cURL 7.10.5.</p>
	 */
	define('CURLOPT_FTP_USE_EPRT', 106);

	/**
	 * <p>Set to <b><code>true</code></b> to first try an <code>EPSV</code> command for FTP transfers before reverting back to <code>PASV</code>. Set to <b><code>false</code></b> to disable <code>EPSV</code>. Available as of cURL 7.9.2.</p>
	 */
	define('CURLOPT_FTP_USE_EPSV', 85);

	/**
	 * <p>Set to <code>1</code> to send a <code>PRET</code> command before <code>PASV</code> (and <code>EPSV</code>). Has no effect when using the active FTP transfers mode. Defaults to <code>0</code>. Available as of cURL 7.20.0.</p>
	 */
	define('CURLOPT_FTP_USE_PRET', 188);

	/**
	 * <p>Set to <b><code>true</code></b> to append to the remote file instead of overwriting it. Available as of cURL 7.1.0 and deprecated as of cURL 7.16.4.</p>
	 */
	define('CURLOPT_FTPAPPEND', 50);

	/**
	 * <p>An alias of <b><code>CURLOPT_TRANSFERTEXT</code></b>. Use that instead. Available as of cURL 7.1, deprecated as of cURL 7.11.1 and last available in cURL 7.15.5. Removed as of PHP 7.3.0.</p>
	 */
	define('CURLOPT_FTPASCII', null);

	/**
	 * <p>Set to <b><code>true</code></b> to only list the names of an FTP directory. Available as of cURL 7.1.0 and deprecated as of cURL 7.16.4.</p>
	 */
	define('CURLOPT_FTPLISTONLY', 48);

	/**
	 * <p>A <code>string</code> which will be used to get the IP address to use for the FTP <code>PORT</code> instruction. The <code>PORT</code> instruction tells the remote server to connect to our specified IP address. The <code>string</code> may be a plain IP address, a hostname, a network interface name (under Unix), or just a plain <code>-</code> to use the system's default IP address. This option accepts a <code>string</code> or <b><code>null</code></b>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_FTPPORT', 10017);

	/**
	 * <p>Set the FTP over SSL authentication method (if activated) to any of the <b><code>CURLFTPAUTH_&#42;</code></b> constants. Defaults to <b><code>CURLFTPAUTH_DEFAULT</code></b>. Available as of cURL 7.12.2.</p>
	 */
	define('CURLOPT_FTPSSLAUTH', 129);

	/**
	 * <p>Set to <b><code>CURLGSSAPI_DELEGATION_FLAG</code></b> to allow unconditional GSSAPI credential delegation. Set to <b><code>CURLGSSAPI_DELEGATION_POLICY_FLAG</code></b> to delegate only if the <code>OK-AS-DELEGATE</code> flag is set in the service ticket. Defaults to <b><code>CURLGSSAPI_DELEGATION_NONE</code></b>. Available as of cURL 7.22.0.</p>
	 */
	define('CURLOPT_GSSAPI_DELEGATION', 210);

	/**
	 * <p>Head start for IPv6 for the happy eyeballs algorithm. Happy eyeballs attempts to connect to both IPv4 and IPv6 addresses for dual-stack hosts, preferring IPv6 first for timeout milliseconds. Defaults to <b><code>CURL_HET_DEFAULT</code></b>, which is currently 200 milliseconds. This option accepts any value that can be cast to a valid <code>int</code>. Available as of PHP 7.3.0 and cURL 7.59.0</p>
	 */
	define('CURLOPT_HAPPY_EYEBALLS_TIMEOUT_MS', 271);

	/**
	 * <p><b><code>true</code></b> to send an HAProxy <code>PROXY</code> protocol v1 header at the start of the connection. The default action is not to send this header. Available as of PHP 7.3.0 and cURL 7.60.0.</p>
	 */
	define('CURLOPT_HAPROXYPROTOCOL', 274);

	/**
	 * <p>Set to <b><code>true</code></b> to include the headers in the output sent to the callback defined by <b><code>CURLOPT_WRITEFUNCTION</code></b>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_HEADER', 42);

	/**
	 * <p>A <code>callable</code> with the following signature:</p> callback(<code>resource</code> <code>$curlHandle</code>, <code>string</code> <code>$headerData</code>): <code>int</code>  <code>curlHandle</code>   The cURL handle.   <code>headerData</code>   The header data which must be written by the callback.    The callback should return the number of bytes written. Available as of cURL 7.7.2.
	 */
	define('CURLOPT_HEADERFUNCTION', 20079);

	/**
	 * <p>Send HTTP headers to both proxy and host or separately. Possible values are any of the <b><code>CURLHEADER_&#42;</code></b> constants. Defaults to <b><code>CURLHEADER_SEPARATE</code></b> as of cURL 7.42.1, and <b><code>CURLHEADER_UNIFIED</code></b> prior to that. Available as of PHP 7.0.7 and cURL 7.37.0.</p>
	 */
	define('CURLOPT_HEADEROPT', 229);

	/**
	 * <p><code>string</code> with HSTS (HTTP Strict Transport Security) cache file name or <b><code>null</code></b> to allow HSTS without reading from or writing to any file and clear the list of files to read HSTS data from. Available as of PHP 8.2.0 and cURL 7.74.0.</p>
	 */
	define('CURLOPT_HSTS', 10300);

	/**
	 * <p>Accepts a bitmask of HSTS (HTTP Strict Transport Security) features defined by the <b><code>CURLHSTS_&#42;</code></b> constants. Available as of PHP 8.2.0 and cURL 7.74.0.</p>
	 */
	define('CURLOPT_HSTS_CTRL', 299);

	/**
	 * <p>Whether to allow HTTP/0.9 responses. Defaults to <b><code>false</code></b> as of cURL 7.66.0; formerly it defaulted to <b><code>true</code></b>. Available as of PHP 7.3.15 and 7.4.3, respectively, and cURL 7.64.0.</p>
	 */
	define('CURLOPT_HTTP09_ALLOWED', 285);

	/**
	 * <p>An <code>array</code> of HTTP <code>200</code> responses that will be treated as valid responses and not as errors. Available as of cURL 7.10.3.</p>
	 */
	define('CURLOPT_HTTP200ALIASES', 10104);

	/**
	 * <p><b><code>false</code></b> to get the raw HTTP response body. Available as of cURL 7.16.2.</p>
	 */
	define('CURLOPT_HTTP_CONTENT_DECODING', 158);

	/**
	 * <p>If set to <code>0</code>, transfer decoding is disabled. If set to <code>1</code>, transfer decoding is enabled. cURL does chunked transfer decoding by default unless this option is set to <code>0</code>. Defaults to <code>1</code>. Available as of cURL 7.16.2.</p>
	 */
	define('CURLOPT_HTTP_TRANSFER_DECODING', 157);

	/**
	 * <p>Set to one of the <b><code>CURL_HTTP_VERSION_&#42;</code></b> constants for cURL to use the specified HTTP version. Available as of cURL 7.9.1.</p>
	 */
	define('CURLOPT_HTTP_VERSION', 84);

	/**
	 * <p>A bitmask of HTTP authentication method(s) to use. The options are: <b><code>CURLAUTH_BASIC</code></b>, <b><code>CURLAUTH_DIGEST</code></b>, <b><code>CURLAUTH_GSSNEGOTIATE</code></b>, <b><code>CURLAUTH_NTLM</code></b>, <b><code>CURLAUTH_AWS_SIGV4</code></b>, <b><code>CURLAUTH_ANY</code></b>, <b><code>CURLAUTH_ANYSAFE</code></b>. If more than one method is used, cURL will poll the server to see what methods it supports and pick the best one. <b><code>CURLAUTH_ANY</code></b> sets all bits. cURL will automatically select the one it finds most secure. <b><code>CURLAUTH_ANYSAFE</code></b> sets all bits except <b><code>CURLAUTH_BASIC</code></b>. cURL will automatically select the one it finds most secure. Available as of cURL 7.10.6.</p>
	 */
	define('CURLOPT_HTTPAUTH', 107);

	/**
	 * <p>Set to <b><code>true</code></b> to reset the HTTP request method to <code>GET</code>. Since <code>GET</code> is the default, this is only necessary if the request method has been changed. Available as of cURL 7.8.1.</p>
	 */
	define('CURLOPT_HTTPGET', 80);

	/**
	 * <p>An <code>array</code> of HTTP header fields to set, in the format <code> array('Content-type: text/plain', 'Content-length: 100') </code> Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_HTTPHEADER', 10023);

	/**
	 * <p><b><code>true</code></b> to tunnel through a given HTTP proxy. Available as of cURL 7.3.0.</p>
	 */
	define('CURLOPT_HTTPPROXYTUNNEL', 61);

	/**
	 * <p>If set to <code>1</code>, ignore the <code>Content-Length</code> header in the HTTP response and ignore asking for or relying on it for FTP transfers. Defaults to <code>0</code>. Available as of cURL 7.14.1.</p>
	 */
	define('CURLOPT_IGNORE_CONTENT_LENGTH', 136);

	/**
	 * <p>Accepts a file handle <code>resource</code> to the file that the transfer should be read from when uploading. Available as of cURL 7.1.0 and deprecated as of cURL 7.9.7. Use <b><code>CURLOPT_READDATA</code></b> instead.</p>
	 */
	define('CURLOPT_INFILE', 10009);

	/**
	 * <p>The expected size, in bytes, of the file when uploading a file to a remote site. Note that using this option will not stop cURL from sending more data, as exactly what is sent depends on <b><code>CURLOPT_READFUNCTION</code></b>. This option accepts any value that can be cast to a valid <code>int</code>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_INFILESIZE', 14);

	/**
	 * <p>Set to a <code>string</code> with the name of the outgoing network interface to use. This can be an interface name, an IP address or a host name. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_INTERFACE', 10062);

	/**
	 * <p>Allows an application to select what kind of IP addresses to use when resolving host names. This is only interesting when using host names that resolve addresses using more than one version of IP. Set to one of the <b><code>CURL_IPRESOLVE_&#42;</code></b> constants. Defaults to <b><code>CURL_IPRESOLVE_WHATEVER</code></b>. Available as of cURL 7.10.8.</p>
	 */
	define('CURLOPT_IPRESOLVE', 113);

	/**
	 * <p>If set to a <code>string</code> naming a file holding a CA certificate in PEM format, an additional check against the peer certificate is performed to verify the issuer is indeed the one associated with the certificate provided by the option. For the result of the check to be considered a failure, this option should be used in combination with the <b><code>CURLOPT_SSL_VERIFYPEER</code></b> option. Available as of cURL 7.19.0.</p>
	 */
	define('CURLOPT_ISSUERCERT', 10170);

	/**
	 * <p>Pass a <code>string</code> with binary data of a CA SSL certificate in PEM format. If set, an additional check against the peer certificate is performed to verify the issuer is the one associated with the certificate provided by the option. Available as of PHP 8.1.0 and cURL 7.71.0.</p>
	 */
	define('CURLOPT_ISSUERCERT_BLOB', 40295);

	/**
	 * <p>Set to <b><code>true</code></b> to keep sending the request body if the HTTP code returned is equal to or larger than <code>300</code>. The default action would be to stop sending and close the stream or connection. Suitable for manual NTLM authentication. Most applications do not need this option. Available as of PHP 7.3.0 and cURL 7.51.0.</p>
	 */
	define('CURLOPT_KEEP_SENDING_ON_ERROR', 245);

	/**
	 * <p>Set to a <code>string</code> with the password required to use the <b><code>CURLOPT_SSLKEY</code></b> or <b><code>CURLOPT_SSH_PRIVATE_KEYFILE</code></b> private key. Setting this option to <b><code>null</code></b> disables using a password for these options. Available as of cURL 7.17.0.</p>
	 */
	define('CURLOPT_KEYPASSWD', 10026);

	/**
	 * <p>The KRB4 (Kerberos 4) security level. Any of the following <code>string</code> values (in order from least to most powerful) are valid: <code>clear</code>, <code>safe</code>, <code>confidential</code>, <code>private</code>. If the <code>string</code> does not match one of these, <code>private</code> is used. Setting this option to <b><code>null</code></b> will disable KRB4 security. Currently KRB4 security only works with FTP transactions. Available as of cURL 7.3.0 and deprecated as of cURL 7.17.0.</p>
	 */
	define('CURLOPT_KRB4LEVEL', 10063);

	/**
	 * <p>Set the kerberos security level for FTP and also enables kerberos awareness. This should be set to one of the following <code>string</code>s: <code>clear</code>, <code>safe</code>, <code>confidential</code>, <code>private</code>. If the <code>string</code> is set but does not match one of these, <code>private</code> is used. Setting this option to <b><code>null</code></b> will disable kerberos support for FTP. Defaults to <b><code>null</code></b>. Available as of cURL 7.16.4.</p>
	 */
	define('CURLOPT_KRBLEVEL', 10063);

	/**
	 * <p>Sets the local port number of the socket used for the connection. This option accepts any value that can be cast to a valid <code>int</code>. Defaults to <code>0</code>. Available as of cURL 7.15.2.</p>
	 */
	define('CURLOPT_LOCALPORT', 139);

	/**
	 * <p>The number of attempts cURL makes to find a working local port number, starting with the one set with <b><code>CURLOPT_LOCALPORT</code></b>. This option accepts any value that can be cast to a valid <code>int</code>. Defaults to <code>1</code>. Available as of cURL 7.15.2.</p>
	 */
	define('CURLOPT_LOCALPORTRANGE', 140);

	/**
	 * <p>Can be used to set protocol specific login options, such as the preferred authentication mechanism via <code>AUTH=NTLM</code> or <code>AUTH=&#42;</code>, and should be used in conjunction with the <b><code>CURLOPT_USERNAME</code></b> option. Available as of PHP 7.0.7 and cURL 7.34.0.</p>
	 */
	define('CURLOPT_LOGIN_OPTIONS', 10224);

	/**
	 * <p>The transfer speed, in bytes per second, that the transfer should be below during the count of <b><code>CURLOPT_LOW_SPEED_TIME</code></b> seconds before PHP considers the transfer too slow and aborts. This option accepts any value that can be cast to a valid <code>int</code>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_LOW_SPEED_LIMIT', 19);

	/**
	 * <p>The number of seconds the transfer speed should be below <b><code>CURLOPT_LOW_SPEED_LIMIT</code></b> before PHP considers the transfer too slow and aborts. This option accepts any value that can be cast to a valid <code>int</code>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_LOW_SPEED_TIME', 20);

	/**
	 * <p>Set a <code>string</code> with the authentication address (identity) of a submitted message that is being relayed to another server. The address should not be specified within a pair of angled brackets (<code>&gt;&lt;</code>). If an empty <code>string</code> is used then a pair of brackets are sent by cURL as required by RFC 2554. Available as of cURL 7.25.0.</p>
	 */
	define('CURLOPT_MAIL_AUTH', 10217);

	/**
	 * <p>Set a <code>string</code> with the sender's email address when sending SMTP mail. The email address should be specified with angled brackets (<code>&gt;&lt;</code>) around it, which if not specified are added automatically. If this parameter is not specified then an empty address is sent to the SMTP server which might cause the email to be rejected. Available as of cURL 7.20.0.</p>
	 */
	define('CURLOPT_MAIL_FROM', 10186);

	/**
	 * <p>Set to an <code>array</code> of <code>string</code>s with the recipients to pass to the server in an SMTP mail request. Each recipient should be specified within a pair of angled brackets (<code>&gt;&lt;</code>). If an angled bracket is not used as the first character, cURL assumes a single email address has been provided and encloses that address within brackets. Available as of cURL 7.20.0.</p>
	 */
	define('CURLOPT_MAIL_RCPT', 10187);

	/**
	 * <p>Set to <code>1</code> to allow <code>RCPT TO</code> command to fail for some recipients which makes cURL ignore errors for individual recipients and proceed with the remaining accepted recipients. If all recipients trigger failures and this flag is specified, cURL aborts the SMTP conversation and returns the error received from to the last <code>RCPT TO</code> command. Replaced by <b><code>CURLOPT_MAIL_RCPT_ALLOWFAILS</code></b> as of cURL 8.2.0. Available as of PHP 8.2.0 and cURL 7.69.0. Deprecated as of cURL 8.2.0.</p>
	 */
	define('CURLOPT_MAIL_RCPT_ALLLOWFAILS', 290);

	/**
	 * <p>If a download exceeds this speed (counted in bytes per second) on cumulative average during the transfer, the transfer will pause to keep the average rate less than or equal to the parameter value. Defaults to unlimited speed. This option accepts any value that can be cast to a valid <code>int</code>. Available as of cURL 7.15.5.</p>
	 */
	define('CURLOPT_MAX_RECV_SPEED_LARGE', 30146);

	/**
	 * <p>If an upload exceeds this speed (counted in bytes per second) on cumulative average during the transfer, the transfer will pause to keep the average rate less than or equal to the parameter value. Defaults to unlimited speed. This option accepts any value that can be cast to a valid <code>int</code>. Available as of cURL 7.15.5.</p>
	 */
	define('CURLOPT_MAX_SEND_SPEED_LARGE', 30145);

	/**
	 * <p>The maximum idle time allowed for an existing connection to be considered for reuse. Default maximum age is set to <code>118</code> seconds. This option accepts any value that can be cast to a valid <code>int</code>. Available as of PHP 8.2.0 and cURL 7.65.0.</p>
	 */
	define('CURLOPT_MAXAGE_CONN', 288);

	/**
	 * <p>The maximum amount of persistent connections that are allowed. When the limit is reached, the oldest one in the cache is closed to prevent increasing the number of open connections. This option accepts any value that can be cast to a valid <code>int</code>. Available as of cURL 7.7.0.</p>
	 */
	define('CURLOPT_MAXCONNECTS', 71);

	/**
	 * <p>Sets the maximum accepted size (in bytes) of a file to download. If the file requested is found larger than this value, the transfer is aborted and <b><code>CURLE_FILESIZE_EXCEEDED</code></b> is returned. Passing <code>0</code> disables this option, and passing a negative size returns a <b><code>CURLE_BAD_FUNCTION_ARGUMENT</code></b>. If the file size is not known prior to the start of download, this option has no effect. For setting a size limit above <code>2GB</code>, <b><code>CURLOPT_MAXFILESIZE_LARGE</code></b> should be used. As of cURL 8.4.0, this option also stops ongoing transfers if they reach this threshold. This option accepts any value that can be cast to a valid <code>int</code>. Defaults to <code>0</code>. Available as of cURL 7.10.8.</p>
	 */
	define('CURLOPT_MAXFILESIZE', 114);

	/**
	 * <p>The maximum file size in bytes allowed to download. If the file requested is found larger than this value, the transfer will not start and <b><code>CURLE_FILESIZE_EXCEEDED</code></b> will be returned. The file size is not always known prior to download, and for such files this option has no effect even if the file transfer ends up being larger than this given limit. This option accepts any value that can be cast to a valid <code>int</code>. Available as of PHP 8.2.0 and cURL 7.11.0.</p>
	 */
	define('CURLOPT_MAXFILESIZE_LARGE', 30117);

	/**
	 * <p>The maximum time in seconds, since the creation of the connection, that is allowed for an existing connection to have for it to be considered for reuse. If a connection is found in the cache that is older than this value, it will instead be closed once any in-progress transfers are complete. Default is <code>0</code> seconds, meaning the option is disabled and all connections are eligible for reuse. This option accepts any value that can be cast to a valid <code>int</code>. Available as of PHP 8.2.0 and cURL 7.80.0.</p>
	 */
	define('CURLOPT_MAXLIFETIME_CONN', 314);

	/**
	 * <p>The maximum amount of HTTP redirections to follow. Use this option alongside <b><code>CURLOPT_FOLLOWLOCATION</code></b>. Default value of <code>20</code> is set to prevent infinite redirects. Setting to <code>-1</code> allows inifinite redirects, and <code>0</code> refuses all redirects. Available as of cURL 7.5.0.</p>
	 */
	define('CURLOPT_MAXREDIRS', 68);

	/**
	 * <p>Set to a bitmask of <b><code>CURLMIMEOPT_&#42;</code></b> constants. Currently there is only one available option: <b><code>CURLMIMEOPT_FORMESCAPE</code></b>. Available as of PHP 8.3.0 and cURL 7.81.0.</p>
	 */
	define('CURLOPT_MIME_OPTIONS', 315);

	/**
	 * <p>Set to <b><code>true</code></b> to be completely silent with regards to the cURL functions. Use <b><code>CURLOPT_RETURNTRANSFER</code></b> instead. Available as of cURL 7.1.0, deprecated as of cURL 7.8.0 and last available in cURL 7.15.5. Removed as of PHP 7.3.0.</p>
	 */
	define('CURLOPT_MUTE', null);

	/**
	 * <p>Set to <b><code>true</code></b> to scan the ~/.netrc file to find a username and password for the remote site that a connection is being established with. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_NETRC', 51);

	/**
	 * <p>Set a <code>string</code> containing the full path name to a <code>.netrc</code> file. If this option is omitted and <b><code>CURLOPT_NETRC</code></b> is set, cURL checks for a <code>.netrc</code> file in the current user's home directory. Available as of cURL 7.11.0.</p>
	 */
	define('CURLOPT_NETRC_FILE', 10118);

	/**
	 * <p>Sets the value of the permissions (<code>int</code>) that is set on newly created directories on the remote server. The default value is <code>0755</code>. The only protocols that can use this are <code>sftp://</code>, <code>scp://</code> and <code>file://</code>. Available as of cURL 7.16.4.</p>
	 */
	define('CURLOPT_NEW_DIRECTORY_PERMS', 160);

	/**
	 * <p>Sets the value of the permissions (as an <code>int</code>) that are set on newly created files on the remote server. The default value is <code>0644</code>. The only protocols that can use this are <code>sftp://</code>, <code>scp://</code> and <code>file://</code>. Available as of cURL 7.16.4.</p>
	 */
	define('CURLOPT_NEW_FILE_PERMS', 159);

	/**
	 * <p>Set to <b><code>true</code></b> to exclude the body from the output. For HTTP(S), cURL makes a HEAD request. For most other protocols, cURL is not asking for the body data at all. Changing this to <b><code>false</code></b> will result in body data being included in the output. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_NOBODY', 44);

	/**
	 * <p>Set to <b><code>true</code></b> to disable the progress meter for cURL transfers.</p><p><b>Note</b>:</p><p>PHP automatically sets this option to <b><code>true</code></b>, this should only be changed for debugging purposes.</p>  Available as of cURL 7.1.0.
	 */
	define('CURLOPT_NOPROGRESS', 43);

	/**
	 * <p>Set a <code>string</code> consisting of a comma separated list of hostnames that do not require a proxy to get reached. Each name in this list is matched as either a domain which contains the hostname or the hostname itself. The only wildcard available in the <code>string</code> is a single <code>&#42;</code> character which matches all hosts, effectively disabling the proxy. Setting this option to an empty <code>string</code> enables the proxy for all hostnames. Since cURL 7.86.0, IP addresses set with this option can be provided using CIDR notation. Available as of cURL 7.19.4.</p>
	 */
	define('CURLOPT_NOPROXY', 10177);

	/**
	 * <p><b><code>true</code></b> to ignore any cURL function that causes a signal to be sent to the PHP process. This is turned on by default in multi-threaded SAPIs so timeout options can still be used. Available as of cURL 7.10.</p>
	 */
	define('CURLOPT_NOSIGNAL', 99);

	/**
	 * <p>A <code>callable</code> with the following signature:</p> callback(<code>resource</code> <code>$curlHandle</code>, <code>string</code> <code>$passwordPrompt</code>, <code>int</code> <code>$maximumPasswordLength</code>): <code>string</code>  <code>curlHandle</code>   The cURL handle.   <code>passwordPrompt</code>   A password prompt.   <code>maximumPasswordLength</code>   The maximum length of the password.    The callback should return a <code>string</code> containing the password. Available as of cURL 7.4.2, deprecated as of cURL 7.11.1 and last available in cURL 7.15.5. Removed as of PHP 7.3.0.
	 */
	define('CURLOPT_PASSWDFUNCTION', null);

	/**
	 * <p>Set to a <code>string</code> with the password to use in authentication. Available as of cURL 7.19.1.</p>
	 */
	define('CURLOPT_PASSWORD', 10174);

	/**
	 * <p>Set to <b><code>true</code></b> for cURL not alter URL paths before passing them on to the server. Defaults to <b><code>false</code></b>, which squashes sequences of <code>/../</code> or <code>/./</code> that may exist in the URL's path part which is supposed to be removed according to RFC 3986 section 5.2.4. Available as of PHP 7.0.7 and cURL 7.42.0.</p>
	 */
	define('CURLOPT_PATH_AS_IS', 234);

	/**
	 * <p>Set a <code>string</code> with the pinned public key. The <code>string</code> can be the file name of the pinned public key in a PEM or DER file format. The <code>string</code> can also be any number of base64 encoded sha256 hashes preceded by <code>sha256//</code> and separated by <code>;</code>. Available as of PHP 7.0.7 and cURL 7.39.0.</p>
	 */
	define('CURLOPT_PINNEDPUBLICKEY', 10230);

	/**
	 * <p>Set to <b><code>true</code></b> to wait for an existing connection to confirm whether it can do multiplexing and use it if it can before creating and using a new connection. Available as of PHP 7.0.7 and cURL 7.43.0.</p>
	 */
	define('CURLOPT_PIPEWAIT', 237);

	/**
	 * <p>An <code>int</code> with an alternative port number to connect to instead of the one specified in the URL or the default port for the used protocol. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_PORT', 3);

	/**
	 * <p>Set to <b><code>true</code></b> to do a HTTP <code>POST</code> request. This request uses the <code>application/x-www-form-urlencoded</code> header. Defaults to <b><code>false</code></b>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_POST', 47);

	/**
	 * <p>The full data to post in a HTTP <code>POST</code> operation. This parameter can either be passed as a urlencoded <code>string</code> like '<code>para1=val1&amp;para2=val2&amp;...</code>' or as an <code>array</code> with the field name as key and field data as value. If <code>value</code> is an <code>array</code>, the <code>Content-Type</code> header will be set to <code>multipart/form-data</code>. Files can be sent using <code>CURLFile</code> or <code>CURLStringFile</code>, in which case <code>value</code> must be an <code>array</code>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_POSTFIELDS', 10015);

	/**
	 * <p>An <code>array</code> of FTP command <code>string</code>s to execute on the server after the FTP request has been performed. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_POSTQUOTE', 10039);

	/**
	 * <p>Set to a bitmask of <b><code>CURL_REDIR_POST_301</code></b>, <b><code>CURL_REDIR_POST_302</code></b> and <b><code>CURL_REDIR_POST_303</code></b> if the HTTP <code>POST</code> method should be maintained when <b><code>CURLOPT_FOLLOWLOCATION</code></b> is set and a specific type of redirect occurs. Available as of cURL 7.19.1.</p>
	 */
	define('CURLOPT_POSTREDIR', 161);

	/**
	 * <p>Set a <code>string</code> holding the host name or dotted numerical IP address to be used as the preproxy that cURL connects to before it connects to the HTTP(S) proxy specified in the <b><code>CURLOPT_PROXY</code></b> option for the upcoming request. The preproxy can only be a SOCKS proxy and it should be prefixed with <code>[scheme]://</code> to specify which kind of socks is used. A numerical IPv6 address must be written within [brackets]. Setting the preproxy to an empty <code>string</code> explicitly disables the use of a preproxy. To specify port number in this <code>string</code>, append <code>:[port]</code> to the end of the host name. The proxy's port number may optionally be specified with the separate option <b><code>CURLOPT_PROXYPORT</code></b>. Defaults to using port 1080 for proxies if a port is not specified. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PRE_PROXY', 10262);

	/**
	 * <p>Set an <code>array</code> of FTP command <code>string</code>s to pass to the server after the transfer type is set. These commands are not performed when a directory listing is performed, only for file transfers. Available as of cURL 7.9.5.</p>
	 */
	define('CURLOPT_PREQUOTE', 10093);

	/**
	 * <p>A <code>callable</code> with the following signature that gets called after the connection is established, but before the request payload (for example, the GET/POST/DELETE request of an HTTP connection) is sent, and can be used to abort or allow the connection depending on the source and destination IP address and port numbers:</p> callback(<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>CurlHandle</code> <code>$curlHandle</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>string</code> <code>$destination_ip</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>string</code> <code>$local_ip</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$destination_port</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$local_port</code><br>): <code>int</code>  <code>curlHandle</code>   The cURL handle.   <code>destination_ip</code>   The primary IP of the remote server established with this connection. For FTP, this is the IP for the control connection. IPv6 addresses are represented without surrounding brackets.   <code>local_ip</code>   The originating IP for this connection. IPv6 addresses are represented without surrounding brackets.   <code>destination_port</code>   The primary port number on the remote server established with this connection. For FTP, this is the port for the control connection. This can be a TCP or a UDP port number depending on the protocol.   <code>local_port</code>   The originating port number for this connection. This can be a TCP or a UDP port number depending on the protocol.    Return <b><code>CURL_PREREQFUNC_OK</code></b> to allow the request, or <b><code>CURL_PREREQFUNC_ABORT</code></b> to abort the transfer. Available as of PHP 8.4.0 and cURL 7.80.0.
	 */
	define('CURLOPT_PREREQFUNCTION', 20312);

	/**
	 * <p>Any data that should be associated with this cURL handle. This data can subsequently be retrieved with the <b><code>CURLINFO_PRIVATE</code></b> option of <code>curl_getinfo()</code>. cURL does nothing with this data. When using a cURL multi handle, this private data is typically a unique key to identify a standard cURL handle. Available as of cURL 7.10.3.</p>
	 */
	define('CURLOPT_PRIVATE', 10103);

	/**
	 * <p>A <code>callable</code> with the following signature:</p> callback(<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>resource</code> <code>$curlHandle</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$bytesToDownload</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$bytesDownloaded</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$bytesToUpload</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$bytesUploaded</code><br>): <code>int</code>  <code>curlHandle</code>   The cURL handle.   <code>bytesToDownload</code>   The total number of bytes expected to be downloaded in this transfer.   <code>bytesDownloaded</code>   The number of bytes downloaded so far.   <code>bytesToUpload</code>   The total number of bytes expected to be uploaded in this transfer.   <code>bytesUploaded</code>   The number of bytes uploaded so far.    The callback should return an <code>int</code> with a non-zero value to abort the transfer and set a <b><code>CURLE_ABORTED_BY_CALLBACK</code></b> error. <p><b>Note</b>:</p><p>The callback is only called when the <b><code>CURLOPT_NOPROGRESS</code></b> option is set to <b><code>false</code></b>.</p>  Available as of cURL 7.1.0 and deprecated as of cURL 7.32.0. Use <b><code>CURLOPT_XFERINFOFUNCTION</code></b> instead.
	 */
	define('CURLOPT_PROGRESSFUNCTION', 20056);

	/**
	 * <p>Bitmask of <b><code>CURLPROTO_&#42;</code></b> values. If used, this bitmask limits what protocols cURL may use in the transfer. Defaults to <b><code>CURLPROTO_ALL</code></b>, ie. cURL will accept all protocols it supports. See also <b><code>CURLOPT_REDIR_PROTOCOLS</code></b>. Available as of cURL 7.19.4 and deprecated as of cURL 7.85.0.</p>
	 */
	define('CURLOPT_PROTOCOLS', 181);

	/**
	 * <p>Set to a <code>string</code> with a comma separated list of case insensitive protocol names (URL schemes) to allow in the transfer. Set to <code>ALL</code> to enable all protocols. By default, cURL accepts all protocols it was built with support for. Available protocols are: <code>DICT</code>, <code>FILE</code>, <code>FTP</code>, <code>FTPS</code>, <code>GOPHER</code>, <code>GOPHERS</code>, <code>HTTP</code>, <code>HTTPS</code>, <code>IMAP</code>, <code>IMAPS</code>, <code>LDAP</code>, <code>LDAPS</code>, <code>MQTT</code>, <code>POP3</code>, <code>POP3S</code>, <code>RTMP</code>, <code>RTMPE</code>, <code>RTMPS</code>, <code>RTMPT</code>, <code>RTMPTE</code>, <code>RTMPTS</code>, <code>RTSP</code>, <code>SCP</code>, <code>SFTP</code>, <code>SMB</code>, <code>SMBS</code>, <code>SMTP</code>, <code>SMTPS</code>, <code>TELNET</code>, <code>TFTP</code>, <code>WS</code>, <code>WSS</code>. Available as of PHP 8.3.0 and cURL 7.85.0.</p>
	 */
	define('CURLOPT_PROTOCOLS_STR', null);

	/**
	 * <p>A <code>string</code> with the HTTP proxy to tunnel requests through. This should be the hostname, the dotted numerical IP address or a numerical IPv6 address written within [brackets]. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_PROXY', 10004);

	/**
	 * <p>The path to proxy Certificate Authority (CA) bundle. Set the path as a <code>string</code> naming a file holding one or more certificates to verify the HTTPS proxy with. This option is for connecting to an HTTPS proxy, not an HTTPS server. Defaults set to the system path where cURL's cacert bundle is assumed to be stored. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_CAINFO', 10246);

	/**
	 * <p>A <code>string</code> with the name of a PEM file holding one or more certificates to verify the HTTPS proxy with. This option is for connecting to an HTTPS proxy, not an HTTPS server. Defaults set to the system path where cURL's cacert bundle is assumed to be stored. Available as of PHP 8.2.0 and cURL 7.77.0.</p>
	 */
	define('CURLOPT_PROXY_CAINFO_BLOB', 40310);

	/**
	 * <p>A <code>string</code> with the directory holding multiple CA certificates to verify the HTTPS proxy with. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_CAPATH', 10247);

	/**
	 * <p>Set to a <code>string</code> with the file name with the concatenation of CRL (Certificate Revocation List) in PEM format to use in the certificate validation that occurs during the SSL exchange. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_CRLFILE', 10260);

	/**
	 * <p>Proxy issuer SSL certificate filename <code>string</code>. Available as of PHP 8.1.0 and cURL 7.71.0.</p>
	 */
	define('CURLOPT_PROXY_ISSUERCERT', 10296);

	/**
	 * <p>A <code>string</code> with the proxy issuer SSL certificate. Available as of PHP 8.1.0 and cURL 7.71.0.</p>
	 */
	define('CURLOPT_PROXY_ISSUERCERT_BLOB', 40297);

	/**
	 * <p>Set the <code>string</code> be used as the password required to use the <b><code>CURLOPT_PROXY_SSLKEY</code></b> private key. A passphrase is not needed to load a certificate but one is needed to load a private key. This option is for connecting to an HTTPS proxy, not an HTTPS server. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_KEYPASSWD', 10258);

	/**
	 * <p>Set the pinned public key for HTTPS proxy. The <code>string</code> can be the file name of the pinned public key which is expected to be in a <code>PEM</code> or <code>DER</code> file format. The <code>string</code> can also be any number of base64 encoded sha256 hashes preceded by <code>sha256//</code> and separated by <code>;</code>. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_PINNEDPUBLICKEY', 10263);

	/**
	 * <p>A <code>string</code> with the proxy authentication service name. Available as of PHP 7.0.7, cURL 7.43.0 (for HTTP proxies) and cURL 7.49.0 (for SOCKS5 proxies).</p>
	 */
	define('CURLOPT_PROXY_SERVICE_NAME', 10235);

	/**
	 * <p>A <code>string</code> with a colon-separated list of ciphers to use for the connection to the HTTPS proxy. When used with OpenSSL commas and spaces are also acceptable as separators, and <code>!</code>, <code>-</code> and <code>+</code> can be used as operators. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_SSL_CIPHER_LIST', 10259);

	/**
	 * <p>Set proxy SSL behavior options, which is a bitmask of the <b><code>CURLSSLOPT_&#42;</code></b> constants. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_SSL_OPTIONS', 261);

	/**
	 * <p>Set to <code>2</code> to verify in the HTTPS proxy's certificate name fields against the proxy name. When set to <code>0</code> the connection succeeds regardless of the names used in the certificate. Use that ability with caution! Set to <code>1</code> in cURL 7.28.0 and earlier as a debug option. Set to <code>1</code> in cURL 7.28.1 to 7.65.3 <b><code>CURLE_BAD_FUNCTION_ARGUMENT</code></b> is returned. As of cURL 7.66.0 <code>1</code> and <code>2</code> is treated as the same value. Defaults to <code>2</code>. In production environments the value of this option should be kept at <code>2</code>. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_SSL_VERIFYHOST', 249);

	/**
	 * <p>Set to <b><code>false</code></b> to stop cURL from verifying the peer's certificate. Alternate certificates to verify against can be specified with the <b><code>CURLOPT_CAINFO</code></b> option or a certificate directory can be specified with the <b><code>CURLOPT_CAPATH</code></b> option. When set to <b><code>false</code></b>, the peer certificate verification succeeds regardless. <b><code>true</code></b> by default. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_SSL_VERIFYPEER', 248);

	/**
	 * <p>A <code>string</code> with the file name of the client certificate used to connect to the HTTPS proxy. The default format is <code>P12</code> on Secure Transport and <code>PEM</code> on other engines, and can be changed with <b><code>CURLOPT_PROXY_SSLCERTTYPE</code></b>. With NSS or Secure Transport, this can also be the nickname of the certificate used for authentication as it is named in the security database. If a file from the current directory is to be used, it must be prefixed with <code>./</code> in order to avoid confusion with a nickname. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_SSLCERT', 10254);

	/**
	 * <p>A <code>string</code> with the SSL proxy client certificate. Available as of PHP 8.1.0 and cURL 7.71.0.</p>
	 */
	define('CURLOPT_PROXY_SSLCERT_BLOB', 40293);

	/**
	 * <p>A <code>string</code> with the format of the client certificate used when connecting to an HTTPS proxy. Supported formats are <code>PEM</code> and <code>DER</code>, except with Secure Transport. OpenSSL (versions 0.9.3 and later) and Secure Transport (on iOS 5 or later, or OS X 10.7 or later) also support <code>P12</code> for PKCS#12-encoded files. Defaults to <code>PEM</code>. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_SSLCERTTYPE', 10255);

	/**
	 * <p>A <code>string</code> with the file name of the private key used for connecting to the HTTPS proxy. The default format is <code>PEM</code> and can be changed with <b><code>CURLOPT_PROXY_SSLKEYTYPE</code></b>. (iOS and Mac OS X only) This option is ignored if cURL was built against Secure Transport. Available if built with TLS enabled. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_SSLKEY', 10256);

	/**
	 * <p>A <code>string</code> with the private key for connecting to the HTTPS proxy. Available as of PHP 8.1.0 and cURL 7.71.0.</p>
	 */
	define('CURLOPT_PROXY_SSLKEY_BLOB', 40294);

	/**
	 * <p>A <code>string</code> with the format of the private key. Supported formats are: <code>PEM</code>, <code>DER</code>, <code>ENG</code>. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_SSLKEYTYPE', 10257);

	/**
	 * <p>Set the preferred HTTPS proxy TLS version to one of the <b><code>CURL_SSLVERSION_&#42;</code></b> constants. Defaults to <b><code>CURL_SSLVERSION_DEFAULT</code></b>.</p><p><b>Warning</b></p> <p>It is better to not set this option and leave the default <b><code>CURL_SSLVERSION_DEFAULT</code></b> which will attempt to figure out the remote SSL protocol version.</p>  Available as of PHP 7.3.0 and cURL 7.52.0.
	 */
	define('CURLOPT_PROXY_SSLVERSION', 250);

	/**
	 * <p>A <code>string</code> with a colon-separated list of ciphers to use for the connection to the TLS 1.3 connection to a proxy. This option is currently used only when cURL is built to use OpenSSL 1.1.1 or later. When using a different SSL backend the TLS 1.3 cipher suites can be set with the <b><code>CURLOPT_PROXY_SSL_CIPHER_LIST</code></b> option. Available as of PHP 7.3.0 and cURL 7.61.0.</p>
	 */
	define('CURLOPT_PROXY_TLS13_CIPHERS', 10277);

	/**
	 * <p>A <code>string</code> with the password to use for the TLS authentication method specified with the <b><code>CURLOPT_PROXY_TLSAUTH_TYPE</code></b> option. Requires that the <b><code>CURLOPT_PROXY_TLSAUTH_USERNAME</code></b> option to also be set. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_TLSAUTH_PASSWORD', 10252);

	/**
	 * <p>The method of the TLS authentication used for the HTTPS connection. Supported method is <code>SRP</code>.</p><p><b>Note</b>:</p><p>Secure Remote Password (SRP) authentication for TLS provides mutual authentication if both sides have a shared secret. To use TLS-SRP, the <b><code>CURLOPT_PROXY_TLSAUTH_USERNAME</code></b> and <b><code>CURLOPT_PROXY_TLSAUTH_PASSWORD</code></b> options must also be set.</p>  Available as of PHP 7.3.0 and cURL 7.52.0.
	 */
	define('CURLOPT_PROXY_TLSAUTH_TYPE', 10253);

	/**
	 * <p>The username to use for the HTTPS proxy TLS authentication method specified with the <b><code>CURLOPT_PROXY_TLSAUTH_TYPE</code></b> option. Requires that the <b><code>CURLOPT_PROXY_TLSAUTH_PASSWORD</code></b> option to also be set. Available as of PHP 7.3.0 and cURL 7.52.0.</p>
	 */
	define('CURLOPT_PROXY_TLSAUTH_USERNAME', 10251);

	/**
	 * <p>Set to <code>1</code> to set the transfer mode (binary or ASCII) for FTP transfers done via an HTTP proxy, by appending <code>type=a</code> or <code>type=i</code> to the URL. Without this setting or it being set to <code>0</code>, <b><code>CURLOPT_TRANSFERTEXT</code></b> has no effect when doing FTP via a proxy. Defaults to <code>0</code>. Available as of cURL 7.18.0.</p>
	 */
	define('CURLOPT_PROXY_TRANSFER_MODE', 166);

	/**
	 * <p>A bitmask of the HTTP authentication method(s) (<b><code>CURLAUTH_&#42;</code></b> constants) to use for the proxy connection. For proxy authentication, only <b><code>CURLAUTH_BASIC</code></b> and <b><code>CURLAUTH_NTLM</code></b> are currently supported. Defaults to <b><code>CURLAUTH_BASIC</code></b>. Available as of cURL 7.10.7.</p>
	 */
	define('CURLOPT_PROXYAUTH', 111);

	/**
	 * <p>An <code>array</code> of custom HTTP header <code>string</code>s to pass to proxies. Available as of PHP 7.0.7 and cURL 7.37.0.</p>
	 */
	define('CURLOPT_PROXYHEADER', 10228);

	/**
	 * <p>Set a <code>string</code> with the password to be used for authentication with the proxy. Available as of cURL 7.19.1.</p>
	 */
	define('CURLOPT_PROXYPASSWORD', 10176);

	/**
	 * <p>An <code>int</code> with the port number of the proxy to connect to. This port number can also be set in <b><code>CURLOPT_PROXY</code></b>. Setting this to zero makes cURL use the default proxy port number or the port number specified in the proxy URL <code>string</code>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_PROXYPORT', 59);

	/**
	 * <p>Sets the type of the proxy to one of the <b><code>CURLPROXY_&#42;</code></b> constants. Defaults to <b><code>CURLPROXY_HTTP</code></b>. Available as of cURL 7.10.</p>
	 */
	define('CURLOPT_PROXYTYPE', 101);

	/**
	 * <p>Set a <code>string</code> with the username to be used for authentication with the proxy. Available as of cURL 7.19.1.</p>
	 */
	define('CURLOPT_PROXYUSERNAME', 10175);

	/**
	 * <p>A <code>string</code> with a username and password formatted as <code>[username]:[password]</code> to use for the connection to the proxy. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_PROXYUSERPWD', 10006);

	/**
	 * <p><b><code>true</code></b> to HTTP PUT a file. The file to PUT must be set with <b><code>CURLOPT_READDATA</code></b> and <b><code>CURLOPT_INFILESIZE</code></b>. Available as of cURL 7.1.0 and deprecated as of cURL 7.12.1.</p>
	 */
	define('CURLOPT_PUT', 54);

	/**
	 * <p>Set to <b><code>true</code></b> for cURL to skip cleanup of resources when recovering from a timeout. This allows for a swift termination of the cURL process at the expense of a possible leak of associated resources. Available as of PHP 8.3.0 and cURL 7.87.0.</p>
	 */
	define('CURLOPT_QUICK_EXIT', null);

	/**
	 * <p>An <code>array</code> of FTP command <code>string</code>s to execute on the server prior to the FTP request. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_QUOTE', 10028);

	/**
	 * <p>A <code>string</code> with a filename to be used to seed the random number generator for SSL. Available as of cURL 7.7.0 and deprecated as of cURL 7.84.0.</p>
	 */
	define('CURLOPT_RANDOM_FILE', 10076);

	/**
	 * <p>A <code>string</code> with the range(s) of data to retrieve in the format <code>X-Y</code> where X or Y are optional. HTTP transfers also support several intervals, separated with commas in the format <code>X-Y,N-M</code>. Set to <b><code>null</code></b> to disable requesting a byte range. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_RANGE', 10007);

	/**
	 * <p>Sets a file pointer <code>resource</code> that will be used by the file read function set with <b><code>CURLOPT_READFUNCTION</code></b>. Available as of cURL 7.9.7.</p>
	 */
	define('CURLOPT_READDATA', 10009);

	/**
	 * <p>A <code>callable</code> with the following signature:</p> callback(<code>resource</code> <code>$curlHandle</code>, <code>resource</code> <code>$streamResource</code>, <code>int</code> <code>$maxAmountOfDataToRead</code>): <code>string</code>  <code>curlHandle</code>   The cURL handle.   <code>streamResource</code>   Stream <code>resource</code> provided to cURL through the option <b><code>CURLOPT_READDATA</code></b>.   <code>maxAmountOfDataToRead</code>   The maximum amount of data to be read.    The callback should return a <code>string</code> with a length equal or smaller than the amount of data requested, typically by reading it from the passed stream <code>resource</code>. It should return an empty <code>string</code> to signal <code>EOF</code>. Available as of cURL 7.1.0.
	 */
	define('CURLOPT_READFUNCTION', 20012);

	/**
	 * <p>Bitmask of <b><code>CURLPROTO_&#42;</code></b> values which limit what protocols cURL may use in a transfer that it follows to in a redirect when <b><code>CURLOPT_FOLLOWLOCATION</code></b> is enabled. This allows limiting specific transfers to only be allowed to use a subset of protocols in redirections. As of cURL 7.19.4, by default cURL will allow all protocols except for <code>FILE</code> and <code>SCP</code>. Prior to cURL 7.19.4, cURL would unconditionally follow to all supported protocols. See also <b><code>CURLOPT_PROTOCOLS</code></b> for protocol constant values. Available as of cURL 7.19.4 and deprecated as of cURL 7.85.0.</p>
	 */
	define('CURLOPT_REDIR_PROTOCOLS', 182);

	/**
	 * <p>Set to a <code>string</code> with a comma separated list of case insensitive protocol names (URL schemes) to allow to follow to in a redirect when <b><code>CURLOPT_FOLLOWLOCATION</code></b> is enabled. Set to <code>ALL</code> to enable all protocols. As of cURL 7.65.2 it defaults to <code>FTP</code>, <code>FTPS</code>, <code>HTTP</code> and <code>HTTPS</code>. From cURL 7.40.0 to 7.65.1, this defaults to all protocols except <code>FILE</code>, <code>SCP</code>, <code>SMB</code> and <code>SMBS</code>. Prior to cURL 7.40.0, this defaults to all protocols except <code>FILE</code> and <code>SCP</code>. Available protocols are: <code>DICT</code>, <code>FILE</code>, <code>FTP</code>, <code>FTPS</code>, <code>GOPHER</code>, <code>GOPHERS</code>, <code>HTTP</code>, <code>HTTPS</code>, <code>IMAP</code>, <code>IMAPS</code>, <code>LDAP</code>, <code>LDAPS</code>, <code>MQTT</code>, <code>POP3</code>, <code>POP3S</code>, <code>RTMP</code>, <code>RTMPE</code>, <code>RTMPS</code>, <code>RTMPT</code>, <code>RTMPTE</code>, <code>RTMPTS</code>, <code>RTSP</code>, <code>SCP</code>, <code>SFTP</code>, <code>SMB</code>, <code>SMBS</code>, <code>SMTP</code>, <code>SMTPS</code>, <code>TELNET</code>, <code>TFTP</code>, <code>WS</code>, <code>WSS</code>. Available as of PHP 8.3.0 and cURL 7.85.0.</p>
	 */
	define('CURLOPT_REDIR_PROTOCOLS_STR', null);

	/**
	 * <p>A <code>string</code> with the contents of the <code>Referer: </code> header to be used in a HTTP request. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_REFERER', 10016);

	/**
	 * <p>A <code>string</code> to use in the upcoming request instead of the path as extracted from the URL. Available as of PHP 7.3.0 and cURL 7.55.0.</p>
	 */
	define('CURLOPT_REQUEST_TARGET', 10266);

	/**
	 * <p>Provide an <code>array</code> of colon-separated <code>string</code>s with custom addresses for specific host and port pairs in the following format: <code> array( "example.com:80:127.0.0.1", "example2.com:443:127.0.0.2", ) </code> Available as of cURL 7.21.3.</p>
	 */
	define('CURLOPT_RESOLVE', 10203);

	/**
	 * <p>The offset, in bytes, to resume a transfer from. This option accepts any value that can be cast to a valid <code>int</code>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_RESUME_FROM', 21);

	/**
	 * <p><b><code>true</code></b> to return the transfer as a <code>string</code> of the return value of <code>curl_exec()</code> instead of outputting it directly.</p>
	 */
	define('CURLOPT_RETURNTRANSFER', 19913);

	/**
	 * <p>Set an <code>int</code> with the CSEQ number to issue for the next RTSP request. Useful if the application is resuming a previously broken connection. The CSEQ increments from this new number henceforth. Defaults to <code>0</code>. Available as of cURL 7.20.0.</p>
	 */
	define('CURLOPT_RTSP_CLIENT_CSEQ', 193);

	/**
	 * <p>Sets the kind of RTSP request to make. Must be one of the <b><code>CURL_RTSPREQ_&#42;</code></b> constants. Available as of cURL 7.20.0.</p>
	 */
	define('CURLOPT_RTSP_REQUEST', 189);

	/**
	 * <p>Set an <code>int</code> with the CSEQ number to expect for the next RTSP Server to Client request. This feature (listening for Server requests) is unimplemented. Defaults to <code>0</code>. Available as of cURL 7.20.0.</p>
	 */
	define('CURLOPT_RTSP_SERVER_CSEQ', 194);

	/**
	 * <p>Set a <code>string</code> with the value of the current RTSP Session ID for the handle. Once this value is set to any non-<b><code>null</code></b> value, cURL returns <b><code>CURLE_RTSP_SESSION_ERROR</code></b> if the ID received from the server does not match. If set to <b><code>null</code></b>, cURL automatically sets the ID the first time the server sets it in a response. Defaults to <b><code>null</code></b> Available as of cURL 7.20.0.</p>
	 */
	define('CURLOPT_RTSP_SESSION_ID', 10190);

	/**
	 * <p>Sets a <code>string</code> with the stream URI to operate on. If not set, cURL defaults to operating on generic server options by passing <code>&#42;</code> in the place of the RTSP Stream URI. When working with RTSP, <code>CURLOPT_RTSP_STREAM_URI</code> indicates what URL to send to the server in the request header while the <code>CURLOPT_URL</code> indicates where to make the connection to. Available as of cURL 7.20.0.</p>
	 */
	define('CURLOPT_RTSP_STREAM_URI', 10191);

	/**
	 * <p>Set the <code>Transport:</code> header for this RTSP session. Available as of cURL 7.20.0.</p>
	 */
	define('CURLOPT_RTSP_TRANSPORT', 10192);

	/**
	 * <p>Always <b><code>true</code></b>, what disables support for the <code>@</code> prefix for uploading files in <b><code>CURLOPT_POSTFIELDS</code></b>, which means that values starting with <code>@</code> can be safely passed as fields. <code>CURLFile</code> may be used for uploads instead.</p>
	 */
	define('CURLOPT_SAFE_UPLOAD', -1);

	/**
	 * <p>The authorization identity (authzid) <code>string</code> for the transfer. Only applicable to the PLAIN SASL authentication mechanism where it is optional. When not specified, only the authentication identity (authcid) as specified by the username will be sent to the server, along with the password. The server will derive the authzid from the authcid when not provided, which it will then use internally. Available as of PHP 8.2.0 and cURL 7.66.0.</p>
	 */
	define('CURLOPT_SASL_AUTHZID', 10289);

	/**
	 * <p><b><code>true</code></b> to enable sending the initial response in the first packet. Available as of PHP 7.0.7 and cURL 7.31.0.</p>
	 */
	define('CURLOPT_SASL_IR', 218);

	/**
	 * <p>A timeout in seconds cURL will wait for a response from an FTP, SFTP, IMAP, SCP, SMTP, or a POP3 server. This option replaces the existing <b><code>CURLOPT_FTP_RESPONSE_TIMEOUT</code></b> option which is deprecated in cURL 7.85.0. Available as of PHP 8.4.0.</p>
	 */
	define('CURLOPT_SERVER_RESPONSE_TIMEOUT', 112);

	/**
	 * <p>A <code>string</code> with the authentication service name. Available as of PHP 7.0.7 and cURL 7.43.0.</p>
	 */
	define('CURLOPT_SERVICE_NAME', 10236);

	/**
	 * <p>A result of <code>curl_share_init()</code>. Makes the cURL handle to use the data from the shared handle. Available as of cURL 7.10.</p>
	 */
	define('CURLOPT_SHARE', 10100);

	/**
	 * <p>The SOCKS5 authentication method(s) to use. The options are: <b><code>CURLAUTH_BASIC</code></b>, <b><code>CURLAUTH_GSSAPI</code></b>, <b><code>CURLAUTH_NONE</code></b>. When more than one method is set, cURL will poll the server to see what methods it supports and pick the best one. Defaults to <code>CURLAUTH_BASIC|CURLAUTH_GSSAPI</code>. Set the actual username and password with the <b><code>CURLOPT_PROXYUSERPWD</code></b> option. Available as of PHP 7.3.0 and cURL 7.55.0.</p>
	 */
	define('CURLOPT_SOCKS5_AUTH', 267);

	/**
	 * <p>Set to <code>1</code> to enable and <code>0</code> to disable the unprotected exchange of the protection mode negotiation as part of the GSSAPI negotiation. Available as of cURL 7.19.4.</p>
	 */
	define('CURLOPT_SOCKS5_GSSAPI_NEC', 180);

	/**
	 * <p>Set a <code>string</code> holding the name of the SOCKS5 service. Defaults to <code>rcmd</code>. Available as of cURL 7.19.4 and deprecated as of cURL 7.49.0. Use <b><code>CURLOPT_PROXY_SERVICE_NAME</code></b> instead.</p>
	 */
	define('CURLOPT_SOCKS5_GSSAPI_SERVICE', 10179);

	/**
	 * <p>A bitmask consisting of one or more of the following constants: <b><code>CURLSSH_AUTH_PUBLICKEY</code></b>, <b><code>CURLSSH_AUTH_PASSWORD</code></b>, <b><code>CURLSSH_AUTH_HOST</code></b>, <b><code>CURLSSH_AUTH_KEYBOARD</code></b>, <b><code>CURLSSH_AUTH_AGENT</code></b>, <b><code>CURLSSH_AUTH_ANY</code></b>. Defaults to <b><code>CURLSSH_AUTH_ANY</code></b>. Available as of cURL 7.16.1.</p>
	 */
	define('CURLOPT_SSH_AUTH_TYPES', 151);

	/**
	 * <p><b><code>true</code></b> to enable, <b><code>false</code></b> to disable built-in SSH compression. Note that the server can disregard this request. Defaults to <b><code>false</code></b>. Available as of PHP 7.3.0 and cURL 7.56.0.</p>
	 */
	define('CURLOPT_SSH_COMPRESSION', 268);

	/**
	 * <p>A <code>string</code> containing 32 hexadecimal digits which should contain the MD5 checksum of the remote host's public key, and cURL will reject the connection to the host unless the md5sums match. This option is only for SCP and SFTP transfers. Available as of cURL 7.17.1.</p>
	 */
	define('CURLOPT_SSH_HOST_PUBLIC_KEY_MD5', 10162);

	/**
	 * <p>A <code>string</code> with the base64-encoded SHA256 hash of the remote host's public key. The transfer will fail if the given hash does not match the hash the remote host provides. Available as of PHP 8.2.0 and cURL 7.80.0.</p>
	 */
	define('CURLOPT_SSH_HOST_PUBLIC_KEY_SHA256', 10311);

	/**
	 * <p>A <code>callable</code> that will be called when SSH host key verification is needed. The callback must have the following signature:</p> callback(<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>resource</code> <code>$curlHandle</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$keyType</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>string</code> <code>$key</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$keyLength</code><br>): <code>int</code>  <code>curlHandle</code>   The cURL handle.   <code>keyType</code>   One of the <b><code>CURLKHTYPE_&#42;</code></b> key types.   <code>key</code>   The key to check.   <code>keyLength</code>   The length of the key in bytes.    This callback overrides <b><code>CURLOPT_SSH_KNOWNHOSTS</code></b>. Available as of PHP 8.3.0 and cURL 7.84.0.
	 */
	define('CURLOPT_SSH_HOSTKEYFUNCTION', null);

	/**
	 * <p>Set to the filename of the known_host file to use which should use the OpenSSH file format as supported by libssh2. Available as of cURL 7.19.6.</p>
	 */
	define('CURLOPT_SSH_KNOWNHOSTS', 10183);

	/**
	 * <p>The file name for a private key. If not used, cURL defaults to $HOME/.ssh/id_dsa if the HOME environment variable is set, and just <code>id_dsa</code> in the current directory if HOME is not set. If the file is password-protected, set the password with <b><code>CURLOPT_KEYPASSWD</code></b>. Available as of cURL 7.16.1.</p>
	 */
	define('CURLOPT_SSH_PRIVATE_KEYFILE', 10153);

	/**
	 * <p>The file name for a public key. If not used, cURL defaults to $HOME/.ssh/id_dsa.pub if the HOME environment variable is set, and just <code>id_dsa.pub</code> in the current directory if HOME is not set. Available as of cURL 7.16.1.</p>
	 */
	define('CURLOPT_SSH_PUBLIC_KEYFILE', 10152);

	/**
	 * <p>A colon-separated <code>string</code> of ciphers to use for the TLS 1.2 (1.1, 1.0) connection. Available as of cURL 7.9.</p>
	 */
	define('CURLOPT_SSL_CIPHER_LIST', 10083);

	/**
	 * <p>A colon delimited list of elliptic curve algorithms. For example, <code>X25519:P-521</code> is a valid list of two elliptic curves. This option defines the client's key exchange algorithms in the SSL handshake, if the SSL backend cURL is built to use supports it. Available as of PHP 8.2.0 and cURL 7.73.0.</p>
	 */
	define('CURLOPT_SSL_EC_CURVES', 10298);

	/**
	 * <p><b><code>false</code></b> to disable ALPN in the SSL handshake (if the SSL backend cURL is built to use supports it), which can be used to negotiate http2. Available as of PHP 7.0.7 and cURL 7.36.0.</p>
	 */
	define('CURLOPT_SSL_ENABLE_ALPN', 226);

	/**
	 * <p><b><code>false</code></b> to disable NPN in the SSL handshake (if the SSL backend cURL is built to use supports it), which can be used to negotiate http2. Available as of PHP 7.0.7 and cURL 7.36.0, and deprecated as of cURL 7.86.0.</p>
	 */
	define('CURLOPT_SSL_ENABLE_NPN', 225);

	/**
	 * <p><b><code>true</code></b> to enable and <b><code>false</code></b> to disable TLS false start which is a mode where a TLS client starts sending application data before verifying the server's <code>Finished</code> message. Available as of PHP 7.0.7 and cURL 7.42.0.</p>
	 */
	define('CURLOPT_SSL_FALSESTART', 233);

	/**
	 * <p>Set SSL behavior options, which is a bitmask of the <b><code>CURLSSLOPT_&#42;</code></b> constants. Defaults to none of the bits being set. Available as of PHP 7.0.7. and cURL 7.25.0.</p>
	 */
	define('CURLOPT_SSL_OPTIONS', 216);

	/**
	 * <p>Set to <code>0</code> to disable and <code>1</code> to enable SSL session-ID caching. By default all transfers are done using the cache enabled. Available as of cURL 7.16.0.</p>
	 */
	define('CURLOPT_SSL_SESSIONID_CACHE', 150);

	/**
	 * <p><code>2</code> to verify that a Common Name field or a Subject Alternate Name field in the SSL peer certificate matches the provided hostname. <code>0</code> to not check the names. <code>1</code> should not be used. In production environments the value of this option should be kept at <code>2</code> (default value). Support for value <code>1</code> removed in cURL 7.28.1. Available as of cURL 7.8.1.</p>
	 */
	define('CURLOPT_SSL_VERIFYHOST', 81);

	/**
	 * <p><b><code>false</code></b> to stop cURL from verifying the peer's certificate. Alternate certificates to verify against can be specified with the <b><code>CURLOPT_CAINFO</code></b> option or a certificate directory can be specified with the <b><code>CURLOPT_CAPATH</code></b> option. Defaults to <b><code>true</code></b> as of cURL 7.10. Default bundle of CA certificates installed as of cURL 7.10. Available as of cURL 7.4.2.</p>
	 */
	define('CURLOPT_SSL_VERIFYPEER', 64);

	/**
	 * <p><b><code>true</code></b> to enable and <b><code>false</code></b> to disable verification of the certificate's status. Available as of PHP 7.0.7 and cURL 7.41.0.</p>
	 */
	define('CURLOPT_SSL_VERIFYSTATUS', 232);

	/**
	 * <p>The name of a file containing a PEM formatted certificate. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_SSLCERT', 10025);

	/**
	 * <p>A <code>string</code> with the SSL client certificate. Available as of PHP 8.1.0 and cURL 7.71.0.</p>
	 */
	define('CURLOPT_SSLCERT_BLOB', 40291);

	/**
	 * <p>The password required to use the <b><code>CURLOPT_SSLCERT</code></b> certificate. Available as of cURL 7.1.0 and deprecated as of cURL 7.17.0.</p>
	 */
	define('CURLOPT_SSLCERTPASSWD', 10026);

	/**
	 * <p>A <code>string</code> with the format of the certificate. Supported formats are: <code>PEM</code>, <code>DER</code>, <code>ENG</code>, <code>P12</code>. <code>P12</code> (for PKCS#12-encoded files) is available as of OpenSSL 0.9.3. Defaults to <code>PEM</code>. Available as of cURL 7.9.3.</p>
	 */
	define('CURLOPT_SSLCERTTYPE', 10086);

	/**
	 * <p>The <code>string</code> identifier for the crypto engine of the private SSL key specified in <b><code>CURLOPT_SSLKEY</code></b>. Available as of cURL 7.9.3.</p>
	 */
	define('CURLOPT_SSLENGINE', 10089);

	/**
	 * <p>The <code>string</code> identifier for the crypto engine used for asymmetric crypto operations. Available as of cURL 7.9.3.</p>
	 */
	define('CURLOPT_SSLENGINE_DEFAULT', 90);

	/**
	 * <p>The name of a file containing a private SSL key. Available as of cURL 7.9.3.</p>
	 */
	define('CURLOPT_SSLKEY', 10087);

	/**
	 * <p>A <code>string</code> private key for client cert. Available as of PHP 8.1.0 and cURL 7.71.0.</p>
	 */
	define('CURLOPT_SSLKEY_BLOB', 40292);

	/**
	 * <p>The secret password needed to use the private SSL key specified in <b><code>CURLOPT_SSLKEY</code></b>.</p><p><b>Note</b>:</p><p>Since this option contains a sensitive password, remember to keep the PHP script it is contained within safe.</p>  Available as of cURL 7.9.3 and deprecated as of cURL 7.17.0.
	 */
	define('CURLOPT_SSLKEYPASSWD', 10026);

	/**
	 * <p>The key type of the private SSL key specified in <b><code>CURLOPT_SSLKEY</code></b>. Supported key types are: <code>PEM</code>, <code>DER</code>, <code>ENG</code>. Defaults to <code>PEM</code>. Available as of cURL 7.9.3.</p>
	 */
	define('CURLOPT_SSLKEYTYPE', 10088);

	/**
	 * <p>One of the <b><code>CURL_SSLVERSION_&#42;</code></b> constants.</p><p><b>Warning</b></p> <p>It is better to not set this option and leave the defaults. As setting this to <b><code>CURL_SSLVERSION_SSLv2</code></b> or <b><code>CURL_SSLVERSION_SSLv3</code></b> is very dangerous, given the known vulnerabilities in SSLv2 and SSLv3.</p>  Defaults to <b><code>CURL_SSLVERSION_DEFAULT</code></b>. Available as of cURL 7.1.0.
	 */
	define('CURLOPT_SSLVERSION', 32);

	/**
	 * <p>Accepts a file handle <code>resource</code> pointing to an alternative location to output errors to instead of <b><code>STDERR</code></b>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_STDERR', 10037);

	/**
	 * <p>Set the numerical stream weight (a number between <code>1</code> and <code>256</code>). Available as of PHP 7.0.7 and cURL 7.46.0.</p>
	 */
	define('CURLOPT_STREAM_WEIGHT', 239);

	/**
	 * <p><b><code>true</code></b> to suppress proxy CONNECT response headers from the user callback functions <b><code>CURLOPT_HEADERFUNCTION</code></b> and <b><code>CURLOPT_WRITEFUNCTION</code></b>, when <b><code>CURLOPT_HTTPPROXYTUNNEL</code></b> is used and a CONNECT request is made. Defaults to <b><code>false</code></b>. Available as of PHP 7.3.0 and cURL 7.54.0.</p>
	 */
	define('CURLOPT_SUPPRESS_CONNECT_HEADERS', 265);

	/**
	 * <p><b><code>true</code></b> to enable and <b><code>false</code></b> to disable TCP Fast Open. Available as of PHP 7.0.7 and cURL 7.49.0.</p>
	 */
	define('CURLOPT_TCP_FASTOPEN', 244);

	/**
	 * <p>If set to <code>1</code>, TCP keepalive probes will be sent. The delay and frequency of these probes can be controlled by the <b><code>CURLOPT_TCP_KEEPIDLE</code></b> and <b><code>CURLOPT_TCP_KEEPINTVL</code></b> options, provided the operating system supports them. If set to <code>0</code> (default) keepalive probes are disabled. The maximum number of probes can be set with the <b><code>CURLOPT_TCP_KEEPCNT</code></b> option. Available as of cURL 7.25.0.</p>
	 */
	define('CURLOPT_TCP_KEEPALIVE', 213);

	/**
	 * <p>Sets the maximum number of TCP keep-alive probes. The default is <code>9</code>. Available as of PHP 8.4.0 and cURL 8.9.0.</p>
	 */
	define('CURLOPT_TCP_KEEPCNT', null);

	/**
	 * <p>Sets the delay, in seconds, that the operating system will wait while the connection is idle before sending keepalive probes, if <b><code>CURLOPT_TCP_KEEPALIVE</code></b> is enabled. Not all operating systems support this option. The default is <code>60</code>. Available as of cURL 7.25.0.</p>
	 */
	define('CURLOPT_TCP_KEEPIDLE', 214);

	/**
	 * <p>Sets the interval, in seconds, that the operating system will wait between sending keepalive probes, if <b><code>CURLOPT_TCP_KEEPALIVE</code></b> is enabled. Not all operating systems support this option. The default is <code>60</code>. Available as of cURL 7.25.0.</p>
	 */
	define('CURLOPT_TCP_KEEPINTVL', 215);

	/**
	 * <p><b><code>true</code></b> to disable TCP's Nagle algorithm, which tries to minimize the number of small packets on the network. Defaults to <b><code>true</code></b>. Available as of cURL 7.11.2.</p>
	 */
	define('CURLOPT_TCP_NODELAY', 121);

	/**
	 * <p>Set an <code>array</code> of <code>string</code>s to pass to the telnet negotiations. The variables should be in the format <code>&gt;option=value&lt;</code>. cURL supports the options <code>TTYPE</code>, <code>XDISPLOC</code> and <code>NEW_ENV</code>. Available as of cURL 7.7.0.</p>
	 */
	define('CURLOPT_TELNETOPTIONS', 10070);

	/**
	 * <p>Set the blocksize to use for TFTP data transmission. Valid range is <code>8</code>-<code>65464</code> bytes. The default of <code>512</code> bytes is used if this option is not specified. The specified block size is only used if supported by the remote server. If the server does not return an option acknowledgment or returns an option acknowledgment with no block size, the default of <code>512</code> bytes is used. Available as of cURL 7.19.4.</p>
	 */
	define('CURLOPT_TFTP_BLKSIZE', 178);

	/**
	 * <p><b><code>true</code></b> to not send TFTP options requests. Defaults to <b><code>false</code></b>. Available as of PHP 7.0.7 and cURL 7.48.0.</p>
	 */
	define('CURLOPT_TFTP_NO_OPTIONS', 242);

	/**
	 * <p>Set how <b><code>CURLOPT_TIMEVALUE</code></b> is treated. Use <b><code>CURL_TIMECOND_IFMODSINCE</code></b> to return the page only if it has been modified since the time specified in <b><code>CURLOPT_TIMEVALUE</code></b>. If it hasn't been modified, a <code>304 Not Modified</code> header will be returned assuming <b><code>CURLOPT_HEADER</code></b> is <b><code>true</code></b>. Use <b><code>CURL_TIMECOND_IFUNMODSINCE</code></b> for the reverse effect. Use <b><code>CURL_TIMECOND_NONE</code></b> to ignore <b><code>CURLOPT_TIMEVALUE</code></b> and always return the page. <b><code>CURL_TIMECOND_NONE</code></b> is the default. Prior to cURL 7.46.0 the default was <b><code>CURL_TIMECOND_IFMODSINCE</code></b>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_TIMECONDITION', 33);

	/**
	 * <p>The maximum number of seconds to allow cURL functions to execute. Defaults to <code>0</code>, meaning that functions never time out during transfer. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_TIMEOUT', 13);

	/**
	 * <p>The maximum number of milliseconds to allow cURL functions to execute. If cURL is built to use the standard system name resolver, that portion of the connect will still use full-second resolution for timeouts with a minimum timeout allowed of one second. Defaults to <code>0</code>, meaning that functions never time out during transfer. Available as of cURL 7.16.2.</p>
	 */
	define('CURLOPT_TIMEOUT_MS', 155);

	/**
	 * <p>The time in seconds since January 1st, 1970. The time will be used by <b><code>CURLOPT_TIMECONDITION</code></b>. Defaults to <code>0</code>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_TIMEVALUE', 34);

	/**
	 * <p>The time in seconds since January 1st, 1970. The time will be used by <b><code>CURLOPT_TIMECONDITION</code></b>. Defaults to zero. The difference between this option and <b><code>CURLOPT_TIMEVALUE</code></b> is the type of the argument. On systems where 'long' is only 32 bit wide, this option has to be used to set dates beyond the year 2038. Available as of PHP 7.3.0 and cURL 7.59.0.</p>
	 */
	define('CURLOPT_TIMEVALUE_LARGE', 30270);

	/**
	 * <p>A <code>string</code> with a colon-separated list of ciphers to use for the connection to the TLS 1.3 connection. This option is currently used only when cURL is built to use OpenSSL 1.1.1 or later. When using a different SSL backend the TLS 1.3 cipher suites can be set with the <b><code>CURLOPT_SSL_CIPHER_LIST</code></b> option. Available as of PHP 7.3.0 and cURL 7.61.0.</p>
	 */
	define('CURLOPT_TLS13_CIPHERS', 10276);

	/**
	 * <p>Set a password to use for the TLS authentication method specified with the <b><code>CURLOPT_TLSAUTH_TYPE</code></b> option. Requires that the <b><code>CURLOPT_TLSAUTH_USERNAME</code></b> option also be set. This feature relies on TLS SRP which does not work with TLS 1.3. Available as of cURL 7.21.4.</p>
	 */
	define('CURLOPT_TLSAUTH_PASSWORD', 10205);

	/**
	 * <p>Set a <code>string</code> with the method of the TLS authentication. Supported method is <code>SRP</code> (TLS Secure Remote Password authentication). Available as of cURL 7.21.4.</p>
	 */
	define('CURLOPT_TLSAUTH_TYPE', 10206);

	/**
	 * <p>Set a <code>string</code> with the username to use for the TLS authentication method specified with the <b><code>CURLOPT_TLSAUTH_TYPE</code></b> option. Requires that the <b><code>CURLOPT_TLSAUTH_PASSWORD</code></b> option also be set. This feature relies on TLS SRP which does not work with TLS 1.3. Available as of cURL 7.21.4.</p>
	 */
	define('CURLOPT_TLSAUTH_USERNAME', 10204);

	/**
	 * <p>Set to <code>1</code> to enable and <code>0</code> to disable requesting compressed <code>Transfer Encoding</code> in the outgoing HTTP request. If the server responds with a compressed <code>Transfer Encoding</code>, cURL will automatically uncompress it on reception. Defaults to <code>0</code>. Available as of cURL 7.21.6.</p>
	 */
	define('CURLOPT_TRANSFER_ENCODING', 207);

	/**
	 * <p><b><code>true</code></b> to use ASCII mode for FTP transfers. For LDAP, it retrieves data in plain text instead of HTML. On Windows systems, it will not set <b><code>STDOUT</code></b> to binary mode. Defaults to <b><code>false</code></b>. Available as of cURL 7.1.1.</p>
	 */
	define('CURLOPT_TRANSFERTEXT', 53);

	/**
	 * <p>Enables the use of Unix domain sockets as connection endpoint and sets the path to the given <code>string</code>. Set to <b><code>null</code></b> to disable. Defaults to <b><code>null</code></b>. Available as of PHP 7.0.7 and cURL 7.40.0.</p>
	 */
	define('CURLOPT_UNIX_SOCKET_PATH', 10231);

	/**
	 * <p><b><code>true</code></b> to keep sending the username and password when following locations (using <b><code>CURLOPT_FOLLOWLOCATION</code></b>), even when the hostname has changed. Defaults to <b><code>false</code></b>. Available as of cURL 7.10.4.</p>
	 */
	define('CURLOPT_UNRESTRICTED_AUTH', 105);

	/**
	 * <p>Some protocols have "connection upkeep" mechanisms. These mechanisms usually send some traffic on existing connections in order to keep them alive. This option defines the connection upkeep interval. Currently, the only protocol with a connection upkeep mechanism is HTTP/2. When the connection upkeep interval is exceeded, an HTTP/2 PING frame is sent on the connection. Defaults to <b><code>CURL_UPKEEP_INTERVAL_DEFAULT</code></b> which is currently <code>60</code> seconds. Available as of PHP 8.2.0 and cURL 7.62.0.</p>
	 */
	define('CURLOPT_UPKEEP_INTERVAL_MS', 281);

	/**
	 * <p><b><code>true</code></b> to prepare for and perform an upload. Defaults to <b><code>false</code></b>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_UPLOAD', 46);

	/**
	 * <p>Preferred buffer size in bytes for the cURL upload buffer. The upload buffer size by default is 64 kilobytes. The maximum buffer size allowed to be set is 2 megabytes. The minimum buffer size allowed to be set is 16 kilobytes. Available as of PHP 8.2.0 and cURL 7.62.0.</p>
	 */
	define('CURLOPT_UPLOAD_BUFFERSIZE', 280);

	/**
	 * <p>The URL to fetch. This can also be set when initializing a session with <code>curl_init()</code>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_URL', 10002);

	/**
	 * <p>Sets the desired level of SSL/TLS for the transfer when using FTP, SMTP, POP3, IMAP, etc. These are all protocols that start out plain text and get "upgraded" to SSL using the STARTTLS command. Set to one of the <b><code>CURLUSESSL_&#42;</code></b> constants. Available as of cURL 7.17.0.</p>
	 */
	define('CURLOPT_USE_SSL', 119);

	/**
	 * <p>The contents of the <code>User-Agent: </code> header to be used in a HTTP request. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_USERAGENT', 10018);

	/**
	 * <p>The user name to use in authentication. Available as of cURL 7.19.1.</p>
	 */
	define('CURLOPT_USERNAME', 10173);

	/**
	 * <p>A username and password formatted as <code>[username]:[password]</code> to use for the connection. Available as cURL 7.1.0.</p>
	 */
	define('CURLOPT_USERPWD', 10005);

	/**
	 * <p><b><code>true</code></b> to output verbose information. Writes output to <b><code>STDERR</code></b>, or the file specified using <b><code>CURLOPT_STDERR</code></b>. Defaults to <b><code>false</code></b>. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_VERBOSE', 41);

	/**
	 * <p>Set to <code>1</code> to transfer multiple files according to a filename pattern. The pattern can be specified as part of the <b><code>CURLOPT_URL</code></b> option, using an fnmatch-like pattern (Shell Pattern Matching) in the last part of URL (filename). Available as of cURL 7.21.0.</p>
	 */
	define('CURLOPT_WILDCARDMATCH', 197);

	/**
	 * <p>A <code>callable</code> with the following signature:</p> callback(<code>resource</code> <code>$curlHandle</code>, <code>string</code> <code>$data</code>): <code>int</code>  <code>curlHandle</code>   The cURL handle.   <code>data</code>   The data to be written.    The data must be saved by the callback and the callback must return the exact number of bytes written or the transfer will be aborted with an error. Available as of cURL 7.1.0.
	 */
	define('CURLOPT_WRITEFUNCTION', 20011);

	/**
	 * <p>Accepts a file handle <code>resource</code> to the file that the header part of the transfer is written to. Available as of cURL 7.1.0.</p>
	 */
	define('CURLOPT_WRITEHEADER', 10029);

	/**
	 * <p>Accepts a bitmask setting WebSocket behavior options. The only available option is <b><code>CURLWS_RAW_MODE</code></b>. Defaults to <code>0</code>. Available as of PHP 8.3.0 and cURL 7.86.0.</p>
	 */
	define('CURLOPT_WS_OPTIONS', null);

	/**
	 * <p>A <code>callable</code> with the following signature:</p> callback(<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>resource</code> <code>$curlHandle</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$bytesToDownload</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$bytesDownloaded</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$bytesToUpload</code>,<br>&#xA0;&#xA0;&#xA0;&#xA0;<code>int</code> <code>$bytesUploaded</code><br>): <code>int</code>  <code>curlHandle</code>   The cURL handle.   <code>bytesToDownload</code>   The total number of bytes expected to be downloaded in this transfer.   <code>bytesDownloaded</code>   The number of bytes downloaded so far.   <code>bytesToUpload</code>   The total number of bytes expected to be uploaded in this transfer.   <code>bytesUploaded</code>   The number of bytes uploaded so far.    Return <code>1</code> to abort the transfer and set a <b><code>CURLE_ABORTED_BY_CALLBACK</code></b> error. Available as of PHP 8.2.0 and cURL 7.32.0.
	 */
	define('CURLOPT_XFERINFOFUNCTION', 20219);

	/**
	 * <p>Specifies the OAuth 2.0 access token. Set to <b><code>null</code></b> to disable. Defaults to <b><code>null</code></b>. Available as of PHP 7.0.7 and cURL 7.33.0.</p>
	 */
	define('CURLOPT_XOAUTH2_BEARER', 10220);

	/**
	 * Pause sending and receiving data. Available as of cURL 7.18.0.
	 */
	define('CURLPAUSE_ALL', 5);

	/**
	 * Unpause sending and receiving data. Available as of cURL 7.18.0.
	 */
	define('CURLPAUSE_CONT', 0);

	/**
	 * Pause receiving data. Available as of cURL 7.18.0.
	 */
	define('CURLPAUSE_RECV', 1);

	/**
	 * Unpause receiving data. Available as of cURL 7.18.0.
	 */
	define('CURLPAUSE_RECV_CONT', 0);

	/**
	 * Pause sending data. Available as of cURL 7.18.0.
	 */
	define('CURLPAUSE_SEND', 4);

	/**
	 * Unpause sending data. Available as of cURL 7.18.0.
	 */
	define('CURLPAUSE_SEND_CONT', 0);

	/**
	 * Available as of cURL 7.43.0.
	 */
	define('CURLPIPE_HTTP1', 1);

	/**
	 * Available as of cURL 7.43.0.
	 */
	define('CURLPIPE_MULTIPLEX', 2);

	/**
	 * Available as of cURL 7.43.0.
	 */
	define('CURLPIPE_NOTHING', 0);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_ALL', -1);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_DICT', 512);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_FILE', 1024);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_FTP', 4);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_FTPS', 8);

	/**
	 * Available as of cURL 7.21.2.
	 */
	define('CURLPROTO_GOPHER', 33554432);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_HTTP', 1);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_HTTPS', 2);

	/**
	 * Available as of cURL 7.20.0.
	 */
	define('CURLPROTO_IMAP', 4096);

	/**
	 * Available as of cURL 7.20.0.
	 */
	define('CURLPROTO_IMAPS', 8192);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_LDAP', 128);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_LDAPS', 256);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.71.0.
	 */
	define('CURLPROTO_MQTT', 268435456);

	/**
	 * Available as of cURL 7.20.0.
	 */
	define('CURLPROTO_POP3', 16384);

	/**
	 * Available as of cURL 7.20.0.
	 */
	define('CURLPROTO_POP3S', 32768);

	/**
	 * Available as of cURL 7.21.0.
	 */
	define('CURLPROTO_RTMP', 524288);

	/**
	 * Available as of cURL 7.21.0.
	 */
	define('CURLPROTO_RTMPE', 2097152);

	/**
	 * Available as of cURL 7.21.0.
	 */
	define('CURLPROTO_RTMPS', 8388608);

	/**
	 * Available as of cURL 7.21.0.
	 */
	define('CURLPROTO_RTMPT', 1048576);

	/**
	 * Available as of cURL 7.21.0.
	 */
	define('CURLPROTO_RTMPTE', 4194304);

	/**
	 * Available as of cURL 7.21.0.
	 */
	define('CURLPROTO_RTMPTS', 16777216);

	/**
	 * Available as of cURL 7.20.0.
	 */
	define('CURLPROTO_RTSP', 262144);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_SCP', 16);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_SFTP', 32);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.40.0.
	 */
	define('CURLPROTO_SMB', 67108864);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.40.0.
	 */
	define('CURLPROTO_SMBS', 134217728);

	/**
	 * Available as of cURL 7.20.0.
	 */
	define('CURLPROTO_SMTP', 65536);

	/**
	 * Available as of cURL 7.20.0.
	 */
	define('CURLPROTO_SMTPS', 131072);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_TELNET', 64);

	/**
	 * Available as of cURL 7.19.4.
	 */
	define('CURLPROTO_TFTP', 2048);

	/**
	 * Available as of cURL 7.10.
	 */
	define('CURLPROXY_HTTP', 0);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.19.3
	 */
	define('CURLPROXY_HTTP_1_0', 1);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.52.0
	 */
	define('CURLPROXY_HTTPS', 2);

	/**
	 * Available as of cURL 7.10.
	 */
	define('CURLPROXY_SOCKS4', 4);

	/**
	 * Available as of cURL 7.18.0.
	 */
	define('CURLPROXY_SOCKS4A', 6);

	/**
	 * Available as of cURL 7.10.
	 */
	define('CURLPROXY_SOCKS5', 5);

	/**
	 * Available as of cURL 7.18.0.
	 */
	define('CURLPROXY_SOCKS5_HOSTNAME', 7);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_BAD_ADDRESS_TYPE', 1);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_BAD_VERSION', 2);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_CLOSED', 3);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_GSSAPI', 4);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_GSSAPI_PERMSG', 5);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_GSSAPI_PROTECTION', 6);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_IDENTD', 7);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_IDENTD_DIFFER', 8);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_LONG_HOSTNAME', 9);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_LONG_PASSWD', 10);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_LONG_USER', 11);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_NO_AUTH', 12);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_OK', 0);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RECV_ADDRESS', 13);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RECV_AUTH', 14);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RECV_CONNECT', 15);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RECV_REQACK', 16);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_ADDRESS_TYPE_NOT_SUPPORTED', 17);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_COMMAND_NOT_SUPPORTED', 18);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_CONNECTION_REFUSED', 19);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_GENERAL_SERVER_FAILURE', 20);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_HOST_UNREACHABLE', 21);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_NETWORK_UNREACHABLE', 22);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_NOT_ALLOWED', 23);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_TTL_EXPIRED', 24);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REPLY_UNASSIGNED', 25);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_REQUEST_FAILED', 26);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_RESOLVE_HOST', 27);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_SEND_AUTH', 28);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_SEND_CONNECT', 29);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_SEND_REQUEST', 30);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_UNKNOWN_FAIL', 31);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_UNKNOWN_MODE', 32);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.73.0
	 */
	define('CURLPX_USER_REJECTED', 33);

	/**
	 * Available as of cURL 7.10.3.
	 */
	define('CURLSHOPT_NONE', 0);

	/**
	 * Specifies a type of data that should be shared. Available as of cURL 7.10.3.
	 */
	define('CURLSHOPT_SHARE', 1);

	/**
	 * Specifies a type of data that will be no longer shared. Available as of cURL 7.10.3.
	 */
	define('CURLSHOPT_UNSHARE', 2);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.28.0
	 */
	define('CURLSSH_AUTH_AGENT', 16);

	define('CURLSSH_AUTH_ANY', -1);

	define('CURLSSH_AUTH_DEFAULT', -1);

	/**
	 * Available as of PHP 7.3.0 and cURL 7.58.0
	 */
	define('CURLSSH_AUTH_GSSAPI', 32);

	define('CURLSSH_AUTH_HOST', 4);

	define('CURLSSH_AUTH_KEYBOARD', 8);

	define('CURLSSH_AUTH_NONE', 0);

	define('CURLSSH_AUTH_PASSWORD', 2);

	define('CURLSSH_AUTH_PUBLICKEY', 1);

	/**
	 * Available as of cURL 7.25.0
	 */
	define('CURLSSLOPT_ALLOW_BEAST', 1);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.77.0
	 */
	define('CURLSSLOPT_AUTO_CLIENT_CERT', 32);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.71.0
	 */
	define('CURLSSLOPT_NATIVE_CA', 16);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.68.0
	 */
	define('CURLSSLOPT_NO_PARTIALCHAIN', 4);

	/**
	 * Available as of PHP 7.0.7 and cURL 7.44.0
	 */
	define('CURLSSLOPT_NO_REVOKE', 2);

	/**
	 * Available as of PHP 8.2.0 and cURL 7.70.0
	 */
	define('CURLSSLOPT_REVOKE_BEST_EFFORT', 8);

	/**
	 * Require SSL for all communication or fail with <b><code>CURLE_USE_SSL_FAILED</code></b>. Available as of cURL 7.17.0.
	 */
	define('CURLUSESSL_ALL', 3);

	/**
	 * Require SSL for the control connection or fail with <b><code>CURLE_USE_SSL_FAILED</code></b>. Available as of cURL 7.17.0.
	 */
	define('CURLUSESSL_CONTROL', 2);

	/**
	 * Do not attempt to use SSL. Available as of cURL 7.17.0.
	 */
	define('CURLUSESSL_NONE', 0);

	/**
	 * Try using SSL, proceed as normal otherwise. Note that server may close the connection if the negotiation does not succeed. Available as of cURL 7.17.0.
	 */
	define('CURLUSESSL_TRY', 1);

	define('CURLVERSION_NOW', 9);

	/**
	 * Available as of PHP 8.3.0 and cURL 7.86.0
	 */
	define('CURLWS_RAW_MODE', null);

}
