#!/usr/bin/perl

use File::Basename;

# This script reads in a gerber plot file created by PCB, and creates an
# aperture table as per AP Circuits specs.


# Copyright 2000  Phil Salkie, Howman Controls  (phil@howman.com)
# This program is released under the GNU General Public License - for
#  a copy of the license, see www.gnu.org





# Parse any command line parameters passed in

while ($ARGV[0] =~ /^-(.+)/ && (shift, ($_ = $1), 1)) {

        next if /^$/;
        s/v// && (++$verbose, redo);
        s/V// && (++$version, redo);
        s/h// && (++$help, redo);
        die "usage:\n    apcaperture.pl [-vVh] filename\n";
}

die "apcaperture.pl Version 1.0, 10 May 2000\n" if $version;

if ($help)
{
	print <<HERE
\n\napcaperture.pl\n
A perl script to take a gerber plot file from the PCB printed circuit board
design package and produce an aperture table file to AP Circuits' specs.\n
Options:
-v	Verbose listing - prints messages to screen about file being processed.
-V	Version - print program version information and exit.
-h 	Help - print this message and exit.\n
Program Copyright 2000 Phil Salkie (phil\@howman.com)
Released under the GNU General Public License - see www.gnu.org for details.\n
HERE
;
	exit;


}

die "usage:\n    apcaperture.pl [-vVh] filename\n" if !$ARGV[0];


print "\nGenerating AP Circuits aperture table for file $ARGV[0].\n\n";


# get directory we're reading from and writing to.
$directory=dirname($ARGV[0]) . "/";
# get base file name 
$in_file_name = basename($ARGV[0], "");

print "Reading from file $ARGV[0]\n";
die "apcaperture.pl Failed: Could not open file $ARGV[0] to read.\n"
                                unless (open(IN_FILE, "<$ARGV[0]"));
        
# IN_FILE now has the handle of the input file to parse out.


print "Writing to aperture table $directory" . "aperture.tbl\n";
die "apcaperture.pl Failed: Could not open file $directory" . "aperture.tbl to write.\n"
                                unless (open(OUT_FILE, ">$directory" . "aperture.tbl"));


while (<IN_FILE>)
{
	if (/^%AMOCT([0-9]+).*,([0-9.]+),22.5\*%\s+$/) # Octagon definition
	{
		$oct_num = $1;
		$oct_size = $2;

		print "Setting octagon mask #$oct_num to $oct_size.\n" if ($verbose);

		print "Warning: Octagonal apertures may not be supported by AP Circuits.\n";

		$oct_hash{$oct_num} = $oct_size;
	} 


	if (/^%AD(D[0-9]+)([A-Z]+)/) # That is, it's an aperture definition
	{
		$ap_number = $1;
		$ap_type = $2;

		print "Found aperture #$ap_number, type $ap_type.\n" if ($verbose); 


		if ($ap_type eq 'C') # Circular Aperture
		{
			/,([0-9.]+)\*%\s+$/;

			$ap_name = "CIRCULAR";
			$x_size = $1;
			$y_size = $1;
		}

		if ($ap_type eq 'R') # Rectangular Aperture
		{
			/,([0-9.]+)X([0-9.]+)\*%\s+$/;

			$ap_name = "RECTANGULAR";
			$x_size = $1;
			$y_size = $2;
		}

		if ($ap_type eq 'OCT') # Octagonal Aperture
		{
			/OCT([0-9.]+)/;

			$ap_name = "OCTAGONAL";
			$x_size = $oct_hash{$1};
			$y_size = $x_size;
		}

		print "Setting aperture dimensions to $x_size by $y_size.\n" if ($verbose);

		printf OUT_FILE "%-4s %-11s %-5d %-5d 0     MULTI 0\r\n",
			$ap_number, $ap_name, $x_size * 1000, $y_size * 1000;

	}


}

close OUT_FILE;

