/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file SuWProtocol.cxx
    \author Romain BONDUE
    \date 07/01/2006 */
#include <sstream>

#include "SuWProtocol.h"
#include "SuWProtocolCodes.h"
#include "ErrorsCode.h"



nsWireless::CSuWProtocol::CSuWProtocol (const std::string& RemoteHandlerName,
                                        const std::string& DeviceName)
                                throw (nsErrors::CSystemExc, std::bad_alloc)
    : m_Su (RemoteHandlerName)
{
    SuDo (nsRoot::CMsg (DeviceName, DeviceNameC));

} // CSuWProtocol()


void nsWireless::CSuWProtocol::SetSSID (const std::string& SSID)
                                                throw (nsErrors::CSystemExc)
{
    SuDo (nsRoot::CMsg (SSID, SSIDC));

} // SetSSID()


void nsWireless::CSuWProtocol::SetFrequency (double Frequency)
                                                throw (nsErrors::CSystemExc)
{
    SuDo (Frequency, FrequencyC);

} // SetFrequency()


void nsWireless::CSuWProtocol::SetChannel (unsigned Channel)
                                                throw (nsErrors::CSystemExc)
{
    SuDo (Channel, ChannelC);

} // SetChannel()


nsWireless::CEncryptionD nsWireless::CSuWProtocol::GetEncryption ()
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
    try
    {
        std::istringstream Is (SuDo (nsRoot::CMsg ("",
                                                   GEncryptionC)).GetText());
        Is.exceptions (std::ios_base::failbit | std::ios_base::badbit);
        int Encrypt; // Can't be EncryptType_e because of the extractor.
        int Auth; // Ditto.
        std::string Key;
        Is >> Encrypt >> Auth;
        Is.ignore(); // The separator.
        getline (Is, Key);
        return CEncryptionD (Key.empty() ? CHexaKey() : CHexaKey (Key),
                             AuthType_e (Auth), EncryptType_e (Encrypt));
    }
    catch (...) // Whatever happens, drop everything.
    {
        throw nsErrors::CSystemExc ("Invalid data read.",
                                    nsErrors::InvalidData);
    }

} // GetEncryption()


void nsWireless::CSuWProtocol::SetEncryption (const CEncryptionD& Descriptor)
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
    std::ostringstream Os;
    Os << Descriptor.GetEncrypt() << ' ' << Descriptor.GetAuth() << ' '
       << Descriptor.GetDefaultKey() << ' ';
    for (unsigned I (0) ; I < CEncryptionD::MaxNbKey ; ++I)
        Os << Descriptor.GetKey (I).GetStr() << '\n';
    SuDo (Os.str(), SEncryptionC);

} // SetEncryption()


void nsWireless::CSuWProtocol::SetMode (Mode_e Mode)
                                                throw (nsErrors::CSystemExc)
{
    SuDo (Mode, ModeC);

} // SetMode()


void nsWireless::CSuWProtocol::SetTxRate (int TxRate)
                                                throw (nsErrors::CSystemExc)
{
    SuDo (TxRate, TxRateC);

} // SetTxRate()


void nsWireless::CSuWProtocol::SetAPMacAddr (const CMacAddress& Address)
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
    SuDo (nsRoot::CMsg (Address.GetStr(), ApMacAddrC));

} // SetAPMacAddr()


void nsWireless::CSuWProtocol::Scan () throw (nsErrors::CSystemExc)
{
    SuDo (nsRoot::CMsg ("", ScanC));

} // Scan()


nsRoot::CMsg nsWireless::CSuWProtocol::SuDo (const nsRoot::CMsg& Msg)
                                                throw (nsErrors::CSystemExc)
{
    try
    {
        const nsRoot::CMsg ReturnedMsg (m_Su.Do (Msg));
        if (ReturnedMsg.GetCode())
            throw nsErrors::CSystemExc (ReturnedMsg.GetText(),
                                        ReturnedMsg.GetCode());
        return ReturnedMsg;
    }
    catch (const nsErrors::CException& Exc)
    {
        throw nsErrors::CSystemExc (Exc.GetMsg(), Exc.GetCode());
    }
    catch (const std::bad_alloc& Exc)
    {
        throw nsErrors::CSystemExc (Exc.what(), nsErrors::OutOfMemory);
    }

} // SuDo()


void nsWireless::CSuWProtocol::Commit () throw (nsErrors::CSystemExc)
{
    SuDo (nsRoot::CMsg ("", CommitC));

} // Scan()
