Developing Rust Decimal
=======================

* [Setup](#setup)
* [Building](#building)
* [Library Versioning](#library-versioning)
* [Formatting / Code Style](#formatting--code-style)
* [Testing](#testing)
* [Fuzzing](#fuzzing)
* [Benchmarking](#benchmarking)
* [Code Coverage](#code-coverage)

## Setup

**Minimum Rust Version:** `1.46.0`

Rust Decimal leverages [cargo make](https://github.com/sagiegurari/cargo-make) to ensure a consistent build/test/release
approach. This also handles installing any additional dependencies in order to execute various commands, making getting set 
up and going relatively straight forward and easy.

Installing this can be done using `cargo`:

```shell
cargo install --force cargo-make
```

Once this has been installed, common tasks can be initiated with the `makers` command. 

## Building

**Useful Make Commands:** `makers build`

Building this library doesn't require any special commands and can be performed using a standard `cargo build`. 

### Building no-std

Builds by default leverage `std`. To disable this functionality you can build Rust Decimal without any default features
enabled:

```shell
cargo build --no-default-features
```

## Library Versioning

**Useful Make Commands:** `makers outdated`

The general rule of thumb with Rust Decimal is to ensure that downstream dependencies are kept up to date, so long as they 
align with the minimum requirements of the Rust Decimal build system. As a helpful measure, a `makers` command has been created
to identify any outdated dependencies within the project. On a regular basis this command is run and results evaluated for potential
updates to be applied.

Please note: because this is a library, the `Cargo.lock` file is not committed into the repository. This means that `cargo update` 
should be run before running `makers outdated`.

## Formatting / Code Style

**Useful Make Commands:** `makers format`, `makers clippy`

To maintain consistent styling across the project, both `clippy` and `cargo fmt` are used. Github actions will
check for consistent styling upon pull request however it is recommended that both of these commands are run before creating
a PR.

## Testing

**Useful Make Commands:** `makers test-all`, `makers test-no-std`, `makers test-maths`, `makers test-serde`, `makers test-db`

Testing is a critical part of the Rust Decimal development lifecycle. It helps ensure that the library is behaving correctly under
a mixture of different inputs and feature configurations. 

There are many different test configurations defined in `Makefile.toml` to help ensure that common test cases are easy to execute locally.
For the full complete list, please review `Makefile.toml` directly for all `test-*` tasks. Some useful test tasks are listed below:

* `test-all`: Tests all test configurations against all features
* `test-default`: Test the default configuration
* `test-no-std`: Test the library against a `no-std` build
* `test-maths`: Test the maths feature of the library using both legacy and default ops.
* `test-serde`: Test serialization and deserialization using a mixture of serde features
* `test-db`: Test a variety of database logic changes, including `diesel` and `postgres` configurations.

When adding new features, please make sure to generate new tasks within the `Makefile.toml` to ensure continued
coverage of your feature going forward.

### Generated tests

Rust Decimal includes a number of generated tests that have been generated outside the scope of this project. These are
effectively CSV files that use randomized data for the `lo`, `mid` and `hi` portions of the Decimal number.

Modification of these files is restricted and should not be committed. If you would like to add further generated tests then
please create new files as you see fit.

## Fuzzing

**Useful Make Commands:** `makers fuzz`

Rust Decimal includes some fuzz testing support also to help capture a wider range of testing scenarios. These can be
run by using `makers fuzz`.

## Benchmarking

**Useful Make Commands:** `makers bench`

In order to ensure that Rust Decimal runs quickly, a number of benchmarks have been created and can be executed using
`makers bench`. 

When adding new benchmarking configurations please ensure that you add a corresponding `Makefile.toml` task to capture the
updated configuration.

## Code Coverage

**Useful Make Commands:** `makers coverage`, `makers codecov-open`

Limited code coverage support has been added to Rust Decimal using `gcov`. A code coverage report
can be generated by running `makers coverage`. Once generated, the report can be opened using `makers codecov-open`.