# Code for the Newt interface to RoutePlanner
#
# Copyright (C) 1996-2001 Chris Lawrence
# This file may be freely distributed under the terms of the RoutePlanner
# license.  A copy should appear as 'LICENSE' in the archive that this
# file was included in.

import rpdbase, string, sys, os, snack, glob, commands, time, rpunits

VERSION = rpdbase.VERSION

def infobox(text, height=7, width=50, title="", screen=None):
    if not screen:
        s = snack.SnackScreen()
    else:
        s = screen
        
    if not title:
        title = VERSION

    t = snack.TextboxReflowed(width, text, maxHeight = s.height - 12)
    g = snack.GridForm(s, title, 1, 2)
    g.add(t, 0, 0, padding = (0, 0, 0, 1))
    g.draw()
    s.refresh()
    if not screen:
        s.finish()

def msgbox(text, height=7, width=60, title="", clear=0, screen=None,
           buttons = None):
    if not screen:
        s = snack.SnackScreen()
    else:
        s = screen
        
    if not title:
        title = VERSION

    if not buttons:
        buttons = ['Ok']
        
    res = snack.ButtonChoiceWindow(s, title, text, buttons, width)
    if not screen:
        s.finish()
    return res

def inputbox(text, prompts, width=60, entrywidth=40, title="", screen=None,
             buttons = ['Ok', 'Cancel']):
    if not title:
        title = VERSION

    if not screen:
        s = snack.SnackScreen()
    else:
        s = screen

    res = snack.EntryWindow(s, title, text, prompts, width=width,
                            entryWidth=entrywidth, buttons=buttons)
    if not screen:
        s.finish()
    return res

def menu(text, height=20, width=60, menuheight=15, menu=None,
         title="", scroll=0, screen=None, startpos=0):
    if not menu: return None
    if not screen:
        s = snack.SnackScreen()
    else:
        s = screen
    if not title:
        title = VERSION

    items = []
    for item in menu:
        items.append(item)

    if len(items) > menuheight: scroll=1
    res = startpos
    button, res = snack.ListboxChoiceWindow(s, title, text, items,
                                            width=width, height=menuheight,
                                            scroll=scroll, default=res)
    if button == 'cancel':
        if not screen:
            s.finish()
        return None
        
    if not screen:
        s.finish()
    return res

def ChooseCity(prompt, db, screen=None, rows=24, columns=79):
    (button, tp) = inputbox(prompt, ['City:'],
                            buttons=['Add to Route', 'Plan Route'],
                            screen=screen)
    if button == 'plan route':
        return None

    infobox('Searching...', screen=screen)
    stuff = tp[0]
    cities = db.CitiesMatching(stuff)
    if not cities:
        msgbox('No cities match '+stuff, screen=screen)
        return ChooseCity(prompt, db, screen)

    if len(cities) == 1:
        return cities[0]

    items = map(str, cities)
    sel = menu('Multiple cities match; please select one:', rows-6,
               columns-10, rows-15, menu = items, screen=screen)

    if sel is not None:
        return cities[sel]

    return None
# End of ChooseCity

def SelectRoutingMethod(screen=None, rows=24, columns=79):
    return menu('Please select a routing method:', rows-6,
                columns-10, rows-15, title = VERSION,
                menu = rpdbase.methods, screen=screen)

def NicerTime(min):
    return '%2d:%02d' % (min / 60, min % 60)

def trip(s, rows, columns, db, units):
    city = ChooseCity('Please select the city you want to start at:', db,
                      screen=s, rows=rows, columns=columns)
    if not city:
        s.finish()
        return 1
    
    path = [city]
    while city:
        city = ChooseCity('Please select destination for next leg:', db,
                          screen=s, rows=rows, columns=columns)
        if city:
            path.append(city)
        elif len(path) < 2:
            s.finish()
            return 1
        
    method = SelectRoutingMethod(s, rows, columns)
    if method is None:
        return 1
    
    msg = 'Planning route ('+rpdbase.methods[method]+'):'
    for entry in path:
        msg = msg+'\n  '+str(entry)
    
    infobox(msg, height=5+len(path), screen=s)
    
    trail = []
    for i in range(len(path)-1):
        trail.append( db.NewNavigate(path[i], path[i+1], method=method,
                                     units=units) )

    mpgs = rpdbase.defeff
    defmpg = rpdbase.defmpg
    speeds = rpdbase.defspeed
    if units != rpunits.UNITS_US:
        defmpg = rpunits.ConvertEfficiency(defmpg, rpunits.UNITS_US, units)
        mpgs = map(rpunits.ConvertEfficiency, mpgs, rpunits.UNITS_US, units)
        speeds = map(lambda x, y=units: rpunits.Distance(x).AsUnits(y))

    displist = rpdbase.ProduceTrail(trail, db, units, mpgs, defmpg, speeds)
    msg = msg + '\n' + rpdbase.FormatRoute(displist, path, units)
    
    while 1:
        res = msgbox(msg, title='Planned route', screen=s,
                     buttons = ['Plan another route...', 'Save to file...',
                                'Exit'])
        if res == 'exit':
            return 0
        elif res == 'plan another route...':
            return trip(s, rows, columns, db, units)
        else:
            (button, tp) = inputbox('Enter a filename:', ['Filename:'],
                                    screen=s)
            if tp and tp[0] and button != 'cancel':
                filename = os.path.expanduser(tp[0])
                fp = open(filename, 'w')
                fp.write(msg+'\n')
                fp.close()

def mainloop(database = None):
    r, c = string.split(commands.getoutput('stty size'))
    rows, columns = int(r), int(c)
    # Reasonable defaults
    rows = rows or 24
    columns = columns or 79

    if not database:
        databases = glob.glob('/usr/share/routeplanner/*.rpl[34]')
        databases = databases + \
                    glob.glob('/usr/share/routeplanner/*.rpl[34].gz')
        databases = map(os.path.basename, databases)
        
        answer = menu("Please select a database:", rows-6,
                      columns-10, rows-15, title = VERSION,
                      menu = databases)
        if answer == None:
            return 1

        database = databases[answer]

    if database and not os.path.exists(database):
        database = '/usr/share/routeplanner/'+database

    s = snack.SnackScreen()
    infobox('Loading database '+database+'...', screen=s)
    db = rpdbase.RPDatabase(database, quiet=1)
    msgbox('%s: %d cities, %d routes\nCreated by %s <%s>' %
           (database, len(db.cities), len(db.routes), db.author,
            db.author_email), height=8, width=60, screen=s)

    units = db.units
    avail_unit_names = rpdbase.UNITS.keys()
    avail_units = rpdbase.UNITS.values()
    selected = avail_units.index(units)
    selected = menu('Please select measurement units to use:', rows-6,
                    columns-10, rows-15, title = VERSION,
                    menu = avail_unit_names, startpos=selected, screen=s)
    if selected is None:
        s.finish()
        return 1
    units = avail_units[selected]
    
    res = trip(s, rows, columns, db, units)
    s.popWindow()
    s.finish()
    return res
