#define GRBS_ROUTE_POSTPROC_TRACE

#undef tprintf
#ifdef GRBS_ROUTE_POSTPROC_TRACE
#include <stdio.h>
#define tprintf printf
#else
#define tprintf grbs_nullprintf
#endif

/* if ang is in arc between arc_sa_ and arc_da_ exclusive, truncate the arc so
   one of the new boundaries is ang. If trunc_end is 0, truncate sa, else
   truncate end angle (da) */
static int grbs_angle_in_arc_trunc(double *arc_sa_, double *arc_da_, double ang, int trunc_end)
{
	double arc_sa = *arc_sa_, arc_da = *arc_da_;
	double arc_ea;
	int swapped = 0;

	/* make sure angle is always positive */
	if (ang < 0)
		ang += 2.0 * M_PI;

	if (arc_da < 0) { /* swap endpoints so da is always positive */
		arc_sa = arc_sa + arc_da;
		arc_da = -arc_da;
		swapped = 1;
	}
	if (arc_sa < 0)
		arc_sa += 2.0*M_PI;

	arc_ea = arc_sa + arc_da;

	/* if arc spans from some high value through zero, the end angle has
	   to be larger than 2*pi; if ang is under both start and end, that may
	   be the case so add a full circle to ang, last chance to get it in
	   range */
	if ((arc_sa > ang) && (arc_ea > ang))
		ang += 2.0*M_PI;

	if ((ang > arc_sa) && (ang < arc_ea))
		goto do_trunc;

	if (arc_ea > 2.0*M_PI) {
		ang += 2.0*M_PI;
		if ((ang > arc_sa) && (ang < arc_ea))
			goto do_trunc;
	}

	return 0;

	do_trunc:;
	/* (ea > sa) and ang is between */
	if (swapped)
		trunc_end = !trunc_end;
	if (trunc_end)
		arc_ea = ang; /* truncate at end */
	else
		arc_sa = ang; /* truncate at start */

	/* now convert back from sa & ea to sa & da, keeping original direction */
	if (*arc_da_ >= 0) { /* positive da */
		*arc_sa_ = arc_sa;
		*arc_da_ = arc_ea - arc_sa;
	}
	else { /* negative da */
		*arc_sa_ = arc_ea;
		*arc_da_ = arc_sa - arc_ea;
	}

	return 1;
}

/* Apply a limit on res at orbit r and l_ang, using effective width ew; returns
   number of changes made */
static int conc_corner_limit(grbs_t *grbs, grbs_arc_t *res, double r, double ew, double l_ang)
{
	double isa, ida, iea; /* limits's start, delta and end angle */
	double ea = asin(ew/r); /* effective angle; see trunk/doc/GRBS/conc_pp_ea.svg */
	int chg = 0;

	/* angle range of limit+-ew at r */
	isa = l_ang - ea;
	iea = l_ang + ea;
	ida = ea * 2;

	/* arc bool: check which end of res is affected by ia */
	if (grbs_angle_in_arc(isa, ida, res->sa, 1)) { /* res start angle is affected */
		chg += grbs_angle_in_arc_trunc(&res->sa, &res->da, isa, 0);
		chg += grbs_angle_in_arc_trunc(&res->sa, &res->da, iea, 0);
	}
	if (grbs_angle_in_arc(isa, ida, res->sa+res->da, 1)) { /* res end angle is affected */
		chg += grbs_angle_in_arc_trunc(&res->sa, &res->da, isa, 1);
		chg += grbs_angle_in_arc_trunc(&res->sa, &res->da, iea, 1);
	}

	return chg;
}

/* Find incident nets from pt that would limit the first "arc"; calculate the
   limited arc in res. Return 0 on success */
static int conc_corner_find_sentinel_limits(grbs_t *grbs, grbs_point_t *pt, grbs_arc_t *arc, grbs_arc_t *res)
{
	grbs_arc_t *i;

	/* initial range is the original arc */
	res->r = arc->r;
	res->sa = arc->sa;
	res->da = arc->da;

	for(i = gdl_first(&(pt->incs)); i != NULL; i = gdl_next(&(pt->incs), i)) {
		double ew;

		if (!i->in_use)
			continue;

		/* effective width: distance between the centerlines of i's net and arc's net */
		ew = arc->copper + i->copper + GRBS_MAX(arc->clearance, i->clearance);
		conc_corner_limit(grbs, res, arc->r, ew, i->sa);
	}
	return 0;
}

/* Update the endpoint of the other (next or previous, depending on tn_dir
   being +1 or -1) arc on the two-net so it heads to the corresponding endpoint
   of 'from' */
void conc_update_other(grbs_t *grbs, grbs_arc_t *from, int tn_dir)
{
	grbs_arc_t *dst = (tn_dir > 0) ? from->link_2net.next : from->link_2net.prev;
	int dst_at_end  = (tn_dir > 0) ? 0 : 1;
	double ex, ey, ea, ang;

	/* endpoint on 'from' */
	ea = dst_at_end ? from->sa : from->sa + from->da;
	ex = from->parent_pt->x + cos(ea) * from->r;
	ey = from->parent_pt->y + sin(ea) * from->r;

	if (dst->r == 0) { /* dst is incident */
		CHG_PRE(grbs, dst);
		dst->sa = atan2(ey - dst->parent_pt->y, ex - dst->parent_pt->x);
		auto_update_seg_sentinel_angles(dst);
		CHG_POST(grbs, dst);
		return;
	}

	if (dst->concave) {
		/* dst is convex or concave */
		double dx = ex - dst->parent_pt->x;
		double dy = ey - dst->parent_pt->y;

		/* see trunk/doc/GRBS/conc_pp.oa.svg for the other-arc-is-convex case, which is the basis for the concave (90 +-degree shift) */

		if (dst->da > 0)
			ang = M_PI/2 - (asin(dst->r / sqrt(dx*dx + dy*dy)) - atan2(dy, dx));
		else
			ang = (asin(dst->r / sqrt(dx*dx + dy*dy)) + atan2(dy, dx)) - M_PI/2;
	}
	else { /* other-arc-is-convex case */
		double a[4];
		grbs_bicycle_angles(
			dst->parent_pt->x, dst->parent_pt->y, dst->r,
			ex, ey, 0, a, 0);
		ang = a[dst->da < 0];
	}

	tprintf("  conc_update_other: ang=%f\n", ang);

	CHG_PRE(grbs, dst);
	if (dst_at_end == 0) {
		double old = dst->sa;
		dst->sa = ang;
		dst->da = grbs_arc_get_delta(dst->sa, old + dst->da, dst->da > 0 ? +1 : -1); /* don't change the other side angle */
	}
	else
		dst->da = grbs_arc_get_delta(dst->sa, ang, dst->da > 0 ? +1 : -1); /* don't change the other side angle */
	auto_update_seg_sentinel_angles(dst);
	CHG_POST(grbs, dst);
}

/* Tune angles of concave arcs so their corners fit with other concave
   arcs and incident nets (runs on all concave segments at a given point) */
void grbs_pp_concave_corners_at(grbs_t *grbs, grbs_point_t *pt)
{
	int segi, sa_chg, ea_chg;
	for(segi = 0; segi < GRBS_MAX_SEG; segi++) {
		grbs_arc_t *prev, *a, *sentinel = gdl_first(&pt->arcs[1][segi]), ra;

		if (sentinel == NULL)
			continue;

		for(a = gdl_next(&(pt->arcs[1][segi]), sentinel); ((a != NULL) && (a->in_use == 0)); a = gdl_next(&(pt->arcs[1][segi]), a)) ;
		if (a == NULL)
			continue;


		tprintf("PP cc %f;%f\n", pt->x, pt->y);
		tprintf("      sentinel: %f + %f; first: %f + %f\n", sentinel->sa, sentinel->da, a->sa, a->da);

		/* for the innermost arc look up incident and other concaves */

		conc_corner_find_sentinel_limits(grbs, pt, a, &ra);

		sa_chg = (ra.sa != a->sa);
		ea_chg = (ra.da != a->da);
		if (sa_chg || ea_chg) {
			tprintf("      truncate: %f + %f\n", ra.sa, ra.da);
			CHG_PRE(grbs, a);
			sentinel->sa = a->sa = ra.sa;
			sentinel->da = a->da = ra.da;
			CHG_POST(grbs, a);
			if (sa_chg)
				conc_update_other(grbs, a, -1);
			if (ea_chg)
				conc_update_other(grbs, a, +1);
		}

		/* from the second concave arc on, it's only the previous arc ends that we need to consider */
		prev = a;
		for(a = gdl_next(&(pt->arcs[1][segi]), a); a != NULL; a = gdl_next(&(pt->arcs[1][segi]), a)) {
			double ew;

			if (!a->in_use)
				continue;

		/* effective width: distance between the centerlines of prev's net and a's net */
			ew = a->copper + prev->copper + GRBS_MAX(a->clearance, prev->clearance);
			sa_chg = conc_corner_limit(grbs, a, a->r, ew, prev->sa);
			ea_chg = conc_corner_limit(grbs, a, a->r, ew, prev->sa + prev->da);

			if (sa_chg)
				conc_update_other(grbs, a, -1);
			if (ea_chg)
				conc_update_other(grbs, a, +1);

			prev = a;
		}
	}
}


/* Tune angles of concave arcs so their corners fit with other concave
   arcs and incident nets (runs on all concave segments at every point) */
void grbs_pp_concave_corners(grbs_t *grbs)
{
	grbs_point_t *p;
	for(p = gdl_first(&grbs->all_points); p != NULL; p = gdl_next(&grbs->all_points, p)) {
		grbs_pp_concave_corners_at(grbs, p);
	}
}

