//crc.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2014
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

#define CRC24_POLY 0x01864CFBL
#define CRC24_INIT 0x00B704CEL

#define ADLER32_BASE 65521L
#define ADLER32_INIT 1L

uint32_t roar_crc24_add(uint32_t state, const void * data, size_t len) {
 const unsigned char * p = data;
 int i;

 if ( data == NULL )
  return CRC24_INIT;

 for (; len; len--) {
  state ^= ((uint32_t)(*p++)) << 16;
  for (i = 0; i < 8; i++) {
   state <<= 1;
   if ( state & 0x01000000 )
    state ^= CRC24_POLY;
  }
 }

 return state & 0x00FFFFFFL;
}

uint32_t roar_adler32_add(uint32_t state, const void * data, size_t len) {
 const unsigned char * p = data;
 uint32_t s1 =  state        & 0xFFFFL;
 uint32_t s2 = (state >> 16) & 0xFFFFL;
 size_t n;

 if ( data == NULL )
  return ADLER32_INIT;

 for (n = 0; n < len; n++) {
  s1 = (s1 + p[n]) % ADLER32_BASE;
  s2 = (s2 + s1  ) % ADLER32_BASE;
 }

 return (s2 << 16) + s1;
}


int roar_hash_crc24_init(void * state) {
 uint32_t * self = state;

 *self = roar_crc24_init();

 return 0;
}

int roar_hash_crc24_digest(void * state, void * digest, size_t * len) {
 uint32_t * self = state;
 unsigned char * out  = digest;
 register uint32_t crc;

 if ( *len < 3 )
  return -1;

 *len = 3;

 crc = *self;

 out[0] = ((crc & 0x00FF0000) >> 16) & 0xFF;
 out[1] = ((crc & 0x0000FF00) >>  8) & 0xFF;
 out[2] = ((crc & 0x000000FF) >>  0) & 0xFF;

 return 0;
}

int roar_hash_crc24_proc(void * state, const void * data, size_t len) {
 uint32_t * self = state;

 *self = roar_crc24_add(*self, data, len);

 return 0;
}


int roar_hash_adler32_init(void * state) {
 uint32_t * self = state;

 *self = roar_adler32_init();

 return 0;
}

int roar_hash_adler32_digest(void * state, void * digest, size_t * len) {
 uint32_t * self = state;
 unsigned char * out  = digest;
 register uint32_t crc;

 if ( *len < 4 )
  return -1;

 *len = 4;

 crc = *self;

 out[0] = ((crc & 0xFF000000) >> 24) & 0xFF;
 out[1] = ((crc & 0x00FF0000) >> 16) & 0xFF;
 out[2] = ((crc & 0x0000FF00) >>  8) & 0xFF;
 out[3] = ((crc & 0x000000FF) >>  0) & 0xFF;

 return 0;
}

int roar_hash_adler32_proc(void * state, const void * data, size_t len) {
 uint32_t * self = state;

 *self = roar_adler32_add(*self, data, len);

 return 0;
}

//ll
