///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// animations using gnuplot
//
// author: Pierre.Saramito@imag.fr
//
// date: 17 november 2002
//
#include "rheolef/branch.h"
#include "rheolef/pstream.h"
#include "rheolef/rheostream.h"
#include "rheolef/basis_on_pointset.h"
#include "field_seq_visu_gnuplot_internal.h"

namespace rheolef {
using namespace std;

// ====================================================
// yet hard-coded ! TODO: use a -select-contour option
// ====================================================
template <class T>
void
put_header_gnuplot (odiststream& ops, const branch_basic<T,sequential>& b)
{
    ostream& out = ops.os();
    delete_macro (b._p_ctrl_out);
    delete_macro (b._p_data_out);
    string basename = iorheo::getbasename(out);
    if (basename == "") basename = "output";
    bool do_verbose = iorheo::getverbose(clog);
    bool do_execute = iorheo::getexecute(clog);
    bool do_clean   = iorheo::getclean  (clog);
    string tmp = (do_clean ? get_tmpdir() + "/" : "");

    string plot_name = tmp+basename + ".plot";
    b._to_clean = plot_name;
    b._p_ctrl_out = new_macro (ofstream(plot_name.c_str()));
    do_verbose && clog << "! file `" << plot_name << "' created" << endl;
    *(b._p_ctrl_out) << setbasename(basename)
                     << setprecision(numeric_limits<T>::digits10)
                     << "dt_pause = 0.25 # in seconds" << endl;
}
template <class T>
void
put_event_gnuplot (odiststream& ops, const branch_basic<T,sequential>& b)
{
  typedef typename branch_basic<T,sequential>::size_type size_type;
  bool do_verbose = iorheo::getverbose(clog);
  if (b.n_field() == 0) return;
  // TODO: assume all fields have the same mesh
  const space_basic<T,sequential>&        Xh    = b[0].second.get_space();
  const geo_basic<T,sequential>&          omega = Xh.get_geo();
  const array<point_basic<T>,sequential>& xdof  = Xh.get_xdofs();

  bool do_clean   = iorheo::getclean  (clog);
  string basename = iorheo::getbasename(ops.os());
  if (basename == "") basename = "output";
  string tmp = (do_clean ? get_tmpdir() + "/" : "");
  // ---------------------------------------------------------
  // 1) range : as computed at the first occurence
  // -----------------------------------------------
  size_t nfld = b.n_field();
  if (b._count_value == 0) {
    T all_min = std::numeric_limits<T>::max();
    T all_max = std::numeric_limits<T>::min();
    for (size_t ifld = 0; ifld < nfld; ifld++) {
      const field_basic<T,sequential>& u = b[ifld].second;
      all_min = std::min (all_min, u.min());
      all_max = std::max (all_max, u.max());
    }
    if (b._u_range.first != std::numeric_limits<T>::min()) {
      all_min = b._u_range.first;
    }
    if (b._u_range.second != std::numeric_limits<T>::max()) {
      all_max = b._u_range.second;
    }
    if (fabs(all_max - all_min) < 1e-7) {
      // perhaps the solution is zero ? try to renormalize
      // TODO: add command-line options for the range
      all_max = all_min + 1;
    }
    *(b._p_ctrl_out) << "set xrange [" << omega.xmin()[0] << ":" << omega.xmax()[0] << "]" << endl
                     << "set yrange [" << all_min << ":" << all_max << "]" << endl;
  }
  // -----------------------------------------------
  // 2) plot commands
  // -----------------------------------------------
  const string& pname = b.parameter_name() + " = " + ftos(b.parameter());
  *(b._p_ctrl_out) << "set title \"i = " << b._count_value << ": " << pname << "\" font \"arial,20\"" << endl
                   << "plot \\" << endl;
  for (size_t ifld = 0; ifld < nfld; ifld++) {
      const string& uname = b[ifld].first;
      string gdat_name = tmp + basename + "_" + uname + "_" + itos(b._count_value) + ".gdat";
      *(b._p_ctrl_out) << (ifld == 0 ? " " : ",")
                       << " \"" << gdat_name << "\""
                       << " u 1:2"
                       << " t \"" << uname << "\""
                       << " w l"
                       << (ifld < nfld-1 ? " \\" : "")
                       << endl;
  }
  *(b._p_ctrl_out) << "pause dt_pause" << endl;
  // -----------------------------------------------
  // 3) data
  // -----------------------------------------------
  // TODO: subdivide option not set by the branch unix command
  size_type subdivide = iorheo::getsubdivide(ops.os());
  bool fill           = iorheo::getfill(ops.os());    // show grid or fill elements
  for (size_t ifld = 0; ifld < nfld; ifld++) {
      const string&                 uname = b[ifld].first;
      const field_basic<T,sequential>& uh = b[ifld].second;
      string gdat_name = tmp + basename + "_" + uname + "_" + itos(b._count_value) + ".gdat";
      ofstream gdat (gdat_name.c_str());
      b._to_clean = b._to_clean + " " + gdat_name;
      if (do_verbose) clog << "! file \"" << gdat_name << "\" created.\n";
      gdat << setprecision(numeric_limits<T>::digits10);
      const geo_basic<T,sequential>& omega = uh.get_geo();
      const numbering<T,sequential>& fem = uh.get_space().get_numbering();
      if (subdivide == 0) { // subdivide is unset: use default
        subdivide = std::max(omega.order(), subdivide);
        subdivide = std::max(fem.degree (), subdivide);
      }
      size_type map_dim = omega.map_dimension();
      size_type used_dim = (fill ? map_dim : 1);
      basis_basic<T> subdivide_pointset ("P"+itos(subdivide));
      basis_on_pointset<T>   geo_on_pointset (subdivide_pointset, omega.get_piola_basis());
      basis_on_pointset<T> field_on_pointset (subdivide_pointset, fem.get_basis());
      bound_type<T> bbox;
      bbox.xmin = omega.xmin();
      bbox.xmax = omega.xmax();
      bbox.umin = uh.min();
      bbox.umax = uh.max();
      for (size_type ie = 0, ne = omega.size(used_dim); ie < ne; ie++) {
        const geo_element& K = omega.get_geo_element(used_dim,ie);
        put (gdat, K, omega, uh, geo_on_pointset, field_on_pointset, subdivide, bbox);
      }
      gdat.close();
  }
}
template <class T>
void
put_finalize_gnuplot (odiststream& ops, const branch_basic<T,sequential>& b)
{
  // is the destructor virtual : ofstream::~ofstream() or opstream::~opstream() ?
  string basename = iorheo::getbasename(ops.os());
  if (basename == "") basename = "output";
  bool do_verbose = iorheo::getverbose(clog);
  bool do_execute = iorheo::getexecute(clog);
  bool do_clean   = iorheo::getclean(clog);
  bool reader_on_stdin = iorheo::getreader_on_stdin(ops.os());
  if (! reader_on_stdin) {
    *(b._p_ctrl_out) << "pause -1 \"<end>\"" << endl;
  }
  string tmp = (do_clean ? get_tmpdir() + "/" : "");
  //
  // run gnuplot
  //
  int status = 0;
  string command;
  if (do_execute) {
      command = "gnuplot ";
      if (reader_on_stdin) command += "-persist ";
      command += tmp + basename + ".plot";
      if (do_verbose) clog << "! " << command << endl;
      cin.sync();
      status = system (command.c_str());
  }
  //
  // clear gnuplot data
  //
  if (do_clean) {
        string tmp = get_tmpdir() + "/";
        if (!do_clean) tmp = "";
	string command = "/bin/rm -f " + b._to_clean;
	// to much files to see:
        // do_verbose && clog << "! " << command << endl;
        do_verbose && clog << "! cleaning temporary .plot and .gdat files" << endl;
	int status = system (command.c_str());
  }
  delete_macro (b._p_ctrl_out);
  delete_macro (b._p_data_out);
  b._p_ctrl_out = 0;
  b._p_data_out = 0;
  b._to_clean = "";
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template void put_header_gnuplot   (odiststream&, const branch_basic<Float,sequential>&);
template void put_event_gnuplot    (odiststream&, const branch_basic<Float,sequential>&);
template void put_finalize_gnuplot (odiststream&, const branch_basic<Float,sequential>&);

} // namespace rheolef
