# ifndef _SKIT_CHEBY_H
# define _SKIT_CHEBY_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*Class:cheby
NAME: cheby - preconditioned Chebyshev method 
SYNOPSIS:
    
    "int cheby" ("const Matrix &A, Vector &x, const Vector &b",
    |   const Preconditioner &M, int &max_iter, Real &tol,
    |  Type eigmin, Type eigmax);

EXAMPLE:

  The simplest call to 'cheby' has the folling form:

  |    int status = cheby(a, x, b, EYE, 100, 1e-7, 0.01, 3);

DESCRIPTION:       

 'cheby' solves the unsymmetric linear system Ax = b 
 using the preconditioned Chebyshev method.

  The return value indicates convergence within max_iter (input)
  iterations (0), or no convergence within max_iter iterations (1).
  Upon successful return, output arguments have the following values:

    "x"  approximate solution to Ax = b

    "max_iter" the number of iterations performed before the tolerance was reached

    "tol" the residual after the final iteration

  Finally, the parameters "eigmin" and "eigmax" are parameters
  provided to estimate an ellipse containing the spectrum of A.
  In the case of positive-definite A, these parameters are real
  and correspond to estimates of the minimal and
  maximal eigenvalues of A, respectively. Note that a poor
  estimates for these values can cause poor convergence
  behavior (including divergence).
SEE ALSO:

  class "csr", class "vec"

NOTE: 

  'cheby' is an iterative template routine.

  'cheby' follows the algorithm described on p. 30 in

	Templates for the Solution of Linear Systems: Building Blocks for Iterative Methods, 
	2nd Edition, 
        R. Barrett, M. Berry, T. F. Chan, J. Demmel, J. Donato, J. Dongarra, V. Eijkhout,
	R. Pozo, C. Romine, H. Van der Vorst,
        SIAM, 1994, "ftp.netlib.org/templates/templates.ps".

  The present implementation is inspired from "IML++ 1.2" iterative method library,
  "http://math.nist.gov/iml++".

AUTHOR: 

    Pierre Saramito
    | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France

DATE: 
    
    12 march 1997

METHODS: @cheby
End:
*/
// ========== [ method implementation ] ====================================

namespace rheolef { 

//<cheby:
template < class Matrix, class Vector, class Preconditioner, class Real,
           class Type >
int 
cheby(const Matrix &A, Vector &x, const Vector &b,
      const Preconditioner &M, int &max_iter, Real &tol,
      Type eigmin, Type eigmax)
{
  Real resid;
  Type alpha=0, beta, c, d;
  Vector p, q, z;

  Real normb = norm(b);
  Vector r = b - A * x;

  if (normb == Real(0))
    normb = 1;
  
  if ((resid = norm(r) / normb) <= tol) {
    tol = resid;
    max_iter = 0;
    return 0;
  }

  c = (eigmax - eigmin) / 2.0;
  d = (eigmax + eigmin) / 2.0;

  for (int i = 1; i <= max_iter; i++) {
    z = M.solve(r);                 // apply preconditioner

    if (i == 1) {
      p = z;
      alpha = 2.0 / d;
    } else {
      beta = c * alpha / 2.0;       // calculate new beta
      beta = beta * beta;
      alpha = 1.0 / (d - beta);     // calculate new alpha
      p = z + beta * p;             // update search direction
    }

    q = A * p;
    x += alpha * p;                 // update approximation vector
    r -= alpha * q;                 // compute residual

    if ((resid = norm(r) / normb) <= tol) {
      tol = resid;
      max_iter = i;
      return 0;                     // convergence
    }
  }

  tol = resid;
  return 1;                         // no convergence
}
//>cheby:
}// namespace rheolef
# endif // _SKIT_CHEBY_H
