/*	Integer_Range

PIRL CVS ID: Integer_Range.java,v 1.14 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Utilities;

/**	An <i>Integer_Range</i> specifies a range of integer values.
<p>
	The range of values is characterized by minimum and maximum values
	that identify the inclusive limits of the range. Single-valued and
	open-ended ranges are defined. An appropriate String representation
	is provided.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.14
*/
public class Integer_Range
	implements Cloneable
{
public static final String
	ID = "PIRL.Utilities.Integer_Range (1.14 2012/04/16 06:18:24)";

/**	The range minimum value.
*/
protected long
	Minimum	= 0;

/**	The range maximum value.
*/
protected long
	Maximum	= 0;

/*==============================================================================
	Constructors
*/
/**	Constructs a default Integer_Range.
<p>
	The range values are initialized to zero.
*/
public Integer_Range ()
{}

/**	Constructs an Integer_Range from limiting values.
<p>
	Note: If the minimum is greater than the maximum then it is
	used as the maximum and the maximum used as the minimum.
<p>
	A range {@link #Is_Open_Ended() is open ended} if either the minimum
	is {@link Long#MIN_VALUE} or the maximum is {@link Long#MAX_VALUE}.
<p>
	@param	minimum	The minimum inclusive limit value of the range.
	@param	maximum	The maximum inclusive limit value of the range.
	@see	#Range(long, long)
*/
public Integer_Range
	(
	long	minimum,
	long	maximum
	)
{Range (minimum, maximum);}

/**	Constructs a single-valued Integer_Range.
<p>
	@param	value	The value of the range.
	@see	#Value(long)
*/
public Integer_Range
	(
	long	value
	)
{Value (value);}

/**	Constructs an Integer_Range as a copy of another Integer_Range.
<p>
	@param	range	The Integer_Range to be copied. If null a default
		Integer_Range is constructed.
	@see	#Range(Integer_Range)
*/
public Integer_Range
	(
	Integer_Range	range
	)
{Range (range);}

/**	Constructs an Integer_Range from a Real_Range.
<p>
	The values of the specified range are truncated and
	clipped to the Long.MIN_VALUE and Long.MAX_VALUE values.
<p>
	@param	range	The Real_Range to be copied. If null a default
		Integer_Range is constructed.
	@see	#Range(Real_Range)
*/
public Integer_Range
	(
	Real_Range	range
	)
{Range (range);}

/**	Clones this Integer_Range.
<p>
	@return	An Integer_Range constructed as a copy of this Integer_Range.
*/
public Object clone ()
{return new Integer_Range (this);}

/*==============================================================================
	Accessors
*/
/**	Sets the range minimum and maximum.
<p>
	Note: If the minimum is greater than the maximum then it is
	used as the maximum and the maximum used as the minimum.
<p>
	@param	minimum	The minimum inclusive value of the range.
	@param	maximum	The maximum inclusive value of the range.
	@return	This Integer_Range.
	@see	#Minimum(long)
	@see	#Maximum(long)
*/
public Integer_Range Range
	(
	long	minimum,
	long	maximum
	)
{
if (minimum <= maximum)
	{
	Minimum (minimum);
	Maximum (maximum);
	}
else
	{
	Minimum (maximum);
	Maximum (minimum);
	}
return this;
}

/**	Sets the range from another Integer_Range.
<p>
	<b>N.B.</b>: The values of the source range are copied without checking.
<p>
	@param	range	The Integer_Range to copy. If null, nothing is done.
	@return	This Integer_Range.
*/
public Integer_Range Range
	(
	Integer_Range	range
	)
{
if (range != null)
	{
	Minimum = range.Minimum;
	Maximum = range.Maximum;
	}
return this;
}

/**	Sets the range from a Real_Range.
<p>
	The values of the Real_Range are truncated (not rounded) and clipped
	to the Long.MIN_VALUE and Long.MAX_VALUE values.
<p>
	@param	range	The Real_Range to be copied. If null nothing is done.
	@return	This Integer_Range.
	@see	#Minimum(long)
	@see	#Maximum(long)
*/
public Integer_Range Range
	(
	Real_Range	range
	)
{
if (range != null)
	{
	if (range.Minimum () < (double)Long.MIN_VALUE)
		Minimum (Long.MIN_VALUE);
	else
	if (range.Minimum () > (double)Long.MAX_VALUE)
		Minimum (Long.MAX_VALUE);
	else
		Minimum ((long)range.Minimum ());

	if (range.Maximum () < (double)Long.MIN_VALUE)
		Maximum (Long.MIN_VALUE);
	else
	if (range.Maximum () > (double)Long.MAX_VALUE)
		Maximum (Long.MAX_VALUE);
	else
		Maximum ((long)range.Maximum ());
	}
return this;
}

/**	Gets the minimum range value.
<p>
	@return	The minimum range value.
*/
public long Minimum ()
{return Minimum;}

/**	Sets the minimum range value.
<p>
	If the new minimum value is greater than the current {@link
	#Maximum() maximum} value, the maximum is reset to the new minimum.
<p>
	@param	minimum	The minimum range value.
	@return	This Integer_Range.
*/
public Integer_Range Minimum
	(
	long	minimum
	)
{
if ((Minimum = minimum) > Maximum)
	 Maximum = minimum;
return this;
}

/**	Gets the maximum range value.
<p>
	@return	The maximum range value.
*/
public long Maximum ()
{return Maximum;}

/**	Sets the maximum range value.
<p>
	If the new maximum range value is less than the current {@link
	#Minimum() minimum} value, the minimum is reset to the new maximum.
<p>
	@param	maximum	The maximum range value.
	@return	This Integer_Range.
*/
public Integer_Range Maximum
	(
	long	maximum
	)
{
if ((Maximum = maximum) < Minimum)
	 Minimum = maximum;
return this;
}

/**	Gets the distance between the range values.
<p>
	@return	The difference between the maximum and minimum limit values,
		or Long.MAX_VALUE if the range is {@link #Is_Open_Ended()
		open-ended}.
*/
public long Distance ()
{
if (Is_Open_Ended ())
	return Long.MAX_VALUE;
return Maximum - Minimum;
}

/**	Tests if the range is open-ended.
<p>
	@return	true	If either the {@link #Minimum() minimum} is
		{@link Long#MIN_VALUE} or the {@link #Maximum() maximum} is
		{@link Long#MAX_VALUE}.
*/
public boolean Is_Open_Ended ()
{
return
	Maximum == Long.MAX_VALUE ||
	Minimum == Long.MIN_VALUE;
}

/**	Tests if the range is for a single value.
<p>
	For a single valued range the {@link #Minimum() minimum} and
	{@link #Maximum() maximum} values are identical.
<p>
	@return true if the range only has a single value; false otherwise.
*/
public boolean Is_Single_Valued ()
{return Minimum == Maximum;}

/**	Sets the range to a single value.
<p>
	Both the minimum and maximum are set to the value.
<p>
	@param	value	The single value for the range.
	@return	This Integer_Range.
	@see	#Minimum(long)
	@see	#Maximum(long)
*/
public Integer_Range Value
	(
	long	value
	)
{
Minimum (value);
return Maximum (value);
}

/**	Compares two Integer_Ranges for equality.
<p>
	The two ranges are equal if, and only if, the argument is not null
	and the {@link #Minimum() minimum} and {@link #Maximum() maximum}
	values of the ranges are equal.
*/
public boolean equals
	(
	Integer_Range	range
	)
{
return
	range != null &&
	range.Minimum == Minimum &&
	range.Maximum == Maximum;
}

/**	Gets a hash code for this Integer_Range.
<p>
	The result is the upper 16 bits of the hash code of the range
	minimum as a Long concatenated with the upper 16 bits of the
	hash code of the range maximum as a Long. That is, the
	hash code is the value of the expression:
<p><blockquote>
	(new Long (Minimum ()).hashCode () & 0xFFFF0000) |
	(new Long (Maximum ()).hashCode () >>> 16)
</blockquote><p>
*/
public int hashCode ()
{
return
	(new Long (Minimum).hashCode () & 0xFFFF0000) |
	(new Long (Maximum).hashCode () >>> 16);
}

/**	Provides a String representation of the Integer_Range.
<p>
	The format of the Integer_Range representation is:
<p><blockquote>
	<i>minimum</i><b>-</b><i>maximum</i>
</blockquote><p>
	If the minimum and maximum are identical then only a single value is
	represented.
<p>
	If the minimum is not {@link Long#MIN_VALUE} or the maximum is {@link
	Long#MAX_VALUE} then the minimum value is represented. Then a dash
	('-') delimiter is provided regardless of whether the minimum is
	represented or not.  If the maximum is not {@link Long#MAX_VALUE}
	then it is included in the representation after the '-' delimiter.
	Note that a range may be open-ended at its minimum or maximum, but
	in all cases at least one limit value will be represented with the
	minimum value being used ifthe range is double open-ended.
<p>
	@return	A String representation of the Integer_Range.
	@see	#Minimum()
	@see	#Maximum()
*/
public String toString ()
{
StringBuffer
	range = new StringBuffer ();
if (Minimum == Maximum)
	range.append (String.valueOf (Minimum));
else
	{
	if (Minimum != Long.MIN_VALUE ||
		Maximum == Long.MAX_VALUE)
		range.append (String.valueOf (Minimum));
	range.append ('-');
	if (Maximum != Long.MAX_VALUE)
		range.append (String.valueOf (Maximum));
	}
return range.toString ();
}

}	//	End of Integer_Range class.
