/*	Database_View

PIRL CVS ID: Database_View.java,v 1.13 2012/04/16 06:08:57 castalia Exp

Copyright (C) 2002-2007  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Database;

import PIRL.PVL.*;
import PIRL.Viewers.*;
import PIRL.Strings.Words;
import PIRL.Configuration.Configuration;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.tree.*;
import javax.swing.table.*;
import javax.swing.event.TableModelListener;
import javax.swing.event.TableModelEvent;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.Iterator;
import java.util.Enumeration;
import java.util.TreeSet;
import java.lang.StringBuffer;
import java.net.URL;

/**	<I>Database_View</I> provides a View of a Database.
<P>
	Database management capabilities are provided.
<P>
	@author	Bradford Castalia, UA/PIRL
	@version 1.13
*/
public class Database_View
	extends JFrame
{
private static final String
	ID = "PIRL.Database.Database_View (1.13 2012/04/16 06:08:57)";

//	Configurable parameters:
private static final int	DEFAULT_VIEW_WIDTH				= 500,
							DEFAULT_SELECTION_PANEL_HEIGHT	= 350,
							DEFAULT_DATA_TABLE_HEIGHT		= 150,
							LIST_WIDTH						= 200,
							LIST_HEIGHT						= 150,
							DEFAULT_LIMIT					= 50;

//	Database.
private Database			The_Database = null;

//	Configuration.
private Configuration		The_Configuration = null;
private Parameter_View		Configuration_View = null;
private Dimension			Configuration_View_Size
								= new Dimension (250, 300);
private View_Locator		Configuration_View_Locator
								= new View_Locator ();

//	GUI elements.
private JTabbedPane			Modes_Panes;


private static Color		Normal_Background_Color = null;

//	SELECT.
private int					SELECT_PANE;
private static ImageIcon	SELECT_ICON
								= null;
private static final String	SELECT_ICON_NAME
								= "Export_24.gif";
private JPanel				Selection_Panels;
private JTree				Fields_Available_Tree,
							Fields_Chosen_Tree;
private DefaultTreeModel	Fields_Available_Tree_Model
								= new DefaultTreeModel
								(new DefaultMutableTreeNode ("Source Fields")),
							Fields_Chosen_Tree_Model;
private JTextArea			Conditions;

//	JSpinner is new with Java 1.4; use it when 1.4 is ubiquitous.
//private JSpinner			Limit;
private JTextField			Limit;
private JButton				Include_Button,
							Exclude_Button,
							Join_Button,
							Reset_Button,
							Update_Button,
							SQL_Button,
							Select_Button;
private static ImageIcon	INCLUDE_ICON;
private static final String	INCLUDE_ICON_NAME
								= "Arrow_Right16.gif";
private static ImageIcon	EXCLUDE_ICON;
private static final String	EXCLUDE_ICON_NAME
								= "Arrow_Left16.gif";
private Data_Table			Select_Table;
private JLabel				Select_Records;
private String				Current_SQL_Command = "";
private View_Locator		SQL_Locator = new View_Locator ();

private static int			UPDATE_DISABLED	= 0,
							UPDATE_PENDING	= 1,
							UPDATE_ENABLED	= -1;
private int					Update_Enabled	= UPDATE_DISABLED;
private static final String	UPDATE_ENABLED_PARAMETER
								= "Update_Enabled";

//	INSERT.
private int					INSERT_PANE;
private static ImageIcon	INSERT_ICON;
private static final String	INSERT_ICON_NAME
								= "Import_24.gif";
private static final Color	INSERT_COLOR = new Color
								((float)0.85, (float)0.75, (float)0.85);
private JPanel				Insertion_Panels;
private JTree				Insert_Fields_Tree;
private DefaultTreeModel	Insert_Fields_Tree_Model
								= new DefaultTreeModel
								(new DefaultMutableTreeNode ("Insert Records"));
private JButton				Insert_into_Button,
							Clear_Button,
							New_Button,
							Insert_Button;
private static ImageIcon	INSERTION_ICON;
private static final String	INSERTION_ICON_NAME
								= "Arrow_Down16.gif";
private Data_Table			Insert_Table;
private JLabel				Insertion_Table;
private boolean				Insert_Enabled = false;
private static final String	INSERT_ENABLED_PARAMETER
								= "Insert_Enabled";

//	Miscellaneous.
private static final String	NL = Database.NL;


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_SETUP			= 1 << 0,
	DEBUG_QUERY			= 1 << 1,
	DEBUG_UI_SELECTION	= 1 << 2,
	DEBUG_SELECT_TABLE	= 1 << 3,
	DEBUG_INSERT_TABLE	= 1 << 4,
	DEBUG_UI_LAYOUT		= 1 << 5,
	DEBUG_SQL			= 1 << 6,
	DEBUG_CONFIGURATION	= 1 << 7,
	DEBUG_UTILITY		= 1 << 8,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
public Database_View
	(
	Database	database
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">>> Database_View");

//	Ensure that the database is open for access.
if (database == null || ! database.is_Open ())
	{
	//	Open the default database.
	try {database.Open (null);}
	catch (Exception exception)
		{
		Dialog_Box.Error ("Unable to Open the default database." + NL
			+ NL
			+ exception.getMessage ());
		throw new Database_Exception
			(ID + NL
			+ exception.getMessage ());
		}
	}
The_Database = database;

//	Setup the configuration.
Configure ();

//	Load the icons.
Load_Icons (this);

/*
//	Build the menu bar.
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("--- Menu_Bar");
setJMenuBar (Menu_Bar ());
*/

//	Setup the Mode tabbed panes.
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("--- Modes_Panes");
Modes_Panes = Modes_Panes ();
Modes_Panes.setEnabledAt (INSERT_PANE, Insert_Enabled);

//	Put the modes panes in the ContentPane.
getContentPane().add (Modes_Panes);

//	Disconnect on close.
addWindowListener (new WindowAdapter ()
	{public void windowClosing (WindowEvent event)
	{
	try {The_Database.Disconnect ();}
	catch (Database_Exception exception)
		{Dialog_Box.Error (exception.getMessage ());}
	}});
pack ();
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("<<< Database_View");
}

private Database_View ()
throws Exception
{
throw new Exception
	(
	getClass().getName() +
	": Hey! You shouldn't be using the default constructor."
	);
}

/*------------------------------------------------------------------------------
	Configure
*/
private void Configure ()
	throws Database_Exception
{
if ((DEBUG & DEBUG_CONFIGURATION) != 0)
	System.out.println (">>> Database_View.Configure");
The_Configuration = The_Database.Data_Port ().Configuration ();

setTitle (The_Configuration.Name ()
	+ " Database at " + The_Configuration.Get_One (Configuration.HOST));

boolean
	case_sensitive = The_Configuration.Case_Sensitive (false);

//	Configuration flags.
Update_Enabled = Data_View.get_flag (The_Configuration,
		UPDATE_ENABLED_PARAMETER, Update_Enabled != UPDATE_DISABLED) ?
			UPDATE_ENABLED : UPDATE_DISABLED;
Insert_Enabled = Data_View.get_flag (The_Configuration,
		INSERT_ENABLED_PARAMETER, Insert_Enabled);

The_Configuration.Case_Sensitive (case_sensitive);
}

/*==============================================================================
	Menus
*/
private JMenuBar Menu_Bar ()
{
JMenuBar menu_bar	= new JMenuBar ();
JMenu view			= new JMenu ("View");
JMenuItem config	= new JMenuItem ("Configuration");
config.setMnemonic ('U');
config.setAccelerator (KeyStroke.getKeyStroke ('U', Event.CTRL_MASK));
config.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{View_Configuration ();}});
view.add (config);
menu_bar.add (view);
return menu_bar;
}

/*==============================================================================
	Menu Actions
*/
//	VIEW the Configuration.
private void View_Configuration ()
{
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println (">>> View_Configuration");
if (Configuration_View == null)
	{
	Configuration_View =
		new Parameter_View
			(
			The_Configuration.Name () + " Database",
			The_Configuration
			);
	Configuration_View
		.addWindowListener (new WindowAdapter ()
			{public void windowClosing (WindowEvent event)
			{Configuration_View = null;}});

	//	Locate the Configuration_View on the screen.
	Configuration_View.setSize (Configuration_View_Size);
	Configuration_View_Locator
		.Offsets    (0, 0)
		.Vertical   (View_Locator.TOP   | View_Locator.INWARD);
	if (Configuration_View_Size.width > getX ())
		Configuration_View_Locator
		.Horizontal (View_Locator.RIGHT  | View_Locator.OUTWARD);
	else
		Configuration_View_Locator
		.Horizontal (View_Locator.LEFT | View_Locator.OUTWARD);
	Configuration_View_Locator.Relocate (Configuration_View, this);
	Configuration_View.setVisible (true);
	}
Configuration_View.toFront ();
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("<<< View_Configuration");
}

/*==============================================================================
	Modes Panels
*/
private JTabbedPane Modes_Panes ()
{
JTabbedPane modes_panes = new JTabbedPane ();

/*..............................................................................
	SELECT
*/
//	Setup the selection panels.
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("--- Selection_Panels");
Selection_Panels = Selection_Panels ();
Selection_Panels.setPreferredSize
	(new Dimension (DEFAULT_VIEW_WIDTH, DEFAULT_SELECTION_PANEL_HEIGHT));
//this.getRootPane ().setDefaultButton (Select_Button);

//	Assemble the field lists.
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("--- Assemble_Fields");
Assemble_Fields ();
Include_Default_Selections ();
Fields_Available_Tree.expandPath
	(new TreePath (Fields_Available_Tree_Model.getRoot ()));
Fields_Chosen_Tree.expandPath
	(new TreePath (Fields_Chosen_Tree_Model.getRoot ()));

//	Prepare the data table panel.
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("--- Data_Table");
Select_Table = new Data_Table ();
Select_Table.Table_Model ().addTableModelListener
	(new Select_Table_Changed ());
Select_Table.setPreferredSize
	(new Dimension (DEFAULT_VIEW_WIDTH, DEFAULT_DATA_TABLE_HEIGHT));

//	Put the selection panel and data table panel in a split panel.
JSplitPane split_pane = new JSplitPane
	(JSplitPane.VERTICAL_SPLIT, Selection_Panels, Select_Table);
split_pane.setOneTouchExpandable (true);
modes_panes.addTab ("Select", SELECT_ICON, split_pane, "Select data");
SELECT_PANE = 0;

/*..............................................................................
	INSERT
*/
//	Setup the insertion panels.
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("--- Insertion_Panels");
Insertion_Panels = Insertion_Panels ();
Insertion_Panels.setPreferredSize
	(new Dimension (DEFAULT_VIEW_WIDTH, DEFAULT_SELECTION_PANEL_HEIGHT));
Insert_Fields_Tree.expandPath
	(new TreePath (Insert_Fields_Tree_Model.getRoot ()));

//	Prepare the data table panel.
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("--- Insert_Table");
Insert_Table = new Data_Table ();
Insert_Table.setPreferredSize
	(new Dimension (DEFAULT_VIEW_WIDTH, DEFAULT_DATA_TABLE_HEIGHT));
Insert_Table.Record_Numbers_Color (INSERT_COLOR);
//	Put the insertion panel and data table panel in a split panel.
split_pane = new JSplitPane
	(JSplitPane.VERTICAL_SPLIT, Insertion_Panels, Insert_Table);
split_pane.setOneTouchExpandable (true);
modes_panes.addTab ("Insert", INSERT_ICON, split_pane, "Insert data");
INSERT_PANE = 1;

modes_panes.setSelectedIndex (SELECT_PANE);
return modes_panes;
}

/*==============================================================================
	SELECT
*/
private JPanel Selection_Panels ()
{
JPanel panels		= new JPanel (new GridBagLayout ());

/*..............................................................................
	Fields panel
*/
JPanel fields_panel	= new JPanel (new GridBagLayout ());
fields_panel.setBorder (new TitledBorder (new EtchedBorder (),
	" Fields ", TitledBorder.LEFT, TitledBorder.TOP));
GridBagConstraints location = new GridBagConstraints ();

//- Row 0
//	Col 0: glue
location.insets		= new Insets (0, 0, 0, 0);
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
fields_panel.add (Box.createHorizontalGlue (), location);

//	Col 1: Available label
location.insets		= new Insets (0, 0, 0, 5);
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
fields_panel.add (new JLabel ("Available"), location);

//	Col 2: Include button
Include_Button		= new JButton ("Include", INCLUDE_ICON);
Include_Button.setHorizontalTextPosition (SwingConstants.LEADING);
Include_Button.setMargin (new Insets (0, 3, 0, 0));
Include_Button.setToolTipText ("Add to fields selections");
Include_Button.setMnemonic ('I');
Include_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Include_Fields ();}});
location.insets		= new Insets (0, 0, 0, 0);
fields_panel.add (Include_Button, location);

//	Col 3: strut (between lists)
location.anchor		= GridBagConstraints.WEST;
fields_panel.add (Box.createHorizontalStrut (5), location);

//	Col 4: Exclude button
Exclude_Button		= new JButton ("Exclude", EXCLUDE_ICON);
Exclude_Button.setMargin (new Insets (0, 0, 0, 3));
Exclude_Button.setToolTipText ("Remove from fields selections");
Exclude_Button.setMnemonic ('E');
Exclude_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Exclude_Fields ();}});
fields_panel.add (Exclude_Button, location);

//	Col 5: Chosen label
location.insets		= new Insets (0, 5, 0, 5);
location.gridwidth	= GridBagConstraints.REMAINDER;
fields_panel.add (new JLabel ("Chosen"), location);

//- Row 1
//	Col 0-2: Source fields tree.
Fields_Available_Tree	= new JTree (Fields_Available_Tree_Model);
((DefaultTreeCellRenderer)Fields_Available_Tree.getCellRenderer ())
	.setLeafIcon (null);
Fields_Available_Tree.setRootVisible (false);
Fields_Available_Tree.setShowsRootHandles (true);
//Fields_Available_Tree.setDragEnabled (true);	// 1.4
ToolTipManager.sharedInstance().registerComponent(Fields_Available_Tree);
((JComponent)Fields_Available_Tree.getCellRenderer ())
	.setToolTipText ("Fields in the database");
JScrollPane source_fields = new JScrollPane (Fields_Available_Tree);
source_fields.setMinimumSize (new Dimension (100, 100));
source_fields.setPreferredSize (new Dimension (LIST_WIDTH, LIST_HEIGHT));
location.insets		= new Insets (0, 5, 5, 0);
location.gridwidth	= 3;
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
fields_panel.add (source_fields, location);

//	Col 3: strut (between lists)
location.insets		= new Insets (0, 0, 5, 0);
location.gridwidth	= 1;
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
location.weighty	= 0.0;
fields_panel.add (Box.createHorizontalStrut (10), location);

//	Col 4-: Selected fields tree.
Fields_Chosen_Tree_Model = new DefaultTreeModel
	(new DefaultMutableTreeNode ("Selected Fields"));
Fields_Chosen_Tree = new JTree (Fields_Chosen_Tree_Model);
((DefaultTreeCellRenderer)Fields_Chosen_Tree.getCellRenderer ())
	.setLeafIcon (null);
Fields_Chosen_Tree.setRootVisible (false);
Fields_Chosen_Tree.setShowsRootHandles (true);
//Fields_Chosen_Tree.setDragEnabled (true);	// 1.4
ToolTipManager.sharedInstance().registerComponent(Fields_Chosen_Tree);
((JComponent)Fields_Chosen_Tree.getCellRenderer ())
	.setToolTipText ("Fields to be selected");
JScrollPane selected_fields = new JScrollPane (Fields_Chosen_Tree);
selected_fields.setMinimumSize (new Dimension (100, 100));
selected_fields.setPreferredSize (new Dimension (LIST_WIDTH, LIST_HEIGHT));
location.insets		= new Insets (0, 0, 5, 5);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
fields_panel.add (selected_fields, location);

location.insets		= new Insets (5, 5, 5, 5);
panels.add (fields_panel, location);

/*..............................................................................
	Conditions panel
*/
JPanel conditions_panel = new JPanel (new GridBagLayout ());
conditions_panel.setBorder (new TitledBorder (new EtchedBorder (),
	" Conditions ", TitledBorder.LEFT, TitledBorder.TOP));

//-	Row 0: Conditions text.
Conditions			= new JTextArea (4, 32);
Conditions.setLineWrap (true);
Conditions.setWrapStyleWord (true);
Conditions.setToolTipText ("Field selection conditions");
JScrollPane conditions_pane = new JScrollPane (Conditions);
conditions_pane.setMinimumSize (new Dimension (200, 50));
location.insets		= new Insets (0, 5, 0, 5);
conditions_panel.add (conditions_pane, location);

//- Row 1
//	Col 0: Joins button.
Join_Button			= new JButton ("Join ...");
Join_Button.setToolTipText ("Join fields of tables");
Join_Button.setMnemonic ('J');
Join_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Join ();}});
location.gridwidth	= 1;
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
location.weighty	= 0.0;
//conditions_panel.add (Join_Button, location);

//	Col 1: Limit field.
Box limits			= Box.createHorizontalBox ();
JLabel limit_label	= new JLabel ("Limit:");
limit_label.setToolTipText ("Maximum records to get");
limits.add (limit_label);
limits.add (Box.createHorizontalStrut (5));
/*
	JSpinner is new with Java 1.4; use it when 1.4 is ubiquitous.
Limit				= new JSpinner (new SpinnerNumberModel
	(new Integer (DEFAULT_LIMIT),	// Initial value
	 new Integer (0),				// Minimum
	 null,							// Maximum (null == unlimited)
	 new Integer (1)));				// Step size
*/
Limit				= new JTextField (6);
Limit.setHorizontalAlignment (JTextField.RIGHT);
Limit.setText (String.valueOf (DEFAULT_LIMIT));
Limit.setMinimumSize (new Dimension (75, 25));
//		How is a JSpinner ToolTip set?
Limit.setToolTipText ("Maximum number of records");
limits.add (Limit);
location.insets		= new Insets (5, 0, 5, 5);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
location.weighty	= 0.0;
conditions_panel.add (limits, location);

location.insets		= new Insets (3, 5, 3, 5);
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
panels.add (conditions_panel, location);

/*..............................................................................
	Buttons panel
*/
//	Col 0: Reset button
JPanel buttons_panel = new JPanel (new GridBagLayout ());
Reset_Button		= new JButton ("Reset");
Reset_Button.setToolTipText ("Reset selections and conditions");
Reset_Button.setMnemonic ('R');
Reset_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Reset_Selections ();}});
location.insets		= new Insets (0, 0, 0, 0);
location.gridwidth	= 1;
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
location.weighty	= 0.0;
buttons_panel.add (Reset_Button, location);

//	Col 1: glue
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
buttons_panel.add (Box.createHorizontalGlue (), location);

//	Col 2: Update button
Update_Button			= new JButton ("Update");
Update_Button.setToolTipText ("Update edited data values");
Update_Button.setMnemonic ('U');
Update_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Update ();}});
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
buttons_panel.add (Update_Button, location);
Update_Button.setEnabled (Update_Enabled != UPDATE_DISABLED);
Normal_Background_Color = Update_Button.getBackground ();

//	Col 1: glue
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
buttons_panel.add (Box.createHorizontalGlue (), location);

//	Col 3: SQL button
SQL_Button			= new JButton ("SQL ...");
SQL_Button.setToolTipText ("Compose custom SQL syntax");
SQL_Button.setMnemonic ('Q');
SQL_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{SQL ();}});
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
buttons_panel.add (SQL_Button, location);

//	Col 4: strut
fields_panel.add (Box.createHorizontalStrut (5), location);

//	Col 5: Select button
Select_Button		= new JButton ("Select");
Select_Button.setToolTipText ("Query the database");
Select_Button.setMnemonic ('S');
Select_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Select ();}});
location.gridwidth	= GridBagConstraints.REMAINDER;
buttons_panel.add (Select_Button, location);

location.insets		= new Insets (0, 7, 5, 7);
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 0.0;
location.weighty	= 0.0;
panels.add (buttons_panel, location);

return panels;
}

/*------------------------------------------------------------------------------
	Fields
*/
private void Assemble_Fields ()
{
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println (">>> Assemble_Fields");
Fields_Available_Tree.clearSelection ();
DefaultMutableTreeNode
	selects = (DefaultMutableTreeNode)Fields_Available_Tree_Model.getRoot (),
	inserts = (DefaultMutableTreeNode)Insert_Fields_Tree_Model.getRoot (),
	selects_catalog,
	inserts_catalog,
	selects_table,
	inserts_table;
selects.removeAllChildren ();
inserts.removeAllChildren ();

TreeSet
	catalogs;
try {catalogs = new TreeSet (The_Database.Catalogs ());}
catch (Database_Exception exception)
	{
	Dialog_Box.Warning ("Unable to access the database catalogs." + NL
		+ NL
		+ ID + NL
		+ exception.getMessage ());
	Fields_Available_Tree_Model.reload ();
	return;
	}
if (catalogs.isEmpty ())
	{
	if ((DEBUG & DEBUG_UI_SELECTION) != 0)
		System.out.println ("<<< Assemble_Fields: no catalogs");
	return;
	}
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("    catalogs: " + catalogs);
Iterator
	catalogs_list = catalogs.iterator ();
while (catalogs_list.hasNext ())
	{
	selects_catalog = new DefaultMutableTreeNode (catalogs_list.next ());
	inserts_catalog = new DefaultMutableTreeNode (Name (selects_catalog));
	TreeSet
		tables;
	try {tables = new TreeSet
			(The_Database.Tables (Name (selects_catalog)));}
	catch (Database_Exception exception)
		{/* Can't access that catalog */ continue;}
	if (tables.isEmpty ())
		//	No accessable tables.
		continue;
	if ((DEBUG & DEBUG_UI_SELECTION) != 0)
		System.out.println
			("    " + Name (selects_catalog) + " tables: " + tables);
	Iterator
		tables_list = tables.iterator ();
	while (tables_list.hasNext ())
		{
		selects_table = new DefaultMutableTreeNode (tables_list.next ());
		inserts_table = new DefaultMutableTreeNode (Name (selects_table));
		Vector
			fields;
		try {fields = The_Database.Field_Names
				(The_Database.Table_Reference
					(Name (selects_catalog), Name (selects_table)));}
		catch (Database_Exception exception)
			{/* Can't access that table */ continue;}
		if (fields.isEmpty ())
			//	No accessable fields.
			continue;
		selects_catalog.add (selects_table);
		inserts_catalog.add (inserts_table);
		if ((DEBUG & DEBUG_UI_SELECTION) != 0)
			System.out.println
				("    " + Name (selects_catalog) + "." + Name (selects_table) +
				" fields: " + fields);
		Iterator
			fields_list = fields.listIterator ();
		String
			field;
		while (fields_list.hasNext ())
			{
			field = (String)fields_list.next ();
			selects_table.add (new DefaultMutableTreeNode (field));
			inserts_table.add (new DefaultMutableTreeNode (field));
			}
		}
	if (selects_catalog.getChildCount () == 0)
		//	Empty catalog.
		continue;
	selects.add (selects_catalog);
	inserts.add (inserts_catalog);
	}
Fields_Available_Tree_Model.reload ();
Insert_Fields_Tree_Model.reload ();
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("<<< Assemble_Fields");
}

private void Include_Default_Selections ()
{
DefaultMutableTreeNode
	root = (DefaultMutableTreeNode)Fields_Available_Tree_Model.getRoot (),
	catalog,
	table,
	field;
TreePath
	selection = new TreePath (root);
catalog = Child (root,
	The_Configuration.Get_Linked_One (Database.CATALOG));
try
	{
	if (catalog != null &&
		The_Database.Catalogs ().contains (Name (catalog)))
		{
		if ((DEBUG & DEBUG_UI_SELECTION) != 0)
			System.out.println
				("  Initial catalog selected: " + Name (catalog));
		selection = selection.pathByAddingChild (catalog);
		table = Child (catalog,
			The_Database.Table_Name (The_Configuration.Get_Linked_One
				(Name (catalog) + The_Configuration.Path_Delimiter ()
				+ Database.TABLE)));
		if (table != null &&
			The_Database.Tables (Name (catalog)).contains (Name (table)))
			{
			if ((DEBUG & DEBUG_UI_SELECTION) != 0)
				System.out.println
					("  Initial table selected: " + Name (table));
			selection = selection.pathByAddingChild (table);
			Vector
				default_fields = The_Configuration.Get_Linked
					(Name (catalog)
					+ The_Configuration.Path_Delimiter ()
					+ The_Database.Table_Name (Name (table))
					+ The_Configuration.Path_Delimiter ()
					+ Database.FIELDS);
			if (default_fields.isEmpty ())
				Fields_Available_Tree.addSelectionPath (selection);
			else
				{
				if ((DEBUG & DEBUG_UI_SELECTION) != 0)
					System.out.println
						("  Initial fields selected: " + default_fields);
				Iterator
					fields_list = default_fields.iterator ();
				while (fields_list.hasNext ())
					if ((field = Child (table, (String)fields_list.next ()))
							!= null)
						Fields_Available_Tree.addSelectionPath
							(selection.pathByAddingChild (field));
				}
			Include_Fields ();
			}
		}
	}
catch (Database_Exception exception)
	{Dialog_Box.Error (exception.getMessage ()); return;}
}

/*==============================================================================
	Selection Actions
*/
//	INCLUDE fields from Available to Chosen.
private void Include_Fields ()
{
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println (">>> Include_Fields");
if (Fields_Available_Tree.isSelectionEmpty ())
	{
	Dialog_Box.Notice ("Select one or more tables and/or fields.");
	return;
	}
TreePath
	selections[] = Fields_Available_Tree.getSelectionPaths (),
	included;
Vector
	expand_paths = new Vector ();
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("    " + selections.length + " selections");
DefaultMutableTreeNode
	root = (DefaultMutableTreeNode)Fields_Chosen_Tree_Model.getRoot ();
for (int index = 0;
	 index < selections.length;
	 index++)
	{
	Object
		segments[] = selections[index].getPath ();
	DefaultMutableTreeNode
		node = (DefaultMutableTreeNode)segments[1];
	String
		name = Name (node);
	DefaultMutableTreeNode
		catalog = Child (root, name),
		table,
		field;
	included = new TreePath (root);
	if ((DEBUG & DEBUG_UI_SELECTION) != 0)
		System.out.println ("    catalog: " + name);
	if (catalog == null &&
		segments.length > 2)	// Ignore catalog selection
		{
		Fields_Chosen_Tree_Model.insertNodeInto
			(catalog = new DefaultMutableTreeNode (name),
			root, root.getChildCount ());
		if ((DEBUG & DEBUG_UI_SELECTION) != 0)
			System.out.println ("           > ADDED");
		}
	if (segments.length > 2)
		{
		included = included.pathByAddingChild (catalog);
		node = (DefaultMutableTreeNode)segments[2];
		if (node.getChildCount () > 0)	// Ignore empty tables.
			{
			name = Name (node);
			if ((DEBUG & DEBUG_UI_SELECTION) != 0)
				System.out.println ("      table: " + name);
			if ((table = Child (catalog, name)) == null)
				{
				Fields_Chosen_Tree_Model.insertNodeInto
					(table = new DefaultMutableTreeNode (name),
					catalog, catalog.getChildCount ());
				if ((DEBUG & DEBUG_UI_SELECTION) != 0)
					System.out.println ("           > ADDED");
				}
			included = included.pathByAddingChild (table);
			if (segments.length > 3)
				{
				name = Name (segments[3]);
				if ((DEBUG & DEBUG_UI_SELECTION) != 0)
					System.out.println ("      field: " + name);
				if ((field = Child (table, name)) == null)
					{
					//	Include the field.
					Fields_Chosen_Tree_Model.insertNodeInto
						(field = new DefaultMutableTreeNode (name),
						table, table.getChildCount ());
					if ((DEBUG & DEBUG_UI_SELECTION) != 0)
						System.out.println ("           > ADDED");
					}
				}
			else
				{
				//	Include all fields of the table.
				Enumeration
					nodes = node.children ();
				while (nodes.hasMoreElements ())
					{
					name = Name (nodes.nextElement ());
					if (Child (table, name) == null)
						table.add (new DefaultMutableTreeNode (name));
					}
				Fields_Chosen_Tree_Model.nodeStructureChanged (table);
				if ((DEBUG & DEBUG_UI_SELECTION) != 0)
					System.out.println ("           > ADDED all fields");
				}
			expand_paths.add (included);
			}
		}
	}
Iterator
	paths = expand_paths.iterator ();
while (paths.hasNext ())
	Fields_Chosen_Tree.expandPath ((TreePath)paths.next ());
if (root.getChildCount () != 0)
	Exclude_Button.setEnabled (true);
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("<<< Include_Fields");
}

//	EXCLUDE fields from Chosen.
private void Exclude_Fields ()
{
DefaultMutableTreeNode
	root = (DefaultMutableTreeNode)Fields_Chosen_Tree_Model.getRoot ();
if (root.getChildCount () == 0)
	return;	// Nothing included.
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println (">>> Exclude_Fields");
if (Fields_Chosen_Tree.isSelectionEmpty ())
	{
	Dialog_Box.Notice ("Select one or more tables and/or fields.");
	return;
	}
TreePath
	selections[] = Fields_Chosen_Tree.getSelectionPaths ();
int
	selection = selections.length;	// Bottom up processing.
while (--selection >= 0)
	{
	DefaultMutableTreeNode
		parent,
		node = (DefaultMutableTreeNode)selections[selection]
			.getLastPathComponent ();
	do
		{
		if ((DEBUG & DEBUG_UI_SELECTION) != 0)
			System.out.println ("    Removing \"" + Name (node) + "\"");
		parent = (DefaultMutableTreeNode)node.getParent ();
		Fields_Chosen_Tree_Model.removeNodeFromParent (node);
		node = parent;
		}
		// Remove non-root empty nodes.
		while (node != root && node.getChildCount () == 0);
	}
if (root.getChildCount () == 0)
	Exclude_Button.setEnabled (false);
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("<<< Exclude_Fields");
}

//	JOIN fields.
private void Join ()
{
}

//	RESET all selections.
private void Reset_Selections ()
{
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println (">>> Reset_Selections");
Fields_Available_Tree.clearSelection ();
Assemble_Fields ();
((DefaultMutableTreeNode)Fields_Chosen_Tree_Model.getRoot ())
	.removeAllChildren ();
Fields_Chosen_Tree_Model.reload ();
Conditions.setText (null);
/*
	Use this with JSpinner.
Limit.setValue (new Integer (DEFAULT_LIMIT));
*/
Limit.setText (String.valueOf (DEFAULT_LIMIT));

if (Update_Enabled > 0)
	{
	//	Restore the edited values.
	Data_Table_Model
		data_table = Select_Table.Table_Model ();
	Vector
		edited_cells = data_table.Edited_Cells ();
	Iterator
		cells = edited_cells.iterator ();
	if ((DEBUG & DEBUG_SELECT_TABLE) != 0)
		System.out.println
			("   Database_View.Reset_Selections: restoring "
			+ edited_cells.size () + " edited cells - ");
	while (cells.hasNext ())
		{
		Cell cell = (Cell)cells.next ();
		data_table.setValueAt (cell.old_value, cell.row, cell.column);
		if ((DEBUG & DEBUG_SELECT_TABLE) != 0)
			System.out.println
				("    " + cell.row + ',' + cell.column + ": "
				+ cell.new_value + '(' + cell.old_value + ')');
		}
	//	Reset all edits.
	data_table.Clear_Edited_Cells ();
	Reset_Update ();
	}
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("<<< Reset_Selections");
}

//	UPDATE edited data.
private void Update ()
{
}

private void Reset_Update ()
{
Update_Button.setBackground (Normal_Background_Color);
if (Update_Enabled != UPDATE_DISABLED)
	Update_Enabled =  UPDATE_ENABLED;
}

//	SELECT the data.
private void Select ()
{
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println (">>> Select");
String
	command = Assemble_Select_Command ();
if (command == null)
	return;

Run_Query (command);
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println ("<<< Select");
}

//	SQL command panel.
private void SQL ()
{
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println (">>> SQL");
String
	command = Current_SQL_Command;
if (command.length () == 0)
	command = Assemble_Select_Command ();
command = SQL.Get_Command ("Select", command, this);
if (command == null)
	return;

Run_Query (command);
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println ("<<< SQL");
}

private void Run_Query
	(
	String	SQL_command
	)
{
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println (">>> Run_Query:" + NL
		+ SQL_command);
final String
	SQL_query = SQL_command;
int
	limit;
/*
	Use this with JSpinner.
try {limit = ((Integer)Limit.getValue ()).intValue ();}
*/
try {limit = new Integer (Limit.getText ()).intValue ();}
catch (NumberFormatException e)
	{
	Dialog_Box.Error ("Use a numeric limit value.");
	Limit.setText (String.valueOf (DEFAULT_LIMIT));
	return;
	}
if (limit == 0)
	limit = -1;	// Unlimited.
final int
	max_records = limit;
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println ("    Limit: " + limit);
Selections_Enabled (false);
final Blinker
	blinking = new Blinker (Select_Button);
blinking.start ();
final Cursor
	cursor = getCursor ();
setCursor (Cursor.getPredefinedCursor (Cursor.WAIT_CURSOR));
Thread
	query = new Thread ()
		{
		private boolean Success = false;
		public void run ()
		{
		try
			{
			String
				command = SQL.Clean (SQL_query);
			if ((DEBUG & DEBUG_SQL) != 0)
				System.out.println ("--- Run_Query:" + NL
					+ command);
			Vector
				table = The_Database.Query (command, max_records);
			Success = true;
			Select_Table.Data (table, Update_Enabled != UPDATE_DISABLED);
			Reset_Update ();
			}
		catch (Database_Exception exception)
			{Dialog_Box.Error (exception.getMessage ());}
		SwingUtilities.invokeLater (new Runnable ()
			{
			public void run ()
			{
			Selections_Enabled (true);
			blinking.stop ();
			setCursor (cursor);
			if (Success)
				Set_Current_SQL_Command (SQL_query);
			}});
		}};
query.start ();
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println ("<<< Run_Query");
}

private String Assemble_Select_Command ()
{
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println (">>> Assemble_Select_Command");
Vector
	tables = Chosen_Tables (),
	fields = Chosen_Fields ();
String
	conditions;
Words
	conditions_words = new Words (SQL.Clean (Conditions.getText ()));
if (conditions_words.Next_Word ().toUpperCase ().equals ("WHERE"))
	conditions = conditions_words.Substring (conditions_words.End_Index);
else
	conditions = conditions_words.toString ();
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println 
		(
		"    Fields: " + fields + NL +
		"    Tables: " + tables + NL +
		"    Conditions: " + conditions
		);
//	Get the SQL command line from the Database.
String
	command = null;
try {command =
		((String)The_Database.Select (tables, fields, conditions, 0)
			.firstElement ());}
catch (Database_Exception e) {}

if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println ("<<< Assemble_Select_Command:" + NL
		+ command);
return command;
}

private void Set_Current_SQL_Command
	(
	String	command
	)
{
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println ("--- Set_Current_SQL_Command:" + NL
		+ command);
if (command == null)
	Current_SQL_Command = "";
else
	Current_SQL_Command = command;
}

//	Enable or disable all selection buttons.
private void Selections_Enabled
	(
	boolean	enable
	)
{
Include_Button.setEnabled (enable);
if (((DefaultMutableTreeNode)Fields_Chosen_Tree_Model.getRoot ())
		.getChildCount () != 0)
	Exclude_Button.setEnabled (enable);
Reset_Button.setEnabled (enable);
SQL_Button.setEnabled (enable);
Select_Button.setEnabled (enable);
}

class Select_Table_Changed
	implements TableModelListener
{
public void tableChanged
	(
	TableModelEvent	event
	)
{
if ((DEBUG & DEBUG_SELECT_TABLE) != 0)
	{
	System.out.println ("--- Database_View.Select_Table_Changed:");
	switch (event.getType ())
		{
		case TableModelEvent.INSERT:
			System.out.print ("INSERT"); break;
		case TableModelEvent.UPDATE:
			System.out.print ("UPDATE"); break;
		case TableModelEvent.DELETE:
			System.out.print ("DELETE"); break;
		default:
			System.out.print ("Type " + event.getType ());
		}
	System.out.println
		(": rows " + event.getFirstRow () + " to " + event.getLastRow ()
		+", column " + event.getColumn ());
	}
if (Update_Enabled == UPDATE_ENABLED &&
	event.getType () == TableModelEvent.UPDATE &&
	event.getColumn () >= 0 &&
	event.getFirstRow () >= 0 &&
	event.getFirstRow () == event.getLastRow ())
	{
	Update_Button.setBackground (Select_Table.Edited_Cell_Color ());
	Update_Enabled = UPDATE_PENDING;
	}
}
}	//	End of class Select_Table_Changed.

/*==============================================================================
	INSERT
*/
private JPanel Insertion_Panels ()
{
JPanel panels		= new JPanel (new GridBagLayout ());

/*..............................................................................
	Fields panel
*/
JPanel fields_panel	= new JPanel (new GridBagLayout ());
fields_panel.setBorder (new TitledBorder (new EtchedBorder (),
	" Fields ", TitledBorder.LEFT, TitledBorder.TOP));
GridBagConstraints location = new GridBagConstraints ();

//-	Row 0
//	Col 0-1: Source fields tree.
Insert_Fields_Tree	= new JTree (Insert_Fields_Tree_Model);
((DefaultTreeCellRenderer)Insert_Fields_Tree.getCellRenderer ())
	.setLeafIcon (null);
Insert_Fields_Tree.setRootVisible (false);
Insert_Fields_Tree.setShowsRootHandles (true);
Insert_Fields_Tree.getSelectionModel ()
	.setSelectionMode (TreeSelectionModel.SINGLE_TREE_SELECTION);
ToolTipManager.sharedInstance().registerComponent(Insert_Fields_Tree);
((JComponent)Insert_Fields_Tree.getCellRenderer ())
	.setToolTipText ("Database tables available for inserts");
JScrollPane source_fields = new JScrollPane (Insert_Fields_Tree);
source_fields.setMinimumSize (new Dimension (100, 100));
source_fields.setPreferredSize (new Dimension (LIST_WIDTH, LIST_HEIGHT));
location.insets		= new Insets (0, 5, 0, 0);
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.BOTH;
location.gridwidth	= 2;
location.weightx	= 1.0;
location.weighty	= 1.0;
fields_panel.add (source_fields, location);

//	Col 1: strut (between lists)
location.insets		= new Insets (0, 0, 0, 0);
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.weighty	= 0.0;
fields_panel.add (Box.createHorizontalStrut (10), location);

//	Col 1: box (corresponds to second list)
location.insets		= new Insets (0, 0, 0, 5);
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.BOTH;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.weighty	= 1.0;
fields_panel.add (new Box.Filler
	(new Dimension (100, 100),
	new Dimension (LIST_WIDTH, LIST_HEIGHT),
	new Dimension (100, 100)),
	location);

//-	Row 1
//	Col 0: To Table button
Insert_into_Button	= new JButton ("To Table", INSERTION_ICON);
Insert_into_Button.setMargin (new Insets (0, 0, 0, 3));
Insert_into_Button.setToolTipText ("Select the table to insert data");
Insert_into_Button.setMnemonic ('T');
Insert_into_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Insert_into_Table ();}});
location.insets		= new Insets (0, 5, 5, 0);
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.weighty	= 0.0;
fields_panel.add (Insert_into_Button, location);

//	Col 1: Chosen table label
Insertion_Table		= new JLabel ();
location.insets		= new Insets (0, 5, 5, 5);
location.fill		= GridBagConstraints.HORIZONTAL;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
fields_panel.add (Insertion_Table, location);

location.insets		= new Insets (5, 5, 3, 5);
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
panels.add (fields_panel, location);

/*..............................................................................
	Buttons panel
*/
JPanel buttons_panel = new JPanel (new GridBagLayout ());

//	Col 0: Clear button
Clear_Button		= new JButton ("Clear");
Clear_Button.setEnabled (false);
Clear_Button.setToolTipText ("Clear pending inserts");
Clear_Button.setMnemonic ('C');
Clear_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Clear_Inserts ();}});
location.insets		= new Insets (0, 0, 0, 0);
location.gridwidth	= 1;
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
location.weighty	= 0.0;
buttons_panel.add (Clear_Button, location);

//	Col 1: glue
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
buttons_panel.add (Box.createHorizontalGlue (), location);

//	Col 2: New button
New_Button			= new JButton ("New");
New_Button.setEnabled (false);
New_Button.setToolTipText ("New record");
New_Button.setMnemonic ('N');
New_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{New_Record ();}});
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
buttons_panel.add (New_Button, location);

//	Col 3: strut
fields_panel.add (Box.createHorizontalStrut (5), location);

//	Col 4: Insert button
Insert_Button		= new JButton ("Insert");
Insert_Button.setEnabled (false);
Insert_Button.setToolTipText ("Insert the records");
Insert_Button.setMnemonic ('I');
Insert_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Insert ();}});
location.gridwidth	= GridBagConstraints.REMAINDER;
buttons_panel.add (Insert_Button, location);

location.insets		= new Insets (0, 7, 5, 7);
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 0.0;
location.weighty	= 0.0;
panels.add (buttons_panel, location);

return panels;
}

/*==============================================================================
	Insertion Actions
*/
private void Insert_into_Table ()
{
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println (">>> Insert_into_Table");
if (Insert_Fields_Tree.isSelectionEmpty ())
	{
	Dialog_Box.Notice ("Select a table.");
	return;
	}
//	Assemble the table reference.
Object
	path_segments[] = Insert_Fields_Tree.getSelectionPath ().getPath ();
String
	table = null;
if (path_segments.length > 2)
	{
	try {table = The_Database.Table_Reference
			(Name (path_segments[1]), Name (path_segments[2]));}
	catch (Database_Exception exception) {/* Won't happen */}
	if ((DEBUG & DEBUG_UI_SELECTION) != 0)
		System.out.println ("    table: " + table);
	}
else
	{
	Dialog_Box.Notice ("Select a table.");
	return;
	}

//	Get the table's field names.
Vector
	field_names;
try {field_names = The_Database.Field_Names (table);}
catch (Database_Exception exception)
	{
	Dialog_Box.Notice ("Table " + table + " is not accessable.");
	return;
	}

//	Provide an initial empty record in the insert table.
Vector
	data = new Vector (1);
data.add (Empty_Record (field_names.size ()));
Insert_Table.Data (data, field_names, true);

//	Post the table reference.
Insertion_Table.setText (table);

//	Enable the buttons.
Clear_Button.setEnabled (true);
New_Button.setEnabled (true);
Insert_Button.setEnabled (true);
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("<<< Insert_into_Table");
}

private void Insert ()
{
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println (">>> Insert");
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("<<< Insert");
}

private void New_Record ()
{
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println (">>> New_Record");
Data_Table_Model
	data_table = Insert_Table.Table_Model ();
data_table.addRow (Empty_Record (data_table.getColumnCount ()));
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("<<< New_Record");
}

private void Clear_Inserts ()
{
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println (">>> Reset_Selections");
Insert_Table.Table_Model ().setRowCount (0);
if ((DEBUG & DEBUG_UI_SELECTION) != 0)
	System.out.println ("<<< Reset_Selections");
}

private Vector Empty_Record
	(
	int		size
	)
{
Vector
	record = new Vector (size);
while (size-- != 0)
	record.add ("");
return record;
}

/*==============================================================================
	Helpers
*/
//	Gets the list of all Chosen tables.
private Vector Chosen_Tables ()
{
Vector
	table_names = new Vector ();
DefaultMutableTreeNode
	root = (DefaultMutableTreeNode)Fields_Chosen_Tree_Model.getRoot ();
if (root.getChildCount () == 0)
	return table_names;
Enumeration
	catalogs = root.children ();
while (catalogs.hasMoreElements ())
	{
	DefaultMutableTreeNode
		catalog = (DefaultMutableTreeNode)catalogs.nextElement ();
	Enumeration
		tables = catalog.children ();
	try {
	while (tables.hasMoreElements ())
		table_names.add (The_Database.Table_Reference
			(Name (catalog),
			(Name ((DefaultMutableTreeNode)tables.nextElement ()))));
		}
	catch (Database_Exception exception) {/* Won't happen */}
	}
return table_names;
}

//	Gets the list of all Chosen fields.
private Vector Chosen_Fields ()
{
Vector
	field_names = new Vector ();
DefaultMutableTreeNode
	root = (DefaultMutableTreeNode)Fields_Chosen_Tree_Model.getRoot ();
if (root.getChildCount () == 0)
	return field_names;
Enumeration
	catalogs;
//	Check for catalogs or tables which requires field name qualification.
boolean
	qualify = false;
if (root.getChildCount () > 1)
	qualify = true;
else
	{
	catalogs = root.children ();
	while (catalogs.hasMoreElements ())
		{
		if (((DefaultMutableTreeNode)catalogs.nextElement ())
			.getChildCount () > 1)
			{
			qualify = true;
			break;
			}
		}
	}
catalogs = root.children ();
while (catalogs.hasMoreElements ())
	{
	DefaultMutableTreeNode
		catalog = (DefaultMutableTreeNode)catalogs.nextElement ();
	Enumeration
		tables = catalog.children ();
	while (tables.hasMoreElements ())
		{
		DefaultMutableTreeNode
			table = (DefaultMutableTreeNode)tables.nextElement ();
		String
			table_name = "";
		if (qualify)
			{
			try {table_name = The_Database.Table_Reference
				(Name (catalog), Name (table)) + ".";}
			catch (Database_Exception exception) {/* Won't happen */}
			}
		Enumeration
			fields = table.children ();
		while (fields.hasMoreElements ())
			field_names.add (table_name
				+ Name ((DefaultMutableTreeNode)fields.nextElement ()));
		}
	}
return field_names;
}

//	Finds a named child of a parent.
private static DefaultMutableTreeNode Child
	(
	DefaultMutableTreeNode	parent,
	String					child_name
	)
{
if (child_name != null)
	{
	if ((DEBUG & DEBUG_UTILITY) != 0)
		System.out.print
			("--> Child: " + child_name + " of " + parent.getUserObject ());
	DefaultMutableTreeNode
		child;
	Enumeration
		nodes = parent.children ();
	while (nodes.hasMoreElements ())
		{
		child = (DefaultMutableTreeNode)nodes.nextElement ();
		if (Name (child).equals (child_name))
			{
			if ((DEBUG & DEBUG_UTILITY) != 0)
				System.out.println (" found.");
			return child;
			}
		}
	}
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println (" not found.");
return null;
}

//	Gets the name of a tree node.
private static String Name
	(
	Object	node
	)
{return (String)((DefaultMutableTreeNode)node).getUserObject ();}

private static void Load_Icons
	(
	Object	object
	)
{
if (SELECT_ICON != null)
	return;
char
	file_separator = System.getProperty ("file.separator").charAt (0);
String
	package_name = object.getClass ().getName ();
package_name = package_name.substring (0, package_name.lastIndexOf ('.'))
		.replace ('.', file_separator)
		+ file_separator + "Icons" + file_separator;
URL
	iconURL;
iconURL = ClassLoader.getSystemResource (package_name + SELECT_ICON_NAME);
if (iconURL != null)
	SELECT_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + INSERT_ICON_NAME);
if (iconURL != null)
	INSERT_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + INCLUDE_ICON_NAME);
if (iconURL != null)
	INCLUDE_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + EXCLUDE_ICON_NAME);
if (iconURL != null)
	EXCLUDE_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + INSERTION_ICON_NAME);
if (iconURL != null)
	INSERTION_ICON = new ImageIcon (iconURL);
}

}	//	End of Database_View class.
