/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * Object used to handle the OpenGL texturing. Depending on the color space of
 * the stored buffer and the capabilities of the GPU, different paths can be
 * used. The plugin supports YCbCr planar FOURCC such as I420 and YV12 through
 * OpenGL fragment programs. If the hardware supports pixel-shaders and
 * multi-texturing with more than 3 texture image units, a fragment program
 * using a texture for each plane is bound. There is at the moment no color
 * space conversion programs for packed FOURCCs. If the hardware does not
 * provide pixel-shaders, the plugin doesn't support YCbCr color spaces.
 *
 * FIXME: The behaviour is currently undefined if the user sends us an image
 *        with an unsupported color space (ie a pixel format which is not
 *        returned by pgm_viewport_get_pixel_formats).
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>     /* memset */
#include "pgmtexture.h"

GST_DEBUG_CATEGORY_EXTERN (pgm_gl_debug);
#define GST_CAT_DEFAULT pgm_gl_debug

/* Texture upload function prototypes */
static void do_rgb_upload       (PgmTexture *texture, void *buffer);
static void do_bgr_upload       (PgmTexture *texture, void *buffer);
static void do_rgba_upload      (PgmTexture *texture, void *buffer);
static void do_bgra_upload      (PgmTexture *texture, void *buffer);
static void do_planar_12_upload (PgmTexture *texture, void *buffer);
static void do_packed_16_upload (PgmTexture *texture, void *buffer);

/* Texture create function prototypes */
static void do_rgb_create       (PgmTexture *texture);
static void do_bgr_create       (PgmTexture *texture);
static void do_rgba_create      (PgmTexture *texture);
static void do_bgra_create      (PgmTexture *texture);
static void do_planar_12_create (PgmTexture *texture);
static void do_packed_16_create (PgmTexture *texture);

/* Texture bind function prototypes */
static void do_rgb_bind         (PgmTexture *texture);
static void do_i420_bind        (PgmTexture *texture);
static void do_yv12_bind        (PgmTexture *texture);
static void do_uyvy_bind        (PgmTexture *texture);

/* Texture unbind function prototypes */
static void do_rgb_unbind       (PgmTexture *texture);
static void do_planar_12_unbind (PgmTexture *texture);
static void do_uyvy_unbind      (PgmTexture *texture);

/* Texturing task function pointers
 * FIXME: 4kb lost for 128b effectively used... */
static PgmTextureCreateFunc create_func[256] = { NULL };
static PgmTextureUploadFunc upload_func[256] = { NULL };
static PgmTextureBindFunc bind_func[256] = { NULL };
static PgmTextureBindFunc unbind_func[256] = { NULL };

/* Texture border colors */
static gfloat rgb_border_color[4] = { 0.0f, 0.0f, 0.0f, 0.0f };
static gfloat y_border_color[4] = { 0.0f, 0.0f, 0.0f, 0.0f };
static gfloat cbcr_border_color[4] = { 0.5f, 0.5f, 0.5f, 0.0f };

/* Keep track of the context and OpenGL procedures */
static const PgmContextProcAddress *gl = NULL;
static const PgmContext *context = NULL;

/* Private functions  */

/* Returns number rounded up to the next power of 2. This is the branch-free
 * implementation detailed in "Hacker's Delight" by Henry S. Warren, Jr. */
static guint
get_upper_power_of_two (guint number)
{
  number = number - 1;
  number = number | (number >> 1);
  number = number | (number >> 2);
  number = number | (number >> 4);
  number = number | (number >> 8);
  number = number | (number >> 16);
  return number + 1;
}

/* Specifies texture parameters of the currently bound texture */
static void
set_texture_parameters (PgmTexture *texture,
                        gfloat *border_color)
{
  /* Texture border color */
  if (border_color)
    gl->tex_parameter_fv (PGM_GL_TEXTURE_2D, PGM_GL_TEXTURE_BORDER_COLOR,
                          border_color);

  /* Texture wrapping */
  gl->tex_parameter_i (PGM_GL_TEXTURE_2D, PGM_GL_TEXTURE_WRAP_S,
                       texture->wrap_s);
  gl->tex_parameter_i (PGM_GL_TEXTURE_2D, PGM_GL_TEXTURE_WRAP_T,
                       texture->wrap_t);

  /* Texture filtering */
  gl->tex_parameter_i (PGM_GL_TEXTURE_2D, PGM_GL_TEXTURE_MIN_FILTER,
                       texture->filter);
  gl->tex_parameter_i (PGM_GL_TEXTURE_2D, PGM_GL_TEXTURE_MAG_FILTER,
                       texture->filter);
}

/* 24 bits RGB texture creation
 * Stored as RGBA so that the part of the texture not used (pot_size - size)
 * is opaque. */
static void
do_rgb_create (PgmTexture *texture)
{
  guint8 *pixels = g_malloc0 (texture->width_pot * texture->height_pot * 4);

  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_image_2d (PGM_GL_TEXTURE_2D, 0, PGM_GL_RGBA,
                    texture->width_pot, texture->height_pot, 0,
                    PGM_GL_RGBA, PGM_GL_UNSIGNED_BYTE, pixels);
  set_texture_parameters (texture, rgb_border_color);

  g_free (pixels);
}

/* 24 bits BGR texture creation.
 * Stored as BGRA so that the part of the texture not used (pot_size - size)
 * is opaque. */
static void
do_bgr_create (PgmTexture *texture)
{
  guint8 *pixels = g_malloc0 (texture->width_pot * texture->height_pot * 4);

  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_image_2d (PGM_GL_TEXTURE_2D, 0, PGM_GL_RGBA,
                    texture->width_pot, texture->height_pot, 0,
                    PGM_GL_BGRA, PGM_GL_UNSIGNED_BYTE, pixels);
  set_texture_parameters (texture, rgb_border_color);

  g_free (pixels);
}

/* 32 bits RGBA texture creation */
static void
do_rgba_create (PgmTexture *texture)
{
  guint8 *pixels = g_malloc0 (texture->width_pot * texture->height_pot * 4);

  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_image_2d (PGM_GL_TEXTURE_2D, 0, PGM_GL_RGBA,
                    texture->width_pot, texture->height_pot, 0,
                    PGM_GL_RGBA, PGM_GL_UNSIGNED_BYTE, pixels);
  set_texture_parameters (texture, rgb_border_color);

  g_free (pixels);
}

/* 32 bits BGRA texture creation */
static void
do_bgra_create (PgmTexture *texture)
{
  guint8 *pixels = g_malloc0 (texture->width_pot * texture->height_pot * 4);

  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_image_2d (PGM_GL_TEXTURE_2D, 0, PGM_GL_RGBA,
                    texture->width_pot, texture->height_pot, 0,
                    PGM_GL_BGRA, PGM_GL_UNSIGNED_BYTE, pixels);
  set_texture_parameters (texture, rgb_border_color);

  g_free (pixels);
}

/* 3 * 8 bits LUMINANCE textures creation to store 12 bits planar YCbCr FourCC
 * formats such as I420 and YV12. Creates 3 textures for respectively the Y,
 * Cb and Cr planes. Fragment program is then used to convert the colorspace. */
static void
do_planar_12_create (PgmTexture *texture)
{
  guint lower_width_pot = texture->width_pot >> 1;
  guint lower_height_pot = texture->height_pot >> 1;
  guint8 *y_pixels, *cbcr_pixels;
  guint size;

  /* Get the correct strides and Cb/Cr plane offsets */
  texture->cbcr_height = texture->height / 2;
  texture->y_stride = GST_ROUND_UP_4 (texture->width);
  texture->cbcr_stride = GST_ROUND_UP_8 (texture->width) / 2;
  texture->cb_offset = texture->y_stride * GST_ROUND_UP_2 (texture->height);
  texture->cr_offset = texture->cb_offset +
    texture->cbcr_stride * GST_ROUND_UP_2 (texture->height) / 2;

  /* Initialization pixels area, the area is initialized with black pixels in
   * the YCbCr color space: (0,0,0)RGB == (16,128,128)YCbCr. We replace in our
   * specific case the Y component which should be 16 by 0. */
  y_pixels = g_malloc0 (texture->width_pot * texture->height_pot);
  size = lower_width_pot * lower_height_pot;
  cbcr_pixels = g_malloc (size);
  memset (cbcr_pixels, 128, size);

  /* Y plane creation */
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_image_2d (PGM_GL_TEXTURE_2D, 0, PGM_GL_LUMINANCE,
                    texture->width_pot, texture->height_pot, 0,
                    PGM_GL_LUMINANCE, PGM_GL_UNSIGNED_BYTE, y_pixels);
  set_texture_parameters (texture, y_border_color);

  /* Cb plane creation */
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[1]);
  gl->tex_image_2d (PGM_GL_TEXTURE_2D, 0, PGM_GL_LUMINANCE,
                    lower_width_pot, lower_height_pot, 0,
                    PGM_GL_LUMINANCE, PGM_GL_UNSIGNED_BYTE, cbcr_pixels);
  set_texture_parameters (texture, cbcr_border_color);

  /* Cr plane creation */
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[2]);
  gl->tex_image_2d (PGM_GL_TEXTURE_2D, 0, PGM_GL_LUMINANCE,
                    lower_width_pot , lower_height_pot, 0,
                    PGM_GL_LUMINANCE, PGM_GL_UNSIGNED_BYTE, cbcr_pixels);
  set_texture_parameters (texture, cbcr_border_color);

  /* Free initialization pixels area */
  g_free (y_pixels);
  g_free (cbcr_pixels);
}

/* FIXME */
static void
do_packed_16_create (PgmTexture *texture)
{
}

/* 24 bits RGB texture upload */
static void
do_rgb_upload (PgmTexture *texture,
               void *buffer)
{
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_sub_image_2d (PGM_GL_TEXTURE_2D, 0, 0, 0, texture->width,
                        texture->height, PGM_GL_RGB, PGM_GL_UNSIGNED_BYTE,
                        buffer);
}

/* 24 bits BGR texture upload */
static void
do_bgr_upload (PgmTexture *texture,
               void *buffer)
{
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_sub_image_2d (PGM_GL_TEXTURE_2D, 0, 0, 0, texture->width,
                        texture->height, PGM_GL_BGR, PGM_GL_UNSIGNED_BYTE,
                        buffer);
}

/* 32 bits RGBA texture upload */
static void
do_rgba_upload (PgmTexture *texture,
                void *buffer)
{
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_sub_image_2d (PGM_GL_TEXTURE_2D, 0, 0, 0, texture->width,
                        texture->height, PGM_GL_RGBA, PGM_GL_UNSIGNED_BYTE,
                        buffer);
}

/* 32 bits BGRA texture upload */
static void
do_bgra_upload (PgmTexture *texture,
                void *buffer)
{
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_sub_image_2d (PGM_GL_TEXTURE_2D, 0, 0, 0, texture->width,
                        texture->height, PGM_GL_BGRA, PGM_GL_UNSIGNED_BYTE,
                        buffer);
}

/* 12 bits planar YCbCr upload */
static void
do_planar_12_upload (PgmTexture *texture,
                     void *buffer)
{
  /* Y plane */
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  gl->tex_sub_image_2d (PGM_GL_TEXTURE_2D, 0, 0, 0, texture->y_stride,
                        texture->height, PGM_GL_LUMINANCE, PGM_GL_UNSIGNED_BYTE,
                        buffer);

  /* Cb plane */
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[1]);
  gl->tex_sub_image_2d (PGM_GL_TEXTURE_2D, 0, 0, 0, texture->cbcr_stride,
                        texture->cbcr_height, PGM_GL_LUMINANCE,
                        PGM_GL_UNSIGNED_BYTE,
                        ((guint8*) buffer) + texture->cb_offset);

  /* Cr plane */
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[2]);
  gl->tex_sub_image_2d (PGM_GL_TEXTURE_2D, 0, 0, 0, texture->cbcr_stride,
                        texture->cbcr_height, PGM_GL_LUMINANCE,
                        PGM_GL_UNSIGNED_BYTE,
                        ((guint8*) buffer) + texture->cr_offset);
}

/* FIXME */
static void
do_packed_16_upload (PgmTexture *texture,
                     void *buffer)
{
}

/* Simple texture name binding */
static void
do_rgb_bind (PgmTexture *texture)
{
  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);

  if (texture->storage == PGM_TEXTURE_SYSTEM_BUFFER)
    pgm_backend_bind_system_buffer_object (context->backend,
                                           texture->data.systembuffer);
}

/* Multi-texture and fragment program binding for the I420 to RGB color
 * space conversion fragment program */
static void
do_i420_bind (PgmTexture *texture)
{
  if (context->feature_mask & PGM_GL_FEAT_PER_PLANE_YCBCR_PROGRAM)
    {
      /* Bind the I420 to RGB color space conversion program */
      pgm_program_bind (pgm_program_get (PGM_PROGRAM_PER_PLANE_I420_RGB));

      /* And bind the texture names to the three texture units */
      gl->active_texture (PGM_GL_TEXTURE0);
      gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
      gl->active_texture (PGM_GL_TEXTURE1);
      gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[1]);
      gl->active_texture (PGM_GL_TEXTURE2);
      gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[2]);
    }
}

/* Multi-texture and fragment program binding for the YV12 to RGB color
 * space conversion fragment program */
static void
do_yv12_bind (PgmTexture *texture)
{
  if (context->feature_mask & PGM_GL_FEAT_PER_PLANE_YCBCR_PROGRAM)
    {
      /* Bind the YV12 to RGB color space conversion program */
      pgm_program_bind (pgm_program_get (PGM_PROGRAM_PER_PLANE_YV12_RGB));

      /* And bind the texture names to the three texture units */
      gl->active_texture (PGM_GL_TEXTURE0);
      gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
      gl->active_texture (PGM_GL_TEXTURE1);
      gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[1]);
      gl->active_texture (PGM_GL_TEXTURE2);
      gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[2]);
    }
}

/* FIXME */
static void
do_uyvy_bind (PgmTexture *texture)
{
}

/* Disable the texture binding the default one */
static void
do_rgb_unbind (PgmTexture *texture)
{
  if (texture->storage == PGM_TEXTURE_SYSTEM_BUFFER)
    pgm_backend_release_system_buffer_object (context->backend,
                                              texture->data.systembuffer);

  gl->bind_texture (PGM_GL_TEXTURE_2D, 0);
}

/* Disable fragment program and multi-texturing */
static void
do_planar_12_unbind (PgmTexture *texture)
{
  /* Unbind the conversion program */
  pgm_program_unbind ();

  /* And disable texture units */
  gl->active_texture (PGM_GL_TEXTURE2);
  gl->bind_texture (PGM_GL_TEXTURE_2D, 0);
  gl->active_texture (PGM_GL_TEXTURE1);
  gl->bind_texture (PGM_GL_TEXTURE_2D, 0);
  gl->active_texture (PGM_GL_TEXTURE0);
  gl->bind_texture (PGM_GL_TEXTURE_2D, 0);
}

/* FIXME */
static void
do_uyvy_unbind (PgmTexture *texture)
{
}

/* Update the normalized size fields */
static void
update_normalized_size (PgmTexture *texture)
{
  texture->norm_width = (gfloat) texture->width / texture->width_pot;
  texture->norm_height = (gfloat) texture->height / texture->height_pot;
  texture->inv_norm_width = 1.0f / texture->norm_width;
  texture->inv_norm_height = 1.0f / texture->norm_height;
}

/* Frees the current stored buffer if any */
static void
free_buffer (PgmTexture *texture)
{
  switch (texture->storage)
    {
    case PGM_TEXTURE_GST_BUFFER:
      if (texture->data.gstbuffer)
        {
          gst_buffer_unref (texture->data.gstbuffer);
          texture->data.gstbuffer = NULL;
        }
      break;

    case PGM_TEXTURE_PIXBUF:
      if (texture->data.pixbuf)
        {
          gdk_pixbuf_unref (texture->data.pixbuf);
          texture->data.pixbuf = NULL;
        }
      break;

    case PGM_TEXTURE_BUFFER:
      if (texture->data.buffer)
        {
          if (!texture->shared)
            g_free (texture->data.buffer);
          texture->data.buffer = NULL;
        }
      break;

    case PGM_TEXTURE_SYSTEM_BUFFER:
      if (texture->data.systembuffer)
        {
          pgm_backend_destroy_system_buffer_object (context->backend,
                                                    texture->data.systembuffer);
          texture->data.systembuffer = NULL;
        }
      break;

    default:
      break;
    }

  texture->storage = PGM_TEXTURE_CLEAN;
}

/* Constructor */
static void
init_texture (PgmTexture *texture)
{
  /* Content */
  texture->storage = PGM_TEXTURE_CLEAN;
  texture->data.gstbuffer = NULL;
  texture->width = -1;
  texture->height = -1;
  texture->width_pot = -1;
  texture->height_pot = -1;
  texture->stride = -1;
  texture->size = -1;
  texture->norm_width = -1.0f;
  texture->norm_height = -1.0f;
  texture->inv_norm_width = -1.0f;
  texture->inv_norm_height = -1.0f;

  /* Parameters */
  texture->matrix = pgm_mat4x4_new_identity ();
  texture->filter = PGM_GL_LINEAR;
  texture->wrap_s = PGM_GL_CLAMP_TO_EDGE;
  texture->wrap_t = PGM_GL_CLAMP_TO_EDGE;

  /* State flags */
  texture->flags = PGM_TEXTURE_IDENTITY_MATRIX;

  /* Indentifiants */
  texture->count = 0;
  texture->id = NULL;

  texture->shared = 0;
}

/* Destructor */
static void
dispose_texture (PgmTexture *texture)
{
  free_buffer (texture);

  if (texture->id)
    pgm_texture_clean (texture);

  pgm_mat4x4_free (texture->matrix);
  texture->matrix = NULL;
}

/* Global pointers initialization */
static void
class_init (PgmContext *_context)
{
  /* Create color space indexed function pointers */
  create_func[PGM_IMAGE_RGB] = GST_DEBUG_FUNCPTR (do_rgb_create);
  create_func[PGM_IMAGE_BGR] = GST_DEBUG_FUNCPTR (do_bgr_create);
  create_func[PGM_IMAGE_RGBA] = GST_DEBUG_FUNCPTR (do_rgba_create);
  create_func[PGM_IMAGE_BGRA] = GST_DEBUG_FUNCPTR (do_bgra_create);
  create_func[PGM_IMAGE_I420] = GST_DEBUG_FUNCPTR (do_planar_12_create);
  create_func[PGM_IMAGE_YV12] = GST_DEBUG_FUNCPTR (do_planar_12_create);
  create_func[PGM_IMAGE_UYVY] = GST_DEBUG_FUNCPTR (do_packed_16_create);
  create_func[PGM_IMAGE_YUYV] = GST_DEBUG_FUNCPTR (do_packed_16_create);

  /* Upload color space indexed function pointers */
  upload_func[PGM_IMAGE_RGB] = GST_DEBUG_FUNCPTR (do_rgb_upload);
  upload_func[PGM_IMAGE_BGR] = GST_DEBUG_FUNCPTR (do_bgr_upload);
  upload_func[PGM_IMAGE_RGBA] = GST_DEBUG_FUNCPTR (do_rgba_upload);
  upload_func[PGM_IMAGE_BGRA] = GST_DEBUG_FUNCPTR (do_bgra_upload);
  upload_func[PGM_IMAGE_I420] = GST_DEBUG_FUNCPTR (do_planar_12_upload);
  upload_func[PGM_IMAGE_YV12] = GST_DEBUG_FUNCPTR (do_planar_12_upload);
  upload_func[PGM_IMAGE_UYVY] = GST_DEBUG_FUNCPTR (do_packed_16_upload);
  upload_func[PGM_IMAGE_YUYV] = GST_DEBUG_FUNCPTR (do_packed_16_upload);

  /* Bind color space indexed function pointers */
  bind_func[PGM_IMAGE_RGB] = GST_DEBUG_FUNCPTR (do_rgb_bind);
  bind_func[PGM_IMAGE_BGR] = GST_DEBUG_FUNCPTR (do_rgb_bind);
  bind_func[PGM_IMAGE_RGBA] = GST_DEBUG_FUNCPTR (do_rgb_bind);
  bind_func[PGM_IMAGE_BGRA] = GST_DEBUG_FUNCPTR (do_rgb_bind);
  bind_func[PGM_IMAGE_I420] = GST_DEBUG_FUNCPTR (do_i420_bind);
  bind_func[PGM_IMAGE_YV12] = GST_DEBUG_FUNCPTR (do_yv12_bind);
  bind_func[PGM_IMAGE_UYVY] = GST_DEBUG_FUNCPTR (do_uyvy_bind);
  bind_func[PGM_IMAGE_YUYV] = GST_DEBUG_FUNCPTR (do_uyvy_bind);

  /* Unbind color space indexed function pointers */
  unbind_func[PGM_IMAGE_RGB] = GST_DEBUG_FUNCPTR (do_rgb_unbind);
  unbind_func[PGM_IMAGE_BGR] = GST_DEBUG_FUNCPTR (do_rgb_unbind);
  unbind_func[PGM_IMAGE_RGBA] = GST_DEBUG_FUNCPTR (do_rgb_unbind);
  unbind_func[PGM_IMAGE_BGRA] = GST_DEBUG_FUNCPTR (do_rgb_unbind);
  unbind_func[PGM_IMAGE_I420] = GST_DEBUG_FUNCPTR (do_planar_12_unbind);
  unbind_func[PGM_IMAGE_YV12] = GST_DEBUG_FUNCPTR (do_planar_12_unbind);
  unbind_func[PGM_IMAGE_UYVY] = GST_DEBUG_FUNCPTR (do_uyvy_unbind);
  unbind_func[PGM_IMAGE_YUYV] = GST_DEBUG_FUNCPTR (do_uyvy_unbind);

  context = _context;
  gl = context->gl;
}

/* Public functions */

PgmTexture *
pgm_texture_new (PgmContext *_context)
{
  PgmTexture *texture;

  /* Initialize the global pointers at first instantiation */
  if (G_UNLIKELY (!context))
    class_init (_context);

  /* And instantiate */
  texture = g_slice_new0 (PgmTexture);
  init_texture (texture);

  return texture;
}

void
pgm_texture_free (PgmTexture *texture)
{
  g_return_if_fail (texture != NULL);

  dispose_texture (texture);
  g_slice_free (PgmTexture, texture);
  texture = NULL;
}

void
pgm_texture_set_buffer (PgmTexture *texture,
                        guchar *buffer,
                        PgmImagePixelFormat csp,
                        guint width,
                        guint height,
                        guint size,
                        guint stride,
                        gboolean share)
{
  free_buffer (texture);
  texture->storage = PGM_TEXTURE_BUFFER;
  if (share)
    {
      texture->data.buffer = buffer;
      texture->shared = 1;
    }
  else
    {
      texture->data.buffer = g_memdup (buffer, size);
      texture->shared = 0;
    }
  texture->width = width;
  texture->height = height;
  texture->stride = stride;
  texture->size = size;
  texture->width_pot = get_upper_power_of_two (width);
  texture->height_pot = get_upper_power_of_two (height);
  texture->csp = csp;
  update_normalized_size (texture);
}

void
pgm_texture_set_system_buffer (PgmTexture *texture,
                               gconstpointer system_buffer,
                               PgmImagePixelFormat csp,
                               guint width,
                               guint height)
{
  free_buffer (texture);

  texture->storage = PGM_TEXTURE_SYSTEM_BUFFER;
  texture->data.systembuffer =
    pgm_backend_create_system_buffer_object (context->backend, system_buffer,
                                             csp);
  texture->width = width;
  texture->height = height;
  texture->csp = csp;
  texture->norm_width = 1.0f;
  texture->norm_height = 1.0f;
  texture->inv_norm_width = 1.0f;
  texture->inv_norm_height = 1.0f;

  /* FIXME: The power-of-two size is biased here since in the GLX backend we
   *        use the non-power-of-two extension. If that's not done here, the
   *        GlImage object generates wrong texture coordinates. That has to
   *        be fixed adding good support for non-power-of-two texture in the
   *        plugin. */
  texture->width_pot = width;
  texture->height_pot = height;
}

void
pgm_texture_set_pixbuf (PgmTexture *texture,
                        GdkPixbuf *pixbuf)
{
  free_buffer (texture);

  texture->storage = PGM_TEXTURE_PIXBUF;
  texture->data.pixbuf = gdk_pixbuf_ref (pixbuf);
  texture->width = gdk_pixbuf_get_width (pixbuf);
  texture->height = gdk_pixbuf_get_height (pixbuf);
  texture->stride = gdk_pixbuf_get_rowstride (pixbuf);
  texture->size = texture->stride * texture->height;
  texture->width_pot = get_upper_power_of_two (texture->width);
  texture->height_pot = get_upper_power_of_two (texture->height);
  if (gdk_pixbuf_get_has_alpha (pixbuf))
    texture->csp = PGM_IMAGE_RGBA;
  else
    texture->csp = PGM_IMAGE_RGB;
  update_normalized_size (texture);
}

void
pgm_texture_set_gst_buffer (PgmTexture *texture,
                            GstBuffer *gstbuffer,
                            PgmImagePixelFormat csp,
                            guint width,
                            guint height,
                            guint stride)
{
  free_buffer (texture);

  texture->storage = PGM_TEXTURE_GST_BUFFER;
  texture->data.gstbuffer = gst_buffer_ref (gstbuffer);
  texture->size = GST_BUFFER_SIZE (gstbuffer);
  texture->width = width;
  texture->width_pot = get_upper_power_of_two (width);
  texture->height = height;
  texture->height_pot = get_upper_power_of_two (height);
  texture->stride = stride;
  texture->csp = csp;
  update_normalized_size (texture);
}

void
pgm_texture_update_gst_buffer (PgmTexture *texture,
                               GstBuffer *gstbuffer)
{
  if (texture->data.gstbuffer)
    gst_buffer_unref (texture->data.gstbuffer);
  texture->data.gstbuffer = gst_buffer_ref (gstbuffer);
  texture->storage = PGM_TEXTURE_GST_BUFFER;
}

void
pgm_texture_bind (PgmTexture *texture)
{
  g_return_if_fail (texture != NULL);

  if (!texture->id)
    return;

  bind_func[texture->csp] (texture);

  /* Push the current texture matrix, and load our matrix */
  if (!(texture->flags & PGM_TEXTURE_IDENTITY_MATRIX))
    {
      gl->matrix_mode (PGM_GL_TEXTURE);
      gl->push_matrix ();
      gl->load_matrix_f (texture->matrix->m);
      gl->matrix_mode (PGM_GL_MODELVIEW);
    }
}

void
pgm_texture_unbind (PgmTexture *texture)
{
  g_return_if_fail (texture != NULL);

  if (!texture->id)
    return;

  unbind_func[texture->csp] (texture);

  /* Pop our texture matrix from the stack */
  if (!(texture->flags & PGM_TEXTURE_IDENTITY_MATRIX))
    {
      gl->matrix_mode (PGM_GL_TEXTURE);
      gl->pop_matrix ();
      gl->matrix_mode (PGM_GL_MODELVIEW);
    }
}

void
pgm_texture_generate (PgmTexture *texture)
{
  pgm_texture_clean (texture);

  /* The fragment programs doing i420 and yv12 to RGB need three textures */
  if (context->feature_mask & PGM_GL_FEAT_PER_PLANE_YCBCR_PROGRAM
      && (texture->csp == PGM_IMAGE_I420 || texture->csp == PGM_IMAGE_YV12))
    texture->count = 3;
  /* Otherwise RGB formats are directly supported and just need one */
  else
    texture->count = 1;

  texture->id = g_slice_alloc0 (sizeof (guint) * texture->count);
  gl->gen_textures (texture->count, texture->id);

  create_func[texture->csp] (texture);
}

void
pgm_texture_clean (PgmTexture *texture)
{
  if (texture->id)
    {
      gl->delete_textures (texture->count, texture->id);
      g_slice_free1 (sizeof (guint) * texture->count, texture->id);
      texture->id = NULL;
      texture->count = 0;
    }
}

void
pgm_texture_upload (PgmTexture *texture)
{
  void *buffer;

  if (G_UNLIKELY (!texture->id))
    return;

  switch (texture->storage)
    {
    case PGM_TEXTURE_GST_BUFFER:
      buffer = GST_BUFFER_DATA (texture->data.gstbuffer);
      break;

    case PGM_TEXTURE_PIXBUF:
      buffer = gdk_pixbuf_get_pixels (texture->data.pixbuf);
      break;

    case PGM_TEXTURE_BUFFER:
      buffer = texture->data.buffer;
      break;

    default:
      return;
    }

  /* FIXME: This test must not be there! At that point the content must be valid
   * in every case. The issue is that it rarely appears that a buffer coming
   * from a GdkPixbuf is NULL, causing a crash in the OpenGL texture update
   * function. That is a workaround to fix this issue, the real source of the
   * problem still need to be found though. */
  if (G_LIKELY (buffer != NULL))
    upload_func[texture->csp] (texture, buffer);

  free_buffer (texture);
}

void
pgm_texture_update (PgmTexture *texture)
{
  if (!texture->id)
    return;

  gl->bind_texture (PGM_GL_TEXTURE_2D, texture->id[0]);
  set_texture_parameters (texture, NULL);
}

void
pgm_texture_set_matrix (PgmTexture *texture,
                        PgmMat4x4 *matrix)
{
  pgm_mat4x4_set_from_mat4x4 (texture->matrix, matrix);

  /* Check identity so that we can avoid pushing our texture matrix */
  if (!pgm_mat4x4_is_identity (matrix))
    {
      PgmMat4x4 *scale, *transform, *transpose;

      /* We need to scale, apply the transformation and unscale to hide
       * the use of power-of-two textures. */
      scale = pgm_mat4x4_new_scale_from_scalars (texture->norm_width,
                                                 texture->norm_height, 1.0f);
      transform = pgm_mat4x4_multiply_mat4x4 (scale, texture->matrix);
      pgm_mat4x4_scale_from_scalars (transform, texture->inv_norm_width,
                                     texture->inv_norm_height, 1.0f);
      transpose = pgm_mat4x4_transpose (transform);
      pgm_mat4x4_set_from_mat4x4 (texture->matrix, transpose);
      pgm_mat4x4_free (scale);
      pgm_mat4x4_free (transform);
      pgm_mat4x4_free (transpose);

      texture->flags &= ~PGM_TEXTURE_IDENTITY_MATRIX;
    }
  else
    texture->flags |= PGM_TEXTURE_IDENTITY_MATRIX;
}
