/*****************************************************************************

Copyright (c) 1994, 2023, Oracle and/or its affiliates.

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License, version 2.0, as published by the
Free Software Foundation.

This program is also distributed with certain software (including but not
limited to OpenSSL) that is licensed under separate terms, as designated in a
particular file or component or in included license documentation. The authors
of MySQL hereby grant you an additional permission to link the program and
your derivative works with the separately licensed software that they have
included with MySQL.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU General Public License, version 2.0,
for more details.

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA

*****************************************************************************/

/** @file ut/ut0rnd.cc
 Random numbers and hashing

 Created 5/11/1994 Heikki Tuuri
 ********************************************************************/

#include "ut0rnd.h"
#include <random>

namespace ut::detail {

/* Changes for each threads for more different sequences. */
thread_local uint64_t random_seed = ut::this_thread_hash + my_timer_cycles();

/* This is a "precomputed" table of random hash values. */
alignas(ut::INNODB_CACHE_LINE_SIZE)
    std::array<std::array<uint64_t, 8>, 256> tab_hash_lookup_table = {{
        {{0x92cbe543cf592811, 0xcbc5ad6697f1e68, 0x9fcdb15c110a0024,
          0x89ea60e0d771f72f, 0xf05806843c1f7b6e, 0xd7e61a504317816e,
          0x16ac1dab21f1c7ec, 0xca459f11f3bc01ea}},
        {{0x1a2be5a5fa110e27, 0x22235b403da39677, 0x647da92f6b114fa1,
          0xb80c4f90c0badc24, 0x54b0b075713a636d, 0xb5eb6fbadd39bf9b,
          0x2c80b3c57f878d15, 0xc5f36215e4326a70}},
        {{0xa08fa3b4236dacc2, 0x71fc8e163e988e43, 0xe327cb46e36d4f8e,
          0xbaf3277a30a7385c, 0xebf215062fb07698, 0x94aee10152799c48,
          0xb38351dbabb29054, 0x90768a5d56e4995e}},
        {{0x640198bd6264ea69, 0x67cc31d5f9933f56, 0x86ffe85c7ba650f3,
          0xfab222ffc76b65f4, 0x19fe1e014cbe3194, 0xc117ca481437a85f,
          0x74009e6fb94826da, 0x3a25f465688aa215}},
        {{0x9f157469193bf70b, 0x9a0d1de3a503044f, 0xa40f014dcc8a6bb,
          0x35103d736f9770d3, 0xbef5cda86d26c2ab, 0x832d472518005024,
          0xdc42f37332a2866c, 0x3fa59e0b4c92c8e1}},
        {{0x2097b400a793bd6c, 0xfcfb172e6f9fd5b1, 0xf152f16a8149020c,
          0x6081d20ea3b40ff0, 0xe13b2c6e813d5f36, 0x9377b21e97ef5051,
          0x9a8d883de2f295cb, 0x6673f6bd0c10e302}},
        {{0xc2f4490f8ca9f19e, 0x1c921f9775dee9be, 0x42587657897451a9,
          0x4999e7fc7af2a212, 0x9787ad5a6ce887d, 0x988f42fc68a15c93,
          0x633398823d4673fb, 0xd6786eea77fb115a}},
        {{0x1c5a2a09dd137b72, 0x122c1bb47b0fb62a, 0x897c86b3ccefbab3,
          0x5afc1125fc0a377a, 0x2233569ed59a393, 0x767d99df5b92a38a,
          0xb3ace35b20ba2807, 0x1b48afad859718af}},
        {{0xd3de696ee6c98c84, 0xe261f8c03721cf11, 0x8ae6563978e6eda6,
          0x8c9e68a489c63d31, 0x82a14e1eaac32eb7, 0x9e0f40f54ae900d7,
          0x417d9ddd93fa27ed, 0xbf0e4bfec8fc0162}},
        {{0xcc6ab7d5d0644ac5, 0x120e0980e5f5280, 0xa793f75c4ea68891,
          0xdf438151ee88d69e, 0xb9e71ed77cca7aeb, 0xdf7d086b6a2a0a1b,
          0x73ff32d2b2546540, 0x5f4429e69aaabe0a}},
        {{0xdd7ea2138404a1cf, 0x542e7d1f0799513a, 0xd3d6e581ec2a80aa,
          0x4ec3223049489f0d, 0x302be28d25ad4914, 0xd16c14457cf2b813,
          0x98c488f5da1316d3, 0xd11395e6eed0a0f7}},
        {{0x7cff1d28f70603f9, 0x61595522ed96cbf9, 0x1d001372ddbb7a54,
          0x30ceae4ac2e96ed, 0xc8da5a0dcc08370b, 0x554fdd9630ad8de1,
          0x923a6c7abcf68d67, 0x2d7ab1417b22fc72}},
        {{0x61a25f39a2e7ab67, 0xc5a36499fa4ce502, 0x7c2f4ff628ed0d9d,
          0x809f606cfa3e9e35, 0x835d4b5758b50503, 0x64ba93a753e5254c,
          0xd9cf81dc2e9fd063, 0x575b8b6245e0ba60}},
        {{0x2faa0fcb4ca1d657, 0x29e76ecaf939754a, 0x7daf1d13ff4fe1b9,
          0xf65d384f5c0813f3, 0x2bfe750050ae566b, 0xa2645ec3da8a1b5f,
          0xfb46635e316b3704, 0xea33183819d6e1d7}},
        {{0xbd8419aac105a936, 0x883f8bf96bb30d9c, 0xfa71d023f1f24fea,
          0x2e1cde3f7fbb9d5a, 0xbc750067db933f61, 0x7c3323893a674606,
          0x4d08de8eb6ed96d0, 0x795aea9795dc2a3a}},
        {{0x8a22b1bb48d317d7, 0xd3f49f9e3b186278, 0xd81669b2ee7d8783,
          0x7b9fe709f0b53ac4, 0xefb130d87724c3f7, 0x4e5eb73086027e4,
          0x48665a5984bd2d6b, 0x7dadda1775014288}},
        {{0x2376f64e72796866, 0x5587f4446d2b7303, 0x773b42246f1d8dad,
          0x20fa15c45ba04398, 0xb0f351e3d9379a17, 0x66e2d254247a4b1d,
          0xa0862f3f7a6f58d7, 0xb6c761e5e32bc78f}},
        {{0xc2bec3bb0e01a40a, 0xd63d0527d011067e, 0xc60011d1806886cb,
          0x691a4165ccad986, 0x92de4bfe32e1212b, 0x1e77c373d9d27ef3,
          0x5ce84acfce2dd73c, 0x11e1c58b194edf3f}},
        {{0x37dfc410687c3e7e, 0xf6146cc605fc07a2, 0x9b8e156cc4fbbe8c,
          0x48ff495d9b25dbe3, 0xed0888202be7e4b7, 0xf371154c9df85f84,
          0x2829ee48c0be5628, 0x6b8fc5a39599e37b}},
        {{0xd3592932f2bcb62a, 0xb8551cee0db082f, 0x1e1f4cc572625b53,
          0x4edb58d0c8c444c7, 0x5f286bc45e80c62a, 0xc2fba0a6ed5efb2c,
          0xe91cb9ad3fe008f0, 0xf6c7a74a8b22f181}},
        {{0x9b63a0dbd75f76d7, 0x3c66fb3afa7b68bc, 0xb3966cfc7c2edc90,
          0x7a2429d7a72afde5, 0x93c8ab9dd6c69601, 0x815dd6ce0dc23968,
          0xb70f7941e3f33bd0, 0x8f6d0a2d12fb7b47}},
        {{0x8c895820ed1862e7, 0x3334732cf7bc51ec, 0x6e3114d9c81873f3,
          0x963a26a2d6a27ff3, 0xa875de101d86666c, 0xbaf804a7cf18872d,
          0xcc09d78a03853346, 0x98478c238c5ba0f2}},
        {{0x6c0a39769d806353, 0x9acf55b2eaf4e046, 0xd3e14ea4e5129488,
          0xf02ecd77f0f0f6d7, 0x3f3cdd02ba19e50b, 0xcdb61aa034ac5817,
          0xaccabf524f615f9a, 0x4c5b0556f342387a}},
        {{0x51fc96b559a79d9d, 0x4adad3352c7e79f8, 0x50531bb38365ed97,
          0x84850b65faae17ef, 0x853af170c1bebb7c, 0xf84417a59d851493,
          0x7518cc395c320824, 0x6353edfb7f94349a}},
        {{0xa6eb0e6e642c5576, 0xf806e62684b62de0, 0xdb582c362b266d0,
          0xb2c6f78da1e11811, 0x1721baaefb1b9d7, 0x6bf7adf240451384,
          0xc018dce061f8b74c, 0x7ce47dc71e0f52ca}},
        {{0x7ed92851ce63e2cd, 0x2deb360bf68b614f, 0xd679b406a4e5468d,
          0x78a9869629e22808, 0xb0805248088979d6, 0xc1740a996f7ba928,
          0x6e2b66697540d908, 0xf83162419199edd0}},
        {{0x6c85a8c21915124d, 0x5c05408ad1a038b2, 0x8d4b4d1f7b517de3,
          0x703f66a99908cfde, 0x623d1658d6b3c9f0, 0x36756c3bf1b16412,
          0x62ffc21e59ab0b93, 0xd976a93ef44c626c}},
        {{0x444962d1347a03d0, 0x7c73a977b7959cff, 0x3108867e344b29aa,
          0x9c9761a8818cfa96, 0x9a4a584da4a8765c, 0xc2b75eb312f906bd,
          0x9ed041d27cf6e44c, 0x2f417173915b4053}},
        {{0xd493c1f696f38ca4, 0x6167664db87d7aed, 0x5086a97897707dd7,
          0xaa7a6acb35db8cc5, 0x2fce222474726934, 0x82eb9a6460ac25e9,
          0x40dacb5f067136ae, 0xd12183b6d058c6c3}},
        {{0x9c5040a796695e4a, 0x6b340b9b434c77d8, 0xeb118ba783d7adc4,
          0x777a238f7f076968, 0x7247cfe8cc2958f5, 0x9e06b8fafda17de5,
          0xaa2c6bd9a7a58126, 0x2bf2021ac2158ef6}},
        {{0x7fef227f31bbff33, 0x9bcb84dc3f01c1e3, 0xaf0a77144e78a9cf,
          0x7b6ee464a6eaec42, 0x25be3ba14072f035, 0xd5eef70e8731047d,
          0xcb6c9d4a5cdcc8ba, 0xe27f833c9f1c64ef}},
        {{0xc9c39cbe0f90d848, 0x3755f5d268c47e42, 0xe4b7d0030060a8a8,
          0xd6b9bf96705d33ef, 0x4a13009bf59eaa28, 0xe97980885a09be36,
          0x2537baea69ad28e1, 0xf81717459276e0cd}},
        {{0xfa3489176c604606, 0x5daaf474646eef31, 0xef10e9e1b37c214e,
          0xb112aad34e796fe6, 0x5010063d9b39a2f0, 0xbfd2dde8791b18d8,
          0x3eee063ee62955b2, 0x5215b8c03212eeec}},
        {{0xf854f20a27c1ed5d, 0xea0b4e31a14df4ee, 0xdf189f810d31c20f,
          0x1cae7d94a6c7d9e1, 0xbfd00b07735f59c3, 0xcab8d1a3e26bda50,
          0x95b563ad37796ad3, 0x2876478c5154ff74}},
        {{0x5ac19063b46ac6cc, 0x112b2d4c496d1ad2, 0x7d5ca24a407b1eca,
          0x24ed96308a0b08be, 0x5cf30f78fed3783a, 0x30fbd78628d082e7,
          0xa325f46d13faf81d, 0xe29e8ba0462ec937}},
        {{0x9dd0d63135daa8f, 0xe26c043ba90058f1, 0xbdc126e6916e39ca,
          0xc0668eb2516a9923, 0xdb3cfe34fbdff212, 0x9c9bb4ff00b08b56,
          0xdf5993c3bae7794b, 0x17c4e681ff923b8c}},
        {{0x6812bb518577c896, 0xd56c640702f3bedc, 0x7bd83f072ccb2713,
          0xfdf41a41890edefe, 0xde11b027a9b4906, 0xa70e33115c2f94d5,
          0x1d65a57c18ef3810, 0x1d61296d36b51679}},
        {{0xcf900feedda15e17, 0x4f8e99093138d91d, 0x7a135c952c9747f7,
          0x7e29089efb7be855, 0x20fcf424ad225bcf, 0xe8d72b96de8f9ac0,
          0x9dbfb7e4deeecffa, 0x812bd706e68a195c}},
        {{0xb88a263e2d36aade, 0xd355ce904d8b5358, 0x44bd849d6902c2c8,
          0xcf4ba4426c326595, 0x32beeab808b3f85d, 0xf3ea26e7addef6f0,
          0xfbff5e2f1e59e94, 0x72efc1979112624b}},
        {{0xdcff654d19c51df5, 0x5e6e64ceacff6643, 0xc72c77861a89a120,
          0x5d84decf7c396825, 0x42cb29147b3eff56, 0x7b5311a7f516bd7e,
          0x79e5d9aa8ebcea5f, 0xa210ec3b3f6f0822}},
        {{0xf1fbb1ce2534d194, 0x44d3185310f7e8ed, 0xb40965f4b7f4e239,
          0xd66a327ae57d6cfc, 0x9765c15e28aff4d2, 0x1401a2819d3a6b8c,
          0xeb0cdefea554e9c4, 0x1daf3bf5a014e58}},
        {{0x52e47e1523411f6e, 0x7ad290bc4d6625cb, 0x1968658f7b82ccac,
          0xf49d851609981e02, 0x2147fb954a75f5e1, 0x41a5cce337cc7f3e,
          0xa281476e634f5a9b, 0x58192f24525820a9}},
        {{0x8463f00cd76ec11b, 0x203b0b0241e1b1a4, 0xfae3e2dcb58c5a37,
          0x13b4dc36508e9de2, 0xebf484e03f8fff8e, 0xdca3af14c2e7c546,
          0xba152839746b57f9, 0x5c89c6df87993255}},
        {{0xe0a7298859325d5a, 0xe2cf3f54e99e7a8f, 0x7c6ab809ed5fafce,
          0x798f24e3eec32eba, 0x978f661d20fe789, 0x7500ee934d734327,
          0xa9300a80f3cd0af6, 0xf1f5831d9b76060e}},
        {{0x2ee879b9cbf2bddc, 0xf43c031b03edc28, 0xac74af4e50aee262,
          0x5a3cd759007a3eba, 0x8643a23567b290cc, 0x674b84b673e369dc,
          0x93c2ae84e18460d, 0xb205b2b4d053ebc7}},
        {{0x2caa6a91ee13a7eb, 0x2fdbf5fc26c24648, 0xfa89e42ddc39992f,
          0x3e7d684202c8b693, 0x521379cca6ec7858, 0xa6544c330303169c,
          0xb2f13566759d59aa, 0x810940db994d088a}},
        {{0x9fdef9e31f4790c2, 0x69bb31ff1fa37572, 0xe800a03c3a6603cb,
          0xeed52b8800c6b2e6, 0x20f72052a3105716, 0x575e439309891944,
          0x8c16a1228750c596, 0x7d4d217fd8285e64}},
        {{0xb69f7114f28ec433, 0xcf34bda4001d219c, 0xe101ed096336b17f,
          0xd4f90437867d668b, 0x2277832405085b13, 0x79bca36d9200ed83,
          0x8872400b7fb71af7, 0x49c8b1191e64c289}},
        {{0xa0c69412b71ce52f, 0xb13c3ef98cdd5445, 0xd6033efad0d67d98,
          0x75fd4375916cb59, 0x3a0e6a2993f6eb4b, 0xaa084dae562f39ed,
          0x58015edeb7d79a0d, 0xc1f140ab5c20caf8}},
        {{0x9919b3c25f252fd2, 0x1a617013fb4c33b5, 0xca5a89a51fa9f4c9,
          0xbb1b6bad05e3a8db, 0x2492e481a3257685, 0xabc0eb8b9860ab63,
          0xec39e5e7457ffccf, 0x71851e82ca7b4211}},
        {{0x176bf1c62ca0aecb, 0x1c26ec8c3ce51e62, 0xf62aaf4dbe4a2bb3,
          0x6a4460206d893feb, 0x1bc29bb3c7f8bec5, 0x4cd758cd937f8e1f,
          0xe496dfe1167e59d, 0xa47e5d6859397a54}},
        {{0xb991c4551fa425e5, 0x4c87cb1853692d17, 0x90f77868d9088fff,
          0x4f9ab853bf6970f4, 0x2f8df26df4dad1b4, 0x8cd736c6e2c15110,
          0x2e8aeee02fef3e15, 0xe6e74d0df65ef869}},
        {{0xcad1755ea717d4c5, 0x56c796adfff759eb, 0xf10629e687bb6b2e,
          0xbfa9f39058eeefb3, 0x6d3088a07ee03049, 0x7730b6688e7aac18,
          0xb7b52ad85749a49d, 0x4b499f5fee1c53d5}},
        {{0x182dcc2dd7662ae, 0x1a7a412d22c5120b, 0x59433b493bc56bc,
          0xe211f0f95a5fef17, 0x36ff30887ad9a2ed, 0xc4b2d1a878556bca,
          0x4a18f402a45f7a78, 0x4b30e5445ec4f94a}},
        {{0x68262b4835cd82a6, 0x500ef4786c622382, 0xa1de782861389f9a,
          0x24732418eb49a4d2, 0xe20600a5f899dc6a, 0x615f92c11952e665,
          0x3948d8ae682639b0, 0xcbc881ebea40c9f9}},
        {{0xc82d7266e1da6bb8, 0x992b18035afa2113, 0xb690b2ace36c3970,
          0x19eae30f62cc3996, 0x59c0bda169dd0654, 0x39a2d24519321987,
          0x818206f3b1e82f5a, 0xf1435119c4d9bda7}},
        {{0x9401bec986afd73e, 0xd100a3f49c1af662, 0x5b66f388b8628d34,
          0x705d4913c3cb155e, 0x717265a7bfd1d745, 0xf1ed14101da638c4,
          0x4e5161c8bcdf521b, 0x9a07373efbbba59f}},
        {{0xf894cafb0401be65, 0xade68c1299cf6c06, 0x1307a7d51f350fc7,
          0xa56ee0b110df7595, 0xdade47ccead9488b, 0x6c8cd67bcb043702,
          0xb4f88671d10e0e74, 0x80fa26ac60e241b3}},
        {{0xcda026efa53919a9, 0xcfffa5a089649c18, 0xd2b4493c52e1c0b3,
          0x9f8a94c17855e9f8, 0x48e95746f07967c6, 0x582ff1d1f128ee98,
          0x6516555e81d92584, 0xe20df01e14d1196}},
        {{0xa2c28921c9b82600, 0x7fb75b81a6690a37, 0xb2a8330f1912cb08,
          0xca90f950f4fa6917, 0x2bf7505284a13766, 0x61956ad0128bf182,
          0x37432dacb89c8842, 0x7aeb9f7f9999c435}},
        {{0xaeb20e890261f9f, 0x5465639dbdfb339f, 0x3e3a90df91c09d9,
          0x3737129fb33d31c6, 0x2cb9375d2e755520, 0x83ae95336eddb4c2,
          0xec416d08f5d0f88d, 0x35a95fec3c344b18}},
        {{0xf83546dfb96c493d, 0x66b830c8aa0efa5f, 0xc12d32ff50b661d6,
          0x5949983f5a0e409c, 0x52f6cacfbb747620, 0xb3c17b66c0427622,
          0x2cff14b6f70ea4b9, 0xbdcbbb7c93b3de2a}},
        {{0xf2132831dfc4a75a, 0xcc4c56d9d7feaa16, 0xb0059e1360d3e794,
          0x58978a8cdfa73247, 0x1a967d94d0e4f16, 0xb47aed0c1503e63a,
          0x1bcb191ea8e9156b, 0x26b457fc120dd746}},
        {{0x357728b95eb33cc, 0x67d7382c54165aa7, 0x17a63ab24d050cd4,
          0xa82886332681cec8, 0x6c77687648537a3, 0xea3020fd335dd703,
          0x8d61fb9012beb870, 0xe17af0cc0d93d81}},
        {{0xd26c120b76a0f9f1, 0x760362a5bfeba79a, 0xc4e4e0d09853889a,
          0x1b04fda13bc18a88, 0xb1816f44e8c56123, 0x92f6ec2fe545843f,
          0x66ee33327a1eb0f0, 0x6cc7e39d2d4e79f7}},
        {{0xb3b1876bec2c71b3, 0xdc7ac06a31927ec4, 0x3a12839e7faff841,
          0xa09333bd0c45a135, 0x2e3272d038a8944c, 0xbd11e43ec3a58ee4,
          0x2df12c36717dcf6b, 0xfde2c2891be6db0f}},
        {{0xe107dc8de9fe8cc5, 0x57a8f2b3ee3689c7, 0xe18fc7ebc2f09a8a,
          0x4991b4923bf5e360, 0x2189ccdca4507cfe, 0x68d186d7e58248d1,
          0xc87b0158ca3ebcba, 0xaa717f5a5e8fc1fd}},
        {{0xaa59ef9f78836b98, 0xb52d82e6cd204436, 0xfac3b2bacef2b078,
          0x26e0f6e2822cb005, 0x368f5d735176e424, 0xf157972fd2d9641a,
          0x52b38774a31caffd, 0xf09e900f5f0a3e7c}},
        {{0x8ae31bb52741b1f4, 0x9edd9442440bf97a, 0x9692d51498fb8d60,
          0xc3c03fdf1febec86, 0x18441c137407e055, 0x3404dbc290822b09,
          0x4c8105f4e51705e8, 0x901255c99984c5f9}},
        {{0x97d81cf3450fe097, 0xd0c0ce7ef347585b, 0xc3a8b8bc33ac7b9a,
          0x9698798f70d1be42, 0xcd502c7e01c8bb36, 0x8547b71db6df5a51,
          0xb834967e915a05b2, 0x940b88d4b90e4312}},
        {{0xca67a8144f3a9658, 0xeb0c629e961c80f5, 0x3b571e24c97ec969,
          0xbc840019cd519fac, 0x7ec68dd9fe24e680, 0xe7015a66a3218f73,
          0x3639c091bc44b09c, 0x3da42ed51b1ef4e}},
        {{0xf8f7325ac1d0ab5b, 0xc3f317f7b4b59c73, 0xadff14de9e56bcf6,
          0x9139caf459c3fc9f, 0x6e10f4c3286ed9c0, 0x96c01eed47ea87eb,
          0x2d671f7599fc5c35, 0x6d9f8578f0c35404}},
        {{0xf88406020f2695d9, 0x9142a5dd4a66c064, 0x59bc2c564b2b984e,
          0xbfaf8f9c0ee3dd85, 0x5d9aad32feb1f7f8, 0x2ecd5a5515fdd691,
          0xadccef8aba937c21, 0x6d45ae6138660b37}},
        {{0x7ec5a3374186c399, 0x608413a21e241f54, 0xc89513ffbb4924e7,
          0x6e83e6dc4835d1d5, 0xbcc9b663a9ad57f0, 0x22f090519abac708,
          0x9562d91eef57c050, 0xe8e1d93ee6f1e796}},
        {{0x5992977b9be32987, 0xc0e674da821f48d5, 0xeb8810eafd0ee609,
          0xd80ebdd799ee63f, 0x3580d4786f0cdd44, 0x2c24256b2eda52b3,
          0x34b7761df7b64c9f, 0xff329e270b81a839}},
        {{0x213fd1b3330e24c9, 0x65019989e21f1372, 0x41edd8476f821045,
          0xdcc15a56f00fa4fa, 0x66cd4ba4fe770eb, 0x583271d00bac074e,
          0x844650f7167e9e7b, 0xa071ece407f5ad98}},
        {{0x52cf99fd73aa993c, 0x6e561b7e097315b2, 0xa928c46e230946ec,
          0x696da8eae5f08ac2, 0x8a1a7de450d6009a, 0x8992901e39f47620,
          0x322a2f4d7a8e5614, 0x3962d089c41d0708}},
        {{0xb397ce607a03c30e, 0xdd965c2a693c3087, 0x5f1211e4ded6e052,
          0xfd1adfc9771a493, 0x790144711fed2d12, 0x7b9d886eac08361b,
          0xf4399321f54d2196, 0xcb5aba27bd9b49b9}},
        {{0x53b54752892c322b, 0x2c2bc284710411dc, 0xb1db38b43b53e1d4,
          0x9f21f747544b60fc, 0xde97708b87e2f9f5, 0xab7ed4dc6204f591,
          0x9b6e5b9fdf3195da, 0xa109943241556807}},
        {{0x22a4b5c8b3e1a42e, 0x209fd78c4beea624, 0x9933b3ac7b6b7c39,
          0xf6a07eae62d61ee, 0x5ad173402748fbc6, 0x6cb0f73ea373073e,
          0xc929a02761adddd2, 0xd5b35d7b20e8d07f}},
        {{0x21dd4bbd824c2b52, 0xdd2378532e02f4d, 0x800afd323acc89f7,
          0x5af8a897d6fb775, 0xbfc0b33003906cda, 0xe1b0907a2f34ba65,
          0xe6fec86a7cd007b2, 0x2093f0bed2e9ad08}},
        {{0xd0de9a1756b202d6, 0x4d7eaba90c9a1675, 0xf93fdeb7e21355af,
          0xbd5458b659f10190, 0x501e8819857ab083, 0xb7ec0b0dfe923944,
          0x76f329d52d8b7f47, 0x79ca98d0d0293eb5}},
        {{0x2417d1fc92492749, 0xad2de8e075047a09, 0xcd87b0aaa560c63b,
          0xf2819a651a4f3e01, 0xe3f5a2f9f554dfbd, 0x73fb4d136a323dc3,
          0x6415e767df5fbce7, 0xc3cc9cc23a46f4f4}},
        {{0x50d7538c5b4a890f, 0x594bb49359692425, 0xa96d8307e8585be4,
          0xb824357d8becb3a0, 0xf51f10da1d09948f, 0x171044b4e2ea6ace,
          0x27f4829e7bcc4bbb, 0xdf5be719230068d9}},
        {{0xc412f800f7bb452, 0x578275f8fc45394e, 0x94c0c6f2101e15be,
          0x9ff3d7bbb0c69413, 0xdf5ab888e8c52785, 0x845ff3c124be2f87,
          0x6402bd3e7f19f646, 0x2180cb1484c49bab}},
        {{0xa597bb26da537cd5, 0xc4a7b925afc68892, 0x8ac07b63552bb48e,
          0x2ea01c66f8638724, 0x11c8eaaebfb21e1d, 0x5b7aeab120391e06,
          0xfd984e22be20948c, 0x5b258a67366e32a3}},
        {{0x782508adc57f8ed5, 0xc1841fbe78fb4503, 0x2329cded6bb4fd12,
          0x9bf82a0f5444b29f, 0x4f49e1adb12c811a, 0xefc7382b3135fdea,
          0xfe0f1bbae8f0f6a5, 0x487bc705334d4296}},
        {{0xe6edd79d372e72e2, 0x42031ffe79f9c8b5, 0x35fdeecb6fbd1bf6,
          0x7dc4f15bd35eee07, 0xb36318825ad0b37b, 0x8bafbd882e4f7122,
          0x4914ccfaea635b17, 0x90434a6f5256514}},
        {{0x6aeb60c1c0bc6cdb, 0x217efefa99ab910e, 0x97f980b37f32ef13,
          0x7d24e114d9fc4c0, 0x8ab5a4df43be84f1, 0x128c1232273c5401,
          0x9e3f7f8dbe155b11, 0x76254177c0d0120a}},
        {{0x13f51949df0e5854, 0xc5bc240e22bc9bd7, 0xd3985eccea298f3c,
          0x8d9975c70bbebe28, 0x39e2cbf96b3af1fb, 0xb5b57248ad7ef99e,
          0x508373f4aba964fa, 0x573f4226ad0a2017}},
        {{0x609a78c0711dd184, 0xa0e2257cf600ca2b, 0xd09a67c2cd23cc04,
          0xaeb310185aeb22dc, 0x9f92178be74fe984, 0x1aa6c8db04391550,
          0x129e392762a8be4f, 0x82d44896d38cb63b}},
        {{0x4075d1fdc829f957, 0x8845b9f49b2f04d6, 0x9d563bf106fd15ef,
          0xb3bdbab3fee75a51, 0xa30dc23d948779a3, 0x5cdb6467a127ef3f,
          0x1b349a7c82523e79, 0x8e459b38665723fc}},
        {{0x89e1e8d55229ffd7, 0xb56d9c82ecd4703, 0xb1e1ef1653caa376,
          0x8b2f31599bc5e570, 0xd9d646a990b2ea1b, 0xfd9dc1cf93fdc2e8,
          0xf9b74b76430e0c15, 0xf1a3c11276c7004d}},
        {{0xe99ac6769c465779, 0xa41d9f186b352c06, 0xa220a4ded52c5aa8,
          0x8cbcf9a042909927, 0x3ad0ef81868ceadd, 0x64f73c7806c6897b,
          0xc9151fa5e90c052d, 0xe8a0abfe7a1f1ef9}},
        {{0xbe3ec1a37dc1888e, 0x8ecc40e28810e514, 0xc482b6c76f43480,
          0x1696b338697488c6, 0xa1d5a313ccd40358, 0x8ce4835783ba834b,
          0x732fec352bf7d823, 0xd5ea377aa08c3076}},
        {{0x374d50ea5b66eebf, 0x570d06db6a4368e7, 0x60f13df058cfc3f5,
          0xa3e2cfe7f27b9cf7, 0x7474d2fd056269da, 0x7062175d51df0ee3,
          0x1930bd08a6317391, 0x377160b1deb11be4}},
        {{0xb4322629ae84e4f2, 0x43c0db0322609612, 0x3c71946d4ee354d4,
          0xdba46bc1b112ffa7, 0xbf01c3bf4b43bcd1, 0xe7dbcb8eaae69c0c,
          0xfcd20bccbe3fc8ad, 0x34dcb498ebfb6cce}},
        {{0x54a8c84d5248846b, 0xac34cf6e89712a6e, 0xfe0a19ca348e5d02,
          0xb07b31ce4a69f1a7, 0x6d48815b8a988e0e, 0x22f2578ef25e5227,
          0x779c8fd0857d7a99, 0x999279090aba41c1}},
        {{0xa5634ebc1ac66bea, 0x7002a5e8c78095d1, 0xa150f4f92ba0c29a,
          0xb1618d3f98404d13, 0x2a651a73e2e019e9, 0xb1727a70f10482bc,
          0x1a4b0e3fddee7b50, 0x7a85367173972fed}},
        {{0x53eb3cc4c5a0e748, 0xfbc32c55e902be4c, 0xec382b4d6886c8be,
          0xde7d2a751d611de1, 0xd0e72ce092ad1f09, 0x2410d3c4bdf091f5,
          0x31528588d3cd34d, 0x2c69db24c30b3833}},
        {{0xd4d8311ba250441b, 0x799669ccb0b13fc2, 0x95a93f26206d73a2,
          0x4086bd4b2d69c816, 0x781ac4aa781fe5f9, 0xbfd301d79a043064,
          0x4d31e7b2b8122df4, 0x2551298702aff43d}},
        {{0x6d8ed26415294055, 0xc8f8f889165becaa, 0x1d29fc16d2a05ce0,
          0x4ec7ff278bc47bb4, 0x49fa506f15c361f8, 0xcf9e398d0ee911c3,
          0x3df509fd253425f0, 0xe43eb268f9054799}},
        {{0xa47198ef54d31bb2, 0x4d4fc4f54b781edc, 0xb6b5e224c8f1a961,
          0xf034666f680dcfc5, 0x7625345a75b016c9, 0xb4b89450c9574195,
          0xc197afde76b172a9, 0x9fd86186f772bae5}},
        {{0xd54696ea0511cb3c, 0x8f65381420c84dbb, 0xff1c626985afc32f,
          0x4dd728f0c6e219de, 0x628de3a5effdbc2, 0x4864e97a3500b844,
          0x62ae2213b7c824d4, 0xe0f55b7848dbea6e}},
        {{0x13cfb14eb3018c11, 0x51c0008e28adffe8, 0xe36a17c6b4fe6e08,
          0xd8d7049f9d99932, 0xb84ed731868f550, 0xfa2688906c9152d9,
          0x2e6c8c1ca67cd145, 0xec9183c10d49623d}},
        {{0x570f338edd90fe18, 0x238703aaeeefe1d7, 0x425db7b566239ac2,
          0xf6b54cad5f56e099, 0xc17e253c34fafda9, 0x73939c68e8a2f46,
          0xc026ddfb2888d54c, 0x7a332b1dfd6ac5cb}},
        {{0x352261d6e81314d1, 0x7e271c8f25dfe6cc, 0x589bd9026d768443,
          0x8ed1c2a697c96236, 0xe4a55440d4c286c4, 0x9e1fdecf97cd06af,
          0x9d390819b42425c8, 0x58af379a7ac83407}},
        {{0x4d8323555071f586, 0xe050e595ec171205, 0xdb5d26542e1f9236,
          0xb192ac34fa83be9e, 0xf79a67b5d6f94717, 0xbf82b6c6f9691d89,
          0x822094d2423602b9, 0x25b41a64a0a0e589}},
        {{0x30e725e77a5cee6e, 0x4779a8a75ec321e9, 0x34b5af83e18fc73e,
          0x995dd90607280d89, 0x938fa68580bfbb3c, 0x18f9b372f43bb1cb,
          0x75a0766892ca82df, 0x4410d8b4d3efbb32}},
        {{0x13d0f66fe136dd28, 0xc38c5d43fb7d1b64, 0x86113ac1a9f8a1b9,
          0xda0043b06957b3a3, 0xa60f46baea5293e7, 0x2e6df50dd2de827d,
          0xc4a02dbaa717f2a6, 0x88ec8c915138ef73}},
        {{0xaa8882fa53967e92, 0x397d3d00c19ed9d6, 0xb3f4bb7dcc5fb909,
          0xe55660f155cc75cf, 0x4025d9a310812adf, 0x840b7009996b78fe,
          0x31eba0a028187363, 0xb53b66945e5d45f7}},
        {{0xa13de6aa30e62dcb, 0xfe86e0ffc6331e74, 0x476107c17caedadf,
          0x7ae2234b6ed21ddb, 0x718a0cb0578e4665, 0x861d37df2ddc9cd2,
          0x44def4a8908d719e, 0x6fa037db75bdda4d}},
        {{0x493fc96220a26afc, 0x920fde9774611f9, 0x3653f28e60017269,
          0xec1e97d70933d3dd, 0x94dbe62db0e09202, 0xad5ea662075b5946,
          0x59ac998fd01e506a, 0x77a0932fffa2e9e8}},
        {{0xb025a5ee666e0f1e, 0xf65f080d1289fae8, 0x862ec75ace14bd4a,
          0x4160f34456c32b99, 0x2cf026ce2631efc5, 0xa5d4d91adc375a04,
          0x96b16967f7067eb1, 0xe59f6b4dabc48ce6}},
        {{0xccc524d02313c3a, 0x5ada17c00e925f23, 0x2ceebea1fa32ebb9,
          0x4278660fcd9f7fa5, 0xe564654027e8fed1, 0x4a1e856218e6cc,
          0x5cadade5ea5b0cf4, 0x61a9aaeb11096cf}},
        {{0xb96e906fd4ad4be9, 0x5334746d0911b26d, 0xe2a6e0d6d07791f9,
          0x2cf1a655c56808bc, 0x30c70f8eb9ec6f03, 0xb79661e769f63d1d,
          0xc5bfd3ca965d9b45, 0xe9769888aae88da}},
        {{0x4806d4997eb8e9c3, 0x1acea026bd3f68e8, 0x27a040cfdab4709a,
          0x79f0f9a25f162213, 0x457a55809a4c8036, 0xfce981dce5853c1f,
          0xb9791b87e332c6f2, 0x45293249082716b6}},
        {{0xc63c112dc2750f33, 0xb2ae20ed06463d3d, 0x5e64979db3dc0d5,
          0xb7b748258496dc44, 0x4918382873ef71c0, 0x7f89c2aacf61ccf7,
          0x603856c3337e254c, 0x31e3db5c40948a99}},
        {{0xc627557ad1b7b11c, 0x93a30f7c90407278, 0xef2dce8c1ec9e33f,
          0x59870da469e1d034, 0x6a87dfc68003a33a, 0x7e81593f47e70c50,
          0x637d9ada7ba38885, 0xe7e3d4b179bab832}},
        {{0x71a626e5da46eec, 0xce5ee44e823dd284, 0xd624990e4a87444c,
          0x6210ba509816059b, 0x9f7d3ebb1cc3a26b, 0xbf7efa86563c4478,
          0x49298cbf78d945e0, 0xb980d28f1a8b0051}},
        {{0x3d09e40b8ca9d1ac, 0xbaa85f4570e0d88f, 0xf9a6cae00b3b0322,
          0x82640489b7123a43, 0x6164a611c6cc636c, 0xac454effd08e256e,
          0x3a71ab7719371b70, 0x743f3c3e7b1cdca0}},
        {{0x2719a868f5a19657, 0x2131151ad47d08f3, 0xdc6d9a7fd99226aa,
          0x5b8b2cd0ff2f1b1e, 0x20026f7664fa0127, 0xdf76a25c98ad5380,
          0x2d818a877dec4da8, 0x94abe74b71cbb572}},
        {{0x9dc0d125923c6b41, 0xd5fab77cb72049cd, 0xb0e4c75b8c93cbd2,
          0xa3387a6f519a6762, 0x8fbc7a5152e2ce19, 0x93a599ae99b65ed5,
          0x12aeab5d2306617e, 0xf9668ab7dd4ef281}},
        {{0xb723a0ba1568ff39, 0xf970b18ed2226e43, 0x62a9e04aeb5ba567,
          0x8a94c4fc0026e62b, 0x840eb8efec2ff900, 0x4546ef13d7bbb5f3,
          0x5cd331a02faa825f, 0x98439d3efc4c914e}},
        {{0x38d0cf651fd90309, 0xdbeda92ec9084c06, 0xc2e94858b415823b,
          0x32cb79d3b60c66ff, 0x92b9ac3011d2e0a1, 0x55b8cfea37b32e09,
          0x67e43f61d12b7d8f, 0x54d87c549fad49fd}},
        {{0x82586e9bed0f76bd, 0x34d7edb21e90f4ff, 0xa18e15d738fd324e,
          0x6bfdf9f75a7f9531, 0xbd9458687b56603a, 0x4db55da62fdc158,
          0xc888291bdb2a49d2, 0xa8fdd29c40d480a1}},
        {{0x95ca152e379fbc0d, 0x929e92e3fac083c5, 0x8980b7e64e02da69,
          0x46efc236caf220a6, 0xa02eb69e12474c36, 0xd7457b9a07a7f4db,
          0x2bf01aa38e36e8b1, 0xa70a4556f8d8bdeb}},
        {{0x471f3479eaee4cc8, 0xc65f1bcdf3888704, 0xb4e47320019663fe,
          0xd07a4c330bd766d, 0x5a32d23b554c22b9, 0x605a712eaf0dd117,
          0x15480dd0d0f34576, 0x6244204453810828}},
        {{0x9350caf07957211d, 0xfaae9a960cdccd5d, 0xdcc554d01fc48a4a,
          0x4fd7ce978fd0feac, 0xe9b1e97dd5afe98a, 0x157de8b597a2038f,
          0x67576cbffb6dba00, 0xe5a49a8f98501b25}},
        {{0x7aee01cd57db31b8, 0xfa2e23c37c213f1, 0x147088487922288c,
          0xd96d839155e03377, 0xd508ce00febd8d88, 0xb74ce789b3d8fdcc,
          0x26825704d8e4ac7a, 0xaa10887b672d4e54}},
        {{0xde953f09f21a9552, 0x723292d65e28e82f, 0xc19bb83e5394caaa,
          0xe11875b30c3560d4, 0xddea5c15fda11ecf, 0x15f700f880e77f76,
          0xa26964b7b3550a3e, 0x30cb783cb7d0add4}},
        {{0xb991ad15ed1f4c86, 0xa656790a024fb2a8, 0xc2bdd72bf179cc8e,
          0xcbc600deb2e7d337, 0x18e3292464ba6ba5, 0xfbdd3f2a1840909b,
          0x296ab90f7f7c4d6d, 0x9e128566b90c22c7}},
        {{0x58419064a7fde5c2, 0x8c0a977b1eb0f68c, 0xaff204c1f9a9a1f4,
          0xa5ed592471dbfc2b, 0x21cf66c28594cddb, 0x9eae139e75b55d08,
          0x98db8f4424204616, 0x748ece0ecd54dca4}},
        {{0x87683986bfbcbfe2, 0x9350f818e9661560, 0xb9e35456451cc2fd,
          0x3cbaa003fc01a3a7, 0x89332e4750053b51, 0xfda258103755910d,
          0x241bdd966bf62481, 0xff84631252956a72}},
        {{0x46774affd3cd6697, 0xa837b0403908c358, 0xc400e1588ffdffd0,
          0xd1f781fe45a2453e, 0xbda6213a03c884cc, 0x7724b062f53f332b,
          0xc699fe6bb2fe6d90, 0x9d15140d95e7a895}},
        {{0x84a9dc81d67ae94f, 0xcee553f0614ed910, 0xf34ec4e433d51fd0,
          0xc20090bdfb922dd8, 0xaf192a4ebab1221f, 0x890c7e2ecc8ed150,
          0xcdf8f11d95b9ecca, 0x7798b9b3d3a5acaf}},
        {{0x3a2d4c4a4036e7bf, 0xc2356ad66e28f3e, 0x1048ea7aafb57fb6,
          0xfdbcbcfd089fb9ba, 0x8b839eaf83043f48, 0xc013e605618c9e84,
          0xe1c3245a883ca02d, 0x6232be8f13fcbcc7}},
        {{0x75ffdaeeb43e8c9d, 0xefffc9701d0cc8f4, 0x4f29052ba4c6b32,
          0xda6e0ad52deade45, 0x88f1624c1f8d2214, 0xeb7337798c18e7fd,
          0x8e1d139dbbb64ffd, 0x97f6ad9f6c2a2c74}},
        {{0x5caf6d7c53438d78, 0x298a762b76e5c1cc, 0x8021c8d0a25ffa14,
          0xc1fd38b6fd3191c4, 0x1f8dee6f7743834f, 0xccda3cb4bbe63111,
          0x61f64137bd1b40, 0x89c005995f47d0dd}},
        {{0xa88d56a5853d617f, 0x68aadc4a9a9a6ed4, 0xf1ccffa81a283616,
          0xfe3448bbebe050f1, 0x69504881943a1a6f, 0x6506dc436573ca28,
          0x218db93aa6899b8a, 0xd3caee953c2a374d}},
        {{0x68e73d3ef02e81fe, 0x82ff0ef46e1eba1f, 0x9e061acedc91df45,
          0xf7ec5acdfd975058, 0x8db10048acae8194, 0x821852b6a78a9449,
          0x4a933a77b8d33944, 0x43bb2d61bfa9334b}},
        {{0xb69238e6c4fec5b6, 0x7278b172b79cc3e7, 0x928e2547bf650ca6,
          0x536fb92f9a5629b5, 0x473ef1eb1594861d, 0xd9c6b3dad0438a44,
          0x5b36a787310e1990, 0x8fc3e7cd57fe9d3f}},
        {{0x898aada42945e535, 0x21b72ee744945d17, 0xf695e2506400604,
          0xa2d46d75eeac06ce, 0x52e8363471d9a537, 0x7305325c78b5864d,
          0xe1348fae97d86bbd, 0xb779c07971143002}},
        {{0x71fee34689ca19cc, 0xfafa7f44fa008e31, 0x810eb751f4d98b3,
          0x8b01b490f9652e9f, 0xbafa469aa79a495a, 0x9b96efdcd4295a6a,
          0x77f4a7459fdb7577, 0x5358c4827d4505cd}},
        {{0x470ef8e78c498025, 0x1bf72250bdc513aa, 0x1f6fdfa774c9f38d,
          0x85e3dfce1856b7a8, 0xcf39e6a35ed727e1, 0xf98d1b56693fede2,
          0x5842139ca088c385, 0x78e97b4b6211389c}},
        {{0xb846b563b47e762, 0x11487ce473a94e2b, 0x96ad4d3d26fd3042,
          0xfba5fc5a931f20fc, 0xc9c9cb2977eccead, 0x70cb060ebd21a4b,
          0x1c11a225d8866986, 0x160c198638cb516a}},
        {{0x84392746ee39a3c8, 0x2276bf6b03a6b6c0, 0xa1f33ac5827b6b1b,
          0xb24a2a1089314977, 0xb1f4592da74d28bf, 0xcc47085ad2d50fe6,
          0x781f81ae73a678e9, 0xb5af1ea09549d0c6}},
        {{0xcc38727809a43627, 0xa7f4a523fff5c0c7, 0x371f955526bc6fe8,
          0xf2d01dcf69b3fe58, 0x4bb73f338defc436, 0x935238f46034092a,
          0x50a76b6d5c6856a, 0xaa513f41e7f4808c}},
        {{0x9932157ad5031808, 0xbb3ab224fd0331e2, 0x991b26ce1e4d9e87,
          0xdb6830f4566cf6eb, 0x9c965570706f67a4, 0x11a60786d7004d67,
          0x2380d774e8d76fe, 0x4b9b595885a43263}},
        {{0xf0ed6547a3b510ca, 0x2437db654a5e8de6, 0x303bb88ddcad569e,
          0x5844b578237e001b, 0x7e27c196b9c8aa55, 0x2c446a285ee0e27d,
          0x2bd16248e4101c7e, 0xd6e874309d4919ba}},
        {{0xe363c0f65c69db0f, 0xebd8c796247b406e, 0x49e838c360c17631,
          0x867853f86bf569a9, 0xb961d858c669a3d2, 0xf7ba9ed029379b4b,
          0xfa794f47c0706c3a, 0x4a76b0118faacad3}},
        {{0x6d923affa769fd20, 0x3309a13b0b201476, 0x4ffa3800bc3bcb63,
          0x64cb9829473eea59, 0x76b4c2a584d23bca, 0x70e5bea9ee6a4e4,
          0xa159c08bcf637d29, 0xc23b73c867f1d580}},
        {{0xa29537370a1689ba, 0x452ed005e95bec02, 0x94726faff401487b,
          0xbe5f4370ad3db286, 0x43b2a48e6a2b5f77, 0x3f280effc5a4c642,
          0x623aeca8205045fe, 0xa51774bf075db945}},
        {{0x3e69d35747ecf426, 0x8aa0f4fb97ddb6eb, 0x19a1ebeeaf1b7011,
          0xe575d2890f45adb2, 0xf8617aa88fa8529b, 0x48f70cac678c8c75,
          0x595c2e7b789666fa, 0x80687075dfc81635}},
        {{0x929fc5a7805cb68f, 0x191178c3706a18bc, 0x53ffdffcb5c49d39,
          0x742de652260cdc22, 0x9ba730c74f719a1d, 0xd27bac8e2f34fe5c,
          0xea24132d102f9f61, 0x30d1b8792c7fe688}},
        {{0xca449dd892e6e1ed, 0x173adb8609c60719, 0x5dda8c95d8a269b1,
          0x873f0523c973be3c, 0xda527dae58337623, 0x222f3a8ed7018434,
          0xde42ab8dac258b8e, 0x7a0eb0d05b2eb873}},
        {{0xa75a30ac1c474d91, 0xef4107a5cdd1c4eb, 0x4aac985d1f77d9d4,
          0xeda5f3af8ff5f926, 0xa65354e15cb76cda, 0x7873b33b8f9e5252,
          0xc60accda4540efc6, 0xf84a4764b5e7fe89}},
        {{0xc710580748aece4b, 0xe8907dfc7e4ba587, 0x9ab236f7ec28b44c,
          0x788d0cb2317825fc, 0x67905e8ec64492f1, 0xabbeb3b9be2e0fd5,
          0xd65d86797b16f76a, 0x3ae39403a55f9f16}},
        {{0x70d4974a5a1847f9, 0x7ef8072f8443344e, 0x3139e1a1e0cecaa1,
          0x65b2a803a3e273c1, 0xad4c83a9bdcee972, 0x6fc816cf27af7c36,
          0xbca0d181f76252d3, 0xa8bfced297da4f8f}},
        {{0x559affe5047a655f, 0x6f238b3f0471b2f3, 0xeaa78c3533c92f4b,
          0xd73380fe7a9641a8, 0xe0d80ec915d4df03, 0xd5c808e5ea7a6785,
          0x30ca1a56eecf4434, 0x179e72dcdd06ae9a}},
        {{0x2eaeaca1a925366f, 0x34d36e1ed44f74d3, 0x101a86744d4a3b74,
          0xbc8046728b5c8ba0, 0x555378de1d82b70, 0xd713136cdacdc820,
          0x97957689692fc346, 0xe0fcf9631d81f118}},
        {{0x53ffbb5408cc36ed, 0x5d6bf192cd595614, 0x3ddabafbb1c983d1,
          0x10f0a129da80ce09, 0x527113f8c3c9ae61, 0x2f20c32ba530a61f,
          0x23e4a3700777e998, 0xc192ea173585bdac}},
        {{0xcb0686448f156e32, 0x276f74cbf133d239, 0x1fb544430804fc26,
          0xfc253c8a5e3b4821, 0x2518de1b96ef0b61, 0xeae0311d4687d57d,
          0xe5940cce0d61696e, 0x5244b765e492e58b}},
        {{0xb801cc2961b70d2e, 0x8910704e9ac33524, 0x1894c2d650e21fd7,
          0x3f767b63c7a06943, 0x847f835e014c7b2e, 0x24144e00720a7aaa,
          0xae1342d9b915d76f, 0x7a4c0ea127f19fcc}},
        {{0xc327d92c638367de, 0xe879bf42fbc2c55f, 0xd555e12f96782e0e,
          0xb0f5596376aaa551, 0x883dd7bbf07fedae, 0xa4775212e3af501d,
          0x1e4ed939b41655b5, 0xe2619c10d630078d}},
        {{0xdaec231af83a6028, 0x4ff6907175ec562d, 0xef3c97ae651414f9,
          0x9de4c6c3086cd85b, 0x85f8ab6f1e2eef9e, 0xfe50a8b29cad4002,
          0xcb621c6f21705197, 0x337b274524639b29}},
        {{0xfbbac8abe1fbd8f2, 0x15e14a4eb85260a4, 0x791cb6e90da48c54,
          0x71bd66f3e290e83f, 0x82d0d1dfeac8b9ca, 0x3d87fbd894fea91c,
          0x9f1e63731cce5d2f, 0xe35900884589b9d}},
        {{0xaae7c4e871eff79d, 0x6144fc557a8a2641, 0x808d98c9fa60bcf8,
          0x15113ecb0a90c803, 0xfb90c584ed3ef9, 0xff9452fd4d2ee02a,
          0xf4164ba2299570d3, 0xc21e63c1d58315cc}},
        {{0x453ee4b6661e34e2, 0x26fdd7df0d192b45, 0x7fd5ce4d6c14746e,
          0x92cbfbdeeb133fd1, 0xcc76ee091e426047, 0x2eab2456d159a686,
          0xe653aa1978b69708, 0x2459677c43f7c090}},
        {{0x9efd3e0f581517f2, 0x19989a0fdaf4d395, 0xf699603055eccf5,
          0x7c05a05f0a67b6c6, 0xa6de6cd962a15816, 0x5be518a7f9e022c9,
          0xd9b756c90024c7d7, 0x57a5ff136774dc2c}},
        {{0xd288ae2c010393ac, 0x6d3310232939b214, 0x9887b1d89a78b034,
          0x62d720f11ab99ac6, 0x8fcedb93336cb089, 0x1a02c7a86b96ba5e,
          0x88e7192e73f367c4, 0xc28d56881ad3596b}},
        {{0xc2f9dfaba153399c, 0xdaf28534290d6ddb, 0x6f1289ac607c329d,
          0xa2357ee4a5b31259, 0x8d07333f0796280b, 0xf11285c28e2ea5a8,
          0x8020ba41780d2284, 0x3fa50d1d76e90943}},
        {{0xca52d451673c9fb, 0x58a826595f807f63, 0x390ee2346a795af3,
          0x2b245a8949d248d4, 0xa1bfe7852c31e91a, 0xe87d8ecc2ebce3ba,
          0x63fe20c68f188c2a, 0x870511190c741a33}},
        {{0x8e167621bcdcc2ba, 0xcb7de8859c7ca4e8, 0x67cbf1aac4c02efd,
          0xd6865ad5eacdf293, 0x7b8cab96bfc4666d, 0xd69cb650b3440f8f,
          0x12d7e75c41139db, 0x2468ff53e184ef97}},
        {{0xf181bf85b2788e, 0x14655a2889804517, 0x5430b1f060610dbd,
          0x4c0fa683caf2d33a, 0xa7cc60271624010d, 0x902fd7786b85356c,
          0x74705471d504113b, 0x8582df781b9713e0}},
        {{0x4822267af2611e1, 0x8c991c87345fe3a6, 0x898ed42fcd7cad08,
          0x8229ee539f0d2e29, 0xfae57dad3e2c4c7d, 0xf997cc37da0732b1,
          0x34e6dc8ee66f126e, 0x8fe4da5818929ed8}},
        {{0xea1d069df977291d, 0x3c4b489ccb567ca0, 0x85843dcde4fb4c21,
          0x472392fdeffee8b5, 0x3f8ebbc66ede5569, 0xc746714ec1a23438,
          0x6eeb4face2ec59b8, 0x96625c4b4afa3250}},
        {{0x2828c1ecf6dfc47b, 0x534d60c43cb6d433, 0xd8547082681bb153,
          0xa73da5fb014e6db4, 0xcad4b954c86ccc35, 0x99e45abe033c147a,
          0x4f9ea4fdbb91cf2c, 0x2e5c615495fcde0d}},
        {{0xf2b2616a5295d0eb, 0x5e87d80af36bd0b6, 0x28745bc1c721585d,
          0x9c716f39452c71e5, 0xd20f0fe065070659, 0xc27c735ed18b03d0,
          0xd44e569943a8f817, 0xbb2481f13c1af256}},
        {{0xa40bfb6c06ecf3ac, 0x8a52f32fd15c605b, 0x1e5ed92d869d5a7d,
          0x11c734ca75e17a2f, 0x505f174f9dc3cf4d, 0x2bc49b3066a462d5,
          0x4a6e9034d477477e, 0xe50e78b6f812eaf}},
        {{0xf661006ec7d5f15, 0x377dad85953c6142, 0xb08b9e5179658a73,
          0x9f9f5029e88c6a5e, 0xdaad00d18198836e, 0x7ec024a2cba6a267,
          0xd130112240987f1c, 0x4f81380650048e61}},
        {{0xae04517ce0bb6575, 0x78e67f6358d2ef0b, 0x8cea0e412d172600,
          0x295b4c0967138199, 0xe40a8fdebe3536ae, 0xcdfff41ab666ca4a,
          0x639efb960a51063e, 0x2d41bb6fd2ae1420}},
        {{0xc912bd21f54497e3, 0x9ea9c855b6614914, 0x2e8b40917e82a57,
          0x1d6b1b51f977c5ae, 0xed87e52a3167f2b, 0x16f7d609bdb28b52,
          0xa8d5b814c9e0255c, 0x8e39f11690b0bb04}},
        {{0xad0db28bd81a6abc, 0xc2fbfe4394706714, 0x5bb8a1d0fbd8656a,
          0x4a49db8977c0a848, 0x6c7d6443d7135230, 0x999c2914dbad984b,
          0xe7e25baa9204d190, 0x490e63a2a07e9a5f}},
        {{0x2a5e84d263dd34ba, 0xd906e0902111aa81, 0x9969de57ee3344f3,
          0xf0a322cb844b001, 0xe7a0199917934b0f, 0x90eb54ea83b72a43,
          0x6457debcf2b7e1bb, 0x90fbc4dd75da0d04}},
        {{0xcb62658f5a375e39, 0x33b74d8f42acbe50, 0x8fe63dd42cf23ae6,
          0xcf92d098b366b7a, 0xf68a9e7c7967fe5, 0xc437d89093584c8f,
          0x44b85fa7c8a88566, 0x5e34e6bbedf9125e}},
        {{0x3092e7ce17f91b41, 0x423153fdb3443db3, 0xb6cd4a44c10a6e93,
          0x1a4175c0314c8b2d, 0x7a1d1e2d70bc4613, 0xa11622c4e426bb7f,
          0xe8e16f0f86ddcf09, 0xf1b2e2c26ac8449c}},
        {{0xbdf9140c9d81f500, 0x374b6b1a02884ffe, 0x62fa57f9f973ec8c,
          0x6ec43c226c662b05, 0xbf5a312aea9054fd, 0xfa7414564801b327,
          0xdc16a9c7855b41a9, 0xd467e5101ee55117}},
        {{0xc01b7235936181fd, 0xe136e00d9ec6706c, 0x4c43813d6f436f62,
          0x69d021bd2b904964, 0x3fedf3622c963674, 0x502b992cbb05a0b9,
          0x7413461a3b56d071, 0x31af387533d70463}},
        {{0xd4175b7d9d8f9c8d, 0x60a02266b65c3784, 0xaa985d03bc6ff6a7,
          0xd3da46536ea51062, 0x417b885d48f3edf2, 0x9cc4d7462dd5a764,
          0xaf758de6c5e4bcf8, 0xaaa35d9ade50da96}},
        {{0x70db2b8d53c7f89, 0x585e9e00b8376861, 0x394da72eaafca11e,
          0x16aeaa5c8811464c, 0x581d89750c90b5c9, 0x386f2ad7792641c2,
          0x77695e06de2e9acb, 0xa97b771e70491731}},
        {{0x529343b70eafb0e8, 0x2084c896178df1b9, 0xc61fbda7da5625a6,
          0x5bb199c6e5150f8a, 0x48bd0a1fa97f72d2, 0x97e3116944d4c206,
          0x2f02616439c37482, 0x38a246cc1f702bdf}},
        {{0x7ed2216afbe155a7, 0x17fe774e5572b265, 0x95e6073456c55dea,
          0xc6cb5e49841232fe, 0xb7ba9cc330f90f1, 0xf5c9acea5c30f111,
          0x3fc4ae742df488d7, 0xdcc16adc8b966da6}},
        {{0x30052a01d4392dfa, 0xbca12d818f74f7c1, 0x849920f033388ee7,
          0x1d536f627000705b, 0x9ec1905b9c6e7c41, 0x11db8129b022f1a,
          0x18a7dbd088537c57, 0xcd689c1a0d952e0}},
        {{0x39feea16dbfadcf3, 0x1d80cf96b89bcf7c, 0xd2ffd609f2dda8c5,
          0xb421aeaab9cf4055, 0x478b155009922755, 0xc3d6736476bc3013,
          0xcadafdd4dcbf02f4, 0x1d558c8a607fe5f6}},
        {{0xf025865a8b39c731, 0x5e9c53ef0281df9e, 0x8cddf8bcdf65c26c,
          0x71c831e107d20230, 0x4df18a4ffd955be2, 0x3cdb50402f908556,
          0x797860c4d69e6c92, 0xf430b3c85d42181d}},
        {{0x82fc0dd4f24e8d6e, 0xa5084e3bee713571, 0xffd80f8fba431f03,
          0x3b140f11377e6631, 0x63da4709ead26706, 0x5315d9fbb74d2106,
          0x2ffd998e54c954e3, 0x77176455d3fd0f6}},
        {{0xbef9dd2007fc442d, 0xca3fa9ec48cf973c, 0x163c63ac896edaa9,
          0xe876cc134d5fc147, 0x9582d1215c79cfa2, 0xc35823f2f1f0036e,
          0x29934707d05e0921, 0xa716c894246f3f7a}},
        {{0x871c7391019c24ab, 0x5329595d2d002173, 0x45a357287e585fbb,
          0x2c2a6bcb12afb4e6, 0xb788389e7bc292fe, 0x111ed19516d4e3f4,
          0xf393bcc4c2dbbce4, 0x594b2e759e373557}},
        {{0x7877f17c931a60b, 0xe6773f98c063f608, 0x89c78865abe58c75,
          0x450c5bf7e44bfa17, 0xb9104e5a1dee071d, 0x8f60b0c673968430,
          0x5bb1de4d0db40a, 0x42cd220ccbd5b040}},
        {{0x8277de8c0f2d5c63, 0x24af35a8260dd3e2, 0x1b312c620f9a0761,
          0x632025a8253f5788, 0xe248b81945d976c7, 0x9a4972a535470498,
          0xd3e66ef51bc0d771, 0x5d06844315fa2326}},
        {{0xdc30d65dd797130e, 0x1c668497a2ac0db4, 0x88f9376168ffb61d,
          0xa6186c056b0871ba, 0x97984dfa2a4d87e, 0x9c87e04fcb19e3d5,
          0x5e3301f5dbe37500, 0x68df95a381fe3f57}},
        {{0xb994e7e5a41ee9e0, 0xfb6ceb4b0790340e, 0x980fa41d7565a42f,
          0x842f494fd3eff575, 0x91ada913b2709a, 0x20650d4ef3a0fdb3,
          0xd948c45e2791c897, 0x9b7bbab75eec1340}},
        {{0x7f44944de1cb905b, 0xc75b33679c983853, 0x2fe95786144a9e6a,
          0x7c9a272cec82fc3, 0xfde69f10bce3c73c, 0x5c32652184aacf39,
          0x90ae49e2bbbd052d, 0xad7e80c1a3ce039a}},
        {{0x84303e7b6150beb5, 0x30ad388f2d161da0, 0x6cb3d05df92e04ab,
          0x569a836b4c36610c, 0xfc07a3d6b19a3729, 0x1aec0c041f4fb7c6,
          0x4cc941b4fc0bc273, 0x158d0522a698be21}},
        {{0xae16bdae26edc96a, 0x6c4fa1f1dfebc3b2, 0xbfd0065dc9340c7f,
          0x63fb0c1956dabd64, 0x699e7e09164d8b43, 0x806ecf245754df32,
          0x69dc0ca51a99ce61, 0x82cc38f12e9b818}},
        {{0xd28ee493f7b62380, 0xa7a1e42b75032141, 0xa33f68bf3db5cce0,
          0xa43db4641364454c, 0x3b2849f63744fe64, 0xf3a6c874bb3e7299,
          0x9c3cdcc59cacd35c, 0x25c1581ba2b59147}},
        {{0x53a30a840a7085e2, 0xce753d2236c594d8, 0x9b8260f37bc76300,
          0xdcc0d91811ac7d9c, 0xef63413af9205223, 0x77a97a7a632195c3,
          0x162d5049389c09f8, 0xb3487e0a893a2dcd}},
        {{0xa70881589abd9868, 0x4d191bd258833b49, 0x5a68722a3b16ed6e,
          0x3dd72a7497babfd4, 0x13e2e95ab100f414, 0xe0102c4eeb36e71a,
          0x8d1b5581e8e68d12, 0x9c15f81056d562f2}},
        {{0xa3e0a100ff020c24, 0xfd41c0dc36038a7e, 0x5cf3425d2098dda5,
          0x16936d64878dd64e, 0x1e10455fa956884d, 0xc2d76118b76e5b14,
          0x730420e87074ec8a, 0xb808c1ab7e4bedd2}},
        {{0x53abd7e19f774732, 0xcd06d9130eaa5b73, 0x664ef4a244464a9b,
          0x8d2fed47537f1086, 0xf06525ec006c97d0, 0x81f9bd58863e87b0,
          0xc0a23b81bab43ccd, 0x26fc2eb03ae46e90}},
        {{0x3be1e9daf46a8f46, 0xddc9c2f9225677bd, 0xbea50e754e1cc4ad,
          0x86709cf11783f304, 0x19eddaecc4a411a9, 0x8a7b6c44785f206f,
          0x6f141065b88e9736, 0xd1fc52f29d1f15b7}},
        {{0x4aa505ea4af2c2c0, 0x3f3bb6a9cb7147b4, 0xdb2e6fe55223c165,
          0xbb23ec3ae82faf7a, 0x98c0220ac348f9f9, 0x34f29fa258020960,
          0x4ccaade45fc20a06, 0x37fd9d8849b9ad12}},
        {{0x8dd94dd8aa6d3cb1, 0x967d04ed246e8013, 0x7867010c2181059f,
          0xc9e4275f549e92e2, 0xf7498e738f7264a, 0x6981a781ee4ca03d,
          0xf874646e4c46cd8a, 0x834f3d111ee4bdbd}},
        {{0xdcd38628cc4c7118, 0xf3fcbb500d720545, 0x9e566ca0cf40446c,
          0xceec8ace4580a5fc, 0x90fda983903e9bf4, 0x4a82adca578e1888,
          0xf757b0eea7e92abb, 0x43ce6d5494edbf8b}},
        {{0xb53927f93ef0868, 0x61eed9380ca76996, 0x460c0661f8929183,
          0xf34025d244408f5a, 0xf9713245ecc8e75f, 0x8edbf871d293fd52,
          0xe1ba7fdfe3126459, 0x3bbb0b98d1fe3d7d}},
        {{0x45d1b551d25d8a85, 0x2bfcd60961c956a6, 0x894d70ff2014b7ec,
          0x38a63287247a9534, 0xee8742c8731a9c58, 0x2d9a4c5729281ed5,
          0x67a832f1db965491, 0x7dc4b8d615ae9b75}},
        {{0x7450d2b7dd0d764a, 0x5d5cda61be938f5b, 0x4f7b112aee852635,
          0x860d11fa0e369ef, 0x3a73e3364552aaa4, 0xf95cebef8c290d7e,
          0xc57f33b75208c866, 0x49590c669769ba43}},
        {{0x130e1d5c812fce44, 0x8c06827c285266ad, 0xa4c5358baafe706f,
          0x39f9421494402116, 0xd3ac860b10605665, 0x7f2ed9c86662c7a9,
          0x50e1af48b327578f, 0xb1ad7955feba36ed}},
        {{0xbcb6d407610e1220, 0x22b13f65b11e6f9c, 0x528c55eb81fb3a29,
          0x323e4151fc1be6b2, 0x198b1587397585e3, 0x1a912242e9399285,
          0x5cdd4c613ac85e35, 0x435abad392b2f0}},
        {{0xa7ffd3bdf68415ca, 0x71f7169f9c53644f, 0x50194b158fbae11,
          0x57cf6cfcae904ce9, 0x897f36e28fc3f6bc, 0x9d9dadb5cac3d4ed,
          0xcce3b09171a6cf16, 0x978765b9ae9aafc}},
        {{0xef3ae09d257a87f6, 0x6706be80a9ca8572, 0xdec5809390f3cf4e,
          0xa9ceeca64a45c38e, 0xb4b1b945089fd936, 0xed9fb4c9d60a2109,
          0x7389e8d1d3cace59, 0xf7873597908f4449}},
        {{0xd2202bd5cc99a4ba, 0xe3e20e68db2a0c5a, 0x755ef209a5632cee,
          0x7deefac387ea9590, 0x356fe3bc1e3cf092, 0xc5b9f2845ba315c,
          0x495be80836e9e668, 0x3c4781ee3e493551}},
        {{0x9587801c98289a6b, 0x19ba8ed05dbf2e62, 0xadc2a3a81644b414,
          0xeba7684c7df7962f, 0x8a003d3d25b3eb84, 0xbd4dc07252e81cdb,
          0x69d02d3d28c83c5, 0xf4ee3eb45058156b}},
        {{0x97f1bcdbd29688b8, 0x99bd8d5cafb0c7bc, 0x8458bd970e3f5ac2,
          0x410a1b3c2ba59320, 0x2adc7a453f78587c, 0x53a3605338b48494,
          0x4ea8f5cd4daf4dab, 0xe17ae6fbdfd8c79}},
        {{0x826ee7f0fe8ea00f, 0x13b5fead37fcc393, 0xabca58829205564f,
          0xd93a11808e0e689d, 0x9c40d4fe5857102, 0x2ee6ba473fbff4c7,
          0x594704b94c182865, 0x83b8a1d12f4b09e4}},
        {{0xaf89d7fb2c1b14b2, 0xe2c9831f1112c7bb, 0xc23643772401e93,
          0x59ffe50e3ff93d6f, 0xbe0728c97dec4f5f, 0x4844d6b90cb4e96e,
          0x89f636bba936cf22, 0xd05d4f7c548899fe}},
        {{0x1f16252f068b98d0, 0x134fb529a6397f00, 0x61c7a6b13f907eee,
          0xdeb20662b0338364, 0xac6e4c78e8abe226, 0xfc1a1b54ea6722b5,
          0x9dcdceb1373bf76, 0x9c26c7b7e8bb6bad}},
        {{0x93b966262b91ced4, 0x92dab731520735f9, 0xb60bb887ab82627a,
          0x5199aa63aeb7dbb5, 0xc5c3d93a6a89506e, 0x317ae3102f88b0ba,
          0xe141f169dbccce13, 0xe38ec448b5e1c068}},
        {{0x1a6580c769aae732, 0x8947d3a76a35ef3e, 0x4f9e6f9777494de4,
          0x289917522f98bc24, 0xe90015e348c4270a, 0xf0ed43bdba42668f,
          0x5e2bb173ba53594b, 0xfda7c100f0b20cfa}},
        {{0x7ff7367d66304a6e, 0x2200219274ba8e7f, 0x139295660e4d0ee4,
          0x3fcaac30f57b9fe8, 0xcada2bd1bd535dce, 0xc8dee542c79d97fd,
          0xe85074e795e1f65b, 0x3983907cb10fb5ac}},
        {{0xfbec47a15099e0a4, 0x8716c65e640a2c92, 0xbaf968408ce4ef4,
          0x8f7286e6b4f0f6e9, 0x2154fd23d2dc93cc, 0x67a2dd49bddeb90f,
          0xc4747d707a369eae, 0x12b142222759a64b}},
        {{0xa117156707f91a01, 0x112f00e1d95b6f65, 0x64ecbbe2e1672e84,
          0xfec97278c849458f, 0x421d214215131044, 0x4404cb521f7311dd,
          0x10d2def235e0ee58, 0xeb24f3b62498e216}},
        {{0xa433bdd2af2966e0, 0x1bc954603cfc9f3f, 0xc0538135f410c12a,
          0x777b1a7a419d2f84, 0xc8cc68b46b85a80d, 0x66e3160a5f79ca76,
          0xaf61899e403484ff, 0xc02eb1c02d97b134}},
        {{0x34e2a7551aed9118, 0x5a2fa0cd845998f1, 0x14613d22725333ee,
          0x5bd18529defe09d4, 0x29dd3b056bf88821, 0xad5f462d6ab9d1de,
          0xb0387ff598bd27f2, 0x148ea1b392980fb7}},
        {{0x6726e0c2e5fdf25c, 0x8eebb69aba514700, 0x6239caff90d45d71,
          0x6030dc08d98002e1, 0x27a86872d5c59178, 0x70eca21fa7c7fece,
          0x5425ed88fb852123, 0xd3e309603c835f4a}},
        {{0x5aa7e2cf79c2fbcc, 0x4eae4592c5f1d37c, 0x74eeb9cae4d88b4e,
          0xc57a058bce91991c, 0x8583272b1901777a, 0x929f15c1f473f31,
          0x50e1f00957d8f8fd, 0xb011e5012731883}},
        {{0x125c727ad5a91683, 0x335d245e020fb1a3, 0xfe4852ef3b096726,
          0x9dd7a0b1f885b82b, 0x19307b779681b2ad, 0xe628f934a01fa645,
          0xf4d93e4de86f7683, 0x9880eb5f5cd462c9}},
        {{0xc1016bfa5ced20c9, 0x96fd2c23bb1d4e09, 0x8bd3efd0b096f5be,
          0xb1fd879a4cf571d, 0xe6c036d13f6c1ac0, 0xf69cd09c4b2f2ab7,
          0x5228a30ad725d1ae, 0xdbf9f423febec722}},
        {{0x2f8e451c08cdcffb, 0xdabf5072976beac3, 0x72cfb49344cbc3e9,
          0x17ceb8c9449b4db5, 0xa95f5ae34a2f0f4, 0xdc8f16542f4e0f56,
          0x5731100b533fe1, 0x9ecf076767e397db}},
        {{0x26789580a01ba8ee, 0x83a87d0b6b7d64c0, 0xbd1c11033b66f89c,
          0x2f32bb583747a0bd, 0x56b4fb33b3bbadc0, 0xdeda6fc0ea91fc96,
          0x5215f442edce5693, 0x2c04ee04247aa24d}},
        {{0x308f84b0d008c1f9, 0xa5fa720535f70a69, 0x57bede9bf052cb2,
          0xa8740a9a3753cbdc, 0xdcd1986ae6bead11, 0x26ab1c6f7e6f59b5,
          0x4b45920df5f718b4, 0xa3b42be5f85182cc}},
        {{0x3a154225facdc0c5, 0xaf6e635ece65657d, 0xc30bc73b771b9fbd,
          0x794f6c11121ea80, 0x830a238287df38dd, 0x9e1ca5eee57bc9b0,
          0x1bc468dd67ef70ab, 0xadbf5cb5d69e629f}},
        {{0xa4c95da5aea26641, 0x38da168b9e525986, 0xa30bd6db6e0fb780,
          0x394a915509e31294, 0xbc393f9320c1f720, 0x44777f91f90158cc,
          0x19123f0e51649bde, 0x29082529862e8168}},
        {{0xc2abdbb8832d83f8, 0x7e517a390ffe58ad, 0xa3aebd2d39a3ace6,
          0x972ada4a8945d2db, 0x6afe60d26cb721b, 0xd0aa3f6ba0056ff5,
          0x25ff530f3f0cc77c, 0x4302da8768bdaad2}},
        {{0xae430fbcd2b23fd8, 0x2903fe26bdd2091d, 0xb798d12e46c00279,
          0xb29ed581e5eafd13, 0x647fc94823966cce, 0xf2dfff2c6519a32f,
          0xbb9551941e4623d6, 0x12f5dee507fdbd93}},
        {{0x10383a9a5fb42536, 0xfb6ae1f3213cee08, 0x612743c207c1214a,
          0x68b489e73597b27d, 0x73d5da11c0128f69, 0x7c09a5995f3323f7,
          0x1cd0c2ed3b2f4fe, 0x2df81d0327498890}},
        {{0xce90cdecaec2b7e2, 0x63724e130d2c6ff3, 0x94dea84775e7430a,
          0x6df71bca317ccc32, 0xe0da6b8b3e3d5678, 0x8105cb7b28350273,
          0x14d46d27d9a1ef6c, 0xb754904a9c3ba32b}},
        {{0xc6400c9ab87729d2, 0xcade2b09ccd6e774, 0xa1a03650650ee7da,
          0x424f4ad947270652, 0xe58f0b82522e5f8b, 0xb432f912a0adf25a,
          0x7c222b3cfcb1534b, 0x4501ce144522369a}},
        {{0x9181483736f42f9b, 0xf92187ae6efa3bb6, 0x6e5baec86ad47134,
          0x5fca16be5c2793a7, 0xc29ed46374ed772c, 0xe5b55322e4cb29d1,
          0xc63e747c49199287, 0x21868971d706c21d}},
        {{0x4bfa6fa24ff9c27e, 0x49dc1848281b4ddf, 0x2025886a388d4ef1,
          0xcee93bb654301544, 0xa7b9052770790b49, 0x6b358213c70d23bf,
          0x53e6800285240d85, 0x9ee14b31af67c9d5}},
        {{0x19eaf72bd7958ff7, 0xb76b02f9b0a9b0ac, 0x1b7c38010fe6439b,
          0xeacc61987e69dd5, 0xfbb17ca81074120f, 0x2f34c20cb10a2686,
          0xccbe0ae157d8e8c7, 0x7b71a948919990f3}},
        {{0x7c1780e00b37829c, 0xc62d20444d843f62, 0x7747abc71d2704c0,
          0x3a226699a4454faa, 0xdf02d99b007bbfe9, 0x34d84012eabed1db,
          0x4a4a1a08de981bef, 0x88cf9b821c80e238}},
        {{0x6f4cb9ffaf753fe4, 0x138b88f965dff23b, 0xb705912a5b598d64,
          0xef6cbc74b1c0ec4f, 0xaaded173eee70f30, 0x51d1a7fd1238fe31,
          0x8886672160464140, 0xb7691a2c840b1d6e}},
        {{0x87a828c6b060a9d5, 0xdf65c4d578844ada, 0xf0e47179e00af4fb,
          0xbfc0da4db4229213, 0x92b404548740a2d8, 0x9fbfa859a0e2066,
          0xdfdf3892a1f76f8c, 0x8f2b489e6d21a09e}},
        {{0x95bc91317ff51f2a, 0x8fc6f76a441bbd1d, 0x6d05849804c2efae,
          0xe9017ea12949f4f8, 0x6a85b2eaf2b1a6ce, 0x8f40bf255ec40fd,
          0xbcdf289447b489fb, 0x3388ea2716b0361c}},
    }};

}  // namespace ut::detail
