# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""Tests for Notes app"""

from __future__ import absolute_import

from testtools.matchers import Equals
from autopilot.matchers import Eventually

from notes_app.tests import NotesAppTestCase

import sqlite3


class TestDelete(NotesAppTestCase):
    """Tests deleting notes"""

    def setUp(self):
        # Setup the database before starting the app
        self.setup_db()
        super(TestDelete, self).setUp()

    def setup_db(self):
        note_data = """{ "elements" : [
            {"content":"This is a note.","type":"text"}
        ]}"""
        path = self.ensure_db()
        conn = sqlite3.connect(path)
        cursor = conn.cursor()
        cursor.execute("DELETE FROM notes")
        cursor.execute("INSERT INTO notes (date, note) "
                       "VALUES ('2013-04-07', '" + note_data + "')")
        conn.commit()
        conn.close()

    def test_slide_to_delete_right(self):
        note = self.main_window.get_notes()[0]
        listitem = self.main_window.get_note_listitem(note)
        listitem.swipe_to_delete('right')
        listitem.confirm_removal()
        self.assertThat(
            lambda: len(self.main_window.get_notes()), Eventually(Equals(0)))

    def test_slide_to_delete_left(self):
        note = self.main_window.get_notes()[0]
        listitem = self.main_window.get_note_listitem(note)
        listitem.swipe_to_delete('left')
        listitem.confirm_removal()
        self.assertThat(
            lambda: len(self.main_window.get_notes()), Eventually(Equals(0)))

    def test_slide_incomplete(self):
        note = self.main_window.get_notes()[0]

        # Verify that sliding the note just a little bit won't remove the note
        note_x, note_y, note_w, note_h = note.globalRect
        self.pointing_device.move(note_x + note_w / 2, note_y + note_h / 2)
        self.pointing_device.press()
        self.pointing_device.move(
            note_x + note_w / 2 + 10, note_y + note_h / 2)
        self.pointing_device.release()

        self.main_window.get_note_listitem(note).swipingState.wait_for("")
        self.assertThat(
            lambda: len(self.main_window.get_notes()), Eventually(Equals(1)))

    def test_slide_and_put_back(self):
        note = self.main_window.get_notes()[0]

        # Verify that sliding the note all the way and then putting it back
        # where it was will not delete it
        note_x, note_y, note_w, note_h = note.globalRect
        self.pointing_device.move(note_x + note_w / 2, note_y + note_h / 2)
        self.pointing_device.press()
        self.pointing_device.move(note_x, note_y + note_h / 2)
        self.pointing_device.move(note_x + note_w / 2, note_y + note_h / 2)
        self.pointing_device.release()

        self.main_window.get_note_listitem(note).swipingState.wait_for("")
        self.assertThat(
            lambda: len(self.main_window.get_notes()), Eventually(Equals(1)))
