/* pdu-gre.c
  
   Builder for GRE PDUs

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "pdu-gre.h"

static void
builder(const GNode *pdu, void *dest)
{
    struct gre_header *gre;
    uint16_t eth_type, *proto;
    const GNode *next_pdu;
    void *field;

    gre = dest;

    gre->version = 0;
    gre->do_cksum = _pb_pdata(pdu, "do_cksum") ? 1 : 0;
    gre->reserved1 = 0;
    gre->reserved2 = 0;

    /* Put in place the protocol type this GRE packet is carrying */
    if (  !(field = _pb_pdata(pdu, "proto") )
	&& (next_pdu = pb_nextpdu(pdu) ) ) {
	/*
	 * User didn't specify the ethertype so we try to guess the right
	 * value based on the next PDU.
	 */

	proto = g_hash_table_lookup(ethertypes, pb_getname(next_pdu) );
	eth_type = proto ? *proto : 0;
    } else
	/*
	 * Use for IP protocol whatever the user specified. Note that if
	 * there is no user-specified IP protocol *and* there is no
	 * next PDU then we end up here, but that's okay because
	 * num_next(NULL) is 0.
	 */
	eth_type = num_next(field);

    eth_type = htons(eth_type);
    SSVAL(gre, offsetof(struct gre_header, protocol), eth_type);
}

#if 0
void
pdu_grehdr_dumper(pdu_t *p, const char *prefix)
{
    struct grehdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Protocol: %s\n", prefix, num_info(hdr_data->protocol) );
    printf("%s    Do checksum: %s\n", prefix,
	   hdr_data->do_cksum ? "yes" : "no");
}
#endif

static void
greopt_cksum(const GNode *option, void *dest, void *curr_pdu_hdr,
	     void *prev_pdu _U_)
{
    struct gre_cksum *cksum;
    uint16_t u16;
    size_t hdr_len, opts_len, payload_len;
    const GNode *pdu;

    cksum = dest;

    pdu = _pb_parent_pdu(option);
    hdr_len = ( (struct node_data *) pdu->data)->_data_pdu.hdr_len;
    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
    payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

    SSVAL(cksum, offsetof(struct gre_cksum, reserved), 0);

    SSVAL(cksum, offsetof(struct gre_cksum, cksum), 0);
    u16 = _pb_cksum(curr_pdu_hdr, hdr_len + opts_len + payload_len);
    SSVAL(cksum, offsetof(struct gre_cksum, cksum), u16);
}


static pdu_t pdu_gre = {
    .name = "gre",
    .description = "GRE packet",
    .documented_in = "RFC 2784",
    .len = sizeof(struct gre_header),
    .fields = (field_t []) {
	{.name = "src", .type = PDU_FTYPE_MACADDR},
	{.name = "dst", .type = PDU_FTYPE_MACADDR},
	{.name = "proto", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &builder
};

static const pdu_option_t pdu_option_grecksum = {
    .name = "gre",
    .description = "GRE checksum",
    .len = sizeof(struct gre_cksum),
    .build = &greopt_cksum
};

void
_pb_register_gre(void)
{
    _pb_register_protocol(&pdu_gre);

    _pb_register_protocol_option("gre", &pdu_option_grecksum);
}
