﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.Collections.Immutable;
using Microsoft.AspNetCore.Razor.Test.Common;
using Microsoft.CodeAnalysis.Razor.Tooltip;
using Microsoft.VisualStudio.Text.Adornments;
using Xunit;
using Xunit.Abstractions;

namespace Microsoft.VisualStudio.LegacyEditor.Razor.Completion;

public class VisualStudioDescriptionFactoryTest(ITestOutputHelper testOutput) : ToolingTestBase(testOutput)
{
    [Fact]
    public void CreateClassifiedDescription_SingleDescription_NoSeparator()
    {
        // Arrange
        var factory = new VisualStudioDescriptionFactory();
        var description = new AggregateBoundAttributeDescription(ImmutableArray.Create(
            new BoundAttributeDescriptionInfo("TheReturnType", "TheTypeName", "ThePropertyName", "The documentation")));

        // Act
        var result = factory.CreateClassifiedDescription(description);

        // Assert
        Assert.DoesNotContain(VisualStudioDescriptionFactory.SeparatorElement, result.Elements);
    }

    [Fact]
    public void CreateClassifiedDescription_MultipleDescription_Separator()
    {
        // Arrange
        var factory = new VisualStudioDescriptionFactory();
        var description = new AggregateBoundAttributeDescription(ImmutableArray.Create(
            new BoundAttributeDescriptionInfo("TheReturnType", "TheTypeName", "ThePropertyName", "The documentation"),
            new BoundAttributeDescriptionInfo("TheReturnType2", "TheTypeName2", "ThePropertyName2", "The documentation2")));

        // Act
        var result = factory.CreateClassifiedDescription(description);

        // Assert
        Assert.Contains(VisualStudioDescriptionFactory.SeparatorElement, result.Elements);
    }

    [Fact]
    public void CreateClassifiedDescription_RepresentsReturnType()
    {
        // Arrange
        var factory = new VisualStudioDescriptionFactory();
        var description = new AggregateBoundAttributeDescription(ImmutableArray.Create(
            new BoundAttributeDescriptionInfo("TheReturnType", "TheTypeName", "ThePropertyName", "The documentation")));

        // Act
        var result = factory.CreateClassifiedDescription(description);

        // Assert
        var flattened = FlattenToStrings(result);
        Assert.Contains(description.DescriptionInfos[0].ReturnTypeName, flattened);
    }

    [Fact]
    public void CreateClassifiedDescription_RepresentsTypeName()
    {
        // Arrange
        var factory = new VisualStudioDescriptionFactory();
        var description = new AggregateBoundAttributeDescription(ImmutableArray.Create(
            new BoundAttributeDescriptionInfo("TheReturnType", "TheTypeName", "ThePropertyName", "The documentation")));

        // Act
        var result = factory.CreateClassifiedDescription(description);

        // Assert
        var flattened = FlattenToStrings(result);
        Assert.Contains(description.DescriptionInfos[0].TypeName, flattened);
    }

    [Fact]
    public void CreateClassifiedDescription_RepresentsPropertyName()
    {
        // Arrange
        var factory = new VisualStudioDescriptionFactory();
        var description = new AggregateBoundAttributeDescription(ImmutableArray.Create(
            new BoundAttributeDescriptionInfo("TheReturnType", "TheTypeName", "ThePropertyName", "The documentation")));

        // Act
        var result = factory.CreateClassifiedDescription(description);

        // Assert
        var flattened = FlattenToStrings(result);
        Assert.Contains(description.DescriptionInfos[0].PropertyName, flattened);
    }

    [Fact]
    public void CreateClassifiedDescription_RepresentsDocumentation()
    {
        // Arrange
        var factory = new VisualStudioDescriptionFactory();
        var description = new AggregateBoundAttributeDescription(ImmutableArray.Create(
            new BoundAttributeDescriptionInfo("TheReturnType", "TheTypeName", "ThePropertyName", "The documentation")));

        // Act
        var result = factory.CreateClassifiedDescription(description);

        // Assert
        var flattened = FlattenToStrings(result);
        Assert.Contains(description.DescriptionInfos[0].Documentation, flattened);
    }

    [Fact]
    public void CreateClassifiedDescription_CanSimplifyKeywordReturnTypes()
    {
        // Arrange
        var factory = new VisualStudioDescriptionFactory();
        var description = new AggregateBoundAttributeDescription(ImmutableArray.Create(
            new BoundAttributeDescriptionInfo("System.String", "TheTypeName", "ThePropertyName", "The documentation")));

        // Act
        var result = factory.CreateClassifiedDescription(description);

        // Assert
        var flattened = FlattenToStrings(result);
        Assert.DoesNotContain(description.DescriptionInfos[0].ReturnTypeName, flattened);
        Assert.Contains("string", flattened);
    }

    [Fact]
    public void CreateClassifiedDescription_CanRepresentMultipleDescriptions()
    {
        // Arrange
        var factory = new VisualStudioDescriptionFactory();
        var description = new AggregateBoundAttributeDescription(ImmutableArray.Create(
            new BoundAttributeDescriptionInfo("System.String", "TheTypeName", "ThePropertyName", "The documentation"),
            new BoundAttributeDescriptionInfo("System.Int32", "TheSecondTypeName", "TheSecondPropertyName", "The second documentation")));

        // Act
        var result = factory.CreateClassifiedDescription(description);

        // Assert
        var flattened = FlattenToStrings(result);
        Assert.Contains(description.DescriptionInfos[0].TypeName, flattened);
        Assert.Contains(description.DescriptionInfos[1].TypeName, flattened);
        Assert.Contains(description.DescriptionInfos[0].Documentation, flattened);
        Assert.Contains(description.DescriptionInfos[1].Documentation, flattened);
    }

    public IReadOnlyList<string> FlattenToStrings(ContainerElement element)
    {
        var flattenedList = new List<string>();
        foreach (var child in element.Elements)
        {
            switch (child)
            {
                case ContainerElement childContainer:
                    var flattened = FlattenToStrings(childContainer);
                    flattenedList.AddRange(flattened);
                    break;
                case ClassifiedTextElement textElement:
                    foreach (var run in textElement.Runs)
                    {
                        flattenedList.Add(run.Text);
                    }

                    break;
            }
        }

        return flattenedList;
    }
}
