#include "elddisp.h"
#include "mixerhandle.h"
#include <QGroupBox>
#include <QHeaderView>
#include <QScrollArea>

eldDisp::eldDisp(snd_hctl_elem_t *elem) {
	m_elem= elem;
	snd_ctl_elem_value_malloc(&m_val);
	snd_ctl_elem_info_malloc(&m_info);
	snd_hctl_elem_info(elem, m_info);
	m_count= snd_ctl_elem_info_get_count(m_info);
	if(m_count)
	{
		setText("Show info...");
		m_win= new QWidget();
		m_win->resize(540, 540);
		m_win->setWindowTitle("ELD on device " + QString::number(snd_hctl_elem_get_device(elem)) + " index " + QString::number(snd_hctl_elem_get_index(elem)));
		QVBoxLayout *lWin= new QVBoxLayout(m_win);
		QScrollArea *s= new QScrollArea;
		lWin->addWidget(s);
		s->setWidgetResizable(1);
		QWidget *wInner= new QWidget;
		s->setWidget(wInner);
		m_lInner= new QVBoxLayout(wInner);
		QGroupBox *gb= new QGroupBox("Basic info");
		m_lInner->addWidget(gb);
		m_lForm= new QFormLayout(gb);
		m_lInner->addWidget(new QLabel("Supported formats (SAD blocks):"));
		m_tSad= new QTableWidget(0, 4, wInner);
		m_tSad->setEditTriggers(QAbstractItemView::NoEditTriggers);
		m_tSad->verticalHeader()->hide();
		m_lInner->addWidget(m_tSad);
		m_lbExt= new QLabel();
		m_lInner->addWidget(m_lbExt);
		connect(this, SIGNAL(pressed()), this, SLOT(showWin()));
	} else
	{
		setText("Not available");
		setDisabled(1);
	}
}

eldDisp::~eldDisp() {
	snd_ctl_elem_value_free(m_val);
	snd_ctl_elem_info_free(m_info);
	snd_hctl_elem_set_callback(m_elem, nullptr);
	snd_hctl_elem_set_callback_private(m_elem, nullptr);
}

QString eldDisp::sratesstring(uint8_t val) {
	QString str= val & 1 ? "32; " : "";
	str+= val & 2 ? "44.1; " : "";
	str+= val & 4 ? "48; " : "";
	str+= val & 8 ? "88.2; " : "";
	str+= val & 16 ? "96; " : "";
	str+= val & 32 ? "176.4; " : "";
	str+= val & 64 ? "192; " : "";
	str+= val & 128 ? "Bit 7 set; " : "";
	return str.left(str.lastIndexOf(';'));
}

QString eldDisp::supportflags(uint8_t val) {
	QString str= val & 1 ? "HDCP; " : "";
	str+= val & 2 ? "ACP/ISRC; " : "";
	return str.left(str.lastIndexOf(';'));
}

QString eldDisp::conntype(uint8_t val) {
	val&= 0b00001100;
	switch(val)
	{
	case 0:
		return "HDMI";
	case 1:
		return "DisplayPort";
	case 2:
		return "Unknown (2)";
	default: // case 3
		return "Unknown (3)";
	}
}

void eldDisp::updat() {
	if(m_win->isHidden()) // no need to updat when not opened
		return;

	QLayoutItem *oldItem;
	while((oldItem= m_lForm->takeAt(0)) != nullptr)
	{
		delete oldItem->widget();
		delete oldItem;
	}
	m_tSad->clear();
	snd_hctl_elem_read(m_elem, m_val);
	ushort i;
	uint8_t tmpbyte;
	QString tmpstr;

	// device name
	tmpstr= "";
	ushort mnLen= (snd_ctl_elem_value_get_byte(m_val, 4) & 0b00011111);
	for(i= 20; i < 20 + mnLen; i++)
		tmpstr.append((char)(snd_ctl_elem_value_get_byte(m_val, i)));
	QString manid= QChar('@' + (snd_ctl_elem_value_get_byte(m_val, 16) & 0b01111100) / 4);
	manid+= QChar('@' + (snd_ctl_elem_value_get_byte(m_val, 16) & 0b00000011) * 8 + (snd_ctl_elem_value_get_byte(m_val, 17) & 0b11100000) / 32);
	manid+= QChar('@' + (snd_ctl_elem_value_get_byte(m_val, 17) & 0b00011111));
	QString prdid= QString("0x%1").arg(snd_ctl_elem_value_get_byte(m_val, 19), 2, 16, '0');
	prdid+= QString("%1").arg(snd_ctl_elem_value_get_byte(m_val, 18), 2, 16, '0');
	m_lForm->addRow("Device:", new QLabel(tmpstr + " (Manufacturer: " + manid + "; Product: " + prdid + ')'));

	m_lForm->addRow("Connection:", new QLabel(conntype(snd_ctl_elem_value_get_byte(m_val, 5))));

	// port id
	tmpstr= "0x";
	for(i= 15; i > 7; i--)
	{
		tmpstr+= QString("%1").arg(snd_ctl_elem_value_get_byte(m_val, i), 2, 16, '0');
	}
	m_lForm->addRow("Port ID:", new QLabel(tmpstr));

	// edid version
	tmpbyte= (snd_ctl_elem_value_get_byte(m_val, 4) & 0b11100000) / 32;
	switch(tmpbyte)
	{
	case 0:
		tmpstr= "None";
		break;
	case 1:
		tmpstr= "CEA-861";
		break;
	case 2:
		tmpstr= "CEA-861A";
		break;
	case 3:
		tmpstr= "CEA-861B/C/D";
		break;
	case 4:
		tmpstr= "Reserved";
		break;
	default:
		tmpstr= "Unknown (" + QString::number(tmpbyte) + ")";
	}
	m_lForm->addRow("EDID version:", new QLabel(tmpstr));

	// eld version
	tmpbyte= (snd_ctl_elem_value_get_byte(m_val, 0) & 0b11111000) / 8;
	switch(tmpbyte)
	{
	case 2:
		tmpstr= "CEA-861D or below";
		break;
	case 31:
		tmpstr= "Non-standard";
		break;
	default:
		tmpstr= "Unknown (" + QString::number(tmpbyte) + ")";
	}
	m_lForm->addRow("ELD version:", new QLabel(tmpstr));

	m_lForm->addRow("Baseline length:", new QLabel(QString::number(snd_ctl_elem_value_get_byte(m_val, 2))));
	m_lForm->addRow("Support flags:", new QLabel(supportflags(snd_ctl_elem_value_get_byte(m_val, 5))));
	tmpbyte= snd_ctl_elem_value_get_byte(m_val, 6);
	if(tmpbyte)
		m_lForm->addRow("A/V sync delay:", new QLabel(QString::number((tmpbyte - 1) * 2) + " ms"));
	tmpbyte= snd_ctl_elem_value_get_byte(m_val, 0) & 0b00000111;
	if(tmpbyte)
		m_lForm->addRow("Channels:", new QLabel(QString::number(tmpbyte + 1)));

	// speaker config
	tmpstr= snd_ctl_elem_value_get_byte(m_val, 7) & 1 ? "Front; " : "";
	tmpstr+= snd_ctl_elem_value_get_byte(m_val, 7) & 2 ? "LFE; " : "";
	tmpstr+= snd_ctl_elem_value_get_byte(m_val, 7) & 4 ? "Center; " : "";
	tmpstr+= snd_ctl_elem_value_get_byte(m_val, 7) & 8 ? "Back L/R; " : "";
	tmpstr+= snd_ctl_elem_value_get_byte(m_val, 7) & 16 ? "Back Center; " : "";
	tmpstr+= snd_ctl_elem_value_get_byte(m_val, 7) & 32 ? "Center L/R; " : "";
	tmpstr+= snd_ctl_elem_value_get_byte(m_val, 7) & 64 ? "Back Center L/R; " : "";
	tmpstr+= snd_ctl_elem_value_get_byte(m_val, 7) & 128 ? "Front Wide; " : "";
	m_lForm->addRow("Speaker config:", new QLabel(tmpstr.left(tmpstr.lastIndexOf(';'))));

	// remaining bytes
	tmpbyte= snd_ctl_elem_value_get_byte(m_val, 1);
	if(tmpbyte)
		m_lForm->addRow("Byte 1:", new QLabel(QString::number(tmpbyte, 16)));
	tmpbyte= snd_ctl_elem_value_get_byte(m_val, 3);
	if(tmpbyte)
		m_lForm->addRow("Byte 3:", new QLabel(QString::number(tmpbyte, 16)));

	QString map_format[]= {
		"Undefined (0)",
		"Linear PCM",
		"AC-3",
		"MPEG-1 layer 1/2",
		"MP3",
		"MPEG-2",
		"AAC-LC",
		"DTS",
		"ATRAC",
		"DSD",
		"Dolby Digital Plus",
		"DTS-HD",
		"Dolby TrueHD / Atmos",
		"DST",
		"WMA Pro"
	};

	QString map_extformat[]= {
		"Undefined (ext. 0)",
		"Unknown (ext. 1)",
		"Unknown (ext. 2)",
		"Unknown (ext. 3)"
		"HE-AAC",
		"HE-AAC v2",
		"AAC-LC (from ext. byte)",
		"DRA",
		"HE-AAC + MPEG Surround",
		"Undefined (ext. 9)",
		"AAC-LC + MPEG Surround",
		"MPEG-H 3D Audio",
		"Dolby AC-4",
		"LPCM 3D",
		"Auro-Cx",
		"USAC"
	};

	tmpbyte= (snd_ctl_elem_value_get_byte(m_val, 5) & 0b11110000) / 16;
	m_tSad->setHorizontalHeaderLabels({ "Channels", "Format", "Sample rates (kHz)", "Format dependent value" });
	uchar sadsStart= 20 + mnLen;
	for(uchar i= 0; i < tmpbyte; i++)
	{
		uchar byte= sadsStart + i * 3;
		m_tSad->insertRow(i);

		// channels
		m_tSad->setItem(i, 0, new QTableWidgetItem(QString::number((snd_ctl_elem_value_get_byte(m_val, byte) & 0b00000111) + 1)));

		// format
		ushort format= (snd_ctl_elem_value_get_byte(m_val, byte) & 0b01111000) / 8;
		if(format == 15)
		{
			tmpstr= map_extformat[(snd_ctl_elem_value_get_byte(m_val, byte + 2 & 0b01111000)) / 8];
			tmpstr+= (snd_ctl_elem_value_get_byte(m_val, byte + 2) & 128) / 128 ? "; Ext. bit 7 set" : "";
		} else
		{
			tmpstr= map_format[format];
			tmpstr+= (snd_ctl_elem_value_get_byte(m_val, byte) & 128) / 128 ? "; Bit 7 set" : "";
		}
		m_tSad->setItem(i, 1, new QTableWidgetItem(tmpstr));

		// sample rates
		m_tSad->setItem(i, 2, new QTableWidgetItem(sratesstring(snd_ctl_elem_value_get_byte(m_val, byte + 1))));

		// format dependent val
		if(format == 1)
		{
			tmpstr= snd_ctl_elem_value_get_byte(m_val, byte + 2) & 1 ? "16-bit; " : "";
			tmpstr+= snd_ctl_elem_value_get_byte(m_val, byte + 2) & 2 ? "20-bit; " : "";
			tmpstr+= snd_ctl_elem_value_get_byte(m_val, byte + 2) & 4 ? "24-bit; " : "";
			tmpstr+= snd_ctl_elem_value_get_byte(m_val, byte + 2) & 0b11111000 ? "High bits set; " : "";
			m_tSad->setItem(i, 3, new QTableWidgetItem(tmpstr.left(tmpstr.lastIndexOf(';'))));
		} else if(format > 1 && format < 9)
		{
			tmpstr= QString::number(snd_ctl_elem_value_get_byte(m_val, byte + 2) * 8);
			m_tSad->setItem(i, 3, new QTableWidgetItem(tmpstr + " kb/s (max)"));
		} else if(format > 8 && format < 15)
			m_tSad->setItem(i, 3, new QTableWidgetItem(QString("0x%1").arg(snd_ctl_elem_value_get_byte(m_val, byte + 2), 2, 16, '0')));
		else
			m_tSad->setItem(i, 3, new QTableWidgetItem(QString("0x%1").arg(snd_ctl_elem_value_get_byte(m_val, byte + 2) & 0b00000111, 2, 16, '0')));
	}
	m_tSad->resizeColumnsToContents();

	// extended data
	tmpstr= "";
	for(uchar i= sadsStart + tmpbyte * 3; i < m_count; i++)
		tmpstr+= QString("0x%1 ").arg(snd_ctl_elem_value_get_byte(m_val, i), 2, 16, '0');
	if(tmpstr.trimmed() == "")
		m_lbExt->hide();
	else
	{
		m_lbExt->show();
		m_lbExt->setText("Extended data: " + tmpstr.trimmed());
	}
}

void eldDisp::showWin() {
	m_win->show();
	updat();
	snd_hctl_elem_set_callback_private(m_elem, new mixerHandle::callbackData{ mixerHandle::EL, this });
	snd_hctl_elem_set_callback(m_elem, callback);
}
