/*  $Id$
 *
 *  Copyright (C) 2003 Choe Hwanjin(krisna@kldp.org)
 *  Copyright (c) 2006 Remco den Breeje <remco@sx.mine.nu>
 *  Copyright (c) 2008 Diego Ongaro <ongardie@gmail.com>
 *  Copyright (c) 2016 Landry Breuil <landry@xfce.org>
 *  Copyright (C) 2020 David Vachulka <arch_dvx@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <libxfce4ui/libxfce4ui.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfce4panel/xfce-panel-plugin.h>
#include "dxreminders.h"
#include "dxreminders-dialogs.h"

#define PLUGIN_WEBSITE "https://dxreminders.dxsolutions.org/plugin"

/* Layouts */
static const gchar *layout_strs[] = {
    N_("Date, then time"),
    N_("Time, then date"),
    N_("Date only"),
    N_("Time only")
};

typedef enum {
    /* standard format item; string is replaced with an example date or time */
    DX_COMBOBOX_ITEM_TYPE_STANDARD,
    /* custom format item; text is translated */
    DX_COMBOBOX_ITEM_TYPE_CUSTOM,
    /* inactive separator */
    DX_COMBOBOX_ITEM_TYPE_SEPARATOR,
} dx_combobox_item_type;

typedef struct {
    gchar *item;
    dx_combobox_item_type type;
} dx_combobox_item;

/*
 * Builtin formats are derived from xfce4-panel-clock.patch by Nick Schermer.
 */
static const dx_combobox_item dx_combobox_date[] = {
    { "%Y-%m-%d",       DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%Y %B %d",       DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "---",            DX_COMBOBOX_ITEM_TYPE_SEPARATOR },  /* placeholder */
    { "%m/%d/%Y",       DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%B %d, %Y",      DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%b %d, %Y",      DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%A, %B %d, %Y",  DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%a, %b %d, %Y",  DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "---",            DX_COMBOBOX_ITEM_TYPE_SEPARATOR },  /* placeholder */
    { "%d/%m/%Y",       DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%d %B %Y",       DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%d %b %Y",       DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%A, %d %B %Y",   DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%a, %d %b %Y",   DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "---",            DX_COMBOBOX_ITEM_TYPE_SEPARATOR },  /* placeholder */
    { "%x",             DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "---",            DX_COMBOBOX_ITEM_TYPE_SEPARATOR },  /* placeholder */
    { N_("Custom..."),  DX_COMBOBOX_ITEM_TYPE_CUSTOM    }
};
#define DX_COMBOBOX_DATE_COUNT (sizeof(dx_combobox_date)/sizeof(dx_combobox_item))

static const dx_combobox_item dx_combobox_time[] = {
    { "%H:%M",          DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%H:%M:%S",       DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "---",            DX_COMBOBOX_ITEM_TYPE_SEPARATOR },  /* placeholder */
    { "%l:%M %P",       DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "%l:%M:%S %P",    DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "---",            DX_COMBOBOX_ITEM_TYPE_SEPARATOR },  /* placeholder */
    { "%X",             DX_COMBOBOX_ITEM_TYPE_STANDARD  },
    { "---",            DX_COMBOBOX_ITEM_TYPE_SEPARATOR },  /* placeholder */
    { N_("Custom..."),  DX_COMBOBOX_ITEM_TYPE_CUSTOM    }
};
#define DX_COMBOBOX_TIME_COUNT (sizeof(dx_combobox_time)/sizeof(dx_combobox_item))

static void dxreminders_configure_response(GtkWidget *dialog, G_GNUC_UNUSED int response, dxremindersPlugin *dxreminders)
{
    /* remove the dialog data from the plugin */
    g_object_set_data(G_OBJECT (dxreminders->plugin), "dialog", NULL);
    /* destroy the properties dialog */
    gtk_widget_destroy(dialog);
    /* save the plugin */
    dxreminders_save(dxreminders->plugin, dxreminders);
}

static GtkWidget *get_frame_box(const gchar *label, GtkWidget **container_return)
{
    GtkWidget *bin;
    GtkWidget *frame;
    frame = xfce_gtk_frame_box_new(label, &bin);
    gtk_widget_set_margin_top(GTK_WIDGET(bin), 6);
    gtk_widget_set_margin_start(GTK_WIDGET(bin), 0);
    G_GNUC_BEGIN_IGNORE_DEPRECATIONS
    gtk_alignment_set_padding(GTK_ALIGNMENT(bin), 0, 0, 12, 0);
    G_GNUC_END_IGNORE_DEPRECATIONS
    *container_return = bin;
    return frame;
}

/*
 * Read layout from combobox and set sensitivity
 */
static void dxreminders_layout_changed(GtkComboBox *cbox, dxremindersPlugin *dx)
{
    dx_layout layout;
    /* read layout */
    layout = gtk_combo_box_get_active(cbox);
    switch(layout)
    {
        case LAYOUT_DATE:
            gtk_widget_show(dx->date_font_hbox);
            gtk_widget_hide(dx->date_tooltip_label);
            gtk_widget_hide(dx->time_font_hbox);
            gtk_widget_show(dx->time_tooltip_label);
            break;
        case LAYOUT_TIME:
            gtk_widget_hide(dx->date_font_hbox);
            gtk_widget_show(dx->date_tooltip_label);
            gtk_widget_show(dx->time_font_hbox);
            gtk_widget_hide(dx->time_tooltip_label);
            break;
        default:
            gtk_widget_show(dx->date_font_hbox);
            gtk_widget_hide(dx->date_tooltip_label);
            gtk_widget_show(dx->time_font_hbox);
            gtk_widget_hide(dx->time_tooltip_label);
    }
    dxreminders_apply_layout(dx, layout);
    dxreminders_update(dx);
}

/*
 * show and read fonts and inform datetime about it
 */
static void dxreminders_font_selection_cb(GtkWidget *widget, dxremindersPlugin *dx)
{
    GtkWidget *dialog;
    gchar *fontname;
    const gchar *previewtext;
    gint target, result;
    gchar *font_name;
    if(widget == dx->date_font_selector)
    {
        target = DATE;
        fontname = dx->date_font;
        previewtext = gtk_label_get_text(GTK_LABEL(dx->date_label));
    }
    else /*time_font_selector */
    {
        target = TIME;
        fontname = dx->time_font;
        previewtext = gtk_label_get_text(GTK_LABEL(dx->time_label));
    }
    dialog = gtk_font_chooser_dialog_new(_("Select font"), GTK_WINDOW(gtk_widget_get_toplevel(widget)));
    gtk_font_chooser_set_font(GTK_FONT_CHOOSER(dialog), fontname);
    if(G_LIKELY(previewtext != NULL))
    {
        gtk_font_chooser_set_preview_text(GTK_FONT_CHOOSER(dialog), previewtext);
    }
    result = gtk_dialog_run(GTK_DIALOG(dialog));
    if(result == GTK_RESPONSE_OK || result == GTK_RESPONSE_ACCEPT)
    {
        font_name = gtk_font_chooser_get_font(GTK_FONT_CHOOSER(dialog));
        if(font_name != NULL)
        {
            gtk_button_set_label(GTK_BUTTON(widget), font_name);
            if(target == DATE)
                dxreminders_apply_font(dx, font_name, NULL);
            else
                dxreminders_apply_font(dx, NULL, font_name);
            g_free (font_name);
        }
    }
    gtk_widget_destroy(dialog);
}

/*
 * Row separator for format-comboboxes of date and time
 * derived from xfce4-panel-clock.patch by Nick Schermer
 */
static gboolean combo_box_row_separator(GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
    const dx_combobox_item *items = (dx_combobox_item *)data;
    gint current;
    GtkTreePath *path;
    path = gtk_tree_model_get_path(model, iter);
    current = gtk_tree_path_get_indices(path)[0];
    gtk_tree_path_free(path);
    return items[current].type == DX_COMBOBOX_ITEM_TYPE_SEPARATOR;
}

/*
 * Read date format from combobox and set sensitivity
 */
static void date_format_changed(GtkComboBox *cbox, dxremindersPlugin *dx)
{
    const gint active = gtk_combo_box_get_active(cbox);
    switch(dx_combobox_date[active].type)
    {
        case DX_COMBOBOX_ITEM_TYPE_STANDARD:
            /* hide custom text entry box and tell datetime which format is selected */
            gtk_widget_hide(dx->date_format_entry);
            dxreminders_apply_format(dx, dx_combobox_date[active].item, NULL);
            break;
        case DX_COMBOBOX_ITEM_TYPE_CUSTOM:
            /* initialize custom text entry box with current format and show the box */
            gtk_entry_set_text(GTK_ENTRY(dx->date_format_entry), dx->date_format);
            gtk_widget_show(dx->date_format_entry);
            break;
        default:
            break; /* separators should never be active */
    }
    dxreminders_update(dx);
}

/*
 * Read time format from combobox and set sensitivity
 */
static void time_format_changed(GtkComboBox *cbox, dxremindersPlugin *dx)
{
    const gint active = gtk_combo_box_get_active(cbox);
    switch(dx_combobox_time[active].type)
    {
        case DX_COMBOBOX_ITEM_TYPE_STANDARD:
            /* hide custom text entry box and tell datetime which format is selected */
            gtk_widget_hide(dx->time_format_entry);
            dxreminders_apply_format(dx, NULL, dx_combobox_time[active].item);
            break;
        case DX_COMBOBOX_ITEM_TYPE_CUSTOM:
            /* initialize custom text entry box with current format and show the box */
            gtk_entry_set_text(GTK_ENTRY(dx->time_format_entry), dx->time_format);
            gtk_widget_show(dx->time_format_entry);
            break;
        default:
            break; /* separators should never be active */
    }
    dxreminders_update(dx);
}

/*
 * read values from date and time entry and inform datetime about it
 */
static gboolean dxreminders_entry_change_cb(GtkWidget *widget, G_GNUC_UNUSED GdkEventFocus *ev, dxremindersPlugin *dx)
{
    const gchar *format;
    format = gtk_entry_get_text(GTK_ENTRY(widget));
    if(format != NULL)
    {
        if(widget == dx->date_format_entry)         /* date */
            dxreminders_apply_format(dx, format, NULL);
        else if(widget == dx->time_format_entry)    /* or time */
            dxreminders_apply_format(dx, NULL, format);
    }
    dxreminders_update(dx);
    return FALSE;
}

/* signal handler for calendar checkbox */
//static void dxreminders_calendar_changed(GtkToggleButton *toggle_button, dxremindersPlugin *dx)
//{
//    if(gtk_toggle_button_get_active(toggle_button))
//        dx->calendar = TRUE;
//    else
//        dx->calendar = FALSE;
//}

void dxreminders_configure(XfcePanelPlugin *plugin, dxremindersPlugin *dxreminders)
{
    guint i;
    gchar *str;
    GtkWidget   *dlg,
                *frame,
                *vbox,
                *hbox,
                *layout_combobox,
                *time_combobox,
                *date_combobox,
                *label,
                *button,
                *entry,
                *bin;
//                *calendar_checkbox;
    GtkSizeGroup  *sg;
    gint i_custom; /* index of custom menu item */
    xfce_textdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");
    /* create the dialog */
    dlg = xfce_titled_dialog_new_with_buttons(_("dxreminders plugin"),
                                              GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(plugin))),
                                              GTK_DIALOG_DESTROY_WITH_PARENT,
                                              "gtk-close", GTK_RESPONSE_OK,
                                              NULL);
    /* center dialog on the screen */
    gtk_window_set_position(GTK_WINDOW(dlg), GTK_WIN_POS_CENTER);
    /* set dialog icon */
    gtk_window_set_icon_name(GTK_WINDOW (dlg), "xfce4-dxreminders-plugin");
    /* link the dialog to the plugin, so we can destroy it when the plugin
     * is closed, but the dialog is still open */
    g_object_set_data(G_OBJECT (plugin), "dialog", dlg);
    gtk_container_set_border_width(GTK_CONTAINER(gtk_dialog_get_content_area(GTK_DIALOG(dlg))), 5);
    gtk_box_set_spacing(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dlg))), 5);
    /* size group */
    sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
    /* layout frame */
    frame = get_frame_box(_("Layout"), &bin);
    gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dlg))), frame, FALSE, FALSE, 0);
    /* vbox */
    vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 6);
    gtk_container_add(GTK_CONTAINER(bin),vbox);
    /* hbox */
    hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 12);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    /* Format label */
    label = gtk_label_new(_("Format:"));
    gtk_label_set_xalign (GTK_LABEL(label), 0.0f);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sg, label);
    /* Layout combobox */
    layout_combobox = gtk_combo_box_text_new();
    gtk_box_pack_start(GTK_BOX(hbox), layout_combobox, TRUE, TRUE, 0);
    for(i=0; i < LAYOUT_COUNT; i++)
        gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(layout_combobox), _(layout_strs[i]));
    gtk_combo_box_set_active(GTK_COMBO_BOX(layout_combobox), dxreminders->layout);
    g_signal_connect(G_OBJECT(layout_combobox), "changed", G_CALLBACK(dxreminders_layout_changed), dxreminders);
    /* show frame */
    gtk_widget_show_all(frame);
    /* Date frame */
    dxreminders->date_frame = get_frame_box(_("Date"), &bin);
    gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dlg))), dxreminders->date_frame, FALSE, FALSE, 0);
    /* vbox */
    vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 6);
    gtk_container_add(GTK_CONTAINER(bin),vbox);
    /* tooltip label */
    str = g_markup_printf_escaped("<span style=\"italic\">%s</span>", _("The date will appear in a tooltip."));
    dxreminders->date_tooltip_label = gtk_label_new(str);
    g_free(str);
    gtk_label_set_xalign (GTK_LABEL(dxreminders->date_tooltip_label), 0.0f);
    gtk_label_set_use_markup(GTK_LABEL(dxreminders->date_tooltip_label), TRUE);
    gtk_box_pack_start(GTK_BOX(vbox), dxreminders->date_tooltip_label, FALSE, FALSE, 0);
    /* hbox */
    dxreminders->date_font_hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 12);
    gtk_box_pack_start(GTK_BOX(vbox), dxreminders->date_font_hbox, FALSE, FALSE, 0);
    /* font label */
    label = gtk_label_new(_("Font:"));
    gtk_label_set_xalign (GTK_LABEL (label), 0.0f);
    gtk_box_pack_start(GTK_BOX(dxreminders->date_font_hbox), label, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sg, label);
    /* font button */
    button = gtk_button_new_with_label(dxreminders->date_font);
    gtk_box_pack_start(GTK_BOX(dxreminders->date_font_hbox), button, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(dxreminders_font_selection_cb), dxreminders);
    dxreminders->date_font_selector = button;
    /* hbox */
    hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 12);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    /* format label */
    label = gtk_label_new(_("Format:"));
    gtk_label_set_xalign (GTK_LABEL (label), 0.0f);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sg, label);
    /* format combobox */
    date_combobox = gtk_combo_box_text_new();
    gtk_box_pack_start(GTK_BOX(hbox), date_combobox, TRUE, TRUE, 0);
    i_custom = 0;
    for(i=0; i < DX_COMBOBOX_DATE_COUNT; i++)
    {
        switch(dx_combobox_date[i].type)
        {
            case DX_COMBOBOX_ITEM_TYPE_STANDARD:
                str = dxreminders_do_utf8strftime(dx_combobox_date[i].item, g_date_time_new_now_local());
                gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(date_combobox), str);
                g_free(str);
                /* set active
                * strcmp isn't fast, but it is done only once while opening the dialog
                */
                if(strcmp(dxreminders->date_format, dx_combobox_date[i].item) == 0)
                    gtk_combo_box_set_active(GTK_COMBO_BOX(date_combobox), i);
                break;
            case DX_COMBOBOX_ITEM_TYPE_CUSTOM:
                gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(date_combobox), _(dx_combobox_date[i].item));
                i_custom = i;
                break;
            case DX_COMBOBOX_ITEM_TYPE_SEPARATOR: /* placeholder item does not need to be translated */
                gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(date_combobox), dx_combobox_date[i].item);
                break;
            default:
                break;
        }
    }
    /* if no item activated -> activate custom item */
    if(gtk_combo_box_get_active(GTK_COMBO_BOX(date_combobox)) < 0)
        gtk_combo_box_set_active(GTK_COMBO_BOX(date_combobox), i_custom);
    gtk_combo_box_set_row_separator_func(GTK_COMBO_BOX(date_combobox), combo_box_row_separator, (gpointer)dx_combobox_date, NULL);
    g_signal_connect(G_OBJECT(date_combobox), "changed", G_CALLBACK(date_format_changed), dxreminders);
    dxreminders->date_format_combobox = date_combobox;

    /* format entry */
    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), dxreminders->date_format);
    gtk_widget_set_halign (GTK_WIDGET (entry), GTK_ALIGN_END);
    gtk_box_pack_end(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(entry), "focus-out-event", G_CALLBACK(dxreminders_entry_change_cb), dxreminders);
    dxreminders->date_format_entry = entry;
    gtk_widget_show_all(dxreminders->date_frame);
    /*
     * time frame
    */
    dxreminders->time_frame = get_frame_box(_("Time"), &bin);
    gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dlg))), dxreminders->time_frame, FALSE, FALSE, 0);
    /* vbox */
    vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 6);
    gtk_container_add(GTK_CONTAINER(bin), vbox);
    /* tooltip label */
    str = g_markup_printf_escaped("<span style=\"italic\">%s</span>", _("The time will appear in a tooltip."));
    dxreminders->time_tooltip_label = gtk_label_new(str);
    g_free(str);
    gtk_label_set_xalign (GTK_LABEL(dxreminders->time_tooltip_label), 0.0f);
    gtk_label_set_use_markup(GTK_LABEL(dxreminders->time_tooltip_label), TRUE);
    gtk_box_pack_start(GTK_BOX(vbox), dxreminders->time_tooltip_label, FALSE, FALSE, 0);
    /* hbox */
    dxreminders->time_font_hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 12);
    gtk_box_pack_start(GTK_BOX(vbox), dxreminders->time_font_hbox, FALSE, FALSE, 0);
    /* font label */
    label = gtk_label_new(_("Font:"));
    gtk_label_set_xalign (GTK_LABEL (label), 0.0f);
    gtk_box_pack_start(GTK_BOX(dxreminders->time_font_hbox), label, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sg, label);
    /* font button */
    button = gtk_button_new_with_label(dxreminders->time_font);
    gtk_box_pack_start(GTK_BOX(dxreminders->time_font_hbox), button, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(dxreminders_font_selection_cb), dxreminders);
    dxreminders->time_font_selector = button;
    /* hbox */
    hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 12);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    /* format label */
    label = gtk_label_new(_("Format:"));
    gtk_label_set_xalign (GTK_LABEL (label), 0.0f);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sg, label);
    /* format combobox */
    time_combobox = gtk_combo_box_text_new();
    gtk_box_pack_start(GTK_BOX(hbox), time_combobox, TRUE, TRUE, 0);
    i_custom = 0;
    for(i=0; i < DX_COMBOBOX_TIME_COUNT; i++)
    {
        switch(dx_combobox_time[i].type)
        {
            case DX_COMBOBOX_ITEM_TYPE_STANDARD:
                str = dxreminders_do_utf8strftime(dx_combobox_time[i].item, g_date_time_new_now_local());
                gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(time_combobox), str);
                g_free(str);
                /* set active
                 * strcmp isn't fast, but it is done only once while opening the dialog
                 */
                if(strcmp(dxreminders->time_format, dx_combobox_time[i].item) == 0)
                    gtk_combo_box_set_active(GTK_COMBO_BOX(time_combobox), i);
                break;
            case DX_COMBOBOX_ITEM_TYPE_CUSTOM:
                gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(time_combobox), _(dx_combobox_time[i].item));
                i_custom = i;
                break;
            case DX_COMBOBOX_ITEM_TYPE_SEPARATOR: /* placeholder item does not need to be translated */
                gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(time_combobox), dx_combobox_time[i].item);
                break;
            default:
                break;
        }
    }
    /* if no item activated -> activate custom item */
    if(gtk_combo_box_get_active(GTK_COMBO_BOX(time_combobox)) < 0)
        gtk_combo_box_set_active(GTK_COMBO_BOX(time_combobox), i_custom);
    gtk_combo_box_set_row_separator_func(GTK_COMBO_BOX(time_combobox), combo_box_row_separator, (gpointer)dx_combobox_time, NULL);
    g_signal_connect(G_OBJECT(time_combobox), "changed", G_CALLBACK(time_format_changed), dxreminders);
    dxreminders->time_format_combobox = time_combobox;
    /* format entry */
    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), dxreminders->time_format);
    gtk_widget_set_halign (GTK_WIDGET (entry), GTK_ALIGN_END);
    gtk_box_pack_end(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(entry), "focus-out-event", G_CALLBACK (dxreminders_entry_change_cb), dxreminders);
    dxreminders->time_format_entry = entry;
    gtk_widget_show_all(dxreminders->time_frame);
//    /* Calendar frame */
//    frame = get_frame_box(_("Mini Calendar"), &bin);
//    gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dlg))), frame, FALSE, FALSE, 0);
//    /* vbox */
//    vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 6);
//    gtk_container_add(GTK_CONTAINER(bin),vbox);
//    /* hbox */
//    hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 12);
//    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
//    /* Calendar label */
//    label = gtk_label_new(_("Enable:"));
//    gtk_label_set_xalign (GTK_LABEL(label), 0.0f);
//    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
//    gtk_size_group_add_widget(sg, label);
//    /* Calendar checkbox */
//    calendar_checkbox = gtk_check_button_new();
//    gtk_box_pack_start(GTK_BOX(hbox), calendar_checkbox, TRUE, TRUE, 0);
//    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(calendar_checkbox), dxreminders->calendar);
//    gtk_widget_set_tooltip_text(calendar_checkbox, _("If enabled, left-click shows calendar instead dxReminders main window"));
//    g_signal_connect (GTK_TOGGLE_BUTTON(calendar_checkbox), "toggled", G_CALLBACK (dxreminders_calendar_changed), dxreminders);
//    /* show frame */
//    gtk_widget_show_all(frame);
    /* connect the reponse signal to the dialog */
    g_signal_connect (G_OBJECT (dlg), "response", G_CALLBACK(dxreminders_configure_response), dxreminders);
    /* set sensitivity for all widgets */
    dxreminders_layout_changed(GTK_COMBO_BOX(layout_combobox), dxreminders);
    date_format_changed(GTK_COMBO_BOX(date_combobox), dxreminders);
    time_format_changed(GTK_COMBO_BOX(time_combobox), dxreminders);
    /* show the entire dialog */
    gtk_widget_show (dlg);
}

void dxreminders_about(G_GNUC_UNUSED XfcePanelPlugin * plugin)
{
   /* about dialog code. you can use the GtkAboutDialog
    * or the XfceAboutInfo widget */
    const gchar *auth[] =
    {
      "David Vachulka <arch_dvx@users.sourceforge.net>",
      NULL
    };
    gtk_show_about_dialog(NULL,
                          "logo-icon-name", "xfce4-dxreminders-plugin",
                          "license",        xfce_get_license_text(XFCE_LICENSE_TEXT_GPL),
                          "version",        PACKAGE_VERSION,
                          "program-name",   PACKAGE_NAME,
                          "comments",       _("dxreminders plugin"),
                          "website",        PLUGIN_WEBSITE,
                          "copyright",      _("Copyright \xc2\xa9 2020 David Vachulka\n"),
                          "authors",        auth,
                          NULL);
}
