+/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
+/* vim: set ts=2 et sw=2 tw=80: */
+/* This Source Code Form is subject to the terms of the Mozilla Public
+ * License, v. 2.0. If a copy of the MPL was not distributed with this file,
+ * You can obtain one at http://mozilla.org/MPL/2.0/. */
+
+#include "nss.h"
+#include "secpkcs7.h"
+
+#include "gtest/gtest.h"
#include "nss_scoped_ptrs.h"

namespace nss_test {

// This is an invalid PKCS7 message. Among other things, it contains some
// unknown hash OIDs. This should fail to parse, but it should be safe to try.
static const uint8_t p7_with_unknown_hashes[] = {
    0x30, 0x4d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x07,
    0x02, 0xa0, 0x40, 0x30, 0x3e, 0x02, 0x01, 0x20, 0x31, 0x27, 0x30, 0x0b,
    0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x05, 0x30,
    0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x05,
    0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02,
    0x04, 0x30, 0x10, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
    0x07, 0x01, 0xa0, 0x03, 0x04, 0x01, 0x00};

// This is an invalid PKCS7 message. It contains multiple hash OIDs (that's not
// what makes it invalid). When it fails to parse, the associated digest data
// structures should be freed correctly.
static const uint8_t p7_with_multiple_hashes[] = {
    0x30, 0x4d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x07,
    0x02, 0xa0, 0x40, 0x30, 0x3e, 0x02, 0x01, 0x20, 0x31, 0x27, 0x30, 0x0b,
    0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x03, 0x30,
    0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x02,
    0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02,
    0x04, 0x30, 0x10, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
    0x07, 0x01, 0xa0, 0x03, 0x04, 0x01, 0x00};

class P7ImportTest : public ::testing::Test {};

TEST_F(P7ImportTest, FailSafeWithUnknownHashes) {
  ScopedSEC_PKCS7DecoderContext dcx(SEC_PKCS7DecoderStart(
      nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr));
  ASSERT_TRUE(dcx);
  SECStatus rv = SEC_PKCS7DecoderUpdate(
      dcx.get(), reinterpret_cast<const char*>(p7_with_unknown_hashes),
      sizeof(p7_with_unknown_hashes));
  ASSERT_EQ(SECFailure, rv);
}

TEST_F(P7ImportTest, NoLeakWithMultipleHashes) {
  ScopedSEC_PKCS7DecoderContext dcx(SEC_PKCS7DecoderStart(
      nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr));
  ASSERT_TRUE(dcx);
  SECStatus rv = SEC_PKCS7DecoderUpdate(
      dcx.get(), reinterpret_cast<const char*>(p7_with_multiple_hashes),
      sizeof(p7_with_multiple_hashes));
  ASSERT_EQ(SECFailure, rv);
}

}  // namespace nss_test