/***************************************************************
 * Name:      calendarpopup.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2020
 * License:   GPL3
 **************************************************************/

#include "calendarpopup.h"
#include "dxdefs.h"
#include "daydialog.h"
#include "data/engine.h"
#include "dxsettings.h"
#include "dxutils.h"

IMPLEMENT_ABSTRACT_CLASS(CalendarPopup, wxMiniFrame)

BEGIN_EVENT_TABLE(CalendarPopup, wxMiniFrame)
    EVT_CLOSE(CalendarPopup::OnClose)
    EVT_CALENDAR(ID_POPUP_CALENDAR, CalendarPopup::OnCalendar)
END_EVENT_TABLE()

CalendarPopup::CalendarPopup(wxWindow *parent)
    : wxMiniFrame(parent, wxID_ANY, _("Calendar"), wxDefaultPosition, wxDefaultSize, wxCAPTION|wxCLOSE_BOX|wxRESIZE_BORDER), m_shown(false)
{
    m_showhidden = dxsettings.hidden();
    if(dxsettings.notaskbar()) SetWindowStyleFlag(wxCAPTION|wxCLOSE_BOX|wxRESIZE_BORDER|wxFRAME_NO_TASKBAR);
    wxPanel *panel = new wxPanel(this);
    m_calendar = new dxCalendarCtrl(panel, ID_POPUP_CALENDAR, true);
    m_calendar->markDays(true, m_showhidden);
    m_mainSizer = new wxBoxSizer(wxVERTICAL);
    m_mainSizer->Add(m_calendar, 0, wxCENTRE|wxALL, 5);
    panel->SetSizer(m_mainSizer);
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
    panel->Bind(wxEVT_CHAR_HOOK, &CalendarPopup::OnKeyUp, this);
}

bool CalendarPopup::shown() const
{
    return m_shown;
}

void CalendarPopup::setShown(bool shown)
{
    m_shown = shown;
}

void CalendarPopup::updateCalendar()
{
    m_calendar->updateColors();
    m_calendar->updateFonts();
    m_calendar->Resize();
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
    if(m_shown)
    {
        SetPosition(m_calpos);
        Hide();
        Show(true);
    }
}

void CalendarPopup::markDays(bool showhidden)
{
    m_showhidden = showhidden;
    m_calendar->markDays(true, showhidden);
}

void CalendarPopup::setPosition(const wxPoint &pos)
{
    m_calpos = pos;
    SetPosition(pos);
}

void CalendarPopup::setValue(const wxDateTime &date)
{
    m_calendar->SetDate(date);
}

void CalendarPopup::OnCalendar(wxCalendarEvent &event)
{
    DayDialog dialog(this, event.GetDate());
    dialog.ShowModal();
    m_calendar->markDays(true, m_showhidden);
}

void CalendarPopup::OnKeyUp(wxKeyEvent &event)
{
    if(event.GetKeyCode() == WXK_ESCAPE)
    {
        Show(false);
        m_shown = false;
    }
    event.Skip();
}

void CalendarPopup::OnClose(wxCloseEvent& /*event*/)
{
    Show(false);
    m_shown = false;
}
