// Copyright (c) 2025 Valentin Lobstein (Chocapikk) <balgogan@protonmail.com>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package scanner

import (
	"fmt"

	"github.com/Chocapikk/wpprobe/internal/logger"
	"github.com/Chocapikk/wpprobe/internal/progress"
)

func getScanMode(mode string) string {
	if mode == "" {
		return "stealthy"
	}
	return mode
}

func performScan(ctx ScanExecutionContext, scanMode string) ([]string, PluginDetectionResult, map[string]string) {
	switch scanMode {
	case "stealthy":
		detected, result := performStealthyScan(ctx)
		return detected, result, nil
	case "bruteforce":
		return performBruteforceScan(ctx)
	case "hybrid":
		return performHybridScan(ctx)
	default:
		logger.DefaultLogger.Warning(
			"Unknown scan mode '" + ctx.Opts.ScanMode + "', defaulting to stealthy",
		)
		detected, result := performStealthyScan(ctx)
		return detected, result, nil
	}
}

func performStealthyScan(ctx ScanExecutionContext) ([]string, PluginDetectionResult) {
	// Check context before starting
	if ctx.Ctx != nil {
		select {
		case <-ctx.Ctx.Done():
			return nil, PluginDetectionResult{}
		default:
		}
	}

	setProgressMessage(ctx.Progress, isFileScan(ctx.Opts), "🔎 Discovering plugins from HTML...")

	httpCfg := HTTPConfigFromOpts(ctx.Opts)
	htmlSlugs, err := discoverPluginsFromHTML(ctx.Ctx, ctx.Target, httpCfg)
	if err != nil {
		// If context was cancelled, return early
		if ctx.Ctx != nil && ctx.Ctx.Err() != nil {
			return nil, PluginDetectionResult{}
		}
		logger.DefaultLogger.Warning(fmt.Sprintf("HTML discovery failed on %s: %v", ctx.Target, err))
	}

	// Check context between steps
	if ctx.Ctx != nil {
		select {
		case <-ctx.Ctx.Done():
			return nil, PluginDetectionResult{}
		default:
		}
	}

	setProgressMessage(ctx.Progress, isFileScan(ctx.Opts), "🔎 Scanning REST API endpoints...")

	endpointsData, err := loadEndpointsData()
	if err != nil {
		return nil, PluginDetectionResult{}
	}

	endpoints := FetchEndpoints(ctx.Ctx, ctx.Target, httpCfg)

	result := buildDetectionResult(endpoints, endpointsData, htmlSlugs)

	if len(result.Detected) == 0 {
		return nil, result
	}
	return result.Detected, result
}

func performBruteforceScan(ctx ScanExecutionContext) ([]string, PluginDetectionResult, map[string]string) {
	plugins, err := LoadPluginsFromFile(ctx.Opts.PluginList)
	if err != nil {
		logger.DefaultLogger.Error("Failed to load plugin list: " + err.Error())
		return nil, PluginDetectionResult{}, nil
	}

	progress := setupBruteforceProgress(ctx.Opts, ctx.Progress, len(plugins))
	defer finishBruteforceProgress(progress, ctx.Opts)

	bruteReq := BruteforceRequest{
		Target:   ctx.Target,
		Plugins:  plugins,
		Threads:  ctx.Opts.Threads,
		Progress: progress,
		HTTP:     HTTPConfigFromOpts(ctx.Opts),
	}
	detected, versions := BruteforcePlugins(bruteReq)

	detectedList, result := buildBruteforceResult(detected, versions)
	return detectedList, result, versions
}

func performHybridScan(ctx ScanExecutionContext) ([]string, PluginDetectionResult, map[string]string) {
	logger.DefaultLogger.Info("Starting hybrid scan: first stealthy, then brute-force...")

	stealthyList, stealthyRes := performStealthyScan(ctx)

	if len(stealthyList) == 0 {
		return performBruteforceScan(ctx)
	}

	finishProgressIfNeeded(ctx.Progress)

	remaining := calculateRemainingPlugins(stealthyList, ctx.Opts)
	if len(remaining) == 0 {
		return stealthyList, stealthyRes, nil
	}

	brutefound, versions := performBruteforceOnRemaining(ctx, remaining)

	detected, result := combineHybridResults(stealthyList, stealthyRes, brutefound)
	return detected, result, versions
}

func performBruteforceOnRemaining(ctx ScanExecutionContext, remaining []string) ([]string, map[string]string) {
	var bruteBar *progress.ProgressManager
	if ctx.Opts.File == "" {
		bruteBar = progress.NewProgressBar(len(remaining), "🔎 Bruteforcing remaining")
		defer bruteBar.Finish()
	}

	bruteReq := BruteforceRequest{
		Target:   ctx.Target,
		Plugins:  remaining,
		Threads:  ctx.Opts.Threads,
		Progress: bruteBar,
		HTTP:     HTTPConfigFromOpts(ctx.Opts),
	}
	detected, versions := BruteforcePlugins(bruteReq)
	return detected, versions
}

