package pagerduty

import (
	"context"
	"fmt"

	"github.com/google/go-querystring/query"
)

// Acknowledgement is the data structure of an acknowledgement of an incident.
type Acknowledgement struct {
	At           string    `json:"at,omitempty"`
	Acknowledger APIObject `json:"acknowledger,omitempty"`
}

// PendingAction is the data structure for any pending actions on an incident.
type PendingAction struct {
	Type string `json:"type,omitempty"`
	At   string `json:"at,omitempty"`
}

// Assignment is the data structure for an assignment of an incident
type Assignment struct {
	At       string    `json:"at,omitempty"`
	Assignee APIObject `json:"assignee,omitempty"`
}

// AlertCounts is the data structure holding a summary of the number of alerts by status of an incident.
type AlertCounts struct {
	Triggered uint `json:"triggered,omitempty"`
	Resolved  uint `json:"resolved,omitempty"`
	All       uint `json:"all,omitempty"`
}

// Priority is the data structure describing the priority of an incident.
type Priority struct {
	APIObject
	Name        string `json:"name,omitempty"`
	Description string `json:"description,omitempty"`
}

// ResolveReason is the data structure describing the reason an incident was resolved
type ResolveReason struct {
	Type     string    `json:"type,omitempty"`
	Incident APIObject `json:"incident"`
}

// IncidentBody is the datastructure containing data describing the incident.
type IncidentBody struct {
	Type    string `json:"type,omitempty"`
	Details string `json:"details,omitempty"`
}

// Assignee is an individual assigned to an incident.
type Assignee struct {
	Assignee APIObject `json:"assignee"`
}

// Occurrence is the data around whether this is a reocurring issue.
type Occurrence struct {
	Count     uint   `json:"count,omitempty"`
	Frequency uint   `json:"frequency,omitempty"`
	Category  string `json:"category,omitempty"`
	Since     string `json:"since,omitempty"`
	Until     string `json:"until,omitempty"`
}

// FirstTriggerLogEntry is the first LogEntry
type FirstTriggerLogEntry struct {
	CommonLogEntryField
	Incident APIObject `json:"incident,omitempty"`
}

// Incident is a normalized, de-duplicated event generated by a PagerDuty integration.
type Incident struct {
	APIObject
	IncidentNumber       uint                 `json:"incident_number,omitempty"`
	Title                string               `json:"title,omitempty"`
	Description          string               `json:"description,omitempty"`
	CreatedAt            string               `json:"created_at,omitempty"`
	PendingActions       []PendingAction      `json:"pending_actions,omitempty"`
	IncidentKey          string               `json:"incident_key,omitempty"`
	Service              APIObject            `json:"service,omitempty"`
	Assignments          []Assignment         `json:"assignments,omitempty"`
	Acknowledgements     []Acknowledgement    `json:"acknowledgements,omitempty"`
	LastStatusChangeAt   string               `json:"last_status_change_at,omitempty"`
	LastStatusChangeBy   APIObject            `json:"last_status_change_by,omitempty"`
	FirstTriggerLogEntry FirstTriggerLogEntry `json:"first_trigger_log_entry,omitempty"`
	EscalationPolicy     APIObject            `json:"escalation_policy,omitempty"`
	Teams                []APIObject          `json:"teams,omitempty"`
	Priority             *Priority            `json:"priority,omitempty"`
	Urgency              string               `json:"urgency,omitempty"`
	Status               string               `json:"status,omitempty"`
	ResolveReason        ResolveReason        `json:"resolve_reason,omitempty"`
	AlertCounts          AlertCounts          `json:"alert_counts,omitempty"`
	Body                 IncidentBody         `json:"body,omitempty"`
	IsMergeable          bool                 `json:"is_mergeable,omitempty"`
	ConferenceBridge     *ConferenceBridge    `json:"conference_bridge,omitempty"`
	AssignedVia          string               `json:"assigned_via,omitempty"`
	Occurrence           *Occurrence          `json:"occurrence,omitempty"`
	IncidentResponders   []IncidentResponders `json:"incidents_responders,omitempty"`
	ResponderRequests    []ResponderRequest   `json:"responder_requests,omitempty"`
	ResolvedAt           string               `json:"resolved_at,omitempty"`
	UpdatedAt            string               `json:"updated_at,omitempty"`
}

// ListIncidentsResponse is the response structure when calling the ListIncident API endpoint.
type ListIncidentsResponse struct {
	APIListObject
	Incidents []Incident `json:"incidents,omitempty"`
}

// ListIncidentsOptions is the structure used when passing parameters to the ListIncident API endpoint.
type ListIncidentsOptions struct {
	// Limit is the pagination parameter that limits the number of results per
	// page. PagerDuty defaults this value to 25 if omitted, and sets an upper
	// bound of 100.
	Limit uint `url:"limit,omitempty"`

	// Offset is the pagination parameter that specifies the offset at which to
	// start pagination results. When trying to request the next page of
	// results, the new Offset value should be currentOffset + Limit.
	Offset uint `url:"offset,omitempty"`

	// Total is the pagination parameter to request that the API return the
	// total count of items in the response. If this field is omitted or set to
	// false, the total number of results will not be sent back from the PagerDuty API.
	//
	// Setting this to true will slow down the API response times, and so it's
	// recommended to omit it unless you've a specific reason for wanting the
	// total count of items in the collection.
	Total bool `url:"total,omitempty"`

	Since       string   `url:"since,omitempty"`
	Until       string   `url:"until,omitempty"`
	DateRange   string   `url:"date_range,omitempty"`
	Statuses    []string `url:"statuses,omitempty,brackets"`
	IncidentKey string   `url:"incident_key,omitempty"`
	ServiceIDs  []string `url:"service_ids,omitempty,brackets"`
	TeamIDs     []string `url:"team_ids,omitempty,brackets"`
	UserIDs     []string `url:"user_ids,omitempty,brackets"`
	Urgencies   []string `url:"urgencies,omitempty,brackets"`
	TimeZone    string   `url:"time_zone,omitempty"`
	SortBy      string   `url:"sort_by,omitempty"`
	Includes    []string `url:"include,omitempty,brackets"`
}

// ConferenceBridge is a struct for the conference_bridge object on an incident
type ConferenceBridge struct {
	// ConferenceNumber is the phone number of the conference call for the
	// conference bridge. Phone numbers should be formatted like
	// +1 415-555-1212,,,,1234#, where a comma (,) represents a one-second
	// wait and pound (#) completes access code input.
	ConferenceNumber string `json:"conference_number,omitempty"`

	// ConferenceURL is the URL for the conference bridge. This could be a link
	// to a video conference or Slack channel.
	ConferenceURL string `json:"conference_url,omitempty"`
}

// ListIncidents lists existing incidents.
//
// Deprecated: Use ListIncidentsWithContext instead.
func (c *Client) ListIncidents(o ListIncidentsOptions) (*ListIncidentsResponse, error) {
	return c.ListIncidentsWithContext(context.Background(), o)
}

// ListIncidentsWithContext lists existing incidents.
func (c *Client) ListIncidentsWithContext(ctx context.Context, o ListIncidentsOptions) (*ListIncidentsResponse, error) {
	v, err := query.Values(o)
	if err != nil {
		return nil, err
	}

	resp, err := c.get(ctx, "/incidents?"+v.Encode(), nil)
	if err != nil {
		return nil, err
	}

	var result ListIncidentsResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, nil
}

// createIncidentResponse is returned from the API when creating a response.
type createIncidentResponse struct {
	Incident Incident `json:"incident"`
}

// CreateIncidentOptions is the structure used when POSTing to the CreateIncident API endpoint.
type CreateIncidentOptions struct {
	// Type is the type of API object this is.
	//
	// Deprecated: Because the Type field can only have the value of "incident"
	// when creating an incident, the CreateIncident* methods always set this
	// value to "incident". Any other value will be overwritten. This will be
	// removed in v2.0.0.
	Type             string            `json:"type"`
	Title            string            `json:"title"`
	Service          *APIReference     `json:"service"`
	Priority         *APIReference     `json:"priority"`
	Urgency          string            `json:"urgency,omitempty"`
	IncidentKey      string            `json:"incident_key,omitempty"`
	Body             *APIDetails       `json:"body,omitempty"`
	EscalationPolicy *APIReference     `json:"escalation_policy,omitempty"`
	Assignments      []Assignee        `json:"assignments,omitempty"`
	ConferenceBridge *ConferenceBridge `json:"conference_bridge,omitempty"`
}

// ManageIncidentsOptions is the structure used when PUTing updates to incidents to the ManageIncidents func
type ManageIncidentsOptions struct {
	ID string `json:"id"`

	// Type is the type of API object this is.
	//
	// Deprecated: Because the Type field can only have the value of "incident"
	// or "incident_reference" when managing an incident, the CreateIncident*
	// methods always set this value to "incident" because this struct is not an
	// incident_reference. Any other value will be overwritten. This will be
	// removed in v2.0.0.
	Type             string            `json:"type"`
	Status           string            `json:"status,omitempty"`
	Title            string            `json:"title,omitempty"`
	Priority         *APIReference     `json:"priority,omitempty"`
	Assignments      []Assignee        `json:"assignments,omitempty"`
	EscalationLevel  uint              `json:"escalation_level,omitempty"`
	EscalationPolicy *APIReference     `json:"escalation_policy,omitempty"`
	Resolution       string            `json:"resolution,omitempty"`
	ConferenceBridge *ConferenceBridge `json:"conference_bridge,omitempty"`
}

// MergeIncidentsOptions is the structure used when merging incidents with MergeIncidents func
type MergeIncidentsOptions struct {
	ID   string `json:"id"`
	Type string `json:"type"`
}

// CreateIncident creates an incident synchronously without a corresponding
// event from a monitoring service.
//
// Deprecated: Use CreateIncidentWithContext instead.
func (c *Client) CreateIncident(from string, o *CreateIncidentOptions) (*Incident, error) {
	return c.CreateIncidentWithContext(context.Background(), from, o)
}

// CreateIncidentWithContext creates an incident synchronously without a
// corresponding event from a monitoring service.
func (c *Client) CreateIncidentWithContext(ctx context.Context, from string, o *CreateIncidentOptions) (*Incident, error) {
	h := map[string]string{
		"From": from,
	}

	// see: https://github.com/PagerDuty/go-pagerduty/issues/390
	o.Type = "incident"

	d := map[string]*CreateIncidentOptions{
		"incident": o,
	}

	resp, err := c.post(ctx, "/incidents", d, h)
	if err != nil {
		return nil, err
	}

	var ii createIncidentResponse
	if err = c.decodeJSON(resp, &ii); err != nil {
		return nil, err
	}

	return &ii.Incident, nil
}

// ManageIncidents acknowledges, resolves, escalates, or reassigns one or more
// incidents.
//
// Deprecated: Use ManageIncidentsWithContext instead.
func (c *Client) ManageIncidents(from string, incidents []ManageIncidentsOptions) (*ListIncidentsResponse, error) {
	return c.ManageIncidentsWithContext(context.Background(), from, incidents)
}

// ManageIncidentsWithContext acknowledges, resolves, escalates, or reassigns
// one or more incidents.
func (c *Client) ManageIncidentsWithContext(ctx context.Context, from string, incidents []ManageIncidentsOptions) (*ListIncidentsResponse, error) {
	// see: https://github.com/PagerDuty/go-pagerduty/issues/390
	for i := range incidents {
		incidents[i].Type = "incident"
	}

	d := map[string][]ManageIncidentsOptions{
		"incidents": incidents,
	}

	h := map[string]string{
		"From": from,
	}

	resp, err := c.put(ctx, "/incidents", d, h)
	if err != nil {
		return nil, err
	}

	var result ListIncidentsResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, nil
}

// MergeIncidents merges a list of source incidents into a specified incident.
//
// Deprecated: Use MergeIncidentsWithContext instead.
func (c *Client) MergeIncidents(from string, id string, sourceIncidents []MergeIncidentsOptions) (*Incident, error) {
	return c.MergeIncidentsWithContext(context.Background(), from, id, sourceIncidents)
}

// MergeIncidentsWithContext merges a list of source incidents into a specified incident.
func (c *Client) MergeIncidentsWithContext(ctx context.Context, from, id string, sourceIncidents []MergeIncidentsOptions) (*Incident, error) {
	d := map[string][]MergeIncidentsOptions{
		"source_incidents": sourceIncidents,
	}

	h := map[string]string{
		"From": from,
	}

	resp, err := c.put(ctx, "/incidents/"+id+"/merge", d, h)
	if err != nil {
		return nil, err
	}

	var result createIncidentResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result.Incident, nil
}

// GetIncident shows detailed information about an incident.
//
// Deprecated: Use GetIncidentWithContext instead.
func (c *Client) GetIncident(id string) (*Incident, error) {
	return c.GetIncidentWithContext(context.Background(), id)
}

// GetIncidentWithContext shows detailed information about an incident.
func (c *Client) GetIncidentWithContext(ctx context.Context, id string) (*Incident, error) {
	resp, err := c.get(ctx, "/incidents/"+id, nil)
	if err != nil {
		return nil, err
	}

	var result map[string]Incident
	if err := c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	i, ok := result["incident"]
	if !ok {
		return nil, fmt.Errorf("JSON response does not have incident field")
	}

	return &i, nil
}

// IncidentNote is a note for the specified incident.
type IncidentNote struct {
	ID        string    `json:"id,omitempty"`
	User      APIObject `json:"user,omitempty"`
	Content   string    `json:"content,omitempty"`
	CreatedAt string    `json:"created_at,omitempty"`
}

// CreateIncidentNoteResponse is returned from the API as a response to creating an incident note.
type CreateIncidentNoteResponse struct {
	IncidentNote IncidentNote `json:"note"`
}

// ListIncidentNotes lists existing notes for the specified incident.
//
// Deprecated: Use ListIncidentNotesWithContext instead.
func (c *Client) ListIncidentNotes(id string) ([]IncidentNote, error) {
	return c.ListIncidentNotesWithContext(context.Background(), id)
}

// ListIncidentNotesWithContext lists existing notes for the specified incident.
func (c *Client) ListIncidentNotesWithContext(ctx context.Context, id string) ([]IncidentNote, error) {
	resp, err := c.get(ctx, "/incidents/"+id+"/notes", nil)
	if err != nil {
		return nil, err
	}

	var result map[string][]IncidentNote
	if err := c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	notes, ok := result["notes"]
	if !ok {
		return nil, fmt.Errorf("JSON response does not have notes field")
	}

	return notes, nil
}

// IncidentAlert is a alert for the specified incident.
type IncidentAlert struct {
	APIObject
	CreatedAt   string                 `json:"created_at,omitempty"`
	Status      string                 `json:"status,omitempty"`
	AlertKey    string                 `json:"alert_key,omitempty"`
	Service     APIObject              `json:"service,omitempty"`
	Body        map[string]interface{} `json:"body,omitempty"`
	Incident    APIReference           `json:"incident,omitempty"`
	Suppressed  bool                   `json:"suppressed,omitempty"`
	Severity    string                 `json:"severity,omitempty"`
	Integration APIObject              `json:"integration,omitempty"`
}

// IncidentAlertResponse is the response of a sincle incident alert
type IncidentAlertResponse struct {
	IncidentAlert *IncidentAlert `json:"alert,omitempty"`
}

// IncidentAlertList is the generic structure of a list of alerts
type IncidentAlertList struct {
	Alerts []IncidentAlert `json:"alerts,omitempty"`
}

// ListAlertsResponse is the response structure when calling the ListAlert API endpoint.
type ListAlertsResponse struct {
	APIListObject
	Alerts []IncidentAlert `json:"alerts,omitempty"`
}

// ListIncidentAlertsOptions is the structure used when passing parameters to the ListIncidentAlerts API endpoint.
type ListIncidentAlertsOptions struct {
	// Limit is the pagination parameter that limits the number of results per
	// page. PagerDuty defaults this value to 25 if omitted, and sets an upper
	// bound of 100.
	Limit uint `url:"limit,omitempty"`

	// Offset is the pagination parameter that specifies the offset at which to
	// start pagination results. When trying to request the next page of
	// results, the new Offset value should be currentOffset + Limit.
	Offset uint `url:"offset,omitempty"`

	// Total is the pagination parameter to request that the API return the
	// total count of items in the response. If this field is omitted or set to
	// false, the total number of results will not be sent back from the PagerDuty API.
	//
	// Setting this to true will slow down the API response times, and so it's
	// recommended to omit it unless you've a specific reason for wanting the
	// total count of items in the collection.
	Total bool `url:"total,omitempty"`

	Statuses []string `url:"statuses,omitempty,brackets"`
	SortBy   string   `url:"sort_by,omitempty"`
	Includes []string `url:"include,omitempty,brackets"`
}

// ListIncidentAlerts lists existing alerts for the specified incident. It's
// recommended to use ListIncidentAlertsWithContext instead.
func (c *Client) ListIncidentAlerts(id string) (*ListAlertsResponse, error) {
	return c.ListIncidentAlertsWithContext(context.Background(), id, ListIncidentAlertsOptions{})
}

// ListIncidentAlertsWithOpts lists existing alerts for the specified incident.
//
// Deprecated: Use ListIncidentAlertsWithContext instead.
func (c *Client) ListIncidentAlertsWithOpts(id string, o ListIncidentAlertsOptions) (*ListAlertsResponse, error) {
	return c.ListIncidentAlertsWithContext(context.Background(), id, o)
}

// ListIncidentAlertsWithContext lists existing alerts for the specified
// incident. If you don't want to filter any of the results, pass in an empty
// ListIncidentAlertOptions.
func (c *Client) ListIncidentAlertsWithContext(ctx context.Context, id string, o ListIncidentAlertsOptions) (*ListAlertsResponse, error) {
	v, err := query.Values(o)
	if err != nil {
		return nil, err
	}

	resp, err := c.get(ctx, "/incidents/"+id+"/alerts?"+v.Encode(), nil)
	if err != nil {
		return nil, err
	}

	var result ListAlertsResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, err
}

// CreateIncidentNoteWithResponse creates a new note for the specified incident.
//
// Deprecated: Use CreateIncidentNoteWithContext instead.
func (c *Client) CreateIncidentNoteWithResponse(id string, note IncidentNote) (*IncidentNote, error) {
	return c.CreateIncidentNoteWithContext(context.Background(), id, note)
}

// CreateIncidentNoteWithContext creates a new note for the specified incident.
func (c *Client) CreateIncidentNoteWithContext(ctx context.Context, id string, note IncidentNote) (*IncidentNote, error) {
	d := map[string]IncidentNote{
		"note": note,
	}

	h := map[string]string{
		"From": note.User.Summary,
	}

	resp, err := c.post(ctx, "/incidents/"+id+"/notes", d, h)
	if err != nil {
		return nil, err
	}

	var result CreateIncidentNoteResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result.IncidentNote, nil
}

// CreateIncidentNote creates a new note for the specified incident.
//
// Deprecated: Use CreateIncidentNoteWithContext instead.
func (c *Client) CreateIncidentNote(id string, note IncidentNote) error {
	data := make(map[string]IncidentNote)
	headers := make(map[string]string)
	headers["From"] = note.User.Summary
	data["note"] = note
	_, err := c.post(context.Background(), "/incidents/"+id+"/notes", data, headers)
	return err
}

// SnoozeIncidentWithResponse sets an incident to not alert for a specified
// period of time.
//
// Deprecated: Use SnoozeIncidentWithContext instead.
func (c *Client) SnoozeIncidentWithResponse(id string, duration uint) (*Incident, error) {
	return c.SnoozeIncidentWithContext(context.Background(), id, duration)
}

// SnoozeIncidentWithContext sets an incident to not alert for a specified period of time.
func (c *Client) SnoozeIncidentWithContext(ctx context.Context, id string, duration uint) (*Incident, error) {
	d := map[string]uint{
		"duration": duration,
	}

	resp, err := c.post(ctx, "/incidents/"+id+"/snooze", d, nil)
	if err != nil {
		return nil, err
	}

	var result createIncidentResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result.Incident, nil
}

// SnoozeIncident sets an incident to not alert for a specified period of time.
//
// Deprecated: Use SnoozeIncidentWithContext instead.
func (c *Client) SnoozeIncident(id string, duration uint) error {
	data := make(map[string]uint)
	data["duration"] = duration
	_, err := c.post(context.Background(), "/incidents/"+id+"/snooze", data, nil)
	return err
}

// ListIncidentLogEntriesResponse is the response structure when calling the ListIncidentLogEntries API endpoint.
type ListIncidentLogEntriesResponse struct {
	APIListObject
	LogEntries []LogEntry `json:"log_entries,omitempty"`
}

// ListIncidentLogEntriesOptions is the structure used when passing parameters to the ListIncidentLogEntries API endpoint.
type ListIncidentLogEntriesOptions struct {
	// Limit is the pagination parameter that limits the number of results per
	// page. PagerDuty defaults this value to 25 if omitted, and sets an upper
	// bound of 100.
	Limit uint `url:"limit,omitempty"`

	// Offset is the pagination parameter that specifies the offset at which to
	// start pagination results. When trying to request the next page of
	// results, the new Offset value should be currentOffset + Limit.
	Offset uint `url:"offset,omitempty"`

	// Total is the pagination parameter to request that the API return the
	// total count of items in the response. If this field is omitted or set to
	// false, the total number of results will not be sent back from the PagerDuty API.
	//
	// Setting this to true will slow down the API response times, and so it's
	// recommended to omit it unless you've a specific reason for wanting the
	// total count of items in the collection.
	Total bool `url:"total,omitempty"`

	Includes   []string `url:"include,omitempty,brackets"`
	IsOverview bool     `url:"is_overview,omitempty"`
	TimeZone   string   `url:"time_zone,omitempty"`
	Since      string   `url:"since,omitempty"`
	Until      string   `url:"until,omitempty"`
}

// ListIncidentLogEntries lists existing log entries for the specified incident.
//
// Deprecated: Use ListIncidentLogEntriesWithContext instead.
func (c *Client) ListIncidentLogEntries(id string, o ListIncidentLogEntriesOptions) (*ListIncidentLogEntriesResponse, error) {
	return c.ListIncidentLogEntriesWithContext(context.Background(), id, o)
}

// ListIncidentLogEntriesWithContext lists existing log entries for the
// specified incident.
func (c *Client) ListIncidentLogEntriesWithContext(ctx context.Context, id string, o ListIncidentLogEntriesOptions) (*ListIncidentLogEntriesResponse, error) {
	v, err := query.Values(o)
	if err != nil {
		return nil, err
	}

	resp, err := c.get(ctx, "/incidents/"+id+"/log_entries?"+v.Encode(), nil)
	if err != nil {
		return nil, err
	}

	var result ListIncidentLogEntriesResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, nil
}

// IncidentResponders contains details about responders to an incident.
type IncidentResponders struct {
	State       string    `json:"state"`
	User        APIObject `json:"user"`
	Incident    APIObject `json:"incident"`
	UpdatedAt   string    `json:"updated_at"`
	Message     string    `json:"message"`
	Requester   APIObject `json:"requester"`
	RequestedAt string    `json:"requested_at"`
}

// ResponderRequestResponse is the response from the API when requesting someone
// respond to an incident.
type ResponderRequestResponse struct {
	ResponderRequest ResponderRequest `json:"responder_request"`
}

// ResponderRequestTarget specifies an individual target for the responder request.
type ResponderRequestTarget struct {
	APIObject
	Responders []IncidentResponders `json:"incidents_responders,omitempty"`
}

// ResponderRequestTargetWrapper is a wrapper for a ResponderRequestTarget.
type ResponderRequestTargetWrapper struct {
	Target ResponderRequestTarget `json:"responder_request_target"`
}

// ResponderRequestOptions defines the input options for the Create Responder function.
type ResponderRequestOptions struct {
	From        string                          `json:"-"`
	Message     string                          `json:"message"`
	RequesterID string                          `json:"requester_id"`
	Targets     []ResponderRequestTargetWrapper `json:"responder_request_targets"`
}

// ResponderRequest contains the API structure for an incident responder request.
type ResponderRequest struct {
	Incident    Incident                        `json:"incident"`
	Requester   User                            `json:"requester,omitempty"`
	RequestedAt string                          `json:"request_at,omitempty"`
	Message     string                          `json:"message,omitempty"`
	Targets     []ResponderRequestTargetWrapper `json:"responder_request_targets"`
}

// ResponderRequest will submit a request to have a responder join an incident.
//
// Deprecated: Use ResponderRequestWithContext instead.
func (c *Client) ResponderRequest(id string, o ResponderRequestOptions) (*ResponderRequestResponse, error) {
	return c.ResponderRequestWithContext(context.Background(), id, o)
}

// ResponderRequestWithContext will submit a request to have a responder join an incident.
func (c *Client) ResponderRequestWithContext(ctx context.Context, id string, o ResponderRequestOptions) (*ResponderRequestResponse, error) {
	h := map[string]string{
		"From": o.From,
	}

	resp, err := c.post(ctx, "/incidents/"+id+"/responder_requests", o, h)
	if err != nil {
		return nil, err
	}

	var result ResponderRequestResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, nil
}

// GetIncidentAlert gets the alert that triggered the incident.
//
// Deprecated: Use GetIncidentAlertWithContext instead.
func (c *Client) GetIncidentAlert(incidentID, alertID string) (*IncidentAlertResponse, error) {
	return c.GetIncidentAlertWithContext(context.Background(), incidentID, alertID)
}

// GetIncidentAlertWithContext gets the alert that triggered the incident.
func (c *Client) GetIncidentAlertWithContext(ctx context.Context, incidentID, alertID string) (*IncidentAlertResponse, error) {
	resp, err := c.get(ctx, "/incidents/"+incidentID+"/alerts/"+alertID, nil)
	if err != nil {
		return nil, err
	}

	var result IncidentAlertResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, nil
}

// ManageIncidentAlerts allows you to manage the alerts of an incident.
func (c *Client) ManageIncidentAlerts(ctx context.Context, incidentID, from string, alerts *IncidentAlertList) (*ListAlertsResponse, error) {
	h := map[string]string{
		"From": from,
	}

	resp, err := c.put(ctx, "/incidents/"+incidentID+"/alerts/", alerts, h)
	if err != nil {
		return nil, err
	}

	var result ListAlertsResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, nil
}

// IncidentStatusUpdate is a status update for the specified incident.
type IncidentStatusUpdate struct {
	ID        string    `json:"id"`
	Message   string    `json:"message"`
	CreatedAt string    `json:"created_at"`
	Sender    APIObject `json:"sender"`
}

// CreateIncidentStatusUpdate creates a new status update for the specified incident.
func (c *Client) CreateIncidentStatusUpdate(ctx context.Context, id string, from string, message string) (IncidentStatusUpdate, error) {
	d := map[string]string{
		"message": message,
	}

	h := map[string]string{
		"From": from,
	}

	resp, err := c.post(ctx, "/incidents/"+id+"/status_updates", d, h)
	if err != nil {
		return IncidentStatusUpdate{}, err
	}

	var result struct {
		IncidentStatusUpdate IncidentStatusUpdate `json:"status_update"`
	}
	if err = c.decodeJSON(resp, &result); err != nil {
		return IncidentStatusUpdate{}, err
	}

	return result.IncidentStatusUpdate, nil
}

// IncidentNotificationSubscriber is a Notification Subscriber on a Incident.
type IncidentNotificationSubscriber struct {
	SubscriberID   string `json:"subscriber_id"`
	SubscriberType string `json:"subscriber_type"`
}

// IncidentNotificationSubscriptionWithContext contains extra context returned from the API.
type IncidentNotificationSubscriptionWithContext struct {
	IncidentNotificationSubscriber
	HasIndirectSubscription bool                                `json:"has_indirect_subscription,omitempty"`
	SubscribedVia           []IncidentNotificationSubscriberVia `json:"subscribed_via,omitempty"`
	SubscribableID          string                              `json:"subscribable_id,omitempty"`
	SubscribableType        string                              `json:"subscribable_type,omitempty"`
	AccountID               string                              `json:"account_id,omitempty"`
	Result                  string                              `json:"result,omitempty"`
}
type IncidentNotificationSubscriberVia struct {
	ID   string `json:"id"`
	Type string `json:"type"`
}

// ListIncidentNotificationSubscribersResponse is the response structure when calling the ListNotificationSubscribers API endpoint.
type ListIncidentNotificationSubscribersResponse struct {
	APIListObject
	Subscribers []IncidentNotificationSubscriptionWithContext `json:"subscribers,omitempty"`
	AccountID   string                                        `json:"account_id,omitempty"`
}

// AddIncidentNotificationSubscribersResponse is the response structure when calling the AddNotificationSubscribers API endpoint.
type AddIncidentNotificationSubscribersResponse struct {
	Subscriptions []IncidentNotificationSubscriptionWithContext `json:"subscriptions,omitempty"`
}

// RemoveIncidentNotificationSubscribersResponse is the response structure when calling the RemoveNotificationSubscriber API endpoint.
type RemoveIncidentNotificationSubscribersResponse struct {
	DeleteCount       uint `json:"deleted_count"`
	UnauthorizedCount uint `json:"unauthorized_count"`
	NonExistentCount  uint `json:"non_existent_count"`
}

// ListIncidentNotificationSubscribersWithContext lists notification subscribers for the specified incident.
func (c *Client) ListIncidentNotificationSubscribersWithContext(ctx context.Context, id string) (*ListIncidentNotificationSubscribersResponse, error) {
	resp, err := c.get(ctx, "/incidents/"+id+"/status_updates/subscribers", nil)
	if err != nil {
		return nil, err
	}

	var result ListIncidentNotificationSubscribersResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, nil
}

// AddIncidentNotificationSubscribersWithContext adds notification subscribers for the specified incident.
func (c *Client) AddIncidentNotificationSubscribersWithContext(ctx context.Context, id string, subscribers []IncidentNotificationSubscriber) (*AddIncidentNotificationSubscribersResponse, error) {
	d := map[string][]IncidentNotificationSubscriber{
		"subscribers": subscribers,
	}

	resp, err := c.post(ctx, "/incidents/"+id+"/status_updates/subscribers", d, nil)
	if err != nil {
		return nil, err
	}

	var result AddIncidentNotificationSubscribersResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, nil
}

// RemoveIncidentNotificationSubscribersWithContext removes notification subscribers for the specified incident.
func (c *Client) RemoveIncidentNotificationSubscribersWithContext(ctx context.Context, id string, subscribers []IncidentNotificationSubscriber) (*RemoveIncidentNotificationSubscribersResponse, error) {
	d := map[string][]IncidentNotificationSubscriber{
		"subscribers": subscribers,
	}

	resp, err := c.post(ctx, "/incidents/"+id+"/status_updates/unsubscribe", d, nil)
	if err != nil {
		return nil, err
	}

	var result RemoveIncidentNotificationSubscribersResponse
	if err = c.decodeJSON(resp, &result); err != nil {
		return nil, err
	}

	return &result, nil
}
