import React, { SVGAttributes, RefAttributes, CSSProperties } from 'react';
import { FlipProp, PullProp, RotateProp, Transform, FaSymbol, IconProp, SizeProp } from '@fortawesome/fontawesome-svg-core';

interface AnimationProps {
    /**
     * Makes the icon spin 360deg clockwise continuously.
     * @see {@link https://docs.fontawesome.com/web/style/animate#spin}
     */
    spin?: boolean | undefined;
    /**
     * Makes the icon spin 360deg clockwise in 8 incremental steps.
     * @see {@link https://docs.fontawesome.com/web/style/animate#spin}
     */
    spinPulse?: boolean | undefined;
    /**
     * When used in conjunction with {@link spin} or {@link spinPulse}, makes the icon spin in reverse.
     * @see {@link https://docs.fontawesome.com/web/style/animate#spin-utilities}
     */
    spinReverse?: boolean | undefined;
    /**
     * @deprecated
     * Will be removed in a future version.
     * Please use {@link spinPulse} instead.
     *
     * Makes the icon spin 360deg clockwise in 8 incremental steps.
     * @see {@link https://docs.fontawesome.com/web/style/animate#spin-utilities}
     */
    pulse?: boolean | undefined;
    /**
     * Makes the icon scale in and out continuously.
     * @see {@link https://docs.fontawesome.com/web/style/animate#beat}
     */
    beat?: boolean | undefined;
    /**
     * Makes the icon fade in and out continuously.
     * @see {@link https://docs.fontawesome.com/web/style/animate#fade}
     */
    fade?: boolean | undefined;
    /**
     * Applies both scaling and fading animations from {@link beat} and {@link fade}.
     * @see {@link https://docs.fontawesome.com/web/style/animate#beat-fade}
     */
    beatFade?: boolean | undefined;
    /**
     * Makes the icon bounce up and down.
     * @see {@link https://docs.fontawesome.com/web/style/animate#bounce}
     */
    bounce?: boolean | undefined;
    /**
     * Makes the icon shake.
     * @see {@link https://docs.fontawesome.com/web/style/animate#shake}
     */
    shake?: boolean | undefined;
}

/**
 * Adds support for FA's CSS variables to the React `style` prop.
 *
 * TODO: Move this to `fontawesome-common-types`
 * @see https://github.com/FortAwesome/react-fontawesome/pull/581#discussion_r2291185167
 */
interface CSSVariables extends FontFamilyVariables {
    '--fa-family'?: string | undefined;
    '--fa-style'?: string | undefined;
    '--fa-display'?: string | undefined;
    '--fa-inverse'?: string | undefined;
    '--fa-width'?: string | undefined;
    '--fa-li-margin'?: string | undefined;
    '--fa-li-width'?: string | undefined;
    '--fa-rotate-angle'?: string | undefined;
    '--fa-animation-delay'?: string | undefined;
    '--fa-animation-direction'?: string | undefined;
    '--fa-animation-duration'?: string | undefined;
    '--fa-animation-iteration-count'?: string | undefined;
    '--fa-animation-timing'?: string | undefined;
    '--fa-beat-scale'?: string | undefined;
    '--fa-fade-opacity'?: string | undefined;
    '--fa-beat-fade-opacity'?: string | undefined;
    '--fa-beat-fade-scale'?: string | undefined;
    '--fa-flip-x'?: string | undefined;
    '--fa-flip-y'?: string | undefined;
    '--fa-flip-z'?: string | undefined;
    '--fa-flip-angle'?: string | undefined;
    '--fa-border-color'?: string | undefined;
    '--fa-border-padding'?: string | undefined;
    '--fa-border-radius'?: string | undefined;
    '--fa-border-style'?: string | undefined;
    '--fa-border-width'?: string | undefined;
    '--fa-pull-margin'?: string | undefined;
    '--fa-stack-z-index'?: string | undefined;
    '--fa-primary-color'?: string | undefined;
    '--fa-primary-opacity'?: string | undefined;
    '--fa-secondary-color'?: string | undefined;
    '--fa-secondary-opacity'?: string | undefined;
}
type FontFamilyVariables = {
    [K in `--fa-font-${string}`]?: string | undefined;
};

interface TransformProps {
    /**
     * @deprecated Simply wrap the icon instead, no need to pass this property.
     *
     * @example
     * ```tsx
     * <ul className='fa-ul'>
     *   <li>
     *     <span className='fa-li'>
     *       <FontAwesomeIcon icon={['fas', 'check']} />
     *     </span>
     *     List item with icon
     *   </li>
     * </ul>
     * ```
     *
     * @see {@link https://docs.fontawesome.com/web/style/lists}
     */
    listItem?: boolean | undefined;
    /** Flip the icon horizontally, vertically or both. */
    flip?: FlipProp | boolean | undefined;
    /** Wrap text around the icon by pulling it left or right. */
    pull?: PullProp | undefined;
    /**
     * The rotation property is used to rotate the icon by 90, 180, or 270 degrees.
     *
     * @see {@link https://docs.fontawesome.com/web/use-with/react/style#rotation}
     */
    rotation?: RotateProp | undefined;
    /**
     * Custom rotation is used to rotate the icon by a specific number of degrees,
     * rather than the standard 90, 180, or 270 degrees available in the `rotation` property.
     *
     * To use this feature, set `rotateBy` to `true` and provide a CSS variable `--fa-rotate-angle`
     * with the desired rotation angle in degrees.
     *
     * @example
     * ```tsx
     * <FontAwesomeIcon
     *   icon="fa-solid fa-coffee"
     *   rotateBy
     *   style={{ '--fa-rotate-angle': '329deg' }}
     * />
     * ```
     *
     * @see {@link https://docs.fontawesome.com/web/use-with/react/style#custom-rotation}
     * @since 7.0.0
     */
    rotateBy?: boolean | undefined;
    /**
     * Apply a custom transform to the icon either using the built-in {@link Transform} config,
     * or a CSS transform string.
     * @see {@link https://docs.fontawesome.com/web/use-with/react/style#power-transforms}
     */
    transform?: string | Transform | undefined;
    /**
     * The icon should render as an SVG symbol rather than a regular element.
     * @see {@link https://docs.fontawesome.com/web/use-with/react/style#use-svg-symbols}
     */
    symbol?: FaSymbol | undefined;
}

interface FontAwesomeIconProps extends AnimationProps, TransformProps, Omit<SVGAttributes<SVGSVGElement>, 'children' | 'mask' | 'transform'>, RefAttributes<SVGSVGElement> {
    /**
     * The icon to render.
     * @see {@link https://docs.fontawesome.com/web/use-with/react/add-icons}
     */
    icon: IconProp;
    /** The size of the icon from a predefined set of sizes. */
    size?: SizeProp | undefined;
    /**
     * Grab the Mask utility when you want to layer two icons but have the inner icon cut out from the icon below so the parent element’s background shows through.
     * @see {@link https://docs.fontawesome.com/web/use-with/react/style#mask}
     */
    mask?: IconProp | undefined;
    /** Accessibility ID for the mask element */
    maskId?: string | undefined;
    /** Any additional CSS classes to apply to the icon */
    className?: string | undefined;
    /** The color of the icon. Can be any valid CSS color value */
    color?: string | undefined;
    /**
     * Applies a border to the icon
     * @see {@link https://docs.fontawesome.com/web/use-with/react/style#bordered-icons}
     */
    border?: boolean | undefined;
    /**
     * @deprecated
     * @since 7.0.0
     *
     * Starting in FontAwesome 7.0.0, all icons are fixed width by default.
     * This property will be removed in a future version.
     *
     * If you want to remove the fixed width to replicate the behavior of
     * previous versions, you can set the new {@link widthAuto} property to `true`.
     *
     * @see {@link FontAwesomeIconProps.widthAuto}
     */
    fixedWidth?: boolean | undefined;
    /**
     * Invert the icon color.
     * @see {@link https://docs.fontawesome.com/web/use-with/react/style#invert-the-icon-color-to-white}
     */
    inverse?: boolean | undefined;
    /** Any custom styles or CSS variable overrides for the icon element. */
    style?: (CSSProperties & CSSVariables) | undefined;
    tabIndex?: number | undefined;
    /** An accessibility title for the icon to be read by screen-readers or assistive technologies. */
    title?: string | undefined;
    /** A unique ID for the accessibility title element. */
    titleId?: string | undefined;
    /**
     * When using Duotone icons, this property will swap the opacity of the two colors.
     * The first color will be rendered with the opacity of the second color, and vice versa
     *
     * @see {@link https://docs.fontawesome.com/web/use-with/react/style#duotone-icons}
     */
    swapOpacity?: boolean | undefined;
    /**
     * When set to `true`, the icon will automatically adjust its width to
     * only the interior symbol and not the entire Icon Canvas.
     *
     * @see {@link https://docs.fontawesome.com/web/style/icon-canvas}
     * @since 7.0.0
     */
    widthAuto?: boolean | undefined;
}

/**
 * FontAwesomeIcon component.
 */
declare const FontAwesomeIcon: React.ForwardRefExoticComponent<Omit<FontAwesomeIconProps, "ref"> & React.RefAttributes<SVGSVGElement>>;

export { type AnimationProps, FontAwesomeIcon, type FontAwesomeIconProps, type TransformProps };
