/*
 * Copyright (c) 2018, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 *
 */

#include "gc/shared/c1/cardTableBarrierSetC1.hpp"
#include "gc/shared/cardTable.hpp"
#include "gc/shared/cardTableBarrierSet.hpp"
#include "gc/shared/gc_globals.hpp"
#include "utilities/macros.hpp"

#ifdef ASSERT
#define __ gen->lir(__FILE__, __LINE__)->
#else
#define __ gen->lir()->
#endif

void CardTableBarrierSetC1::store_at_resolved(LIRAccess& access, LIR_Opr value) {
  DecoratorSet decorators = access.decorators();
  bool is_array = (decorators & IS_ARRAY) != 0;
  bool on_anonymous = (decorators & ON_UNKNOWN_OOP_REF) != 0;

  if (access.is_oop()) {
    pre_barrier(access, access.resolved_addr(),
                LIR_OprFact::illegalOpr /* pre_val */, access.patch_emit_info());
  }

  BarrierSetC1::store_at_resolved(access, value);

  if (access.is_oop()) {
    bool precise = is_array || on_anonymous;
    LIR_Opr post_addr = precise ? access.resolved_addr() : access.base().opr();
    post_barrier(access, post_addr, value);
  }
}

LIR_Opr CardTableBarrierSetC1::atomic_cmpxchg_at_resolved(LIRAccess& access, LIRItem& cmp_value, LIRItem& new_value) {
  if (access.is_oop()) {
    pre_barrier(access, access.resolved_addr(),
                LIR_OprFact::illegalOpr /* pre_val */, nullptr);
  }

  LIR_Opr result = BarrierSetC1::atomic_cmpxchg_at_resolved(access, cmp_value, new_value);

  if (access.is_oop()) {
    post_barrier(access, access.resolved_addr(), new_value.result());
  }

  return result;
}

LIR_Opr CardTableBarrierSetC1::atomic_xchg_at_resolved(LIRAccess& access, LIRItem& value) {
  if (access.is_oop()) {
    pre_barrier(access, access.resolved_addr(),
                LIR_OprFact::illegalOpr /* pre_val */, nullptr);
  }

  LIR_Opr result = BarrierSetC1::atomic_xchg_at_resolved(access, value);

  if (access.is_oop()) {
    post_barrier(access, access.resolved_addr(), value.result());
  }

  return result;
}

// This overrides the default to resolve the address into a register,
// assuming it will be used by a write barrier anyway.
LIR_Opr CardTableBarrierSetC1::resolve_address(LIRAccess& access, bool resolve_in_register) {
  DecoratorSet decorators = access.decorators();
  bool needs_patching = (decorators & C1_NEEDS_PATCHING) != 0;
  bool is_write = (decorators & ACCESS_WRITE) != 0;
  bool is_array = (decorators & IS_ARRAY) != 0;
  bool on_anonymous = (decorators & ON_UNKNOWN_OOP_REF) != 0;
  bool precise = is_array || on_anonymous;
  resolve_in_register |= !needs_patching && is_write && access.is_oop() && precise;
  return BarrierSetC1::resolve_address(access, resolve_in_register);
}

void CardTableBarrierSetC1::post_barrier(LIRAccess& access, LIR_Opr addr, LIR_Opr new_val) {
  DecoratorSet decorators = access.decorators();
  LIRGenerator* gen = access.gen();
  bool in_heap = (decorators & IN_HEAP) != 0;
  if (!in_heap) {
    return;
  }

  CardTableBarrierSet* ctbs = barrier_set_cast<CardTableBarrierSet>(BarrierSet::barrier_set());
  LIR_Const* card_table_base = new LIR_Const(ctbs->card_table_base_const());

  if (addr->is_address()) {
    LIR_Address* address = addr->as_address_ptr();
    // ptr cannot be an object because we use this barrier for array card marks
    // and addr can point in the middle of an array.
    LIR_Opr ptr = gen->new_pointer_register();
    if (!address->index()->is_valid() && address->disp() == 0) {
      __ move(address->base(), ptr);
    } else {
      assert(address->disp() != max_jint, "lea doesn't support patched addresses!");
      __ leal(addr, ptr);
    }
    addr = ptr;
  }
  assert(addr->is_register(), "must be a register at this point");

#ifdef CARDTABLEBARRIERSET_POST_BARRIER_HELPER
  gen->CardTableBarrierSet_post_barrier_helper(addr, card_table_base);
#else
  LIR_Opr tmp = gen->new_pointer_register();
  if (two_operand_lir_form) {
    LIR_Opr addr_opr = LIR_OprFact::address(new LIR_Address(addr, addr->type()));
    __ leal(addr_opr, tmp);
    __ unsigned_shift_right(tmp, CardTable::card_shift(), tmp);
  } else {
    __ unsigned_shift_right(addr, CardTable::card_shift(), tmp);
  }

  LIR_Address* card_addr;
  if (gen->can_inline_as_constant(card_table_base)) {
    card_addr = new LIR_Address(tmp, card_table_base->as_jint(), T_BYTE);
  } else {
    card_addr = new LIR_Address(tmp, gen->load_constant(card_table_base), T_BYTE);
  }

  LIR_Opr dirty = LIR_OprFact::intConst(CardTable::dirty_card_val());
  if (UseCondCardMark) {
    LIR_Opr cur_value = gen->new_register(T_INT);
    __ move(card_addr, cur_value);

    LabelObj* L_already_dirty = new LabelObj();
    __ cmp(lir_cond_equal, cur_value, dirty);
    __ branch(lir_cond_equal, L_already_dirty->label());
    __ move(dirty, card_addr);
    __ branch_destination(L_already_dirty->label());
  } else {
    __ move(dirty, card_addr);
  }
#endif
}
