// Copyright 2019 The TCMalloc Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "absl/types/span.h"
#include "tcmalloc/common.h"
#include "tcmalloc/internal/config.h"
#include "tcmalloc/size_class_info.h"

GOOGLE_MALLOC_SECTION_BEGIN
namespace tcmalloc {
namespace tcmalloc_internal {

// Columns in the following tables:
// - bytes: size of the size class
// - pages: number of pages per span
// - batch: preferred number of objects for transfers between caches
// - cap: maximum cpu cache capacity
// - class: size class number
// - objs: number of objects per span
// - waste/fixed: fixed per-size-class overhead due to end-of-span fragmentation
//   and other factors. For instance, if we have a 96 byte size class, and use
//   a single 8KiB page, then we will hold 85 objects per span, and have 32
//   bytes left over. There is also a fixed component of 48 bytes of TCMalloc
//   metadata per span. Together, the fixed overhead would be wasted/allocated
//   = (32 + 48) / (8192 - 32) ~= 0.98%.
// - waste/sampling: overhead due to heap sampling
//   (rounding to page size, proxy object, metadata).
// - inc: increment from the previous size class. This caps the dynamic
//   overhead component based on mismatches between the number of bytes
//   requested and the number of bytes provided by the size class. Together
//   they sum to the total overhead; for instance if you asked for a 50-byte
//   allocation that rounds up to a 64-byte size class, the dynamic overhead
//   would be 28%, and if waste were 22% it would mean (on average) 25 bytes
//   of overhead for allocations of that size.

// clang-format off
#if defined(__cpp_aligned_new) && __STDCPP_DEFAULT_NEW_ALIGNMENT__ <= 8
#if TCMALLOC_PAGE_SHIFT == 13
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2025},  //  0  1024  0.58%    0.42%   0.00%
  {    16,    1,   32, 2025},  //  1   512  0.58%    0.42% 100.00%
  {    32,    1,   32, 2025},  //  2   256  0.58%    0.42% 100.00%
  {    64,    1,   32, 2025},  //  3   128  0.58%    0.42% 100.00%
  {    72,    1,   32, 1276},  //  4   113  1.26%    0.42%  12.50%
  {    80,    1,   32, 2025},  //  5   102  0.97%    0.42%  11.11%
  {    88,    1,   32, 1032},  //  6    93  0.68%    0.42%  10.00%
  {    96,    1,   32, 1207},  //  7    85  0.97%    0.42%   9.09%
  {   104,    1,   32,  490},  //  8    78  1.55%    0.42%   8.33%
  {   112,    1,   32,  804},  //  9    73  0.78%    0.42%   7.69%
  {   120,    1,   32,  504},  // 10    68  0.97%    0.42%   7.14%
  {   128,    1,   32,  957},  // 11    64  0.58%    0.42%   6.67%
  {   144,    1,   32,  873},  // 12    56  2.14%    0.42%  12.50%
  {   160,    1,   32,  721},  // 13    51  0.97%    0.42%  11.11%
  {   176,    1,   32,  378},  // 14    46  1.75%    0.42%  10.00%
  {   192,    1,   32,  492},  // 15    42  2.14%    0.42%   9.09%
  {   208,    1,   32,  326},  // 16    39  1.55%    0.42%   8.33%
  {   224,    1,   32,  283},  // 17    36  2.14%    0.42%   7.69%
  {   240,    1,   32,  266},  // 18    34  0.97%    0.42%   7.14%
  {   256,    1,   32,  613},  // 19    32  0.58%    0.42%   6.67%
  {   280,    1,   32,  320},  // 20    29  1.46%    0.42%   9.38%
  {   312,    1,   32,  347},  // 21    26  1.55%    0.42%  11.43%
  {   336,    1,   32,  360},  // 22    24  2.14%    0.42%   7.69%
  {   352,    1,   32,  188},  // 23    23  1.75%    0.42%   4.76%
  {   384,    1,   32,  244},  // 24    21  2.14%    0.42%   9.09%
  {   416,    2,   32,  235},  // 25    39  1.27%    0.42%   8.33%
  {   448,    1,   32,  245},  // 26    18  2.14%    0.42%   7.69%
  {   512,    1,   32,  477},  // 27    16  0.58%    0.42%  14.29%
  {   584,    1,   32,  255},  // 28    14  0.78%    0.42%  14.06%
  {   640,    4,   32,  212},  // 29    51  0.54%    0.42%   9.59%
  {   712,    2,   32,  190},  // 30    23  0.34%    0.42%  11.25%
  {   768,    3,   32,  182},  // 31    32  0.19%    0.42%   7.87%
  {   904,    1,   32,  206},  // 32     9  1.26%    0.42%  17.71%
  {  1024,    1,   32,  373},  // 33     8  0.58%    0.42%  13.27%
  {  1152,    1,   32,  192},  // 34     7  2.14%    0.43%  12.50%
  {  1280,    3,   32,  170},  // 35    19  1.23%    0.42%  11.11%
  {  1408,    4,   32,  160},  // 36    23  1.32%    0.43%  10.00%
  {  1536,    3,   32,  166},  // 37    16  0.19%    0.42%   9.09%
  {  1792,    2,   32,  162},  // 38     9  1.85%    0.43%  16.67%
  {  1920,    4,   32,  140},  // 39    17  0.54%    0.42%   7.14%
  {  2048,    1,   32,  187},  // 40     4  0.58%    0.42%   6.67%
  {  2176,    4,   30,  142},  // 41    15  0.54%    0.42%   6.25%
  {  2432,    3,   26,  150},  // 42    10  1.23%    0.43%  11.76%
  {  2688,    1,   24,  141},  // 43     3  2.14%    0.43%  10.53%
  {  3072,    3,   21,  145},  // 44     8  0.19%    0.42%  14.29%
  {  3328,    9,   19,  135},  // 45    22  0.76%    0.43%   8.33%
  {  3712,    5,   17,  142},  // 46    11  0.43%    0.42%  11.54%
  {  4096,    1,   16,  341},  // 47     2  0.58%    0.43%  10.34%
  {  4736,    7,   13,  139},  // 48    12  0.98%    0.43%  15.62%
  {  5376,    2,   12,  133},  // 49     3  1.85%    0.43%  13.51%
  {  6144,    3,   10,  139},  // 50     4  0.19%    0.42%  14.29%
  {  7168,    7,    9,  135},  // 51     8  0.08%    0.42%  16.67%
  {  8192,    1,    8,  207},  // 52     1  0.58%    0.03%  14.29%
  {  9344,    8,    7,  134},  // 53     7  0.27%    0.82%  14.06%
  { 10496,    9,    6,  131},  // 54     7  0.41%    0.82%  12.33%
  { 12288,    3,    5,  134},  // 55     2  0.19%    0.82%  17.07%
  { 13568,    5,    4,  129},  // 56     3  0.74%    0.82%  10.42%
  { 14720,    9,    4,  128},  // 57     5  0.24%    0.82%   8.49%
  { 16384,    2,    4,  141},  // 58     1  0.29%    0.03%  11.30%
  { 17408,   17,    3,  130},  // 59     8  0.03%    1.21%   6.25%
  { 19072,    7,    3,  128},  // 60     3  0.31%    1.21%   9.56%
  { 21760,    8,    3,  131},  // 61     3  0.46%    1.21%  14.09%
  { 24576,    3,    2,  129},  // 62     1  0.19%    0.03%  12.94%
  { 26624,   13,    2,  128},  // 63     4  0.05%    1.60%   8.33%
  { 28672,    7,    2,  130},  // 64     2  0.08%    1.60%   7.69%
  { 30720,   15,    2,  132},  // 65     4  0.04%    1.60%   7.14%
  { 32768,    4,    2,  137},  // 66     1  0.15%    0.03%   6.67%
  { 34048,   25,    2,  128},  // 67     6  0.27%    1.99%   3.91%
  { 40960,    5,    2,  129},  // 68     1  0.12%    0.03%  20.30%
  { 49152,    6,    2,  128},  // 69     1  0.10%    0.03%  20.00%
  { 57344,    7,    2,  128},  // 70     1  0.08%    0.03%  16.67%
  { 65536,    8,    2,  134},  // 71     1  0.07%    0.03%  14.29%
  { 68224,   25,    2,  130},  // 72     3  0.09%    3.55%   4.10%
  { 81920,   10,    2,  129},  // 73     1  0.06%    0.03%  20.08%
  { 98304,   12,    2,  128},  // 74     1  0.05%    0.03%  20.00%
  {114688,   14,    2,  128},  // 75     1  0.04%    0.03%  16.67%
  {133760,   49,    2,  129},  // 76     3  0.04%    6.68%  16.63%
  {155648,   19,    2,  121},  // 77     1  0.03%    0.03%  16.36%
  {180224,   22,    2,  127},  // 78     1  0.03%    0.03%  15.79%
  {212992,   26,    2,  127},  // 79     1  0.02%    0.03%  18.18%
  {262144,   32,    2,  127},  // 80     1  0.02%    0.03%  23.08%
};
#elif TCMALLOC_PAGE_SHIFT == 15
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 1824},  //  0  4096  0.15%    1.60%   0.00%
  {    16,    1,   32, 1824},  //  1  2048  0.15%    1.60% 100.00%
  {    32,    1,   32, 1824},  //  2  1024  0.15%    1.60% 100.00%
  {    64,    1,   32, 1824},  //  3   512  0.15%    1.60% 100.00%
  {    72,    1,   32, 1241},  //  4   455  0.17%    1.60%  12.50%
  {    80,    1,   32, 1824},  //  5   409  0.29%    1.60%  11.11%
  {    88,    1,   32, 1268},  //  6   372  0.24%    1.60%  10.00%
  {    96,    1,   32, 1590},  //  7   341  0.24%    1.60%   9.09%
  {   104,    1,   32,  718},  //  8   315  0.17%    1.60%   8.33%
  {   112,    1,   32,  844},  //  9   292  0.34%    1.60%   7.69%
  {   120,    1,   32,  678},  // 10   273  0.17%    1.60%   7.14%
  {   128,    1,   32, 1448},  // 11   256  0.15%    1.60%   6.67%
  {   144,    1,   32,  909},  // 12   227  0.39%    1.60%  12.50%
  {   160,    1,   32,  744},  // 13   204  0.54%    1.60%  11.11%
  {   176,    1,   32,  461},  // 14   186  0.24%    1.60%  10.00%
  {   192,    1,   32,  603},  // 15   170  0.54%    1.60%   9.09%
  {   208,    1,   32,  298},  // 16   157  0.49%    1.60%   8.33%
  {   240,    1,   32,  686},  // 17   136  0.54%    1.60%  15.38%
  {   256,    1,   32,  812},  // 18   128  0.15%    1.60%   6.67%
  {   280,    1,   32,  385},  // 19   117  0.17%    1.60%   9.38%
  {   304,    1,   32,  290},  // 20   107  0.88%    1.60%   8.57%
  {   320,    1,   32,  203},  // 21   102  0.54%    1.60%   5.26%
  {   352,    1,   32,  398},  // 22    93  0.24%    1.60%  10.00%
  {   400,    1,   32,  298},  // 23    81  1.27%    1.60%  13.64%
  {   448,    1,   32,  255},  // 24    73  0.34%    1.60%  12.00%
  {   512,    1,   32,  480},  // 25    64  0.15%    1.60%  14.29%
  {   544,    1,   32,  180},  // 26    60  0.54%    1.60%   6.25%
  {   584,    1,   32,  181},  // 27    56  0.34%    1.60%   7.35%
  {   640,    1,   32,  276},  // 28    51  0.54%    1.60%   9.59%
  {   712,    1,   32,  224},  // 29    46  0.20%    1.60%  11.25%
  {   816,    1,   32,  265},  // 30    40  0.54%    1.60%  14.61%
  {   904,    1,   32,  192},  // 31    36  0.83%    1.60%  10.78%
  {  1024,    1,   32,  362},  // 32    32  0.15%    1.60%  13.27%
  {  1152,    1,   32,  197},  // 33    28  1.71%    1.60%  12.50%
  {  1408,    1,   32,  235},  // 34    23  1.32%    1.60%  22.22%
  {  1536,    1,   32,  163},  // 35    21  1.71%    1.60%   9.09%
  {  1792,    1,   32,  158},  // 36    18  1.71%    1.60%  16.67%
  {  2048,    1,   32,  183},  // 37    16  0.15%    1.60%  14.29%
  {  2176,    1,   30,  162},  // 38    15  0.54%    1.60%   6.25%
  {  2432,    3,   26,  143},  // 39    40  1.09%    1.60%  11.76%
  {  2688,    1,   24,  141},  // 40    12  1.71%    1.60%  10.53%
  {  3072,    3,   21,  134},  // 41    32  0.05%    1.60%  14.29%
  {  3584,    1,   18,  135},  // 42     9  1.71%    1.60%  16.67%
  {  4096,    1,   16,  321},  // 43     8  0.15%    1.60%  14.29%
  {  4224,    3,   15,  120},  // 44    23  1.22%    1.60%   3.12%
  {  4608,    1,   14,  129},  // 45     7  1.71%    1.60%   9.09%
  {  5376,    1,   12,  128},  // 46     6  1.71%    1.60%  16.67%
  {  6528,    1,   10,  143},  // 47     5  0.54%    1.60%  21.43%
  {  8192,    1,    8,  165},  // 48     4  0.15%    1.60%  25.49%
  {  9344,    2,    7,  127},  // 49     7  0.27%    1.60%  14.06%
  { 10880,    1,    6,  120},  // 50     3  0.54%    1.60%  16.44%
  { 13568,    5,    4,  123},  // 51    12  0.65%    1.60%  24.71%
  { 16384,    1,    4,  147},  // 52     2  0.15%    1.60%  20.75%
  { 18688,    4,    3,  124},  // 53     7  0.23%    1.60%  14.06%
  { 21760,    2,    3,  117},  // 54     3  0.46%    1.60%  16.44%
  { 24576,    3,    2,  117},  // 55     4  0.05%    1.60%  12.94%
  { 27264,    5,    2,  116},  // 56     6  0.19%    1.60%  10.94%
  { 32768,    1,    2,  141},  // 57     1  0.15%    0.03%  20.19%
  { 35200,   14,    2,  116},  // 58    13  0.26%    3.16%   7.42%
  { 40960,    5,    2,  115},  // 59     4  0.03%    3.16%  16.36%
  { 49152,    3,    2,  115},  // 60     2  0.05%    3.16%  20.00%
  { 54528,    5,    2,  115},  // 61     3  0.19%    3.16%  10.94%
  { 65536,    2,    2,  124},  // 62     1  0.07%    0.03%  20.19%
  { 70912,   13,    2,  116},  // 63     6  0.13%    4.73%   8.20%
  { 81920,    5,    2,  116},  // 64     2  0.03%    4.72%  15.52%
  { 98304,    3,    2,  115},  // 65     1  0.05%    0.03%  20.00%
  {114688,    7,    2,  115},  // 66     2  0.02%    6.28%  16.67%
  {131072,    4,    2,  142},  // 67     1  0.04%    0.03%  14.29%
  {141952,   13,    2,  114},  // 68     3  0.04%    7.85%   8.30%
  {163840,    5,    2,  114},  // 69     1  0.03%    0.03%  15.42%
  {196608,    6,    2,  115},  // 70     1  0.02%    0.03%  20.00%
  {229376,    7,    2,  111},  // 71     1  0.02%    0.03%  16.67%
  {262144,    8,    2,  117},  // 72     1  0.02%    0.03%  14.29%
};
#elif TCMALLOC_PAGE_SHIFT == 18
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 1912},  //  0 32768  0.02%   12.53%   0.00%
  {    16,    1,   32, 1912},  //  1 16384  0.02%   12.53% 100.00%
  {    32,    1,   32, 1912},  //  2  8192  0.02%   12.53% 100.00%
  {    64,    1,   32, 1920},  //  3  4096  0.02%   12.53% 100.00%
  {    72,    1,   32, 1912},  //  4  3640  0.04%   12.53%  12.50%
  {    80,    1,   32, 1691},  //  5  3276  0.04%   12.53%  11.11%
  {    88,    1,   32,  632},  //  6  2978  0.05%   12.53%  10.00%
  {    96,    1,   32,  898},  //  7  2730  0.04%   12.53%   9.09%
  {   104,    1,   32,  510},  //  8  2520  0.04%   12.53%   8.33%
  {   112,    1,   32,  758},  //  9  2340  0.04%   12.53%   7.69%
  {   128,    1,   32, 1197},  // 10  2048  0.02%   12.53%  14.29%
  {   144,    1,   32,  992},  // 11  1820  0.04%   12.53%  12.50%
  {   160,    1,   32,  841},  // 12  1638  0.04%   12.53%  11.11%
  {   192,    1,   32,  642},  // 13  1365  0.04%   12.53%  20.00%
  {   208,    1,   32,  299},  // 14  1260  0.04%   12.53%   8.33%
  {   232,    1,   32,  624},  // 15  1129  0.10%   12.53%  11.54%
  {   256,    1,   32,  737},  // 16  1024  0.02%   12.53%  10.34%
  {   288,    1,   32,  564},  // 17   910  0.04%   12.53%  12.50%
  {   336,    1,   32,  667},  // 18   780  0.04%   12.53%  16.67%
  {   376,    1,   32,  220},  // 19   697  0.05%   12.53%  11.90%
  {   416,    1,   32,  295},  // 20   630  0.04%   12.53%  10.64%
  {   472,    1,   32,  275},  // 21   555  0.09%   12.53%  13.46%
  {   512,    1,   32,  339},  // 22   512  0.02%   12.53%   8.47%
  {   576,    1,   32,  266},  // 23   455  0.04%   12.53%  12.50%
  {   680,    1,   32,  287},  // 24   385  0.15%   12.53%  18.06%
  {   768,    1,   32,  214},  // 25   341  0.12%   12.53%  12.94%
  {   896,    1,   32,  212},  // 26   292  0.21%   12.53%  16.67%
  {  1024,    1,   32,  340},  // 27   256  0.02%   12.53%  14.29%
  {  1152,    1,   32,  194},  // 28   227  0.26%   12.53%  12.50%
  {  1280,    1,   32,  170},  // 29   204  0.41%   12.53%  11.11%
  {  1408,    1,   32,  148},  // 30   186  0.12%   12.53%  10.00%
  {  1664,    1,   32,  258},  // 31   157  0.36%   12.53%  18.18%
  {  1920,    1,   32,  212},  // 32   136  0.41%   12.53%  15.38%
  {  2048,    1,   32,  183},  // 33   128  0.02%   12.53%   6.67%
  {  2176,    1,   30,  312},  // 34   120  0.41%   12.53%   6.25%
  {  2304,    1,   28,  153},  // 35   113  0.70%   12.53%   5.88%
  {  2560,    1,   25,  146},  // 36   102  0.41%   12.53%  11.11%
  {  2816,    1,   23,  129},  // 37    93  0.12%   12.53%  10.00%
  {  3072,    1,   21,  130},  // 38    85  0.41%   12.53%   9.09%
  {  3328,    1,   19,  147},  // 39    78  0.99%   12.54%   8.33%
  {  3584,    1,   18,  126},  // 40    73  0.21%   12.53%   7.69%
  {  3840,    1,   17,  126},  // 41    68  0.41%   12.53%   7.14%
  {  4096,    1,   16,  273},  // 42    64  0.02%   12.53%   6.67%
  {  4224,    1,   15,  132},  // 43    62  0.12%   12.53%   3.12%
  {  4736,    1,   13,  136},  // 44    55  0.65%   12.54%  12.12%
  {  5504,    1,   11,  150},  // 45    47  1.34%   12.54%  16.22%
  {  6144,    1,   10,  130},  // 46    42  1.58%   12.54%  11.63%
  {  7168,    1,    9,  133},  // 47    36  1.58%   12.54%  16.67%
  {  8192,    1,    8,  167},  // 48    32  0.02%   12.53%  14.29%
  {  9344,    1,    7,  130},  // 49    28  0.21%   12.53%  14.06%
  { 10368,    1,    6,  124},  // 50    25  1.14%   12.54%  10.96%
  { 11392,    1,    5,  128},  // 51    23  0.07%   12.53%   9.88%
  { 12416,    1,    5,  126},  // 52    21  0.56%   12.54%   8.99%
  { 13440,    2,    4,  123},  // 53    39  0.03%   12.53%   8.25%
  { 14976,    2,    4,  123},  // 54    35  0.03%   12.53%  11.43%
  { 16384,    1,    4,  140},  // 55    16  0.02%   12.53%   9.40%
  { 18048,    2,    3,  127},  // 56    29  0.18%   12.54%  10.16%
  { 20096,    1,    3,  122},  // 57    13  0.36%   12.54%  11.35%
  { 21760,    1,    3,  121},  // 58    12  0.41%   12.54%   8.28%
  { 23808,    1,    2,  125},  // 59    11  0.12%   12.53%   9.41%
  { 26112,    1,    2,  122},  // 60    10  0.41%   12.54%   9.68%
  { 29056,    1,    2,  120},  // 61     9  0.26%   12.54%  11.27%
  { 32768,    1,    2,  170},  // 62     8  0.02%   12.53%  12.78%
  { 34944,    2,    2,  121},  // 63    15  0.03%   12.53%   6.64%
  { 39296,    3,    2,  120},  // 64    20  0.07%   12.53%  12.45%
  { 43648,    1,    2,  120},  // 65     6  0.12%   12.54%  11.07%
  { 47616,    2,    2,  119},  // 66    11  0.11%   12.54%   9.09%
  { 52352,    1,    2,  120},  // 67     5  0.16%   12.54%   9.95%
  { 58240,    2,    2,  119},  // 68     9  0.03%   12.53%  11.25%
  { 65536,    1,    2,  122},  // 69     4  0.02%   12.53%  12.53%
  { 69888,    4,    2,  119},  // 70    15  0.03%   12.53%   6.64%
  { 74880,    2,    2,  119},  // 71     7  0.03%   12.53%   7.14%
  { 78592,    3,    2,  119},  // 72    10  0.07%   12.54%   4.96%
  { 87296,    1,    2,  119},  // 73     3  0.12%   12.54%  11.07%
  { 98304,    3,    2,  119},  // 74     8  0.01%   12.53%  12.61%
  {104832,    2,    2,  120},  // 75     5  0.03%   12.54%   6.64%
  {116480,    4,    2,  119},  // 76     9  0.03%   12.54%  11.11%
  {131072,    1,    2,  120},  // 77     2  0.02%   12.53%  12.53%
  {142976,    6,    2,  119},  // 78    11  0.01%   12.53%   9.08%
  {157184,    3,    2,  119},  // 79     5  0.07%   12.54%   9.94%
  {174720,    2,    2,  119},  // 80     3  0.03%   12.54%  11.16%
  {196608,    3,    2,  119},  // 81     4  0.01%   12.53%  12.53%
  {218368,    5,    2,  119},  // 82     6  0.04%   12.54%  11.07%
  {262144,    1,    2,  122},  // 83     1  0.02%    0.03%  20.05%
};
#elif TCMALLOC_PAGE_SHIFT == 12
static_assert(kMaxSize == 8192, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = false,
  .span_size = 48,
  .sampling_rate = 524288,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2630},  //  0   512  1.16%    0.92%   0.00%
  {    16,    1,   32, 2630},  //  1   256  1.16%    0.92% 100.00%
  {    32,    1,   32, 2633},  //  2   128  1.16%    0.92% 100.00%
  {    64,    1,   32, 2630},  //  3    64  1.16%    0.92% 100.00%
  {    72,    1,   32,  927},  //  4    56  2.70%    0.92%  12.50%
  {    80,    1,   32, 2630},  //  5    51  1.54%    0.92%  11.11%
  {    88,    1,   32, 1083},  //  6    46  2.32%    0.92%  10.00%
  {    96,    1,   32, 1257},  //  7    42  2.70%    0.92%   9.09%
  {   104,    1,   32,  669},  //  8    39  2.12%    0.92%   8.33%
  {   112,    1,   32, 1198},  //  9    36  2.70%    0.92%   7.69%
  {   128,    1,   32, 1610},  // 10    32  1.16%    0.92%  14.29%
  {   144,    1,   32, 1294},  // 11    28  2.70%    0.92%  12.50%
  {   160,    2,   32, 1169},  // 12    51  0.97%    0.92%  11.11%
  {   176,    1,   32,  562},  // 13    23  2.32%    0.92%  10.00%
  {   192,    1,   32,  607},  // 14    21  2.70%    0.92%   9.09%
  {   208,    2,   32,  392},  // 15    39  1.55%    0.92%   8.33%
  {   216,    2,   32,  330},  // 16    37  3.01%    0.92%   3.85%
  {   240,    1,   32,  533},  // 17    17  1.54%    0.92%  11.11%
  {   248,    2,   32,  231},  // 18    33  0.68%    0.92%   3.33%
  {   264,    2,   32,  564},  // 19    31  0.68%    0.92%   6.45%
  {   280,    2,   32,  258},  // 20    29  1.46%    0.92%   6.06%
  {   304,    3,   32,  348},  // 21    40  1.43%    0.92%   8.57%
  {   336,    1,   32,  483},  // 22    12  2.70%    0.92%  10.53%
  {   392,    3,   32,  239},  // 23    31  1.49%    0.92%  16.67%
  {   448,    1,   32,  231},  // 24     9  2.70%    0.92%  14.29%
  {   528,    3,   32,  762},  // 25    23  1.56%    0.92%  17.86%
  {   624,    2,   32,  331},  // 26    13  1.55%    0.92%  18.18%
  {   768,    3,   32,  343},  // 27    16  0.39%    0.92%  23.08%
  {   904,    2,   32,  289},  // 28     9  1.26%    0.92%  17.71%
  {  1024,    1,   32,  959},  // 29     4  1.16%    0.92%  13.27%
  {  1280,    5,   32,  213},  // 30    16  0.23%    0.92%  25.00%
  {  1408,   10,   32,  168},  // 31    29  0.43%    0.92%  10.00%
  {  1664,    9,   32,  196},  // 32    22  0.82%    0.92%  18.18%
  {  2048,    1,   32,  518},  // 33     2  1.16%    0.92%  23.08%
  {  2432,    3,   26,  208},  // 34     5  1.43%    0.92%  18.75%
  {  3200,    4,   20,  167},  // 35     5  2.63%    0.93%  31.58%
  {  4096,    1,   16,  623},  // 36     1  1.16%    0.13%  28.00%
  {  4736,    7,   13,  209},  // 37     6  1.06%    1.71%  15.62%
  {  6144,    3,   10,  164},  // 38     2  0.39%    1.70%  29.73%
  {  7168,    7,    9,  166},  // 39     4  0.17%    1.70%  16.67%
  {  8192,    2,    8,  232},  // 40     1  0.58%    0.13%  14.29%
};
#else
#error "Unsupported TCMALLOC_PAGE_SHIFT value!"
#endif
#else
#if TCMALLOC_PAGE_SHIFT == 13
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2369},  //  0  1024  0.58%    0.42%   0.00%
  {    16,    1,   32, 2369},  //  1   512  0.58%    0.42% 100.00%
  {    32,    1,   32, 2372},  //  2   256  0.58%    0.42% 100.00%
  {    64,    1,   32, 2369},  //  3   128  0.58%    0.42% 100.00%
  {    80,    1,   32, 2369},  //  4   102  0.97%    0.42%  25.00%
  {    96,    1,   32, 1597},  //  5    85  0.97%    0.42%  20.00%
  {   112,    1,   32,  910},  //  6    73  0.78%    0.42%  16.67%
  {   128,    1,   32, 1035},  //  7    64  0.58%    0.42%  14.29%
  {   144,    1,   32,  699},  //  8    56  2.14%    0.42%  12.50%
  {   160,    1,   32,  587},  //  9    51  0.97%    0.42%  11.11%
  {   176,    1,   32,  333},  // 10    46  1.75%    0.42%  10.00%
  {   192,    1,   32,  418},  // 11    42  2.14%    0.42%   9.09%
  {   208,    1,   32,  295},  // 12    39  1.55%    0.42%   8.33%
  {   224,    1,   32,  264},  // 13    36  2.14%    0.42%   7.69%
  {   240,    1,   32,  251},  // 14    34  0.97%    0.42%   7.14%
  {   256,    1,   32,  507},  // 15    32  0.58%    0.42%   6.67%
  {   272,    1,   32,  230},  // 16    30  0.97%    0.42%   6.25%
  {   288,    1,   32,  264},  // 17    28  2.14%    0.42%   5.88%
  {   320,    2,   32,  307},  // 18    51  0.68%    0.42%  11.11%
  {   336,    1,   32,  270},  // 19    24  2.14%    0.42%   5.00%
  {   352,    1,   32,  193},  // 20    23  1.75%    0.42%   4.76%
  {   384,    1,   32,  235},  // 21    21  2.14%    0.42%   9.09%
  {   416,    2,   32,  228},  // 22    39  1.27%    0.42%   8.33%
  {   448,    1,   32,  236},  // 23    18  2.14%    0.42%   7.69%
  {   512,    1,   32,  406},  // 24    16  0.58%    0.42%  14.29%
  {   576,    1,   32,  241},  // 25    14  2.14%    0.42%  12.50%
  {   640,    4,   32,  213},  // 26    51  0.54%    0.42%  11.11%
  {   704,    2,   32,  193},  // 27    23  1.46%    0.42%  10.00%
  {   768,    3,   32,  191},  // 28    32  0.19%    0.42%   9.09%
  {   848,    2,   32,  184},  // 29    19  1.95%    0.42%  10.42%
  {   944,    3,   32,  182},  // 30    26  0.32%    0.42%  11.32%
  {  1024,    1,   32,  319},  // 31     8  0.58%    0.42%   8.47%
  {  1152,    1,   32,  197},  // 32     7  2.14%    0.43%  12.50%
  {  1280,    3,   32,  180},  // 33    19  1.23%    0.42%  11.11%
  {  1408,    4,   32,  172},  // 34    23  1.32%    0.43%  10.00%
  {  1536,    3,   32,  178},  // 35    16  0.19%    0.42%   9.09%
  {  1792,    2,   32,  175},  // 36     9  1.85%    0.43%  16.67%
  {  1920,    4,   32,  158},  // 37    17  0.54%    0.42%   7.14%
  {  2048,    1,   32,  192},  // 38     4  0.58%    0.42%   6.67%
  {  2176,    4,   30,  160},  // 39    15  0.54%    0.42%   6.25%
  {  2304,    2,   28,  160},  // 40     7  1.85%    0.43%   5.88%
  {  2560,    5,   25,  161},  // 41    16  0.12%    0.42%  11.11%
  {  2816,   10,   23,  158},  // 42    29  0.37%    0.42%  10.00%
  {  3072,    3,   21,  156},  // 43     8  0.19%    0.42%   9.09%
  {  3328,    9,   19,  154},  // 44    22  0.76%    0.43%   8.33%
  {  3712,    5,   17,  160},  // 45    11  0.43%    0.42%  11.54%
  {  4096,    1,   16,  307},  // 46     2  0.58%    0.43%  10.34%
  {  4352,    7,   15,  153},  // 47    13  1.42%    0.43%   6.25%
  {  4864,    3,   13,  154},  // 48     5  1.23%    0.43%  11.76%
  {  5376,    2,   12,  152},  // 49     3  1.85%    0.43%  10.53%
  {  6144,    3,   10,  158},  // 50     4  0.19%    0.42%  14.29%
  {  7168,    7,    9,  154},  // 51     8  0.08%    0.42%  16.67%
  {  8192,    1,    8,  207},  // 52     1  0.58%    0.03%  14.29%
  {  9344,    8,    7,  153},  // 53     7  0.27%    0.82%  14.06%
  { 10496,    9,    6,  152},  // 54     7  0.41%    0.82%  12.33%
  { 12288,    3,    5,  154},  // 55     2  0.19%    0.82%  17.07%
  { 13568,    5,    4,  150},  // 56     3  0.74%    0.82%  10.42%
  { 14720,    9,    4,  149},  // 57     5  0.24%    0.82%   8.49%
  { 16384,    2,    4,  158},  // 58     1  0.29%    0.03%  11.30%
  { 17408,   17,    3,  150},  // 59     8  0.03%    1.21%   6.25%
  { 19072,    7,    3,  149},  // 60     3  0.31%    1.21%   9.56%
  { 21760,    8,    3,  152},  // 61     3  0.46%    1.21%  14.09%
  { 24576,    3,    2,  149},  // 62     1  0.19%    0.03%  12.94%
  { 26624,   13,    2,  149},  // 63     4  0.05%    1.60%   8.33%
  { 28672,    7,    2,  150},  // 64     2  0.08%    1.60%   7.69%
  { 30720,   15,    2,  153},  // 65     4  0.04%    1.60%   7.14%
  { 32768,    4,    2,  156},  // 66     1  0.15%    0.03%   6.67%
  { 33408,   45,    2,  148},  // 67    11  0.33%    1.99%   1.95%
  { 35456,   13,    2,  148},  // 68     3  0.17%    1.99%   6.13%
  { 40960,    5,    2,  149},  // 69     1  0.12%    0.03%  15.52%
  { 49152,    6,    2,  149},  // 70     1  0.10%    0.03%  20.00%
  { 57344,    7,    2,  149},  // 71     1  0.08%    0.03%  16.67%
  { 65536,    8,    2,  153},  // 72     1  0.07%    0.03%  14.29%
  { 68224,   25,    2,  150},  // 73     3  0.09%    3.55%   4.10%
  { 81920,   10,    2,  149},  // 74     1  0.06%    0.03%  20.08%
  { 98304,   12,    2,  149},  // 75     1  0.05%    0.03%  20.00%
  {114688,   14,    2,  149},  // 76     1  0.04%    0.03%  16.67%
  {131072,   16,    2,  148},  // 77     1  0.04%    0.03%  14.29%
  {139264,   17,    2,  149},  // 78     1  0.03%    0.03%   6.25%
  {155648,   19,    2,  148},  // 79     1  0.03%    0.03%  11.76%
  {172032,   21,    2,  148},  // 80     1  0.03%    0.03%  10.53%
  {204800,   25,    2,  148},  // 81     1  0.02%    0.03%  19.05%
  {229376,   28,    2,  148},  // 82     1  0.02%    0.03%  12.00%
  {262144,   32,    2,  148},  // 83     1  0.02%    0.03%  14.29%
};
#elif TCMALLOC_PAGE_SHIFT == 15
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2249},  //  0  4096  0.15%    1.60%   0.00%
  {    16,    1,   32, 2249},  //  1  2048  0.15%    1.60% 100.00%
  {    32,    1,   32, 2249},  //  2  1024  0.15%    1.60% 100.00%
  {    64,    1,   32, 2249},  //  3   512  0.15%    1.60% 100.00%
  {    80,    1,   32, 2249},  //  4   409  0.29%    1.60%  25.00%
  {    96,    1,   32, 2100},  //  5   341  0.24%    1.60%  20.00%
  {   112,    1,   32, 1138},  //  6   292  0.34%    1.60%  16.67%
  {   128,    1,   32, 1563},  //  7   256  0.15%    1.60%  14.29%
  {   144,    1,   32,  739},  //  8   227  0.39%    1.60%  12.50%
  {   160,    1,   32,  615},  //  9   204  0.54%    1.60%  11.11%
  {   176,    1,   32,  402},  // 10   186  0.24%    1.60%  10.00%
  {   192,    1,   32,  509},  // 11   170  0.54%    1.60%   9.09%
  {   208,    1,   32,  279},  // 12   157  0.49%    1.60%   8.33%
  {   240,    1,   32,  571},  // 13   136  0.54%    1.60%  15.38%
  {   256,    1,   32,  666},  // 14   128  0.15%    1.60%   6.67%
  {   288,    1,   32,  382},  // 15   113  0.83%    1.60%  12.50%
  {   304,    1,   32,  234},  // 16   107  0.88%    1.60%   5.56%
  {   320,    1,   32,  208},  // 17   102  0.54%    1.60%   5.26%
  {   352,    1,   32,  355},  // 18    93  0.24%    1.60%  10.00%
  {   400,    1,   32,  279},  // 19    81  1.27%    1.60%  13.64%
  {   448,    1,   32,  246},  // 20    73  0.34%    1.60%  12.00%
  {   512,    1,   32,  416},  // 21    64  0.15%    1.60%  14.29%
  {   544,    1,   32,  190},  // 22    60  0.54%    1.60%   6.25%
  {   592,    1,   32,  209},  // 23    55  0.78%    1.60%   8.82%
  {   640,    1,   32,  245},  // 24    51  0.54%    1.60%   8.11%
  {   704,    1,   32,  222},  // 25    46  1.32%    1.60%  10.00%
  {   816,    1,   32,  255},  // 26    40  0.54%    1.60%  15.91%
  {   896,    1,   32,  198},  // 27    36  1.71%    1.60%   9.80%
  {  1024,    1,   32,  328},  // 28    32  0.15%    1.60%  14.29%
  {  1152,    1,   32,  203},  // 29    28  1.71%    1.60%  12.50%
  {  1280,    2,   32,  187},  // 30    51  0.46%    1.60%  11.11%
  {  1408,    1,   32,  187},  // 31    23  1.32%    1.60%  10.00%
  {  1536,    1,   32,  178},  // 32    21  1.71%    1.60%   9.09%
  {  1792,    1,   32,  174},  // 33    18  1.71%    1.60%  16.67%
  {  1920,    1,   32,  150},  // 34    17  0.54%    1.60%   7.14%
  {  2048,    1,   32,  183},  // 35    16  0.15%    1.60%   6.67%
  {  2176,    1,   30,  177},  // 36    15  0.54%    1.60%   6.25%
  {  2432,    3,   26,  163},  // 37    40  1.09%    1.60%  11.76%
  {  2688,    1,   24,  160},  // 38    12  1.71%    1.60%  10.53%
  {  3072,    3,   21,  155},  // 39    32  0.05%    1.60%  14.29%
  {  3584,    1,   18,  157},  // 40     9  1.71%    1.60%  16.67%
  {  4096,    1,   16,  297},  // 41     8  0.15%    1.60%  14.29%
  {  4224,    3,   15,  146},  // 42    23  1.22%    1.60%   3.12%
  {  4608,    1,   14,  152},  // 43     7  1.71%    1.60%   9.09%
  {  5120,    3,   12,  148},  // 44    19  1.09%    1.60%  11.11%
  {  5760,    3,   11,  150},  // 45    17  0.44%    1.60%  12.50%
  {  6528,    1,   10,  155},  // 46     5  0.54%    1.60%  13.33%
  {  7168,    2,    9,  143},  // 47     9  1.63%    1.60%   9.80%
  {  8192,    1,    8,  177},  // 48     4  0.15%    1.60%  14.29%
  {  9344,    2,    7,  150},  // 49     7  0.27%    1.60%  14.06%
  { 10880,    1,    6,  145},  // 50     3  0.54%    1.60%  16.44%
  { 12288,    3,    5,  144},  // 51     8  0.05%    1.60%  12.94%
  { 13568,    5,    4,  145},  // 52    12  0.65%    1.60%  10.42%
  { 16384,    1,    4,  165},  // 53     2  0.15%    1.60%  20.75%
  { 18688,    4,    3,  148},  // 54     7  0.23%    1.60%  14.06%
  { 21760,    2,    3,  143},  // 55     3  0.46%    1.60%  16.44%
  { 24576,    3,    2,  143},  // 56     4  0.05%    1.60%  12.94%
  { 27264,    5,    2,  142},  // 57     6  0.19%    1.60%  10.94%
  { 32768,    1,    2,  160},  // 58     1  0.15%    0.03%  20.19%
  { 33792,   31,    2,  142},  // 59    30  0.21%    3.16%   3.12%
  { 37376,    8,    2,  142},  // 60     7  0.21%    3.16%  10.61%
  { 40960,    5,    2,  141},  // 61     4  0.03%    3.16%   9.59%
  { 49152,    3,    2,  142},  // 62     2  0.05%    3.16%  20.00%
  { 54528,    5,    2,  142},  // 63     3  0.19%    3.16%  10.94%
  { 65536,    2,    2,  148},  // 64     1  0.07%    0.03%  20.19%
  { 70912,   13,    2,  142},  // 65     6  0.13%    4.73%   8.20%
  { 81920,    5,    2,  142},  // 66     2  0.03%    4.72%  15.52%
  { 98304,    3,    2,  142},  // 67     1  0.05%    0.03%  20.00%
  {114688,    7,    2,  142},  // 68     2  0.02%    6.28%  16.67%
  {131072,    4,    2,  161},  // 69     1  0.04%    0.03%  14.29%
  {141952,   13,    2,  141},  // 70     3  0.04%    7.85%   8.30%
  {163840,    5,    2,  141},  // 71     1  0.03%    0.03%  15.42%
  {196608,    6,    2,  142},  // 72     1  0.02%    0.03%  20.00%
  {212992,   13,    2,  141},  // 73     2  0.01%   10.97%   8.33%
  {229376,    7,    2,  137},  // 74     1  0.02%    0.03%   7.69%
  {262144,    8,    2,  143},  // 75     1  0.02%    0.03%  14.29%
};
#elif TCMALLOC_PAGE_SHIFT == 18
static_assert(kMaxSize == 262144, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = true,
  .span_size = 48,
  .sampling_rate = 2097152,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2368},  //  0 32768  0.02%   12.53%   0.00%
  {    16,    1,   32, 2368},  //  1 16384  0.02%   12.53% 100.00%
  {    32,    1,   32, 2368},  //  2  8192  0.02%   12.53% 100.00%
  {    64,    1,   32, 2372},  //  3  4096  0.02%   12.53% 100.00%
  {    80,    1,   32, 2368},  //  4  3276  0.04%   12.53%  25.00%
  {    96,    1,   32, 1006},  //  5  2730  0.04%   12.53%  20.00%
  {   112,    1,   32,  834},  //  6  2340  0.04%   12.53%  16.67%
  {   128,    1,   32,  871},  //  7  2048  0.02%   12.53%  14.29%
  {   144,    1,   32,  733},  //  8  1820  0.04%   12.53%  12.50%
  {   160,    1,   32,  633},  //  9  1638  0.04%   12.53%  11.11%
  {   176,    1,   32,  302},  // 10  1489  0.05%   12.53%  10.00%
  {   192,    1,   32,  347},  // 11  1365  0.04%   12.53%   9.09%
  {   208,    1,   32,  268},  // 12  1260  0.04%   12.53%   8.33%
  {   224,    1,   32,  466},  // 13  1170  0.04%   12.53%   7.69%
  {   256,    1,   32,  584},  // 14  1024  0.02%   12.53%  14.29%
  {   288,    1,   32,  446},  // 15   910  0.04%   12.53%  12.50%
  {   336,    1,   32,  516},  // 16   780  0.04%   12.53%  16.67%
  {   384,    1,   32,  261},  // 17   682  0.12%   12.53%  14.29%
  {   432,    1,   32,  289},  // 18   606  0.15%   12.53%  12.50%
  {   480,    1,   32,  205},  // 19   546  0.04%   12.53%  11.11%
  {   512,    1,   32,  275},  // 20   512  0.02%   12.53%   6.67%
  {   576,    1,   32,  246},  // 21   455  0.04%   12.53%  12.50%
  {   640,    1,   32,  235},  // 22   409  0.16%   12.53%  11.11%
  {   704,    1,   32,  197},  // 23   372  0.12%   12.53%  10.00%
  {   800,    1,   32,  219},  // 24   327  0.23%   12.53%  13.64%
  {   896,    1,   32,  180},  // 25   292  0.21%   12.53%  12.00%
  {  1024,    1,   32,  296},  // 26   256  0.02%   12.53%  14.29%
  {  1152,    1,   32,  198},  // 27   227  0.26%   12.53%  12.50%
  {  1280,    1,   32,  182},  // 28   204  0.41%   12.53%  11.11%
  {  1408,    1,   32,  168},  // 29   186  0.12%   12.53%  10.00%
  {  1664,    1,   32,  241},  // 30   157  0.36%   12.53%  18.18%
  {  1920,    1,   32,  209},  // 31   136  0.41%   12.53%  15.38%
  {  2048,    1,   32,  191},  // 32   128  0.02%   12.53%   6.67%
  {  2176,    1,   30,  278},  // 33   120  0.41%   12.53%   6.25%
  {  2304,    1,   28,  171},  // 34   113  0.70%   12.53%   5.88%
  {  2560,    1,   25,  165},  // 35   102  0.41%   12.53%  11.11%
  {  2816,    1,   23,  155},  // 36    93  0.12%   12.53%  10.00%
  {  3072,    1,   21,  155},  // 37    85  0.41%   12.53%   9.09%
  {  3328,    1,   19,  167},  // 38    78  0.99%   12.54%   8.33%
  {  3584,    1,   18,  153},  // 39    73  0.21%   12.53%   7.69%
  {  3840,    1,   17,  153},  // 40    68  0.41%   12.53%   7.14%
  {  4096,    1,   16,  251},  // 41    64  0.02%   12.53%   6.67%
  {  4224,    1,   15,  156},  // 42    62  0.12%   12.53%   3.12%
  {  4736,    1,   13,  160},  // 43    55  0.65%   12.54%  12.12%
  {  5248,    1,   12,  167},  // 44    49  1.92%   12.54%  10.81%
  {  5760,    1,   11,  153},  // 45    45  1.14%   12.54%   9.76%
  {  6528,    1,   10,  158},  // 46    40  0.41%   12.53%  13.33%
  {  7168,    1,    9,  150},  // 47    36  1.58%   12.54%   9.80%
  {  8192,    1,    8,  180},  // 48    32  0.02%   12.53%  14.29%
  {  9344,    1,    7,  155},  // 49    28  0.21%   12.53%  14.06%
  { 10368,    1,    6,  151},  // 50    25  1.14%   12.54%  10.96%
  { 11392,    1,    5,  154},  // 51    23  0.07%   12.53%   9.88%
  { 12416,    1,    5,  153},  // 52    21  0.56%   12.54%   8.99%
  { 13440,    2,    4,  150},  // 53    39  0.03%   12.53%   8.25%
  { 14976,    2,    4,  150},  // 54    35  0.03%   12.53%  11.43%
  { 16384,    1,    4,  162},  // 55    16  0.02%   12.53%   9.40%
  { 17408,    1,    3,  150},  // 56    15  0.41%   12.54%   6.25%
  { 18688,    1,    3,  151},  // 57    14  0.21%   12.54%   7.35%
  { 20096,    1,    3,  149},  // 58    13  0.36%   12.54%   7.53%
  { 21760,    1,    3,  149},  // 59    12  0.41%   12.54%   8.28%
  { 23808,    1,    2,  151},  // 60    11  0.12%   12.53%   9.41%
  { 26112,    1,    2,  150},  // 61    10  0.41%   12.54%   9.68%
  { 29056,    1,    2,  149},  // 62     9  0.26%   12.54%  11.27%
  { 32768,    1,    2,  182},  // 63     8  0.02%   12.53%  12.78%
  { 34944,    2,    2,  149},  // 64    15  0.03%   12.53%   6.64%
  { 37376,    1,    2,  149},  // 65     7  0.21%   12.54%   6.96%
  { 40320,    2,    2,  148},  // 66    13  0.03%   12.53%   7.88%
  { 43648,    1,    2,  148},  // 67     6  0.12%   12.54%   8.25%
  { 47616,    2,    2,  148},  // 68    11  0.11%   12.54%   9.09%
  { 52352,    1,    2,  148},  // 69     5  0.16%   12.54%   9.95%
  { 58240,    2,    2,  148},  // 70     9  0.03%   12.53%  11.25%
  { 65536,    1,    2,  150},  // 71     4  0.02%   12.53%  12.53%
  { 69888,    4,    2,  148},  // 72    15  0.03%   12.53%   6.64%
  { 74880,    2,    2,  148},  // 73     7  0.03%   12.53%   7.14%
  { 78592,    3,    2,  148},  // 74    10  0.07%   12.54%   4.96%
  { 87296,    1,    2,  148},  // 75     3  0.12%   12.54%  11.07%
  { 98304,    3,    2,  148},  // 76     8  0.01%   12.53%  12.61%
  {104832,    2,    2,  148},  // 77     5  0.03%   12.54%   6.64%
  {116480,    4,    2,  148},  // 78     9  0.03%   12.54%  11.11%
  {131072,    1,    2,  148},  // 79     2  0.02%   12.53%  12.53%
  {142976,    6,    2,  148},  // 80    11  0.01%   12.53%   9.08%
  {157184,    3,    2,  148},  // 81     5  0.07%   12.54%   9.94%
  {174720,    2,    2,  148},  // 82     3  0.03%   12.54%  11.16%
  {196608,    3,    2,  148},  // 83     4  0.01%   12.53%  12.53%
  {209664,    4,    2,  148},  // 84     5  0.03%   12.54%   6.64%
  {232960,    8,    2,  148},  // 85     9  0.03%   12.54%  11.11%
  {262144,    1,    2,  150},  // 86     1  0.02%    0.03%  12.53%
};
#elif TCMALLOC_PAGE_SHIFT == 12
static_assert(kMaxSize == 8192, "kMaxSize mismatch");
static constexpr SizeClassAssumptions Assumptions{
  .has_expanded_classes = false,
  .span_size = 48,
  .sampling_rate = 524288,
  .large_size = 1024,
  .large_size_alignment = 128,
};
static constexpr SizeClassInfo List[] = {
//                                         |    waste     |
//  bytes pages batch   cap    class  objs |fixed sampling|    inc
  {     0,    0,    0,    0},  //  0     0  0.00%    0.00%   0.00%
  {     8,    1,   32, 2905},  //  0   512  1.16%    0.92%   0.00%
  {    16,    1,   32, 2905},  //  1   256  1.16%    0.92% 100.00%
  {    32,    1,   32, 2905},  //  2   128  1.16%    0.92% 100.00%
  {    64,    1,   32, 2905},  //  3    64  1.16%    0.92% 100.00%
  {    80,    1,   32, 2905},  //  4    51  1.54%    0.92%  25.00%
  {    96,    1,   32, 1879},  //  5    42  2.70%    0.92%  20.00%
  {   112,    1,   32, 1491},  //  6    36  2.70%    0.92%  16.67%
  {   128,    1,   32, 1411},  //  7    32  1.16%    0.92%  14.29%
  {   144,    1,   32, 1143},  //  8    28  2.70%    0.92%  12.50%
  {   160,    2,   32, 1037},  //  9    51  0.97%    0.92%  11.11%
  {   176,    1,   32,  525},  // 10    23  2.32%    0.92%  10.00%
  {   192,    1,   32,  563},  // 11    21  2.70%    0.92%   9.09%
  {   208,    2,   32,  380},  // 12    39  1.55%    0.92%   8.33%
  {   224,    1,   32,  513},  // 13    18  2.70%    0.92%   7.69%
  {   240,    1,   32,  316},  // 14    17  1.54%    0.92%   7.14%
  {   256,    1,   32,  553},  // 15    16  1.16%    0.92%   6.67%
  {   272,    1,   32,  267},  // 16    15  1.54%    0.92%   6.25%
  {   288,    1,   32,  302},  // 17    14  2.70%    0.92%   5.88%
  {   304,    3,   32,  261},  // 18    40  1.43%    0.92%   5.56%
  {   336,    1,   32,  457},  // 19    12  2.70%    0.92%  10.53%
  {   384,    3,   32,  248},  // 20    32  0.39%    0.92%  14.29%
  {   416,    3,   32,  194},  // 21    29  2.20%    0.92%   8.33%
  {   448,    1,   32,  234},  // 22     9  2.70%    0.92%   7.69%
  {   512,    1,   32,  674},  // 23     8  1.16%    0.92%  14.29%
  {   528,    3,   32,  200},  // 24    23  1.56%    0.92%   3.12%
  {   624,    2,   32,  329},  // 25    13  1.55%    0.92%  18.18%
  {   768,    3,   32,  339},  // 26    16  0.39%    0.92%  23.08%
  {   816,    1,   32,  213},  // 27     5  1.54%    0.92%   6.25%
  {   896,    2,   32,  257},  // 28     9  2.14%    0.92%   9.80%
  {  1024,    1,   32,  864},  // 29     4  1.16%    0.92%  14.29%
  {  1152,    2,   32,  224},  // 30     7  2.14%    0.92%  12.50%
  {  1280,    5,   32,  186},  // 31    16  0.23%    0.92%  11.11%
  {  1408,   10,   32,  193},  // 32    29  0.43%    0.92%  10.00%
  {  1664,    9,   32,  216},  // 33    22  0.82%    0.92%  18.18%
  {  2048,    1,   32,  486},  // 34     2  1.16%    0.92%  23.08%
  {  2432,    3,   26,  225},  // 35     5  1.43%    0.92%  18.75%
  {  3072,    3,   21,  189},  // 36     4  0.39%    0.92%  26.32%
  {  3584,    7,   18,  187},  // 37     8  0.17%    0.92%  16.67%
  {  4096,    1,   16,  570},  // 38     1  1.16%    0.13%  14.29%
  {  4608,    9,   14,  227},  // 39     8  0.13%    1.70%  12.50%
  {  5120,    5,   12,  178},  // 40     4  0.23%    1.70%  11.11%
  {  6144,    3,   10,  187},  // 41     2  0.39%    1.70%  20.00%
  {  7168,    7,    9,  190},  // 42     4  0.17%    1.70%  16.67%
  {  8192,    2,    8,  247},  // 43     1  0.58%    0.13%  14.29%
};
#else
#error "Unsupported TCMALLOC_PAGE_SHIFT value!"
#endif
#endif
// clang-format on

static_assert(sizeof(List) / sizeof(List[0]) <= kNumBaseClasses);
extern constexpr SizeClasses kLowFragSizeClasses{List, Assumptions};

}  // namespace tcmalloc_internal
}  // namespace tcmalloc
GOOGLE_MALLOC_SECTION_END
