/**
 * Tests that sharding or moving a buckets collection missing the `timeseries` options fails.
 * Those inconsistent collections may be generated by implicit creation bugs (e.g. SERVER-87678).
 */

import {configureFailPoint} from "jstests/libs/fail_point_util.js";
import {FeatureFlagUtil} from "jstests/libs/feature_flag_util.js";
import {FixtureHelpers} from "jstests/libs/fixture_helpers.js";

// Attempting to track a buckets collection without timeseries options triggers a tassert.
// Disable the behavior of aborting on exit so that we can stop the sharded cluster normally.
TestData.testingDiagnosticsEnabled = false;

var st = new ShardingTest({shards: 2, mongos: 1, config: 1});

const db = st.s.getDB(jsTestName());
const coll = db.coll;
const bucketsColl = db.system.buckets.coll;

assert.commandWorked(
    st.s.adminCommand({enableSharding: db.getName(), primaryShard: st.shard0.shardName}));

if (FeatureFlagUtil.isPresentAndEnabled(db, "DisallowBucketCollectionWithoutTimeseriesOptions")) {
    configureFailPoint(st.rs0.getPrimary(), "skipCreateTimeseriesBucketsWithoutOptionsCheck");
}

assert.commandWorked(db.createCollection(bucketsColl.getName()));

// Attempting to shard the collection either directly or through the buckets namespace will fail.
assert.commandFailedWithCode(db.adminCommand({shardCollection: coll.getFullName(), key: {meta: 1}}),
                             [9934501 /* Binary 8.0 */, 6159000 /* Binary 7.0 */]);
assert.commandFailedWithCode(
    db.adminCommand({shardCollection: bucketsColl.getFullName(), key: {meta: 1}}), 5731501);

// Attempting to move the collection either directly or through the buckets namespace will fail.
if (FeatureFlagUtil.isPresentAndEnabled(db, "MoveCollection")) {
    assert.commandFailedWithCode(
        db.adminCommand({moveCollection: coll.getFullName(), toShard: st.shard1.shardName}),
        9934501);
    assert.commandFailedWithCode(
        db.adminCommand({moveCollection: bucketsColl.getFullName(), toShard: st.shard1.shardName}),
        9934501);
}

// Make sure none of the operations tracked the collection despite failing.
assert(!FixtureHelpers.isTracked(coll));
assert(!FixtureHelpers.isTracked(bucketsColl));

// The view can't be created as the buckets collection is incompatible.
assert.commandFailedWithCode(
    db.createCollection(coll.getName(), {timeseries: {timeField: 't', metaField: 'm'}}),
    ErrorCodes.NamespaceExists);

assert(bucketsColl.drop());

st.stop();
