/*
 *  Copyright (C) 2005-2018 Team Kodi
 *  This file is part of Kodi - https://kodi.tv
 *
 *  SPDX-License-Identifier: GPL-2.0-or-later
 *  See LICENSES/README.md for more information.
 */

#include "BinaryAddonManager.h"

#include "addons/addoninfo/AddonInfo.h"
#include "addons/binary-addons/AddonDll.h"
#include "addons/binary-addons/BinaryAddonBase.h"
#include "utils/log.h"

#include <mutex>

using namespace ADDON;

std::shared_ptr<CBinaryAddonBase> CBinaryAddonManager::GetAddonBase(
    const AddonInfoPtr& addonInfo,
    IAddonInstanceHandler* handler,
    std::shared_ptr<CAddonDll>& addon)
{
  std::unique_lock lock(m_critSection);

  std::shared_ptr<CBinaryAddonBase> addonBase;

  const auto& addonInstances = m_runningAddons.find(addonInfo->ID());
  if (addonInstances != m_runningAddons.end())
  {
    addonBase = addonInstances->second;
  }
  else
  {
    addonBase = std::make_shared<CBinaryAddonBase>(addonInfo);

    m_runningAddons.emplace(addonInfo->ID(), addonBase);
  }

  if (addonBase)
  {
    addon = addonBase->GetAddon(handler);
  }
  if (!addon)
  {
    CLog::LogF(LOGFATAL, "Unable to get add-on '{}'!", addonInfo->ID());
  }

  return addonBase;
}

void CBinaryAddonManager::ReleaseAddonBase(const std::shared_ptr<CBinaryAddonBase>& addonBase,
                                           IAddonInstanceHandler* handler)
{
  const auto& addon = m_runningAddons.find(addonBase->ID());
  if (addon == m_runningAddons.end())
    return;

  addonBase->ReleaseAddon(handler);

  if (addonBase->UsedInstanceCount() > 0)
    return;

  m_runningAddons.erase(addon);
}

std::shared_ptr<CBinaryAddonBase> CBinaryAddonManager::GetRunningAddonBase(
    const std::string& addonId) const
{
  std::unique_lock lock(m_critSection);

  const auto& addonInstances = m_runningAddons.find(addonId);
  if (addonInstances != m_runningAddons.end())
    return addonInstances->second;

  return nullptr;
}

AddonPtr CBinaryAddonManager::GetRunningAddon(const std::string& addonId) const
{
  std::unique_lock lock(m_critSection);

  const std::shared_ptr<CBinaryAddonBase> base = GetRunningAddonBase(addonId);
  if (base)
    return base->GetActiveAddon();

  return nullptr;
}
