/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Copyright (c) 1995, 2004 IBM Corporation. All rights reserved.             */
/* Copyright (c) 2005-2014 Rexx Language Association. All rights reserved.    */
/*                                                                            */
/* This program and the accompanying materials are made available under       */
/* the terms of the Common Public License v1.0 which accompanies this         */
/* distribution. A copy is also available at the following address:           */
/* https://www.oorexx.org/license.html                                        */
/*                                                                            */
/* Redistribution and use in source and binary forms, with or                 */
/* without modification, are permitted provided that the following            */
/* conditions are met:                                                        */
/*                                                                            */
/* Redistributions of source code must retain the above copyright             */
/* notice, this list of conditions and the following disclaimer.              */
/* Redistributions in binary form must reproduce the above copyright          */
/* notice, this list of conditions and the following disclaimer in            */
/* the documentation and/or other materials provided with the distribution.   */
/*                                                                            */
/* Neither the name of Rexx Language Association nor the names                */
/* of its contributors may be used to endorse or promote products             */
/* derived from this software without specific prior written permission.      */
/*                                                                            */
/* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS        */
/* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT          */
/* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS          */
/* FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT   */
/* OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,      */
/* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED   */
/* TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,        */
/* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY     */
/* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING    */
/* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS         */
/* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.               */
/*                                                                            */
/*----------------------------------------------------------------------------*/
/******************************************************************************/
/* REXX Kernel                                               BufferClass.hpp  */
/*                                                                            */
/* Primitive Buffer Class Definitions                                         */
/*                                                                            */
/******************************************************************************/
#ifndef Included_BufferClass
#define Included_BufferClass

#include <string.h>

class BufferClassBase : public RexxObject
{
public:

    inline BufferClassBase() {;};

    inline size_t getDataLength() { return this->dataLength; }
    inline size_t getBufferSize() { return this->bufferSize; }
    inline void   setDataLength(size_t l) { this->dataLength = l; }
    virtual char *getData() = 0;

    inline void copyData(size_t offset, const char *string, size_t l) { memmove(this->getData() + offset, string, l); }
    inline void copyData(CONSTRXSTRING &r) { copyData(0, r.strptr, r.strlength); }
    inline void copyData(RXSTRING &r) { copyData(0, r.strptr, r.strlength); }
    inline void openGap(size_t offset, size_t _size, size_t tailSize)
    {
        memmove(getData() + offset + _size, getData() + offset, tailSize);
    }
    inline void closeGap(size_t offset, size_t _size, size_t tailSize)
    {
        memmove(getData() + offset, getData() + offset + _size, tailSize);
    }

    inline void adjustGap(size_t offset, size_t _size, size_t _newSize, size_t _dataLength)
    {
        memmove(getData() + offset + _newSize, getData() + offset + _size, _dataLength - (offset + _size));
    }
    inline void setData(size_t offset, char character, size_t l)
    {
        memset(getData() + offset, character, l);
    }
protected:

    // the following field is padding required to get the start of the of the
    // buffer data aligned on an object grain boundary.  Since we unflatten saved programs
    // by reducing the size of the surrounding buffer to reveal the exposed data, we need
    // to ensure appropriate data alignment.  Fortunately, because the sizes of all of the
    // fields doubles when going to 64-bit, this single padding item is sufficient to
    // get everything lined up on all platforms.
    size_t reserved;
    size_t bufferSize;                  // size of the buffer
    size_t dataLength;                  // length of the buffer data (freqently the same)
};


class BufferClass : public BufferClassBase
{
public:
    void *operator new(size_t, size_t);
    inline void  operator delete(void *) { ; }

    inline BufferClass(size_t length)
    {
        // initialize the length values
        bufferSize = length;
        dataLength = length;
        // buffers have no references.
        setHasNoReferences();
    }

    inline BufferClass(RESTORETYPE restoreType) { ; }

    BufferClass *expand(size_t);
    RexxObject *newRexx(RexxObject **args, size_t argc);
    char *getData() override { return data; }

    static void createInstance();

    static RexxClass *classInstance;   // singleton class instance

protected:
    char data[4];                       /* actual data length                */
};


inline BufferClass *new_buffer(size_t s) { return new (s) BufferClass(s); }
inline BufferClass *new_buffer(CONSTRXSTRING &r)
{
    BufferClass *b = new_buffer(r.strlength);
    b->copyData(r);
    return b;
}

inline BufferClass *new_buffer(RXSTRING &r)
{
    BufferClass *b = new_buffer(r.strlength);
    b->copyData(r);
    return b;
}

inline BufferClass *new_buffer(const char *data, size_t length)
{
    BufferClass *b = new_buffer(length);
    b->copyData(0, data, length);
    return b;
}
#endif
