//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
/// \file optical/LXe/src/LXePMTSD.cc
/// \brief Implementation of the LXePMTSD class
//
//
#include "LXePMTSD.hh"

#include "LXeDetectorConstruction.hh"
#include "LXePMTHit.hh"
#include "LXeUserTrackInformation.hh"

#include "G4LogicalVolume.hh"
#include "G4ParticleDefinition.hh"
#include "G4ParticleTypes.hh"
#include "G4SDManager.hh"
#include "G4Step.hh"
#include "G4TouchableHistory.hh"
#include "G4Track.hh"
#include "G4VPhysicalVolume.hh"
#include "G4VTouchable.hh"
#include "G4ios.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

LXePMTSD::LXePMTSD(G4String name) : G4VSensitiveDetector(name)
{
  collectionName.insert("pmtHitCollection");
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

LXePMTSD::~LXePMTSD()
{
  delete fPMTPositionsX;
  delete fPMTPositionsY;
  delete fPMTPositionsZ;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void LXePMTSD::SetPmtPositions(const std::vector<G4ThreeVector>& positions)
{
  for (size_t i = 0; i < positions.size(); ++i) {
    if (fPMTPositionsX) fPMTPositionsX->push_back(positions[i].x());
    if (fPMTPositionsY) fPMTPositionsY->push_back(positions[i].y());
    if (fPMTPositionsZ) fPMTPositionsZ->push_back(positions[i].z());
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void LXePMTSD::Initialize(G4HCofThisEvent* hitsCE)
{
  fPMTHitCollection = new LXePMTHitsCollection(SensitiveDetectorName, collectionName[0]);

  if (fHitCID < 0) {
    fHitCID = G4SDManager::GetSDMpointer()->GetCollectionID(fPMTHitCollection);
  }
  hitsCE->AddHitsCollection(fHitCID, fPMTHitCollection);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4bool LXePMTSD::ProcessHits(G4Step*, G4TouchableHistory*)
{
  return false;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

// Generates a hit and uses the postStepPoint's mother volume replica number
// PostStepPoint because the hit is generated manually when the photon is
// absorbed by the photocathode

G4bool LXePMTSD::ProcessHits_boundary(const G4Step* aStep, G4TouchableHistory*)
{
  // need to know if this is an optical photon
  if (aStep->GetTrack()->GetDefinition() != G4OpticalPhoton::OpticalPhotonDefinition())
    return false;

  // User replica number 1 since photocathode is a daughter volume
  // to the pmt which was replicated
  G4int pmtNumber = aStep->GetPostStepPoint()->GetTouchable()->GetReplicaNumber(1);
  G4VPhysicalVolume* physVol = aStep->GetPostStepPoint()->GetTouchable()->GetVolume(1);

  // Find the correct hit collection
  size_t n = fPMTHitCollection->entries();
  LXePMTHit* hit = nullptr;
  for (size_t i = 0; i < n; ++i) {
    if ((*fPMTHitCollection)[i]->GetPMTNumber() == pmtNumber) {
      hit = (*fPMTHitCollection)[i];
      break;
    }
  }

  if (hit == nullptr) {  // this pmt wasn't previously hit in this event
    hit = new LXePMTHit();  // so create new hit
    hit->SetPMTNumber(pmtNumber);
    hit->SetPMTPhysVol(physVol);
    fPMTHitCollection->insert(hit);
    hit->SetPMTPos((*fPMTPositionsX)[pmtNumber], (*fPMTPositionsY)[pmtNumber],
                   (*fPMTPositionsZ)[pmtNumber]);
  }

  hit->IncPhotonCount();  // increment hit for the selected pmt

  if (!LXeDetectorConstruction::GetSphereOn()) {
    hit->SetDrawit(true);
    // If the sphere is disabled then this hit is automaticaly drawn
  }
  else {  // sphere enabled
    auto trackInfo = (LXeUserTrackInformation*)aStep->GetTrack()->GetUserInformation();
    if (trackInfo->GetTrackStatus() & hitSphere)
      // only draw this hit if the photon has hit the sphere first
      hit->SetDrawit(true);
  }

  return true;
}
