// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.openqa.selenium.bidi.module;

import java.io.StringReader;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.bidi.BiDi;
import org.openqa.selenium.bidi.Event;
import org.openqa.selenium.bidi.HasBiDi;
import org.openqa.selenium.bidi.browsingcontext.BrowsingContextInfo;
import org.openqa.selenium.bidi.browsingcontext.DownloadEnded;
import org.openqa.selenium.bidi.browsingcontext.DownloadInfo;
import org.openqa.selenium.bidi.browsingcontext.HistoryUpdated;
import org.openqa.selenium.bidi.browsingcontext.NavigationInfo;
import org.openqa.selenium.bidi.browsingcontext.UserPromptClosed;
import org.openqa.selenium.bidi.browsingcontext.UserPromptOpened;
import org.openqa.selenium.internal.Require;
import org.openqa.selenium.json.Json;
import org.openqa.selenium.json.JsonInput;

public class BrowsingContextInspector implements AutoCloseable {

  private static final Json JSON = new Json();

  private final Set<String> browsingContextIds;

  private final BiDi bidi;

  private static final Function<Map<String, Object>, BrowsingContextInfo>
      browsingContextInfoMapper =
          params -> {
            try (StringReader reader = new StringReader(JSON.toJson(params));
                JsonInput input = JSON.newInput(reader)) {
              return input.readNonNull(BrowsingContextInfo.class);
            }
          };

  private static final Function<Map<String, Object>, NavigationInfo> navigationInfoMapper =
      params -> {
        try (StringReader reader = new StringReader(JSON.toJson(params));
            JsonInput input = JSON.newInput(reader)) {
          return input.readNonNull(NavigationInfo.class);
        }
      };

  private static final Function<Map<String, Object>, DownloadInfo> downloadWillBeginMapper =
      params -> {
        try (StringReader reader = new StringReader(JSON.toJson(params));
            JsonInput input = JSON.newInput(reader)) {
          return input.readNonNull(DownloadInfo.class);
        }
      };

  private static final Function<Map<String, Object>, DownloadEnded> downloadEndMapper =
      params -> {
        try (StringReader reader = new StringReader(JSON.toJson(params));
            JsonInput input = JSON.newInput(reader)) {
          return input.readNonNull(DownloadEnded.class);
        }
      };

  private final Event<BrowsingContextInfo> browsingContextCreated =
      new Event<>("browsingContext.contextCreated", browsingContextInfoMapper);

  private static final Event<BrowsingContextInfo> browsingContextDestroyed =
      new Event<>("browsingContext.contextDestroyed", browsingContextInfoMapper);

  private static final Event<UserPromptClosed> userPromptClosed =
      new Event<>(
          "browsingContext.userPromptClosed",
          params -> {
            try (StringReader reader = new StringReader(JSON.toJson(params));
                JsonInput input = JSON.newInput(reader)) {
              return input.readNonNull(UserPromptClosed.class);
            }
          });

  private final Set<Event<NavigationInfo>> navigationEventSet = new HashSet<>();

  private static final Event<DownloadInfo> downloadWillBeginEvent =
      new Event<>("browsingContext.downloadWillBegin", downloadWillBeginMapper);

  private static final Event<DownloadEnded> downloadEndEvent =
      new Event<>("browsingContext.downloadEnd", downloadEndMapper);

  private static final Event<UserPromptOpened> userPromptOpened =
      new Event<>(
          "browsingContext.userPromptOpened",
          params -> {
            try (StringReader reader = new StringReader(JSON.toJson(params));
                JsonInput input = JSON.newInput(reader)) {
              return input.readNonNull(UserPromptOpened.class);
            }
          });

  private static final Event<HistoryUpdated> historyUpdated =
      new Event<>(
          "browsingContext.historyUpdated",
          params -> {
            try (StringReader reader = new StringReader(JSON.toJson(params));
                JsonInput input = JSON.newInput(reader)) {
              return input.readNonNull(HistoryUpdated.class);
            }
          });

  public BrowsingContextInspector(WebDriver driver) {
    this(new HashSet<>(), driver);
  }

  public BrowsingContextInspector(String browsingContextId, WebDriver driver) {
    this(Collections.singleton(Require.nonNull("Browsing context id", browsingContextId)), driver);
  }

  public BrowsingContextInspector(Set<String> browsingContextIds, WebDriver driver) {
    Require.nonNull("WebDriver", driver);
    Require.nonNull("Browsing context id list", browsingContextIds);

    if (!(driver instanceof HasBiDi)) {
      throw new IllegalArgumentException("WebDriver instance must support BiDi protocol");
    }

    this.bidi = ((HasBiDi) driver).getBiDi();
    this.browsingContextIds = browsingContextIds;
  }

  public void onBrowsingContextCreated(Consumer<BrowsingContextInfo> consumer) {
    if (browsingContextIds.isEmpty()) {
      this.bidi.addListener(browsingContextCreated, consumer);
    } else {
      this.bidi.addListener(browsingContextIds, browsingContextCreated, consumer);
    }
  }

  public void onBrowsingContextDestroyed(Consumer<BrowsingContextInfo> consumer) {
    if (browsingContextIds.isEmpty()) {
      this.bidi.addListener(browsingContextDestroyed, consumer);
    } else {
      this.bidi.addListener(browsingContextIds, browsingContextDestroyed, consumer);
    }
  }

  public void onNavigationStarted(Consumer<NavigationInfo> consumer) {
    addNavigationEventListener("browsingContext.navigationStarted", consumer);
  }

  public void onFragmentNavigated(Consumer<NavigationInfo> consumer) {
    addNavigationEventListener("browsingContext.fragmentNavigated", consumer);
  }

  public void onDomContentLoaded(Consumer<NavigationInfo> consumer) {
    addNavigationEventListener("browsingContext.domContentLoaded", consumer);
  }

  public void onBrowsingContextLoaded(Consumer<NavigationInfo> consumer) {
    addNavigationEventListener("browsingContext.load", consumer);
  }

  public void onDownloadWillBegin(Consumer<DownloadInfo> consumer) {
    if (browsingContextIds.isEmpty()) {
      this.bidi.addListener(downloadWillBeginEvent, consumer);
    } else {
      this.bidi.addListener(browsingContextIds, downloadWillBeginEvent, consumer);
    }
  }

  public void onDownloadEnd(Consumer<DownloadEnded> consumer) {
    if (browsingContextIds.isEmpty()) {
      this.bidi.addListener(downloadEndEvent, consumer);
    } else {
      this.bidi.addListener(browsingContextIds, downloadEndEvent, consumer);
    }
  }

  public void onNavigationAborted(Consumer<NavigationInfo> consumer) {
    addNavigationEventListener("browsingContext.navigationAborted", consumer);
  }

  public void onNavigationFailed(Consumer<NavigationInfo> consumer) {
    addNavigationEventListener("browsingContext.navigationFailed", consumer);
  }

  public void onNavigationCommitted(Consumer<NavigationInfo> consumer) {
    addNavigationEventListener("browsingContext.navigationCommitted", consumer);
  }

  public void onUserPromptClosed(Consumer<UserPromptClosed> consumer) {
    if (browsingContextIds.isEmpty()) {
      this.bidi.addListener(userPromptClosed, consumer);
    } else {
      this.bidi.addListener(browsingContextIds, userPromptClosed, consumer);
    }
  }

  public void onUserPromptOpened(Consumer<UserPromptOpened> consumer) {
    if (browsingContextIds.isEmpty()) {
      this.bidi.addListener(userPromptOpened, consumer);
    } else {
      this.bidi.addListener(browsingContextIds, userPromptOpened, consumer);
    }
  }

  public void onHistoryUpdated(Consumer<HistoryUpdated> consumer) {
    if (browsingContextIds.isEmpty()) {
      this.bidi.addListener(historyUpdated, consumer);
    } else {
      this.bidi.addListener(browsingContextIds, historyUpdated, consumer);
    }
  }

  private void addNavigationEventListener(String name, Consumer<NavigationInfo> consumer) {
    Event<NavigationInfo> navigationEvent = new Event<>(name, navigationInfoMapper);

    navigationEventSet.add(navigationEvent);

    if (browsingContextIds.isEmpty()) {
      this.bidi.addListener(navigationEvent, consumer);
    } else {
      this.bidi.addListener(browsingContextIds, navigationEvent, consumer);
    }
  }

  @Override
  public void close() {
    this.bidi.clearListener(browsingContextCreated);
    this.bidi.clearListener(browsingContextDestroyed);
    this.bidi.clearListener(userPromptOpened);
    this.bidi.clearListener(userPromptClosed);
    this.bidi.clearListener(historyUpdated);
    this.bidi.clearListener(downloadWillBeginEvent);
    this.bidi.clearListener(downloadEndEvent);

    navigationEventSet.forEach(this.bidi::clearListener);
  }
}
