// <copyright file="BrowsingContextTests.cs" company="Selenium Committers">
// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
// </copyright>

using System.Linq;
using System.Threading.Tasks;
using NUnit.Framework;

namespace OpenQA.Selenium.BiDi.BrowsingContext;

internal class BrowsingContextTests : BiDiTestFixture
{
    [Test]
    public async Task CanCreateNewTab()
    {
        var tab = await bidi.BrowsingContext.CreateAsync(ContextType.Tab);

        Assert.That(tab, Is.Not.Null);
    }

    [Test]
    public async Task CanCreateNewTabWithReferencedContext()
    {
        var tab = await bidi.BrowsingContext.CreateAsync(ContextType.Tab, new()
        {
            ReferenceContext = context
        });

        Assert.That(tab, Is.Not.Null);
    }

    [Test]
    public async Task CanCreateNewWindow()
    {
        var window = await bidi.BrowsingContext.CreateAsync(ContextType.Window);

        Assert.That(window, Is.Not.Null);
    }

    [Test]
    public async Task CanCreateNewWindowWithReferencedContext()
    {
        var window = await bidi.BrowsingContext.CreateAsync(ContextType.Window, new()
        {
            ReferenceContext = context
        });

        Assert.That(window, Is.Not.Null);
    }

    [Test]
    public async Task CanCreateContextWithAllParameters()
    {
        var userContext = await bidi.Browser.CreateUserContextAsync();

        var window = await bidi.BrowsingContext.CreateAsync(ContextType.Window, new()
        {
            ReferenceContext = context,
            UserContext = userContext.UserContext,
            Background = true
        });

        Assert.That(window, Is.Not.Null);
    }

    [Test]
    public async Task CanNavigateToUrl()
    {
        var info = await context.NavigateAsync(UrlBuilder.WhereIs("/bidi/logEntryAdded.html"));

        Assert.That(info.Url, Does.Contain("/bidi/logEntryAdded.html"));
    }

    [Test]
    public async Task CanNavigateToUrlWithReadinessState()
    {
        var info = await context.NavigateAsync(UrlBuilder.WhereIs("/bidi/logEntryAdded.html"), new()
        {
            Wait = ReadinessState.Complete
        });

        Assert.That(info, Is.Not.Null);
        Assert.That(info.Url, Does.Contain("/bidi/logEntryAdded.html"));
    }

    [Test]
    public async Task CanGetTreeWithChild()
    {
        await context.NavigateAsync(UrlBuilder.WhereIs("iframes.html"), new() { Wait = ReadinessState.Complete });

        var tree = await context.GetTreeAsync();

        Assert.That(tree.Contexts, Has.Count.EqualTo(1));
        Assert.That(tree.Contexts[0].Context, Is.EqualTo(context));
        Assert.That(tree.Contexts[0].Children, Has.Count.EqualTo(1));
        Assert.That(tree.Contexts[0].Children[0].Url, Does.Contain("formPage.html"));
    }

    [Test]
    public async Task CanGetTreeWithDepth()
    {
        await context.NavigateAsync(UrlBuilder.WhereIs("iframes.html"), new() { Wait = ReadinessState.Complete });

        var tree = await context.GetTreeAsync(new() { MaxDepth = 0 });

        Assert.That(tree.Contexts, Has.Count.EqualTo(1));
        Assert.That(tree.Contexts[0].Context, Is.EqualTo(context));
        Assert.That(tree.Contexts[0].Children, Is.Null);
    }

    [Test]
    public async Task CanGetTreeTopLevel()
    {
        var window1 = await bidi.BrowsingContext.CreateAsync(ContextType.Window);
        var window2 = await bidi.BrowsingContext.CreateAsync(ContextType.Window);

        var tree = await bidi.BrowsingContext.GetTreeAsync();

        Assert.That(tree.Contexts, Has.Count.GreaterThanOrEqualTo(2));
    }

    [Test]
    public async Task CanCloseWindow()
    {
        var window = await bidi.BrowsingContext.CreateAsync(ContextType.Window);

        await window.Context.CloseAsync();

        var tree = await bidi.BrowsingContext.GetTreeAsync();

        Assert.That(tree.Contexts.Select(i => i.Context), Does.Not.Contain(window));
    }

    [Test]
    public async Task CanCloseTab()
    {
        var tab = await bidi.BrowsingContext.CreateAsync(ContextType.Tab);

        await tab.Context.CloseAsync();

        var tree = await bidi.BrowsingContext.GetTreeAsync();

        Assert.That(tree.Contexts.Select(i => i.Context), Does.Not.Contain(tab));
    }

    [Test]
    public async Task CanActivate()
    {
        await context.ActivateAsync();

        // TODO: Add assertion when https://w3c.github.io/webdriver-bidi/#type-browser-ClientWindowInfo is implemented
    }

    [Test]
    public async Task CanReload()
    {
        string url = UrlBuilder.WhereIs("/bidi/logEntryAdded.html");

        await context.NavigateAsync(url, new() { Wait = ReadinessState.Complete });

        var info = await context.ReloadAsync();

        Assert.That(info, Is.Not.Null);
        Assert.That(info.Url, Does.Contain("/bidi/logEntryAdded.html"));
    }

    [Test]
    public async Task CanReloadWithReadinessState()
    {
        string url = UrlBuilder.WhereIs("/bidi/logEntryAdded.html");

        await context.NavigateAsync(url, new() { Wait = ReadinessState.Complete });

        var info = await context.ReloadAsync(new()
        {
            Wait = ReadinessState.Complete
        });

        Assert.That(info, Is.Not.Null);
        Assert.That(info.Url, Does.Contain("/bidi/logEntryAdded.html"));
    }

    [Test]
    public async Task CanHandleUserPrompt()
    {
        await context.NavigateAsync(UrlBuilder.WhereIs("alerts.html"), new() { Wait = ReadinessState.Complete });

        driver.FindElement(By.Id("alert")).Click();

        await context.HandleUserPromptAsync();
    }

    [Test]
    public async Task CanAcceptUserPrompt()
    {
        await context.NavigateAsync(UrlBuilder.WhereIs("alerts.html"), new() { Wait = ReadinessState.Complete });

        driver.FindElement(By.Id("alert")).Click();

        await context.HandleUserPromptAsync(new()
        {
            Accept = true
        });
    }

    [Test]
    public async Task CanDismissUserPrompt()
    {
        await context.NavigateAsync(UrlBuilder.WhereIs("alerts.html"), new() { Wait = ReadinessState.Complete });

        driver.FindElement(By.Id("alert")).Click();

        await context.HandleUserPromptAsync(new()
        {
            Accept = false
        });
    }

    [Test]
    public async Task CanPassUserTextToPrompt()
    {
        await context.NavigateAsync(UrlBuilder.WhereIs("alerts.html"), new() { Wait = ReadinessState.Complete });

        driver.FindElement(By.Id("alert")).Click();

        await context.HandleUserPromptAsync(new()
        {
            UserText = "Selenium automates browsers"
        });
    }

    [Test]
    public async Task CanCaptureScreenshot()
    {
        var screenshot = await context.CaptureScreenshotAsync();

        Assert.That(screenshot, Is.Not.Null);
        Assert.That(screenshot.Data.Length, Is.Not.Zero);
    }

    [Test]
    public async Task CanCaptureScreenshotWithParameters()
    {
        var screenshot = await context.CaptureScreenshotAsync(new()
        {
            Origin = ScreenshotOrigin.Document,
            Clip = new BoxClipRectangle(5, 5, 10, 10)
        });

        Assert.That(screenshot, Is.Not.Null);
        Assert.That(screenshot.Data.Length, Is.Not.Zero);
    }

    [Test]
    public async Task CanCaptureScreenshotOfViewport()
    {
        var screenshot = await context.CaptureScreenshotAsync(new()
        {
            Origin = ScreenshotOrigin.Viewport,
            Clip = new BoxClipRectangle(5, 5, 10, 10)
        });

        Assert.That(screenshot, Is.Not.Null);
        Assert.That(screenshot.Data.Length, Is.Not.Zero);
    }

    [Test]
    public async Task CanCaptureScreenshotOfElement()
    {
        await context.NavigateAsync(UrlBuilder.WhereIs("formPage.html"), new() { Wait = ReadinessState.Complete });

        var nodesResult = await context.LocateNodesAsync(new CssLocator("#checky"));

        var screenshot = await context.CaptureScreenshotAsync(new()
        {
            Clip = new ElementClipRectangle(nodesResult.Nodes[0])
        });

        Assert.That(screenshot, Is.Not.Null);
        Assert.That(screenshot.Data.Length, Is.Not.Zero);
    }

    [Test]
    public async Task CanSetViewport()
    {
        Task<int> GetWidthAsync() => context.Script.EvaluateAsync<int>("window.innerWidth", false);
        Task<int> GetHeightAsync() => context.Script.EvaluateAsync<int>("window.innerHeight", false);

        var defaultWidth = await GetWidthAsync();
        var defaultHeight = await GetHeightAsync();

        await context.SetViewportAsync(new() { Viewport = new Viewport(250, 300) });

        Assert.That(await GetWidthAsync(), Is.EqualTo(250));
        Assert.That(await GetHeightAsync(), Is.EqualTo(300));

        await context.SetViewportAsync(new() { Viewport = new Viewport(250, 300) });
        await context.SetViewportAsync(); // Sends nothing

        Assert.That(await GetWidthAsync(), Is.EqualTo(250));
        Assert.That(await GetHeightAsync(), Is.EqualTo(300));

        await context.SetViewportAsync(new() { Viewport = new Viewport(250, 300) });
        await context.SetViewportAsync(new() { Viewport = default }); // Sends nothing

        Assert.That(await GetWidthAsync(), Is.EqualTo(250));
        Assert.That(await GetHeightAsync(), Is.EqualTo(300));

        await context.SetViewportAsync(new() { Viewport = new Viewport(250, 300) });
        await context.SetViewportAsync(new() { Viewport = default(Viewport?) }); // Explicitly sends "null", resetting to default

        Assert.That(await GetWidthAsync(), Is.EqualTo(defaultWidth));
        Assert.That(await GetHeightAsync(), Is.EqualTo(defaultHeight));
    }

    [Test]
    public async Task CanSetViewportWithDevicePixelRatio()
    {
        await context.SetViewportAsync(new() { Viewport = new Viewport(250, 300), DevicePixelRatio = 5 });
    }

    [Test]
    public async Task CanPrintPage()
    {
        var pdf = await context.PrintAsync();

        Assert.That(pdf, Is.Not.Null);
        Assert.That(pdf.Data.Length, Is.Not.Zero);
    }
}
