/*
 * MSMPEG4 backend for encoder and decoder
 * copyright (c) 2001 Fabrice Bellard
 * copyright (c) 2002-2004 Michael Niedermayer <michaelni@gmx.at>
 *
 * msmpeg4v1 & v2 stuff by Michael Niedermayer <michaelni@gmx.at>
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file
 * MSMPEG4 data tables.
 */

#include "h263data.h"
#include "mpeg4videodata.h"
#include "msmpeg4data.h"
#include "rl.h"

uint32_t ff_v2_dc_lum_table[512][2];
uint32_t ff_v2_dc_chroma_table[512][2];

/* non intra picture macroblock coded block pattern + mb type */
const uint32_t ff_table_mb_non_intra[128][2] = {
    {   0x40,  7 }, {   0x13c9, 13 }, {   0x9fd, 12 }, {   0x1fc, 15 },
    {  0x9fc, 12 }, {    0xa83, 18 }, { 0x12d34, 17 }, {  0x83bc, 16 },
    {  0x83a, 12 }, {    0x7f8, 17 }, {   0x3fd, 16 }, {   0x3ff, 16 },
    {   0x79, 13 }, {    0xa82, 18 }, {  0x969d, 16 }, {   0x2a4, 16 },
    {  0x978, 12 }, {    0x543, 17 }, {  0x41df, 15 }, {   0x7f9, 17 },
    { 0x12f3, 13 }, {  0x25a6b, 18 }, { 0x25ef9, 18 }, {   0x3fa, 16 },
    { 0x20ee, 14 }, {  0x969ab, 20 }, {  0x969c, 16 }, { 0x25ef8, 18 },
    { 0x12d2, 13 }, {    0xa85, 18 }, {  0x969e, 16 }, {  0x4bc8, 15 },
    {   0x3d, 12 }, {  0x12f7f, 17 }, {   0x2a2, 16 }, {  0x969f, 16 },
    { 0x25ee, 14 }, { 0x12d355, 21 }, { 0x12f7d, 17 }, { 0x12f7e, 17 },
    {  0x9e5, 12 }, {    0xa81, 18 }, { 0x4b4d4, 19 }, {  0x83bd, 16 },
    {   0x78, 13 }, {   0x969b, 16 }, {   0x3fe, 16 }, {   0x2a5, 16 },
    {   0x7e, 13 }, {    0xa80, 18 }, {   0x2a3, 16 }, {   0x3fb, 16 },
    { 0x1076, 13 }, {    0xa84, 18 }, {   0x153, 15 }, {  0x4bc9, 15 },
    {   0x55, 13 }, { 0x12d354, 21 }, {  0x4bde, 15 }, {  0x25e5, 14 },
    {  0x25b, 10 }, {   0x4b4c, 15 }, {   0x96b, 12 }, {   0x96a, 12 },
    {    0x1,  2 }, {      0x0,  7 }, {    0x26,  6 }, {   0x12b,  9 },
    {    0x7,  3 }, {    0x20f, 10 }, {     0x4,  9 }, {    0x28, 12 },
    {    0x6,  3 }, {    0x20a, 10 }, {   0x128,  9 }, {    0x2b, 12 },
    {   0x11,  5 }, {     0x1b, 11 }, {   0x13a,  9 }, {   0x4ff, 11 },
    {    0x3,  4 }, {    0x277, 10 }, {   0x106,  9 }, {   0x839, 12 },
    {    0xb,  4 }, {    0x27b, 10 }, {   0x12c,  9 }, {   0x4bf, 11 },
    {    0x9,  6 }, {     0x35, 12 }, {   0x27e, 10 }, {  0x13c8, 13 },
    {    0x1,  6 }, {    0x4aa, 11 }, {   0x208, 10 }, {    0x29, 12 },
    {    0x1,  4 }, {    0x254, 10 }, {   0x12e,  9 }, {   0x838, 12 },
    {   0x24,  6 }, {    0x4f3, 11 }, {   0x276, 10 }, {  0x12f6, 13 },
    {    0x1,  5 }, {    0x27a, 10 }, {   0x13e,  9 }, {    0x3e, 12 },
    {    0x8,  6 }, {    0x413, 11 }, {     0xc, 10 }, {   0x4be, 11 },
    {   0x14,  5 }, {    0x412, 11 }, {   0x253, 10 }, {   0x97a, 12 },
    {   0x21,  6 }, {    0x4ab, 11 }, {   0x20b, 10 }, {    0x34, 12 },
    {   0x15,  5 }, {    0x278, 10 }, {   0x252, 10 }, {   0x968, 12 },
    {    0x5,  5 }, {      0xb, 10 }, {    0x9c,  8 }, {     0xe, 10 },
};

/* vlc table 0, for intra luma */

static const uint16_t table0_vlc[133][2] = {
    {   0x1,  2 }, {   0x6,  3 }, {   0xf,  4 }, {  0x16,  5 },
    {  0x20,  6 }, {  0x18,  7 }, {   0x8,  8 }, {  0x9a,  8 },
    {  0x56,  9 }, { 0x13e,  9 }, {  0xf0, 10 }, { 0x3a5, 10 },
    {  0x77, 11 }, { 0x1ef, 11 }, {  0x9a, 12 }, {  0x5d, 13 },
    {   0x1,  4 }, {  0x11,  5 }, {   0x2,  7 }, {   0xb,  8 },
    {  0x12,  9 }, { 0x1d6,  9 }, { 0x27e, 10 }, { 0x191, 11 },
    {  0xea, 12 }, { 0x3dc, 12 }, { 0x13b, 13 }, {   0x4,  5 },
    {  0x14,  7 }, {  0x9e,  8 }, {   0x9, 10 }, { 0x1ac, 11 },
    { 0x1e2, 11 }, { 0x3ca, 12 }, {  0x5f, 13 }, {  0x17,  5 },
    {  0x4e,  7 }, {  0x5e,  9 }, {  0xf3, 10 }, { 0x1ad, 11 },
    {  0xec, 12 }, { 0x5f0, 13 }, {   0xe,  6 }, {  0xe1,  8 },
    { 0x3a4, 10 }, {  0x9c, 12 }, { 0x13d, 13 }, {  0x3b,  6 },
    {  0x1c,  9 }, {  0x14, 11 }, { 0x9be, 12 }, {   0x6,  7 },
    {  0x7a,  9 }, { 0x190, 11 }, { 0x137, 13 }, {  0x1b,  7 },
    {   0x8, 10 }, { 0x75c, 11 }, {  0x71,  7 }, {  0xd7, 10 },
    { 0x9bf, 12 }, {   0x7,  8 }, {  0xaf, 10 }, { 0x4cc, 11 },
    {  0x34,  8 }, { 0x265, 10 }, {  0x9f, 12 }, {  0xe0,  8 },
    {  0x16, 11 }, { 0x327, 12 }, {  0x15,  9 }, { 0x17d, 11 },
    { 0xebb, 12 }, {  0x14,  9 }, {  0xf6, 10 }, { 0x1e4, 11 },
    {  0xcb, 10 }, { 0x99d, 12 }, {  0xca, 10 }, { 0x2fc, 12 },
    { 0x17f, 11 }, { 0x4cd, 11 }, { 0x2fd, 12 }, { 0x4fe, 11 },
    { 0x13a, 13 }, {   0xa,  4 }, {  0x42,  7 }, { 0x1d3,  9 },
    { 0x4dd, 11 }, {  0x12,  5 }, {  0xe8,  8 }, {  0x4c, 11 },
    { 0x136, 13 }, {  0x39,  6 }, { 0x264, 10 }, { 0xeba, 12 },
    {   0x0,  7 }, {  0xae, 10 }, { 0x99c, 12 }, {  0x1f,  7 },
    { 0x4de, 11 }, {  0x43,  7 }, { 0x4dc, 11 }, {   0x3,  8 },
    { 0x3cb, 12 }, {   0x6,  8 }, { 0x99e, 12 }, {  0x2a,  8 },
    { 0x5f1, 13 }, {   0xf,  8 }, { 0x9fe, 12 }, {  0x33,  8 },
    { 0x9ff, 12 }, {  0x98,  8 }, { 0x99f, 12 }, {  0xea,  8 },
    { 0x13c, 13 }, {  0x2e,  8 }, { 0x192, 11 }, { 0x136,  9 },
    {  0x6a,  9 }, {  0x15, 11 }, { 0x3af, 10 }, { 0x1e3, 11 },
    {  0x74, 11 }, {  0xeb, 12 }, { 0x2f9, 12 }, {  0x5c, 13 },
    {  0xed, 12 }, { 0x3dd, 12 }, { 0x326, 12 }, {  0x5e, 13 },
    {  0x16,  7 },
};

static const int8_t table0_level[132] = {
    1,  2,  3,  4,  5,  6,  7,  8,
    9, 10, 11, 12, 13, 14, 15, 16,
    1,  2,  3,  4,  5,  6,  7,  8,
    9, 10, 11,  1,  2,  3,  4,  5,
    6,  7,  8,  1,  2,  3,  4,  5,
    6,  7,  1,  2,  3,  4,  5,  1,
    2,  3,  4,  1,  2,  3,  4,  1,
    2,  3,  1,  2,  3,  1,  2,  3,
    1,  2,  3,  1,  2,  3,  1,  2,
    3,  1,  2,  3,  1,  2,  1,  2,
    1,  1,  1,  1,  1,  1,  2,  3,
    4,  1,  2,  3,  4,  1,  2,  3,
    1,  2,  3,  1,  2,  1,  2,  1,
    2,  1,  2,  1,  2,  1,  2,  1,
    2,  1,  2,  1,  2,  1,  2,  1,
    1,  1,  1,  1,  1,  1,  1,  1,
    1,  1,  1,  1,
};

static const int8_t table0_run[132] = {
     0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,
     1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  2,  2,  2,  2,  2,
     2,  2,  2,  3,  3,  3,  3,  3,
     3,  3,  4,  4,  4,  4,  4,  5,
     5,  5,  5,  6,  6,  6,  6,  7,
     7,  7,  8,  8,  8,  9,  9,  9,
    10, 10, 10, 11, 11, 11, 12, 12,
    12, 13, 13, 13, 14, 14, 15, 15,
    16, 17, 18, 19, 20,  0,  0,  0,
     0,  1,  1,  1,  1,  2,  2,  2,
     3,  3,  3,  4,  4,  5,  5,  6,
     6,  7,  7,  8,  8,  9,  9, 10,
    10, 11, 11, 12, 12, 13, 13, 14,
    15, 16, 17, 18, 19, 20, 21, 22,
    23, 24, 25, 26,
};

/* vlc table 1, for intra chroma and P macroblocks */

static const uint16_t table1_vlc[149][2] = {
    {    0x4,  3 }, {   0x14,  5 }, {   0x17,  7 }, {   0x7f,  8 },
    {  0x154,  9 }, {  0x1f2, 10 }, {   0xbf, 11 }, {   0x65, 12 },
    {  0xaaa, 12 }, {  0x630, 13 }, { 0x1597, 13 }, {  0x3b7, 14 },
    { 0x2b22, 14 }, {  0xbe6, 15 }, {    0xb,  4 }, {   0x37,  7 },
    {   0x62,  9 }, {    0x7, 11 }, {  0x166, 12 }, {   0xce, 13 },
    { 0x1590, 13 }, {  0x5f6, 14 }, {  0xbe7, 15 }, {    0x7,  5 },
    {   0x6d,  8 }, {    0x3, 11 }, {  0x31f, 12 }, {  0x5f2, 14 },
    {    0x2,  6 }, {   0x61,  9 }, {   0x55, 12 }, {  0x1df, 14 },
    {   0x1a,  6 }, {   0x1e, 10 }, {  0xac9, 12 }, { 0x2b23, 14 },
    {   0x1e,  6 }, {   0x1f, 10 }, {  0xac3, 12 }, { 0x2b2b, 14 },
    {    0x6,  7 }, {    0x4, 11 }, {  0x2f8, 13 }, {   0x19,  7 },
    {    0x6, 11 }, {  0x63d, 13 }, {   0x57,  7 }, {  0x182, 11 },
    { 0x2aa2, 14 }, {    0x4,  8 }, {  0x180, 11 }, {  0x59c, 14 },
    {   0x7d,  8 }, {  0x164, 12 }, {  0x76d, 15 }, {    0x2,  9 },
    {  0x18d, 11 }, { 0x1581, 13 }, {   0xad,  8 }, {   0x60, 12 },
    {  0xc67, 14 }, {   0x1c,  9 }, {   0xee, 13 }, {    0x3,  9 },
    {  0x2cf, 13 }, {   0xd9,  9 }, { 0x1580, 13 }, {    0x2, 11 },
    {  0x183, 11 }, {   0x57, 12 }, {   0x61, 12 }, {   0x31, 11 },
    {   0x66, 12 }, {  0x631, 13 }, {  0x632, 13 }, {   0xac, 13 },
    {  0x31d, 12 }, {   0x76, 12 }, {   0x3a, 11 }, {  0x165, 12 },
    {  0xc66, 14 }, {    0x3,  2 }, {   0x54,  7 }, {  0x2ab, 10 },
    {   0x16, 13 }, {  0x5f7, 14 }, {    0x5,  4 }, {   0xf8,  9 },
    {  0xaa9, 12 }, {   0x5f, 15 }, {    0x4,  4 }, {   0x1c, 10 },
    { 0x1550, 13 }, {    0x4,  5 }, {   0x77, 11 }, {  0x76c, 15 },
    {    0xe,  5 }, {    0xa, 12 }, {    0xc,  5 }, {  0x562, 11 },
    {    0x4,  6 }, {  0x31c, 12 }, {    0x6,  6 }, {   0xc8, 13 },
    {    0xd,  6 }, {  0x1da, 13 }, {    0x7,  6 }, {   0xc9, 13 },
    {    0x1,  7 }, {   0x2e, 14 }, {   0x14,  7 }, { 0x1596, 13 },
    {    0xa,  7 }, {  0xac2, 12 }, {   0x16,  7 }, {  0x15b, 14 },
    {   0x15,  7 }, {  0x15a, 14 }, {    0xf,  8 }, {   0x5e, 15 },
    {   0x7e,  8 }, {   0xab,  8 }, {   0x2d,  9 }, {   0xd8,  9 },
    {    0xb,  9 }, {   0x14, 10 }, {  0x2b3, 10 }, {  0x1f3, 10 },
    {   0x3a, 10 }, {    0x0, 10 }, {   0x58, 10 }, {   0x2e,  9 },
    {   0x5e, 10 }, {  0x563, 11 }, {   0xec, 12 }, {   0x54, 12 },
    {  0xac1, 12 }, { 0x1556, 13 }, {  0x2fa, 13 }, {  0x181, 11 },
    { 0x1557, 13 }, {  0x59d, 14 }, { 0x2aa3, 14 }, { 0x2b2a, 14 },
    {  0x1de, 14 }, {  0x63c, 13 }, {   0xcf, 13 }, { 0x1594, 13 },
    {    0xd,  9 },
};

static const int8_t table1_level[148] = {
    1,  2,  3,  4,  5,  6, 7, 8,
    9, 10, 11, 12, 13, 14, 1, 2,
    3,  4,  5,  6,  7,  8, 9, 1,
    2,  3,  4,  5,  1,  2, 3, 4,
    1,  2,  3,  4,  1,  2, 3, 4,
    1,  2,  3,  1,  2,  3, 1, 2,
    3,  1,  2,  3,  1,  2, 3, 1,
    2,  3,  1,  2,  3,  1, 2, 1,
    2,  1,  2,  1,  1,  1, 1, 1,
    1,  1,  1,  1,  1,  1, 1, 1,
    1,  1,  2,  3,  4,  5, 1, 2,
    3,  4,  1,  2,  3,  1, 2, 3,
    1,  2,  1,  2,  1,  2, 1, 2,
    1,  2,  1,  2,  1,  2, 1, 2,
    1,  2,  1,  2,  1,  2, 1, 2,
    1,  1,  1,  1,  1,  1, 1, 1,
    1,  1,  1,  1,  1,  1, 1, 1,
    1,  1,  1,  1,  1,  1, 1, 1,
    1,  1,  1,  1,
};

static const int8_t table1_run[148] = {
     0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  2,
     2,  2,  2,  2,  3,  3,  3,  3,
     4,  4,  4,  4,  5,  5,  5,  5,
     6,  6,  6,  7,  7,  7,  8,  8,
     8,  9,  9,  9, 10, 10, 10, 11,
    11, 11, 12, 12, 12, 13, 13, 14,
    14, 15, 15, 16, 17, 18, 19, 20,
    21, 22, 23, 24, 25, 26, 27, 28,
    29,  0,  0,  0,  0,  0,  1,  1,
     1,  1,  2,  2,  2,  3,  3,  3,
     4,  4,  5,  5,  6,  6,  7,  7,
     8,  8,  9,  9, 10, 10, 11, 11,
    12, 12, 13, 13, 14, 14, 15, 15,
    16, 17, 18, 19, 20, 21, 22, 23,
    24, 25, 26, 27, 28, 29, 30, 31,
    32, 33, 34, 35, 36, 37, 38, 39,
    40, 41, 42, 43,
};

/* third vlc table */

static const uint16_t table2_vlc[186][2] = {
    {    0x1,  2 }, {    0x5,  3 }, {    0xd,  4 }, {   0x12,  5 },
    {    0xe,  6 }, {   0x15,  7 }, {   0x13,  8 }, {   0x3f,  8 },
    {   0x4b,  9 }, {  0x11f,  9 }, {   0xb8, 10 }, {  0x3e3, 10 },
    {  0x172, 11 }, {  0x24d, 12 }, {  0x3da, 12 }, {  0x2dd, 13 },
    { 0x1f55, 13 }, {  0x5b9, 14 }, { 0x3eae, 14 }, {    0x0,  4 },
    {   0x10,  5 }, {    0x8,  7 }, {   0x20,  8 }, {   0x29,  9 },
    {  0x1f4,  9 }, {  0x233, 10 }, {  0x1e0, 11 }, {  0x12a, 12 },
    {  0x3dd, 12 }, {  0x50a, 13 }, { 0x1f29, 13 }, {  0xa42, 14 },
    { 0x1272, 15 }, { 0x1737, 15 }, {    0x3,  5 }, {   0x11,  7 },
    {   0xc4,  8 }, {   0x4b, 10 }, {   0xb4, 11 }, {  0x7d4, 11 },
    {  0x345, 12 }, {  0x2d7, 13 }, {  0x7bf, 13 }, {  0x938, 14 },
    {  0xbbb, 14 }, {  0x95e, 15 }, {   0x13,  5 }, {   0x78,  7 },
    {   0x69,  9 }, {  0x232, 10 }, {  0x461, 11 }, {  0x3ec, 12 },
    {  0x520, 13 }, { 0x1f2a, 13 }, { 0x3e50, 14 }, { 0x3e51, 14 },
    { 0x1486, 15 }, {    0xc,  6 }, {   0x24,  9 }, {   0x94, 11 },
    {  0x8c0, 12 }, {  0xf09, 14 }, { 0x1ef0, 15 }, {   0x3d,  6 },
    {   0x53,  9 }, {  0x1a0, 11 }, {  0x2d6, 13 }, {  0xf08, 14 },
    {   0x13,  7 }, {   0x7c,  9 }, {  0x7c1, 11 }, {  0x4ac, 14 },
    {   0x1b,  7 }, {   0xa0, 10 }, {  0x344, 12 }, {  0xf79, 14 },
    {   0x79,  7 }, {  0x3e1, 10 }, {  0x2d4, 13 }, { 0x2306, 14 },
    {   0x21,  8 }, {  0x23c, 10 }, {  0xfae, 12 }, { 0x23de, 14 },
    {   0x35,  8 }, {  0x175, 11 }, {  0x7b3, 13 }, {   0xc5,  8 },
    {  0x174, 11 }, {  0x785, 13 }, {   0x48,  9 }, {  0x1a3, 11 },
    {  0x49e, 13 }, {   0x2c,  9 }, {   0xfa, 10 }, {  0x7d6, 11 },
    {   0x92, 10 }, {  0x5cc, 13 }, { 0x1ef1, 15 }, {   0xa3, 10 },
    {  0x3ed, 12 }, {  0x93e, 14 }, {  0x1e2, 11 }, { 0x1273, 15 },
    {  0x7c4, 11 }, { 0x1487, 15 }, {  0x291, 12 }, {  0x293, 12 },
    {  0xf8a, 12 }, {  0x509, 13 }, {  0x508, 13 }, {  0x78d, 13 },
    {  0x7be, 13 }, {  0x78c, 13 }, {  0x4ae, 14 }, {  0xbba, 14 },
    { 0x2307, 14 }, {  0xb9a, 14 }, { 0x1736, 15 }, {    0xe,  4 },
    {   0x45,  7 }, {  0x1f3,  9 }, {  0x47a, 11 }, {  0x5dc, 13 },
    { 0x23df, 14 }, {   0x19,  5 }, {   0x28,  9 }, {  0x176, 11 },
    {  0x49d, 13 }, { 0x23dd, 14 }, {   0x30,  6 }, {   0xa2, 10 },
    {  0x2ef, 12 }, {  0x5b8, 14 }, {   0x3f,  6 }, {   0xa5, 10 },
    {  0x3db, 12 }, {  0x93f, 14 }, {   0x44,  7 }, {  0x7cb, 11 },
    {  0x95f, 15 }, {   0x63,  7 }, {  0x3c3, 12 }, {   0x15,  8 },
    {  0x8f6, 12 }, {   0x17,  8 }, {  0x498, 13 }, {   0x2c,  8 },
    {  0x7b2, 13 }, {   0x2f,  8 }, { 0x1f54, 13 }, {   0x8d,  8 },
    {  0x7bd, 13 }, {   0x8e,  8 }, { 0x1182, 13 }, {   0xfb,  8 },
    {  0x50b, 13 }, {   0x2d,  8 }, {  0x7c0, 11 }, {   0x79,  9 },
    { 0x1f5f, 13 }, {   0x7a,  9 }, { 0x1f56, 13 }, {  0x231, 10 },
    {  0x3e4, 10 }, {  0x1a1, 11 }, {  0x143, 11 }, {  0x1f7, 11 },
    {  0x16f, 12 }, {  0x292, 12 }, {  0x2e7, 12 }, {  0x16c, 12 },
    {  0x16d, 12 }, {  0x3dc, 12 }, {  0xf8b, 12 }, {  0x499, 13 },
    {  0x3d8, 12 }, {  0x78e, 13 }, {  0x2d5, 13 }, { 0x1f5e, 13 },
    { 0x1f2b, 13 }, {  0x78f, 13 }, {  0x4ad, 14 }, { 0x3eaf, 14 },
    { 0x23dc, 14 }, {   0x4a,  9 },
};

static const int8_t table2_level[185] = {
     1,  2,  3,  4,  5,  6,  7,  8,
     9, 10, 11, 12, 13, 14, 15, 16,
    17, 18, 19,  1,  2,  3,  4,  5,
     6,  7,  8,  9, 10, 11, 12, 13,
    14, 15,  1,  2,  3,  4,  5,  6,
     7,  8,  9, 10, 11, 12,  1,  2,
     3,  4,  5,  6,  7,  8,  9, 10,
    11,  1,  2,  3,  4,  5,  6,  1,
     2,  3,  4,  5,  1,  2,  3,  4,
     1,  2,  3,  4,  1,  2,  3,  4,
     1,  2,  3,  4,  1,  2,  3,  1,
     2,  3,  1,  2,  3,  1,  2,  3,
     1,  2,  3,  1,  2,  3,  1,  2,
     1,  2,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,
     2,  3,  4,  5,  6,  1,  2,  3,
     4,  5,  1,  2,  3,  4,  1,  2,
     3,  4,  1,  2,  3,  1,  2,  1,
     2,  1,  2,  1,  2,  1,  2,  1,
     2,  1,  2,  1,  2,  1,  2,  1,
     2,  1,  2,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,
     1,
};

static const int8_t table2_run[185] = {
     0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  2,  2,  2,  2,  2,  2,
     2,  2,  2,  2,  2,  2,  3,  3,
     3,  3,  3,  3,  3,  3,  3,  3,
     3,  4,  4,  4,  4,  4,  4,  5,
     5,  5,  5,  5,  6,  6,  6,  6,
     7,  7,  7,  7,  8,  8,  8,  8,
     9,  9,  9,  9, 10, 10, 10, 11,
    11, 11, 12, 12, 12, 13, 13, 13,
    14, 14, 14, 15, 15, 15, 16, 16,
    17, 17, 18, 19, 20, 21, 22, 23,
    24, 25, 26, 27, 28, 29, 30,  0,
     0,  0,  0,  0,  0,  1,  1,  1,
     1,  1,  2,  2,  2,  2,  3,  3,
     3,  3,  4,  4,  4,  5,  5,  6,
     6,  7,  7,  8,  8,  9,  9, 10,
    10, 11, 11, 12, 12, 13, 13, 14,
    14, 15, 15, 16, 17, 18, 19, 20,
    21, 22, 23, 24, 25, 26, 27, 28,
    29, 30, 31, 32, 33, 34, 35, 36,
    37,
};

/* second non intra vlc table */
static const uint16_t table4_vlc[169][2] = {
    {    0x0,  3 }, {    0x3,  4 }, {    0xb,  5 }, {   0x14,  6 },
    {   0x3f,  6 }, {   0x5d,  7 }, {   0xa2,  8 }, {   0xac,  9 },
    {  0x16e,  9 }, {  0x20a, 10 }, {  0x2e2, 10 }, {  0x432, 11 },
    {  0x5c9, 11 }, {  0x827, 12 }, {  0xb54, 12 }, {  0x4e6, 13 },
    { 0x105f, 13 }, { 0x172a, 13 }, { 0x20b2, 14 }, { 0x2d4e, 14 },
    { 0x39f0, 14 }, { 0x4175, 15 }, { 0x5a9e, 15 }, {    0x4,  4 },
    {   0x1e,  5 }, {   0x42,  7 }, {   0xb6,  8 }, {  0x173,  9 },
    {  0x395, 10 }, {  0x72e, 11 }, {  0xb94, 12 }, { 0x16a4, 13 },
    { 0x20b3, 14 }, { 0x2e45, 14 }, {    0x5,  5 }, {   0x40,  7 },
    {   0x49,  9 }, {  0x28f, 10 }, {  0x5cb, 11 }, {  0x48a, 13 },
    {  0x9dd, 14 }, { 0x73e2, 15 }, {   0x18,  5 }, {   0x25,  8 },
    {   0x8a, 10 }, {  0x51b, 11 }, {  0xe5f, 12 }, {  0x9c9, 14 },
    { 0x139c, 15 }, {   0x29,  6 }, {   0x4f,  9 }, {  0x412, 11 },
    {  0x48d, 13 }, { 0x2e41, 14 }, {   0x38,  6 }, {  0x10e,  9 },
    {  0x5a8, 11 }, { 0x105c, 13 }, { 0x39f2, 14 }, {   0x58,  7 },
    {  0x21f, 10 }, {  0xe7e, 12 }, { 0x39ff, 14 }, {   0x23,  8 },
    {  0x2e3, 10 }, {  0x4e5, 13 }, { 0x2e40, 14 }, {   0xa1,  8 },
    {  0x5be, 11 }, {  0x9c8, 14 }, {   0x83,  8 }, {  0x13a, 11 },
    { 0x1721, 13 }, {   0x44,  9 }, {  0x276, 12 }, { 0x39f6, 14 },
    {   0x8b, 10 }, {  0x4ef, 13 }, { 0x5a9b, 15 }, {  0x208, 10 },
    { 0x1cfe, 13 }, {  0x399, 10 }, { 0x1cb4, 13 }, {  0x39e, 10 },
    { 0x39f3, 14 }, {  0x5ab, 11 }, { 0x73e3, 15 }, {  0x737, 11 },
    { 0x5a9f, 15 }, {  0x82d, 12 }, {  0xe69, 12 }, {  0xe68, 12 },
    {  0x433, 11 }, {  0xb7b, 12 }, { 0x2df8, 14 }, { 0x2e56, 14 },
    { 0x2e57, 14 }, { 0x39f7, 14 }, { 0x51a5, 15 }, {    0x3,  3 },
    {   0x2a,  6 }, {   0xe4,  8 }, {  0x28e, 10 }, {  0x735, 11 },
    { 0x1058, 13 }, { 0x1cfa, 13 }, { 0x2df9, 14 }, { 0x4174, 15 },
    {    0x9,  4 }, {   0x54,  8 }, {  0x398, 10 }, {  0x48b, 13 },
    { 0x139d, 15 }, {    0xd,  4 }, {   0xad,  9 }, {  0x826, 12 },
    { 0x2d4c, 14 }, {   0x11,  5 }, {  0x16b,  9 }, {  0xb7f, 12 },
    { 0x51a4, 15 }, {   0x19,  5 }, {  0x21b, 10 }, { 0x16fd, 13 },
    {   0x1d,  5 }, {  0x394, 10 }, { 0x28d3, 14 }, {   0x2b,  6 },
    {  0x5bc, 11 }, { 0x5a9a, 15 }, {   0x2f,  6 }, {  0x247, 12 },
    {   0x10,  7 }, {  0xa35, 12 }, {   0x3e,  6 }, {  0xb7a, 12 },
    {   0x59,  7 }, { 0x105e, 13 }, {   0x26,  8 }, {  0x9cf, 14 },
    {   0x55,  8 }, { 0x1cb5, 13 }, {   0x57,  8 }, {  0xe5b, 12 },
    {   0xa0,  8 }, { 0x1468, 13 }, {  0x170,  9 }, {   0x90, 10 },
    {  0x1ce,  9 }, {  0x21a, 10 }, {  0x218, 10 }, {  0x168,  9 },
    {  0x21e, 10 }, {  0x244, 12 }, {  0x736, 11 }, {  0x138, 11 },
    {  0x519, 11 }, {  0xe5e, 12 }, {  0x72c, 11 }, {  0xb55, 12 },
    {  0x9dc, 14 }, { 0x20bb, 14 }, {  0x48c, 13 }, { 0x1723, 13 },
    { 0x2e44, 14 }, { 0x16a5, 13 }, {  0x518, 11 }, { 0x39fe, 14 },
    {  0x169,  9 },
};

static const int8_t table4_level[168] = {
     1,  2,  3,  4,  5,  6,  7,  8,
     9, 10, 11, 12, 13, 14, 15, 16,
    17, 18, 19, 20, 21, 22, 23,  1,
     2,  3,  4,  5,  6,  7,  8,  9,
    10, 11,  1,  2,  3,  4,  5,  6,
     7,  8,  1,  2,  3,  4,  5,  6,
     7,  1,  2,  3,  4,  5,  1,  2,
     3,  4,  5,  1,  2,  3,  4,  1,
     2,  3,  4,  1,  2,  3,  1,  2,
     3,  1,  2,  3,  1,  2,  3,  1,
     2,  1,  2,  1,  2,  1,  2,  1,
     2,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  2,  3,  4,  5,
     6,  7,  8,  9,  1,  2,  3,  4,
     5,  1,  2,  3,  4,  1,  2,  3,
     4,  1,  2,  3,  1,  2,  3,  1,
     2,  3,  1,  2,  1,  2,  1,  2,
     1,  2,  1,  2,  1,  2,  1,  2,
     1,  2,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,
};

static const int8_t table4_run[168] = {
     0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  1,
     1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  2,  2,  2,  2,  2,  2,
     2,  2,  3,  3,  3,  3,  3,  3,
     3,  4,  4,  4,  4,  4,  5,  5,
     5,  5,  5,  6,  6,  6,  6,  7,
     7,  7,  7,  8,  8,  8,  9,  9,
     9, 10, 10, 10, 11, 11, 11, 12,
    12, 13, 13, 14, 14, 15, 15, 16,
    16, 17, 18, 19, 20, 21, 22, 23,
    24, 25, 26,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  1,  1,  1,  1,
     1,  2,  2,  2,  2,  3,  3,  3,
     3,  4,  4,  4,  5,  5,  5,  6,
     6,  6,  7,  7,  8,  8,  9,  9,
    10, 10, 11, 11, 12, 12, 13, 13,
    14, 14, 15, 16, 17, 18, 19, 20,
    21, 22, 23, 24, 25, 26, 27, 28,
    29, 30, 31, 32, 33, 34, 35, 36,
};

RLTable ff_rl_table[NB_RL_TABLES] = {
    /* intra luminance tables */
    /* low motion  */
    { 132,
      85,
      table0_vlc,
      table0_run,
      table0_level, },
    /* high motion */
    { 185,
      119,
      table2_vlc,
      table2_run,
      table2_level, },
    /* mid-rate */
    { 102,
      67,
      ff_mpeg4_intra_vlc,
      ff_mpeg4_intra_run,
      ff_mpeg4_intra_level, },
    /* intra chrominance / non intra tables */
    /* low motion inter */
    { 148,
      81,
      table1_vlc,
      table1_run,
      table1_level, },
    /* high motion inter */
    { 168,
      99,
      table4_vlc,
      table4_run,
      table4_level, },
    /* mid rate inter */
    { 102,
      58,
      ff_inter_vlc,
      ff_inter_run,
      ff_inter_level, },
};

/* motion vector table 0 */

#define E(mvx, mvy) ((mvx << 8) | mvy)
#define ESCAPE 0

const uint16_t ff_msmp4_mv_table0[MSMPEG4_MV_TABLES_NB_ELEMS] = {
    ESCAPE,    E(31, 39), E(38, 31), E(24, 32), E(23, 30), E(34, 20),
    E(33, 22), E(21, 29), E(28, 34), E(33, 24), E(49, 32), E(27, 41),
    E(32,  2), E(40, 33), E(34, 39), E(37, 25), E(29, 26), E(38, 28),
    E(43, 31), E(37, 33), E(31, 35), E(32, 35), E(25, 29), E(26, 28),
    E(32, 42), E(33, 23), E(42, 31), E(35, 36), E(27, 31), E(36, 30),
    E(21, 32), E(23, 32), E(43, 32), E(36, 32), E(39, 29), E(38, 30),
    E(36, 34), E(34, 36), E(19, 30), E(20, 38), E(41, 41), E(26, 44),
    E(41, 53), E(42, 11), E(42, 20), E(42, 25), E(26, 45), E(26, 46),
    E(26, 47), E(26, 48), E( 1, 39), E(26, 52), E(37, 46), E(37, 49),
    E(37, 63), E(23, 55), E(34, 44), E(42, 38), E(27, 13), E(43, 13),
    E(34, 57), E(27, 22), E(31, 51), E(24, 16), E(33, 12), E(16, 28),
    E( 3, 35), E(31, 57), E(24, 25), E(31, 33), E(33, 33), E(34, 32),
    E(31, 34), E(34, 31), E(33, 20), E(24, 26), E( 4, 28), E(44, 28),
    E(44, 29), E(11, 32), E(44, 31), E(31, 62), E(13, 34), E(13, 35),
    E(44, 35), E(45, 19), E(13, 49), E(25, 48), E(22, 39), E(38, 40),
    E(26, 18), E(38, 43), E(38, 46), E(39, 11), E(32,  6), E(30, 48),
    E(39, 19), E(30, 49), E(22, 41), E(32, 10), E(26, 23), E(30, 58),
    E(47, 17), E(47, 21), E(47, 23), E(19, 34), E(47, 30), E(30, 60),
    E(31,  0), E(35, 44), E( 8, 34), E(23, 26), E(47, 37), E(47, 46),
    E(27, 43), E(35, 49), E(47, 59), E(31,  4), E(48, 34), E(35, 59),
    E(19, 37), E(36, 22), E(49, 25), E(49, 28), E(33, 46), E(31,  6),
    E(39, 40), E(27, 59), E(39, 42), E(49, 36), E(49, 61), E(50, 28),
    E(50, 30), E(50, 31), E(39, 43), E(31, 10), E(51, 22), E(51, 23),
    E(39, 47), E(28, 20), E(33, 52), E(33, 55), E(21, 36), E(40, 25),
    E(31, 16), E(52,  1), E(53, 11), E(40, 27), E(53, 29), E( 9,  5),
    E(33, 63), E(31, 18), E(53, 41), E(54, 31), E(54, 34), E(54, 38),
    E(55,  1), E(55,  5), E(34, 13), E(15, 28), E(56, 31), E(25, 17),
    E(56, 38), E(21, 39), E(21, 41), E(40, 36), E(40, 37), E(25, 22),
    E(40, 39), E(58, 33), E(36, 43), E( 5, 43), E(41, 15), E(41, 17),
    E(12, 49), E(60, 30), E(41, 21), E(41, 22), E(37, 20), E(22, 10),
    E(61, 17), E(18, 25), E(29, 54), E(29, 57), E(30,  3), E(61, 34),
    E(30,  8), E(61, 36), E(62, 25), E(62, 31), E(30, 14), E(30, 15),
    E(63, 19), E(18, 29), E(13, 25), E(30, 18), E(45, 30), E( 1, 63),
    E(34, 16), E(31, 42), E(22, 28), E(35, 13), E(15, 33), E(46, 33),
    E(46, 35), E(35, 19), E(31, 46), E(28, 43), E(29, 17), E(29, 19),
    E(23, 36), E(41, 39), E(27, 24), E(14, 31), E(53, 32), E(53, 33),
    E(27, 26), E(24, 28), E(32, 62), E(57, 33), E(32, 63), E(17, 33),
    E(42, 39), E(37, 19), E(29, 49), E(33, 17), E( 1, 31), E(25, 43),
    E(32, 13), E(32, 15), E(63, 29), E(26, 25), E(40, 35), E(23, 34),
    E(27, 25), E(22, 33), E(24, 29), E(22, 31), E(39, 39), E(43, 35),
    E(27, 36), E(37, 24), E( 6, 32), E(42, 30), E(24, 33), E(40, 31),
    E(28, 39), E(32,  1), E(32, 41), E(41, 32), E(36, 31), E(28, 32),
    E(35, 31), E(29, 31), E(38, 32), E(23, 31), E(27, 30), E(32, 59),
    E(32, 22), E(32, 21), E(30, 39), E(35, 25), E(26, 34), E(34, 38),
    E(30, 28), E(44, 33), E(36, 23), E(47, 33), E(26, 27), E(30, 44),
    E(31, 60), E(41, 30), E(39, 36), E(33, 19), E(33, 29), E(32, 36),
    E(22, 34), E(45, 31), E(36, 24), E(32, 16), E(45, 35), E(19, 32),
    E(36, 27), E(32,  4), E(43, 33), E(60, 32), E(37, 31), E(35, 33),
    E(35, 32), E(34, 29), E(33, 28), E(25, 33), E(25, 34), E(21, 33),
    E(36, 35), E(37, 32), E(29, 39), E(31, 24), E(30, 41), E(31,  3),
    E(41, 35), E(22, 32), E(32, 39), E(32, 30), E(33, 32), E(32, 29),
    E(33, 35), E(31, 28), E(28, 30), E(26, 33), E(31, 49), E(39, 41),
    E(20, 28), E(20, 29), E(44, 34), E(23, 38), E(33,  7), E(34, 61),
    E(29, 29), E(31, 38), E(35, 11), E(33, 50), E(33, 13), E(28, 41),
    E(41, 19), E(23, 47), E(41, 25), E(41, 26), E(29, 15), E(25, 42),
    E(26, 41), E(49, 29), E(29, 45), E(24, 27), E(37, 17), E(49, 35),
    E(34, 33), E(30, 35), E(50, 32), E(51, 29), E(51, 32), E(26, 24),
    E(39, 13), E(25, 26), E(26, 26), E(15, 31), E(39, 24), E(18, 33),
    E(42, 28), E(17, 30), E( 4, 31), E(31, 11), E(32, 45), E(32, 46),
    E(60, 33), E( 1, 33), E(42, 35), E(32, 57), E( 0, 32), E(12, 32),
    E(19, 35), E(35, 45), E(22, 38), E(30, 21), E( 5, 33), E( 5, 31),
    E(17, 31), E(29, 21), E(22, 35), E(31,  5), E(31, 17), E(43, 29),
    E(37, 38), E(27, 38), E(32, 18), E(32, 58), E(32, 19), E(33, 43),
    E(29, 41), E(33, 41), E(31, 37), E(30, 37), E(41, 33), E( 7, 32),
    E(28, 31), E(34, 27), E(30, 27), E(37, 28), E(26, 36), E(34, 30),
    E(31, 29), E(33, 30), E(27, 32), E(38, 29), E(30, 25), E(36, 28),
    E(37, 36), E(21, 34), E(23, 39), E(19, 53), E(34, 41), E(36, 33),
    E(32, 28), E(25, 32), E(29, 32), E(29, 33), E(33, 27), E(29, 27),
    E(43, 30), E(35, 41), E(29, 43), E(26, 39), E(22, 29), E(32,  7),
    E(19, 29), E(32, 17), E(25, 28), E(31, 19), E(41, 28), E(49, 31),
    E(28, 25), E(34, 19), E(28, 40), E(39, 26), E(34, 21), E(19, 33),
    E(37, 39), E(38, 23), E(29, 40), E(21, 30), E(36, 39), E(42, 34),
    E(27, 28), E(59, 33), E(38, 33), E(35, 28), E(33, 34), E(30, 32),
    E(30, 36), E(23, 35), E(33, 21), E(35, 23), E(31, 21), E(30, 26),
    E(41, 31), E(39, 32), E(32, 27), E(26, 30), E(31, 22), E(37, 37),
    E(38, 27), E(39, 27), E(32, 43), E(23, 29), E(32, 60), E(27, 39),
    E(25, 35), E(61, 32), E(30, 33), E(30, 34), E(34, 28), E(33, 36),
    E(37, 27), E(20, 22), E(20, 27), E(41, 20), E(13, 29), E( 2, 33),
    E( 5, 41), E( 5, 42), E( 1, 35), E(32, 24), E(34, 24), E( 5, 62),
    E(32, 50), E( 6, 29), E(32, 53), E(35, 50), E(20, 35), E(35, 55),
    E(27, 42), E(35, 61), E(35, 63), E(36,  4), E(36,  7), E(13, 42),
    E(36, 21), E(41, 46), E(41, 47), E(33, 37), E(28, 33), E(41, 49),
    E( 3, 27), E(30, 46), E(42, 17), E(27, 55), E(20, 41), E(30, 50),
    E(27, 63), E(28,  4), E(30, 56), E(21, 18), E(33,  8), E(33, 10),
    E(14, 18), E(24, 51), E(30, 63), E(42, 36), E(24, 55), E(33, 18),
    E(25,  5), E(42, 55), E(43,  9), E(14, 29), E(43, 17), E(43, 21),
    E(14, 30), E(43, 27), E( 7,  1), E(36, 59), E(37,  7), E(37, 11),
    E(37, 12), E(37, 15), E( 7,  5), E(14, 42), E(25, 24), E(43, 41),
    E(43, 43), E(44,  7), E(15, 27), E(37, 22), E( 7, 25), E( 7, 31),
    E( 3, 30), E( 1, 37), E( 8, 22), E(15, 39), E(15, 53), E( 8, 29),
    E(44, 36), E(44, 37), E(44, 48), E(45,  0), E(45,  5), E(45, 13),
    E(45, 17), E( 8, 31), E( 8, 32), E(45, 26), E(21, 46), E(45, 28),
    E(28, 44), E(28, 45), E(28, 46), E(21, 53), E(28, 49), E(28, 51),
    E(22,  3), E(37, 41), E(46,  3), E(46, 22), E(37, 42), E( 8, 33),
    E(29,  5), E(37, 45), E(29,  7), E(22, 22), E(46, 40), E(37, 53),
    E(22, 24), E(29, 14), E(47, 25), E(47, 27), E(38, 10), E(38, 12),
    E(16, 34), E(38, 16), E(38, 17), E(33, 53), E(38, 20), E(22, 26),
    E(47, 39), E(47, 45), E(29, 18), E(25, 45), E(16, 35), E( 0, 33),
    E(48, 31), E( 1, 49), E(34,  4), E(48, 39), E(48, 42), E(34, 11),
    E(25, 51), E(26,  5), E(26, 13), E(49, 26), E(49, 27), E(26, 17),
    E( 1, 52), E(49, 30), E( 4,  0), E(26, 22), E( 4, 12), E(49, 34),
    E( 1, 62), E( 9, 36), E(49, 38), E(49, 41), E(49, 47), E(10, 30),
    E(49, 63), E(17, 38), E(38, 41), E(17, 43), E(17, 59), E(50, 33),
    E(38, 45), E(50, 37), E(50, 38), E(22, 40), E(38, 47), E(51, 24),
    E(38, 48), E(38, 49), E(51, 30), E(39,  1), E(39, 10), E(18, 22),
    E(22, 49), E(39, 15), E(51, 39), E(22, 59), E(52, 31), E(52, 32),
    E(52, 33), E(39, 18), E(53, 13), E(53, 15), E(10, 31), E(23, 21),
    E(18, 27), E(29, 44), E(10, 42), E(53, 39), E(29, 46), E(54, 17),
    E(11, 26), E(54, 32), E(11, 30), E(11, 31), E(55,  0), E( 1, 30),
    E(18, 34), E(55,  9), E(55, 12), E(55, 13), E(55, 25), E(55, 31),
    E(55, 32), E(29, 55), E(34, 43), E(18, 35), E(29, 61), E(56, 33),
    E(56, 35), E(34, 46), E(57, 25), E(34, 47), E(34, 48), E(29, 62),
    E(29, 63), E(57, 38), E(57, 61), E(35,  1), E(35,  2), E(35,  5),
    E(35,  7), E(35,  9), E(39, 46), E(35, 10), E(59, 34), E(59, 35),
    E(18, 36), E(59, 55), E(39, 51), E(30,  7), E(18, 40), E(40, 23),
    E(60, 34), E(60, 37), E(61,  5), E(30, 13), E(18, 42), E(61, 18),
    E(61, 25), E(19, 27), E(19, 28), E(41, 11), E(17, 37), E(26, 42),
    E(20, 39), E(45, 37), E(46, 31), E(32, 49), E(37, 21), E(22, 27),
    E(47, 29), E(29, 47), E(31, 53), E(27, 40), E(29, 51), E(30, 16),
    E(21, 26), E(33,  1), E(35, 40), E(18, 30), E(25, 41), E(33, 44),
    E(50, 34), E(51, 25), E(42, 27), E( 2, 31), E(15, 35), E(51, 35),
    E(53, 31), E(33, 15), E(25, 49), E(29,  1), E(55, 35), E(37, 40),
    E(38,  5), E(33, 58), E(38, 21), E(59, 29), E(38, 22), E(33, 59),
    E(39, 45), E(13, 31), E(32,  9), E(40, 26), E(61,  9), E(61, 29),
    E(32, 11), E( 9, 32), E(44, 30), E( 3,  3), E(31, 13), E(29, 20),
    E(25, 18), E(24, 35), E(48, 32), E(16, 32), E(28, 24), E(49, 33),
    E(22, 30), E(34, 40), E(31, 47), E(38, 39), E(39, 37), E(44, 32),
    E(11, 33), E(35, 24), E(30, 23), E(33, 45), E(33, 47), E(36, 25),
    E(31, 59), E(40, 34), E(25, 27), E(27, 27), E(38, 26), E(31, 40),
    E(36, 37), E(28, 27), E(32, 31), E(31, 31), E(34, 35), E(32, 37),
    E(34, 34), E(24, 31), E(39, 34), E(34, 37), E(27, 33), E(28, 35),
    E(31, 61), E(36, 26), E(25, 39), E(45, 33), E(25, 31), E(35, 34),
    E( 3, 31), E(34, 22), E(38, 37), E(39, 25), E(40, 30), E(38, 25),
    E( 1, 32), E(25, 30), E(39, 35), E(30, 38), E(37, 29), E(32, 40),
    E(28, 28), E(29, 38), E(32,  5), E(37, 34), E(31, 27), E(35, 27),
    E(27, 29), E(23, 27), E(28, 23), E(33, 42), E(46, 32), E(15, 32),
    E(13, 33), E(24, 36), E(41, 37), E(24, 37), E(34, 17), E(34, 18),
    E(30, 22), E(26, 40), E(24, 38), E(14, 32), E(60, 31), E(21, 35),
    E(29, 24), E(39, 17), E(23, 25), E(35, 17), E(37, 23), E(30, 24),
    E(32, 61), E(19, 31), E(24, 34), E(25, 25), E(26, 38), E(45, 32),
    E(38, 35), E(36, 29), E( 3, 32), E(26, 29), E(36, 36), E(30, 31),
    E(59, 32), E(28, 36), E(63, 32), E(26, 32), E(38, 34), E(63, 33),
    E(31, 20), E(27, 11), E(20, 32), E(24, 30), E(33,  3), E(37, 35),
    E(28, 38), E(39, 30), E(32,  3), E(31, 26), E(39, 31), E(35, 29),
    E(31, 30), E(32, 38), E(23, 33), E(23,  3), E( 9, 29), E(25, 40),
    E( 9, 31), E(29, 53), E(20, 30), E(30, 17), E(30, 20), E(42, 37),
    E(18, 31), E(28, 42), E(25, 47), E(28, 47), E(28, 54), E(21, 38),
    E(29,  9), E(43, 34), E(29, 13), E(43, 37), E(26, 21), E(44, 25),
    E(44, 27), E(21, 43), E(38, 42), E(35, 30), E(21, 45), E(24, 40),
    E(45, 25), E(45, 27), E(35, 46), E(39, 22), E(22, 25), E(35, 53),
    E(36, 20), E(34,  2), E(34, 14), E(45, 39), E(34, 15), E(29, 22),
    E(31, 44), E(46, 34), E(25, 21), E(46, 38), E(16, 33), E(17, 27),
    E(31, 48), E(20, 34), E(32, 52), E(47, 35), E(47, 47), E(47, 49),
    E(32, 54), E(49,  2), E(49, 13), E(49, 23), E(31, 52), E( 1, 29),
    E(27, 45), E(28, 22), E(39, 49), E(39, 54), E(21, 21), E(36, 40),
    E(23, 42), E(51, 31), E( 2, 30), E(40, 29), E(51, 34), E(32,  0),
    E(53, 25), E(24, 23), E(30, 51), E(24, 24), E(30, 59), E(21, 28),
    E(40, 38), E(57, 29), E(57, 31), E(31,  2), E(41, 13), E(58, 31),
    E(32,  8), E(12, 31), E( 4, 33), E(32, 12), E(34, 45), E(59, 41),
    E(31,  7), E(32, 14), E(13, 30), E( 9, 25), E(35, 18), E(26, 43),
    E(35, 20), E(37, 43), E(61, 35), E(37, 44), E(63,  1), E(26, 49),
    E(29, 42), E(41, 42), E(45, 34), E(33, 51), E(34, 42), E(47, 31),
    E(41, 36), E(22, 36), E(42, 29), E(35, 21), E(35, 22), E(23, 37),
    E(32, 44), E(35, 43), E(43, 25), E(32, 47), E( 7, 33), E(31, 45),
    E(41, 27), E(20, 31), E(31, 58), E(16, 31), E(13, 32), E(63, 31),
    E(25, 38), E(30, 43), E(33, 61), E(31, 43), E(41, 29), E(34, 23),
    E(39, 28), E(47, 32), E(40, 28), E(58, 32), E(59, 31), E(41, 34),
    E(33,  5), E( 3, 33), E(17, 32), E(61, 33), E(42, 33), E(21, 31),
    E(26, 35), E(35, 26), E(29, 36), E(26, 31), E(33, 38), E(30, 30),
    E(31, 32), E(33, 31), E(37, 30), E(31, 23), E(30, 42), E(26, 37),
    E(33, 26), E(27, 35), E(31, 36), E(27, 34), E(35, 37), E(29, 23),
    E(36, 38), E(61, 31), E(33, 40), E(37, 26), E(62, 32), E(35, 38),
    E(57, 32), E(31, 41), E(34, 26), E(34, 25), E(40, 32), E(35, 35),
    E(29, 34), E(32, 25), E(29, 30), E(28, 26), E(38, 18), E(23, 41),
    E(32, 20), E(29, 28), E(29, 25), E(28, 37), E(42, 32), E(33, 39),
    E(32, 26), E(31, 25), E(30, 40), E(35, 47), E(30, 45), E(30, 54),
    E(31,  1), E(32, 23), E(33, 25), E(42, 26), E(47, 34), E(39, 38),
    E(38, 14), E(40, 22), E(40, 24), E(51, 33), E(55, 34), E(56, 32),
    E(18, 32), E(21, 37), E(39, 21), E(39, 23), E(33, 49), E(17, 35),
    E(41, 23), E(23, 28), E(24, 39), E(43, 39), E(25, 23), E(31, 55),
    E(20, 33), E(25, 37), E(38, 38), E(25, 36), E(27, 37), E(29, 37),
    E( 4, 32), E(39, 33), E( 5, 32), E(28, 29), E(38, 24), E(17, 29),
    E(35, 39), E( 2, 32), E(38, 36), E(30, 29), E(29, 35), E(32, 34),
    E(32, 33), E(32, 32),
};

const uint8_t ff_msmp4_mv_table0_lens[MSMPEG4_MV_TABLES_NB_ELEMS] = {
     8, 12, 12, 13, 15, 15, 15, 15, 12, 15, 15, 15, 15, 14, 14, 14, 14, 14,
    14, 11,  9,  8, 13, 14, 14, 14, 14, 13, 11, 12, 12, 12, 12, 10, 13, 13,
    12, 12, 16, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,  6,  6,  7,
     8,  8, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 14, 14, 14, 14, 13, 12, 11, 10,
     9,  9, 11, 13, 13, 14, 14, 13, 13, 13, 13, 13, 12, 16, 16, 15, 15, 15,
    15, 15, 15, 15,  9, 10, 15, 15, 15, 15, 15, 15, 14, 14, 14, 13, 11,  9,
     8, 10, 11, 12, 14, 14, 13, 10, 13, 14, 14, 14, 14, 13, 11,  7,  5,  8,
     9, 11, 12, 13, 16, 16, 16, 16, 16, 16, 16, 16, 11, 12, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,  8, 10, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    14, 14, 11, 12, 13, 13, 11, 12, 13, 14, 14,  9,  9,  8, 10, 13, 13, 13,
    14, 16, 17, 17, 15, 11, 10, 10,  8,  9, 11, 13, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    14, 14, 12, 12,  8,  7, 11, 14, 14, 14, 14, 13, 13, 10, 10, 13, 14, 14,
    14, 14, 14, 14, 14, 14, 13, 12,  8,  9, 11, 11, 14, 17, 17, 17, 17, 17,
    17, 17, 17, 13, 13, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 11, 11, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 14, 14, 14, 14, 14, 14, 14,
    14, 14,  4,  5, 10, 10,  9, 13, 13, 12, 11, 12, 14, 14, 14, 14, 11, 10,
    14, 14, 14, 14, 14, 14, 13, 13, 13, 12, 12, 13, 13, 13, 13, 12, 11, 12,
    12, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 14, 14, 14, 14, 14, 14, 14, 13, 12, 12, 13, 13,  8,
    12, 13, 13, 11, 12, 15, 16, 16, 14, 14, 14, 12, 13, 13, 12, 12, 11, 10,
     8, 11, 13, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 10, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 13,
    13, 13, 12, 12, 12,  9,  4,  5, 12, 13, 14, 14, 12, 12, 10, 12, 12, 14,
    14, 14, 14, 14, 14, 13, 13, 13, 12, 12, 12, 10, 10, 10, 10, 14, 16, 16,
    15, 13, 13, 13, 12, 12, 11, 11, 13, 15, 15, 15, 15, 12, 11, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 14, 14, 14, 13, 12, 12, 11, 12, 12, 14, 14, 13, 13, 13, 10, 10,  6,
     4,  1,
};

/* motion vector table 1 */

const uint16_t ff_msmp4_mv_table1[MSMPEG4_MV_TABLES_NB_ELEMS] = {
    E(32, 32), E(42, 39), E(35, 53), E( 3, 28), E(51, 39), E(35, 57),
    E(16, 46), E(35, 10), E(30, 52), E(63, 27), E(57, 35), E(39, 42),
    E(39, 45), E(25, 59), E(35, 15), E(51, 34), E(31,  9), E(48, 34),
    E( 2, 33), E( 6, 31), E(34, 49), E(30, 43), E(28, 42), E(33, 51),
    E(39, 27), E(33, 42), E(37, 25), E(60, 34), E(38, 60), E( 9, 38),
    E(38, 33), E(51, 32), E(30, 22), E(33, 23), E(32, 24), E(32, 26),
    E(32, 29), E(40, 29), E(27, 23), E(46, 29), E(31, 14), E( 9, 30),
    E(35, 45), E(29, 49), E(31, 11), E(47, 32), E(32, 48), E(39, 35),
    E(37, 31), E(25, 34), E(30, 25), E(24, 30), E(35, 25), E(61, 32),
    E(39, 24), E(16, 24), E(36, 23), E(50, 32), E(32, 45), E(19, 33),
    E(59, 33), E(29, 25), E(30, 36), E(63, 33), E(45, 39), E(31, 12),
    E(60, 29), E(14, 24), E(34,  0), E(42, 20), E(24, 16), E( 0, 28),
    E(37, 43), E(18, 35), E(61, 28), E(57,  1), E(30, 30), E(31, 33),
    E(31, 35), E(26, 32), E(37, 49), E(28,  4), E(53, 21), E(20, 42),
    E(46, 39), E(14, 29), E(47,  3), E(38, 44), E(38, 21), E(38, 53),
    E( 9, 55), E(34, 11), E(39,  5), E(43,  3), E(17, 39), E(39, 53),
    E(59, 28), E( 5, 25), E(27, 19), E( 0, 34), E(55, 31), E(46, 28),
    E(19, 35), E(37, 41), E(62, 30), E(34, 44), E(30, 18), E(16, 29),
    E(19, 34), E(49, 35), E(30, 41), E(25, 27), E(43, 32), E( 4, 32),
    E(34, 25), E(27, 25), E(33, 19), E(32, 41), E(25, 29), E(62, 31),
    E(24, 35), E(32, 38), E(30, 32), E(34, 33), E(34, 29), E(41, 36),
    E(12, 28), E(27, 21), E(41, 25), E(39, 26), E(30, 24), E(37, 21),
    E(14, 31), E( 5, 30), E(13, 34), E(35, 19), E(55, 34), E(30, 44),
    E(40, 36), E(40, 38), E(42, 25), E(31, 19), E(27, 36), E(30, 42),
    E(37, 27), E(31, 17), E(21, 33), E(45, 31), E(32, 44), E(39, 28),
    E(20, 33), E(46, 33), E(47, 31), E(35, 23), E(19, 31), E(33, 30),
    E(28, 32), E(36, 32), E(33, 32), E(34, 40), E( 8, 31), E(20, 28),
    E(35, 39), E(43, 33), E(25, 23), E(45, 37), E(35,  3), E(33, 55),
    E(35, 29), E(36, 41), E(54, 30), E(38, 22), E(27, 41), E(37, 24),
    E(62, 33), E(40, 26), E(15, 35), E(59, 35), E(49, 30), E( 1, 33),
    E(40, 33), E(32,  0), E(29, 37), E(32,  2), E( 3, 32), E(42, 32),
    E(49, 31), E(21, 34), E(21, 30), E(14, 32), E( 5, 31), E(28, 36),
    E(31, 21), E(51, 33), E(31, 55), E(29, 51), E(31, 38), E(62, 32),
    E(34, 47), E(34, 48), E(39, 38), E(24, 24), E(32, 30), E(29, 32),
    E(30, 34), E(35, 34), E(29, 15), E(25, 21), E(26, 36), E(33, 61),
    E(31, 26), E(32, 34), E(30, 33), E(27, 31), E(23, 35), E(29, 50),
    E(21, 25), E(31, 47), E(36, 37), E(35, 41), E(58, 31), E(29, 23),
    E(29, 41), E(21, 29), E(20, 30), E(33, 24), E(32, 22), E(42, 35),
    E(29, 46), E(10, 30), E(61, 29), E(33, 10), E(49, 34), E(35, 42),
    E(34, 45), E( 3, 29), E(38, 42), E(39, 22), E(29,  3), E( 7, 33),
    E(41, 27), E( 1, 34), E(30, 31), E(13, 32), E(28, 25), E(12, 32),
    E(40, 28), E(25, 35), E(31, 61), E(28, 31), E(29, 34), E(34, 35),
    E(28, 33), E(26, 33), E(35,  1), E( 4, 34), E(31, 58), E(33,  2),
    E(39, 37), E(21, 31), E(60, 33), E(23, 28), E(15, 34), E(38, 30),
    E(37, 33), E(34, 34), E(31, 28), E(23, 36), E(11, 35), E(37, 23),
    E(28, 23), E(35, 13), E(30, 58), E(30,  3), E(24, 26), E( 1, 63),
    E(43, 25), E(56, 31), E( 8, 34), E(32, 14), E(44, 33), E(32, 35),
    E(26, 31), E(31,  5), E(33, 49), E(31, 20), E(36, 33), E(32,  1),
    E(27, 33), E(31, 34), E(31, 30), E(28, 39), E(37, 36), E(43, 31),
    E(31, 59), E(34, 39), E(31, 27), E(34, 61), E(42, 38), E(24, 39),
    E(45, 25), E(30, 61), E(20, 29), E(23, 37), E( 8, 30), E(38, 41),
    E(22, 26), E(33, 48), E(17, 28), E(52, 33), E(34, 50), E(29, 21),
    E(32,  9), E(30, 23), E(49, 33), E(43, 35), E(28, 24), E(39, 39),
    E(33, 21), E(33, 34), E(32, 28), E(30, 29), E(25, 37), E(32, 11),
    E(48, 33), E( 2, 30), E(15, 31), E(20, 31), E(22, 35), E(53, 30),
    E(43, 37), E(44, 29), E(29, 30), E( 1, 29), E(27, 40), E(31,  6),
    E(41, 39), E(43, 28), E(41, 28), E(30, 19), E(31,  8), E(36, 22),
    E(30, 45), E(25, 28), E(31, 51), E(28, 40), E(32, 25), E(21, 35),
    E(24, 29), E(25, 25), E(15, 33), E(20, 32), E(40, 30), E(39, 34),
    E(31, 46), E(17, 35), E(22, 32), E(39, 31), E(33, 28), E(29, 29),
    E(21, 32), E(37, 37), E(43, 30), E(18, 30), E(32,  8), E(37, 32),
    E(31, 32), E(31, 22), E(31, 40), E(57, 32), E(38, 29), E(33, 37),
    E(35, 35), E(33, 63), E( 5, 34), E(55, 35), E(31, 10), E(33, 58),
    E(30, 17), E(40, 37), E(23, 39), E(15, 30), E(29, 18), E(34, 54),
    E(34, 14), E(59, 29), E(34, 16), E(30, 59), E(11, 30), E(13, 35),
    E(38, 23), E( 5, 35), E(24, 28), E(33, 15), E(37, 30), E(25, 31),
    E(36, 40), E(13, 33), E(33, 45), E(31, 49), E(32, 16), E(42, 31),
    E(25, 33), E(32, 36), E(39, 25), E(11, 32), E(33,  5), E(61, 31),
    E(63, 32), E(31,  4), E( 0, 30), E(30, 47), E(32, 62), E(37, 29),
    E(31,  2), E(42, 36), E( 5, 29), E(30, 55), E(29, 20), E(29, 35),
    E(34, 12), E(10, 31), E(44, 35), E(30, 14), E(63, 29), E(35, 18),
    E(12, 34), E( 3, 34), E(33, 47), E(32, 51), E(39, 36), E(37, 39),
    E(32, 53), E(32, 10), E(29, 24), E(17, 29), E(32, 20), E(29, 39),
    E(33, 27), E(31, 36), E(32, 27), E(31,  3), E(30, 39), E(44, 32),
    E(31, 37), E( 5, 33), E(44, 30), E(46, 31), E(32, 40), E(36, 25),
    E(35, 40), E(26, 38), E(37, 38), E(33, 38), E(25, 32), E( 2, 32),
    E(58, 33), E(26, 24), E(15, 29), E(31,  1), E(31, 25), E(17, 26),
    E(25, 22), E(34,  2), E(43, 27), E(39, 23), E(34,  8), E(56, 30),
    E(29, 55), E(23, 38), E(30, 60), E(12, 31), E(30,  4), E(10, 34),
    E(35, 49), E( 9, 34), E(24, 27), E(58, 34), E(10, 33), E(12, 30),
    E(54, 31), E(32, 56), E(34, 23), E(38, 27), E( 8, 32), E(22, 31),
    E(34, 24), E(36, 36), E(17, 32), E(38, 36), E(33, 25), E(32, 33),
    E(32, 31), E(58, 32), E(38, 28), E(38, 37), E(38, 26), E(22, 29),
    E(29, 43), E(32, 61), E(25, 36), E(16, 33), E(35, 24), E(32, 55),
    E(35, 43), E(33,  9), E(43, 29), E(18, 34), E(46, 34), E(38, 24),
    E(35, 26), E(32,  6), E(60, 31), E( 1, 32), E(23, 33), E(30, 35),
    E(27, 32), E(33, 26), E(41, 31), E(35, 38), E(33, 54), E(37, 42),
    E(20, 35), E(41, 38), E(35, 46), E(29, 44), E(18, 26), E(61, 27),
    E(30,  9), E(40, 39), E(24, 25), E(39, 21), E(39, 32), E(53, 29),
    E(62, 28), E(26, 42), E(29, 13), E(33,  0), E(34,  5), E(53, 34),
    E(49, 37), E(21, 28), E(27, 24), E(11, 29), E(63, 63), E(50, 33),
    E(33, 18), E(18, 33), E(53, 33), E(60, 30), E(13, 31), E(32, 50),
    E(36, 24), E(24, 34), E(27, 34), E(32, 58), E(38, 35), E(35, 28),
    E(16, 32), E(33, 60), E(51, 30), E(32,  4), E(36, 39), E(43, 34),
    E(45, 30), E(27, 39), E(45, 33), E(42, 33), E(30, 38), E(32, 17),
    E(31, 43), E(20, 26), E(26, 22), E(23, 27), E( 7, 29), E(13, 25),
    E( 6, 33), E(47, 25), E(29, 61), E(30,  0), E( 9, 35), E(51, 25),
    E(22, 36), E(34, 15), E(21, 27), E(25, 40), E(33, 53), E(32, 37),
    E(56, 33), E(57,  3), E(30, 10), E(38, 16), E(51, 37), E(51, 38),
    E(38, 18), E(63, 30), E(28, 46), E(40, 27), E(35,  9), E(33,  6),
    E(42, 28), E(29, 22), E(24, 38), E(30,  2), E(25, 26), E(31, 63),
    E(52, 32), E(31, 57), E(29, 26), E(35, 31), E(32,  5), E(41, 32),
    E(35, 21), E(38, 39), E( 4, 31), E(30, 40), E(17, 31), E( 9, 33),
    E(22, 28), E(34, 18), E( 4, 30), E(17, 34), E(28, 22), E(55, 33),
    E(42, 29), E(40, 34), E(46, 32), E(38, 34), E(48, 32), E(63, 31),
    E(23, 31), E(39, 33), E(33,  3), E( 3, 31), E(18, 29), E(33, 62),
    E(33,  8), E(24, 42), E(58, 28), E(58, 29), E(34,  3), E(49, 25),
    E(29, 16), E(43, 26), E( 4, 29), E( 1, 35), E( 4, 28), E(42, 27),
    E(35, 51), E(35, 61), E(30, 48), E(17, 37), E( 5,  9), E(56, 34),
    E(25, 41), E(17, 30), E(20, 34), E(47, 35), E(34, 21), E(33, 13),
    E(16, 34), E(40, 35), E(32, 19), E(28, 35), E(33, 36), E(36, 30),
    E(25, 39), E(16, 30), E(42, 30), E(19, 32), E(30, 46), E(53, 32),
    E(32, 23), E(29, 42), E(10, 32), E(11, 31), E(14, 33), E(34, 38),
    E(32, 39), E(41, 29), E(26, 26), E(61,  7), E(25, 49), E(22, 33),
    E(28, 38), E(36, 38), E(45, 32), E(34, 27), E(28, 30), E(34, 28),
    E(33, 59), E(37, 45), E(36, 20), E(55, 29), E(28, 21), E(35,  5),
    E(29,  5), E(50, 29), E(48, 28), E(52, 34), E( 2, 29), E(42, 24),
    E(34, 10), E(40, 24), E(46, 35), E(46, 36), E(43, 38), E(33, 11),
    E( 4, 33), E(33, 40), E(32, 18), E(36, 34), E(27, 35), E(35, 22),
    E(35, 55), E(29, 11), E(29, 38), E(41, 33), E(29, 28), E( 7, 32),
    E(44, 31), E(26, 25), E(39, 29), E(32,  3), E(16, 31), E(31, 53),
    E(26, 27), E(34, 43), E(38, 25), E(29, 40), E(41, 35), E(35, 27),
    E(36, 29), E(38, 31), E(29, 27), E(32, 43), E(27, 29), E(30, 37),
    E(24, 32), ESCAPE,    E(32, 63), E(24, 31), E(42, 34), E(48, 36),
    E(20, 38), E(29, 53), E(31, 54), E(61, 33), E(41, 26), E( 7, 30),
    E(30, 49), E(35, 20), E(19, 27), E(14, 30), E(21, 39), E( 8, 33),
    E(39, 41), E(39, 49), E(40, 22), E(46, 38), E(55, 38), E(34,  4),
    E( 6, 30), E(30,  8), E(34,  9), E(37,  3), E(25, 24), E(37, 22),
    E(33, 50), E(22, 37), E(44, 36), E(52, 31), E(17, 27), E(35,  2),
    E(31, 50), E(30, 21), E(24, 36), E(35, 33), E(22, 38), E(61, 30),
    E(32, 12), E( 9, 31), E(45, 34), E(34, 20), E(31, 15), E(19, 29),
    E( 9, 32), E(31, 62), E(18, 32), E(33, 17), E(33,  1), E(37, 34),
    E(32, 15), E(22, 30), E(26, 30), E(59, 31), E(29,  9), E(36, 42),
    E(46, 30), E(31, 13), E(35, 17), E(54, 32), E(29, 19), E(57, 31),
    E(30, 20), E(50, 31), E( 3, 30), E(31,  7), E(63,  1), E(34, 17),
    E(47, 34), E(41, 37), E(35, 63), E(40, 25), E(25, 30), E(37, 28),
    E( 1, 31), E(26, 28), E(22, 34), E(35, 37), E(34, 32), E(60, 32),
    E(27, 30), E(37, 19), E(28, 44), E(30,  1), E(50, 28), E(14, 28),
    E(28, 48), E(55, 30), E( 6, 34), E(23, 41), E(19, 41), E(14, 38),
    E(30, 12), E( 3, 27), E(30, 15), E(28,  0), E(28, 16), E(61, 34),
    E(61, 35), E(47, 38), E(45, 28), E(48, 29), E(40, 40), E(40, 42),
    E(34, 51), E(34, 52), E(25, 45), E(30, 54), E(29, 59), E(35, 59),
    E( 4, 42), E(51, 31), E(18, 31), E(31, 44), E(14, 34), E(37, 26),
    E(36, 35), E(37, 35), E(30, 26), E(31, 41), E(31, 39), E(23, 32),
    E(23, 29), E(38, 40), E(31, 16), E(24, 37), E(32, 52), E(31, 48),
    E(50, 30), E(28, 34), E(32, 21), E(33, 20), E(31, 18), E(49, 32),
    E(34, 37), E(33, 29), E(31, 29), E(40, 31), E( 3, 33), E(28, 27),
    E(26, 35), E(28, 28), E(33, 39), E(34, 26), E(26, 44), E(53, 37),
    E(50, 26), E(26, 46), E(41, 24), E(27, 15), E(59, 39), E(27, 22),
    E(29, 31), E(38, 46), E(39, 19), E( 3, 35), E(50, 38), E(43, 17),
    E(47, 37), E(23, 23), E(33, 52), E(55, 55), E(35, 50), E(21, 37),
    E(23, 26), E(35, 11), E(61, 37), E(33, 12), E(46, 24), E(52, 30),
    E(35, 16), E(34, 13), E(24, 22), E(30, 13), E(43, 36), E(16, 35),
    E(37, 40), E(21, 41), E( 2, 34), E(54, 33), E(27, 38), E( 9, 29),
    E(33,  7), E(23, 25), E(19, 30), E(32, 54), E(29, 45), E(29, 47),
    E(33, 46), E(28, 41), E(27, 27), E(32, 49), E(39, 30), E(33, 31),
    E(59, 32), E(32, 42), E(33, 22), E(47, 29), E(49, 29), E(32, 59),
    E(40, 32), E(31, 24), E(27, 37), E(23, 34), E(28, 37), E(36, 26),
    E(32,  7), E(38, 32), E(29, 33), E(15, 32), E(30, 16), E(47, 30),
    E(33, 14), E(27, 26), E(25, 38), E(54, 34), E(44, 34), E(45, 29),
    E(50, 34), E(58, 30), E(51, 29), E(48, 30), E(33, 57), E(59, 30),
    E( 6, 32), E(34, 22), E(27, 28), E(31, 45), E(30, 27), E(24, 33),
    E(26, 29), E(33,  4), E(24, 41), E(45, 26), E(23, 30), E( 2, 31),
    E(28, 29), E(35, 36), E(30, 28), E(34, 53), E(30, 51), E(55,  5),
    E(55, 21), E( 1, 30), E(29,  2), E(29, 63), E(26, 41), E(28, 20),
    E(25,  1), E(31, 56), E(36, 21), E(22, 24), E(17, 25), E(30,  5),
    E(52, 28), E( 2, 35), E(44, 26), E(44, 28), E(57, 30), E(26, 18),
    E(62, 29), E(41, 23), E(39, 40), E(57, 34), E(26, 20), E(46, 42),
    E(49, 39), E(11, 34), E(16, 28), E(19, 61), E( 5, 38), E(34, 46),
    E(57, 33), E(13, 29), E(26, 39), E(33, 44), E(13, 30), E(48, 31),
    E(30, 50), E(11, 33), E( 7, 31), E(56, 32), E(34, 19), E(33, 43),
    E(32, 46), E(34, 41), E(41, 30), E(36, 28), E(34, 31), E(33, 33),
    E(33, 35), E(26, 34), E(28, 26), E(32, 13), E(41, 34), E( 0, 32),
    E(32, 57), E(35, 30), E(35, 32), E(55, 32), E(26, 40), E( 0, 31),
    E(26, 37), E(35, 47), E( 1,  1), E(51, 35), E(33, 16), E(53, 31),
    E(31,  0), E(45, 35), E(12, 33), E(29,  1), E(29, 17), E(36, 27),
    E( 5, 32), E(38, 38), E(59, 27), E(38, 62), E(39,  9), E(58, 36),
    E(34, 42), E(31, 42), E(36, 31), E(47, 33), E(32, 47), E(17, 33),
    E(31, 60), E(31, 23), E(33, 41), E(34, 30), E(34, 36), E(32, 60),
    E(29, 36), E(38, 17), E(25,  9), E(49, 49), E(50, 24), E(22, 25),
    E(28, 54), E(59, 34), E( 9, 39), E(59, 37), E( 7,  3), E(28,  1),
    E(55, 25), E(17, 38), E( 4, 38), E(35, 44), E(50, 35), E(21, 36),
    E(29, 12), E(44, 60), E(47, 36), E(18, 38), E(24, 40), E(19, 25),
    E(25, 43), E(31, 31),
};

const uint8_t ff_msmp4_mv_table1_lens[MSMPEG4_MV_TABLES_NB_ELEMS] = {
     2, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 13, 13, 13, 14, 15, 15, 11, 13, 13, 12, 11, 10,
     8, 14, 14, 14, 14, 14, 14, 14, 14, 13, 13, 12, 10, 12, 12, 12, 12, 10,
    14, 15, 15, 13, 13, 13, 12, 12, 11, 13, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15,  9,  6,  9, 10, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 13, 13, 11, 11, 12, 13, 13, 11, 12, 13, 13, 10,  7,  8, 10, 14,
    15, 15, 14, 14, 12, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 13, 13, 13,
    13, 13, 12, 13, 13, 13, 13, 13, 13, 13, 13,  8,  9,  9,  5, 13, 14, 14,
    12, 12, 14, 14, 14, 14, 10, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 13,
    12, 12, 12, 11, 10, 11, 13, 13, 13, 13, 12, 12, 12, 13, 14, 14, 11, 12,
    14, 14, 14, 14,  7,  8,  9, 10, 14, 14, 13, 12, 11,  7,  8, 10, 13, 14,
    14, 13, 13, 13, 13, 13, 13, 13, 13, 12, 11, 13, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14,  8, 13, 13, 13, 13, 12, 12, 10, 10, 10,
    10, 11, 14, 14, 13, 13, 13, 12, 13, 14, 14, 11, 10,  9, 10, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 13, 13,  8, 11, 12, 13, 13, 10, 10,
    10,  8,  8, 13, 13, 12, 12, 12, 10, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 13, 13, 13, 13, 13, 13, 13, 12,  8,  9, 10, 13, 13,
    13, 13, 13, 13, 14, 14, 14, 14, 10, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 13, 13, 13,  9, 13, 13, 13, 13, 12, 12, 12, 13, 13, 11, 10, 10, 10,
    11, 13, 13, 13, 13,  9,  4, 12, 12, 12, 12, 10, 10, 13, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 13, 13, 11, 10,
    13, 13, 13, 13, 12, 12, 10,  9, 13, 13, 12, 11, 11, 13, 14, 14, 12, 11,
    12, 14, 14, 14, 14, 10, 14, 14, 14, 14, 14, 14, 14, 14, 13, 13, 13, 13,
    13, 13, 13, 13, 12, 12, 10, 10,  9, 11, 12, 12, 10, 12, 13, 13, 11, 13,
    13, 13, 13, 11,  9, 11, 13, 14, 14, 12, 10, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 13, 13, 13, 13, 12,
    12, 12, 12, 12, 10,  5,  4, 12, 12, 13, 13, 13, 13, 10, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 12, 12, 12, 11, 11, 10,  9, 11, 11, 12, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14,  9, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 13, 13, 13, 13, 13, 13, 13, 13, 12, 11, 12, 12, 11,
    12, 13, 13, 10, 13, 13, 13, 13, 12, 12, 11, 12, 12, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,  9, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 13, 13, 13, 13, 13, 13, 13, 13, 12,  8, 10, 10,
    13, 13, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13, 12, 12, 11, 12, 12,
    11, 10, 11, 11, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 13, 13, 13, 13, 13, 13, 13, 13, 12, 11, 10, 10,
    13, 13, 12, 12, 13, 13, 10, 13, 13, 13, 13, 11,  9, 12, 13, 14, 14, 12,
    12, 12, 12, 11, 10, 10, 12, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 13, 13, 12, 12, 10, 11, 13, 14, 14, 12, 11, 11, 12,
    13, 13, 11,  9, 13, 13, 13, 13, 13, 13, 12, 11, 11, 10, 11, 11, 11, 11,
    10,  4, 11, 11, 12, 14, 14, 14, 14, 11, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    13, 13, 13,  8, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 12, 12, 12, 11,
    12, 12, 11, 11, 14, 14, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 14, 14, 11, 12, 12, 12, 12, 11,  6, 10, 11, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 13, 13, 13, 13, 13, 11, 11, 11, 11, 10, 10,
    12, 13, 13, 13, 13, 13, 13, 10, 10, 13, 13, 12, 11,  8,  8, 11, 11, 12,
    12, 11, 10, 11, 14, 14, 14, 14, 14, 14, 14, 14,  8, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 12, 12, 11,  5,
    10, 11, 12, 13, 13, 10, 10, 11, 12, 12, 12, 12, 11,  9,  8, 12, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 12, 12, 12, 12, 11, 11,
    12, 13, 14, 14, 12, 12, 11, 11, 10, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 12,
    12, 12, 12, 11,  7,  5,  8, 11, 12, 12, 12, 12, 11,  9,  7, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 12, 10, 12, 14, 14, 14, 14,
    12, 12,  9, 12, 12, 12, 12, 11, 11,  8, 10, 10, 11, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14,  4,
};

const uint8_t ff_v2_mb_type[8][2] = {
    { 1, 1 }, {    0, 2 }, {    3, 3 }, {    9, 5 },
    { 5, 4 }, { 0x21, 7 }, { 0x20, 7 }, { 0x11, 6 },
};

const uint8_t ff_v2_intra_cbpc[4][2] = {
    { 1, 1 }, { 0, 3 }, { 1, 3 }, { 1, 2 },
};

const uint8_t ff_wmv1_y_dc_scale_table[32] = {
     0,  8,  8,  8,  8,  8,  9,  9, 10, 10, 11, 11, 12, 12, 13, 13,
    14, 14, 15, 15, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21
};
const uint8_t ff_wmv1_c_dc_scale_table[32] = {
     0,  8,  8,  8,  8,  9,  9, 10, 10, 11, 11, 12, 12, 13, 13, 14,
    14, 15, 15, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22
};

const uint8_t ff_old_ff_y_dc_scale_table[32] = {
     0,  8,  8,  8,  8, 10, 12, 14, 16, 17, 18, 19, 20, 21, 22, 23,
    24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39
};


const uint8_t ff_table_inter_intra[4][2] = {
    { 0, 1 } /* Luma-Left Chroma-Left */,
    { 2, 2 } /* Luma-Top  Chroma-Left */,
    { 6, 3 } /* luma-Left Chroma-Top  */,
    { 7, 3 } /* luma-Top  Chroma-Top  */
};

static const uint32_t table_mb_non_intra2[128][2] = {
    { 0x0000A7, 14 }, { 0x01B2B8, 18 }, { 0x01B28E, 18 }, { 0x036575, 19 },
    { 0x006CAC, 16 }, { 0x000A69, 18 }, { 0x002934, 20 }, { 0x00526B, 21 },
    { 0x006CA1, 16 }, { 0x01B2B9, 18 }, { 0x0029AD, 20 }, { 0x029353, 24 },
    { 0x006CA7, 16 }, { 0x006CAB, 16 }, { 0x01B2BB, 18 }, { 0x00029B, 16 },
    { 0x00D944, 17 }, { 0x000A6A, 18 }, { 0x0149A8, 23 }, { 0x03651F, 19 },
    { 0x006CAF, 16 }, { 0x000A4C, 18 }, { 0x03651E, 19 }, { 0x000A48, 18 },
    { 0x00299C, 20 }, { 0x00299F, 20 }, { 0x029352, 24 }, { 0x0029AC, 20 },
    { 0x000296, 16 }, { 0x00D946, 17 }, { 0x000A68, 18 }, { 0x000298, 16 },
    { 0x000527, 17 }, { 0x00D94D, 17 }, { 0x0014D7, 19 }, { 0x036574, 19 },
    { 0x000A5C, 18 }, { 0x01B299, 18 }, { 0x00299D, 20 }, { 0x00299E, 20 },
    { 0x000525, 17 }, { 0x000A66, 18 }, { 0x00A4D5, 22 }, { 0x00149B, 19 },
    { 0x000295, 16 }, { 0x006CAD, 16 }, { 0x000A49, 18 }, { 0x000521, 17 },
    { 0x006CAA, 16 }, { 0x00D945, 17 }, { 0x01B298, 18 }, { 0x00052F, 17 },
    { 0x003654, 15 }, { 0x006CA0, 16 }, { 0x000532, 17 }, { 0x000291, 16 },
    { 0x003652, 15 }, { 0x000520, 17 }, { 0x000A5D, 18 }, { 0x000294, 16 },
    { 0x00009B, 11 }, { 0x0006E2, 12 }, { 0x000028, 12 }, { 0x0001B0, 10 },
    { 0x000001,  3 }, { 0x000010,  8 }, { 0x00002F,  6 }, { 0x00004C, 10 },
    { 0x00000D,  4 }, { 0x000000, 10 }, { 0x000006,  9 }, { 0x000134, 12 },
    { 0x00000C,  4 }, { 0x000007, 10 }, { 0x000007,  9 }, { 0x0006E1, 12 },
    { 0x00000E,  5 }, { 0x0000DA,  9 }, { 0x000022,  9 }, { 0x000364, 11 },
    { 0x00000F,  4 }, { 0x000006, 10 }, { 0x00000F,  9 }, { 0x000135, 12 },
    { 0x000014,  5 }, { 0x0000DD,  9 }, { 0x000004,  9 }, { 0x000015, 11 },
    { 0x00001A,  6 }, { 0x0001B3, 10 }, { 0x000005, 10 }, { 0x0006E3, 12 },
    { 0x00000C,  5 }, { 0x0000B9,  8 }, { 0x000004,  8 }, { 0x0000DB,  9 },
    { 0x00000E,  4 }, { 0x00000B, 10 }, { 0x000023,  9 }, { 0x0006CB, 12 },
    { 0x000005,  6 }, { 0x0001B1, 10 }, { 0x000001, 10 }, { 0x0006E0, 12 },
    { 0x000011,  5 }, { 0x0000DF,  9 }, { 0x00000E,  9 }, { 0x000373, 11 },
    { 0x000003,  5 }, { 0x0000B8,  8 }, { 0x000006,  8 }, { 0x000175,  9 },
    { 0x000015,  5 }, { 0x000174,  9 }, { 0x000027,  9 }, { 0x000372, 11 },
    { 0x000010,  5 }, { 0x0000BB,  8 }, { 0x000005,  8 }, { 0x0000DE,  9 },
    { 0x00000F,  5 }, { 0x000001,  9 }, { 0x000012,  8 }, { 0x000004, 10 },
    { 0x000002,  3 }, { 0x000016,  5 }, { 0x000009,  4 }, { 0x000001,  5 },
};

static const uint32_t table_mb_non_intra3[128][2] = {
    { 0x0002A1, 10 }, { 0x005740, 15 }, { 0x01A0BF, 18 }, { 0x015D19, 17 },
    { 0x001514, 13 }, { 0x00461E, 15 }, { 0x015176, 17 }, { 0x015177, 17 },
    { 0x0011AD, 13 }, { 0x00682E, 16 }, { 0x0682F9, 20 }, { 0x03417D, 19 },
    { 0x001A36, 14 }, { 0x002A2D, 14 }, { 0x00D05E, 17 }, { 0x006824, 16 },
    { 0x001515, 13 }, { 0x00545C, 15 }, { 0x0230E9, 18 }, { 0x011AFA, 17 },
    { 0x0015D7, 13 }, { 0x005747, 15 }, { 0x008D79, 16 }, { 0x006825, 16 },
    { 0x002BA2, 14 }, { 0x00A8BA, 16 }, { 0x0235F6, 18 }, { 0x015D18, 17 },
    { 0x0011AE, 13 }, { 0x00346F, 15 }, { 0x008C3B, 16 }, { 0x00346E, 15 },
    { 0x000D1A, 13 }, { 0x00461F, 15 }, { 0x0682F8, 20 }, { 0x011875, 17 },
    { 0x002BA1, 14 }, { 0x008D61, 16 }, { 0x0235F7, 18 }, { 0x0230E8, 18 },
    { 0x001513, 13 }, { 0x008D7B, 16 }, { 0x011AF4, 17 }, { 0x011AF5, 17 },
    { 0x001185, 13 }, { 0x0046BF, 15 }, { 0x008D60, 16 }, { 0x008D7C, 16 },
    { 0x001512, 13 }, { 0x00461C, 15 }, { 0x00AE8D, 16 }, { 0x008D78, 16 },
    { 0x000D0E, 13 }, { 0x003413, 15 }, { 0x0046B1, 15 }, { 0x003416, 15 },
    { 0x000AEA, 12 }, { 0x002A2C, 14 }, { 0x005741, 15 }, { 0x002A2F, 14 },
    { 0x000158,  9 }, { 0x0008D2, 12 }, { 0x00054C, 11 }, { 0x000686, 12 },
    { 0x000000,  2 }, { 0x000069,  8 }, { 0x00006B,  8 }, { 0x00068C, 12 },
    { 0x000007,  3 }, { 0x00015E,  9 }, { 0x0002A3, 10 }, { 0x000AE9, 12 },
    { 0x000006,  3 }, { 0x000231, 10 }, { 0x0002B8, 10 }, { 0x001A08, 14 },
    { 0x000010,  5 }, { 0x0001A9, 10 }, { 0x000342, 11 }, { 0x000A88, 12 },
    { 0x000004,  4 }, { 0x0001A2, 10 }, { 0x0002A4, 10 }, { 0x001184, 13 },
    { 0x000012,  5 }, { 0x000232, 10 }, { 0x0002B2, 10 }, { 0x000680, 12 },
    { 0x00001B,  6 }, { 0x00046A, 11 }, { 0x00068E, 12 }, { 0x002359, 14 },
    { 0x000016,  5 }, { 0x00015F,  9 }, { 0x0002A0, 10 }, { 0x00054D, 11 },
    { 0x000005,  4 }, { 0x000233, 10 }, { 0x0002B9, 10 }, { 0x0015D6, 13 },
    { 0x000022,  6 }, { 0x000468, 11 }, { 0x000683, 12 }, { 0x001A0A, 14 },
    { 0x000013,  5 }, { 0x000236, 10 }, { 0x0002BB, 10 }, { 0x001186, 13 },
    { 0x000017,  5 }, { 0x0001AB, 10 }, { 0x0002A7, 10 }, { 0x0008D3, 12 },
    { 0x000014,  5 }, { 0x000237, 10 }, { 0x000460, 11 }, { 0x000D0F, 13 },
    { 0x000019,  6 }, { 0x0001AA, 10 }, { 0x0002B3, 10 }, { 0x000681, 12 },
    { 0x000018,  6 }, { 0x0001A8, 10 }, { 0x0002A5, 10 }, { 0x00068F, 12 },
    { 0x000007,  4 }, { 0x000055,  7 }, { 0x000047,  7 }, { 0x0000AD,  8 },
};

static const uint32_t table_mb_non_intra4[128][2] = {
    { 0x0000D4,  8 }, { 0x0021C5, 14 }, { 0x00F18A, 16 }, { 0x00D5BC, 16 },
    { 0x000879, 12 }, { 0x00354D, 14 }, { 0x010E3F, 17 }, { 0x010F54, 17 },
    { 0x000866, 12 }, { 0x00356E, 14 }, { 0x010F55, 17 }, { 0x010E3E, 17 },
    { 0x0010CE, 13 }, { 0x003C84, 14 }, { 0x00D5BD, 16 }, { 0x00F18B, 16 },
    { 0x000868, 12 }, { 0x00438C, 15 }, { 0x0087AB, 16 }, { 0x00790B, 15 },
    { 0x000F10, 12 }, { 0x00433D, 15 }, { 0x006AD3, 15 }, { 0x00790A, 15 },
    { 0x001AA7, 13 }, { 0x0043D4, 15 }, { 0x00871E, 16 }, { 0x006ADF, 15 },
    { 0x000D7C, 12 }, { 0x003C94, 14 }, { 0x00438D, 15 }, { 0x006AD2, 15 },
    { 0x0006BC, 11 }, { 0x0021E9, 14 }, { 0x006ADA, 15 }, { 0x006A99, 15 },
    { 0x0010F7, 13 }, { 0x004389, 15 }, { 0x006ADB, 15 }, { 0x0078C4, 15 },
    { 0x000D56, 12 }, { 0x0035F7, 14 }, { 0x00438E, 15 }, { 0x006A98, 15 },
    { 0x000D52, 12 }, { 0x003C95, 14 }, { 0x004388, 15 }, { 0x00433C, 15 },
    { 0x000D54, 12 }, { 0x001E4B, 13 }, { 0x003C63, 14 }, { 0x003C83, 14 },
    { 0x000861, 12 }, { 0x0021EB, 14 }, { 0x00356C, 14 }, { 0x0035F6, 14 },
    { 0x000863, 12 }, { 0x00219F, 14 }, { 0x003568, 14 }, { 0x003C82, 14 },
    { 0x0001AE,  9 }, { 0x0010C0, 13 }, { 0x000F11, 12 }, { 0x001AFA, 13 },
    { 0x000000,  1 }, { 0x0000F0,  8 }, { 0x0001AD,  9 }, { 0x0010C1, 13 },
    { 0x00000A,  4 }, { 0x0003C5, 10 }, { 0x000789, 11 }, { 0x001AB5, 13 },
    { 0x000009,  4 }, { 0x000435, 11 }, { 0x000793, 11 }, { 0x001E40, 13 },
    { 0x00001D,  5 }, { 0x0003CB, 10 }, { 0x000878, 12 }, { 0x001AAF, 13 },
    { 0x00000B,  4 }, { 0x0003C7, 10 }, { 0x000791, 11 }, { 0x001AAB, 13 },
    { 0x00001F,  5 }, { 0x000436, 11 }, { 0x0006BF, 11 }, { 0x000F19, 12 },
    { 0x00003D,  6 }, { 0x000D51, 12 }, { 0x0010C4, 13 }, { 0x0021E8, 14 },
    { 0x000036,  6 }, { 0x000437, 11 }, { 0x0006AF, 11 }, { 0x0010C5, 13 },
    { 0x00000C,  4 }, { 0x000432, 11 }, { 0x000794, 11 }, { 0x001E30, 13 },
    { 0x000042,  7 }, { 0x000870, 12 }, { 0x000F24, 12 }, { 0x001E43, 13 },
    { 0x000020,  6 }, { 0x00043E, 11 }, { 0x000795, 11 }, { 0x001AAA, 13 },
    { 0x000037,  6 }, { 0x0006AC, 11 }, { 0x0006AE, 11 }, { 0x0010F6, 13 },
    { 0x000034,  6 }, { 0x00043A, 11 }, { 0x000D50, 12 }, { 0x001AAE, 13 },
    { 0x000039,  6 }, { 0x00043F, 11 }, { 0x00078D, 11 }, { 0x0010D2, 13 },
    { 0x000038,  6 }, { 0x00043B, 11 }, { 0x0006BD, 11 }, { 0x0010D3, 13 },
    { 0x000011,  5 }, { 0x0001AC,  9 }, { 0x0000F3,  8 }, { 0x000439, 11 },
};

const uint32_t (*const ff_wmv2_inter_table[WMV2_INTER_CBP_TABLE_COUNT])[2] = {
    table_mb_non_intra2,
    table_mb_non_intra3,
    table_mb_non_intra4,
    ff_table_mb_non_intra,
};
