// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief K0 and Lambda spectra at 14, 22 and 34 GeV.
  class TASSO_1985_I205119 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1985_I205119);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(Beam(), "Beams");
      declare(UnstableParticles(), "UFS");

      // Book histograms
      size_t ih = 0;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en+"kaon_x"],    1+ih,1,1);
        book(_h[en+"lambda_x"],  4+ih,1,1);
        book(_h[en+"kaon_p"],    7+ih,1,1);
        book(_h[en+"lambda_p"], 10+ih,1,1);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
      MSG_DEBUG("Avg beam momentum = " << meanBeamMom);

      for (const Particle& p : apply<UnstableParticles>(event, "UFS").particles(Cuts::abspid==PID::LAMBDA ||
                                                                                Cuts::pid==130 || Cuts::pid==310)) {
        const double xE = p.E()/meanBeamMom;
        const double modp = p.p3().mod();
        const double beta = modp/p.E();
        if (p.abspid()==PID::LAMBDA) {
          _h[_sqs+"lambda_x"]->fill(xE,1./beta);
          _h[_sqs+"lambda_p"]->fill(modp,1.);
        }
        else {
          _h[_sqs+"kaon_x"]->fill(xE,1./beta);
          _h[_sqs+"kaon_p"]->fill(modp,1.);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        scale({_h[en+"kaon_x"], _h[en+"lambda_x"]}, sqr(eVal)*crossSection()/microbarn/sumOfWeights());
        scale({_h[en+"kaon_p"], _h[en+"lambda_p"]}, crossSection()/nanobarn/sumOfWeights());
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1985_I205119);
}
