// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Sphericity.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief Add a short analysis description here
  class TASSO_1980_I153511 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1980_I153511);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      declare(Beam(), "Beams");

      const ChargedFinalState cfs;
      declare(cfs, "CFS");

      // Thrust and sphericity
      declare(Sphericity(cfs), "Sphericity");

      // Book histograms
      size_t ih = 1;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en+"S"],   ih, 1, 1);
        book(_h[en+"A"], 2+ih, 1, 1);
        book(_h[en+"x"], 4+ih, 1, 1);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
      const ChargedFinalState& cfs = apply<ChargedFinalState>(event, "CFS");
      const Sphericity& sphericity = apply<Sphericity>(event, "Sphericity");
      _h[_sqs+"S"]->fill(sphericity.sphericity());
      _h[_sqs+"A"]->fill(sphericity.aplanarity());
      for (const Particle& p : cfs.particles()) {
        const Vector3 mom3 = p.p3();
        const double mom = mom3.mod();
        const double xp = mom/meanBeamMom;
        _h[_sqs+"x"]->fill(xp);
      }

    }


    /// Normalise histograms etc., after the run
    void finalize() {
      normalize(_h);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1980_I153511);
}
