/*
 * Copyright (C) 2017 ~ 2018 Deepin Technology Co., Ltd.
 *
 * Author:     kirigaya <kirigaya@mkacg.com>
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "basewidget.h"
#include "ddlog.h"

#define ICON_SIZE 420,264

BaseWidget::BaseWidget(QWidget *parent)
    : DFrame(parent)
    , m_borderWidget(new BorderWidget(this))
    , m_layout(new QVBoxLayout(this))
    , m_title(new DLabel(this))
{
    qCDebug(app) << "BaseWidget constructor started";

    m_layout->setContentsMargins(0, 0, 0, 0);
    m_layout->setSpacing(0);
    m_layout->addWidget(m_borderWidget, 0, Qt::AlignCenter);
    m_layout->addSpacing(10);

    qCDebug(app) << "Creating title layout";
    QHBoxLayout *titleLayout = new QHBoxLayout(this);
    m_radioBtn = new DRadioButton();
    titleLayout->addSpacing(3);
    titleLayout->addWidget(m_radioBtn);
    titleLayout->addSpacing(8);
    titleLayout->addWidget(m_title);
    titleLayout->addStretch();

    m_layout->addLayout(titleLayout);
    m_layout->addStretch();

    qCDebug(app) << "Connecting radio button signals";
    connect(m_radioBtn, &DRadioButton::clicked, [=](bool click){
        if (click)
            emit clicked();
        else
            m_radioBtn->setChecked(true);
    });

    setLineWidth(0);
    setLayout(m_layout);
    qCDebug(app) << "BaseWidget initialization completed";
}

void BaseWidget::setLayoutSpacing(int i)
{
    m_layout->setSpacing(i);
}

/*******************************************************************************
 1. @函数:    updateInterface
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    设置边框大小
*******************************************************************************/
void BaseWidget::updateInterface(float f)
{
    m_borderWidget->updateInterface(f);
    m_borderWidget->setFixedSize(m_size * f);
}

/*******************************************************************************
 1. @函数:    setBigPixmap
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    设置第一次启动图片函数封装
*******************************************************************************/
void BaseWidget::setBigPixmap(const QString &url)
{
    qCDebug(app) << "Setting big pixmap from URL:" << url;

    setBigPixmap(DHiDPIHelper::loadNxPixmap(url));
}

/*******************************************************************************
 1. @函数:    setSmallPixmap
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    设置日常启动图片函数封装
*******************************************************************************/
void BaseWidget::setSmallPixmap(const QString &url)
{
    qCDebug(app) << "Setting small pixmap from URL:" << url;

    setSmallPixmap(QPixmap(DHiDPIHelper::loadNxPixmap(url)));
}

/*******************************************************************************
 1. @函数:    setPixmap
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    类外设置图片函数
*******************************************************************************/
void BaseWidget::setPixmap(const QPixmap &pixmap)
{
    qCDebug(app) << "Setting direct pixmap, size:" << pixmap.size();

    m_borderWidget->setFixedSize(QSize(ICON_SIZE) + QSize(10, 10));
    m_borderWidget->setPixmap(pixmap);

    setFixedWidth(m_borderWidget->width());
}

/*******************************************************************************
 1. @函数:    setBigPixmap
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    设置第一次启动图片函数,设置图片到边框界面时，将边框对应的放大，并设置主界面的高度和宽度
*******************************************************************************/
void BaseWidget::setBigPixmap(const QPixmap &pixmap)
{
    qCDebug(app) << "Setting big pixmap, original size:" << pixmap.size();

    QSize scaleSize = QSize(ICON_SIZE) * devicePixelRatioF();
    QPixmap scaledPixmap = pixmap.scaled(scaleSize, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
    m_borderWidget->setFixedSize(QSize(ICON_SIZE) + QSize(10, 10));
    m_borderWidget->setPixmap(pixmap);
    m_size = QSize(ICON_SIZE) + QSize(10, 12);

    setFixedWidth(m_borderWidget->width());
    setFixedHeight(m_borderWidget->height() + m_title->height() + 10);
}

/*******************************************************************************
 1. @函数:    setSmallPixmap
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    设置日常启动图片函数,设置图片到边框界面时，将边框对应的放大，并设置主界面的高度和宽度
*******************************************************************************/
void BaseWidget::setSmallPixmap(const QPixmap &pixmap)
{
    qCDebug(app) << "Setting small pixmap, original size:" << pixmap.size();

    QSize scaleSize = QSize(240, 150) * devicePixelRatioF();
    QPixmap scaledPixmap = pixmap.scaled(scaleSize, Qt::IgnoreAspectRatio);
    m_borderWidget->setFixedSize(QSize(240, 150) + QSize(10, 10));
    m_borderWidget->setPixmap(scaledPixmap);
    m_size = QSize(240, 150) + QSize(10, 10);

    setFixedWidth(m_borderWidget->width());
}

/*******************************************************************************
 1. @函数:    setTitle
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    设置图片的标题
*******************************************************************************/
void BaseWidget::setTitle(const QString &title)
{
    qCDebug(app) << "Setting widget title to:" << title;

    m_title->setText(title);
}

/*******************************************************************************
 1. @函数:    setChecked
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    设置边框可检查状态
*******************************************************************************/
void BaseWidget::setChecked(bool checked)
{
    qCDebug(app) << "Setting checked state to:" << checked;

    m_borderWidget->setChecked(checked);
    m_radioBtn->setChecked(checked);
}

/*******************************************************************************
 1. @函数:    mouseReleaseEvent
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    模仿鼠标点击功能
*******************************************************************************/
void BaseWidget::mouseReleaseEvent(QMouseEvent *event)
{
    QFrame::mouseReleaseEvent(event);

    emit clicked();
}

void BaseWidget::mouseDoubleClickEvent(QMouseEvent *event)
{
    qCDebug(app) << "Mouse double click event received";

    Q_UNUSED(event);
    return;
}

void BaseWidget::resizeEvent(QResizeEvent *event)
{
    DFrame::resizeEvent(event);

    emit sizeChanged();
}
